#
# Copyright 2017 Russell Smiley
#
# This file is part of registerMap.
#
# registerMap is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# registerMap is distributed in the hope that it will be useful
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with registerMap.  If not, see <http://www.gnu.org/licenses/>.
#

import collections


# https://code.activestate.com/recipes/576694/
class OrderedSet( collections.MutableSet ) :
    def __init__( self, iterable = None ) :
        self.end = end = [ ]
        end += [ None, end, end ]  # sentinel node for doubly linked list
        self.map = { }  # key --> [key, prev, next]
        if iterable is not None :
            self |= iterable


    def __len__( self ) :
        return len( self.map )


    def __contains__( self, key ) :
        return key in self.map


    def add( self, key ) :
        if key not in self.map :
            end = self.end
            curr = end[ 1 ]
            curr[ 2 ] = end[ 1 ] = self.map[ key ] = [ key, curr, end ]


    def discard( self, key ) :
        if key in self.map :
            key, prev, next = self.map.pop( key )
            prev[ 2 ] = next
            next[ 1 ] = prev


    def remove( self, key ) :
        if key in self.map :
            key, prev, next = self.map.pop( key )
            prev[ 2 ] = next
            next[ 1 ] = prev
        else :
            raise KeyError( 'item not in set' )


    def __iter__( self ) :
        end = self.end
        curr = end[ 2 ]
        while curr is not end :
            yield curr[ 0 ]
            curr = curr[ 2 ]


    def __reversed__( self ) :
        end = self.end
        curr = end[ 1 ]
        while curr is not end :
            yield curr[ 0 ]
            curr = curr[ 1 ]


    def pop( self, last = True ) :
        if not self :
            raise KeyError( 'set is empty' )
        key = self.end[ 1 ][ 0 ] if last else self.end[ 2 ][ 0 ]
        self.discard( key )
        return key


    def __repr__( self ) :
        if not self :
            return '%s()' % (self.__class__.__name__,)
        return '%s(%r)' % (self.__class__.__name__, list( self ))


    def __eq__( self, other ) :
        if isinstance( other, OrderedSet ) :
            return len( self ) == len( other ) and list( self ) == list( other )
        return set( self ) == set( other )


class ElementSet( OrderedSet ) :
    def __init__( self ) :
        super().__init__()


    def find( self, name ) :
        """
        Find Element's in the ElementSet with the specified name. Assumes Element's have a name property for comparison.

        :param name: Name of Element to find in set.

        :return: Set of Element.
        """
        foundFromName = { x for x in self if x[ 'name' ] == name }

        return foundFromName
