# Load required packages

library(dplyr)
library(tibble)

library(devtools)
install_github("dsrobertson/onlineFDR")
library(onlineFDR)


################################################################################
############################ Simulation study ##################################
################################################################################

set.seed(7)

N = 3000

F1 = rnorm(N, sd = sqrt(2*log(N)))

pi.theta = sample(c(rep(1, 600), rep(0, 2400)))

theta = rep(0,N)
theta[pi.theta==1] = F1[pi.theta==1]

Z = theta + rnorm(N)

pval = 2*pnorm(-abs(Z))

d.sim = data.frame(id = 1:N, pval)


# BH method
pval.BH = p.adjust(pval, method = "BH")
R.BH = pval.BH <= 0.05

R.BH.false = sum(R.BH[pi.theta == 0] == 1)
R.BH.true = sum(R.BH[pi.theta == 1] == 1)

BH.FDR = sum(R.BH[pi.theta == 0] == 1)/sum(R.BH == 1)

# BH (dep)
pval.BY = p.adjust(pval, method = "BY")
R.BY = pval.BY <= 0.05

R.BY.false = sum(R.BY[pi.theta == 0] == 1)
R.BY.true = sum(R.BY[pi.theta == 1] == 1)

BY.FDR = sum(R.BY[pi.theta == 0] == 1)/sum(R.BY == 1)

# LORD 3
LORD3.sim = LORD(d.sim, random = FALSE)
R.LORD3 = LORD3.sim$R

R.LORD3.false = sum(R.LORD3[pi.theta == 0] == 1)
R.LORD3.true = sum(R.LORD3[pi.theta == 1] == 1)

LORD3.FDR = sum(R.LORD3[pi.theta == 0] == 1)/sum(R.LORD3 == 1)

# LORD++
LORD.plus.sim = LORD(d.sim, version = '++', random = FALSE)
R.LORD.plus = LORD.plus.sim$R

R.LORD.plus.false = sum(R.LORD.plus[pi.theta == 0] == 1)
R.LORD.plus.true = sum(R.LORD.plus[pi.theta == 1] == 1)

LORD.plus.FDR = sum(R.LORD.plus[pi.theta == 0] == 1)/sum(R.LORD.plus == 1)

# LORD 2
LORD2.sim = LORD(d.sim, version = 2, random = FALSE)
R.LORD2 = LORD2.sim$R

R.LORD2.false = sum(R.LORD2[pi.theta == 0] == 1)
R.LORD2.true = sum(R.LORD2[pi.theta == 1] == 1)

LORD2.FDR = sum(R.LORD2[pi.theta == 0] == 1)/sum(R.LORD2 == 1)

# LOND
LOND.sim = LOND(d.sim, random = FALSE)
R.LOND = LOND.sim$R

R.LOND.false = sum(R.LOND[pi.theta == 0] == 1)
R.LOND.true = sum(R.LOND[pi.theta == 1] == 1)

LOND.FDR = sum(R.LOND[pi.theta == 0] == 1)/sum(R.LOND == 1)

# LORD (dep)
LORDdep.sim = LORDdep(d.sim, random = FALSE)
R.LORDdep = LORDdep.sim$R

R.LORDdep.false = sum(R.LORDdep[pi.theta == 0] == 1)
R.LORDdep.true = sum(R.LORDdep[pi.theta == 1] == 1)

LORDdep.FDR = sum(R.LORDdep[pi.theta == 0] == 1)/sum(R.LORDdep == 1)

# LOND (dep)
LONDdep.sim = LOND(d.sim, dep = TRUE, random = FALSE)
R.LONDdep = LONDdep.sim$R

R.LONDdep.false = sum(R.LONDdep[pi.theta == 0] == 1)
R.LONDdep.true = sum(R.LONDdep[pi.theta == 1] == 1)

LONDdep.FDR = sum(R.LONDdep[pi.theta == 0] == 1)/sum(R.LONDdep == 1)

# Bonferroni
bonf.sim = bonfInfinite(d.sim, random = FALSE)
R.bonf = bonf.sim$R

R.bonf.false = sum(R.bonf[pi.theta == 0] == 1)
R.bonf.true = sum(R.bonf[pi.theta == 1] == 1)

bonf.FDR = sum(R.bonf[pi.theta == 0] == 1)/sum(R.bonf == 1)

sim.total.rejections = matrix(c(sum(pi.theta), sum(R.BH), sum(R.LORD3),
                                sum(R.LORD.plus), sum(R.LORD2), sum(R.LOND),
                                sum(R.BY), sum(R.LORDdep), sum(R.LONDdep),
                                sum(R.bonf), N),
                              nrow = 1,
                              dimnames = list(c(""),
                                              c("Actual", "BH", "LORD 3",
                                                "LORD++", "LORD 2", "LOND",
                                                "BY", "LORD (dep)",
                                                "LOND (dep)",
                                                "Bonferroni", "N")))

sim.total.rejections



sim.false.rejections = matrix(c(R.BH.false, R.LORD3.false, R.LORD.plus.false,
                                R.LORD2.false, R.LOND.false, R.BY.false,
                                R.LORDdep.false, R.LONDdep.false, R.bonf.false),
                              nrow = 1,
                              dimnames = list(c(""),
                                              c("BH", "LORD 3", "LORD++",
                                                "LORD 2", "LOND", "BY",
                                                "LORD (dep)", "LOND (dep)",
                                                "Bonferroni")))

sim.false.rejections


sim.true.rejections = matrix(c(R.BH.true, R.LORD3.true, R.LORD.plus.true,
                               R.LORD2.true, R.LOND.true, R.BY.true,
                               R.LORDdep.true, R.LONDdep.true, R.bonf.true),
                             nrow = 1,
                             dimnames = list(c(""),
                                             c("BH", "LORD 3", "LORD++",
                                               "LORD 2", "LOND", "BY",
                                               "LORD (dep)", "LOND (dep)",
                                               "Bonferroni")))

sim.true.rejections


sim.FDR = matrix(c(BH.FDR, LORD3.FDR, LORD.plus.FDR, LORD2.FDR, LOND.FDR, 
                   BY.FDR, LORDdep.FDR, LONDdep.FDR, bonf.FDR), nrow = 1,
                 dimnames = list(c(""),
                                 c("BH", "LORD 3", "LORD++",
                                   "LORD 2", "LOND", "BY",
                                   "LORD (dep)", "LOND (dep)",
                                   "Bonferroni")))

sim.FDR


################################################################################
# Plot of empirical FDR

LORD3.FDR = LORD.plus.FDR = LORD2.FDR = LOND.FDR = LORDdep.FDR = 
  LONDdep.FDR = bonf.FDR = rep(0, N)


for(j in 1:N){
  
  if(sum(R.LORD3[1:j] == 1) > 0) {
    LORD3.FDR[j] = sum(R.LORD3[which(pi.theta[1:j] == 0)] == 1)/
      sum(R.LORD3[1:j] == 1)
    
  } else { LORD3.FDR[j] = 0 }
  
  if(sum(R.LORD.plus[1:j] == 1) > 0) {
    LORD.plus.FDR[j] = sum(R.LORD.plus[which(pi.theta[1:j] == 0)] == 1)/
      sum(R.LORD.plus[1:j] == 1)
    
  } else { LORD.plus.FDR[j] = 0 }
  
  if(sum(R.LORD2[1:j] == 1) > 0) {
    LORD2.FDR[j] = sum(R.LORD2[which(pi.theta[1:j] == 0)] == 1)/
      sum(R.LORD2[1:j] == 1)
    
  } else { LORD2.FDR[j] = 0 }
  
  if(sum(R.LOND[1:j] == 1) > 0) {
    LOND.FDR[j] = sum(R.LOND[which(pi.theta[1:j] == 0)] == 1)/
      sum(R.LOND[1:j] == 1)
    
  } else { LOND.FDR[j] = 0 }
  
  if(sum(R.LORDdep[1:j] == 1) > 0) {
    LORDdep.FDR[j] = sum(R.LORDdep[which(pi.theta[1:j] == 0)] == 1)/
      sum(R.LORDdep[1:j] == 1)
    
  } else { LORDdep.FDR[j] = 0 }
  
  if(sum(R.LONDdep[1:j] == 1) > 0) {
    LONDdep.FDR[j] = sum(R.LONDdep[which(pi.theta[1:j] == 0)] == 1)/
      sum(R.LONDdep[1:j] == 1)
    
  } else { LONDdep.FDR[j] = 0 }
  
  if(sum(R.bonf[1:j] == 1) > 0) {
    bonf.FDR[j] = sum(R.bonf[which(pi.theta[1:j] == 0)] == 1)/
      sum(R.bonf[1:j] == 1)
    
  } else { bonf.FDR[j] = 0 }
}


png(filename="sim_FDR_plot.png", width = 800, height = 600)

plot(100*LORD3.FDR, pch = 'x', col = 'green4',
     xlab = 'Number of hypotheses tested', ylab = 'Empirical FDR (%)',
     cex.axis = 1.5, cex.lab = 1.5, xaxs = 'i', yaxs = 'i',
     xlim = c(-1, N), ylim = 100*c(0, 0.051), font.lab = 2)

points(100*LORD.plus.FDR, pch = 'x', col = 'blue3')

points(100*LORD2.FDR, pch = 'x', col = 'red3')

legend('topright', c('LORD 3', 'LORD++', 'LORD 2'), pch = rep('x', 3),
       col = c('green4', 'blue3', 'red3'), cex = 1.5)

dev.off()


################################################################################
################################# IMPC data ####################################
################################################################################

IMPC.data.full = read.csv("IMPC_ProcessedData_Continuous.csv")
IMPC.data.full = as_tibble(IMPC.data.full)

IMPC.data = select(IMPC.data.full, Latest.Mutant.Assay.Date,
                   Genotype.Contribution, Sd.Contribution)

IMPC.data = arrange(IMPC.data, Latest.Mutant.Assay.Date)

alpha = 0.05

################################################################################
### Genotype p-values

genotype.pval = pull(select(IMPC.data, Genotype.Contribution))

genotype.df = data.frame(id = 1:(length(genotype.pval)),
                         select(IMPC.data, date = Latest.Mutant.Assay.Date,
                                pval = Genotype.Contribution))

genotype.N = length(genotype.pval)

# Fixed
genotype.R.fixed = genotype.pval < 0.0001

# BH method
genotype.pval.BH = p.adjust(genotype.pval, method = "BH")
genotype.R.BH = genotype.pval.BH <= alpha

# BH (dep)
genotype.pval.BY = p.adjust(genotype.pval, method = "BY")
genotype.R.BY = genotype.pval.BY <= alpha

# LORD 3
genotype.LORD3 = LORD(genotype.df, random = FALSE)
genotype.R.LORD3 = genotype.LORD3$R

# LORD++
genotype.LORD.plus = LORD(genotype.df, version = '++', random = FALSE)
genotype.R.LORD.plus = genotype.LORD.plus$R

# LORD 2
genotype.LORD2 = LORD(genotype.df, version = 2, random = FALSE)
genotype.R.LORD2 = genotype.LORD2$R

# LOND
genotype.LOND = LOND(genotype.df, random = FALSE)
genotype.R.LOND = genotype.LOND$R

# LORD (dep)
genotype.LORDdep = LORDdep(genotype.df, random = FALSE)
genotype.R.LORDdep = genotype.LORDdep$R

# LOND (dep)
genotype.LONDdep = LOND(genotype.df, dep = TRUE, random = FALSE)
genotype.R.LONDdep = genotype.LONDdep$R

# Bonferroni
genotype.bonf = bonfInfinite(genotype.df, random = FALSE)
genotype.R.bonf = genotype.bonf$R

# Summary
genotype.results = matrix(c(sum(genotype.R.fixed), sum(genotype.R.BH),
                            sum(genotype.R.LORD3), sum(genotype.R.LORD.plus),
                            sum(genotype.R.LORD2), sum(genotype.R.LOND),
                            sum(genotype.R.BY), sum(genotype.R.LORDdep),
                            sum(genotype.R.LONDdep), sum(genotype.R.bonf),
                            genotype.N), nrow = 1,
                          dimnames = list(c(""),
                                          c("Fixed", "BH", "LORD 3", "LORD++",
                                            "LORD 2", "LOND", "BY",
                                            "LORD (dep)", "LOND (dep)",
                                            "Bonferroni", "N")))

genotype.results


# Comparison of _sets_ of discoveries
table(as.numeric(genotype.R.fixed), as.numeric(genotype.R.BH),
      dnn = c("Fixed","BH"))
table(as.numeric(genotype.R.fixed), genotype.R.LORD3,
      dnn = c("Fixed","LORD3"))
table(as.numeric(genotype.R.fixed), genotype.R.LORD.plus,
      dnn = c("Fixed","LORD++"))
table(as.numeric(genotype.R.fixed), genotype.R.LORD2,
      dnn = c("Fixed","LORD2"))
table(as.numeric(genotype.R.fixed), genotype.R.LOND,
      dnn = c("Fixed","LOND"))
table(as.numeric(genotype.R.fixed), genotype.R.LONDdep,
      dnn = c("Fixed","LOND (dep)"))
table(as.numeric(genotype.R.fixed), genotype.R.LORDdep,
      dnn = c("Fixed","LORD (dep)"))
table(as.numeric(genotype.R.fixed), genotype.R.bonf,
      dnn = c("Fixed","Bonferroni-like"))

table(as.numeric(genotype.R.BH), genotype.R.LORD3,
      dnn = c("BH","LORD3"))
table(as.numeric(genotype.R.BH), genotype.R.LORD.plus,
      dnn = c("BH","LORD++"))
table(as.numeric(genotype.R.BH), genotype.R.LORD2,
      dnn = c("BH","LORD2"))
table(as.numeric(genotype.R.BH), genotype.R.LOND,
      dnn = c("BH","LOND"))
table(as.numeric(genotype.R.BH), genotype.R.LONDdep,
      dnn = c("BH","LOND (dep)"))
table(as.numeric(genotype.R.BH), genotype.R.LORDdep,
      dnn = c("BH","LORD (dep)"))
table(as.numeric(genotype.R.BH), genotype.R.bonf,
      dnn = c("BH","Bonferroni-like"))

table(genotype.R.LORD3, genotype.R.LORD.plus, dnn = c("LORD3","LORD++"))
table(genotype.R.LORD3, genotype.R.LORD2, dnn = c("LORD3","LORD2"))
table(genotype.R.LORD3, genotype.R.LOND, dnn = c("LORD3","LOND"))
table(genotype.R.LORD3, genotype.R.LONDdep, dnn = c("LORD3","LOND (dep)"))
table(genotype.R.LORD3, genotype.R.LORDdep, dnn = c("LORD3","LORD (dep)"))
table(genotype.R.LORD3, genotype.R.bonf, dnn = c("LORD3","Bonferroni-like"))

table(genotype.R.LORD.plus, genotype.R.LORD2, dnn = c("LORD++","LORD2"))
table(genotype.R.LORD.plus, genotype.R.LOND, dnn = c("LORD++","LOND"))
table(genotype.R.LORD.plus, genotype.R.LONDdep,
      dnn = c("LORD++","LOND (dep)"))
table(genotype.R.LORD.plus, genotype.R.LORDdep,
      dnn = c("LORD++","LORD (dep)"))
table(genotype.R.LORD.plus, genotype.R.bonf,
      dnn = c("LORD++","Bonferroni-like"))

table(genotype.R.LOND, genotype.R.LONDdep, dnn = c("LOND","LOND (dep)"))
table(genotype.R.LOND, genotype.R.LORDdep, dnn = c("LOND","LORD (dep)"))
table(genotype.R.LOND, genotype.R.bonf, dnn = c("LOND","Bonferroni-like"))

table(genotype.R.LONDdep, genotype.R.LORDdep,
      dnn = c("LOND (dep)","LORD (dep)"))
table(genotype.R.LONDdep, genotype.R.bonf,
      dnn = c("LOND (dep)","Bonferroni-like"))


################################################################################
# Plots of adjusted test levels for the genotype effect data

png(filename="genotype_plot.png", width = 1200, height = 1200)

mfact = 2.5

x1.tick = seq(0, 150000, 50000)
x1.lab = c(TRUE, FALSE, TRUE, FALSE)

par(oma=c(3,3,3,3))  # all sides have 3 lines of space  
par(mar=c(5,4,4,2) + 0.1)   

par(mfrow=c(2,2))


# Plot 1
plot(log10(genotype.LORD3$alphai), pch = '.', col = 'green4',
     yaxt="n", xaxt = "n",
     xlab = " ", ylab = " ")

# y1.tick = seq(0, 0.03, by=0.01)
y1.tick = seq(-6, -1, by=1)

axis(1, at=x1.tick, labels=x1.lab, padj=0.5, cex.axis = 2.2)
axis(2, at=y1.tick, padj = 0.02, cex.axis=2.2)


mtext("a) LORD 3", side=3, line=2, cex=mfact, font = 2)  

# mtext("Order", side=1, line=1, adj = 0.22, padj = -0.5,
#       cex=mfact, outer=TRUE, font = 2)  

mtext(expression(paste(log[10]," adjusted test levels")),
      side=2, line=1, cex=mfact, outer=TRUE, padj = 0.3, adj = 0.85, font=2)  


# Plot 2
par(mar=c(5,4,4,0) + 0.1)   

plot(log10(genotype.LORD.plus$alphai), pch = '.', col = 'purple3',
     yaxt = "n", xaxt = "n", ylim = range(log10(genotype.LORD3$alphai)),
     ylab = " ", xlab = " ")

# y2.tick = seq(0, 0.006, by = 0.002)
axis(1, at=x1.tick, labels=x1.lab, padj=0.5, cex.axis = 2.2)
axis(2, at=y1.tick, padj = 0.02, cex.axis=2.2)

mtext("b) LORD++", side=3, line=2, cex=mfact, font = 2)  

# mtext("Order", side=1, line=1, adj = 0.8, padj = -0.5,
#       cex=mfact, outer=TRUE, font = 2)  


# Plot 3
par(mar=c(5,4,4,2) + 0.1)

plot(log10(genotype.LOND$alphai), pch = '+', col = 'blue3',
     yaxt="n", xaxt = "n", ylim = c(-8, -2.5),
     xlab = " ", ylab = " ")

points(log10(genotype.LONDdep$alphai), pch = 'x', col = 'red4')

y3.tick = seq(-8, -3, by=1)
axis(1, at=x1.tick, labels=x1.lab, padj=0.5, cex.axis = 2.2)
axis(2, at=y3.tick, padj = 0.02, cex.axis=2.2)

legend('topright', c('Independent', 'Dependent'), col = c('blue3', 'red4'),
       pch = c('+', 'x'), cex = mfact)


mtext("c) LOND", side=3, line=2, cex=mfact, font = 2)  

mtext("Index", side=1, line=1, adj = 0.22, padj = -0.5,
      cex=mfact, outer=TRUE, font = 2)  

mtext(expression(paste(log[10]," adjusted test levels")),
      side=2, line=1, cex=mfact, outer=TRUE, padj = 0.3, adj = 0.15, font=2)  


# Plot 4
par(mar=c(5,4,4,0) + 0.1)  
plot(log10(genotype.bonf$alphai), pch = 15, col = 'yellow4',
     yaxt = "n", xaxt = "n",
     ylab = " ", xlab = " ")

axis(1, at=x1.tick, labels=x1.lab, padj=0.5, cex.axis = 2.2)
axis(2, at=y3.tick, padj = 0.02, cex.axis=2.2)

mtext("d) Bonferroni-like", side=3, line=2, cex=mfact, font = 2)  

mtext("Index", side=1, line=1, adj = 0.8, padj = -0.5,
      cex=mfact, outer=TRUE, font = 2)  


dev.off()


################################################################################
### Sd p-values

sd.pval = pull(select(IMPC.data, Sd.Contribution))

sd.df = data.frame(id = 1:(length(genotype.pval)),
                   select(IMPC.data, date = Latest.Mutant.Assay.Date,
                          pval = Sd.Contribution))

sd.N = length(sd.pval)

# Fixed
sd.R.fixed = sd.pval < 0.0001

# BH method
sd.pval.BH = p.adjust(sd.pval, method = "BH")
sd.R.BH = sd.pval.BH <= alpha

# BH (dep)
sd.pval.BY = p.adjust(sd.pval, method = "BY")
sd.R.BY = sd.pval.BY <= alpha

# LORD 3
sd.LORD3 = LORD(sd.df, random = FALSE)
sd.R.LORD3 = sd.LORD3$R

# LORD++
sd.LORD.plus = LORD(sd.df, version = '++', random = FALSE)
sd.R.LORD.plus = sd.LORD.plus$R

# LORD 2
sd.LORD2 = LORD(sd.df, version = 2, random = FALSE)
sd.R.LORD2 = sd.LORD2$R

# LOND
sd.LOND = LOND(sd.df, random = FALSE)
sd.R.LOND = sd.LOND$R

# LORD (dep)
sd.LORDdep = LORDdep(sd.df, random = FALSE)
sd.R.LORDdep = sd.LORDdep$R

# LOND (dep)
sd.LONDdep = LOND(sd.df, dep = TRUE, random = FALSE)
sd.R.LONDdep = sd.LONDdep$R

# Bonferroni
sd.bonf = bonfInfinite(sd.df, random = FALSE)
sd.R.bonf = sd.bonf$R


# Summary
sd.results = matrix(c(sum(sd.R.fixed), sum(sd.R.BH),
                      sum(sd.R.LORD3), sum(sd.R.LORD.plus),
                      sum(sd.R.LORD2), sum(sd.R.LOND),
                      sum(sd.R.BY), sum(sd.R.LORDdep), sum(sd.R.LONDdep),
                      sum(sd.R.bonf), sd.N), nrow = 1,
                    dimnames = list(c(""),
                                    c("Fixed", "BH", "LORD 3", "LORD++",
                                      "LORD 2", "LOND", "BY",
                                      "LORDdep", "LONDdep",
                                      "Bonferroni", "N")))

sd.results


# Comparisons of _sets_ of discoveries
table(as.numeric(sd.R.fixed), as.numeric(sd.R.BH), dnn = c("Fixed","BH"))
table(as.numeric(sd.R.fixed), sd.R.LORD3, dnn = c("Fixed","LORD3"))
table(as.numeric(sd.R.fixed), sd.R.LORD.plus, dnn = c("Fixed","LORD++"))
table(as.numeric(sd.R.fixed), sd.R.LORD2, dnn = c("Fixed","LORD2"))
table(as.numeric(sd.R.fixed), sd.R.LOND, dnn = c("Fixed","LOND"))
table(as.numeric(sd.R.fixed), sd.R.LONDdep, dnn = c("Fixed","LOND (dep)"))
table(as.numeric(sd.R.fixed), sd.R.LORDdep, dnn = c("Fixed","LORD (dep)"))
table(as.numeric(sd.R.fixed), sd.R.bonf, dnn = c("Fixed","Bonferroni-like"))

table(as.numeric(sd.R.BH), sd.R.LORD3, dnn = c("BH","LORD3"))
table(as.numeric(sd.R.BH), sd.R.LORD.plus, dnn = c("BH","LORD++"))
table(as.numeric(sd.R.BH), sd.R.LORD2, dnn = c("BH","LORD2"))
table(as.numeric(sd.R.BH), sd.R.LOND, dnn = c("BH","LOND"))
table(as.numeric(sd.R.BH), sd.R.LONDdep, dnn = c("BH","LOND (dep)"))
table(as.numeric(sd.R.BH), sd.R.LORDdep, dnn = c("BH","LORD (dep)"))
table(as.numeric(sd.R.BH), sd.R.bonf, dnn = c("BH","Bonferroni-like"))

table(sd.R.LORD3, sd.R.LORD.plus, dnn = c("LORD3","LORD++"))
table(sd.R.LORD3, sd.R.LORD2, dnn = c("LORD3","LORD2"))
table(sd.R.LORD3, sd.R.LOND, dnn = c("LORD3","LOND"))
table(sd.R.LORD3, sd.R.LONDdep, dnn = c("LORD3","LOND (dep)"))
table(sd.R.LORD3, sd.R.LORDdep, dnn = c("LORD3","LORD (dep)"))
table(sd.R.LORD3, sd.R.bonf, dnn = c("LORD3","Bonferroni-like"))

table(sd.R.LORD.plus, sd.R.LORD2, dnn = c("LORD++","LORD2"))
table(sd.R.LORD.plus, sd.R.LOND, dnn = c("LORD++","LOND"))
table(sd.R.LORD.plus, sd.R.LONDdep, dnn = c("LORD++","LOND (dep)"))
table(sd.R.LORD.plus, sd.R.LORDdep, dnn = c("LORD++","LORD (dep)"))
table(sd.R.LORD.plus, sd.R.bonf, dnn = c("LORD++","Bonferroni-like"))

table(sd.R.LOND, sd.R.LONDdep, dnn = c("LOND","LOND (dep)"))
table(sd.R.LOND, sd.R.LORDdep, dnn = c("LOND","LORD (dep)"))
table(sd.R.LOND, sd.R.bonf, dnn = c("LOND","Bonferroni-like"))

table(sd.R.LONDdep, sd.R.LORDdep, dnn = c("LOND (dep)","LORD (dep)"))
table(sd.R.LONDdep, sd.R.bonf, dnn = c("LOND (dep)","Bonferroni-like"))



################################################################################
################################ Yeast data ####################################
################################################################################

yeast.data = read.csv("Yeast_AssembledData3Columns.csv")

names(yeast.data)[names(yeast.data)=="strain.compound.id"] = "id"
names(yeast.data)[names(yeast.data)=="pvalue"] = "pval"

yeast.N = length(yeast.data$id)


# Fixed
yeast.R.fixed = yeast.data$pval < pnorm(-4)

# BH method
yeast.pval.BH = p.adjust(yeast.data$pval, method = "BH")
yeast.R.BH = yeast.pval.BH <= alpha

# BH (dep)
yeast.pval.BY = p.adjust(yeast.data$pval, method = "BY")
yeast.R.BY = yeast.pval.BY <= alpha

# LORD 3
yeast.LORD3 = LORD(yeast.data, random = FALSE)
yeast.R.LORD3 = yeast.LORD3$R

# LORD++
yeast.LORD.plus = LORD(yeast.data, version = '++', random = FALSE)
yeast.R.LORD.plus = yeast.LORD.plus$R

# LORD 2
yeast.LORD2 = LORD(yeast.data, version = 2, random = FALSE)
yeast.R.LORD2 = yeast.LORD2$R

# LOND
yeast.LOND = LOND(yeast.data, random = FALSE)
yeast.R.LOND = yeast.LOND$R

# LORD (dep)
yeast.LORDdep = LORDdep(yeast.data, random = FALSE)
yeast.R.LORDdep = yeast.LORDdep$R

# LOND (dep)
yeast.LONDdep = LOND(yeast.data, dep = TRUE, random = FALSE)
yeast.R.LONDdep = yeast.LONDdep$R

# Bonferroni
yeast.bonf = bonfInfinite(yeast.data, random = FALSE)
yeast.R.bonf = yeast.bonf$R


# Summary
yeast.results = matrix(c(sum(yeast.R.fixed), sum(yeast.R.BH),
                         sum(yeast.R.LORD3), sum(yeast.R.LORD.plus),
                         sum(yeast.R.LORD2), sum(yeast.R.LOND),
                         sum(yeast.R.BY), sum(yeast.R.LORDdep),
                         sum(yeast.R.LONDdep), sum(yeast.R.bonf), yeast.N),
                       nrow = 1,
                       dimnames = list(c(""),
                                       c("Fixed", "BH", "LORD 3", "LORD++",
                                         "LORD 2", "LOND", "BY",
                                         "LORDdep", "LONDdep",
                                         "Bonferroni", "N")))

yeast.results


# Comparisons of _sets_ of discoveries
table(as.numeric(yeast.R.fixed), as.numeric(yeast.R.BH),
      dnn = c("Fixed","BH"))
table(as.numeric(yeast.R.fixed), yeast.R.LORD3,
      dnn = c("Fixed","LORD3"))
table(as.numeric(yeast.R.fixed), yeast.R.LORD.plus,
      dnn = c("Fixed","LORD++"))
table(as.numeric(yeast.R.fixed), yeast.R.LORD2,
      dnn = c("Fixed","LORD2"))
table(as.numeric(yeast.R.fixed), yeast.R.LOND,
      dnn = c("Fixed","LOND"))
table(as.numeric(yeast.R.fixed), yeast.R.LONDdep,
      dnn = c("Fixed","LOND (dep)"))
table(as.numeric(yeast.R.fixed), yeast.R.LORDdep,
      dnn = c("Fixed","LORD (dep)"))
table(as.numeric(yeast.R.fixed), yeast.R.bonf,
      dnn = c("Fixed","Bonferroni-like"))

table(as.numeric(yeast.R.BH), yeast.R.LORD3, dnn = c("BH","LORD3"))
table(as.numeric(yeast.R.BH), yeast.R.LORD.plus, dnn = c("BH","LORD++"))
table(as.numeric(yeast.R.BH), yeast.R.LORD2, dnn = c("BH","LORD2"))
table(as.numeric(yeast.R.BH), yeast.R.LOND, dnn = c("BH","LOND"))
table(as.numeric(yeast.R.BH), yeast.R.LONDdep, dnn = c("BH","LOND (dep)"))
table(as.numeric(yeast.R.BH), yeast.R.LORDdep, dnn = c("BH","LORD (dep)"))
table(as.numeric(yeast.R.BH), yeast.R.bonf, dnn = c("BH","Bonferroni-like"))

table(yeast.R.LORD3, yeast.R.LORD.plus, dnn = c("LORD3","LORD++"))
table(yeast.R.LORD3, yeast.R.LORD2, dnn = c("LORD3","LORD2"))
table(yeast.R.LORD3, yeast.R.LOND, dnn = c("LORD3","LOND"))
table(yeast.R.LORD3, yeast.R.LONDdep, dnn = c("LORD3","LOND (dep)"))
table(yeast.R.LORD3, yeast.R.LORDdep, dnn = c("LORD3","LORD (dep)"))
table(yeast.R.LORD3, yeast.R.bonf, dnn = c("LORD3","Bonferroni-like"))

table(yeast.R.LORD.plus, yeast.R.LORD2, dnn = c("LORD++","LORD2"))
table(yeast.R.LORD.plus, yeast.R.LOND, dnn = c("LORD++","LOND"))
table(yeast.R.LORD.plus, yeast.R.LONDdep, dnn = c("LORD++","LOND (dep)"))
table(yeast.R.LORD.plus, yeast.R.LORDdep, dnn = c("LORD++","LORD (dep)"))
table(yeast.R.LORD.plus, yeast.R.bonf, dnn = c("LORD++","Bonferroni-like"))

table(yeast.R.LOND, yeast.R.LONDdep, dnn = c("LOND","LOND (dep)"))
table(yeast.R.LOND, yeast.R.LORDdep, dnn = c("LOND","LORD (dep)"))
table(yeast.R.LOND, yeast.R.bonf, dnn = c("LOND","Bonferroni-like"))

table(yeast.R.LONDdep, yeast.R.LORDdep, dnn = c("LOND (dep)","LORD (dep)"))
table(yeast.R.LONDdep, yeast.R.bonf, dnn = c("LOND (dep)","Bonferroni-like"))
