/*
 * biojs-io-biom
 * https://github.com/molbiodiv/biojs-io-biom
 *
 * Copyright (c) 2015 Markus J. Ankenbrand
 * Licensed under the MIT license.
 */


// browserify build config
var buildDir = "build";
var outputFile = "biom";

// packages
var gulp   = require('gulp');

// es6 compilation
var babel = require('gulp-babel');
require('babel-core/register');

// browser builds
var browserify = require('browserify');
var watchify = require('watchify')
var uglify = require('gulp-uglify');


// testing
var mocha = require('gulp-mocha');
// coverage
var istanbul = require('gulp-istanbul');
var isparta = require('isparta');
var runSequence = require('run-sequence');
var coveralls = require('gulp-coveralls');

// code style 
var jshint = require('gulp-jshint'); 

// gulp helper
var source = require('vinyl-source-stream'); // converts node streams into vinyl streams
var gzip = require('gulp-gzip');
var rename = require('gulp-rename');
var chmod = require('gulp-chmod');
var streamify = require('gulp-streamify'); // converts streams into buffers (legacy support for old plugins)

// path tools
var fs = require('fs');
var path = require('path');
var join = path.join;
var mkdirp = require('mkdirp');
var del = require('del');

// auto config
var outputFileMin = join(buildDir,outputFile + ".min.js");
var packageConfig = require('./package.json');

// a failing test breaks the whole build chain
gulp.task('build', ['babel', 'build-browser', 'build-browser-gzip']);
gulp.task('default', ['lint','test',  'build']);



gulp.task('lint', function() {
  return gulp.src('./src/*.js')
    .pipe(jshint())
    .pipe(jshint.reporter('default'));
});




gulp.task('coverage:instrument', function () {
  return gulp.src(['src/**/*.js'])
    // Covering files
    .pipe(istanbul({instrumenter: isparta.Instrumenter}))
    // Force `require` to return covered files
    .pipe(istanbul.hookRequire());
});

gulp.task('test-unit', function () {

    return gulp.src('./test/**/*.js', {read: false})

        .pipe(mocha({reporter: 'spec',
                    useColors: true}))
});

gulp.task('coverage:report', function(done) {
  return gulp.src(['src/**/*.js'], {read: false})
      .pipe(istanbul.writeReports({
        // Istanbul configuration (see https://github.com/SBoudrias/gulp-istanbul#istanbulwritereportsopt)
        // ...
      }));
});

gulp.task('test', function(done) {
  runSequence('coverage:instrument', 'test-unit', 'coverage:report', 'coveralls', done);
});

gulp.task('coveralls', function () {
  if (!process.env.CI) {
    return;
  }

  return gulp.src('./coverage/lcov.info')
      .pipe(coveralls());
});

gulp.task('test-watch', function() {
   gulp.watch(['./src/**/*.js','./lib/**/*.js', './test/**/*.js'], ['test']);
});





// will remove everything in build
gulp.task('clean', function(cb) {
  del([buildDir], cb);
});

// just makes sure that the build dir exists
gulp.task('init', ['clean'], function() {
  mkdirp(buildDir, function (err) {
    if (err) console.error(err)
  });
});

// browserify debug
gulp.task('build-browser',['init'], function() {
  var b = browserify({debug: true,hasExports: true});
  exposeBundles(b);
  return b.bundle()
    .pipe(source(outputFile + ".js"))
    .pipe(chmod(644))
    .pipe(gulp.dest(buildDir));
});

// browserify min
gulp.task('build-browser-min',['init'], function() {
  var b = browserify({hasExports: true});
  exposeBundles(b);
  return b.bundle()
    .pipe(source(outputFile + ".min.js"))
    .pipe(chmod(644))
    .pipe(streamify(uglify()))
    .pipe(gulp.dest(buildDir));
});
 
gulp.task('build-browser-gzip', ['build-browser-min'], function() {
  return gulp.src(outputFileMin)
    .pipe(gzip({append: false, gzipOptions: { level: 9 }}))
    .pipe(rename(outputFile + ".min.gz.js"))
    .pipe(gulp.dest(buildDir));
});

// exposes the main package
// + checks the config whether it should expose other packages
function exposeBundles(b){
  b.add("./" + packageConfig.main, {expose: packageConfig.name });
  if(packageConfig.sniper !== undefined && packageConfig.sniper.exposed !== undefined){
    for(var i=0; i<packageConfig.sniper.exposed.length; i++){
      b.require(packageConfig.sniper.exposed[i]);
    }
  }
}

// watch task for browserify 
// watchify has an internal cache -> subsequent builds are faster
gulp.task('watch', function() {
  var util = require('gulp-util')

  var b = browserify({debug: true,hasExports: true, cache: {}, packageCache: {} });
  b.add("./" + packageConfig.main, {expose: packageConfig.name});
  // expose other bundles
  exposeBundles(b);

  function rebundle(ids){
    b.bundle()
    .on("error", function(error) {
      util.log(util.colors.red("Error: "), error);
     })
    .pipe(source(outputFile + ".js"))
    .pipe(chmod(644))
    .pipe(gulp.dest(buildDir));
  }

  var watcher = watchify(b);
  watcher.on("update", rebundle)
   .on("log", function(message) {
      util.log("Refreshed:", message);
  });
  return rebundle();
});

gulp.task('babel', function () {
  return gulp.src("src/biojs-io-biom.js")
    .pipe(babel())
    .pipe(gulp.dest("lib"));
});
