# Code for the baseline ASF case study model, as contained in Lange et al. 2014
# See "code to generate figures.R" for details of how this code was used to generate results for the Final Report
# This code is a minor modification of the code provided by Hans-Hermann Thulke
# The code was modified so as to allow for the computation of omnibus space-time interaction tests within the Lange et al. 2014 framework (as explained at the end of Section 6.5.3 in the Final Report)
# Stephen Catterall 16/6/16

endemicity.test <- function(table, x=1, y=2, t=3, num.permutations = 100, dist=1:10, time=1:10, format="ll") {
  orig <- data.frame( x=table[,x], y=table[,y], t=table[,t] )
  rows <- nrow(orig)
  perm <- list()
  for(i in 1:num.permutations) {
    p <- orig[,1:2]
    p$t <- sample(orig$t,rows)
    perm[[i]] <- p
  }
  
  distFun <- dist.sq.arr
  if(format == "ll") {
    distFun <- dist.sq.ll.arr
  }
  dist.matrix <- array(0,dim=c(rows,rows))
  for( i in 1:rows ) {
    dist.matrix[i,] <- distFun(orig, orig[i,])
  }		
  orig.arr <- array(0,dim=c(length(dist),length(time)))
  perm.arr <- array(0,dim=c(length(dist),length(time),num.permutations))
  
  p01.arr <- array(0,dim=c(length(dist),length(time)))
  p05.arr <- array(0,dim=c(length(dist),length(time)))
  p95.arr <- array(0,dim=c(length(dist),length(time)))
  p99.arr <- array(0,dim=c(length(dist),length(time)))
  mean.arr <- array(0,dim=c(length(dist),length(time)))
  min.arr <- array(0,dim=c(length(dist),length(time)))
  max.arr <- array(0,dim=c(length(dist),length(time)))
  signif.arr <- array(0,dim=c(length(dist),length(time)))
  rank.arr <- array(0,dim=c(length(dist),length(time)))
  perm.sums <- rep(0,num.permutations) # new!!! summation for omnibus test
  
  
  for(i in 1:length(dist)) {
    for(j in 1:length(time)) {
      print(paste("d:", dist[i], "t:", time[j] ))
      orig.arr[i,j] <- count.endemic.once(orig, dist[i], time[j], dist.matrix)
      values <- rep(0,num.permutations)
      for( k in 1:num.permutations ) {
        values[k] <-  count.endemic.once(perm[[k]], dist[i], time[j], dist.matrix)
        perm.sums[k]=perm.sums[k]+values[k]; 
      }
      values.sorted <- sort(values)
      q <- quantile(values, probs=c(0.01, 0.05, 0.95, 0.99), type=1)
      p01.arr[i,j] <- q[[1]]
      p05.arr[i,j] <- q[[2]]
      p95.arr[i,j] <- q[[3]]
      p99.arr[i,j] <- q[[4]]
      mean.arr[i,j] <- mean(values)
      min.arr[i,j] <- min(values)
      max.arr[i,j] <- max(values)
      perm.arr[i,j,] <- values
      if( orig.arr[i,j] < p01.arr[i,j] ) {
        signif.arr[i,j] <- 2
      } else if( orig.arr[i,j] < p05.arr[i,j] ) {
        signif.arr[i,j] <- 1
      }
      rank.arr[i,j] <- 100 * length(which(values.sorted <= orig.arr[i,j])) / num.permutations
    }
  }
  
  return(list(
    data=orig,
    orig=orig.arr,
    psum=perm.sums,
    #perm=perm.arr,
    dist.matrix=sqrt(dist.matrix),
    dist=dist,
    time=time,
    mean=mean.arr,
    min=min.arr,
    max=max.arr,
    p01=p01.arr,
    p05=p05.arr,
    p95=p95.arr,
    p99=p99.arr,
    significance=signif.arr,
    quantile = rank.arr,
    format=format))
}

count.endemic.once <- function( table, dist, time, dist.matrix ) {
  cnt <- 0
  distsq <- dist^2 
  x <- table[,1]
  y <- table[,2]
  t <- table[,3]
  for(j in 1:nrow(table)) {
    is.end <- (t[j] - t >= time) & (dist.matrix[j,] <= distsq)
    other <- NULL
    if( j>1 ) { 
      other <- 1:(j-1)
    }
    if( j<length(is.end) ) {
      other <- c(other, (j+1):length(is.end))
    }
    if(any(is.end[ other ])) {
      cnt <- cnt + 1
    }
  }
  return( cnt / nrow(table) )
}

dist.sq <- function( p1, p2 ) {
  return( (p1[1] - p2[1])^2 + (p1[2] - p2[2])^2 )
}
dist.sq.arr <- function( p1arr, p2 ) {
  return( (p1arr[,1] - p2[1,1])^2 + (p1arr[,2] - p2[1,2])^2 )
}
dist.sq.ll.arr <- function( p1arr, p2 ) {
  R <- 6371
  dLat <- deg.to.rad(p1arr[,1] - p2[1,1])
  dLon <- deg.to.rad(p1arr[,2] - p2[1,2])
  lat1 <- deg.to.rad(p1arr[,1])
  lat2 <- deg.to.rad(p2[1,1])
  
  a <- sin(dLat/2) * sin(dLat/2) +  sin(dLon/2) * sin(dLon/2) * cos(lat1) * cos(lat2)
  c <- 2 * atan2(sqrt(a), sqrt(1-a))
  d <- R * c
  return( d^2 )
}
deg.to.rad <- function(v) {
  return( 2 * pi * v / 360 )
}

