import numpy as np

def simulate_xrd(a, c, name):
    # Cu K-alpha wavelength
    wavelength = 1.5406 
    
    # Miller Indices (h, k, l) for the most prominent Hexagonal peaks
    peaks = [
        (1, 0, 0), (0, 0, 2), (1, 0, 1), 
        (1, 0, 2), (1, 1, 0), (1, 0, 3)
    ]
    
    print(f"\n--- XRD Simulation: {name} ---")
    print(f"{'Plane (hkl)':<12} | {'d-spacing (Å)':<15} | {'2-Theta (deg)':<15}")
    print("-" * 45)
    
    for (h, k, l) in peaks:
        # Calculate d-spacing for Hexagonal Crystal System
        # 1/d^2 = 4/3 * (h^2 + hk + k^2)/a^2 + l^2/c^2
        inv_d_sq = (4/3) * (h**2 + h*k + k**2) / (a**2) + (l**2) / (c**2)
        d = np.sqrt(1 / inv_d_sq)
        
        # Calculate 2-Theta using Bragg's Law
        # theta = arcsin(lambda / 2d)
        if (wavelength / (2 * d)) <= 1:
            theta = np.arcsin(wavelength / (2 * d))
            two_theta = np.degrees(theta) * 2
            print(f"({h} {k} {l}){' ':<9} | {d:<15.4f} | {two_theta:<15.2f}")

# Data from your previous 'geometries.py' run
simulate_xrd(4.561, 7.448, "Nd-Killer (6th Node)")
simulate_xrd(4.801, 7.841, "Dy-Killer (7th Node)")

# (base) brendanlynch@Brendans-Laptop rareEarth % python xrd.py

# --- XRD Simulation: Nd-Killer (6th Node) ---
# Plane (hkl)  | d-spacing (Å)   | 2-Theta (deg)  
# ---------------------------------------------
# (1 0 0)          | 3.9499          | 22.49          
# (0 0 2)          | 3.7240          | 23.88          
# (1 0 1)          | 3.4896          | 25.51          
# (1 0 2)          | 2.7096          | 33.03          
# (1 1 0)          | 2.2805          | 39.48          
# (1 0 3)          | 2.1020          | 43.00          

# --- XRD Simulation: Dy-Killer (7th Node) ---
# Plane (hkl)  | d-spacing (Å)   | 2-Theta (deg)  
# ---------------------------------------------
# (1 0 0)          | 4.1578          | 21.35          
# (0 0 2)          | 3.9205          | 22.66          
# (1 0 1)          | 3.6733          | 24.21          
# (1 0 2)          | 2.8524          | 31.33          
# (1 1 0)          | 2.4005          | 37.43          
# (1 0 3)          | 2.2128          | 40.74          
# (base) brendanlynch@Brendans-Laptop rareEarth % 