import numpy as np
from pyscf import gto, dft, grad
from ase import Atoms
from ase.md.velocitydistribution import MaxwellBoltzmannDistribution
from ase.md.verlet import VelocityVerlet
from ase.units import fs
from ase.calculators.calculator import Calculator, all_changes

class PySCF_Direct(Calculator):
    implemented_properties = ['energy', 'forces']
    def __init__(self, **kwargs):
        Calculator.__init__(self, **kwargs)
        self.dm = None 

    def calculate(self, atoms=None, properties=['energy'], system_changes=all_changes):
        Calculator.calculate(self, atoms, properties, system_changes)
        pos = self.atoms.get_positions()
        mol_list = [[self.atoms.symbols[i], pos[i]] for i in range(len(self.atoms))]
        
        # Using the standard 6-31g* basis set as in your successful CLi9 run
        mol = gto.M(atom=mol_list, basis='6-31g*', charge=0, spin=1, verbose=0)
        mf = dft.UKS(mol).set(xc='b3lyp')
        
        # Applying the "Lynch-Shift" used in CLi9 to prevent SCF divergence
        mf.level_shift = 0.3  
        mf.conv_tol = 1e-5    
        mf.max_cycle = 200    
        
        if self.dm is None:
            e_tot = mf.kernel()
        else:
            e_tot = mf.kernel(dm0=self.dm)
        
        self.dm = mf.make_rdm1() 
        
        if not mf.converged:
            mf = mf.newton()
            e_tot = mf.kernel(dm0=self.dm)
            
        g = mf.Gradients().kernel()
        self.results['energy'] = e_tot * 27.2114 
        self.results['forces'] = -g * 27.2114 / 0.529177

# INITIALIZE ArLi5 ATOMS (Using the ACI-Corrected Nodal Displacement)
initial_coords = [
    ('Ar', ( 0.000000,  0.000000,  0.000000)),
    ('Li', ( 1.581200,  0.003119,  0.000000)),
    ('Li', (-1.581200, -0.003119,  0.000000)),
    ('Li', ( 0.003119,  1.581200,  0.000000)),
    ('Li', (-0.003119, -1.581200,  0.000000)),
    ('Li', ( 0.000000,  0.000000,  1.722457))
]

symbols = [c[0] for c in initial_coords]
positions = [c[1] for c in initial_coords]
atoms = Atoms(symbols=symbols, positions=positions)

atoms.calc = PySCF_Direct()
print("Initializing Maxwell-Boltzmann Distribution at 300K...")
MaxwellBoltzmannDistribution(atoms, temperature_K=300)

# Running 200 steps (0.5 fs each) as per the CLi9 success protocol
dyn = VelocityVerlet(atoms, timestep=0.5 * fs, trajectory='ArLi5_thermal_lock.traj')

def print_status():
    print(f"Step: {dyn.get_number_of_steps()} | Energy: {atoms.get_potential_energy():.4f} eV")

dyn.attach(print_status, interval=1)

print("\n--- STARTING ArLi5 THERMAL STABILITY RUN ---")
dyn.run(200)

print("\nSimulation complete. Open 'ArLi5_thermal_lock.traj' to confirm 'Ar' remains caged.")

# (base) brendanlynch@Brendans-Laptop ArLi5 % python thermalStability.py
# /Users/brendanlynch/miniconda3/lib/python3.12/site-packages/pyscf/dft/libxc.py:771: UserWarning: Since PySCF-2.3, B3LYP (and B3P86) are changed to the VWN-RPA variant, corresponding to the original definition by Stephens et al. (issue 1480) and the same as the B3LYP functional in Gaussian. To restore the VWN5 definition, you can put the setting "B3LYP_WITH_VWN5 = True" in pyscf_conf.py
#   warnings.warn('Since PySCF-2.3, B3LYP (and B3P86) are changed to the VWN-RPA variant, '
# Initializing Maxwell-Boltzmann Distribution at 300K...

# --- STARTING ArLi5 THERMAL STABILITY RUN ---
# Step: 0 | Energy: -15357.1559 eV
# Step: 1 | Energy: -15357.2901 eV
# Step: 2 | Energy: -15357.6439 eV
# Step: 3 | Energy: -15358.1972 eV
# Step: 4 | Energy: -15358.9224 eV
# Step: 5 | Energy: -15359.7875 eV
# Step: 6 | Energy: -15360.7326 eV
# Step: 7 | Energy: -15362.2884 eV
# Step: 8 | Energy: -15363.2679 eV
# Step: 9 | Energy: -15364.2372 eV
# Step: 10 | Energy: -15365.1756 eV
# Step: 11 | Energy: -15366.0692 eV
# Step: 12 | Energy: -15366.9087 eV
# Step: 13 | Energy: -15367.6896 eV
# Step: 14 | Energy: -15368.4098 eV
# Step: 15 | Energy: -15369.0704 eV
# Step: 16 | Energy: -15369.6728 eV
# Step: 17 | Energy: -15370.2214 eV
# Step: 18 | Energy: -15370.7205 eV
# Step: 19 | Energy: -15371.1882 eV
# Step: 20 | Energy: -15371.5913 eV
# Step: 21 | Energy: -15371.9461 eV
# Step: 22 | Energy: -15372.2603 eV
# Step: 23 | Energy: -15372.5364 eV
# Step: 24 | Energy: -15372.7768 eV
# Step: 25 | Energy: -15372.9852 eV
# Step: 26 | Energy: -15373.1633 eV
# Step: 27 | Energy: -15373.3147 eV
# Step: 28 | Energy: -15373.4413 eV
# Step: 29 | Energy: -15373.5462 eV
# Step: 30 | Energy: -15373.6314 eV
# Step: 31 | Energy: -15373.6992 eV
# Step: 32 | Energy: -15373.7517 eV
# Step: 33 | Energy: -15373.7907 eV
# Step: 34 | Energy: -15373.8178 eV
# Step: 35 | Energy: -15373.8346 eV
# Step: 36 | Energy: -15373.8425 eV
# Step: 37 | Energy: -15373.8430 eV
# Step: 38 | Energy: -15373.8372 eV
# Step: 39 | Energy: -15373.8269 eV
# Step: 40 | Energy: -15373.8152 eV
# Step: 41 | Energy: -15373.8020 eV
# Step: 42 | Energy: -15373.7913 eV
# Step: 43 | Energy: -15373.7888 eV
# Step: 44 | Energy: -15373.8359 eV
# Step: 45 | Energy: -15373.8294 eV
# Step: 46 | Energy: -15373.8228 eV
# Step: 47 | Energy: -15373.8158 eV
# Step: 48 | Energy: -15373.8088 eV
# Step: 49 | Energy: -15373.8011 eV
# Step: 50 | Energy: -15373.7933 eV
# Step: 51 | Energy: -15373.7855 eV
# Step: 52 | Energy: -15373.7775 eV
# Step: 53 | Energy: -15373.7695 eV
# Step: 54 | Energy: -15373.7615 eV
# Step: 55 | Energy: -15373.7536 eV
# Step: 56 | Energy: -15373.7458 eV
# Step: 57 | Energy: -15373.7380 eV
# Step: 58 | Energy: -15373.7305 eV
# Step: 59 | Energy: -15373.7231 eV
# Step: 60 | Energy: -15373.7160 eV
# Step: 61 | Energy: -15373.7090 eV
# Step: 62 | Energy: -15373.7023 eV
# Step: 63 | Energy: -15373.6958 eV
# Step: 64 | Energy: -15373.6895 eV
# Step: 65 | Energy: -15373.6835 eV
# Step: 66 | Energy: -15373.6777 eV
# Step: 67 | Energy: -15373.6722 eV
# Step: 68 | Energy: -15373.6669 eV
# Step: 69 | Energy: -15373.6618 eV
# Step: 70 | Energy: -15373.6570 eV
# Step: 71 | Energy: -15373.6525 eV
# Step: 72 | Energy: -15373.6481 eV
# Step: 73 | Energy: -15373.6440 eV
# Step: 74 | Energy: -15373.6400 eV
# Step: 75 | Energy: -15373.6363 eV
# Step: 76 | Energy: -15373.6327 eV
# Step: 77 | Energy: -15373.6294 eV
# Step: 78 | Energy: -15373.6262 eV
# Step: 79 | Energy: -15373.6231 eV
# Step: 80 | Energy: -15373.6202 eV
# Step: 81 | Energy: -15373.6175 eV
# Step: 82 | Energy: -15373.6149 eV
# Step: 83 | Energy: -15373.6124 eV
# Step: 84 | Energy: -15373.6101 eV
# Step: 85 | Energy: -15373.6079 eV
# Step: 86 | Energy: -15373.6057 eV
# Step: 87 | Energy: -15373.6037 eV
# Step: 88 | Energy: -15373.6018 eV
# Step: 89 | Energy: -15373.5999 eV
# Step: 90 | Energy: -15373.5982 eV
# Step: 91 | Energy: -15373.5965 eV
# Step: 92 | Energy: -15373.5949 eV
# Step: 93 | Energy: -15373.5934 eV
# Step: 94 | Energy: -15373.5919 eV
# Step: 95 | Energy: -15373.5905 eV
# Step: 96 | Energy: -15373.5892 eV
# Step: 97 | Energy: -15373.5879 eV
# Step: 98 | Energy: -15373.5866 eV
# Step: 99 | Energy: -15373.5855 eV
# Step: 100 | Energy: -15373.5843 eV
# Step: 101 | Energy: -15373.5833 eV
# Step: 102 | Energy: -15373.5822 eV
# Step: 103 | Energy: -15373.5813 eV
# Step: 104 | Energy: -15373.5803 eV
# Step: 105 | Energy: -15373.5794 eV
# Step: 106 | Energy: -15373.5786 eV
# Step: 107 | Energy: -15373.5778 eV
# Step: 108 | Energy: -15373.5770 eV
# Step: 109 | Energy: -15373.5763 eV
# Step: 110 | Energy: -15373.5756 eV
# Step: 111 | Energy: -15373.5749 eV
# Step: 112 | Energy: -15373.5743 eV
# Step: 113 | Energy: -15373.5737 eV
# Step: 114 | Energy: -15373.5731 eV
# Step: 115 | Energy: -15373.5726 eV
# Step: 116 | Energy: -15373.5721 eV
# Step: 117 | Energy: -15373.5716 eV
# Step: 118 | Energy: -15373.5711 eV
# Step: 119 | Energy: -15373.5707 eV
# Step: 120 | Energy: -15373.5703 eV
# Step: 121 | Energy: -15373.5699 eV
# Step: 122 | Energy: -15373.5696 eV
# Step: 123 | Energy: -15373.5693 eV
# Step: 124 | Energy: -15373.5690 eV
# Step: 125 | Energy: -15373.5687 eV
# Step: 126 | Energy: -15373.5684 eV
# Step: 127 | Energy: -15373.5682 eV
# Step: 128 | Energy: -15373.5680 eV
# Step: 129 | Energy: -15373.5678 eV
# Step: 130 | Energy: -15373.5676 eV
# Step: 131 | Energy: -15373.5674 eV
# Step: 132 | Energy: -15373.5672 eV
# Step: 133 | Energy: -15373.5670 eV
# Step: 134 | Energy: -15373.5669 eV
# Step: 135 | Energy: -15373.5668 eV
# Step: 136 | Energy: -15373.5667 eV
# Step: 137 | Energy: -15373.5665 eV
# Step: 138 | Energy: -15373.5665 eV
# Step: 139 | Energy: -15373.5664 eV
# Step: 140 | Energy: -15373.5663 eV
# Step: 141 | Energy: -15373.5662 eV
# Step: 142 | Energy: -15373.5662 eV
# Step: 143 | Energy: -15373.5661 eV
# Step: 144 | Energy: -15373.5661 eV
# Step: 145 | Energy: -15373.5660 eV
# Step: 146 | Energy: -15373.5660 eV
# Step: 147 | Energy: -15373.5659 eV
# Step: 148 | Energy: -15373.5659 eV
# Step: 149 | Energy: -15373.5659 eV
# Step: 150 | Energy: -15373.5658 eV
# Step: 151 | Energy: -15373.5658 eV
# Step: 152 | Energy: -15373.5658 eV
# Step: 153 | Energy: -15373.5658 eV
# Step: 154 | Energy: -15373.5657 eV
# Step: 155 | Energy: -15373.5657 eV
# Step: 156 | Energy: -15373.5657 eV
# Step: 157 | Energy: -15373.5657 eV
# Step: 158 | Energy: -15373.5657 eV
# Step: 159 | Energy: -15373.5657 eV
# Step: 160 | Energy: -15373.5657 eV
# Step: 161 | Energy: -15373.5657 eV
# Step: 162 | Energy: -15373.5657 eV
# Step: 163 | Energy: -15373.5656 eV
# Step: 164 | Energy: -15373.5656 eV
# Step: 165 | Energy: -15373.5656 eV
# Step: 166 | Energy: -15373.5656 eV
# Step: 167 | Energy: -15373.5655 eV
# Step: 168 | Energy: -15373.5655 eV
# Step: 169 | Energy: -15373.5655 eV
# Step: 170 | Energy: -15373.5654 eV
# Step: 171 | Energy: -15373.5654 eV
# Step: 172 | Energy: -15373.5653 eV
# Step: 173 | Energy: -15373.5653 eV
# Step: 174 | Energy: -15373.5652 eV
# Step: 175 | Energy: -15373.5652 eV
# Step: 176 | Energy: -15373.5651 eV
# Step: 177 | Energy: -15373.5651 eV
# Step: 178 | Energy: -15373.5651 eV
# Step: 179 | Energy: -15373.5650 eV
# Step: 180 | Energy: -15373.5650 eV
# Step: 181 | Energy: -15373.5650 eV
# Step: 182 | Energy: -15373.5650 eV
# Step: 183 | Energy: -15373.5650 eV
# Step: 184 | Energy: -15373.5651 eV
# Step: 185 | Energy: -15373.5651 eV
# Step: 186 | Energy: -15373.5651 eV
# Step: 187 | Energy: -15373.5652 eV
# Step: 188 | Energy: -15373.5653 eV
# Step: 189 | Energy: -15373.5653 eV
# Step: 190 | Energy: -15373.5654 eV
# Step: 191 | Energy: -15373.5655 eV
# Step: 192 | Energy: -15373.5656 eV
# Step: 193 | Energy: -15373.5657 eV
# Step: 194 | Energy: -15373.5658 eV
# Step: 195 | Energy: -15373.5659 eV
# Step: 196 | Energy: -15373.5660 eV
# Step: 197 | Energy: -15373.5661 eV
# Step: 198 | Energy: -15373.5662 eV
# Step: 199 | Energy: -15373.5663 eV
# Step: 200 | Energy: -15373.5663 eV

# Simulation complete. Open 'ArLi5_thermal_lock.traj' to confirm 'Ar' remains caged.
# (base) brendanlynch@Brendans-Laptop ArLi5 % 