import numpy as np

def test_manufacturing_tolerances(impurity_level_ppm):
    """
    Calculates the impact of interstitial 'noise' (Na/Mg) on the 
    Lynch-Node c/a ratio.
    """
    # Base Nodal Stability (ACI)
    base_stability = 0.9998
    
    # Interference Constant: How much 100ppm of Na distorts the lattice
    # Sodium ions are larger than Nitrogen, causing 'Lattice Swell'
    distortion_factor = impurity_level_ppm * 1.5e-5
    
    effective_ratio = 1.6330 + distortion_factor
    ideal_node = 1.63299
    
    # Magnetocrystalline Anisotropy (K1) Collapse Logic
    # K1 drops exponentially as the lattice is pushed out of the 1.633 manifold
    k1_at_purity = 9.0 * np.exp(-abs(effective_ratio - ideal_node) * 500)
    
    status = "STABLE" if k1_at_purity > 7.5 else "CRITICAL COLLAPSE"
    
    print(f"--- Impurity Report: {impurity_level_ppm} ppm Na ---")
    print(f"Lattice Distortion: Δ{distortion_factor:.5f}")
    print(f"Effective K1:       {k1_at_purity:.2f} MJ/m^3")
    print(f"Structural Status:  {status}")
    return k1_at_purity

# Scenario A: Ultra-High Selectivity MOF (Your current Bipyridine-UFT-F)
test_manufacturing_tolerances(impurity_level_ppm=150)

# Scenario B: Standard MOF (Higher interference)
test_manufacturing_tolerances(impurity_level_ppm=1200)

# (base) brendanlynch@Brendans-Laptop rareEarth % python test2.py
# --- Impurity Report: 150 ppm Na ---
# Lattice Distortion: Δ0.00225
# Effective K1:       2.91 MJ/m^3
# Structural Status:  CRITICAL COLLAPSE
# --- Impurity Report: 1200 ppm Na ---
# Lattice Distortion: Δ0.01800
# Effective K1:       0.00 MJ/m^3
# Structural Status:  CRITICAL COLLAPSE
# (base) brendanlynch@Brendans-Laptop rareEarth %