import numpy as np
from pyscf import gto, dft

# =================================================================
# UFT-F RESONANCE ENGINE: CLi9 (STABLE CARBONIA) CLOSURE
# Target: ACI Zero-Loss Node at the E8-Lattice Intersection
# =================================================================

def calculate_cli9_resonance_lock():
    # Placeholder coordinates for the optimized 1:9 geometry
    # In practice, these are derived from your 119 GPa formation node
    cli9_coords = [
        ['C',  (0.000000,  0.000000,  0.000000)],
        ['Li', (1.450000,  0.000000,  0.000000)],
        ['Li', (-1.450000, 0.000000,  0.000000)],
        ['Li', (0.000000,  1.450000,  0.000000)],
        ['Li', (0.000000, -1.450000,  0.000000)],
        ['Li', (0.725000,  0.725000,  1.025000)],
        ['Li', (-0.725000, -0.725000, 1.025000)],
        ['Li', (0.725000, -0.725000, -1.025000)],
        ['Li', (-0.725000, 0.725000, -1.025000)],
        ['Li', (0.000000,  0.000000,  1.600000)]
    ]

    mol = gto.M(
        atom=cli9_coords,
        basis='6-31g*',
        charge=0,
        spin=1  # Validating the radical state as in HeN5
    )

    print("--- EXTRACTING CLi9 RESONANCE DATA FOR PATENT ---")
    mf = dft.UKS(mol)
    mf.xc = 'pbe'
    energy = mf.kernel()

    # 1. ALPHA RESONANCE GAP (The Electronic Signature)
    n_alpha = mol.nelec[0]
    gap_a = (mf.mo_energy[0][n_alpha] - mf.mo_energy[0][n_alpha-1]) * 27.2114
    
    # 2. UFT-F STABILITY SCORE 
    # Logic: 100 * (1 - |Energy % c_UFT-F|)
    c_uft_f = 0.003119337
    stability_score = 100 * (1 - abs(energy % c_uft_f))

    print(f"\n[PATENT DATA: CLi9]")
    print(f"Total SCF Energy: {energy:.10f} Hartree")
    print(f"Alpha Resonance Gap: {gap_a:.4f} eV")
    print(f"UFT-F Resonance Stability: {stability_score:.4f}%")
    
    return stability_score

if __name__ == "__main__":
    calculate_cli9_resonance_lock()

#     (base) brendanlynch@Brendans-Laptop CLi9 % python stability.py
# /Users/brendanlynch/miniconda3/lib/python3.12/site-packages/pyscf/dft/libxc.py:771: UserWarning: Since PySCF-2.3, B3LYP (and B3P86) are changed to the VWN-RPA variant, corresponding to the original definition by Stephens et al. (issue 1480) and the same as the B3LYP functional in Gaussian. To restore the VWN5 definition, you can put the setting "B3LYP_WITH_VWN5 = True" in pyscf_conf.py
#   warnings.warn('Since PySCF-2.3, B3LYP (and B3P86) are changed to the VWN-RPA variant, '
# --- EXTRACTING CLi9 RESONANCE DATA FOR PATENT ---
# converged SCF energy = -104.062237817858  <S^2> = 1.955202  2S+1 = 2.9699845

# [PATENT DATA: CLi9]
# Total SCF Energy: -104.0622378179 Hartree
# Alpha Resonance Gap: 0.2232 eV
# UFT-F Resonance Stability: 99.8036%
# (base) brendanlynch@Brendans-Laptop CLi9 % 