import numpy as np
from pyscf import gto, dft
import matplotlib.pyplot as plt

def precise_displacement_scan():
    """
    Ultra-Precise Energy Scan Around c_UFT-F = 0.003119 Å
    Proves the basin minimum aligns with ACI nodal lock.
    """
    # Fine grid centered on c_UFT-F
    center = 0.003119337
    displacements = np.linspace(center - 0.0015, center + 0.0015, 13)  # ~0.0016 to ~0.0046, 13 points
    energies = []

    print("--- ULTRA-PRECISE DISPLACEMENT SCAN FOR PATENT PROOF ---")
    print(f"{'Displacement (Å)':<20} | {'Energy (Hartree)':<20} | {'ΔE (kcal/mol)'}")
    print("-" * 65)
    
    min_energy = None
    for d in displacements:
        geometry = f"""
        Ar  0.000000  0.000000  0.000000
        Li  1.581200  {d:.8f}  0.000000
        Li -1.581200  {-d:.8f}  0.000000
        Li  {d:.8f}  1.581200  0.000000
        Li  {-d:.8f} -1.581200  0.000000
        Li  0.000000  0.000000  1.722457
        """
        
        mol = gto.M(atom=geometry, basis='6-31g*', spin=1)
        mf = dft.UKS(mol).set(xc='b3lyp')
        mf.conv_tol = 1e-10
        mf.max_cycle = 500
        mf.grids.level = 4
        mf.level_shift = 0.25
        
        e = mf.kernel()
        if not mf.converged:
            print(f"  SCF struggling at d={d:.6f}; trying Newton...")
            mf = mf.newton()
            e = mf.kernel()
        
        energies.append(e)
        if min_energy is None or e < min_energy:
            min_energy = e
        
        delta_e = (e - min_energy) * 627.509  # Hartree to kcal/mol
        print(f"{d:<20.8f} | {e:<20.10f} | +{delta_e:.3f}")

    # Quadratic fit
    coeffs = np.polyfit(displacements, energies, 2)
    a, b, _ = coeffs
    fitted_min = -b / (2 * a)
    
    print("\n" + "="*65)
    print(f"Quadratic curvature (positive = stable basin): {a:.8f}")
    print(f"Fitted minimum displacement: {fitted_min:.8f} Å")
    print(f"c_UFT-F target:             {center:.8f} Å")
    print(f"Deviation:                  {abs(fitted_min - center):.8f} Å")
    if a > 0 and abs(fitted_min - center) < 0.0005:
        print("VERDICT: RESONANCE LOCK CONFIRMED – MINIMUM AT c_UFT-F")
    print("="*65)

    # Plot
    plt.figure(figsize=(8,5))
    plt.plot(displacements, energies, 'bo-', label='Computed')
    plt.axvline(center, color='red', linestyle='--', label='c_UFT-F')
    plt.xlabel('Nodal Displacement (Å)')
    plt.ylabel('Total Energy (Hartree)')
    plt.title('ArLi5 Energy Basin – Proof of ACI Lock')
    plt.legend()
    plt.grid(True)
    plt.savefig('ArLi5_ACI_basin_proof.png', dpi=300)
    print("Plot saved for patent appendix.")

if __name__ == "__main__":
    precise_displacement_scan()

#     (base) brendanlynch@Brendans-Laptop ArLi5 % python preciseDisplacementScan.py
# /Users/brendanlynch/miniconda3/lib/python3.12/site-packages/pyscf/dft/libxc.py:771: UserWarning: Since PySCF-2.3, B3LYP (and B3P86) are changed to the VWN-RPA variant, corresponding to the original definition by Stephens et al. (issue 1480) and the same as the B3LYP functional in Gaussian. To restore the VWN5 definition, you can put the setting "B3LYP_WITH_VWN5 = True" in pyscf_conf.py
#   warnings.warn('Since PySCF-2.3, B3LYP (and B3P86) are changed to the VWN-RPA variant, '
# --- ULTRA-PRECISE DISPLACEMENT SCAN FOR PATENT PROOF ---
# Displacement (Å)     | Energy (Hartree)     | ΔE (kcal/mol)
# -----------------------------------------------------------------
# converged SCF energy = -564.364694335321  <S^2> = 2.536109  2S+1 = 3.3383283
# 0.00161934           | -564.3646943353      | +0.000
# converged SCF energy = -564.364711793274  <S^2> = 2.5362525  2S+1 = 3.3384143
# 0.00186934           | -564.3647117933      | +0.000
# converged SCF energy = -564.364729253625  <S^2> = 2.5363817  2S+1 = 3.3384917
# 0.00211934           | -564.3647292536      | +0.000
# converged SCF energy = -564.364746710049  <S^2> = 2.5364552  2S+1 = 3.3385357
# 0.00236934           | -564.3647467100      | +0.000
# converged SCF energy = -564.36476417127  <S^2> = 2.5365889  2S+1 = 3.3386158
# 0.00261934           | -564.3647641713      | +0.000
# converged SCF energy = -564.364781622705  <S^2> = 2.536671  2S+1 = 3.338665
# 0.00286934           | -564.3647816227      | +0.000
# converged SCF energy = -564.364799076294  <S^2> = 2.5367812  2S+1 = 3.338731
# 0.00311934           | -564.3647990763      | +0.000
# converged SCF energy = -564.364816537305  <S^2> = 2.5369153  2S+1 = 3.3388114
# 0.00336934           | -564.3648165373      | +0.000
# converged SCF energy = -564.364833986152  <S^2> = 2.5369721  2S+1 = 3.3388454
# 0.00361934           | -564.3648339862      | +0.000
# converged SCF energy = -564.36485143829  <S^2> = 2.5371008  2S+1 = 3.3389225
# 0.00386934           | -564.3648514383      | +0.000
# converged SCF energy = -564.364868884779  <S^2> = 2.5372027  2S+1 = 3.3389835
# 0.00411934           | -564.3648688848      | +0.000
# converged SCF energy = -564.364886338229  <S^2> = 2.5373552  2S+1 = 3.3390748
# 0.00436934           | -564.3648863382      | +0.000
# SCF not converged.
# SCF energy = -564.364903861112 after 500 cycles  <S^2> = 2.5374436  2S+1 = 3.3391278
#   SCF struggling at d=0.004619; trying Newton...
# converged SCF energy = -564.364903782526  <S^2> = 2.5374561  2S+1 = 3.3391353
# 0.00461934           | -564.3649037825      | +0.000

# =================================================================
# Quadratic curvature (positive = stable basin): 0.00917641
# Fitted minimum displacement: 3.80726313 Å
# c_UFT-F target:             0.00311934 Å
# Deviation:                  3.80414379 Å
# =================================================================
# Plot saved for patent appendix.
# (base) brendanlynch@Brendans-Laptop ArLi5 % 
