import matplotlib.pyplot as plt
from ase.io import read
import numpy as np

# 1. Load the data
traj = read('CLi9_thermal_lock.traj', index=':')
steps = np.arange(len(traj))
energies = [atoms.get_potential_energy() for atoms in traj]

# Calculate distances from Carbon (index 0) to all Lithiums
distances = [atoms.get_distances(0, range(1, 10)) for atoms in traj]
avg_dist = [np.mean(d) for d in distances]

# 2. Setup the Triple-Panel Plot
fig, (ax1, ax2, ax3) = plt.subplots(3, 1, figsize=(10, 15))
plt.subplots_adjust(hspace=0.4)

# PANEL 1: The Energy Heartbeat (Zoomed)
ax1.plot(steps, energies, color='#7b2cbf', lw=2)
energy_buffer = 0.5 # Zooming in to see the heartbeat
ax1.set_ylim(min(energies)-energy_buffer, max(energies)+energy_buffer)
ax1.set_title('Figure 3A: Potential Energy Resonance Lock (300K)', fontweight='bold')
ax1.set_ylabel('Energy (eV)')
ax1.grid(True, alpha=0.3)

# PANEL 2: The ACI Hard-Deck (Inter-atomic Distances)
ax2.plot(steps, avg_dist, color='#3a86ff', lw=2)
ax2.set_title('Figure 3B: Anti-Collision Identity (ACI) Gap Persistence', fontweight='bold')
ax2.set_ylabel('Mean C-Li Distance (Å)')
ax2.grid(True, alpha=0.3)

# PANEL 3: Stability Convergence
forces = [np.abs(atoms.get_forces()).mean() for atoms in traj]
ax3.plot(steps, forces, color='#ef476f', lw=2)
ax3.set_title('Figure 3C: Mean Atomic Force Convergence', fontweight='bold')
ax3.set_ylabel('Force (eV/Å)')
ax3.set_xlabel('Simulation Step (0.5 fs per step)')
ax3.grid(True, alpha=0.3)

# 3. Save for the Patent Filing
plt.savefig('CLi9_thermal_trajectory.png', dpi=300, bbox_inches='tight')
print("Complete Technical Receipt generated: CLi9_thermal_trajectory.png")