import numpy as np
import matplotlib.pyplot as plt
from scipy.fft import fft2, fftshift

# --- UFT-F Constants ---
C_UFT_F = 0.003119337

def get_psd1d(image):
    """Computes the 1D Radial Power Spectrum of an image."""
    h, w = image.shape
    f_shift = fftshift(fft2(image))
    psd2d = np.abs(f_shift)**2
    
    y, x = np.indices(psd2d.shape)
    center = np.array([(x.max()-x.min())/2.0, (y.max()-y.min())/2.0])
    r = np.hypot(x - center[0], y - center[1]).astype(int)
    
    tbin = np.bincount(r.ravel(), psd2d.ravel())
    nr = np.bincount(r.ravel())
    radialprofile = tbin / nr
    return radialprofile

def ultimate_physics_test():
    size = 512
    # 1. Create a Fractal-Like Biological Structure (Turbulent Phantom)
    # This mimics the complex branching of micro-vasculature.
    y, x = np.mgrid[:size, :size]
    phantom = np.sin(x/10) * np.cos(y/10) * (np.sqrt((x-256)**2 + (y-256)**2) < 150)
    
    # 2. Add White Noise (The Clinical Challenge)
    noise = np.random.normal(0, 0.5, (size, size))
    noisy_input = phantom + noise
    
    # 3. Axiom-Rad Reconstruction
    f_domain = fftshift(fft2(noisy_input))
    dist_sq = (x-256)**2 + (y-256)**2
    k = np.sqrt(dist_sq)
    # The ACI-Closure Filter
    eta = C_UFT_F * 10
    axiom_filter = (k + 1.0)**(-5/3) * np.exp(-k * eta)
    axiom_filter /= np.max(axiom_filter)
    axiom_recon = np.abs(np.fft.ifft2(np.fft.ifftshift(f_domain * axiom_filter)))

    # 4. PSD Calculation for Comparison
    psd_truth = get_psd1d(phantom)
    psd_noisy = get_psd1d(noisy_input)
    psd_axiom = get_psd1d(axiom_recon)

    # 5. The "Shadow of a Doubt" Plot
    plt.figure(figsize=(10, 6))
    plt.loglog(psd_truth, label="Ground Truth (Anatomy)", color='green', lw=2)
    plt.loglog(psd_noisy, label="Noisy Input (Raw MRI)", color='red', alpha=0.5)
    plt.loglog(psd_axiom, label="Axiom-Rad (UFT-F)", color='blue', lw=2, linestyle='--')
    
    # The -5/3 Reference Line (The Law of Physics)
    k_ref = np.arange(10, 100)
    plt.loglog(k_ref, 1e7 * k_ref**(-5/3), color='black', label="Kolmogorov -5/3 Law", linestyle=':')

    plt.title("Spectral Energy Conservation Test")
    plt.xlabel("Wavenumber (k)")
    plt.ylabel("Power Spectral Density")
    plt.legend()
    plt.grid(True, which="both", ls="-", alpha=0.5)
    plt.show()

if __name__ == "__main__":
    ultimate_physics_test()