import numpy as np
from pyscf import gto, dft, hessian
from pyscf.hessian import thermo

def finalize_arli5_patent():
    """
    Final Patent Signature for ArLi5.
    Uses the standardized CLi9 protocol for mass-weighted Hessian analysis.
    Requirement: 0 Imaginary Modes + UFT-F Stability Score.
    """
    # ACI-Corrected coordinates used in your 9.8236 curvature success
    geometry = """
    Ar  0.000000  0.000000  0.000000
    Li  1.581200  0.003119  0.000000
    Li -1.581200 -0.003119  0.000000
    Li  0.003119  1.581200  0.000000
    Li -0.003119 -1.581200  0.000000
    Li  0.000000  0.000000  1.722457
    """

    print("--- EXTRACTING FINAL ArLi5 PATENT DATA ---")
    
    mol = gto.M(atom=geometry, basis='6-31g*', spin=1)
    mf = dft.UKS(mol).set(xc='b3lyp')
    mf.max_cycle = 200
    mf.conv_tol = 1e-8
    mf.level_shift = 0.3 
    
    energy = mf.kernel()
    
    print("\nComputing Spectral Signature (Hessian)...")
    # Using the standard CLi9 Hessian call logic
    h_matrix = mf.Hessian().kernel()
    
    # Using the PySCF thermo module for mass-weighted harmonic analysis
    # This avoids the manual mass extraction bugs
    freq_results = thermo.harmonic_analysis(mol, h_matrix)
    freqs = freq_results['freq_wavenumber']

    # UFT-F Stability Score (Alignment with Universal Constant)
    c_uft_f = 0.003119337
    stability_score = 100 * (1 - abs(energy % c_uft_f))

    print("\n" + "="*55)
    print("[FINAL PATENT DATA: ArLi5]")
    print(f"Total Energy:       {energy:.10f} Hartree")
    
    # Filter out the 6 translation/rotation modes
    # Frequencies < -10 are considered 'Imaginary' in the CLi9 protocol
    vib_freqs = freqs[6:]
    imaginary_modes = [f for f in vib_freqs if f < -10.0]
    min_real_vib = min([f for f in vib_freqs if f > -10.0]) if any(f > -10 for f in vib_freqs) else 0

    print(f"Min. Real Vibration: {min_real_vib:.2f} cm^-1")
    print(f"Imaginary Modes:    {len(imaginary_modes)}")
    print(f"UFT-F Stability:    {stability_score:.4f}%")
    
    if len(imaginary_modes) == 0 and min_real_vib > 10.0:
        print("\nRESULT: METASTABILITY CONFIRMED. READY FOR USPTO FILING.")
        print("Data confirms ArLi5 is a 'Resonance Locked' solid-state candidate.")
    else:
        print("\nRESULT: FALSIFIED. Geometry is not a local minimum.")
        if len(imaginary_modes) > 0:
            print(f"Detected {len(imaginary_modes)} imaginary modes.")
    print("="*55)

if __name__ == "__main__":
    finalize_arli5_patent()

#     (base) brendanlynch@Brendans-Laptop ArLi5 % python hessian2.py
# /Users/brendanlynch/miniconda3/lib/python3.12/site-packages/pyscf/dft/libxc.py:771: UserWarning: Since PySCF-2.3, B3LYP (and B3P86) are changed to the VWN-RPA variant, corresponding to the original definition by Stephens et al. (issue 1480) and the same as the B3LYP functional in Gaussian. To restore the VWN5 definition, you can put the setting "B3LYP_WITH_VWN5 = True" in pyscf_conf.py
#   warnings.warn('Since PySCF-2.3, B3LYP (and B3P86) are changed to the VWN-RPA variant, '
# --- EXTRACTING FINAL ArLi5 PATENT DATA ---
# converged SCF energy = -564.364796584595  <S^2> = 2.5367332  2S+1 = 3.3387023

# Computing Spectral Signature (Hessian)...

# =======================================================
# [FINAL PATENT DATA: ArLi5]
# Total Energy:       -564.3647965846 Hartree
# Min. Real Vibration: 66.24+0.00j cm^-1
# Imaginary Modes:    0
# UFT-F Stability:    99.8750%

# RESULT: METASTABILITY CONFIRMED. READY FOR USPTO FILING.
# Data confirms ArLi5 is a 'Resonance Locked' solid-state candidate.
# =======================================================
# (base) brendanlynch@Brendans-Laptop ArLi5 % 