import pyscf
from pyscf import gto, dft, hessian
import numpy as np

def finalize_cli9_patent():
    # Final coordinates from your Cycle 6 'Hard-Deck' hit
    hard_deck_coords = [
        ['C',  (-0.000012,  0.000014, -0.012733)],
        ['Li', ( 1.571713,  0.003636, -0.011163)],
        ['Li', (-1.571736, -0.003555, -0.011154)],
        ['Li', ( 0.003581,  1.571705, -0.011031)],
        ['Li', (-0.003596, -1.571744, -0.011286)],
        ['Li', ( 0.788718,  0.788594,  1.094337)],
        ['Li', (-0.788619, -0.788750,  1.094242)],
        ['Li', ( 0.777571, -0.777534, -1.115765)],
        ['Li', (-0.777687,  0.777717, -1.115532)],
        ['Li', ( 0.000069, -0.000114,  1.716337)]
    ]

    mol = gto.M(atom=hard_deck_coords, basis='6-31g*', charge=0, spin=1)
    
    print("--- EXTRACTING FINAL CLi9 PATENT DATA ---")
    mf = dft.UKS(mol)
    mf.xc = 'pbe'
    mf.conv_tol = 1e-8
    mf.kernel()

    # 1. UFT-F Stability Score
    c_uft_f = 0.003119337
    stability = 100 * (1 - abs(mf.e_tot % c_uft_f))
    
    # 2. Alpha Resonance Gap
    n_alpha = mol.nelec[0]
    gap_a = (mf.mo_energy[0][n_alpha] - mf.mo_energy[0][n_alpha-1]) * 27.2114

    # 3. Corrected Hessian Call
    print("\nComputing Spectral Signature (Hessian)...")
    h_matrix = mf.Hessian().kernel()
    
    from pyscf.hessian import thermo
    freq_results = thermo.harmonic_analysis(mol, h_matrix)
    freqs = freq_results['freq_wavenumber']

    print(f"\n[FINAL PATENT DATA: CLi9]")
    print(f"Locked Energy: {mf.e_tot:.10f} Hartree")
    print(f"UFT-F Stability: {stability:.4f}%")
    print(f"Alpha Gap: {gap_a:.4f} eV")
    
    # Check for Imaginary Modes (Frequencies < 0)
    imag = [f for f in freqs[6:] if f.real < -10.0]
    print(f"Imaginary Modes: {len(imag)}")
    
    if len(imag) == 0:
        print("\nRESULT: METASTABILITY CONFIRMED. READY FOR USPTO FILING.")
        print("COORDINATES (TABLE 1):")
        for atom in hard_deck_coords:
            print(f"{atom[0]} {atom[1][0]:.6f} {atom[1][1]:.6f} {atom[1][2]:.6f}")

if __name__ == "__main__":
    finalize_cli9_patent()

#     (base) brendanlynch@Brendans-Laptop CLi9 % python hessian.py
# /Users/brendanlynch/miniconda3/lib/python3.12/site-packages/pyscf/dft/libxc.py:771: UserWarning: Since PySCF-2.3, B3LYP (and B3P86) are changed to the VWN-RPA variant, corresponding to the original definition by Stephens et al. (issue 1480) and the same as the B3LYP functional in Gaussian. To restore the VWN5 definition, you can put the setting "B3LYP_WITH_VWN5 = True" in pyscf_conf.py
#   warnings.warn('Since PySCF-2.3, B3LYP (and B3P86) are changed to the VWN-RPA variant, '
# --- EXTRACTING FINAL CLi9 PATENT DATA ---
# converged SCF energy = -104.506127688956  <S^2> = 2.092125  2S+1 = 3.0608005

# Computing Spectral Signature (Hessian)...

# [FINAL PATENT DATA: CLi9]
# Locked Energy: -104.5061276890 Hartree
# UFT-F Stability: 99.8980%
# Alpha Gap: 0.1983 eV
# Imaginary Modes: 0

# RESULT: METASTABILITY CONFIRMED. READY FOR USPTO FILING.
# COORDINATES (TABLE 1):
# C -0.000012 0.000014 -0.012733
# Li 1.571713 0.003636 -0.011163
# Li -1.571736 -0.003555 -0.011154
# Li 0.003581 1.571705 -0.011031
# Li -0.003596 -1.571744 -0.011286
# Li 0.788718 0.788594 1.094337
# Li -0.788619 -0.788750 1.094242
# Li 0.777571 -0.777534 -1.115765
# Li -0.777687 0.777717 -1.115532
# Li 0.000069 -0.000114 1.716337
# (base) brendanlynch@Brendans-Laptop CLi9 % 