import numpy as np

def generate_exact_geometries(node_name, mw_eff):
    """
    Derives the Unit Cell parameters based on the Nodal Mass Density.
    Translates UFT-F Spectral Nodes into Angstrom-level coordinates.
    """
    # Nodal scaling factor (derived from the Base-24 manifold)
    # This translates mass-density to spatial-volume
    V_node = (mw_eff / 24.0) ** (1/3)
    
    # Standard Hexagonal Lattice Relationship: c/a ratio
    # For high-coercivity, we look for the 'Golden Ratio' stretch (~1.633)
    a = 2.51 * V_node  # Base dimension in Angstroms
    c = a * 1.633      # Height dimension (The 'Easy Axis' stretch)
    
    # Wyckoff Positions for Space Group 194 (Hexagonal)
    # This is the 'falsifiable' geometry for a chemist
    geometry = {
        'Space Group': 'P63/mmc (194)',
        'Lattice a': round(a, 4),
        'Lattice c': round(c, 4),
        'Volume': round(0.866 * (a**2) * c, 3),
        'Positions': {
            'W (Anchor)': '2b (0, 0, 1/4)',
            'Co/Fe (Magnetic)': '6h (x, 2x, 1/4) where x=0.833',
            'N (Interstitial)': '2a (0, 0, 0)'
        }
    }
    return geometry

# Evaluate your 'Locked' Nodes
nodes = {
    "Nd-Killer (6th Node)": 144.0049753,
    "Dy-Killer (7th Node)": 168.0049753
}

print(f"{'Node Target':<25} | {'Lattice a (Å)':<15} | {'Lattice c (Å)':<15} | {'Cell Vol (Å³)'}")
print("-" * 80)
for name, mw in nodes.items():
    geo = generate_exact_geometries(name, mw)
    print(f"{name:<25} | {geo['Lattice a']:<15} | {geo['Lattice c']:<15} | {geo['Volume']}")

#     (base) brendanlynch@Brendans-Laptop rareEarth % python geometries.py
# Node Target               | Lattice a (Å)   | Lattice c (Å)   | Cell Vol (Å³)
# --------------------------------------------------------------------------------
# Nd-Killer (6th Node)      | 4.561           | 7.4482          | 134.181
# Dy-Killer (7th Node)      | 4.8015          | 7.8409          | 156.544
# (base) brendanlynch@Brendans-Laptop rareEarth % 