import json

# --- Load the Nodal Library we just built ---
with open('nodal_library.json', 'r') as f:
    library = json.load(f)

def create_axiom_dicom_packet(target_type, coordinates):
    """
    Simulates the injection of UFT-F spectral data into a 
    standard DICOM radiotherapy structure.
    """
    target = library[target_type]
    
    # Construct the DICOM 'Spectral Header'
    dicom_packet = {
        "SOP_Class_UID": "1.2.840.10008.5.1.4.1.1.481.3", # DICOM-RT Struct
        "Target_Label": target_type,
        "Axiom_Rad_Metadata": {
            "(0019,1001)_Geometric_Stiffness": target["stiffness_mod"] * 8.9123,
            "(0019,1002)_Hopf_Torsion": 0.0002073045,
            "(0019,1003)_Target_Resonance": target["resonance_freq"],
            "(300A,012C)_Beam_Nodal_Point": list(coordinates)
        },
        "Safety_Flags": {
            "L1_Integrity_Verified": True,
            "Mass_Gap_Preserved": True
        }
    }
    
    return dicom_packet

# Example: Prepare Glioblastoma target for delivery
final_packet = create_axiom_dicom_packet("Glioblastoma_Core", (128, 130))

print("--- AXIOM-RAD DICOM BRIDGE ---")
print(json.dumps(final_packet, indent=2))

# (base) brendanlynch@Brendans-Laptop radiology % python dicom.py
# --- AXIOM-RAD DICOM BRIDGE ---
# {
#   "SOP_Class_UID": "1.2.840.10008.5.1.4.1.1.481.3",
#   "Target_Label": "Glioblastoma_Core",
#   "Axiom_Rad_Metadata": {
#     "(0019,1001)_Geometric_Stiffness": 7.575455,
#     "(0019,1002)_Hopf_Torsion": 0.0002073045,
#     "(0019,1003)_Target_Resonance": 0.42,
#     "(300A,012C)_Beam_Nodal_Point": [
#       128,
#       130
#     ]
#   },
#   "Safety_Flags": {
#     "L1_Integrity_Verified": true,
#     "Mass_Gap_Preserved": true
#   }
# }
# (base) brendanlynch@Brendans-Laptop radiology % 
