import numpy as np
import matplotlib.pyplot as plt
from scipy.fft import fft2, ifft2, fftshift

# --- UFT-F Constants ---
C_UFT_F = 0.003119337

def stress_test_axiom_rad():
    size = 512
    # 1. CREATE THE 'UNSOLVABLE' PHANTOM
    # A tiny high-contrast nodule (2 pixels) near a sharp edge
    # This creates a 'Gibbs Singularity' for standard FFT filters.
    phantom = np.zeros((size, size))
    yy, xx = np.mgrid[:size, :size]
    phantom[200:312, 200:210] = 1.0  # Sharp vertical edge
    phantom[256, 225] = 1.0          # The 'Sub-pixel' Nodule (The Test Case)
    
    # 2. SIMULATE DATA LOSS (UNDERSAMPLING)
    # We remove 80% of the high-frequency data, simulating a 5x faster MRI scan.
    k_space = fftshift(fft2(phantom))
    mask = np.zeros((size, size))
    mask[size//2-40:size//2+40, size//2-40:size//2+40] = 1.0
    undersampled_k = k_space * mask
    
    # 3. STANDARD RECONSTRUCTION (SINC-BLUR)
    # This represents what a standard scanner would show.
    std_recon = np.abs(ifft2(fftshift(undersampled_k)))
    
    # 4. AXIOM-RAD RECONSTRUCTION (KOLMOGOROV-ACI CLOSURE)
    # We use the -5/3 law to 'infer' the missing turbulence in k-space.
    y, x = np.ogrid[-size//2:size//2, -size//2:size//2]
    k_radius = np.sqrt(x**2 + y**2)
    eta = C_UFT_F * 5
    
    # The ACI-Corrected Filter (Non-linear closure)
    # This attempts to restore the nodal point by enforcing L1 stability.
    axiom_filter = (k_radius + 1.0)**(-5/6) * np.exp(-k_radius * eta)
    axiom_filter /= np.max(axiom_filter)
    
    axiom_recon = np.abs(ifft2(fftshift(undersampled_k * axiom_filter)))

    # 5. THE FALSIFIABILITY METRIC: SIGNAL-TO-RECONSTRUCTION RATIO (SRR)
    # We check if the tiny nodule at [256, 225] is visible or lost in the blur.
    val_std = std_recon[256, 225]
    val_axiom = axiom_recon[256, 225]
    
    print("--- HARDEST ROBUST STRESS TEST ---")
    print(f"Standard Detection Intensity: {val_std:.6f}")
    print(f"Axiom-Rad Detection Intensity: {val_axiom:.6f}")
    print(f"Resolution Gain: {(val_axiom / (val_std + 1e-9)):.2f}x")
    
    # Visualizing the 'Ringing' vs 'Stability'
    plt.figure(figsize=(12, 6))
    plt.subplot(121); plt.imshow(std_recon[230:280, 200:250], cmap='gray')
    plt.title("Standard: Gibbs Ringing & Loss")
    plt.subplot(122); plt.imshow(axiom_recon[230:280, 200:250], cmap='magma')
    plt.title("Axiom-Rad: Nodal Preservation")
    plt.show()

if __name__ == "__main__":
    stress_test_axiom_rad()

#     (base) brendanlynch@Brendans-Laptop radiology % python axiomRadStresstest.py
# --- HARDEST ROBUST STRESS TEST ---
# Standard Detection Intensity: 0.014359
# Axiom-Rad Detection Intensity: 0.037564
# Resolution Gain: 2.62x
# 2026-01-02 05:46:09.561 python[39083:74456260] The class 'NSSavePanel' overrides the method identifier.  This method is implemented by class 'NSWindow'
# (base) brendanlynch@Brendans-Laptop radiology % 