import numpy as np
from pyscf import gto, dft, grad
from ase import Atoms
from ase.md.velocitydistribution import MaxwellBoltzmannDistribution
from ase.md.verlet import VelocityVerlet
from ase.units import fs
from ase.calculators.calculator import Calculator, all_changes

# =================================================================
# CLi9 THERMAL STABILITY BENCHMARK (300K)
# This script proves the "Resonance Lock" survives room temperature.
# =================================================================

class PySCF_Direct(Calculator):
    implemented_properties = ['energy', 'forces']
    def __init__(self, **kwargs):
        Calculator.__init__(self, **kwargs)
        self.dm = None  # To store the electron cloud state

    def calculate(self, atoms=None, properties=['energy'], system_changes=all_changes):
        Calculator.calculate(self, atoms, properties, system_changes)
        pos = self.atoms.get_positions()
        mol_list = [[self.atoms.symbols[i], pos[i]] for i in range(len(self.atoms))]
        
        # 1. Use a more robust Basis Set (matching your HeN5 patent [cite: 402])
        mol = gto.M(atom=mol_list, basis='6-31g*', charge=0, spin=1, verbose=0)
        mf = dft.UKS(mol).set(xc='pbe')
        
        # 2. Add the "Lynch-Shift" (Level Shifting prevents the Step 15 snap)
        mf.level_shift = 0.3  # Forces stability in the Lithium cage
        mf.conv_tol = 1e-5    # Tighter tolerance for starship-grade data
        mf.max_cycle = 200    # More time to find the 'Lock'
        
        # 3. RESTART LOGIC: Use the previous density matrix (The 'Memory' fix)
        if self.dm is None:
            e_tot = mf.kernel()
        else:
            e_tot = mf.kernel(dm0=self.dm)
        
        self.dm = mf.make_rdm1() # Save for the next femtosecond
        
        if not mf.converged:
            # Fallback to Second-Order SCF (Newton) if it struggles
            mf = mf.newton()
            e_tot = mf.kernel(dm0=self.dm)
            
        g = mf.Gradients().kernel()
        self.results['energy'] = e_tot * 27.2114 
        self.results['forces'] = -g * 27.2114 / 0.529177

# 1. INITIALIZE CLi9 ATOMS (Using Table 1 Patent Coordinates)
initial_coords = [
    ('C',  ( -0.000012,  0.000014, -0.012733)),
    ('Li', (  1.571713,  0.003636, -0.011163)),
    ('Li', ( -1.571736, -0.003555, -0.011154)),
    ('Li', (  0.003581,  1.571705, -0.011031)),
    ('Li', ( -0.003596, -1.571744, -0.011286)),
    ('Li', (  0.788718,  0.788594,  1.094337)),
    ('Li', ( -0.788619, -0.788750,  1.094242)),
    ('Li', (  0.777571, -0.777534, -1.115765)),
    ('Li', ( -0.777687,  0.777717, -1.115532)),
    ('Li', (  0.000069, -0.000114,  1.716337))
]

symbols = [c[0] for c in initial_coords]
positions = [c[1] for c in initial_coords]
atoms = Atoms(symbols=symbols, positions=positions)

# 2. ATTACH CALCULATOR & INITIALIZE THERMAL STATE
atoms.calc = PySCF_Direct()
print("Initializing Maxwell-Boltzmann Distribution at 300K...")
MaxwellBoltzmannDistribution(atoms, temperature_K=300)

# 3. CONFIGURE VERLET INTEGRATOR (1.0 fs steps)
# This will save a .traj file you can open in Avogadro to see the vibrations.
dyn = VelocityVerlet(atoms, timestep=0.5 * fs, trajectory='CLi9_thermal_lock.traj')

def print_status():
    print(f"Step: {dyn.get_number_of_steps()} | Energy: {atoms.get_potential_energy():.4f} eV")

dyn.attach(print_status, interval=1)

print("\n--- STARTING CLi9 THERMAL STABILITY RUN ---")
# Run 200 steps (200 femtoseconds) to establish initial thermal equilibrium
dyn.run(200)

print("\nSimulation complete. Open 'CLi9_thermal_lock.traj' to confirm 'C' stays caged.")

# (base) brendanlynch@Brendans-Laptop CLi9 % python amd.py
# /Users/brendanlynch/miniconda3/lib/python3.12/site-packages/pyscf/dft/libxc.py:771: UserWarning: Since PySCF-2.3, B3LYP (and B3P86) are changed to the VWN-RPA variant, corresponding to the original definition by Stephens et al. (issue 1480) and the same as the B3LYP functional in Gaussian. To restore the VWN5 definition, you can put the setting "B3LYP_WITH_VWN5 = True" in pyscf_conf.py
#   warnings.warn('Since PySCF-2.3, B3LYP (and B3P86) are changed to the VWN-RPA variant, '
# Initializing Maxwell-Boltzmann Distribution at 300K...

# --- STARTING CLi9 THERMAL STABILITY RUN ---
# Step: 0 | Energy: -2843.7546 eV
# Step: 1 | Energy: -2843.8527 eV
# Step: 2 | Energy: -2844.2154 eV
# Step: 3 | Energy: -2844.8317 eV
# Step: 4 | Energy: -2845.6803 eV
# Step: 5 | Energy: -2846.7364 eV
# Step: 6 | Energy: -2847.9386 eV
# Step: 7 | Energy: -2849.2584 eV
# Step: 8 | Energy: -2850.6642 eV
# Step: 9 | Energy: -2852.1161 eV
# Step: 10 | Energy: -2853.5748 eV
# Step: 11 | Energy: -2855.0113 eV
# Step: 12 | Energy: -2856.3994 eV
# Step: 13 | Energy: -2857.7188 eV
# Step: 14 | Energy: -2858.9536 eV
# Step: 15 | Energy: -2860.0932 eV
# Step: 16 | Energy: -2861.1308 eV
# Step: 17 | Energy: -2862.0636 eV
# Step: 18 | Energy: -2862.8917 eV
# Step: 19 | Energy: -2863.6177 eV
# Step: 20 | Energy: -2864.2462 eV
# Step: 21 | Energy: -2864.7825 eV
# Step: 22 | Energy: -2865.2330 eV
# Step: 23 | Energy: -2865.6051 eV
# Step: 24 | Energy: -2865.9051 eV
# Step: 25 | Energy: -2866.1407 eV
# Step: 26 | Energy: -2866.3179 eV
# Step: 27 | Energy: -2866.4436 eV
# Step: 28 | Energy: -2866.5233 eV
# Step: 29 | Energy: -2866.5628 eV
# Step: 30 | Energy: -2866.5666 eV
# Step: 31 | Energy: -2866.5400 eV
# Step: 32 | Energy: -2866.4886 eV
# Step: 33 | Energy: -2866.4081 eV
# Step: 34 | Energy: -2866.3138 eV
# Step: 35 | Energy: -2866.1975 eV
# Step: 36 | Energy: -2866.0715 eV
# Step: 37 | Energy: -2865.9262 eV
# Step: 38 | Energy: -2865.7749 eV
# Step: 39 | Energy: -2865.6120 eV
# Step: 40 | Energy: -2865.4436 eV
# Step: 41 | Energy: -2865.2633 eV
# Step: 42 | Energy: -2865.0844 eV
# Step: 43 | Energy: -2864.8869 eV
# Step: 44 | Energy: -2864.7045 eV
# Step: 45 | Energy: -2864.4919 eV
# Step: 46 | Energy: -2864.2947 eV
# Step: 47 | Energy: -2864.0892 eV
# Step: 48 | Energy: -2863.8721 eV
# Step: 49 | Energy: -2863.6783 eV
# Step: 50 | Energy: -2863.4686 eV
# Step: 51 | Energy: -2863.2944 eV
# Step: 52 | Energy: -2863.0816 eV
# Step: 53 | Energy: -2862.9368 eV
# Step: 54 | Energy: -2862.8319 eV
# Step: 55 | Energy: -2862.6521 eV
# Step: 56 | Energy: -2862.4843 eV
# Step: 57 | Energy: -2862.3016 eV
# Step: 58 | Energy: -2862.1402 eV
# Step: 59 | Energy: -2861.9573 eV
# Step: 60 | Energy: -2861.8035 eV
# Step: 61 | Energy: -2861.6281 eV
# Step: 62 | Energy: -2861.4764 eV
# Step: 63 | Energy: -2861.3090 eV
# Step: 64 | Energy: -2861.1612 eV
# Step: 65 | Energy: -2860.9974 eV
# Step: 66 | Energy: -2860.8545 eV
# Step: 67 | Energy: -2860.6936 eV
# Step: 68 | Energy: -2860.5573 eV
# Step: 69 | Energy: -2860.4038 eV
# Step: 70 | Energy: -2860.2719 eV
# Step: 71 | Energy: -2860.1263 eV
# Step: 72 | Energy: -2859.9980 eV
# Step: 73 | Energy: -2859.8606 eV
# Step: 74 | Energy: -2859.7376 eV
# Step: 75 | Energy: -2859.6165 eV
# Step: 76 | Energy: -2859.5012 eV
# Step: 77 | Energy: -2859.3790 eV
# Step: 78 | Energy: -2859.2610 eV
# Step: 79 | Energy: -2859.1364 eV
# Step: 80 | Energy: -2859.0237 eV
# Step: 81 | Energy: -2858.9045 eV
# Step: 82 | Energy: -2858.7968 eV
# Step: 83 | Energy: -2858.6821 eV
# Step: 84 | Energy: -2858.5796 eV
# Step: 85 | Energy: -2858.4698 eV
# Step: 86 | Energy: -2858.3752 eV
# Step: 87 | Energy: -2858.2712 eV
# Step: 88 | Energy: -2858.1772 eV
# Step: 89 | Energy: -2858.0782 eV
# Step: 90 | Energy: -2857.9918 eV
# Step: 91 | Energy: -2857.8981 eV
# Step: 92 | Energy: -2857.8124 eV
# Step: 93 | Energy: -2857.7239 eV
# Step: 94 | Energy: -2857.6452 eV
# Step: 95 | Energy: -2857.5616 eV
# Step: 96 | Energy: -2857.4864 eV
# Step: 97 | Energy: -2857.4076 eV
# Step: 98 | Energy: -2857.3358 eV
# Step: 99 | Energy: -2857.2540 eV
# Step: 100 | Energy: -2857.1846 eV
# Step: 101 | Energy: -2857.1162 eV
# Step: 102 | Energy: -2857.0490 eV
# Step: 103 | Energy: -2856.9798 eV
# Step: 104 | Energy: -2856.9117 eV
# Step: 105 | Energy: -2856.8499 eV
# Step: 106 | Energy: -2856.7890 eV
# Step: 107 | Energy: -2856.7415 eV
# Step: 108 | Energy: -2856.6692 eV
# Step: 109 | Energy: -2856.6409 eV
# Step: 110 | Energy: -2856.5384 eV
# Step: 111 | Energy: -2856.5430 eV
# Step: 112 | Energy: -2856.4088 eV
# Step: 113 | Energy: -2856.4253 eV
# Step: 114 | Energy: -2856.2883 eV
# Step: 115 | Energy: -2856.3552 eV
# Step: 116 | Energy: -2856.1341 eV
# Step: 117 | Energy: -2856.2626 eV
# Step: 118 | Energy: -2856.0736 eV
# Step: 119 | Energy: -2856.1848 eV
# Step: 120 | Energy: -2855.9700 eV
# Step: 121 | Energy: -2856.0974 eV
# Step: 122 | Energy: -2855.9423 eV
# Step: 123 | Energy: -2856.0349 eV
# Step: 124 | Energy: -2855.8933 eV
# Step: 125 | Energy: -2855.9252 eV
# Step: 126 | Energy: -2855.9107 eV
# Step: 127 | Energy: -2855.8274 eV
# Step: 128 | Energy: -2855.8771 eV
# Step: 129 | Energy: -2854.8121 eV
# Step: 130 | Energy: -2855.8724 eV
# Step: 131 | Energy: -2852.0495 eV
# Step: 132 | Energy: -2855.5496 eV
# Step: 133 | Energy: -2855.1767 eV
# Step: 134 | Energy: -2853.0586 eV
# Step: 135 | Energy: -2852.0500 eV
# Step: 136 | Energy: -2852.1183 eV
# Step: 137 | Energy: -2851.8875 eV
# Step: 138 | Energy: -2852.3890 eV
# Step: 139 | Energy: -2851.7926 eV
# Step: 140 | Energy: -2852.4651 eV
# Step: 141 | Energy: -2851.7541 eV
# Step: 142 | Energy: -2851.8524 eV
# Step: 143 | Energy: -2851.7167 eV
# Step: 144 | Energy: -2854.7782 eV
# Step: 145 | Energy: -2851.6667 eV
# Step: 146 | Energy: -2851.5933 eV
# Step: 147 | Energy: -2854.2330 eV
# Step: 148 | Energy: -2851.5553 eV
# Step: 149 | Energy: -2851.4307 eV
# Step: 150 | Energy: -2854.2483 eV
# Step: 151 | Energy: -2851.4460 eV
# Step: 152 | Energy: -2851.3825 eV
# Step: 153 | Energy: -2854.2039 eV
# Step: 154 | Energy: -2851.3559 eV
# Step: 155 | Energy: -2855.5180 eV
# Step: 156 | Energy: -2851.1795 eV
# Step: 157 | Energy: -2854.8837 eV
# Step: 158 | Energy: -2851.1096 eV
# Step: 159 | Energy: -2855.3061 eV
# Step: 160 | Energy: -2851.0890 eV
# Step: 161 | Energy: -2855.1790 eV
# Step: 162 | Energy: -2851.0567 eV
# Step: 163 | Energy: -2855.1908 eV
# Step: 164 | Energy: -2850.9947 eV
# Step: 165 | Energy: -2855.2337 eV
# Step: 166 | Energy: -2850.9532 eV
# Step: 167 | Energy: -2855.1656 eV
# Step: 168 | Energy: -2850.9139 eV
# Step: 169 | Energy: -2855.1548 eV
# Step: 170 | Energy: -2850.8629 eV
# Step: 171 | Energy: -2855.1654 eV
# Step: 172 | Energy: -2850.8181 eV
# Step: 173 | Energy: -2851.0711 eV
# Step: 174 | Energy: -2850.7799 eV
# Step: 175 | Energy: -2851.4712 eV
# Step: 176 | Energy: -2850.7408 eV
# Step: 177 | Energy: -2850.8796 eV
# Step: 178 | Energy: -2851.2663 eV
# Step: 179 | Energy: -2850.6939 eV
# Step: 180 | Energy: -2855.1038 eV
# Step: 181 | Energy: -2850.6341 eV
# Step: 182 | Energy: -2850.8658 eV
# Step: 183 | Energy: -2850.7055 eV
# Step: 184 | Energy: -2855.0778 eV
# Step: 185 | Energy: -2850.5895 eV
# Step: 186 | Energy: -2850.7580 eV
# Step: 187 | Energy: -2851.0933 eV
# Step: 188 | Energy: -2850.5673 eV
# Step: 189 | Energy: -2850.6802 eV
# Step: 190 | Energy: -2855.0156 eV
# Step: 191 | Energy: -2850.6338 eV
# Step: 192 | Energy: -2850.5301 eV
# Step: 193 | Energy: -2850.6511 eV
# Step: 194 | Energy: -2855.0060 eV
# Step: 195 | Energy: -2855.0150 eV
# Step: 196 | Energy: -2850.4955 eV
# Step: 197 | Energy: -2850.5672 eV
# Step: 198 | Energy: -2850.4232 eV
# Step: 199 | Energy: -2855.0271 eV
# Step: 200 | Energy: -2850.4563 eV

# Simulation complete. Open 'CLi9_thermal_lock.traj' to confirm 'C' stays caged.
# (base) brendanlynch@Brendans-Laptop CLi9 % from ase.io import read, write traj = read('CLi9_thermal_lock.traj', index=':') write('CLi9_animation.xyz', traj)
# zsh: unknown file attribute: C
# (base) brendanlynch@Brendans-Laptop CLi9 % python -c "from ase.io import read, write; traj = read('CLi9_thermal_lock.traj', index=':'); write('CLi9_animation.xyz', traj)"
# (base) brendanlynch@Brendans-Laptop CLi9 % 