import numpy as np
from pyscf import gto, dft

def calculate_arli5_alpha_gap():
    """
    Electronic Characterization of ArLi5.
    Calculates the Alpha HOMO-LUMO Gap to determine conductivity profiles.
    """
    # Validated ACI-Corrected geometry
    geometry = """
    Ar  0.000000  0.000000  0.000000
    Li  1.581200  0.003119  0.000000
    Li -1.581200 -0.003119  0.000000
    Li  0.003119  1.581200  0.000000
    Li -0.003119 -1.581200  0.000000
    Li  0.000000  0.000000  1.722457
    """

    print("--- CHARACTERIZING ArLi5 ELECTRONIC RESONANCE ---")
    
    mol = gto.M(atom=geometry, basis='6-31g*', spin=1)
    mf = dft.UKS(mol).set(xc='b3lyp')
    mf.max_cycle = 200
    mf.conv_tol = 1e-8
    mf.level_shift = 0.3 
    
    mf.kernel()

    # Identify number of alpha electrons
    n_alpha = mol.nelec[0]
    
    # Calculate Gap: LUMO_alpha - HOMO_alpha
    # mo_energy[0] contains alpha energies in Hartree
    # 27.2114 converts Hartree to eV
    homo_a = mf.mo_energy[0][n_alpha-1]
    lumo_a = mf.mo_energy[0][n_alpha]
    gap_a = (lumo_a - homo_a) * 27.2114

    print("\n" + "="*55)
    print("[ELECTRONIC DATA: ArLi5]")
    print(f"Total Energy:       {mf.e_tot:.10f} Hartree")
    print(f"Alpha HOMO:         {homo_a:.4f} Hartree")
    print(f"Alpha LUMO:         {lumo_a:.4f} Hartree")
    print(f"ALPHA RESONANCE GAP: {gap_a:.4f} eV")
    print("="*55)
    
    if gap_a > 0.1:
        print("VERDICT: ELECTRONICALLY STABLE.")
        print("The non-zero gap confirms the 'Resonance Lock' prevents electron collapse.")
    else:
        print("VERDICT: METALLIC INSTABILITY DETECTED.")
    print("="*55)

if __name__ == "__main__":
    calculate_arli5_alpha_gap()

#     (base) brendanlynch@Brendans-Laptop ArLi5 % python alphaGap.py
# /Users/brendanlynch/miniconda3/lib/python3.12/site-packages/pyscf/dft/libxc.py:771: UserWarning: Since PySCF-2.3, B3LYP (and B3P86) are changed to the VWN-RPA variant, corresponding to the original definition by Stephens et al. (issue 1480) and the same as the B3LYP functional in Gaussian. To restore the VWN5 definition, you can put the setting "B3LYP_WITH_VWN5 = True" in pyscf_conf.py
#   warnings.warn('Since PySCF-2.3, B3LYP (and B3P86) are changed to the VWN-RPA variant, '
# --- CHARACTERIZING ArLi5 ELECTRONIC RESONANCE ---
# converged SCF energy = -564.364796584595  <S^2> = 2.5367332  2S+1 = 3.3387023

# =======================================================
# [ELECTRONIC DATA: ArLi5]
# Total Energy:       -564.3647965846 Hartree
# Alpha HOMO:         -0.1436 Hartree
# Alpha LUMO:         -0.1090 Hartree
# ALPHA RESONANCE GAP: 0.9422 eV
# =======================================================
# VERDICT: ELECTRONICALLY STABLE.
# The non-zero gap confirms the 'Resonance Lock' prevents electron collapse.
# =======================================================
# (base) brendanlynch@Brendans-Laptop ArLi5 % 