import numpy as np

def simulate_mof_extraction():
    # Constants for Seawater Concentrations (mol/L approx)
    seawater_nodes = {
        'W':  5.4e-10,  # Tungsten (Target)
        'Co': 4.0e-10,  # Cobalt (Target)
        'Na': 4.7e-1,   # Sodium (The "Noise")
        'Mg': 5.3e-2    # Magnesium (The "Noise")
    }

    # Organic Linker Candidates
    # We look for the "Binding Energy Delta"
    # A negative delta means the target is preferred over salt.
    linkers = [
        {"name": "Amidoxime-Enhanced", "affinity_W": -14.5, "affinity_Na": -1.2},
        {"name": "Porphyrin-Cage",     "affinity_W": -18.2, "affinity_Na": -0.8},
        {"name": "Bipyridine-UFT-F",   "affinity_W": -22.4, "affinity_Na": -0.4}
    ]

    print(f"{'Linker Chemistry':<20} | {'W-Selectivity':<15} | {'Extraction Feasibility'}")
    print("-" * 65)

    for l in linkers:
        # Selectivity Ratio: How much more the MOF "wants" W over Na
        # Adjusted by the UFT-F Stability Constant (1 + 1/240)
        selectivity = np.exp(abs(l['affinity_W'] - l['affinity_Na'])) * (1 + 1/240)
        
        # We need a selectivity > 10^6 to overcome the high Na concentration
        status = "VIABLE" if selectivity > 1e6 else "INSUFFICIENT"
        
        print(f"{l['name']:<20} | {selectivity:<15.2e} | {status}")

simulate_mof_extraction()

# (base) brendanlynch@Brendans-Laptop rareEarth % python MOFFromOcean.py
# Linker Chemistry     | W-Selectivity   | Extraction Feasibility
# -----------------------------------------------------------------
# Amidoxime-Enhanced   | 6.00e+05        | INSUFFICIENT
# Porphyrin-Cage       | 3.62e+07        | VIABLE
# Bipyridine-UFT-F     | 3.60e+09        | VIABLE
# (base) brendanlynch@Brendans-Laptop rareEarth % 

