// __BEGIN_LICENSE__
//  Copyright (c) 2009-2013, United States Government as represented by the
//  Administrator of the National Aeronautics and Space Administration. All
//  rights reserved.
//
//  The NGT platform is licensed under the Apache License, Version 2.0 (the
//  "License"); you may not use this file except in compliance with the
//  License. You may obtain a copy of the License at
//  http://www.apache.org/licenses/LICENSE-2.0
//
//  Unless required by applicable law or agreed to in writing, software
//  distributed under the License is distributed on an "AS IS" BASIS,
//  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//  See the License for the specific language governing permissions and
//  limitations under the License.
// __END_LICENSE__

/// \file stereo_parse.cc
///
/// This program is to allow python access to stereo settings.

#include <asp/Core/StereoSettings.h>
#include <asp/Core/StereoTiling.h>
#include <asp/Core/Macros.h>
#include <asp/Tools/stereo.h>
#include <asp/Sessions/StereoSession.h>
#include <asp/Sessions/StereoSessionFactory.h>
#include <asp/Core/DisparityProcessing.h>

#include <vw/Stereo/DisparityMap.h>
#include <vw/Cartography/GeoReferenceUtils.h>
#include <vw/Stereo/CorrelationView.h>
#include <vw/Cartography/shapeFile.h>
#include <vw/FileIO/DiskImageView.h>

#include <xercesc/util/PlatformUtils.hpp>

using namespace vw;
using namespace asp;
namespace fs = boost::filesystem;

// Find the tile at given location for a parallel_stereo run with local epipolar
// alignment.
void find_tile_at_loc(std::string const& tile_at_loc, ASPGlobalOptions const& opt) {

  if (opt.input_dem != "") 
    vw_throw(ArgumentErr() << "Option --tile-at-location does not work "
             << "with mapprojected images.\n");

  std::istringstream iss(tile_at_loc);
  double lon, lat, h;
  if (!(iss >> lon >> lat >> h))
    vw_throw(ArgumentErr() << "Could not parse --tile-at-location.\n");

  vw::cartography::GeoReference georef = opt.session->get_georef();

  vw::CamPtr left_camera_model, right_camera_model;
  opt.session->camera_models(left_camera_model, right_camera_model);
  
  Vector3 xyz = georef.datum().geodetic_to_cartesian(Vector3(lon, lat, h));
  Vector2 pix = left_camera_model->point_to_pixel(xyz);

  vw::TransformPtr tx_left = opt.session->tx_left();

  pix = tx_left->forward(pix);

  // Read the tiles
  std::string line;
  std::string dir_list = opt.out_prefix + "-dirList.txt";
  vw_out() << "Reading list of tiles: " << dir_list << "\n";
  std::ifstream ifs(dir_list);
  std::vector<BBox2i> boxes;
  bool success = false;
  while (ifs >> line) {
    std::string::size_type pos = line.find(opt.out_prefix);
    if (pos == std::string::npos) 
      vw_throw(ArgumentErr() << "Could not find the output prefix in " << dir_list << ".\n");

    std::string tile_name = line;
    line.replace(pos, opt.out_prefix.size() + 1, ""); // add 1 to replace the dash

    int start_x, start_y, wid_x, wid_y;
    int ans = sscanf(line.c_str(), "%d_%d_%d_%d", &start_x, &start_y, &wid_x, &wid_y);
    if (ans != 4) 
      vw_throw(ArgumentErr() << "Error parsing 4 numbers from string: " << line);

    BBox2i box(start_x, start_y, wid_x, wid_y);
    if (box.contains(pix)) {
      vw::vw_out() << "Tile with location: " << tile_name << "\n";
      success = true;
    }
  }

  if (!success)
    vw_out() << "No tile found at location.\n"; 
}

int main(int argc, char* argv[]) {

  try {
    xercesc::XMLPlatformUtils::Initialize();
    stereo_register_sessions();

    bool verbose = true;
    std::vector<ASPGlobalOptions> opt_vec;
    std::string output_prefix;
    asp::parseStereoArgs(argc, argv, ParseDescription(),
                         verbose, output_prefix, opt_vec);
    if (opt_vec.empty())
      return 1;

    ASPGlobalOptions opt = opt_vec[0];

    // This is needed for the stereo_dist distributed processing program. Save
    // the needed list of tiles to disk (and their mapprojected versions as a
    // shapefile) and exit.
    if (asp::stereo_settings().stereo_dist_tile_params != vw::Vector2i(0, 0)) {
      produceDistTileList(opt.in_file1, opt.in_file2, output_prefix,
                          asp::stereo_settings().stereo_dist_tile_params);
      return 0;
    }

    // Use this as a small utility in debugging parallel_stereo runs
    if (!stereo_settings().tile_at_loc.empty()) {
      find_tile_at_loc(stereo_settings().tile_at_loc, opt);
      return 0;
    }
    
    vw_out() << "in_file1,"        << opt.in_file1        << "\n";
    vw_out() << "in_file2,"        << opt.in_file2        << "\n";
    vw_out() << "cam_file1,"       << opt.cam_file1       << "\n";
    vw_out() << "cam_file2,"       << opt.cam_file2       << "\n";
    vw_out() << "input_dem,"       << opt.input_dem       << "\n";
    vw_out() << "extra_argument1," << opt.extra_argument1 << "\n";
    vw_out() << "extra_argument2," << opt.extra_argument2 << "\n";
    vw_out() << "extra_argument3," << opt.extra_argument3 << "\n";

    vw_out() << "stereo_session,"   << opt.stereo_session << "\n";
    vw_out() << "stereo_default_filename," << opt.stereo_default_filename << "\n";
    vw_out() << "left_image_crop_win,"
             << stereo_settings().left_image_crop_win.min().x() << ","
             << stereo_settings().left_image_crop_win.min().y() << ","
             << stereo_settings().left_image_crop_win.width()   << ","
             << stereo_settings().left_image_crop_win.height()  << "\n";

    // This is different from opt.out_prefix for multiview
    vw_out() << "out_prefix," << output_prefix << "\n";

    Vector2i left_image_size  = file_image_size(opt.in_file1),
             right_image_size = file_image_size(opt.in_file2);
    vw_out() << "left_image_size,"  << left_image_size.x()  << "," 
             << left_image_size.y()  << "\n";
    vw_out() << "right_image_size," << right_image_size.x() << "," 
             << right_image_size.y() << "\n";

    std::string trans_left_image  = opt.out_prefix+"-L.tif";
    std::string trans_right_image = opt.out_prefix+"-R.tif";
    vw::vw_out() << "trans_left_image,"  << trans_left_image  << "\n";
    vw::vw_out() << "trans_right_image," << trans_right_image << "\n";
    
    // This is needed to create tiles in parallel_stereo
    vw::Vector2 trans_left_image_size(0, 0);
    if (fs::exists(trans_left_image))
      trans_left_image_size = file_image_size(trans_left_image);
    vw_out() << "trans_left_image_size," << trans_left_image_size.x() << "," 
             << trans_left_image_size.y() << "\n";

    // This separator needs to be in sync with parallel_stereo. It is needed to
    // extract from a line of text the portion after the separator.
    std::string sep = "--non-comma-separator--";
     
    cartography::GeoReference georef = opt.session->get_georef();
    vw_out() << "WKT" << sep << georef.get_wkt() << "\n";

    // Write the geotransform as a string as expected by GDAL's vrt xml format
    // TODO: Not sure if this will be useful as a member function in GeoReference.
    Matrix3x3 T = georef.transform();
    std::ostringstream os;
    os.precision(18);
    os << " "  << T(0, 2) << ",  " << T(0, 0) << ",  " << T(0, 1) << ",  "
       << " "  << T(1, 2) << ",  " << T(1, 0) << ",  " << T(1, 1);
    vw_out() << "GeoTransform" << sep << os.str() << "\n";

    // Some care is needed below. The transformed_window will be used
    // by parallel_stereo to parallelize stereo on a given user-specified
    // region. If both --left-image-crop-win and --right-image-crop-win
    // is specified, we already chopped the input images to these windows,
    // and created L.tif. Hence, we will work on the bd box on L.tif. However,
    // if just left-image-crop-win was set, we still use the full images,
    // but just the domain of computation is restricted. Hence we take user's
    // crop window, transform it to be in the L.tif coordinates, and use that one.
    bool crop_left = (stereo_settings().left_image_crop_win  != BBox2i(0, 0, 0, 0));
    BBox2i transformed_window;
    if (crop_left) {
      transformed_window.min() = Vector2(0, 0);
      transformed_window.max() = trans_left_image_size;
    }else{
      transformed_window = transformed_crop_win(opt);
    }
    vw_out() << "transformed_window," << transformed_window.min().x() << ","
             << transformed_window.min().y() << ","
             << transformed_window.width()   << ","
             << transformed_window.height()  << "\n";

    //vw_out() << "corr_tile_size," << ASPGlobalOptions::corr_tile_size() << "\n";
    vw_out() << "corr_tile_size," << stereo_settings().corr_tile_size_ovr << "\n";
    vw_out() << "rfne_tile_size," << ASPGlobalOptions::rfne_tile_size() << "\n";
    vw_out() << "tri_tile_size,"  << ASPGlobalOptions::tri_tile_size()  << "\n";
    vw_out() << "stereo_algorithm," << stereo_settings().stereo_algorithm << "\n";
    vw_out() << "alignment_method,"  << stereo_settings().alignment_method << "\n";
    vw_out() << "subpixel_mode," << stereo_settings().subpixel_mode << "\n";

    // Apart from default block matching, correlation will be done tiles
    // with padding, which is called here collar_size.
    vw::stereo::CorrelationAlgorithm stereo_alg
      = asp::stereo_alg_to_num(stereo_settings().stereo_algorithm);
    bool using_tiles = (stereo_alg > vw::stereo::VW_CORRELATION_BM ||
                        stereo_settings().alignment_method == "local_epipolar");
    
    int sgm_collar_size = 0;
    if (using_tiles)
      sgm_collar_size = stereo_settings().sgm_collar_size;
    vw_out() << "collar_size," << sgm_collar_size << "\n";

    vw_out() << "corr_memory_limit_mb," << stereo_settings().corr_memory_limit_mb << "\n";
    vw_out() << "save_lr_disp_diff," << stereo_settings().save_lr_disp_diff << "\n";
    vw_out() << "correlator_mode," << stereo_settings().correlator_mode << "\n";

    if (asp::stereo_settings().parallel_tile_size != vw::Vector2i(0, 0))
      produceTiles(opt.session->isMapProjected(), output_prefix, trans_left_image_size,
                   asp::stereo_settings().parallel_tile_size, sgm_collar_size);
    
    // Attach a georeference to this disparity. 
    // TODO(oalexan1): Make this into a function
    std::string left_image_file = opt.out_prefix + "-L.tif";
    if (stereo_settings().attach_georeference_to_lowres_disparity &&
        fs::exists(left_image_file)) {

      cartography::GeoReference left_georef, left_sub_georef;
      bool   has_left_georef = read_georeference(left_georef, left_image_file);
      bool   has_nodata      = false;
      double output_nodata   = -32768.0;
      if (has_left_georef) {

        DiskImageView<float> left_image(left_image_file);
        for (int i = 0; i < 2; i++) {
          std::string d_sub_file = opt.out_prefix + "-D_sub.tif";
          if (i == 1) d_sub_file = opt.out_prefix + "-D_sub_spread.tif";
          if (!fs::exists(d_sub_file)) 
            continue;

          bool has_sub_georef = read_georeference(left_sub_georef, d_sub_file);
          if (has_sub_georef) {
            // If D_sub already has a georef, as with seed-mode 3, don't overwrite it.
            continue;
          }

          vw::ImageView<PixelMask<Vector2f>> d_sub;
          vw::read_image(d_sub, d_sub_file);
          // Account for scale.
          double left_scale = 0.5*(double(d_sub.cols())/left_image.cols() +
                                   double(d_sub.rows())/left_image.rows());
          left_sub_georef = resample(left_georef, left_scale);
          vw::cartography::block_write_gdal_image(d_sub_file, d_sub,
                                      has_left_georef, left_sub_georef,
                                      has_nodata, output_nodata,
                                      opt, TerminalProgressCallback("asp", "\t    D_sub: "));
        } // End i loop
      } // End has_left_georef
    } // End georef attach 

    xercesc::XMLPlatformUtils::Terminate();
  } ASP_STANDARD_CATCHES;

  return 0;
}
