// __BEGIN_LICENSE__
//  Copyright (c) 2009-2026, United States Government as represented by the
//  Administrator of the National Aeronautics and Space Administration. All
//  rights reserved.
//
//  The NGT platform is licensed under the Apache License, Version 2.0 (the
//  "License"); you may not use this file except in compliance with the
//  License. You may obtain a copy of the License at
//  http://www.apache.org/licenses/LICENSE-2.0
//
//  Unless required by applicable law or agreed to in writing, software
//  distributed under the License is distributed on an "AS IS" BASIS,
//  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//  See the License for the specific language governing permissions and
//  limitations under the License.
// __END_LICENSE__

/// \file opencv_calibrate.cpp
///
/// Camera calibration tool using OpenCV's chessboard/circle grid pattern detection.
/// Computes intrinsic camera parameters and distortion coefficients from a set of
/// calibration images.

#include <vw/Image/ImageView.h>
#include <vw/Image/ImageIO.h>
#include <vw/FileIO/DiskImageResource.h>
#include <vw/Core/Log.h>

#include <opencv2/core/core.hpp>
#include <opencv2/imgproc/imgproc.hpp>
#include <opencv2/calib3d/calib3d.hpp>

#include <cctype>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <iostream>

const char * usage =
" \nexample command line for calibration from a list of stored images:\n"
"   imagelist_creator image_list.xml *.png\n"
"   calibration -w 4 -h 5 -s 0.025 -o camera.yml -op -oe image_list.xml\n"
" where image_list.xml is the standard OpenCV XML/YAML\n"
" use imagelist_creator to create the xml or yaml list\n"
" file consisting of the list of strings, e.g.:\n"
" \n"
"<?xml version=\"1.0\"?>\n"
"<opencv_storage>\n"
"<images>\n"
"view000.png\n"
"view001.png\n"
"<!-- view002.png -->\n"
"view003.png\n"
"view010.png\n"
"one_extra_view.jpg\n"
"</images>\n"
"</opencv_storage>\n";


static void help() {
  vw::vw_out() << "This is a camera calibration sample.\n"
    "Usage: calibration\n"
    "     -w <board_width>         # the number of inner corners per one of\n"
    "                              #   board dimension\n"
    "     -h <board_height>        # the number of inner corners per another\n"
    "                              #   board dimension\n"
    "     [-pt <pattern>]          # the type of pattern: chessboard or\n"
    "                              #   circles' grid\n"
    "     [-n <number_of_frames>]  # the number of frames to use for\n"
    "                              #   calibration (if not specified, it\n"
    "                              #   will be set to the number of board\n"
    "                              #   views actually available)\n"
    "     [-s <squareSize>]       # square size in some user-defined units\n"
    "                              #   (1 by default)\n"
    "     [-o <out_camera_params>] # the output filename for intrinsic\n"
    "                              #   [and extrinsic] parameters\n"
    "     [-op]                    # write detected feature points\n"
    "     [-oe]                    # write extrinsic parameters\n"
    "     [-zt]                    # assume zero tangential distortion\n"
    "     [-a <aspectRatio>]      # fix aspect ratio (fx/fy)\n"
    "     [-p]                     # fix the principal point at the center\n"
    "     [-v]                     # flip the captured images around the\n"
    "                              #   horizontal axis\n"
    "     [input_data]             # input data, one of the following:\n"
    "                              #  - text file with a list of the images\n"
    "                              #    of the board\n"
    "                              #    the text file can be generated with\n"
    "                              #    imagelist_creator\n"
    "\n";
  vw::vw_out() << "\n" << usage;
}

enum { DETECTION = 0, CAPTURING = 1, CALIBRATED = 2 };
enum BoardPattern { CHESSBOARD, CIRCLES_GRID, ASYMMETRIC_CIRCLES_GRID };

static double computeReprojectionErrors(
    const std::vector<std::vector<cv::Point3f>>& objectPoints,
    const std::vector<std::vector<cv::Point2f>>& imagePoints,
    const std::vector<cv::Mat>& rvecs, const std::vector<cv::Mat>& tvecs,
    const cv::Mat& cameraMatrix, const cv::Mat& distCoeffs,
    std::vector<float>& perViewErrors) {

  std::vector<cv::Point2f> imagePoints2;
  int i, totalPoints = 0;
  double totalErr = 0, err;
  perViewErrors.resize(objectPoints.size());

  for (i = 0; i < (int)objectPoints.size(); i++) {
    cv::projectPoints(cv::Mat(objectPoints[i]), rvecs[i], tvecs[i],
                      cameraMatrix, distCoeffs, imagePoints2);
    err = cv::norm(cv::Mat(imagePoints[i]), cv::Mat(imagePoints2), cv::NORM_L2);
    int n = (int)objectPoints[i].size();
    perViewErrors[i] = (float)std::sqrt(err*err/n);
    totalErr += err*err;
    totalPoints += n;
  }

  return std::sqrt(totalErr/totalPoints);
}

static void calcChessboardCorners(cv::Size boardSize, float squareSize,
                                  std::vector<cv::Point3f>& corners,
                                  BoardPattern patternType = CHESSBOARD) {
  corners.resize(0);

  switch (patternType) {
    case CHESSBOARD:
    case CIRCLES_GRID:
      for (int i = 0; i < boardSize.height; i++)
        for (int j = 0; j < boardSize.width; j++)
          corners.push_back(cv::Point3f(float(j*squareSize),
                                        float(i*squareSize), 0));
      break;

    case ASYMMETRIC_CIRCLES_GRID:
      for (int i = 0; i < boardSize.height; i++)
        for (int j = 0; j < boardSize.width; j++)
          corners.push_back(cv::Point3f(float((2*j + i % 2)*squareSize),
                                        float(i*squareSize), 0));
      break;

    default:
      CV_Error(cv::Error::StsBadArg, "Unknown pattern type\n");
  }
}

bool runCalibration(std::vector<std::vector<cv::Point2f>> imagePoints,
                    cv::Size imageSize, cv::Size boardSize, BoardPattern patternType,
                    float squareSize, float aspectRatio,
                    int flags, cv::Mat& cameraMatrix, cv::Mat& distCoeffs,
                    std::vector<cv::Mat>& rvecs, std::vector<cv::Mat>& tvecs,
                    std::vector<float>& reprojErrs,
                    double& totalAvgErr) {
  cameraMatrix = cv::Mat::eye(3, 3, CV_64F);
  if (flags & cv::CALIB_FIX_ASPECT_RATIO)
    cameraMatrix.at<double>(0,0) = aspectRatio;

  const int NUM_DISTORTION_COEFFS = 4;
  distCoeffs = cv::Mat::zeros(NUM_DISTORTION_COEFFS, 1, CV_64F);

  std::vector<std::vector<cv::Point3f>> objectPoints(1);
  calcChessboardCorners(boardSize, squareSize, objectPoints[0], patternType);

  objectPoints.resize(imagePoints.size(), objectPoints[0]);

  double rms = cv::calibrateCamera(objectPoints, imagePoints, imageSize, cameraMatrix,
                                    distCoeffs, rvecs, tvecs, flags|cv::CALIB_FIX_K3);
  vw::vw_out() << "RMS error reported by calibrateCamera: " << rms << "\n";

  bool ok = cv::checkRange(cameraMatrix) && cv::checkRange(distCoeffs);

  totalAvgErr = computeReprojectionErrors(objectPoints, imagePoints, rvecs, tvecs,
                                          cameraMatrix, distCoeffs, reprojErrs);

  return ok;
}

static void saveCameraParams(const std::string& filename,
                             cv::Size imageSize, cv::Size boardSize,
                             float squareSize, float aspectRatio, int flags,
                             const cv::Mat& cameraMatrix,
                             const cv::Mat& distCoeffs,
                             const std::vector<cv::Mat>& rvecs,
                             const std::vector<cv::Mat>& tvecs,
                             const std::vector<float>& reprojErrs,
                             const std::vector<std::vector<cv::Point2f>>& imagePoints,
                             double totalAvgErr) {
  cv::FileStorage fs(filename, cv::FileStorage::WRITE);

  time_t tt;
  time(&tt);
  struct tm *t2 = localtime(&tt);
  char buf[1024];
  strftime(buf, sizeof(buf)-1, "%c", t2);

  fs << "calibration_time" << buf;

  if (!rvecs.empty() || !reprojErrs.empty())
    fs << "nframes" << (int)std::max(rvecs.size(), reprojErrs.size());
  fs << "image_width" << imageSize.width;
  fs << "image_height" << imageSize.height;
  fs << "board_width" << boardSize.width;
  fs << "board_height" << boardSize.height;
  fs << "square_size" << squareSize;

  if (flags & cv::CALIB_FIX_ASPECT_RATIO)
    fs << "aspectRatio" << aspectRatio;

  if (flags != 0) {
    sprintf(buf, "flags: %s%s%s%s",
            flags & cv::CALIB_USE_INTRINSIC_GUESS ? "+use_intrinsic_guess" : "",
            flags & cv::CALIB_FIX_ASPECT_RATIO ? "+fix_aspectRatio" : "",
            flags & cv::CALIB_FIX_PRINCIPAL_POINT ? "+fix_principal_point" : "",
            flags & cv::CALIB_ZERO_TANGENT_DIST ? "+zero_tangent_dist" : "");
  }

  fs << "flags" << flags;

  fs << "camera_matrix" << cameraMatrix;
  fs << "distortion_coefficients" << distCoeffs;

  fs << "avg_reprojection_error" << totalAvgErr;
  if (!reprojErrs.empty())
    fs << "per_view_reprojection_errors" << cv::Mat(reprojErrs);

  if (!rvecs.empty() && !tvecs.empty()) {
    CV_Assert(rvecs[0].type() == tvecs[0].type());
    cv::Mat bigmat((int)rvecs.size(), 6, rvecs[0].type());
    for (int i = 0; i < (int)rvecs.size(); i++) {
      cv::Mat r = bigmat(cv::Range(i, i+1), cv::Range(0,3));
      cv::Mat t = bigmat(cv::Range(i, i+1), cv::Range(3,6));

      CV_Assert(rvecs[i].rows == 3 && rvecs[i].cols == 1);
      CV_Assert(tvecs[i].rows == 3 && tvecs[i].cols == 1);
      r = rvecs[i].t();
      t = tvecs[i].t();
    }
    fs << "extrinsic_parameters" << bigmat;
  }

  if (!imagePoints.empty()) {
    cv::Mat imagePtMat((int)imagePoints.size(), (int)imagePoints[0].size(), CV_32FC2);
    for (int i = 0; i < (int)imagePoints.size(); i++) {
      cv::Mat r = imagePtMat.row(i).reshape(2, imagePtMat.cols);
      cv::Mat imgpti(imagePoints[i]);
      imgpti.copyTo(r);
    }
    fs << "image_points" << imagePtMat;
  }
}

static bool readStringList(const std::string& filename, const std::string& tempFile,
                           std::vector<std::string>& l) {
  l.resize(0);
  cv::FileStorage fs(filename, cv::FileStorage::READ);
  if (!fs.isOpened())
    return false;
  cv::FileNode n = fs.getFirstTopLevelNode();
  if (n.type() != cv::FileNode::SEQ)
    return false;
  cv::FileNodeIterator it = n.begin(), it_end = n.end();
  for (; it != it_end; ++it)
    l.push_back((std::string)*it);

  return true;
}

bool runAndSave(const std::string& outputFilename,
                const std::vector<std::vector<cv::Point2f>>& imagePoints,
                cv::Size imageSize, cv::Size boardSize, BoardPattern patternType,
                float squareSize, float aspectRatio, int flags, cv::Mat& cameraMatrix,
                cv::Mat& distCoeffs, bool writeExtrinsics, bool writePoints) {
  std::vector<cv::Mat> rvecs, tvecs;
  std::vector<float> reprojErrs;
  double totalAvgErr = 0;

  bool ok = runCalibration(imagePoints, imageSize, boardSize, patternType, squareSize,
                           aspectRatio, flags, cameraMatrix, distCoeffs,
                           rvecs, tvecs, reprojErrs, totalAvgErr);
  vw::vw_out() << (ok ? "Calibration succeeded" : "Calibration failed")
               << ". avg reprojection error = " << totalAvgErr << "\n";

  if (ok)
    saveCameraParams(outputFilename, imageSize,
                     boardSize, squareSize, aspectRatio,
                     flags, cameraMatrix, distCoeffs,
                     writeExtrinsics ? rvecs : std::vector<cv::Mat>(),
                     writeExtrinsics ? tvecs : std::vector<cv::Mat>(),
                     writeExtrinsics ? reprojErrs : std::vector<float>(),
                     writePoints ? imagePoints : std::vector<std::vector<cv::Point2f>>(),
                     totalAvgErr);
  return ok;
}

cv::Mat vw_imread(const std::string fileName,
                  vw::ImageView<vw::uint8> &gray_buffer) {
  vw::read_image(gray_buffer, fileName);

  // Figure out the image buffer parameters
  void*   raw_data_ptr = reinterpret_cast<void*>(gray_buffer.data());
  size_t  pixel_size   = sizeof(vw::uint8);
  size_t  step_size    = gray_buffer.cols() * pixel_size;

  // Create an OpenCV wrapper for the buffer image
  cv::Mat cv_image(gray_buffer.rows(), gray_buffer.cols(), CV_8UC1,
                   raw_data_ptr, step_size);
  return cv_image;
}

int main(int argc, char** argv) {
  cv::Size boardSize, imageSize;
  float squareSize = 1.f, aspectRatio = 1.f;
  cv::Mat cameraMatrix, distCoeffs;
  const char* outputFilename = "out_camera_data.yml";
  const char* inputFilename = 0;

  int i, nframes = 10;
  bool writeExtrinsics = false, writePoints = false;
  int flags = 0;
  int mode = DETECTION;
  std::vector<std::vector<cv::Point2f>> imagePoints;
  std::vector<std::string> imageList;
  BoardPattern pattern = CHESSBOARD;

  if (argc < 2) {
    help();
    return 0;
  }

  for (i = 1; i < argc; i++) {
    const char* s = argv[i];
    if (strcmp(s, "-w") == 0) {
      if (sscanf(argv[++i], "%u", &boardSize.width) != 1 || boardSize.width <= 0) {
        vw::vw_out() << "Invalid board width\n";
        return -1;
      }
    } else if (strcmp(s, "-h") == 0) {
      if (sscanf(argv[++i], "%u", &boardSize.height) != 1 || boardSize.height <= 0) {
        vw::vw_out() << "Invalid board height\n";
        return -1;
      }
    } else if (strcmp(s, "-pt") == 0) {
      i++;
      if (!strcmp(argv[i], "circles"))
        pattern = CIRCLES_GRID;
      else if (!strcmp(argv[i], "acircles"))
        pattern = ASYMMETRIC_CIRCLES_GRID;
      else if (!strcmp(argv[i], "chessboard"))
        pattern = CHESSBOARD;
      else {
        vw::vw_out() << "Invalid pattern type: must be chessboard or circles\n";
        return -1;
      }
    } else if (strcmp(s, "-s") == 0) {
      if (sscanf(argv[++i], "%f", &squareSize) != 1 || squareSize <= 0) {
        vw::vw_out() << "Invalid board square width\n";
        return -1;
      }
    } else if (strcmp(s, "-n") == 0) {
      if (sscanf(argv[++i], "%u", &nframes) != 1 || nframes <= 3) {
        vw::vw_out() << "Invalid number of images\n";
        return -1;
      }
    } else if (strcmp(s, "-a") == 0) {
      if (sscanf(argv[++i], "%f", &aspectRatio) != 1 || aspectRatio <= 0) {
        vw::vw_out() << "Invalid aspect ratio\n";
        return -1;
      }
      flags |= cv::CALIB_FIX_ASPECT_RATIO;
    } else if (strcmp(s, "-op") == 0) {
      writePoints = true;
    } else if (strcmp(s, "-oe") == 0) {
      writeExtrinsics = true;
    } else if (strcmp(s, "-zt") == 0) {
      flags |= cv::CALIB_ZERO_TANGENT_DIST;
    } else if (strcmp(s, "-p") == 0) {
      flags |= cv::CALIB_FIX_PRINCIPAL_POINT;
    } else if (strcmp(s, "-o") == 0) {
      outputFilename = argv[++i];
    } else if (s[0] != '-') {
      inputFilename = s;
    } else {
      vw::vw_out() << "Unknown option " << s << "\n";
      return -1;
    }
  }

  std::string tempFile = outputFilename;
  tempFile += ".temp";

  if (inputFilename) {
    if (readStringList(inputFilename, tempFile, imageList))
      mode = CAPTURING;
    else {
      vw::vw_out() << "Error openening file " << inputFilename << "\n";
      return -1;
    }
  } else {
    vw::vw_out() << "Missing input filename.\n";
    return -1;
  }

  if (!imageList.empty())
    nframes = (int)imageList.size();

  for (i = 0; ; i++) {
    cv::Mat view;
    vw::ImageView<vw::uint8> gray_buffer;

    if (i < (int)imageList.size()) {
      view = vw_imread(imageList[i], gray_buffer);
      if (!view.data) {
        vw::vw_out() << "Failed to read image data from input file "
                     << imageList[i] << "\n";
        return -1;
      }
    }

    if (!view.data) {
      if (imagePoints.size() > 0)
        runAndSave(outputFilename, imagePoints, imageSize,
                   boardSize, pattern, squareSize, aspectRatio,
                   flags, cameraMatrix, distCoeffs,
                   writeExtrinsics, writePoints);
      break;
    }

    imageSize = view.size();

    std::vector<cv::Point2f> pointbuf;
    cv::Mat viewGray;
    if (view.channels() > 1)
      cv::cvtColor(view, viewGray, cv::COLOR_BGR2GRAY);
    else
      view.copyTo(viewGray);

    bool found;
    switch (pattern) {
      case CHESSBOARD:
        found = cv::findChessboardCorners(view, boardSize, pointbuf,
                                          cv::CALIB_CB_ADAPTIVE_THRESH |
                                          cv::CALIB_CB_FAST_CHECK |
                                          cv::CALIB_CB_NORMALIZE_IMAGE);
        break;
      case CIRCLES_GRID:
        found = cv::findCirclesGrid(view, boardSize, pointbuf);
        break;
      case ASYMMETRIC_CIRCLES_GRID:
        found = cv::findCirclesGrid(view, boardSize, pointbuf,
                                    cv::CALIB_CB_ASYMMETRIC_GRID);
        break;
      default:
        vw::vw_out() << "Unknown pattern type\n";
        return -1;
    }

    int winSize = 11;
    if (pattern == CHESSBOARD && found)
      cv::cornerSubPix(viewGray, pointbuf, cv::Size(winSize, winSize),
                       cv::Size(-1, -1),
                       cv::TermCriteria(cv::TermCriteria::EPS +
                                        cv::TermCriteria::COUNT, 30, 0.0001));

    if (mode == CAPTURING && found) {
      imagePoints.push_back(pointbuf);
    }

    if (found)
      cv::drawChessboardCorners(view, boardSize, cv::Mat(pointbuf), found);

    std::string msg = mode == CAPTURING ? "100/100" :
        mode == CALIBRATED ? "Calibrated" : "Press 'g' to start";
    int baseLine = 0;
    cv::Size textSize = cv::getTextSize(msg, 1, 1, 1, &baseLine);
    cv::Point textOrigin(view.cols - 2*textSize.width - 10, view.rows - 2*baseLine - 10);

    if (mode == CAPTURING) {
      msg = cv::format("%d/%d", (int)imagePoints.size(), nframes);
    }

    cv::putText(view, msg, textOrigin, 1, 1,
                mode != CALIBRATED ? cv::Scalar(0, 0, 255) : cv::Scalar(0, 255, 0));

    if (mode == CAPTURING && imagePoints.size() >= (unsigned)nframes) {
      if (runAndSave(outputFilename, imagePoints, imageSize,
                     boardSize, pattern, squareSize, aspectRatio,
                     flags, cameraMatrix, distCoeffs,
                     writeExtrinsics, writePoints))
        mode = CALIBRATED;
      else
        mode = DETECTION;
      break;
    }
  }

  return 0;
}
