// __BEGIN_LICENSE__
//  Copyright (c) 2009-2025, United States Government as represented by the
//  Administrator of the National Aeronautics and Space Administration. All
//  rights reserved.
//
//  The NGT platform is licensed under the Apache License, Version 2.0 (the
//  "License"); you may not use this file except in compliance with the
//  License. You may obtain a copy of the License at
//  http://www.apache.org/licenses/LICENSE-2.0
//
//  Unless required by applicable law or agreed to in writing, software
//  distributed under the License is distributed on an "AS IS" BASIS,
//  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//  See the License for the specific language governing permissions and
//  limitations under the License.
// __END_LICENSE__

// \file SfsModel.h
// Modeling reflectance, intensity, albedo for SfS

#ifndef __ASP_SFS_SFS_MODEL_H__
#define __ASP_SFS_SFS_MODEL_H__

#include <asp/SfS/SfsImageProc.h>

#include <vw/Camera/CameraModel.h>
#include <vw/Image/MaskViews.h>
#include <vw/Image/ImageViewRef.h>
#include <vw/Math/Vector.h>

namespace vw { namespace cartography {
  class GeoReference;
}} // namespace vw::cartography

namespace asp {

// Maximum number of reflectance model and haze coefficients
const size_t g_num_model_coeffs = 16;
const size_t g_max_num_haze_coeffs = 6;

enum REFL_TYPE {NO_REFL = 0, LAMBERT, LUNAR_LAMBERT, HAPKE, ARBITRARY_MODEL, CHARON};

class HeightErrEstim;
class SfsOptions;

struct ReflParams {
  int reflectanceType;
  // Two parameters used in the formula for the Lunar-Lambertian
  // reflectance
  double phaseCoeffC1, phaseCoeffC2;
};

// Reflectance formula that is nonlinear if there is more than one haze coefficient
// (that is experimental).
double nonlinReflectance(double reflectance, double exposure,
                         double steepness_factor,
                         double const* haze, int num_haze_coeffs);

// Computes the ground reflectance with a desired reflectance model.
double calcReflectance(vw::Vector3 const& cameraPosition, vw::Vector3 const& normal,
                       vw::Vector3 const& xyz, vw::Vector3 const& sun_position,
                       ReflParams const& refl_params,
                       const double * refl_coeffs);

// Computed intensity:
// albedo * nonlinReflectance(reflectance_i, exposures[i], haze, num_haze_coeffs) + haze[0]
// Cost function:
// sum_i | I_i - sim_intensity_i|^2
double calcSimIntensity(double albedo, double reflectance, double exposure,
                     double steepness_factor, double const* haze, int num_haze_coeffs);

// TODO(oalexan1): Should one mark the no-data values rather than setting
// them to 0?
void calcSimIntensity(vw::ImageView<double> const& albedo,
                      MaskedDblImgT const& reflectance,
                      double exposure,
                      double steepness_factor,
                      std::vector<double> const& haze,
                      int num_haze_coeffs,
                      int num_threads,
                      bool show_progress,
                      MaskedDblImgT & sim_intensity);

// Calc albedo given the intensity. See calcSimIntensity().
double calcAlbedo(double intensity, double reflectance, double exposure,
                  double steepness_factor, double const* haze, int num_haze_coeffs);

// Calculate current ECEF position and normal vector for a given DEM pixel.
// This is an auxiliary function needed to compute the reflectance.
void calcPointAndNormal(int col, int row,
                        double left_h, double center_h, double right_h,
                        double bottom_h, double top_h,
                        bool use_pq, double p, double q, // dem partial derivatives
                        vw::cartography::GeoReference const& geo,
                        double gridx, double gridy,
                        // Outputs
                        vw::Vector3 & xyz, vw::Vector3 & normal);

// Compute the reflectance and intensity at a single pixel. Compute the slope and/or
// height error estimation if the pointer is not NULL.
bool calcPixReflectanceInten(double left_h, double center_h, double right_h,
                             double bottom_h, double top_h,
                             bool use_pq, double p, double q, // dem partial derivatives
                             int col, int row,
                             DblImgT         const& dem,
                             vw::cartography::GeoReference const& geo,
                             bool model_shadows,
                             double max_dem_height,
                             double gridx, double gridy,
                             vw::Vector3       const & sunPosition,
                             asp::ReflParams   const & refl_params,
                             vw::BBox2i        const & crop_box,
                             MaskedImgRefT     const & image,
                             DblImgT        const & blend_weight,
                             bool blend_weight_is_ground_weight,
                             vw::CamPtr camera,
                             vw::PixelMask<double>   & reflectance,
                             vw::PixelMask<double>   & intensity,
                             double                  & ground_weight,
                             double            const * refl_coeffs,
                             asp::SfsOptions   const & opt,
                             asp::HeightErrEstim     * heightErrEstim = NULL);

// The value stored in the output intensity(i, j) is the one at entry
// (i - 1) * sample_col_rate + 1, (j - 1) * sample_row_rate + 1
// in the full image. For i = 0 or j = 0 invalid values are stored.
void computeReflectanceAndIntensity(DblImgT const& dem,
                                    vw::ImageView<vw::Vector2> const& pq,
                                    vw::cartography::GeoReference const& geo,
                                    bool model_shadows,
                                    bool show_progress,
                                    double & max_dem_height, // alias
                                    double gridx, double gridy,
                                    int sample_col_rate, int sample_row_rate,
                                    vw::Vector3 const& sunPosition,
                                    asp::ReflParams const& refl_params,
                                    vw::BBox2i const& crop_box,
                                    asp::MaskedImgRefT const  & image,
                                    asp::DblImgT const  & blend_weight,
                                    bool blend_weight_is_ground_weight,
                                    vw::CamPtr camera,
                                    MaskedDblImgT & reflectance,
                                    MaskedDblImgT & intensity,
                                    DblImgT & ground_weight,
                                    double const * refl_coeffs,
                                    asp::SfsOptions const & opt,
                                    asp::HeightErrEstim * heightErrEstim = NULL);

// Initalize the reflectance parameters based on user input
void setupReflectance(asp::ReflParams & refl_params, asp::SfsOptions & opt);

// This function computes / saves measured and simulated intensities, and
// handles related estimations. If opt.allow_borderline_data is true, create for
// each image that will not be skipped a weight matrix with dimensions equal to
// DEM dimensions, that will be used instead of weights in the camera image
// space. These are balanced among each other and give more weight to barely lit
// and unlit nearby pixels.
void calcIntenEstimHeights(SfsOptions & opt,
                           vw::ImageView<double> const& dem,
                           vw::ImageView<double> const& albedo,
                           vw::cartography::GeoReference const& geo,
                           bool show_progress,
                           double max_dem_height,
                           double gridx, double gridy,
                           std::vector<vw::Vector3> const& sunPosition,
                           asp::ReflParams const& refl_params,
                           std::vector<vw::BBox2i> const& crop_boxes,
                           std::vector<MaskedImgRefT> const& masked_images,
                           std::vector<vw::ImageView<double>> const& blend_weights,
                           bool blend_weight_is_ground_weight,
                           std::vector<vw::CamPtr> const& cameras,
                           float img_nodata_val,
                           // Outputs
                           vw::ImageView<int> & lit_image_mask,
                           std::vector<vw::ImageView<double>> & ground_weights,
                           std::vector<MaskedDblImgT> & meas_intensities,
                           std::vector<MaskedDblImgT> & sim_intensities);

} // end namespace asp

#endif // __ASP_SFS_SFS_MODEL_H__
