// __BEGIN_LICENSE__
//  Copyright (c) 2009-2026, United States Government as represented by the
//  Administrator of the National Aeronautics and Space Administration. All
//  rights reserved.
//
//  The NGT platform is licensed under the Apache License, Version 2.0 (the
//  "License"); you may not use this file except in compliance with the
//  License. You may obtain a copy of the License at
//  http://www.apache.org/licenses/LICENSE-2.0
//
//  Unless required by applicable law or agreed to in writing, software
//  distributed under the License is distributed on an "AS IS" BASIS,
//  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//  See the License for the specific language governing permissions and
//  limitations under the License.
// __END_LICENSE__

#ifndef __ASP_CAMERA_BUNDLE_ADJUST_OPTIONS_H__
#define __ASP_CAMERA_BUNDLE_ADJUST_OPTIONS_H__

#include <asp/Camera/BundleAdjustCamera.h>

#include <vw/BundleAdjustment/ControlNetwork.h>
#include <vw/Cartography/Datum.h>

// Functions that need the BaOptions class. 

namespace asp {
  
struct BaOptions: public asp::BaBaseOptions {
  std::string cnet_file,
    cost_function, mapprojected_data, gcp_from_mapprojected;
  int ip_per_tile, ip_per_image, matches_per_tile;
  double overlap_exponent, ip_triangulation_max_error;
  int num_parallel_jobs, job_id, num_random_passes, ip_num_ransac_iterations;
  bool save_intermediate_cameras, approximate_pinhole_intrinsics,
    init_camera_using_gcp, disable_pinhole_gcp_init,
    transform_cameras_with_shared_gcp, transform_cameras_using_gcp,
    solve_intrinsics, ip_normalize_tiles, ip_debug_images, stop_after_stats, 
    calc_normalization_bounds, calc_ip, stop_after_matching,
    skip_matching, apply_initial_transform_only, save_vwip;
  std::string camera_position_file, initial_transform_file, dem_file_for_overlap;
  double semi_major, semi_minor, position_filter_dist, max_gcp_reproj_err;
  std::string remove_outliers_params_str;
  std::vector<double> intrinsics_limits;
  boost::shared_ptr<vw::ba::ControlNetwork> cnet;
  int ip_detect_method, num_scales;
  double epipolar_threshold; // Max distance from epipolar line to search for IP matches.
  double ip_inlier_factor, ip_uniqueness_thresh, nodata_value, 
    auto_overlap_buffer, pct_for_overlap, min_distortion, ip_nodata_radius;
  bool skip_rough_homography, enable_rough_homography, disable_tri_filtering,
    enable_tri_filtering, no_datum, individually_normalize,
    force_reuse_match_files, no_poses_from_nvm, save_cnet_as_csv, aster_use_csm,
    query_num_image_pairs;
  vw::Vector2 elevation_limit;   // Expected range of elevation to limit results to.
  vw::BBox2 lon_lat_limit;       // Limit the triangulated interest points to this lonlat range
  vw::Matrix<double> initial_transform;
  std::string fixed_cameras_indices_str, flann_method, fixed_distortion_indices_str;
  std::set<int> fixed_cameras_indices;
  std::vector<int> fixed_distortion_indices;
  asp::IntrinsicOptions intrinsics_options;
  vw::Vector2i matches_per_tile_params;
  
  // Make sure all values are initialized, even though they will be
  // over-written later.
  BaOptions(): ip_per_tile(0), ip_per_image(0), 
             overlap_exponent(0), 
             save_intermediate_cameras(false),
             solve_intrinsics(false), 
             semi_major(0), semi_minor(0), position_filter_dist(-1),
             max_gcp_reproj_err(-1.0),
             ip_detect_method(0), num_scales(-1), 
             pct_for_overlap(-1), skip_rough_homography(false),
             individually_normalize(false), 
             force_reuse_match_files(false), no_poses_from_nvm(false),
             save_cnet_as_csv(false), aster_use_csm(false), query_num_image_pairs(false) {}

  /// Bundle adjustment settings that must be passed to the asp settings
  void copy_to_asp_settings() const;
  
}; // End class BaOptions

// Read the mapprojected data, if having the option --mapprojected-data.
void setupMapprojectedData(asp::BaOptions & opt,
                           bool need_no_matches,
                           std::vector<std::string> & map_files,
                           std::string & mapproj_dem);

/// Looks in the input camera position file to generate a GCC position for
/// each input camera.
/// - If no match is found, the coordinate is (0,0,0)
int loadEstimCameraPositions(asp::BaOptions &opt,
                             std::vector<vw::Vector3> & estimated_camera_gcc);

// Compute statistics for the designated range of images (or mapprojected
// images), and perhaps the footprints. Or, compute the ip per image (before
// matching). These distinct operations use much shared logic, so are put in the
// same function. In parallel_bundle_adjust this function is called separately
// for different ranges.
void computeStatsOrIp(asp::BaOptions const& opt, 
                      std::vector<std::string> const& files_for_stats,
                      std::string const& dem_file_for_overlap, 
                      std::string const& normalization_bounds_file, 
                      bool calcIp);

} // end namespace asp

#endif // __ASP_CAMERA_BUNDLE_ADJUST_OPTIONS_H__
