# LFM Canonical Derivations Reference

**Version**: 3.0 (v12.0 equations)
**Date**: February 10, 2026  
**Purpose**: Machine-verifiable derivations for the Lattice Field Medium framework

This document provides step-by-step derivations that can be verified with any symbolic math package (SymPy, Mathematica, Maple). Each derivation has numbered equations and explicit intermediate steps.

---

## Table of Contents

0. [Discrete Lattice Foundation](#0-discrete-lattice-foundation)
0.5. [Spinor Extension: GOV-01-S](#05-spinor-extension-gov-01-s)
1. [Complex Wave Decomposition](#1-complex-wave-decomposition)
2. [Current Conservation from GOV-01](#2-current-conservation-from-gov-01)
3. [Phase Interference and Coulomb Force](#3-phase-interference-and-coulomb-force)
4. [Noether Current Derivation](#4-noether-current-derivation)
5. [Why 1/r² Force Law Emerges](#5-why-1r-force-law-emerges)
6. [Charge Quantization](#6-charge-quantization)

---

## 0. Discrete Lattice Foundation

### CRITICAL: LFM is Fundamentally Discrete

The governing equations are often written in continuous PDE notation:

$$\frac{\partial^2 \Psi}{\partial t^2} = c^2 \nabla^2 \Psi - \chi^2 \Psi \quad \text{(GOV-01)}$$

$$\frac{\partial^2 \chi}{\partial t^2} = c^2 \nabla^2 \chi - \kappa\left(\sum_a |\Psi_a|^2 + \varepsilon_W \cdot j - E_0^2\right) + \lambda(-\chi)^3 \Theta(-\chi) \quad \text{(GOV-02)}$$

**This notation is SHORTHAND.** The fundamental equations are discrete lattice updates.

### GOV-02 Term Definitions

- $\kappa = 1/(4\chi_0 - 13) = 1/63$ = gravitational coupling (derived from χ₀)
- $\sum_a |\Psi_a|^2$ = total energy density (sources gravity, colorblind/phase-blind)
- $j = \sum_a \text{Im}(\Psi_a^* \nabla \Psi_a)$ = Noether current (momentum density, sources frame dragging)
- $\varepsilon_W = 2/(\chi_0+1) = 0.1$ = helicity coupling (derived from χ₀)
- $\lambda = \chi_0 - 9 = 10$ = floor stiffness (derived from χ₀)
- $\Theta(x)$ = Heaviside step function (1 if x > 0, else 0)
- Floor term only activates when χ < 0 (black hole interiors)

### The True Governing Equations

At each lattice point $i$ and time step $n$:

**GOV-01 (Discrete - FUNDAMENTAL):**
$$\boxed{\Psi_i^{n+1} = 2\Psi_i^n - \Psi_i^{n-1} + (\Delta t)^2 \left[ \frac{c^2}{(\Delta x)^2}\left(\Psi_{i-1}^n - 2\Psi_i^n + \Psi_{i+1}^n\right) - (\chi_i^n)^2 \Psi_i^n \right]}$$

**GOV-02 (Discrete - FUNDAMENTAL, COMPLETE):**
$$\boxed{\chi_i^{n+1} = 2\chi_i^n - \chi_i^{n-1} + (\Delta t)^2 \left[ \frac{c^2}{(\Delta x)^2}\left(\chi_{i-1}^n - 2\chi_i^n + \chi_{i+1}^n\right) - \kappa\left(|\Psi_i^n|^2 + \varepsilon_W j_i^n - E_0^2\right) + \lambda(-\chi_i^n)^3 \Theta(-\chi_i^n) \right]}$$

where $j_i^n = \text{Im}\left(\Psi_i^{n*} \cdot \frac{\Psi_{i+1}^n - \Psi_{i-1}^n}{2\Delta x}\right)$ is the discrete momentum density.

Where:
- $i$ = lattice point index (spatial)
- $n$ = time step index (temporal)
- $\Delta x$ = lattice spacing (Planck length scale)
- $\Delta t$ = time step (Planck time scale)
- $\Psi_i^n \in \mathbb{C}$ = matter wave field at point $i$, time $n$
- $\chi_i^n \in \mathbb{R}$ = field stiffness at point $i$, time $n$

### Why This Matters

1. **Each point updates using ONLY neighbors** — No global coordination required
2. **Causality is built in** — The Courant condition $c \cdot \Delta t \leq \Delta x$ ensures $c$ is maximum speed
3. **No infinities** — Discrete lattice has no UV divergence
4. **Purely local physics** — Point $i$ knows nothing about distant points

### The Continuum Limit

The PDE form is recovered when $\Delta x, \Delta t \to 0$:

$$\frac{\Psi_i^{n+1} - 2\Psi_i^n + \Psi_i^{n-1}}{(\Delta t)^2} \xrightarrow{\Delta t \to 0} \frac{\partial^2 \Psi}{\partial t^2}$$

$$\frac{\Psi_{i+1}^n - 2\Psi_i^n + \Psi_{i-1}^n}{(\Delta x)^2} \xrightarrow{\Delta x \to 0} \frac{\partial^2 \Psi}{\partial x^2}$$

The continuum equations are valid approximations at scales $\gg \Delta x$, which is why they match observed physics.

### No Coordinator Needed

A common question: "What runs the loop? What makes time advance?"

**Answer:** Nothing external. The update rule is a constraint that states satisfy. Time index $n$ is not "advanced by" something — it IS the sequence of states. This is no different from asking "what makes 3 come after 2?"

The discrete form makes clear: **each point contains its state and updates locally.** Physics IS this computation. There is no meta-level running it.

### 3D Extension

In three dimensions, the discrete Laplacian becomes:
$$\mathcal{L}[\Psi]_{i,j,k} = \frac{1}{(\Delta x)^2}\left(\Psi_{i+1,j,k} + \Psi_{i-1,j,k} + \Psi_{i,j+1,k} + \Psi_{i,j-1,k} + \Psi_{i,j,k+1} + \Psi_{i,j,k-1} - 6\Psi_{i,j,k}\right)$$

The update rules remain purely local — each point uses only its 6 nearest neighbors.

---

## 0.5 Spinor Extension: GOV-01-S

### Why Spinors?

The Klein-Gordon equation (GOV-01-K) describes spin-0 particles (bosons). For spin-1/2 particles (fermions: electrons, quarks), we need the Dirac equation.

**Key insight**: The Klein-Gordon equation IS the square of the Dirac equation. This is not an approximation — it's a mathematical identity from Clifford algebra.

### GOV-01-S: The Dirac Equation with Position-Dependent Mass

$$(i\gamma^\mu \partial_\mu - \chi(x,t))\psi = 0 \tag{0.5.1}$$

where:
- $\psi \in \mathbb{C}^4$ is a 4-component Dirac spinor
- $\gamma^\mu$ are the Dirac matrices satisfying $\{\gamma^\mu, \gamma^\nu\} = 2\eta^{\mu\nu}$
- $\chi(x,t)$ is the spacetime-varying effective mass (evolves via GOV-02)

### Derivation: Dirac² = Klein-Gordon

Starting from GOV-01-S, apply $(i\gamma^\mu \partial_\mu + \chi)$ from the left:

$$(i\gamma^\mu \partial_\mu + \chi)(i\gamma^\nu \partial_\nu - \chi)\psi = 0 \tag{0.5.2}$$

Using the Clifford algebra relation $\gamma^\mu \gamma^\nu + \gamma^\nu \gamma^\mu = 2\eta^{\mu\nu}$:

$$(i\gamma^\mu \partial_\mu)(i\gamma^\nu \partial_\nu) = -\gamma^\mu \gamma^\nu \partial_\mu \partial_\nu = -\eta^{\mu\nu} \partial_\mu \partial_\nu = -\Box \tag{0.5.3}$$

where $\Box = \partial_t^2 - c^2 \nabla^2$ is the d'Alembertian.

Therefore:
$$(-\Box - \chi^2)\psi = 0 \tag{0.5.4}$$

Rearranging:
$$\boxed{\frac{\partial^2 \psi}{\partial t^2} = c^2 \nabla^2 \psi - \chi^2 \psi} \tag{0.5.5}$$

This IS GOV-01-K! Each spinor component satisfies Klein-Gordon.

### Field Representation Hierarchy

| Level | Field | Components | GOV-01 Form | What It Describes |
|-------|-------|------------|-------------|-------------------|
| 0 | $E \in \mathbb{R}$ | 1 real | Klein-Gordon | Neutral bosonic matter |
| 1 | $\Psi \in \mathbb{C}$ | 1 complex | Klein-Gordon | Charged bosonic matter |
| 2 | $\Psi_a \in \mathbb{C}^3$ | 3 complex | Klein-Gordon | Colored bosonic matter |
| 3 | $\psi \in \mathbb{C}^4$ | 4 spinor | **Dirac** | Single-color fermions |
| 4 | $\psi_a^\alpha$ | 12 complex | Dirac + color | Full Standard Model |

**Rule**: Use the simplest level that captures your physics. Start at Level 0; increase only when needed.

### What Spinors Enable

1. **Spin-1/2 particles**: Electrons, quarks, neutrinos
2. **Pauli exclusion**: Antisymmetric wavefunctions
3. **720° periodicity**: Spinor double-covering of rotations
4. **Dirac algebra**: Spin operators, magnetic moments
5. **Bell correlations**: Non-classical angular correlations (Paper 70)

### Discrete Spinor Update

On the lattice, GOV-01-S becomes a 4-component update:

$$\psi_i^{n+1} = 2\psi_i^n - \psi_i^{n-1} + (\Delta t)^2 \left[ \frac{c^2}{(\Delta x)^2} \mathcal{L}[\psi]_i^n - (\chi_i^n)^2 \psi_i^n \right]$$

where $\psi_i^n \in \mathbb{C}^4$ and $\mathcal{L}$ is the discrete Laplacian applied component-wise.

**Note**: This is the second-order form. The first-order Dirac form requires staggered lattice techniques.

---

## 1. Complex Wave Decomposition

### Setup

Starting with the complex wave field:

$$\Psi(x,t) = R(x,t) \cdot e^{i\theta(x,t)} \tag{1.1}$$

where $R$ is the real amplitude and $\theta$ is the real phase.

### First Time Derivative

$$\frac{\partial \Psi}{\partial t} = \frac{\partial}{\partial t}\left(R \cdot e^{i\theta}\right) \tag{1.2}$$

Using the product rule:

$$\frac{\partial \Psi}{\partial t} = \dot{R} \cdot e^{i\theta} + R \cdot i\dot{\theta} \cdot e^{i\theta} \tag{1.3}$$

$$\frac{\partial \Psi}{\partial t} = \left(\dot{R} + iR\dot{\theta}\right) e^{i\theta} \tag{1.4}$$

### Second Time Derivative

$$\frac{\partial^2 \Psi}{\partial t^2} = \frac{\partial}{\partial t}\left[\left(\dot{R} + iR\dot{\theta}\right) e^{i\theta}\right] \tag{1.5}$$

Applying product rule again:

$$\frac{\partial^2 \Psi}{\partial t^2} = \left(\ddot{R} + i\dot{R}\dot{\theta} + i\dot{R}\dot{\theta} + iR\ddot{\theta} - R\dot{\theta}^2\right) e^{i\theta} \tag{1.6}$$

Collecting terms:

$$\boxed{\frac{\partial^2 \Psi}{\partial t^2} = \left[\left(\ddot{R} - R\dot{\theta}^2\right) + i\left(2\dot{R}\dot{\theta} + R\ddot{\theta}\right)\right] e^{i\theta}} \tag{1.7}$$

### Second Spatial Derivative

By identical reasoning (replace $t \to x$):

$$\boxed{\frac{\partial^2 \Psi}{\partial x^2} = \left[\left(R_{xx} - R\theta_x^2\right) + i\left(2R_x\theta_x + R\theta_{xx}\right)\right] e^{i\theta}} \tag{1.8}$$

### Verification

These formulas can be verified with SymPy:

```python
from sympy import *
t, x = symbols('t x', real=True)
R = Function('R')(x, t)
theta = Function('theta')(x, t)
Psi = R * exp(I * theta)
print(simplify(diff(Psi, t, 2) / exp(I*theta)))
# Output confirms (1.7)
```

---

## 2. Current Conservation from GOV-01

### GOV-01 for Complex Field

$$\frac{\partial^2 \Psi}{\partial t^2} = c^2 \nabla^2 \Psi - \chi^2 \Psi \tag{2.1}$$

### Substitution

Substituting (1.7) and (1.8) into (2.1) and dividing by $e^{i\theta}$:

$$\left(\ddot{R} - R\dot{\theta}^2\right) + i\left(2\dot{R}\dot{\theta} + R\ddot{\theta}\right) = c^2\left[\left(R_{xx} - R\theta_x^2\right) + i\left(2R_x\theta_x + R\theta_{xx}\right)\right] - \chi^2 R \tag{2.2}$$

### Separating Real and Imaginary Parts

**Real Part (Amplitude Equation):**

$$\boxed{\ddot{R} - R\dot{\theta}^2 = c^2\left(R_{xx} - R\theta_x^2\right) - \chi^2 R} \tag{2.3}$$

**Imaginary Part (Phase Equation):**

$$\boxed{2\dot{R}\dot{\theta} + R\ddot{\theta} = c^2\left(2R_x\theta_x + R\theta_{xx}\right)} \tag{2.4}$$

### Rewriting as Continuity Equation

Consider the left side of (2.4):

$$2\dot{R}\dot{\theta} + R\ddot{\theta} = \frac{1}{R} \cdot \frac{\partial}{\partial t}\left(R^2 \dot{\theta}\right) \tag{2.5}$$

**Proof of (2.5):**
$$\frac{\partial}{\partial t}\left(R^2 \dot{\theta}\right) = 2R\dot{R}\dot{\theta} + R^2\ddot{\theta}$$
$$\frac{1}{R} \cdot \frac{\partial}{\partial t}\left(R^2 \dot{\theta}\right) = 2\dot{R}\dot{\theta} + R\ddot{\theta} \quad \checkmark$$

Similarly for the right side:

$$2R_x\theta_x + R\theta_{xx} = \frac{1}{R} \cdot \frac{\partial}{\partial x}\left(R^2 \theta_x\right) \tag{2.6}$$

Multiplying (2.4) by $R$:

$$\frac{\partial}{\partial t}\left(R^2 \dot{\theta}\right) = c^2 \frac{\partial}{\partial x}\left(R^2 \theta_x\right) \tag{2.7}$$

In 3D:

$$\boxed{\frac{\partial}{\partial t}\left(R^2 \dot{\theta}\right) = c^2 \nabla \cdot \left(R^2 \nabla\theta\right)} \tag{2.8}$$

### Identification as Continuity Equation

Defining:
- $\rho = R^2 = |\Psi|^2$ (probability/charge density)
- $\mathbf{j} = -c^2 R^2 \nabla\theta$ (probability/charge current)

Equation (2.8) becomes:

$$\boxed{\frac{\partial \rho}{\partial t} + \nabla \cdot \mathbf{j} = 0} \tag{2.9}$$

**This is exactly the continuity equation, derived purely from GOV-01.**

---

## 3. Phase Interference and Coulomb Force

### Superposition of Two Particles

Consider two wave packets:

$$\Psi_1 = R_1 e^{i\theta_1}, \quad \Psi_2 = R_2 e^{i\theta_2} \tag{3.1}$$

The total field is:

$$\Psi_{total} = \Psi_1 + \Psi_2 \tag{3.2}$$

### Energy Density

The energy density is proportional to $|\Psi_{total}|^2$:

$$|\Psi_1 + \Psi_2|^2 = |\Psi_1|^2 + |\Psi_2|^2 + 2\,\text{Re}(\Psi_1^* \Psi_2) \tag{3.3}$$

The interference term is:

$$2\,\text{Re}(\Psi_1^* \Psi_2) = 2R_1 R_2 \cos(\theta_2 - \theta_1) \tag{3.4}$$

### Phase Cases

| Relative Phase | $\cos(\Delta\theta)$ | Energy Change | Interpretation |
|----------------|----------------------|---------------|----------------|
| $\Delta\theta = 0$ (same) | $+1$ | Increases | **Repulsion** |
| $\Delta\theta = \pi$ (opposite) | $-1$ | Decreases | **Attraction** |

### Force from Energy Gradient

Force is the negative gradient of energy:

$$\mathbf{F} = -\nabla E_{total} \tag{3.5}$$

- Same phase → Energy increases as particles approach → $\mathbf{F}$ points apart → **Repulsion**
- Opposite phase → Energy decreases as particles approach → $\mathbf{F}$ points together → **Attraction**

**This reproduces Coulomb's law behavior without injecting it.**

---

## 4. Noether Current Derivation

### The Noether Current

For a complex field $\Psi$ with U(1) symmetry, the conserved Noether current is:

$$j^\mu = \text{Im}(\Psi^* \partial^\mu \Psi) \tag{4.1}$$

In component form:
- $\rho = \text{Im}(\Psi^* \dot{\Psi})$ (charge density)
- $\mathbf{j} = \text{Im}(\Psi^* \nabla\Psi)$ (current density)

### Verification

Using $\Psi = R e^{i\theta}$:

$$\Psi^* \nabla\Psi = R e^{-i\theta} \cdot \nabla(R e^{i\theta}) \tag{4.2}$$

$$= R e^{-i\theta} \cdot \left((\nabla R) e^{i\theta} + R \cdot i(\nabla\theta) e^{i\theta}\right) \tag{4.3}$$

$$= R \nabla R + i R^2 \nabla\theta \tag{4.4}$$

Taking the imaginary part:

$$\boxed{\mathbf{j} = \text{Im}(\Psi^* \nabla\Psi) = R^2 \nabla\theta} \tag{4.5}$$

This confirms the identification in Section 2.

---

## 5. Why 1/r² Force Law Emerges

### Geometric Origin

The 1/r² dependence in Coulomb's law is NOT specific to electromagnetism—it's a consequence of **3D geometry**.

### Energy Density Spreads on Spheres

When two wave packets overlap, the interference term (Eq. 3.4) contributes to the total energy:

$$\Delta E = \int 2R_1 R_2 \cos(\Delta\theta) \, d^3x \tag{5.1}$$

For localized packets separated by distance $r$:
- The overlap region has volume $\propto r^0$ (contact area)
- But energy contributions from a point source spread over a sphere of area $4\pi r^2$

### The Gradient Gives Force

The force is $\mathbf{F} = -\nabla E$. For energy that goes as $1/r$:

$$E(r) \propto \frac{1}{r} \implies F = -\frac{dE}{dr} \propto \frac{1}{r^2} \tag{5.2}$$

### Why Energy Goes as 1/r

For a point-like source with amplitude $R \propto 1/r$ (to conserve total probability on expanding spheres):

$$\int R^2 \cdot 4\pi r^2 \, dr = \text{const} \implies R \propto \frac{1}{r} \tag{5.3}$$

The interference energy between two such sources:

$$\Delta E \propto R_1 R_2 \propto \frac{1}{r_1} \cdot \frac{1}{r_2} \tag{5.4}$$

At the midpoint where they overlap: $\Delta E \propto 1/r$.

**Summary**: The 1/r² force law comes from 3D geometry (surface area of spheres), not from any specific property of electromagnetism.

---

## 6. Charge Quantization

### Why Only θ = 0 or θ = π?

In principle, θ could take any value. But stable particles require:

1. **Phase coherence**: The wave must maintain a well-defined phase
2. **Topological stability**: Only phase differences that are multiples of π give stable interference

### Discrete Phase Values

For a standing wave bound state, the boundary conditions require:

$$\theta = n\pi, \quad n \in \mathbb{Z} \tag{6.1}$$

This gives only two distinguishable charge states:
- $n$ even → θ = 0 (mod 2π) → **negative charge**
- $n$ odd → θ = π (mod 2π) → **positive charge**

### Connection to Topology

The phase θ lives on a circle (0 to 2π). Stable configurations are those where θ returns to itself after going around any closed loop—this requires θ to be constant or change by multiples of 2π.

**Result**: Charge is quantized because phase is topological.

---

## Appendix A: Common Errors

### Error: Computing ∇²(R sin θ) incorrectly

**WRONG** (claimed by some critics):
$$\nabla^2(R \sin\theta) = \frac{1}{R}\left[\sin\theta + \frac{\cos^2\theta}{\sin\theta}\right]$$

This formula:
1. Has wrong dimensions (1/R vs $R_{xx}$)
2. Is missing spatial derivatives of R and θ
3. Blows up when sin θ = 0

**CORRECT:**
$$\nabla^2(R \sin\theta) = (R_{xx} - R\theta_x^2)\sin\theta + (2R_x\theta_x + R\theta_{xx})\cos\theta \tag{A.1}$$

**Derivation:**
$$\frac{\partial}{\partial x}(R \sin\theta) = R_x \sin\theta + R \cos\theta \cdot \theta_x$$

$$\frac{\partial^2}{\partial x^2}(R \sin\theta) = R_{xx}\sin\theta + R_x\cos\theta \cdot \theta_x + R_x\cos\theta \cdot \theta_x + R(-\sin\theta)\theta_x^2 + R\cos\theta \cdot \theta_{xx}$$

$$= (R_{xx} - R\theta_x^2)\sin\theta + (2R_x\theta_x + R\theta_{xx})\cos\theta \quad \checkmark$$

---

## Appendix B: Verification Scripts

All derivations in this document can be verified using the regression test suite:

```
pytest tests/test_lfm_em_regression.py -v
```

The tests verify:
1. Time/space derivative structure (Eqs. 1.7, 1.8)
2. Real/imaginary separation (Eqs. 2.3, 2.4)
3. Continuity equation emergence (Eq. 2.9)
4. Noether current form (Eq. 4.5)
5. Phase interference signs (Section 3)

---

## References

1. Klein, O. (1926). Quantentheorie und fünfdimensionale Relativitätstheorie. *Zeitschrift für Physik*, 37(12), 895-906.
2. Gordon, W. (1926). Der Comptoneffekt nach der Schrödingerschen Theorie. *Zeitschrift für Physik*, 40(1-2), 117-133.
3. Noether, E. (1918). Invariante Variationsprobleme. *Nachrichten von der Gesellschaft der Wissenschaften zu Göttingen*, 235-257.

---

*Document maintained at: `analysis/LFM_CANONICAL_DERIVATIONS.md`*  
*Regression tests: `tests/test_lfm_em_regression.py`*
