#!/bin/bash

# Eolisa Space - Zenodo Package Verification Script
# ==================================================
# This script verifies the integrity and completeness of the Zenodo package
# before upload.

echo "============================================================"
echo "EOLISA SPACE - ZENODO PACKAGE VERIFICATION"
echo "============================================================"
echo ""

# Color codes
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

# Counters
PASS=0
FAIL=0
WARN=0

# Function to check file exists
check_file() {
    if [ -f "$1" ]; then
        echo -e "${GREEN}✓${NC} $1"
        ((PASS++))
        return 0
    else
        echo -e "${RED}✗${NC} $1 - MISSING"
        ((FAIL++))
        return 1
    fi
}

# Function to check directory exists
check_dir() {
    if [ -d "$1" ]; then
        echo -e "${GREEN}✓${NC} $1/"
        ((PASS++))
        return 0
    else
        echo -e "${RED}✗${NC} $1/ - MISSING"
        ((FAIL++))
        return 1
    fi
}

# Function to check file size
check_size() {
    FILE=$1
    MIN_SIZE=$2
    
    if [ -f "$FILE" ]; then
        SIZE=$(stat -f%z "$FILE" 2>/dev/null || stat -c%s "$FILE" 2>/dev/null)
        if [ "$SIZE" -gt "$MIN_SIZE" ]; then
            echo -e "${GREEN}✓${NC} $FILE ($(numfmt --to=iec $SIZE))"
            ((PASS++))
            return 0
        else
            echo -e "${YELLOW}⚠${NC} $FILE - size too small ($(numfmt --to=iec $SIZE))"
            ((WARN++))
            return 1
        fi
    else
        echo -e "${RED}✗${NC} $FILE - MISSING"
        ((FAIL++))
        return 1
    fi
}

echo "1. Checking Core Files..."
echo "─────────────────────────────────────────────────────────"
check_file "README.md"
check_file "LICENSE"
check_file "CITATION.cff"
check_file ".zenodo.json"
check_file "requirements.txt"
check_file "manuscript.pdf"
check_file "PUBLICATION_CHECKLIST.md"

echo ""
echo "2. Checking Directory Structure..."
echo "─────────────────────────────────────────────────────────"
check_dir "code"
check_dir "data"
check_dir "documentation"
check_dir "supplementary"

echo ""
echo "3. Checking Code Files..."
echo "─────────────────────────────────────────────────────────"
check_file "code/image_enhancement_pipeline.py"
check_file "code/grmhd_comparison.py"
check_file "code/hotspot_analysis.py"
check_file "code/download_data.py"
check_file "code/run_full_analysis.py"

echo ""
echo "4. Checking Documentation..."
echo "─────────────────────────────────────────────────────────"
check_file "documentation/DATA_NOTES.md"

echo ""
echo "5. Checking File Sizes (minimum thresholds)..."
echo "─────────────────────────────────────────────────────────"
check_size "README.md" 5000
check_size "manuscript.pdf" 10000
check_size "code/image_enhancement_pipeline.py" 15000
check_size "code/grmhd_comparison.py" 15000
check_size "code/hotspot_analysis.py" 15000
check_size "code/run_full_analysis.py" 10000
check_size "documentation/DATA_NOTES.md" 8000

echo ""
echo "6. Checking Python Syntax..."
echo "─────────────────────────────────────────────────────────"

for pyfile in code/*.py; do
    if python3 -m py_compile "$pyfile" 2>/dev/null; then
        echo -e "${GREEN}✓${NC} $pyfile - syntax OK"
        ((PASS++))
    else
        echo -e "${RED}✗${NC} $pyfile - SYNTAX ERROR"
        ((FAIL++))
    fi
done

echo ""
echo "7. Checking Metadata Completeness..."
echo "─────────────────────────────────────────────────────────"

# Check .zenodo.json
if grep -q "\"title\"" .zenodo.json && \
   grep -q "\"description\"" .zenodo.json && \
   grep -q "\"creators\"" .zenodo.json && \
   grep -q "\"keywords\"" .zenodo.json; then
    echo -e "${GREEN}✓${NC} .zenodo.json - metadata complete"
    ((PASS++))
else
    echo -e "${RED}✗${NC} .zenodo.json - metadata incomplete"
    ((FAIL++))
fi

# Check CITATION.cff
if grep -q "cff-version:" CITATION.cff && \
   grep -q "title:" CITATION.cff && \
   grep -q "authors:" CITATION.cff; then
    echo -e "${GREEN}✓${NC} CITATION.cff - metadata complete"
    ((PASS++))
else
    echo -e "${RED}✗${NC} CITATION.cff - metadata incomplete"
    ((FAIL++))
fi

echo ""
echo "8. Checking Contact Information..."
echo "─────────────────────────────────────────────────────────"

# Check for contact email
if grep -r "sentinelalpha@eolisaspace.com" README.md >/dev/null; then
    echo -e "${GREEN}✓${NC} Contact email present in README"
    ((PASS++))
else
    echo -e "${RED}✗${NC} Contact email missing in README"
    ((FAIL++))
fi

# Check for Eolisa Space attribution
if grep -r "Eolisa Space" README.md >/dev/null; then
    echo -e "${GREEN}✓${NC} Eolisa Space attribution present"
    ((PASS++))
else
    echo -e "${RED}✗${NC} Eolisa Space attribution missing"
    ((FAIL++))
fi

echo ""
echo "9. Computing Checksums..."
echo "─────────────────────────────────────────────────────────"

if command -v sha256sum &> /dev/null; then
    find . -type f ! -name "SHA256SUMS.txt" -exec sha256sum {} \; > SHA256SUMS.txt 2>/dev/null
    echo -e "${GREEN}✓${NC} SHA256 checksums generated (SHA256SUMS.txt)"
    ((PASS++))
else
    echo -e "${YELLOW}⚠${NC} sha256sum not available, skipping"
    ((WARN++))
fi

echo ""
echo "10. Package Size..."
echo "─────────────────────────────────────────────────────────"

TOTAL_SIZE=$(du -sh . 2>/dev/null | awk '{print $1}')
echo "Total package size: $TOTAL_SIZE"

if [ -f "../Eolisa_Space_SgrA_Wormhole_Analysis_v1.0.tar.gz" ]; then
    ARCHIVE_SIZE=$(ls -lh ../Eolisa_Space_SgrA_Wormhole_Analysis_v1.0.tar.gz | awk '{print $5}')
    echo "Compressed archive: $ARCHIVE_SIZE"
    echo -e "${GREEN}✓${NC} Archive ready for upload"
    ((PASS++))
else
    echo -e "${YELLOW}⚠${NC} Compressed archive not found"
    echo "Run: tar -czf ../Eolisa_Space_SgrA_Wormhole_Analysis_v1.0.tar.gz ."
    ((WARN++))
fi

echo ""
echo "============================================================"
echo "VERIFICATION SUMMARY"
echo "============================================================"
echo -e "${GREEN}PASSED:${NC}  $PASS"
echo -e "${YELLOW}WARNINGS:${NC} $WARN"
echo -e "${RED}FAILED:${NC}  $FAIL"
echo ""

if [ $FAIL -eq 0 ]; then
    echo -e "${GREEN}═══════════════════════════════════════════════════════════${NC}"
    echo -e "${GREEN}✓ PACKAGE VERIFICATION COMPLETE - READY FOR ZENODO UPLOAD${NC}"
    echo -e "${GREEN}═══════════════════════════════════════════════════════════${NC}"
    echo ""
    echo "Next steps:"
    echo "1. Upload to Zenodo: https://zenodo.org/"
    echo "2. Obtain DOI: 10.5281/zenodo.XXXXXXX"
    echo "3. Update manuscript and GitHub with DOI"
    echo "4. Submit to arXiv"
    echo ""
    exit 0
else
    echo -e "${RED}═══════════════════════════════════════════════════════════${NC}"
    echo -e "${RED}✗ VERIFICATION FAILED - FIX ERRORS BEFORE UPLOADING${NC}"
    echo -e "${RED}═══════════════════════════════════════════════════════════${NC}"
    echo ""
    exit 1
fi
