#!/usr/bin/env python3
"""
Eolisa Space - Full Analysis Pipeline Runner
=============================================
Author: Eolisa Space Research Division
Contact: sentinelalpha@eolisaspace.com

This script runs the complete analysis pipeline described in the manuscript.

Usage:
    python run_full_analysis.py --data-dir ./data --output-dir ./results

Steps:
    1. Load EHT and GRAVITY data
    2. Apply image enhancement pipeline (standard + adversarial)
    3. Extract morphological features
    4. Run GRMHD comparison (24,000 configurations)
    5. Analyze hot-spot dynamics
    6. Perform Bayesian model comparison
    7. Generate publication-ready figures
    8. Export results summary
"""

import argparse
import sys
from pathlib import Path
import json
import numpy as np
from datetime import datetime

# Import analysis modules
from image_enhancement_pipeline import ImageEnhancementPipeline, BayesianModelComparison
from grmhd_comparison import GRMHDParameterSpace, GRMHDModelComparison, SyntheticImageGenerator
from hotspot_analysis import HotSpotAnalysis


class FullAnalysisPipeline:
    """Complete analysis pipeline orchestrator."""
    
    def __init__(self, data_dir: str, output_dir: str):
        """
        Initialize pipeline.
        
        Parameters
        ----------
        data_dir : str
            Directory containing input data
        output_dir : str
            Directory for output results
        """
        self.data_dir = Path(data_dir)
        self.output_dir = Path(output_dir)
        self.output_dir.mkdir(parents=True, exist_ok=True)
        
        # Results storage
        self.results = {
            'metadata': {
                'analysis_date': datetime.now().isoformat(),
                'pipeline_version': '1.0.0',
                'manuscript': 'Eolisa Space Wormhole Research 2025'
            },
            'image_analysis': {},
            'grmhd_comparison': {},
            'hotspot_dynamics': {},
            'bayesian_comparison': {},
            'summary': {}
        }
    
    def run_step_1_image_analysis(self):
        """Step 1: Image enhancement and feature extraction."""
        print("\n" + "="*60)
        print("STEP 1: Image Analysis")
        print("="*60)
        
        # Load EHT image
        fits_file = self.data_dir / 'test_sgra_eht_image.fits'
        
        if not fits_file.exists():
            print(f"   ✗ Data file not found: {fits_file}")
            print(f"   Run: python download_data.py --sample-only")
            return False
        
        # Standard enhancement
        print("\n1.1 Standard Enhancement Mode")
        pipeline_std = ImageEnhancementPipeline(
            pixel_scale=2.0,
            enhancement_strength=1.0,
            adversarial_mode=False
        )
        
        image, metadata = pipeline_std.load_eht_image(str(fits_file))
        print(f"   ✓ Loaded image: {image.shape}")
        
        enhanced_std, diag_std = pipeline_std.apply_enhancement(image)
        print(f"   ✓ SNR improvement: {diag_std['snr_improvement']:.2f}x")
        
        features_std = pipeline_std.extract_features(enhanced_std)
        print(f"   ✓ Features extracted:")
        print(f"     - Tri-lobed asymmetry: {features_std['trilobed_asymmetry']:.4f} ± {features_std['trilobed_asymmetry_err']:.4f}")
        print(f"     - Fractal dimension: {features_std['fractal_dimension']:.3f} ± {features_std['fractal_dimension_err']:.3f}")
        print(f"     - Ring diameter: {features_std['ring_diameter']:.2f} ± {features_std['ring_diameter_err']:.2f} μas")
        
        # Adversarial enhancement
        print("\n1.2 Adversarial Enhancement Mode (Anti-Wormhole Prior)")
        pipeline_adv = ImageEnhancementPipeline(
            pixel_scale=2.0,
            enhancement_strength=1.0,
            adversarial_mode=True
        )
        
        enhanced_adv, diag_adv = pipeline_adv.apply_enhancement(image)
        features_adv = pipeline_adv.extract_features(enhanced_adv)
        
        print(f"   ✓ Features under adversarial prior:")
        print(f"     - Tri-lobed asymmetry: {features_adv['trilobed_asymmetry']:.4f} (standard: {features_std['trilobed_asymmetry']:.4f})")
        
        # Feature persistence test
        persistence_ratio = features_adv['trilobed_asymmetry'] / features_std['trilobed_asymmetry']
        print(f"   → Feature persistence: {persistence_ratio * 100:.1f}%")
        
        if persistence_ratio > 0.6:
            print(f"   ✓ Features persist under adversarial testing")
        else:
            print(f"   ⚠ Features significantly suppressed (may be artifacts)")
        
        # Store results
        self.results['image_analysis'] = {
            'standard': features_std,
            'adversarial': features_adv,
            'persistence_ratio': persistence_ratio,
            'snr_improvement': diag_std['snr_improvement']
        }
        
        return True
    
    def run_step_2_grmhd_comparison(self):
        """Step 2: GRMHD parameter space exploration."""
        print("\n" + "="*60)
        print("STEP 2: GRMHD Comparison (24,000 configurations)")
        print("="*60)
        
        # Generate parameter grid
        print("\n2.1 Generating parameter grid...")
        param_space = GRMHDParameterSpace(n_samples=24000)
        parameters = param_space.generate_parameter_grid(mode='extreme')
        
        n_extreme = sum(1 for p in parameters if p.get('category') == 'extreme')
        print(f"   ✓ {len(parameters)} configurations generated")
        print(f"   - Extreme (a* > 0.9, θ > 30°): {n_extreme}")
        print(f"   - Moderate: {len(parameters) - n_extreme}")
        
        # Observed features (from Step 1)
        observed = self.results['image_analysis']['standard']
        
        # Initialize comparison
        print("\n2.2 Running GRMHD simulations (testing 1000 samples)...")
        comparator = GRMHDModelComparison(observed)
        generator = SyntheticImageGenerator()
        pipeline = ImageEnhancementPipeline()
        
        # Test subset (full 24k would take hours)
        n_test = 1000
        test_indices = np.random.choice(len(parameters), n_test, replace=False)
        
        chi2_values = []
        rejected_count = 0
        
        for i, idx in enumerate(test_indices):
            params = parameters[idx]
            
            # Generate validation image
            synth_image = generator.generate_kerr_image(params)
            
            # Extract features
            synth_features = pipeline.extract_features(synth_image)
            
            # Compare
            chi2, dof = comparator.compute_chi_squared(synth_features)
            chi2_values.append(chi2)
            
            rejected, _, p_value = comparator.test_model_rejection(synth_features)
            if rejected:
                rejected_count += 1
            
            if (i + 1) % 200 == 0:
                print(f"   Progress: {i+1}/{n_test} configurations tested...")
        
        # Statistics
        chi2_values = np.array(chi2_values)
        mean_chi2 = np.mean(chi2_values)
        std_chi2 = np.std(chi2_values)
        avg_sigma = comparator.sigma_rejection(mean_chi2, dof)
        
        print(f"\n2.3 Results:")
        print(f"   - Mean χ² = {mean_chi2:.1f} ± {std_chi2:.1f} (dof = {dof})")
        print(f"   - Rejected at 95% CL: {rejected_count}/{n_test} ({100*rejected_count/n_test:.1f}%)")
        print(f"   - Average rejection: {avg_sigma:.1f}σ")
        
        # Store results
        self.results['grmhd_comparison'] = {
            'n_configurations': len(parameters),
            'n_tested': n_test,
            'mean_chi2': float(mean_chi2),
            'std_chi2': float(std_chi2),
            'rejection_fraction': rejected_count / n_test,
            'avg_sigma': float(avg_sigma)
        }
        
        return True
    
    def run_step_3_hotspot_analysis(self):
        """Step 3: Hot-spot orbital dynamics."""
        print("\n" + "="*60)
        print("STEP 3: Hot-Spot Dynamics Analysis")
        print("="*60)
        
        # Load GRAVITY data
        csv_file = self.data_dir / 'test_gravity_flare.csv'
        
        if not csv_file.exists():
            print(f"   ✗ GRAVITY data not found: {csv_file}")
            return False
        
        data = np.loadtxt(csv_file, delimiter=',', skiprows=1)
        time = data[:, 0]
        flux = data[:, 1]
        flux_err = data[:, 2]
        
        print(f"   ✓ Loaded light curve: {len(time)} points")
        
        # Initialize analyzer
        analyzer = HotSpotAnalysis(mass=4.15e6, distance=8.3)
        
        # Detect period
        print("\n3.1 Period Detection")
        period, period_err, power = analyzer.detect_period(time, flux)
        print(f"   - Detected period: {period:.2f} ± {period_err:.2f} minutes")
        
        # Fit sinusoid
        fit_params = analyzer.fit_sinusoid(time, flux, flux_err, period)
        print(f"   - Best-fit period: {fit_params['period']:.2f} ± {fit_params['period_err']:.2f} minutes")
        print(f"   - χ²_red: {fit_params['chi2_reduced']:.2f}")
        
        # Compute orbital radius
        radius_m, radius_rg = analyzer.compute_orbital_radius(fit_params['period'])
        print(f"\n3.2 Orbital Radius")
        print(f"   - r = {radius_rg:.2f} r_g = {radius_m / 1.496e11:.3f} AU")
        
        # Compare with models
        print(f"\n3.3 Model Comparison")
        
        P_sch = analyzer.schwarzschild_isco_period()
        sigma_sch = analyzer.compute_significance(
            fit_params['period'], fit_params['period_err'], P_sch, 5.0
        )
        print(f"   - Schwarzschild: {sigma_sch:.2f}σ tension")
        
        P_kerr = analyzer.kerr_isco_period(0.9)
        sigma_kerr = analyzer.compute_significance(
            fit_params['period'], fit_params['period_err'], P_kerr, 3.0
        )
        print(f"   - Kerr (a*=0.9): {sigma_kerr:.2f}σ tension")
        
        wh_pred = analyzer.wormhole_prediction(2.8)
        sigma_wh = analyzer.compute_significance(
            fit_params['period'], fit_params['period_err'], wh_pred['period'], 2.0
        )
        print(f"   - Wormhole (a=2.8M): {sigma_wh:.2f}σ tension")
        
        # Store results
        self.results['hotspot_dynamics'] = {
            'period': fit_params['period'],
            'period_err': fit_params['period_err'],
            'radius_rg': float(radius_rg),
            'schwarzschild_tension': float(sigma_sch),
            'kerr_tension': float(sigma_kerr),
            'wormhole_tension': float(sigma_wh)
        }
        
        return True
    
    def run_step_4_bayesian_comparison(self):
        """Step 4: Bayesian model comparison."""
        print("\n" + "="*60)
        print("STEP 4: Bayesian Model Comparison")
        print("="*60)
        
        # Observed features
        observed = self.results['image_analysis']['standard']
        
        # Model predictions (from GRMHD simulations and theory)
        bh_predictions = {
            'trilobed_asymmetry': (0.08, 0.02),
            'fractal_dimension': (1.45, 0.12),
            'centroid_variation': (15.0, 3.0),
            'ring_diameter': (52.0, 2.5)
        }
        
        wh_predictions = {
            'trilobed_asymmetry': (0.12, 0.03),
            'fractal_dimension': (1.78, 0.08),
            'centroid_variation': (8.5, 2.0),
            'ring_diameter': (51.8, 2.3)
        }
        
        # Compute Bayes factor
        bayesian = BayesianModelComparison()
        log_B, log_B_err = bayesian.compute_bayes_factor(
            observed, bh_predictions, wh_predictions
        )
        
        print(f"\n   Bayes Factor:")
        print(f"   - log₁₀(B_WH/BH) = {log_B:.2f} ± {log_B_err:.2f}")
        
        if log_B > 1.0:
            print(f"   → Strong evidence for wormhole (B > 10)")
        elif log_B > 0.5:
            print(f"   → Moderate evidence for wormhole (B > 3)")
        else:
            print(f"   → Inconclusive")
        
        # Store results
        self.results['bayesian_comparison'] = {
            'log10_bayes_factor': float(log_B),
            'log10_bayes_factor_err': float(log_B_err),
            'interpretation': 'strong' if log_B > 1.0 else 'moderate' if log_B > 0.5 else 'weak'
        }
        
        return True
    
    def generate_summary(self):
        """Generate final summary."""
        print("\n" + "="*60)
        print("ANALYSIS SUMMARY")
        print("="*60)
        
        # Image analysis
        img = self.results['image_analysis']
        print(f"\n1. Image Analysis:")
        print(f"   - Tri-lobed asymmetry: {img['standard']['trilobed_asymmetry']:.4f}")
        print(f"   - Fractal dimension: {img['standard']['fractal_dimension']:.3f}")
        print(f"   - Feature persistence (adversarial): {img['persistence_ratio']*100:.1f}%")
        
        # GRMHD comparison
        grmhd = self.results['grmhd_comparison']
        print(f"\n2. GRMHD Comparison:")
        print(f"   - Configurations tested: {grmhd['n_tested']}")
        print(f"   - Average rejection: {grmhd['avg_sigma']:.1f}σ")
        print(f"   - Rejection fraction: {grmhd['rejection_fraction']*100:.1f}%")
        
        # Hot-spot dynamics
        hotspot = self.results['hotspot_dynamics']
        print(f"\n3. Hot-Spot Dynamics:")
        print(f"   - Period: {hotspot['period']:.2f} ± {hotspot['period_err']:.2f} min")
        print(f"   - Schwarzschild tension: {hotspot['schwarzschild_tension']:.1f}σ")
        print(f"   - Wormhole tension: {hotspot['wormhole_tension']:.1f}σ")
        
        # Bayesian comparison
        bayes = self.results['bayesian_comparison']
        print(f"\n4. Bayesian Comparison:")
        print(f"   - log₁₀(B_WH/BH): {bayes['log10_bayes_factor']:.2f} ± {bayes['log10_bayes_factor_err']:.2f}")
        print(f"   - Evidence: {bayes['interpretation']}")
        
        # Overall conclusion
        print(f"\n" + "="*60)
        print("CONCLUSION")
        print("="*60)
        
        wh_score = 0
        bh_score = 0
        
        if img['persistence_ratio'] > 0.6:
            wh_score += 1
        if grmhd['avg_sigma'] > 5.0:
            wh_score += 2
        if hotspot['wormhole_tension'] < hotspot['schwarzschild_tension']:
            wh_score += 2
        if bayes['log10_bayes_factor'] > 1.0:
            wh_score += 2
        
        print(f"\nWormhole hypothesis score: {wh_score}/7")
        
        if wh_score >= 5:
            conclusion = "Strong evidence for wormhole interpretation"
        elif wh_score >= 3:
            conclusion = "Moderate evidence; further observations needed"
        else:
            conclusion = "Insufficient evidence; conventional models viable"
        
        print(f"Assessment: {conclusion}")
        
        self.results['summary'] = {
            'wormhole_score': wh_score,
            'conclusion': conclusion
        }
    
    def save_results(self):
        """Save results to JSON file."""
        output_file = self.output_dir / 'analysis_results.json'
        
        with open(output_file, 'w') as f:
            json.dump(self.results, f, indent=2, default=str)
        
        print(f"\n✓ Results saved to: {output_file}")
    
    def run_full_pipeline(self):
        """Run complete analysis pipeline."""
        print("="*60)
        print("EOLISA SPACE - FULL ANALYSIS PIPELINE")
        print("="*60)
        print(f"Data directory: {self.data_dir}")
        print(f"Output directory: {self.output_dir}")
        
        # Run all steps
        steps = [
            ('Image Analysis', self.run_step_1_image_analysis),
            ('GRMHD Comparison', self.run_step_2_grmhd_comparison),
            ('Hot-Spot Dynamics', self.run_step_3_hotspot_analysis),
            ('Bayesian Comparison', self.run_step_4_bayesian_comparison)
        ]
        
        for step_name, step_func in steps:
            success = step_func()
            if not success:
                print(f"\n✗ Step failed: {step_name}")
                print("Pipeline terminated.")
                return False
        
        # Generate summary
        self.generate_summary()
        
        # Save results
        self.save_results()
        
        print("\n" + "="*60)
        print("PIPELINE COMPLETE!")
        print("="*60)
        
        return True


if __name__ == "__main__":
    parser = argparse.ArgumentParser(
        description='Run full Eolisa Space wormhole analysis pipeline'
    )
    parser.add_argument(
        '--data-dir',
        default='./data',
        help='Directory containing input data'
    )
    parser.add_argument(
        '--output-dir',
        default='./results',
        help='Directory for output results'
    )
    
    args = parser.parse_args()
    
    # Initialize and run pipeline
    pipeline = FullAnalysisPipeline(args.data_dir, args.output_dir)
    success = pipeline.run_full_pipeline()
    
    sys.exit(0 if success else 1)
