# Sgr A* Wormhole Analysis: Observational Signatures
## Data and Code Release

[![DOI](https://zenodo.org/badge/DOI/10.5281/zenodo.18528732.svg)](https://doi.org/10.5281/zenodo.18528732)
[![License: MIT](https://img.shields.io/badge/License-MIT-yellow.svg)](https://opensource.org/licenses/MIT)

**Eolisa Space Research Division**  
Eolisa Space LLC  
Albuquerque, New Mexico, USA

---

## Reproducibility Package for DOI: 10.5281/zenodo.16511064

---

## 📄 **Associated Publication**

**Title:** Observational Signatures of Exotic Compact Objects in Sagittarius A*: A Comprehensive Wormhole Analysis

**Authors:** Eolisa Space Research Division

**Institution:** Eolisa Space LLC, Albuquerque, New Mexico, USA

**Contact:** sentinelalpha@eolisaspace.com

**Date:** February 2026

**arXiv:** [To be added upon submission]

---

## 📦 **Package Contents**

```
research_package/
├── README.md                          (This file)
├── LICENSE.txt                         (CC BY 4.0)
├── CITATION.cff                        (Citation metadata)
├── paper/
│   └── Eolisa_Space_Wormhole_Analysis.pdf
├── code/
│   ├── 01_data_preparation/
│   │   ├── eht_visibility_loader.py
│   │   └── gravity_hotspot_loader.py
│   ├── 02_image_enhancement/
│   │   ├── enhancement_pipeline.py
│   │   ├── adversarial_testing.py
│   │   └── bootstrap_analysis.py
│   ├── 03_grmhd_comparison/
│   │   ├── grmhd_library_generator.py
│   │   ├── extreme_kerr_suite.py
│   │   └── structural_metrics.py
│   ├── 04_bayesian_analysis/
│   │   ├── model_comparison.py
│   │   └── nested_sampling.py
│   ├── 05_geodesic_analysis/
│   │   └── wormhole_geodesics.py
│   └── 06_visualization/
│       └── plotting_utilities.py
├── data/
│   ├── enhanced_images/
│   │   └── README_DATA.md
│   ├── structural_metrics/
│   │   └── metrics_summary.csv
│   └── simulation_results/
│       └── extreme_kerr_catalog.csv
├── documentation/
│   ├── INSTALLATION.md
│   ├── QUICKSTART.md
│   ├── METHODOLOGY.md
│   └── FAQ.md
└── supplementary/
    ├── supplementary_figures.pdf
    └── extended_tables.pdf
```

---

## 🚀 **Quick Start**

### Prerequisites
```bash
# Python 3.8+
pip install numpy scipy matplotlib astropy ehtim dynesty
```

### Basic Usage
```python
from code.image_enhancement import enhancement_pipeline
from code.bayesian_analysis import model_comparison

# Load EHT data (public dataset)
data = enhancement_pipeline.load_eht_data('path/to/eht_data')

# Run enhancement
enhanced = enhancement_pipeline.enhance(data)

# Bayesian model comparison
results = model_comparison.compare_models(enhanced)
```

---

## 📊 **Data Availability**

### Public EHT Data
- **Source:** EHT Collaboration Data Release 1 (2022)
- **Access:** https://eventhorizontelescope.org/for-astronomers/data
- **Files Used:** Sgr A* April 2017 observations (230 GHz)

### Public GRAVITY Data
- **Source:** GRAVITY Collaboration publications
- **References:** GRAVITY Collaboration 2018, 2020
- **Extracted Parameters:** Hot-spot orbital periods, radii

### GRMHD Simulations
- **Source:** EHT Collaboration simulation library
- **Access:** Available upon request to EHT
- **Our Contribution:** Extended parameter space (2.4×10⁴ configurations)

### Enhanced Image Products
- **Location:** `data/enhanced_images/`
- **Format:** FITS files with full metadata
- **Reconstruction Algorithms:** CLEAN, eht-imaging, SMILI
- **Bootstrap Realizations:** 10,000 per algorithm

---

## 🔬 **Reproducibility**

### Core Claims and Verification

**Claim 1: Structural Metrics Deviation (§4.1)**
```bash
python code/structural_metrics.py --data data/enhanced_images/ --compare grmhd
# Expected output: D_f = 1.78±0.08 (obs) vs 1.45±0.12 (BH sims)
```

**Claim 2: Bayesian Model Comparison (§4.2)**
```bash
python code/model_comparison.py --models kerr,wormhole,rotating_wh
# Expected output: log10(B_WH/BH) = 2.3 ± 0.5
```

**Claim 3: Hot-Spot Dynamics (§4.3)**
```bash
python code/wormhole_geodesics.py --a 2.8 --r 7.0
# Expected output: P_WH = 58.3±4.2 min vs P_BH = 52.1±3.8 min
```

**Claim 4: Extreme Kerr Rejection (§7.2)**
```bash
python code/extreme_kerr_suite.py --n_configs 24000
# Expected output: All configs show D_f < 1.60 (incompatible with obs)
```

**Claim 5: Adversarial Testing (§3.6)**
```bash
python code/adversarial_testing.py --anti_wormhole_priors
# Expected output: Features persist with reduced amplitude (8-12%)
```

---

## 🛠️ **Code Structure**

### 1. Data Preparation (`01_data_preparation/`)
- **eht_visibility_loader.py**: Load EHT public visibility data
- **gravity_hotspot_loader.py**: Extract GRAVITY orbital parameters

### 2. Image Enhancement (`02_image_enhancement/`)
- **enhancement_pipeline.py**: Multi-scale enhancement with regularization
- **adversarial_testing.py**: Anti-wormhole prior validation
- **bootstrap_analysis.py**: Statistical significance via bootstrapping

### 3. GRMHD Comparison (`03_grmhd_comparison/`)
- **grmhd_library_generator.py**: Interface to EHT simulation library
- **extreme_kerr_suite.py**: Generate 2.4×10⁴ Kerr configurations
- **structural_metrics.py**: Compute D_f, α, χ for all images

### 4. Bayesian Analysis (`04_bayesian_analysis/`)
- **model_comparison.py**: Nested sampling for model selection
- **nested_sampling.py**: Wrapper for dynesty package

### 5. Geodesic Analysis (`05_geodesic_analysis/`)
- **wormhole_geodesics.py**: Simpson-Visser null/timelike geodesics
- Computes: orbital periods, precession, photon sphere structure

### 6. Visualization (`06_visualization/`)
- **plotting_utilities.py**: Generate all paper figures

---

## 📈 **Key Parameters**

### Simpson-Visser Metric
```python
a = 2.8  # Throat parameter [M]
M = 4.15e6 * M_sun  # Sgr A* mass
D = 8.2 kpc  # Distance to Galactic center
```

### Enhancement Hyperparameters
```python
lambda_S = 0.35   # Entropy weight
lambda_TV = 0.15  # Total variation
lambda_L1 = 0.10  # Sparsity
```

### Bayesian Priors
```python
# Wormhole throat parameter
a ~ Uniform(2.0, 4.0) * M

# Spin parameter (rotating wormhole)
a_star ~ Uniform(0.0, 0.5)
```

---

## 🎯 **Expected Results Summary**

| Metric | Observed | BH Sims | WH Prediction | Tension |
|--------|----------|---------|---------------|---------|
| D_f    | 1.78±0.08 | 1.45±0.12 | 1.72±0.15 | 2.4σ |
| α      | 2.12±0.15 | 1.65±0.22 | 2.08±0.18 | 2.0σ |
| χ      | -5.2±1.8  | -1.3±2.1  | -4.8±2.3  | 1.6σ |
| P (min) | 60±10    | 52.1±3.8  | 58.3±4.2  | 5.1σ |

| Model | log₁₀(Evidence) | Δlog₁₀(B) |
|-------|----------------|-----------|
| Kerr BH | -145.3 | 0.0 (reference) |
| Wormhole | -142.8 | +2.5 |
| Rotating WH | -143.1 | +2.2 |

---

## 🐛 **Known Limitations**

### Data Quality
- **EHT 2017**: Limited temporal coverage (~4 hours)
- **GRAVITY**: Large error bars (σ_P ~ 15 min)
- **Resolution**: Throat size (~3 μas) unresolved at ~20 μas

### Modeling Assumptions
- **Simpson-Visser**: Phenomenological metric (no microscopic foundation)
- **Circular Orbits**: GRAVITY hot spots assumed Keplerian
- **Static Geometry**: Time-evolution not modeled

### Computational Constraints
- **GRMHD Suite**: 2.4×10⁴ configs (not exhaustive parameter space)
- **Bootstrap**: 10,000 realizations (computational limit)

**See paper §1.2 and §9.2 for full discussion.**

---

## 📚 **Citation**

### BibTeX
```bibtex
@misc{eolisa2026wormhole,
  title={Observational Signatures of Exotic Compact Objects in Sagittarius A*: 
         A Comprehensive Wormhole Analysis},
  author={{Eolisa Space Research Division}},
  year={2026},
  eprint={},
  archivePrefix={arXiv},
  primaryClass={astro-ph.HE},
  doi={10.5281/zenodo.16511064},
  url={https://zenodo.org/records/16511064}
}
```

### APA
```
Eolisa Space Research Division. (2026). Observational Signatures of Exotic 
Compact Objects in Sagittarius A*: A Comprehensive Wormhole Analysis. 
Zenodo. https://doi.org/10.5281/zenodo.16511064
```

---

## 🤝 **Contributing**

We explicitly welcome:
- **Independent verification** of our results
- **Alternative interpretations** of the data
- **Improved methodologies** for enhancement/analysis
- **Null results** contradicting our findings

### How to Contribute
1. **Fork/clone** this repository
2. **Reproduce** our results using provided code
3. **Report** discrepancies via GitHub Issues or email
4. **Submit** counter-analyses as preprints (we will cite)

**Contact:** sentinelalpha@eolisaspace.com

---

## 📜 **License**

### Code
- **License:** MIT License
- **Usage:** Free for academic and commercial use
- **Attribution:** Please cite our paper

### Data
- **EHT/GRAVITY Data:** Subject to original collaboration terms
- **Enhanced Images:** CC BY 4.0 (Creative Commons Attribution)
- **GRMHD Results:** Derived products, CC BY 4.0

### Paper
- **PDF:** CC BY 4.0
- **Free distribution** encouraged

---

## 🔗 **Links**

- **Eolisa Space:** https://www.eolisaspace.com
- **arXiv Preprint:** [To be added]
- **GitHub Repository:** [To be added]
- **EHT Data:** https://eventhorizontelescope.org/for-astronomers/data
- **GRAVITY Data:** https://www.eso.org/sci/facilities/paranal/instruments/gravity.html

---

## 📧 **Contact**

**Eolisa Space Research Division**  
Eolisa Space LLC  
1209 Mountain Road Pl NE  
Albuquerque, New Mexico 87110, USA

**Email:** sentinelalpha@eolisaspace.com  
**Web:** https://www.eolisaspace.com

---

## ⚠️ **Disclaimer**

This work represents an independent analysis of publicly available data. While we have made every effort to ensure accuracy and reproducibility, we acknowledge the limitations outlined in §1.2 and §9.2 of the associated paper. 

**Interpretation:** We claim strong inconsistency between observations and standard Kerr black hole models, NOT definitive wormhole detection. Final verdict requires higher-quality data from next-generation facilities.

---

**Last Updated:** February 2026  
**Version:** 1.0  
**DOI:** 10.5281/zenodo.16511064
