# This file is part of the TaylorIntegration.jl package; MIT licensed

# jetcoeffs!
@doc doc"""
    jetcoeffs!(eqsdiff, t, x, params)

Returns an updated `x` using the recursion relation of the
derivatives obtained from the differential equations
``\dot{x}=dx/dt=f(x, p, t)``.

`eqsdiff` is the function defining the RHS of the ODE,
`x` contains the Taylor1 expansion of the dependent variable(s) and
`t` is the independent variable, and `params` are the parameters appearing on the
function defining the differential equation. See [`taylorinteg`](@ref) for examples
and convention for `eqsdiff`. Note that `x` is of type `Taylor1{U}` where
`U<:Number`; `t` is of type `Taylor1{T}` where `T<:Real`.

Initially, `x` contains only the 0-th order Taylor coefficient of
the current system state (the initial conditions), and `jetcoeffs!`
computes recursively the high-order derivates back into `x`.

"""
function jetcoeffs!(
    eqsdiff,
    t::Taylor1{T},
    x::Taylor1{U},
    params,
) where {T<:Real,U<:Number}
    order = x.order
    for ord = 0:order-1
        ordnext = ord + 1

        # Set `xaux` auxiliary Taylor1 variables to order `ord`
        @inbounds xaux = Taylor1(@views x.coeffs[begin:ordnext])

        # Equations of motion
        dx = eqsdiff(xaux, params, t)

        # Recursion relation
        solcoeff!(x, dx, ordnext)
    end
    nothing
end

@doc doc"""
    jetcoeffs!(eqsdiff!, t, x, dx, xaux, params)

Mutates `x` in-place using the recursion relation of the
derivatives obtained from the differential equations
``\dot{x}=dx/dt=f(x, p, t)``.

`eqsdiff!` is the function defining the RHS of the ODE,
`x` contains the Taylor1 expansion of the dependent variables and
`t` is the independent variable, and `params` are the parameters appearing on the
function defining the differential equation. See [`taylorinteg`](@ref) for examples
and convention for `eqsdiff`. Note that `x` is of type `Vector{Taylor1{U}}`
where `U<:Number`; `t` is of type `Taylor1{T}` where `T<:Real`. In this case,
two auxiliary containers `dx` and `xaux` (both of the same type as `x`) are
needed to avoid allocations.

Initially, `x` contains only the 0-th order Taylor coefficient of
the current system state (the initial conditions), and `jetcoeffs!`
computes recursively the high-order derivates back into `x`.

"""
function jetcoeffs!(
    eqsdiff!,
    t::Taylor1{T},
    x::AbstractArray{Taylor1{U},N},
    dx::AbstractArray{Taylor1{U},N},
    xaux::AbstractArray{Taylor1{U},N},
    params,
) where {T<:Real,U<:Number,N}
    order = x[1].order
    for ord = 0:order-1
        ordnext = ord + 1

        # Set `xaux`, auxiliary vector of Taylor1 to order `ord`
        for j in eachindex(x)
            @inbounds xaux[j] = Taylor1(@views x[j].coeffs[begin:ordnext])
        end

        # Equations of motion
        eqsdiff!(dx, xaux, params, t)

        # Recursion relations
        for j in eachindex(x)
            @inbounds solcoeff!(x[j], dx[j], ordnext)
        end
    end
    nothing
end


# __jetcoeffs
"""
    __jetcoeffs!(::Val{false}, f, t, x, params, rv)
    __jetcoeffs!(::Val{true}, f, t, x, params, rv)
    __jetcoeffs!(::Val{false}, f, t, x, dx, xaux, params, rv)
    __jetcoeffs!(::Val{true}, f, t, x, dx, xaux, params, rv)

Chooses a method of [`jetcoeffs!`](@ref) (hard-coded) or the generated by
[`@taylorize`](@ref)) depending on `Val{bool}` (`bool::Bool`).
"""
@inline __jetcoeffs!(
    ::Val{false},
    f,
    t,
    x::Taylor1{U},
    params,
    ::RetAlloc{Taylor1{U}},
) where {U} = jetcoeffs!(f, t, x, params)
@inline __jetcoeffs!(
    ::Val{true},
    f,
    t,
    x::Taylor1{U},
    params,
    rv::RetAlloc{Taylor1{U}},
) where {U} = jetcoeffs!(Val(f), t, x, params, rv)

@inline __jetcoeffs!(::Val{false}, f, t, x, dx, xaux, params, rv) =
    jetcoeffs!(f, t, x, dx, xaux, params)
@inline __jetcoeffs!(::Val{true}, f, t, x, dx, xaux, params, rv) =
    jetcoeffs!(Val(f), t, x, dx, params, rv)


# _allocate_jetcoeffs! fallbacks; needed to define the corresponding parsed method
@inline _allocate_jetcoeffs!(::Taylor1{T}, x::Taylor1{S}, params) where {T,S} =
    RetAlloc{Taylor1{S}}()
@inline _allocate_jetcoeffs!(
    ::Taylor1{T},
    x::AbstractArray{Taylor1{S},N},
    ::AbstractArray{Taylor1{S},N},
    params,
) where {T,S,N} = RetAlloc{Taylor1{S}}()


# _determine_parsing!
"""
    _determine_parsing!(parse_eqs::Bool, f, t, x, params)
    _determine_parsing!(parse_eqs::Bool, f, t, x, dx, params)

Check if the parsed method of `jetcoeffs!` exists and check it
runs without error.
"""
function _determine_parsing!(parse_eqs::Bool, f, t, x, params)
    parse_eqs =
        parse_eqs &&
        !isempty(methodswith(Val{f}, jetcoeffs!)) &&
        !isempty(methodswith(Val{f}, _allocate_jetcoeffs!))

    # tmpTaylor, arrTaylor = _allocate_jetcoeffs!(t, x, params)
    rv = _allocate_jetcoeffs!(t, x, params)

    if parse_eqs
        try
            rv = _allocate_jetcoeffs!(Val(f), t, x, params)
            jetcoeffs!(Val(f), t, x, params, rv)
        catch
            @warn("""Unable to use the parsed method of `jetcoeffs!` for `$f`,
            despite of having `parse_eqs=true`, due to some internal error.
            Using `parse_eqs = false`.""")
            parse_eqs = false
        end
    end

    return parse_eqs, rv
end
function _determine_parsing!(parse_eqs::Bool, f, t, x, dx, params)

    parse_eqs =
        parse_eqs &&
        !isempty(methodswith(Val{f}, jetcoeffs!)) &&
        !isempty(methodswith(Val{f}, _allocate_jetcoeffs!))

    rv = _allocate_jetcoeffs!(t, x, dx, params)

    if parse_eqs
        try
            rv = _allocate_jetcoeffs!(Val(f), t, x, dx, params)
            jetcoeffs!(Val(f), t, x, dx, params, rv)
        catch
            @warn("""Unable to use the parsed method of `jetcoeffs!` for `$f`,
            despite of having `parse_eqs=true`, due to some internal error.
            Using `parse_eqs = false`.""")
            parse_eqs = false
        end
    end

    return parse_eqs, rv
end
