<!-- Macros and functions for EU PPP reports, including appendix E format -->

<#--
	appendixEstudies prints information of a list of studies (ENDPOINT_STUDY_RECORDs, FLEXIBLE_RECORDs) following the structure 
	set in the Appendix E of EFSA's 2019 admin guidance for PPPs.

	Inputs:
		- subject: ENTITY or list of ENTITIES that studies belong to
		- docSubTypes: string or list of strings defining document's sub type e.g. "AnalyticalMethods"
		- context: list of hashmaps containing the conditions to be used to select which document(s) to output
			The hashmaps are definied in the ftls of the individual reports that use the appendixEstudies macro
			E.g. one example containing one condition for analytical methods for risk assessment is
				[{"path": "AdministrativeData.Endpoint", "val" : ["risk"], "qual" : "lk", "type" : "picklist"}] 
			Fur further description see 'filterStudyListbyContext' function
		- name: string with the name of the document to be displayed; if empty the name will be taken from the document subtype
		- section: string or list of strings containing the section(s) where the document needs to be cross-referenced in order to be output
			This is only applicable to the analytical methods section.
			E.g. section="6" or section=["7.1", "7.3"]
		- includeMetabolites: bool; if true, studies of the selected type in metabolite datasets will also be output
			This requires the global variable _metabolites to exist and have content.
			It is applicable for reports on the active substance (where metabolite studies should be also listed)
		- includeStudies: bool; if true studies are printed. In combination with includeWaivers it can control if to print studies, waivers, or both. 
		- includeWaivers: bool; if true waivers are printed. In combination with includeStudies it can control if to print studies, waivers, or both. 
		- includeRichTextFields: bool; if true print rich text fields of materials and methods, results, and overall remarks sections
		- includeImages: bool; if true print attached images
		- includeAnnotations: bool; to include or not annotations created with IUCLID annotations tool
		- sectionNb: text value to display the regulatory data point
		- includeImpurities: bool; if true, studies of the selected type in impurity datasets will also be output
			This requires the global variable _impurities to exist and have content.
			It is applicable for reports on the active substance (where impurity studies should be also listed)
		- entitiesHashKey: key for the subject (see above) in the 'entities' hash
		
-->
<#macro appendixEstudies subject docSubTypes context="" name="" section="" includeMetabolites=true includeStudies=true includeWaivers=true includeRichTextFields=true includeImages=true includeAnnotations=false sectionNb="" includeImpurities=true entitiesHashKey="main" >
	<#compress>

		<#-- Convert docSubTypes to list, in case only one name provided -->
		<#if !docSubTypes?is_sequence>
			<#local docSubTypes=[docSubTypes]/>
		</#if>

		<#-- Get requirement name to display-->
		<#local name = getRequirementName(name, docSubTypes)>

		<#-- Convert subject always to a list -->
		<#if !subject?is_sequence>
			<#local subject=[subject]/>
		</#if>

		<#-- Create list of entities to loop for studies, based on subject -->
		<#local entities = {}/>
		<#list subject as entity>
			<#local entities = com.addToHash(entities, entitiesHashKey, entity)/>
		</#list>

		<#-- Add metabolites to the list of entities to loop for studies, if required -->
		<#if includeMetabolites && _metabolites?? && _metabolites?has_content>
			<#if _metabolites?is_sequence>
				<#list _metabolites as metabolite>
					<#local entities = com.addToHash(entities, "metab", metabolite)/>
				</#list>
			</#if>
		</#if>

		<#-- Add impurities to the list of entities to loop for studies, if required -->
		<#if includeImpurities && _impurities?? && _impurities?has_content>
			<#if _impurities?is_sequence>
				<#list _impurities as impurity>
					<#local entities = com.addToHash(entities, "impur", impurity)/>
				</#list>
			</#if>
		</#if>

		<#-- Get a hash of studies for results and data waivers for all entities-->
		<#local studyHash = createStudyHash(entities, docSubTypes, context, section)/>

		<#-- DATA WAIVING: if flag is true and there are data waivings, output waivers -->
		<#local waiversFound = false/>
		<#if includeWaivers>
			<#list studyHash['dw'] as entityType, entityStudyHash_dw> <#-- "main", "metab", "impur" -->
				<#if entityStudyHash_dw?has_content>
					<#if !waiversFound>					
						<@com.emptyLine/>
						<para><emphasis role="HEAD-KeyA">Data waiving</emphasis></para>

						<#local waiversFound = true>
					</#if>	

					<#list entityStudyHash_dw as entityName, entityData>

						<#local dataWaivingStudyList=entityData['records']/>

						<#-- print a header for metabolites -->
						<#if entityType == "metab">
							<@com.emptyLine/>
							<para><emphasis role="underline">----- Metabolite <emphasis role="bold">${entityName}</emphasis> -----</emphasis></para>
							<@com.emptyLine/>
						</#if>

						<#-- print a header for impurities -->
						<#if entityType == "impur">
							<@com.emptyLine/>
							<para><emphasis role="underline">----- Relevant Impurity <emphasis role="bold">${entityName}</emphasis> -----</emphasis></para>
							<@com.emptyLine/>
						</#if>						

						<@dataWaiver dataWaivingStudyList name includeAnnotations/>
					</#list>
				</#if>	
			</#list>	
		</#if>

		<#-- STUDY RESULTS: if flag is true, output studies -->
		<#if includeStudies>
			<#-- check if there are studies -->
			<#local studiesFound = false>
			<#list studyHash['st'] as entityType, entityStudyHash_st> <#-- "main", "metab", "impur" -->
				<#if entityStudyHash_st?has_content>
					<#local studiesFound = true>
					<#break>
				</#if>
			</#list>

			<#-- if there are no waivers or there are studies, print "Studies" title -->
			<#if !waiversFound || studiesFound>
				<@com.emptyLine/>
				<para><emphasis role="HEAD-KeyA">Studies</emphasis></para>
			</#if>

			<#-- if there are no studies, print text -->
			<#if !studiesFound>
				<#if !waiversFound>
					<#--  <@com.emptyLine/>  -->
					<para>No individual studies available for ${name}.</para>
					<#--  <@com.emptyLine/>  -->
				</#if>
			<#else>

				<#-- print summary text for studies -->
				<para><@resultStudyHashSummaryText studyHash['st'] name/></para>
				<@com.emptyLine/>

				<#-- loop over entities -->
				<#local studyIndex=0/>
				<#list studyHash['st'] as entityType, entityStudyHash_st> <#-- "main", "metab", "impur" -->
					<#list entityStudyHash_st as entityName, resultData>
		
						<#local resultStudyList=resultData['records']/>
						<#local entity=resultData['entity']/>

						<#-- print a header for metabolites -->
						<#if entityType == "metab">
							<para><emphasis role="underline">----- Metabolite <emphasis role="bold">${entityName}</emphasis> -----</emphasis></para>
							<@com.emptyLine/>
						</#if>

						<#-- print a header for impurities -->
						<#if entityType == "impur">
							<para><emphasis role="underline">----- Relevant Impurity <emphasis role="bold">${entityName}</emphasis> -----</emphasis></para>
							<@com.emptyLine/>
						</#if>	

						<#-- loop over studies -->	
						<#list resultStudyList as study>
							<#assign sectionIndex = 0 /> <#-- index to govern a), b), c) printing  -->
							<#local studyIndex=studyIndex+1/>

							<#-- create section and assign a label
								note: role="NotInToc" makes the section not appear in the main ToCs, based on the default stylesheet
							-->
							<sect4 xml:id="${study.documentKey.uuid!}" label="/${studyIndex}" role="NotInToc">

								<#local docUrl=iuclid.webUrl.documentView(study.documentKey) />

								<#-- set the title with hyperlink to the actual document in IUCLID -->
								<title  role="HEAD-5"><ulink url="${docUrl}"><@com.text study.name ""/></ulink></title>

								<#-- 1. AppendixE header-->
								<para><emphasis role="HEAD-KeyA">1. Information on the study</emphasis></para>
								<@appendixEheader study entity sectionNb studyIndex/>

								<@com.emptyLine/>

								<#-- special case for analytical information (TO BE CHANGED AFTER FORMAT UPDATE) -->
								<#if study.documentSubType=="AnalyticalInformation" && keyPhyschem??>
									<@keyPhyschem.results_analyticalInformation study/>
								<#else>
									<#-- 2. AppendixE M&M and Results-->
									<para><emphasis role="HEAD-KeyA">2. Full summary of the study according to OECD format</emphasis></para>

									<#-- 2a. Executive summary -->
									<para><emphasis role="bold">${letterIndex(sectionIndex)} Executive summary:</emphasis></para>
									<#if study.hasElement("ApplicantSummaryAndConclusion.ExecutiveSummary") && study.ApplicantSummaryAndConclusion.ExecutiveSummary?has_content>
										<para style="background-color:#f7f7f7"><@com.richText study.ApplicantSummaryAndConclusion.ExecutiveSummary/></para>
									<#elseif study.hasElement("ApplicantSSummaryAndConclusion.ExecutiveSummary.ExecutiveSummary") && study.ApplicantSSummaryAndConclusion.ExecutiveSummary.ExecutiveSummary?has_content>
										<#--for intermediate effects-->
										<para style="background-color:#f7f7f7"><@com.richText study.ApplicantSSummaryAndConclusion.ExecutiveSummary.ExecutiveSummary/></para>
									</#if>

									<#assign sectionIndex += 1 />

									<@com.emptyLine/>

									<#-- 2b. Materials and methods
										There are different paths for materials and methods (or similar). The most standard
										is "MaterialsAndMethods" but in other documents there are different or additional ones.
									#-->

									<#-- background -->
									<#if study.hasElement("Background") && study.Background?has_content>
										<#if study.documentSubType=="EfficacyData">
											<para><emphasis role="bold">${letterIndex(sectionIndex)} Basic information:</emphasis></para>
											<@keyEfficacy.basicinfo study/>
											<#assign sectionIndex += 1 />								
										<#elseif study.Background.BackgroundInformation?has_content>
											<para><emphasis role="bold">${letterIndex(sectionIndex)} Background information:</emphasis></para>
											<para role="indent"><@com.text study.Background.BackgroundInformation/></para>
											<#assign sectionIndex += 1 />
										</#if>
									</#if>

									<#-- general information -->
									<#if study.hasElement("GeneralInformation") && study.documentSubType=="EffectivenessAgainstTargetOrganisms" && keyBioPropMicro??>
										<para><emphasis role="bold">${letterIndex(sectionIndex)} General information:</emphasis></para>
										<@keyBioPropMicro.generalInfo_effectivenessTargetOrg study/>
										<#assign sectionIndex += 1 />
									</#if>

									<#-- materials and methods -->
									<#if study.hasElement("MaterialsAndMethods")>
										<@appendixEmethods study entity includeRichTextFields/>
										<#assign sectionIndex += 1 />
									</#if>

									<#-- for intermediate effects -->
									<#if study.hasElement("EffectIdentification") && keyTox??>
										<@keyTox.intermediateEffectIdentification study/>
									</#if>

									<@com.emptyLine/>

									<#-- 2c. Results, including the overall remarks section -->
									<@appendixEresults study includeRichTextFields/>
									<#assign sectionIndex += 1 />

									<#if includeRichTextFields || includeImages>
										<@appendixEoverallRemarksAttachments study includeRichTextFields includeImages/>
									</#if>

									<@com.emptyLine/>

									<#-- 3. AppendixE Assessment and Conclusion -->
									<@appendixEconclusion study includeAnnotations />

								</#if>
							</sect4>
							<@com.emptyLine/>
						</#list>
					</#list>
				</#list>
			</#if>

		</#if>
	</#compress>
</#macro>

<#-- appendixEheader prints some summary information of study records, following Appendix E structure.
	It  prints information from:
		- Administrateive data block
		- Materials and methods (guidelines, and glp)
		- Literature references (including NoSID)
		- Change log
	NOTE: Additional values from administrative data or LITERATURE could be eventually added (e.g study period, type of study, company owner, etc)

	Inputs:
		- study: DOCUMENT object
		- entity: ENTITY object
		- sectionNb: SECTION number
		- studyNb: STUDY number
#-->
<#macro appendixEheader study entity sectionNb studyNb>
	<#compress>

		<#--Get first study report literature reference, if references exist (if more than one reference, sort and take first study report)-->
		<#local reference=getStudyReference(study)/>

		<table border="1">
			<title> </title>
			<col width="30%" />
			<col width="70%" />
			<tbody valign="middle">
			<tr>
				<th><?dbfo bgcolor="#d3d3d3" ?><emphasis role="bold">Data point</emphasis></th>
				<td>
					<#--  <#if docName_IotS??>${docName_IotS} <#elseif _prefix?? && _prefix=="C.">${_prefix}</#if>${sectionNb}/${studyNb}.  -->
					<#if _reqSetType??>${_reqSetType} <#elseif _prefix?? && _prefix=="C.">${_prefix}</#if>${sectionNb}/${studyNb}.
				</td>
			</tr>
			<tr>
				<th><?dbfo bgcolor="#d3d3d3" ?><emphasis role="bold">Study type</emphasis></th>
				<td>
					<#if study.documentSubType=="IntermediateEffects">
						intermediate effects: <@com.picklist study.AdministrativeData.StudyResultType/>
					<#elseif study.documentSubType=="AnalyticalInformation">
						spectra (UV/VIS, IR, NMR, MS), molar extinction at relevant wavelengths, optical purity
					<#else>
						<@com.picklist study.AdministrativeData.Endpoint/>
					</#if>
				</td>
			</tr>
			<tr>
				<th><?dbfo bgcolor="#d3d3d3" ?><emphasis role="bold">Testing period</emphasis></th>
				<td>
					<#if study.hasElement("AdministrativeData.StudyPeriodStartDate") && study.AdministrativeData.StudyPeriodStartDate?has_content>
						<#local testingStartDate><@com.value study.AdministrativeData.StudyPeriodStartDate "" /></#local>
					</#if>
					<#if study.hasElement("AdministrativeData.StudyPeriodEndDate") && study.AdministrativeData.StudyPeriodEndDate?has_content>
						<#local testingEndDate><@com.value study.AdministrativeData.StudyPeriodEndDate "" /></#local>
					</#if>
					<#if testingStartDate?? || testingEndDate??>
						<#if testingStartDate??> ${testingStartDate}</#if><#if testingEndDate??> - ${testingEndDate}</#if>
					</#if>
				</td>
			</tr>
			<tr>
				<th><?dbfo bgcolor="#d3d3d3" ?><emphasis role="bold">Report type</emphasis></th>
				<td>
					${getReferenceElement(reference, "GeneralInfo.LiteratureType")}
				</td>
			</tr>
			<tr>
				<th><?dbfo bgcolor="#d3d3d3" ?><emphasis role="bold">Report author</emphasis></th>
				<td>
					${getReferenceElement(reference, "GeneralInfo.Author")}
				</td>
			</tr>
			<tr>
				<th><?dbfo bgcolor="#d3d3d3" ?><emphasis role="bold">Report year</emphasis></th>
				<td>
					${getReferenceElement(reference, "GeneralInfo.ReferenceYear")}
				</td>
			</tr>
			<tr>
				<th><?dbfo bgcolor="#d3d3d3" ?><emphasis role="bold">Report title</emphasis></th>
				<td>
					${getReferenceElement(reference, "GeneralInfo.Name", true)}
				</td>
			</tr>
			<tr>
				<th><?dbfo bgcolor="#d3d3d3" ?><emphasis role="bold">Report No</emphasis></th>
				<td>
					${getReferenceElement(reference, "GeneralInfo.ReportNo")}
				</td>
			</tr>
			<tr>
				<th><?dbfo bgcolor="#d3d3d3" ?><emphasis role="bold">Document No</emphasis></th>
				<td>
					${getReferenceElement(reference, "GeneralInfo.CompanyOwnerStudyNo")}
				</td>
			</tr>
			<tr>
				<th><?dbfo bgcolor="#d3d3d3" ?><emphasis role="bold">Testing facility</emphasis></th>
				<td>
					${getReferenceElement(reference, "GeneralInfo.TestLab")}
				</td>
			</tr>
			<tr>
				<th><?dbfo bgcolor="#d3d3d3" ?><emphasis role="bold">Notification of Study Id</emphasis></th>
				<td>
					${getNoSid(reference)}
				</td>
			</tr>
			<tr>
				<th><?dbfo bgcolor="#d3d3d3" ?><emphasis role="bold">Guidelines followed in study</emphasis></th>
				<td>
					${getGuidelines(study)}
				</td>
			</tr>
			<tr>
				<th><?dbfo bgcolor="#d3d3d3" ?><emphasis role="bold">Deviations from current test guideline</emphasis></th>
				<td>
					${getGuidelinesDeviation(study)}
				</td>
			</tr>
			<tr>
				<th><?dbfo bgcolor="#d3d3d3" ?><emphasis role="bold">Previous evaluation</emphasis></th>
				<td>
					${getChangeLogStatus(study, entity)}
				</td>
			</tr>
			<tr>
				<th><?dbfo bgcolor="#d3d3d3" ?><emphasis role="bold">GLP/Officially recognised testing facilities</emphasis></th>
				<td>
					${getGLP(study)}
				</td>
			</tr>
			<tr>
				<th><?dbfo bgcolor="#d3d3d3" ?><emphasis role="bold">Adequacy of study</emphasis></th>
				<td>
					<#if study.hasElement("AdministrativeData.PurposeFlag")>
						<@com.picklist study.AdministrativeData.PurposeFlag/>
					<#else>
						n.a.
					</#if>
				</td>
			</tr>
			<tr>
				<th><?dbfo bgcolor="#d3d3d3" ?><emphasis role="bold">Acceptability/Reliability</emphasis></th>
				<td>
					<#if study.hasElement("AdministrativeData.Reliability")>
						<@com.picklist study.AdministrativeData.Reliability/><#if study.AdministrativeData.RationalReliability?has_content>: <@com.picklist picklistValue=study.AdministrativeData.RationalReliability printDescription=false/></#if>
					<#else>
						n.a.
					</#if>
				</td>
			</tr>
			</tbody>
		</table>
	</#compress>
</#macro>

<#-- appendixEconclusion prints information in the Applicant's summary and conclusion section

	Eventually, it could include the RMS opinion by including information from the annotation.
	
	Inputs:
		- study: DOCUMENT object
		- includeAnnotations: if true, include annotations by the RMS or Evaluator (TO BE IMPLEMENTED)
#-->
<#macro appendixEconclusion study includeAnnotations=false>
	<#compress>
		<para><emphasis role="HEAD-KeyA">3. Assessment and conclusion </emphasis> </para>

		<para><emphasis role="bold">a) Assessment and conclusion by applicant:</emphasis></para>

		<#-- Interpretation of results 
				NOTE: does not exist for all documents
		-->
		<#if study.hasElement("ApplicantSummaryAndConclusion.InterpretationOfResults") && study.ApplicantSummaryAndConclusion.InterpretationOfResults?has_content>
			<para>
				<emphasis role="bold">Interpretation of results: </emphasis><@com.picklist study.ApplicantSummaryAndConclusion.InterpretationOfResults/>
			</para>
			<#-- for intermediate effects -->
		<#elseif study.hasElement("ApplicantSSummaryAndConclusion.InterpretationOfResultsObservations") && study.ApplicantSSummaryAndConclusion.InterpretationOfResultsObservations?has_content>
			
			<para><emphasis role="bold">Interpretation of results:</emphasis></para>
			
			<#local int=study.ApplicantSSummaryAndConclusion.InterpretationOfResultsObservations/>
			
			<#if int.TypeOfResult?has_content>
				<para role="indent">Type of result: <@com.picklist int.TypeOfResult/></para>
			</#if>
			
			<#if int.EffectConcentrationChoice?has_content>
				<para role="indent">Effect concentration: <@com.picklist int.EffectConcentrationChoice/>
					<#if int.Concentration?has_content>
						= <@com.range int.Concentration/>
					</#if>
				</para>
			</#if>
			
			<#if int.Remarks?has_content>
				<para role="indent">Remarks: <@com.text int.Remarks/></para>
			</#if>

			<#if int.OverallResults?has_content>
				<para>
					<emphasis role="bold">Conclusion:</emphasis>
					<@com.text int.OverallResults/>
				</para>
			</#if>
		</#if>

		<#-- Validity of criteria
				NOTE: does not exist for all documents
		-->
		<#if study.hasElement("ApplicantSummaryAndConclusion.ValidityCriteria") && study.ApplicantSummaryAndConclusion.ValidityCriteria?has_content>
			<para role="small"><@printValidityCriteriaTable study.ApplicantSummaryAndConclusion.ValidityCriteria/></para>
		</#if>	

		<#-- Conclusion -->
		<#if study.hasElement("ApplicantSummaryAndConclusion.Conclusions") && study.ApplicantSummaryAndConclusion.Conclusions?has_content>
			<para><emphasis role="bold">Conclusion:</emphasis></para>
			<para role="indent"><@com.text study.ApplicantSummaryAndConclusion.Conclusions/> </para>
		</#if>

		<#--  Annotations  -->
		<#if includeAnnotations>
			<#local annotationsList = getAnnotationsListFromDoc(study)/>
		</#if>
		<#if annotationsList?? && annotationsList?size gt 0>
			<@printAnnotationsforDoc study/>
		<#else>
			<#local boxTitle = "b) Assessment and conclusion by RMS:">
			<#if workingContext?? && workingContext=="MRL">
				<#local boxTitle = "b) Assessment and conclusion by Evaluating Member State:">
			</#if>	
			<@com.printEmptyBox title=boxTitle color="#DAE9F7"/>
		</#if>
	</#compress>
</#macro>

<#-- Macro that prints the new Validity Criteria Repeatables Table -->
<#macro printValidityCriteriaTable vcTable bgcolor="#d3d3d3">

	<#-- Get Validity Criteria information -->
	<#local vcList = studyandsummaryCom.getValidityCriteria(vcTable)/>

    <#--  CREATE TABLE  -->
    <table border="1">
        <#--  Assign title  -->
        <title>Validity criteria</title>

        <#--  Set columns width  -->
        <col width="33%"/>
        <col width="33%"/>
        <col width="33%"/>
            
        <#--  Define table header  -->
        <thead align="center" valign="middle">
            <tr><?dbfo bgcolor="${bgcolor}" ?>
                <th>
                    <emphasis role="bold">Validity criteria</emphasis>
                </th>
                <th>
                    <emphasis role="bold">Observed value</emphasis>
                </th>
                <th>
                    <emphasis role="bold">Validity criteria fulfilled</emphasis>
                </th>                  
            </tr>
        </thead>
            
        <#--  Define table body  -->
        <tbody valign="middle">
            <#--  Iterate over rows  -->
            <#list vcList as vcHash>
                <tr>
                    <#-- Validity criteria -->
                    <td>
                        ${vcHash['vc']}
                    </td>

                    <#-- Observed value -->
                    <td>
                        ${vcHash['ov']}    
                    </td>

                    <#-- Validity criteria fulfilled (with remarks) -->
                    <td>
                        ${vcHash['vcf']}
                    </td>                       
                </tr>
            </#list>
        </tbody>
    </table>
</#macro>

<#-- appendixEmethods prints the materials and methods section of a study. 
	
	The part on test material is common to most templates and it's printed using this macro
	(including any fields contained within this path, which can very across documents).
	Specific m&m fields are printed using macros of the corresponding common_module for each 
	section (Tox, Ecotox, etc), depending on the document subtype.

	Printing the rich-text field on "Any other information on material and methods incl. tables" can be activated or deactivated.

	Inputs:
	- study: document object
	- includeRichTextFields: if true, it prints rich-text sections like "Any other results incl. tables"
-->
<#macro appendixEmethods study entity includeRichTextFields=true>

	<para><emphasis role="bold">${letterIndex(sectionIndex)} Materials and methods:</emphasis></para>

	<#-- Test material and additional test material -->
	<#if study.hasElement("MaterialsAndMethods.TestMaterials")>
		<#local testMat=study.MaterialsAndMethods.TestMaterials/>
		<#if testMat.TestMaterialInformation?has_content>
			<para><emphasis role="bold">Test material:</emphasis></para>
			<@studyandsummaryCom.testMaterialInformation documentKey=testMat.TestMaterialInformation entity=entity/>
		</#if>	

		<#if testMat.AdditionalTestMaterialInformation?has_content>
			<#local testMatIndex=0/>
			<para><emphasis role="bold">Additional test material(s):</emphasis></para>
			<#list testMat.AdditionalTestMaterialInformation as addTestMat>
				<#local testMatIndex=testMatIndex+1/>
				<@studyandsummaryCom.testMaterialInformation documentKey=addTestMat entity=entity testMatIndex=testMatIndex/>
				<#if addTestMat?has_next><@com.emptyLine/></#if>
			</#list>
		</#if>

		<#-- additional details on test materials -->
		<#if testMat.SpecificDetailsOnTestMaterialUsedForTheStudy?has_content>
			<@com.emptyLine/>
			<informaltable frame="none">
				<col width="25%" />
				<col width="75%" />
				<tbody>
					<tr>
						<td>Specific details on test material(s):</td>
						<td>
							<@com.value testMat.SpecificDetailsOnTestMaterialUsedForTheStudy/>
						</td>	
					</tr>
				</tbody>
			</informaltable>
		</#if>

		<#-- additional details on test materials CONFIDENTIAL -->
		<#--  <#if testMat.SpecificDetailsOnTestMaterialUsedForTheStudyConfidential?has_content>
			<@com.emptyLine/>
			<informaltable frame="none">
				<col width="25%" />
				<col width="75%" />
				<tbody>
					<tr>
						<td>Specific details [confidential]:</td>
						<td>
							<@com.value testMat.SpecificDetailsOnTestMaterialUsedForTheStudyConfidential/>
						</td>	
					</tr>
				</tbody>
			</informaltable>
		</#if>  -->

		<#-- print any children fields of the testMaterial, if they exist (NOT repeatable tables/lists) 
		NOTE: in Fate - FieldStudies and Tox - BasicToxicokinetics additional repeatable 
		test material fields are handled in specific method macros below.-->
		<@com.children path=testMat exclude=["TestMaterialInformation", "SpecificDetailsOnTestMaterialUsedForTheStudy",
												"SpecificDetailsOnTestMaterialUsedForTheStudyConfidential", "AdditionalTestMaterialInformation", 
												"DetailsOnSoilCharacteristics"] style="none" tabular=true/>
	</#if>

	<#-- specific methods: uses global variable _section -->
	<#if _section??>
		<#if _section=="Tox" && keyTox??>
			<@keyTox.toxMethodPPP study/>
		<#elseif _section=="Ecotox" && keyEcotox??>
			<@keyEcotox.ecotoxMethod study/>
		<#elseif _section=="Physchem" && keyPhyschem??>
			<@keyPhyschem.physchemMethod study/>
		<#elseif _section=="Residues" && keyRes??>
			<@keyRes.residuesMethod study/>
		<#elseif _section=="Anmeth" && keyAnMeth??>
			<@keyAnMeth.analyticalMethodsMethod study/>
		<#elseif _section=="Fate" && keyFate??>
			<@keyFate.fateMethod study/>
		<#elseif _section=="Efficacy" && keyEfficacy??>
			<@keyEfficacy.efficacyMethod study/>
		</#if>
	</#if>

	<#--  (Q)SAR model and software block -->
	<#if study.MaterialsAndMethods.hasElement("ModelAndSoftware") && study.MaterialsAndMethods.ModelAndSoftware?has_content>
		<@iuclid.label for=study.MaterialsAndMethods.ModelAndSoftware var="modSoftLabel"/>

		<para><emphasis role="bold">${modSoftLabel}:</emphasis></para>
		<@studyandsummaryCom.methods_qsar study.MaterialsAndMethods.ModelAndSoftware />
	</#if>

	<#if includeRichTextFields>
		<#--Any other information-->
		<#if study.MaterialsAndMethods.hasElement("AnyOtherInformationOnMaterialsAndMethodsInclTables") && study.MaterialsAndMethods.AnyOtherInformationOnMaterialsAndMethodsInclTables.OtherInformation?has_content>
			<para><emphasis role="bold">Other information:</emphasis></para>
			<para><para style="background-color:#f7f7f7" role="indent"><@com.richText study.MaterialsAndMethods.AnyOtherInformationOnMaterialsAndMethodsInclTables.OtherInformation/></para></para>
		</#if>
	</#if>
</#macro>

<#-- appendixEresults prints the results section of a study. 
	
	Specific result fields are printed using macros of the corresponding common_module for each 
	section (Tox, Ecotox, etc), depending on the document subtype.

	Printing the rich-text field on "Any other information on results incl. tables" can be activated or deactivated.

	Inputs:
	- study: document object
	- includeRichTextFields: if true, it prints rich-text sections like "Any other information on results incl. tables"
-->
<#macro appendixEresults study includeRichTextFields=true>
	<#compress>
		<para><emphasis role="bold">${letterIndex(sectionIndex)} Results:</emphasis></para>

		<#-- 1. Section-dependent specific result lists 
			NOTE: it seems in some cases ("TechnicalCharacteristics") this section has a different path
		-->

		<#--2. Physchem-->
		<#if study.documentSubType=='Melting'>
			<#if study.ResultsAndDiscussion.MeltingPoint?has_content>
				<para><emphasis role="bold">Melting point:</emphasis></para>
				<@keyPhyschem.meltingPointList studyandsummaryCom.orderByKeyResult(study.ResultsAndDiscussion.MeltingPoint)/>
			</#if>
		<#elseif study.documentSubType=='BoilingPoint'>
			<#if study.ResultsAndDiscussion.BoilingPoint?has_content>
				<para><emphasis role="bold">Boiling point:</emphasis></para>
				<@keyPhyschem.boilingPointList studyandsummaryCom.orderByKeyResult(study.ResultsAndDiscussion.BoilingPoint)/>
			</#if>
		<#elseif study.documentSubType=='Vapour'>
			<#if study.ResultsAndDiscussion.Vapourpr?has_content>
				<para><emphasis role="bold">Vapour pressure:</emphasis></para>
				<@keyPhyschem.vapourprList studyandsummaryCom.orderByKeyResult(study.ResultsAndDiscussion.Vapourpr)/>
			</#if>
			<#if study.ResultsAndDiscussion.VapourPressureAt50CIfRelevantForClassificationOfGasUnderPressure?has_content>
				<para><emphasis role="bold">Vapour pressure at 50°C:</emphasis></para>
				<@keyPhyschem.vapourprAt50List studyandsummaryCom.orderByKeyResult(study.ResultsAndDiscussion.VapourPressureAt50CIfRelevantForClassificationOfGasUnderPressure)/>
			</#if>
			<#if study.ResultsAndDiscussion.Transition?has_content>
				<para><emphasis role="bold">Transition / decomposition:</emphasis></para>
				<@keyPhyschem.vapourTransitionList studyandsummaryCom.orderByKeyResult(study.ResultsAndDiscussion.Transition)/>
			</#if>
		<#elseif study.documentSubType=="HenrysLawConstant">
			<#if study.ResultsAndDiscussion.HenrysLawConstantH?has_content>
				<para><emphasis role="bold">Henry's Law Constant H:</emphasis></para>
				<@keyPhyschem.henrysLawConstantHList studyandsummaryCom.orderByKeyResult(study.ResultsAndDiscussion.HenrysLawConstantH)/>
			</#if>
		<#elseif study.documentSubType=="GeneralInformation">
			<#if study.ResultsAndDiscussion.SubstancePhysicalState?has_content>
				<para><emphasis role="bold">Physical state at 20°C and 1013 hPa:</emphasis></para>
				<para role="indent"><@com.picklist study.ResultsAndDiscussion.SubstancePhysicalState/></para>
			</#if>
			<#if study.ResultsAndDiscussion.SubstanceType?has_content>
				<para><emphasis role="bold">Substance type:</emphasis></para>
				<para role="indent"><@com.picklist study.ResultsAndDiscussion.SubstanceType/></para>
			</#if>
			<#if study.ResultsAndDiscussion.FormBlock?has_content>
				<para><emphasis role="bold">Form / colour/ odour:</emphasis></para>
				<@keyPhyschem.appearanceList studyandsummaryCom.orderByKeyResult(study.ResultsAndDiscussion.FormBlock)/>
			</#if>
		<#elseif study.documentSubType=="WaterSolubility">
			<#if study.ResultsAndDiscussion.WaterSolubility?has_content>
				<para><emphasis role="bold">Water solubility:</emphasis></para>
				<@keyPhyschem.waterSolubilityList studyandsummaryCom.orderByKeyResult(study.ResultsAndDiscussion.WaterSolubility)/>
			</#if>
			<#if study.ResultsAndDiscussion.SolubilityOfMetalIonsInAqueousMedia?has_content>
				<para><emphasis role="bold">Solubility of metal ions in aqueous media:</emphasis></para>
				<@keyPhyschem.solubilityMetalList studyandsummaryCom.orderByKeyResult(study.ResultsAndDiscussion.SolubilityOfMetalIonsInAqueousMedia)/>
			</#if>
			<#if study.ResultsAndDiscussion.DetailsOnResults?has_content>
				<para><emphasis role="bold">Details on results:</emphasis></para>
				<para role="indent"><@com.text study.ResultsAndDiscussion.DetailsOnResults/></para>
			</#if>
		<#elseif study.documentSubType=="SolubilityOrganic">
			<#if study.ResultsAndDiscussion.SolubilityOrganic?has_content>
				<para><emphasis role="bold">Solubility in organic solvents / fat solubility:</emphasis></para>
				<@keyPhyschem.solubilityOrganicList studyandsummaryCom.orderByKeyResult(study.ResultsAndDiscussion.SolubilityOrganic)/>
			</#if>
			<#if study.ResultsAndDiscussion.TestSubstanceTable?has_content>
				<para><emphasis role="bold">Test substance stable:</emphasis></para>
				<para role="indent"><@com.picklist study.ResultsAndDiscussion.TestSubstanceTable/></para>
			</#if>
			<#if study.ResultsAndDiscussion.DetailsOnResults?has_content>
				<para><emphasis role="bold">Details on results:</emphasis></para>
				<para role="indent"><@com.text study.ResultsAndDiscussion.DetailsOnResults/></para>
			</#if>
		<#elseif study.documentSubType=="Partition">
			<#if study.ResultsAndDiscussion.Partcoeff?has_content>
				<para><emphasis role="bold">Partition coefficient:</emphasis></para>
				<@keyPhyschem.partcoeffList studyandsummaryCom.orderByKeyResult(study.ResultsAndDiscussion.Partcoeff)/>
			</#if>
			<#if study.ResultsAndDiscussion.DetailsOnResults?has_content>
				<para><emphasis role="bold">Details on results:</emphasis></para>
				<para role="indent"><@com.text study.ResultsAndDiscussion.DetailsOnResults/></para>
			</#if>
		<#elseif study.documentSubType=="DissociationConstant">
			<#if study.ResultsAndDiscussion.DissociatingProperties?has_content>
				<para><emphasis role="bold">Dissociating properties:</emphasis></para>
				<para role="indent"><@com.picklist study.ResultsAndDiscussion.DissociatingProperties/></para>
			</#if>
			<#if study.ResultsAndDiscussion.IdentityOfDissociatedSpecies?has_content>
				<para><emphasis role="bold">Identity of dissociated species:</emphasis></para>
				<para role="indent"><@com.value study.ResultsAndDiscussion.IdentityOfDissociatedSpecies/></para>
			</#if>
			<#if study.ResultsAndDiscussion.DissociationConstant?has_content>
				<para><emphasis role="bold">Dissociation constant:</emphasis></para>
				<@keyPhyschem.dissociationConstantList studyandsummaryCom.orderByKeyResult(study.ResultsAndDiscussion.DissociationConstant)/>
			</#if>
		<#elseif study.documentSubType=="Flammability">
			<#if study.ResultsAndDiscussion.FlammableGasesLowerAndUpperExplosionLimit?has_content>
				<para><emphasis role="bold">Flammable gasses (lower and upper explosion limits):</emphasis></para>
				<@keyPhyschem.flammableGasesList studyandsummaryCom.orderByKeyResult(study.ResultsAndDiscussion.FlammableGasesLowerAndUpperExplosionLimit)/>
			</#if>
			<#if study.ResultsAndDiscussion.Aerosols?has_content>
				<para><emphasis role="bold">Aerosols:</emphasis></para>
				<@keyPhyschem.aerosolsList studyandsummaryCom.orderByKeyResult(study.ResultsAndDiscussion.Aerosols)/>
			</#if>
			<#if study.ResultsAndDiscussion.FlammableSolids?has_content>
				<para><emphasis role="bold">Flammable solids:</emphasis></para>
				<@keyPhyschem.flammableSolidsList studyandsummaryCom.orderByKeyResult(study.ResultsAndDiscussion.FlammableSolids)/>
			</#if>
			<#if study.ResultsAndDiscussion.PyrophoricSolids?has_content>
				<para><emphasis role="bold">Pyrophoric solids:</emphasis></para>
				<@keyPhyschem.pyrophoricSolidsList studyandsummaryCom.orderByKeyResult(study.ResultsAndDiscussion.PyrophoricSolids)/>
			</#if>
			<#if study.ResultsAndDiscussion.PyrophoricLiquids?has_content>
				<para><emphasis role="bold">Pyrophoric liquid:</emphasis></para>
				<@keyPhyschem.pyrophoricLiquidList studyandsummaryCom.orderByKeyResult(study.ResultsAndDiscussion.PyrophoricLiquids)/>
			</#if>
			<#if study.ResultsAndDiscussion.SelfHeatingSubstancesMixtures?has_content>
				<para><emphasis role="bold">Self-heating substances/mixtures:</emphasis></para>
				<@keyPhyschem.selfHeatingSubstancesMixturesList studyandsummaryCom.orderByKeyResult(study.ResultsAndDiscussion.SelfHeatingSubstancesMixtures)/>
			</#if>
			<#if study.ResultsAndDiscussion.SubstancesMixturesWhichInContactWithWaterEmitFlammableGases?has_content>
				<para><emphasis role="bold">Substances/ mixture which in contact with water emit flammable gases:</emphasis></para>
				<@keyPhyschem.substancesMixturesWithWaterEmitFlammableGasesList studyandsummaryCom.orderByKeyResult(study.ResultsAndDiscussion.SubstancesMixturesWhichInContactWithWaterEmitFlammableGases)/>
			</#if>
		<#elseif study.documentSubType=="AutoFlammability">
			<#if study.ResultsAndDiscussion.AutoFlammability?has_content>
				<para><emphasis role="bold">Auto-ignition temperature (liquids / gases):</emphasis></para>
				<@keyPhyschem.autoFlammabilityList studyandsummaryCom.orderByKeyResult(study.ResultsAndDiscussion.AutoFlammability)/>
			</#if>
			<#if study.ResultsAndDiscussion.RelativeSelfIgnitionTemperatureSolids?has_content>
				<para><emphasis role="bold">Relative self-ignition temperature (solids):</emphasis></para>
				<@keyPhyschem.relativeSelfIgnitionTemperatureSolidsList studyandsummaryCom.orderByKeyResult(study.ResultsAndDiscussion.RelativeSelfIgnitionTemperatureSolids)/>
			</#if>
			<#if study.ResultsAndDiscussion.SelfIgnitionTemperatureOfDustAccumulation?has_content>
				<para><emphasis role="bold">Self-ignition temperature of dust accumulation:</emphasis></para>
				<@keyPhyschem.selfIgnitionTemperatureOfDustAccumulationList studyandsummaryCom.orderByKeyResult(study.ResultsAndDiscussion.SelfIgnitionTemperatureOfDustAccumulation)/>
			</#if>
		<#elseif study.documentSubType=="FlashPoint">
			<#if study.ResultsAndDiscussion.FlashPoint?has_content>
				<para><emphasis role="bold">Flash point:</emphasis></para>
				<@keyPhyschem.flashPointList studyandsummaryCom.orderByKeyResult(study.ResultsAndDiscussion.FlashPoint)/>
			</#if>
			<#if study.ResultsAndDiscussion.SustainingCombustibility?has_content>
				<para><emphasis role="bold">Sustaining combustibility:</emphasis></para>
				<@keyPhyschem.sustainingCombustabilityList studyandsummaryCom.orderByKeyResult(study.ResultsAndDiscussion.SustainingCombustibility)/>
			</#if>
		<#elseif study.documentSubType=='Explosiveness'>
			<#if study.ResultsAndDiscussion.SmallScalePreliminaryTests?has_content>
				<para><emphasis role="bold">Small-scale preliminary tests:</emphasis></para>
				<@keyPhyschem.smallScalePrelimTestsList studyandsummaryCom.orderByKeyResult(study.ResultsAndDiscussion.SmallScalePreliminaryTests)/>
			</#if>
			<#if study.ResultsAndDiscussion.ResultsOfTestSeriesForExplosives?has_content>
				<para><emphasis role="bold">All Test series for explosives:</emphasis></para>
				<#-- NOTE: i would change the way data display e.g. changing Test series for explosives by the actual test series-->
				<@keyPhyschem.resultsOfTestSeriesExplosivesList studyandsummaryCom.orderByKeyResult(study.ResultsAndDiscussion.ResultsOfTestSeriesForExplosives)/>
			</#if>
		<#elseif study.documentSubType=='SurfaceTension'>
			<#if study.ResultsAndDiscussion.SurfaceTension?has_content>
				<para><emphasis role="bold">SurfaceTension:</emphasis></para>
				<@keyPhyschem.surfaceTensionList studyandsummaryCom.orderByKeyResult(study.ResultsAndDiscussion.SurfaceTension)/>
			</#if>
		<#elseif study.documentSubType=="OxidisingProperties">
			<#if study.ResultsAndDiscussion.TestResultOxidisingGases?has_content>
				<para><emphasis role="bold">Oxidising gases:</emphasis></para>
				<@keyPhyschem.oxidisingGasesList studyandsummaryCom.orderByKeyResult(study.ResultsAndDiscussion.TestResultOxidisingGases)/>
			</#if>
			<#if study.ResultsAndDiscussion.TestResultsOxidisingLiquids?has_content>
				<para><emphasis role="bold">Oxidising liquids:</emphasis></para>
				<@keyPhyschem.oxidisingLiquidsList studyandsummaryCom.orderByKeyResult(study.ResultsAndDiscussion.TestResultsOxidisingLiquids)/>
			</#if>
			<#if study.ResultsAndDiscussion.TestResultsOxidisingSolids?has_content>
				<para><emphasis role="bold">Oxidising solids:</emphasis></para>
				<@keyPhyschem.oxidisingSolidsList studyandsummaryCom.orderByKeyResult(study.ResultsAndDiscussion.TestResultsOxidisingSolids)/>
			</#if>
		<#elseif study.documentSubType=="Ph">
			<#if study.ResultsAndDiscussion.phValue?has_content>
				<para><emphasis role="bold">pH:</emphasis></para>
				<@keyPhyschem.phList studyandsummaryCom.orderByKeyResult(study.ResultsAndDiscussion.phValue)/>
			</#if>
			<#if study.ResultsAndDiscussion.AcidityOrAlkalinity?has_content>
				<para><emphasis role="bold">Acidity / alkalinity:</emphasis></para>
				<@keyPhyschem.acidityList studyandsummaryCom.orderByKeyResult(study.ResultsAndDiscussion.AcidityOrAlkalinity)/>
			</#if>
		<#elseif study.documentSubType=="Viscosity">
			<#if study.ResultsAndDiscussion.Viscosity?has_content>
				<para><emphasis role="bold">Viscosity:</emphasis></para>
				<@keyPhyschem.viscosityList studyandsummaryCom.orderByKeyResult(study.ResultsAndDiscussion.Viscosity)/>
			</#if>
		<#elseif study.documentSubType=="Density">
			<#if study.ResultsAndDiscussion.Density?has_content>
				<para><emphasis role="bold">Density:</emphasis></para>
				<@keyPhyschem.densityList studyandsummaryCom.orderByKeyResult(study.ResultsAndDiscussion.Density)/>
			</#if>
		<#elseif study.documentSubType=="StorageStability">
			<#if study.ResultsAndDiscussion.Results?has_content>
				<para><@com.text study.ResultsAndDiscussion.Results/></para>
			</#if>
			<#if study.ResultsAndDiscussion.TransformationProducts?has_content || study.ResultsAndDiscussion.IdentityTransformation?has_content>
				<para><emphasis role="bold">Transformation products: </emphasis><@com.picklist study.ResultsAndDiscussion.TransformationProducts/><?linebreak?>
					<@keyPhyschem.transformationList studyandsummaryCom.orderByKeyResult(study.ResultsAndDiscussion.IdentityTransformation)/>
				</para>
			</#if>
			<#if study.ResultsAndDiscussion.ContainerMaterial?has_content>
				<para><emphasis role="bold">Reactivity towards container material:</emphasis></para>
				<@com.picklistMultiple study.ResultsAndDiscussion.ContainerMaterial/>
			</#if>

		<#elseif study.documentSubType=="StabilityThermal">
			<#if study.ResultsAndDiscussion.ThermalStability.TestSubstanceThermallyStable?has_content || study.ResultsAndDiscussion.ThermalStability.OperatingTemperature?has_content ||
					study.ResultsAndDiscussion.ThermalStability.Sublimation?has_content || study.ResultsAndDiscussion.ThermalStability.TransformationProducts?has_content || study.ResultsAndDiscussion.ThermalStability.IdentityTransformation?has_content >
				<para><emphasis role="HEAD-WoutNo">Thermal stability:</emphasis></para>
				<#if study.ResultsAndDiscussion.ThermalStability.TestSubstanceThermallyStable?has_content>
					<para>Test substance thermally stable: <@com.picklist study.ResultsAndDiscussion.ThermalStability.TestSubstanceThermallyStable/></para>
				</#if>
				<#if study.ResultsAndDiscussion.ThermalStability.OperatingTemperature?has_content>
					<para><emphasis role="bold">Operating temperature: </emphasis>
						<@keyPhyschem.operatingTempList studyandsummaryCom.orderByKeyResult(study.ResultsAndDiscussion.ThermalStability.OperatingTemperature)/>
					</para>
				</#if>
				<#if study.ResultsAndDiscussion.ThermalStability.Sublimation?has_content>
					<para>Sublimation:<@com.picklist study.ResultsAndDiscussion.ThermalStability.Sublimation/></para>
				</#if>
				<#if study.ResultsAndDiscussion.ThermalStability.TransformationProducts?has_content || study.ResultsAndDiscussion.ThermalStability.IdentityTransformation?has_content>
					<para><emphasis role="bold">Transformation products: </emphasis><@com.picklist study.ResultsAndDiscussion.ThermalStability.TransformationProducts/><?linebreak?>
						<@keyPhyschem.transformationList studyandsummaryCom.orderByKeyResult(study.ResultsAndDiscussion.ThermalStability.IdentityTransformation)/>
					</para>
				</#if>
			</#if>
			<#if study.ResultsAndDiscussion.StudyOnStability.StableToSunlight?has_content>
				<para><emphasis role="HEAD-WoutNo">Stability to sunlight</emphasis></para>
				<para>Test substance stable to sunlight: <@com.picklist study.ResultsAndDiscussion.StudyOnStability.StableToSunlight/></para>
			</#if>
			<#if study.ResultsAndDiscussion.StabilityToMetals.StableToMetals?has_content>
				<para><emphasis role="HEAD-WoutNo">Stability to metals</emphasis></para>
				<para>Test substance stable to metals / metal ions: <@com.picklist study.ResultsAndDiscussion.StabilityToMetals.StableToMetals/></para>
			</#if>
		<#elseif study.documentSubType=="DegreeOfDissolutionAndDilutionStability">
			<#if study.ResultsAndDiscussion.DegreeOfDissolution?has_content>
				<para><emphasis role="bold">Degree of dissolution: </emphasis>
					<@keyPhyschem.degreeDissolutionList studyandsummaryCom.orderByKeyResult(study.ResultsAndDiscussion.DegreeOfDissolution)/>
				</para>
			</#if>
			<#if study.ResultsAndDiscussion.DilutionStability?has_content || study.ResultsAndDiscussion.DilutionStabilityMT41?has_content>
				<para><emphasis role="bold">Dilution stability: </emphasis>
					<@keyPhyschem.dilutionStabilityList studyandsummaryCom.orderByKeyResult(study.ResultsAndDiscussion.DilutionStabilityMT41)/>
					<@keyPhyschem.dilutionStability2List studyandsummaryCom.orderByKeyResult(study.ResultsAndDiscussion.DilutionStability)/>
				</para>
			</#if>
			<#if study.ResultsAndDiscussion.DetailsOnResults?has_content>
				<para>Details:
					<@com.text study.ResultsAndDiscussion.DetailsOnResults/>
				</para>
			</#if>
		<#elseif study.documentSubType=="AdditionalPhysicoChemical">
			<#if study.ResultsAndDiscussion.Results?has_content>
				<para><@com.text study.ResultsAndDiscussion.Results/></para>
			</#if>

		<#elseif study.documentSubType=="EffectivenessAgainstTargetOrganisms">
			<#if study.ResultsAndDiscussion.DetailsOnResults?has_content>
				<para>Details:</para>
				<para role="indent"><@com.text study.ResultsAndDiscussion.DetailsOnResults/></para>
			</#if>

		<#--NOTE: this is for microorg, but at least in chemicals, it's used in ecotox-->
		<#elseif study.documentSubType=="ToxicityToOtherAboveGroundOrganisms" && keyBioPropMicro??>
			<#if study.ResultsAndDiscussion.EffectConcentrations?has_content>
				<para>Effect concentrations:
					<@keyBioPropMicro.effectList study.ResultsAndDiscussion.EffectConcentrations/>
				</para>
			</#if>
			<#if study.ResultsAndDiscussion.ResultsDetails?has_content>
				<para>Details:
					<@com.text study.ResultsAndDiscussion.ResultsDetails/>
				</para>
			</#if>
			<#if study.ResultsAndDiscussion.ResultsRefSubstance?has_content>
				<para>Results with reference substance:
					<@com.text study.ResultsAndDiscussion.ResultsRefSubstance/>
				</para>
			</#if>
			<#if study.ResultsAndDiscussion.Statistics?has_content>
				<para>Statistics:
					<@com.text study.ResultsAndDiscussion.Statistics/>
				</para>
			</#if>

		<#--5. Tox-->
		<#elseif study.documentSubType=="BasicToxicokinetics">
			<@keyTox.results_basicToxicokinetics study/>

		<#elseif study.documentSubType=="AcuteToxicityOral" || study.documentSubType=="AcuteToxicityDermal" ||  study.documentSubType=="AcuteToxicityInhalation" || study.documentSubType=="AcuteToxicityOtherRoutes" >
			<@keyTox.results_acuteToxicity study/>

		<#elseif study.documentSubType=="EyeIrritation">
			<@keyTox.results_eyeIrritation study/>

		<#elseif study.documentSubType=="SkinIrritationCorrosion">
			<@keyTox.results_skinIrritation study/>

		<#elseif study.documentSubType=="SkinSensitisation">
			<@keyTox.results_skinSensitisation study />

		<#elseif study.documentSubType=="PhototoxicityVitro">
			<@keyTox.results_phototoxicity study/>

		<#elseif study.documentSubType=="RepeatedDoseToxicityOral" || study.documentSubType=="RepeatedDoseToxicityInhalation" || study.documentSubType=="RepeatedDoseToxicityDermal" ||
					study.documentSubType=="RepeatedDoseToxicityOther" || study.documentSubType=="Carcinogenicity" || study.documentSubType=="Neurotoxicity" || study.documentSubType=="Immunotoxicity">
			<@keyTox.results_repDoseCarciNeuroImmuno study/>

		<#elseif study.documentSubType=="GeneticToxicityVitro" || study.documentSubType=="GeneticToxicityVivo">
			<@keyTox.results_geneticToxicity study/>
			<#-- NOTE: changed from existing macros-->

		<#elseif study.documentSubType=="ToxicityReproduction">
			<@keyTox.results_toxicityReproduction study/>

		<#elseif study.documentSubType=="ToxicityReproductionOther">
			<@keyTox.results_toxicityReproductionOther study/>

		<#elseif study.documentSubType=="DevelopmentalToxicityTeratogenicity">
			<@keyTox.results_developmentalToxicity study/>

		<#elseif study.documentSubType=="ToxicEffectsLivestock">
			<@keyTox.results_effectsLivestock study/>

		<#elseif study.documentSubType=="SpecificInvestigations">
			<@keyTox.results_specificInvestigation study/>

		<#elseif study.documentSubType=="EndocrineDisrupterMammalianScreening">
			<@keyTox.results_endocrineDisrupterMammalianScreening study/>

		<#elseif study.documentSubType=="EndocrineDisrupterScreeningInVitro">
			<@keyTox.results_endocrineDisrupterScreeningInVitro study/>

		<#elseif study.documentSubType=="DermalAbsorption">
			<@keyTox.results_dermalAbsorption study/>

		<#elseif study.documentSubType=="HealthSurveillanceData" ||  study.documentSubType=="ExposureRelatedObservationsOther" || study.documentSubType=="EpidemiologicalData" >
			<@keyTox.results_healthSurvExposureEpidemiological study/>

		<#elseif study.documentSubType=="DirectObservationsClinicalCases" || study.documentSubType="SensitisationData">
			<@keyTox.results_directObs study/>

		<#elseif study.documentSubType=="CellCultureStudy">
			<@keyTox.results_cellCulture study/>

		<#elseif study.documentSubType=="IntermediateEffects">
			<@keyTox.results_intermediateEffects study/>

		<#--8. Ecotox results -->
		<#elseif study.documentSubType=="ToxicityToBirds">
			<#--NOTE: probably could be merged with next-->
			<@keyEcotox.results_toxicityToBirds study/>

		<#elseif study.documentSubType=="ToxicityToOtherAboveGroundOrganisms" || study.documentSubType=="ShortTermToxicityToFish" ||
					study.documentSubType=="LongTermToxToFish" || study.documentSubType=="EndocrineDisrupterTestingInAqua" ||
					study.documentSubType=="ShortTermToxicityToAquaInv" || study.documentSubType=="LongTermToxicityToAquaInv"||
					study.documentSubType=="SedimentToxicity" || study.documentSubType=="ToxicityToAquaticAlgae" ||
					study.documentSubType=="ToxicityToAquaticPlant" || study.documentSubType=="ToxicityToTerrestrialArthropods" ||
					study.documentSubType=="ToxicityToSoilMacroorganismsExceptArthropods" ||
					study.documentSubType=="ToxicityToSoilMicroorganisms" || study.documentSubType=="ToxicityToTerrestrialPlants" ||
					study.documentSubType=="ToxicityToMicroorganisms" || study.documentSubType=="ToxicityToBees" ||
					study.documentSubType=="ToxicityToTerrestrialArthropodsOtherThanBees" || study.documentSubType=="ToxicityToSoilArthropods">
			<@keyEcotox.results_envToxicity study/>

		<#elseif study.documentSubType=="BioaccumulationAquaticSediment" || study.documentSubType=="BioaccumulationTerrestrial">
			<@keyEcotox.results_bioaccumulation study/>

		<#-- 6. Residues-->
		<#elseif study.documentSubType=="StabilityOfResiduesInStoredCommod">
			<@keyRes.results_stabilityOfResiduesInStoredCommod study/>

		<#elseif study.documentSubType=="ResiduesInLivestock">
			<@keyRes.results_residuesInLivestock study/>

		<#elseif study.documentSubType=="NatureResiduesInProcessedCommod">
			<@keyRes.results_natureResiduesInProcessedCommod study/>

		<#elseif study.documentSubType=="MigrationOfResidues">
			<@keyRes.results_migrationOfResidues study/>

		<#elseif study.documentSubType=="AdditionalInfoOnResiduesInFood" || study.documentSubType=="ExpectedExposureAndProposedAcceptableResidues">
			<#if study.ResultsAndDiscussion.DetailsOnResults?has_content>
				<para>Details:</para>
				<para role="indent"><@com.text study.ResultsAndDiscussion.DetailsOnResults/></para>
			</#if>

		<#elseif study.documentSubType=="ResiduesInRotationalCrops">
			<@keyRes.results_residuesInRotationalCrops study/>

		<#elseif study.documentSubType=="MagnitudeResidInProcessedComm">
			<@keyRes.results_magnitudeResidInProcessedComm study/>

		<#elseif study.documentSubType=="MetabolismInCrops" || study.documentSubType=="MetabolismInLivestock">
			<@keyRes.results_metabolismLivestock study/>

		<#elseif study.documentSubType=="ResiduesProcessedCommodities">
			<@keyRes.results_residuesProcessedCommodities study/>

		<#--Analytical methods-->
		<#elseif study.documentSubType=="AnalyticalMethods">
			<@keyAnMeth.results_analyticalMethods study/>

		<#--FATE-->
		<#elseif study.documentSubType=="BiodegradationInSoil">
			<@keyFate.results_biodegradationInSoil study/>
			<#--NOTE: maybe could be combined with the other biodegradations-->

		<#elseif study.documentSubType=="PhotoTransformationInSoil" || study.documentSubType=="Phototransformation" ||
					study.documentSubType=="PhototransformationInAir">
			<@keyFate.results_phototransformation study/>
		
		<#elseif study.documentSubType=="FieldStudies">
			<@keyFate.results_fieldStudies study/>
		
		<#elseif study.documentSubType=="AdsorptionDesorption">
			<@keyFate.results_adsorptionDesorption study/>

		<#elseif study.documentSubType=="Hydrolysis">
			<@keyFate.results_hydrolysis study/>

		<#elseif study.documentSubType=="BiodegradationInWaterScreeningTests">
			<@keyFate.results_biodegradationWaterScreening study/>

		<#elseif study.documentSubType=="BiodegradationInWaterAndSedimentSimulationTests">
			<@keyFate.results_biodegradationWaterSedimentSimulation study/>

		<#elseif study.documentSubType=="MonitoringData">
			<@keyFate.results_monitoring study/>

		<#--Efficacy-->
		<#elseif study.documentSubType=="EfficacyData">
			<@keyEfficacy.results_efficacyData study/>

		<#--Analytical profile of batches-->
		<#elseif study.documentSubType=="AnalyticalProfileOfBatches">
			<@keyComp.batchAnalysisSummary study/>
		</#if>

		<#--  (Q)SAR predictions block -->
		<#if study.hasElement("ResultsAndDiscussion")>
			<#local study_results = study.ResultsAndDiscussion />
		<#elseif study.hasElement("ResultsAndDiscussions")>
			<#local study_results = study.ResultsAndDiscussions />
		</#if>

		<#if study_results??>
			<#if study_results.hasElement("AdditionalInformationAboutApplicabilityDomainAndReliabilityOfQSARPredictions") && study_results.AdditionalInformationAboutApplicabilityDomainAndReliabilityOfQSARPredictions?has_content>
				<@iuclid.label for=study_results.AdditionalInformationAboutApplicabilityDomainAndReliabilityOfQSARPredictions var="qsarLabel"/>

				<para><emphasis role="bold">${qsarLabel}:</emphasis></para>
				<@studyandsummaryCom.results_qsar study_results.AdditionalInformationAboutApplicabilityDomainAndReliabilityOfQSARPredictions />
			</#if>
		</#if>

		<#-- Without specific results: "AgedSorption", "OtherDistributionData", "TransportViaAir",
			"AdditionalInformationOnEnvironmentalFateAndBehaviour"-->

		<#-- 2. Other information including tables + other HTML fields -->
		<#if includeRichTextFields>
			<@appendixEresults_richText study=study excludeOtherInfoDocSubTypes=[]/><#-- before excluded: "MetabolismInCrops","MetabolismInLivestock"] -->
		</#if>
	</#compress>
</#macro>

<#-- appendixEresults_richText prints the rich text fields in the results section, in most cases:
		- ResultsAndDiscussion.AnyOtherInformationOnResultsInclTables.OtherInformation
	but sometimes additional Rich Text fields can be found in results sections, and these are also
	considered. Also a few times the path might be a bit different so it is considered e.g. "ResultsAndDiscussions" 
	instead of "ResultsAndDiscussion"

	Inputs:
	- study: document object
	- excludeOtherInfoDocSubTypes: list of document sub types to exclude for printing the html in "Other information"
-->
<#macro appendixEresults_richText study excludeOtherInfoDocSubTypes>
	<#compress>

	<#-- get the right path for results -->
	<#if study.hasElement("ResultsAndDiscussion")>
		<#local resultsPath=study.ResultsAndDiscussion/>
	<#elseif study.hasElement("ResultsAndDiscussions")>
		<#local resultsPath=study.ResultsAndDiscussions/>
	</#if>

	<#-- get the right path to the main Rich Text field: OtherInformation is the most common field but there might be other 
		 also, exclude paths that should NOT be output
	-->
	<#if resultsPath.hasElement("AnyOtherInformationOnResultsInclTables")>
		
		<#if resultsPath.AnyOtherInformationOnResultsInclTables.hasElement("OtherInformation") &&
				resultsPath.AnyOtherInformationOnResultsInclTables.OtherInformation?has_content &&
				!(excludeOtherInfoDocSubTypes?seq_contains(study.documentSubType))>

			<para><emphasis role="bold">Other information:</emphasis></para>
			<para><para style="background-color:#f7f7f7" role="indent"><@com.richText resultsPath.AnyOtherInformationOnResultsInclTables.OtherInformation/></para></para>
		</#if>
	</#if>

	<#-- Iterate over all other HTML fields if any -->
	<#--  <#list resultsPath?children as child>
		<#if child?node_type?contains("html") && child?node_name!="OtherInformation" && child?has_content>
			<#assign childName=child?node_name?replace("([A-Z]{1})", " $1", "r")?lower_case?cap_first/>
			<para>${childName}:</para>
			<para><para style="background-color:#f7f7f7" role="indent"><@com.richText child/></para></para>
		</#if>
	</#list>  -->

	</#compress>

</#macro>

<#-- appendixEoverallRemarksAttachments prints the Overall Remarks section of ESR, which is rich-text.
	If selected, it prints images included in the attachment section.
	Eventually, this macro could be used to print metadata of the existing attachments 

	Inputs:
	- study: document object
	- includeImages: if true, print images of the attachments block
-->
<#macro appendixEoverallRemarksAttachments study includeRichTextFields=true includeImages=true>
	<#compress>

	<#if includeRichTextFields && study.OverallRemarksAttachments.RemarksOnResults?has_content>
		<para><emphasis role="bold">Overall remarks:</emphasis></para>
		<para style="background-color:#f7f7f7"><@com.richText study.OverallRemarksAttachments.RemarksOnResults/></para>
	</#if>
	
	<#if includeImages>
		<#if study.hasElement("OverallRemarksAttachments.AttachedBackgroundMaterial") && study.OverallRemarksAttachments.AttachedBackgroundMaterial?has_content>
			<para><emphasis role="bold">Illustrations / pictures / graphs / attachments:</emphasis></para>
			<#local attachIndex=0/>
			<#list study.OverallRemarksAttachments.AttachedBackgroundMaterial as row>
				<#local attachIndex = attachIndex+1/>
				<#if attachIndex gt 1><@com.emptyLine /></#if>

				<informaltable frame="none">
					<col width="25%" />
					<col width="75%" />
					<tbody>

						<tr>
							<td>#${attachIndex}</td>
							<td></td>
						</tr>
						

						<#if row.Type?has_content>
						<tr>
							<td>Attachment type:</td>
							<td><@com.value row.Type/></td>
						</tr>
						</#if>

						<#if row.Remarks?has_content>
						<tr>
							<td>Remarks:</td>
							<td><@com.value row.Remarks/></td>
						</tr>
						</#if>
					</tbody>
				</informaltable>
				
				<#if row.AttachedSanitisedDocsForPublication?has_content>
					<@com.printImage attachmentPath=row.AttachedSanitisedDocsForPublication imageWidthPerc=70 printTitle=false informal=true url=iuclid.webUrl.documentView(study.documentKey) printMetaDataIfNotImage=true/>
				<#else>
					<para>No sanitised version of attachment available.</para>	
				</#if>
			</#list>
		</#if>
	</#if>

	</#compress>

</#macro>

<#--getSummaryAttachments prints the Attachments section from a summary document object.

    Inputs:
    - summary:            The summary document object to extract attachments from.
    - paths:              A list of string paths to check for available attachment data in the summary.
    - printSummaryTitle:  (Optional, boolean) If true, prints the summary title with a link above the attachments. Default is true.
-->
<#function getSummaryAttachments summary paths printSummaryTitle=true>

	<#-- initialise output -->
	<#local attachments = ''/>

	<#-- iterate over possible paths and create full path when found -->
	<#local pathObject = studyandsummaryCom.getObjectFromPathOptions(summary, paths)/>

	<#-- get summary Title -->
	<#local summaryUrl=iuclid.webUrl.documentView(summary.documentKey) />
	<#local summaryName><@com.text summary.name ""/></#local>
	<#local summaryLink><ulink url="${summaryUrl}">${summaryName}</ulink></#local>

	<#if pathObject?has_content>
		<#local attachments>
			<#if printSummaryTitle><@com.emptyLine/><para>${summaryLink}</para></#if>
			<#local attachIndex=0/>
			<#list pathObject as row>
				<#local attachIndex = attachIndex+1/>
				<#if attachIndex gt 1><@com.emptyLine /></#if>

				<informaltable frame="none">
					<col width="25%" />
					<col width="75%" />
					<tbody>

						<tr>
							<td>#${attachIndex}</td>
							<td></td>
						</tr>

						<#if row.Remarks?has_content>
						<tr>
							<td>Remarks:</td>
							<td><@com.value row.Remarks/></td>
						</tr>
						</#if>
					</tbody>
				</informaltable>
				
				<#if row.AttachedSanitisedDocsForPublication?has_content>
					<@com.printImage attachmentPath=row.AttachedSanitisedDocsForPublication imageWidthPerc=70 printTitle=false informal=true url=summaryUrl printMetaDataIfNotImage=true/>
				<#else>
					<para>No sanitised version of attachment available.</para>	
				</#if>
			</#list>
		</#local>	
	</#if>

	<#return attachments/>
</#function>

<#-- resultStudyHashSummaryText prints a summary of the studies retrieved.

	In case of substances, it should indicate how many studies have been reported for the active substance 
	and each of the metabolites and impurities.
	In case of product, only how many studies in total.

	This distinction is based on an assumption: for a product there is only one entity in the hash, while
	for substances there might be more than one (active substance, which is the subject, and metabolites)

	Inputs:
	- studyHash: hash containing study results i.e. studyHash['st']
	- name: str with requirement name to print
-->
<#macro resultStudyHashSummaryText studyHash name>
	<#compress>
		<#local text=''/>
		<#local size=0/>

		<#-- count how many different entities are in studyHash -->
		<#local nMain=0><#if studyHash["main"]?has_content><#local nMain=1></#if>
		<#local nMetab=0><#if studyHash["metab"]?has_content><#local nMetab=studyHash["metab"]?keys?size></#if>
		<#local nImp=0><#if studyHash["impur"]?has_content><#local nImp=studyHash["impur"]?keys?size></#if>
		<#if (nMain + nMetab + nImp) gt 1>
			<#local includeText = true>
		</#if>

		<#list studyHash as entityType, entityStudyHash_st> <#-- "main", "metab", "impur" -->
			<#list entityStudyHash_st as entityName, resultData>
						
				<#local resultStudyList=resultData['records']/>

				<#local size = size + resultStudyList?size/>
				
				<#if includeText??> <#-- exists only if there is more than one entity in the studyHash -->
					<#local textEntity>
						<para role="indent">-
							${resultStudyList?size} for
							<#if entityType == "main">active substance<#else><#if entityType == "metab">metabolite<#else>relevant impurity</#if> ${entityName}</#if>
						</para>
					</#local>	
					<#local text = text + textEntity/>
				</#if>
			</#list>	
		</#list>

		${size} individual <#if size==1>study<#else>studies</#if> for ${name} <#if size==1>is<#else>are</#if> summarised below: ${text}

	</#compress>
</#macro>

<#--
	substanceEntitiesStudies prints all summaries and studies for a list of metabolites or impurities. 
	If selected, first all the summaries will be printed and then all the studies. 
	There are no sub-sections, and all documents are printed sequentially for EACH metabolite or impurity.

	This is used in the Tox section, only.
	NOTE: to be considered if to display results in the tox section as in the rest of the the sections i.e. together with the active substance.

	Inputs:
		- entityList: list of SUBSTANCE entities corresponding to metabolites or impurities
		- mixture: MIXTURE entity
		- activeSubstance: SUBSTANCE entity corresponding to the active substance
		- studySubTypes: string or list of strings defining document's sub type e.g. "AnalyticalMethods"
		- studyContext: list of hashmaps containing the conditions to be used to select which document(s) to output
			The hashmaps are definied in the ftls of the individual reports that use the appendixEstudies macro
			E.g. one example containing one condition for analytical methods for risk assessment is
				[{"path": "AdministrativeData.Endpoint", "val" : ["risk"], "qual" : "lk", "type" : "picklist"}] 
			Fur further description see 'filterStudyListbyContext' function
		- studyName: string with the name of the document to be displayed; if empty the name will be taken from the document subtype
		- summarySubTypes: list of summary document subtypes to be considered 
		- summaryMacroCall: name of the macro to be used to print summaries
		- includeStudies: bool; if true studies are printed. In combination with includeWaivers it can control if to print studies, waivers, or both. 
		- includeWaivers: bool; if true waivers are printed. In combination with includeStudies it can control if to print studies, waivers, or both. 
		- includeRichTextFields: bool; if true print rich text fields of materials and methods, results, and overall remarks sections
		- includeImages: bool; if true print attached images
		- relevantImp: bool, if true entityList is the list of relevant impurities else of metabolites
-->
<#macro substanceEntitiesStudies entityList=[] mixture="" activeSubstance="" studySubTypes=[] studyContext="" studyName="" summarySubTypes=[] summaryMacroCall="keyAppendixE.PPPsummary" 
				includeStudies=true includeWaivers=true includeRichTextFields=true, includeImages=true includeAnnotations=false sectionNb="" relevantImp=false>
	<#compress>

		<#-- if list of substance entities is empty, retrieve from the main mixture -->
		<#if (!entityList?has_content) && mixture?has_content>
			<#if relevantImp>
				<#local filterImpTypes = ["relevant"]/>
    			<#local entityList = com.getImpurities(_subject, filterImpTypes)/>
			<#else>
				<#if activeSubstance?has_content>
					<#local entityList=com.getMetabolites(mixture, activeSubstance, true)/>
				<#else>
					<#local entityList=com.getMetabolites(mixture)/>
				</#if>
			</#if>	
		</#if>

		<#if !(workingContext?? && workingContext == "MRL" || _section == "Ecotox")>
			<para><emphasis role="HEAD-KeyA"><#if relevantImp>Relevant Impurities<#else>Metabolites</#if></emphasis></para>
		</#if>	

		<#if entityList?has_content>

			<#-- create extendedSummarySubTypes to search summary documents with containsDoc -->
			<#local extendedSummarySubTypes = extendSubTypes(summarySubTypes)/>

			<#-- create extendedStudySubTypes to search study documents with containsDoc -->
			<#local extendedStudySubTypes = extendSubTypes(studySubTypes, false)/>

			<#-- find which substance entities have relevant information -->
			<#local relEntityList = []>
			<#list entityList as entity>
				<#local entityFound = false>
				<#if summarySubTypes?has_content && containsDoc(entity, extendedSummarySubTypes, "", false, "", false)>
					<#local relEntityList += [entity]>
					<#local entityFound = true>
				</#if>	
				<#if !entityFound && includeStudies && containsDoc(entity, extendedStudySubTypes, studyContext, false, "", false)>
					<#local relEntityList += [entity]>
				</#if>		
			</#list> 

			<@com.emptyLine/>

			<#-- print how many substance entities with relevant information have been found and list their names -->
			<para>
				<#local summaryText>There <#if entityList?size gt 1>are<#else> is</#if> ${entityList?size} dataset<#if entityList?size gt 1>s</#if> provided<#if activeSubstance?has_content>for ${activeSubstance.ChemicalName}</#if></#local>
				<#if relEntityList?size == 0>
					<#local summaryText>${summaryText}, but<#if entityList?size gt 1> none have<#else> it does not have any </#if> relevant information.</#local>		
				<#else>
					<#local summaryText>${summaryText}<#if relEntityList?size == 1 && entityList?size == 1>, and it has relevant information: <#elseif relEntityList?size == entityList?size>, all with relevant information: <#else>, of which ${relEntityList?size}<#if relEntityList?size == 1> has<#else> have</#if> relevant information: </#if></#local>
					<#if relEntityList?size gt 1>
						<#local summaryText>${summaryText}
							<#list relEntityList as entity>
								<para role="indent">
									- <command linkend="${entity.documentKey.uuid!}_${includeStudies?c}">${entity.ChemicalName}</command>
								</para>
							</#list>
						</#local>	
					<#else>
						<#local summaryText>${summaryText}<command linkend="${relEntityList[0].documentKey.uuid!}">${relEntityList[0].ChemicalName}</command></#local>
					</#if>
				</#if>
				${summaryText}
			</para>

			<@com.emptyLine/>
			
			<#if relEntityList?size gt 0> 
				<#-- iterate over the list of substance entities and get all studies -->
				<#list relEntityList as entity>
					<#if containsDoc(entity, extendedSummarySubTypes, "", false, "", false) || containsDoc(entity, extendedStudySubTypes, studyContext, false, "", false)>
						<#global substanceEntityIndex += 1/>
						<#-- create a section for each substance entity -->
						<sect3 xml:id="${entity.documentKey.uuid!}_${includeStudies?c}" label="-${substanceEntityIndex}">
							<title role="HEAD-3"><#if relevantImp>Relevant Impurity<#else>Metabolite</#if> ${entity.ChemicalName}</title>

							<#-- Get all summaries -->
							<#if summaryMacroCall?has_content>
								<#local summaryFirst=true/>
								<#list summarySubTypes as summarySubType>
									<#local endSummary>ENDPOINT_SUMMARY.${summarySubType}</#local>
									<#local flexSummary>FLEXIBLE_SUMMARY.${summarySubType}</#local>
									<#if containsDoc(entity, [endSummary, flexSummary], "", false, "", false)>

										<#if _section?? && _section="Ecotox" && summarySubType?contains("EU_PPP")>
											<#local summaryMacroCall = "keyEcotox.ecotoxPPPsummary"/>
										</#if>

										<#local summaryCallString = "<@" + summaryMacroCall + " subject=entity docSubTypes=summarySubType merge=false includeMetabolites=false includeImpurities=false/>"/>
										<#local summaryCall=summaryCallString?interpret/>
										<#local summary><@summaryCall/></#local>
	
										<#if summary?has_content>
											<#if summaryFirst>
												<#local summaryFirst=false/>
												<@com.emptyLine/>
												<para><emphasis role="HEAD-KeyA">Summaries</emphasis></para>
												<para>Summaries for studies on <#if relevantImp>relevant impurity<#else>metabolite</#if> ${entity.ChemicalName} are provided below:</para>
											</#if>
								
											<#local summaryDocFullName=summarySubType?replace("_EU_PPP", "")?replace("([A-Z]{1})", " $1", "r")?lower_case?cap_first/>
											<#if (summarySubTypes?size > 1)>
												<@com.emptyLine/>
												${summary?replace('<emphasis role="HEAD-KeyA">Summary</emphasis>',
															'-- for <emphasis role="HEAD-KeyA">${summaryDocFullName}</emphasis>:')}
											<#else>
												${summary?replace('<para><emphasis role="HEAD-KeyA">Summary</emphasis></para>', '')}
											</#if>
	
										</#if>
										<#if !summaryFirst && summarySubType_index==(summarySubTypes?size-1)>
											<@com.emptyLine/>
										</#if>
									</#if>
								</#list>
							</#if>

							<#local entitySectionNb = sectionNb + "-" + substanceEntityIndex + "."/>
							<#if relevantImp>
								<#local entityType="relevant impurity"/>
								<#local entitiesHashKey="impur"/>
							<#else>
								<#local entityType="metabolite"/>
								<#local entitiesHashKey="metab"/>
							</#if>
							<#--Get all studies-->
							<@keyAppendixE.appendixEstudies subject=entity
					    		docSubTypes=studySubTypes context=studyContext name="Toxicity studies of ${entityType} ${entity.ChemicalName}"
					    		includeMetabolites=false includeStudies=includeStudies includeWaivers=includeWaivers includeRichTextFields=includeRichTextFields 
					    		includeImages=includeImages includeAnnotations=includeAnnotations sectionNb=entitySectionNb includeImpurities=false entitiesHashKey=entitiesHashKey/>
						</sect3>
					</#if>	
				</#list>
			</#if>	
		<#else>
			<para>There are no datasets provided.</para>	
		</#if>
	</#compress>
</#macro>

<#-- AppendixE (MODIFIED: taking into consideration impurities to create a nested hash structure)

	createStudyHash creates a hash of studies for results and data waivers for all entities, taking into account
	the required context and/or section, name to select studies/waivers.

	Inputs:
		- entities: list of ENTITIES that documents belong to
		- docSubTypes: string or list of strings defining document's sub type e.g. "AnalyticalMethods"
		- context: list of hashmaps containing the conditions to be used to select which document(s) to output
			The hashmaps are definied in the ftls of the individual reports that use the appendixEstudies macro
			E.g. one example containing one condition for analytical methods for risk assessment is
				[{"path": "AdministrativeData.Endpoint", "val" : ["risk"], "qual" : "lk", "type" : "picklist"}] 
			Fur further description see 'createStudyHash' function
		- section: string or list of strings containing the section(s) where the document needs to be cross-referenced in order to be output
			This is only applicable to the analytical methods section.
			E.g. section="6" or section=["7.1", "7.3"].
			
	Returns:
		- studyHash: hashmap, with two hash entries, one for studies 'st' and one for waivers 'dw' with the following nested structure:

	studyHash = {
    'st': {studyHash_st},
    'dw': {studyHash_dw}
	}
	├── st (Actual Studies)
	│   ├── key1 (e.g., 'main'): {keyStudyHash_st}
	│   │   ├── entityName1: {
	│   │       ├── 'entity': entity1,
	│   │       └── 'records': [resultStudyList1]
	│   │   }
	│   │  
	│   └── key2 (e.g., 'metab'): {keyStudyHash_st}
	│       ├── entityName3: {
	│           ├── 'entity': entity3,
	│           └── 'records': [resultStudyList3]
	│       }
	│       ├── entityName4: {
	│           ├── 'entity': entity4,
	│           └── 'records': [resultStudyList4]
	│       }
	│
	└── dw (Data Waivers)
		├── key1 (e.g., 'main'): {keyStudyHash_dw}
		│   ├── entityName1: {
		│       ├── 'entity': entity1,
		│       └── 'records': [dataWaivingStudyList1]
		│   }
-->
<#function createStudyHash entities docSubTypes="" context="" section="">

    <#local studyHash_st = {}/>
    <#local studyHash_dw = {}/>

    <#list entities?keys as key> <#-- "main", "metab", "impur" -->
        <#local entityStudyHash_st = {} />
        <#local entityStudyHash_dw = {} />
        
        <#list entities[key] as entity>
            <#local entityName = getEntityName(entity) />

            <#-- filter by type, context, section and specific data points requirements according to DAR Vol3 templates -->
            <#local studyList = getStudyListByType(entity, docSubTypes) />
            <#local studyList = filterStudyListbyContext(studyList, context) />	

            <#if _doc2SectHashMap??>
                <#local studyList = filterStudyListbySection(studyList, section, _doc2SectHashMap) />
            </#if>

            <#-- separate studies into real studies and data waiving documents -->
            <#local resultStudyList = getResultStudies(studyList) />
            <#if resultStudyList?has_content>
                <#local entityStudyHash_st = entityStudyHash_st + {entityName: {'entity': entity, 'records': resultStudyList}} />
            </#if>

            <#local dataWaivingStudyList = getDataWaivings(studyList) />
            <#if dataWaivingStudyList?has_content>
                <#local entityStudyHash_dw = entityStudyHash_dw + {entityName: {'entity': entity, 'records': dataWaivingStudyList}} />
            </#if>
        </#list>
        
        <#local studyHash_st = studyHash_st + {key: entityStudyHash_st} />
        <#local studyHash_dw = studyHash_dw + {key: entityStudyHash_dw} />
    </#list>

    <#local studyHash = {'st': studyHash_st, 'dw': studyHash_dw} />

    <#return studyHash />
</#function>

<#-- getStudyListByType selects documents of the required sub type in an entity 
	The document types currently supported are FLEXIBLE_RECORD and ENDPOINT_STUDY_RECORD
	(NOTE: this assumes that a flexible record can never have the same sub type as an endpoint study record)

	Inputs:
		- entity: ENTITY that documents belong to
		- docSubTypes: string or list of strings defining document's sub type e.g. "AnalyticalMethods"
		
	Returns:
		- allStudyList: list of studies of the right type
-->
<#function getStudyListByType entity docSubTypes="">

	<#local allStudyList=[]/>

	<#local flexibleRecordSubTypes = ["IntermediateEffects", "AnalyticalInformation"]> 

	<#list docSubTypes as docSubType>
		<#if flexibleRecordSubTypes?seq_contains(docSubType)>
			<#local studyList = iuclid.getSectionDocumentsForParentKey(entity.documentKey, "FLEXIBLE_RECORD", docSubType) />
		<#else>
			<#local studyList = iuclid.getSectionDocumentsForParentKey(entity.documentKey, "ENDPOINT_STUDY_RECORD", docSubType) />
		</#if>
		<#local allStudyList = allStudyList + studyList/>
	</#list>

	<#return allStudyList/>
</#function>

<#--
	filterStudyListbyContext filters a list of studies based on context rules provided.

	Inputs:
	- myStudyList: list of documents
	- context: hashmap containing a list of rules to be a applied to select documents
		Hashmaps are found in the specific ftls of the reports.
		The keys of each hashmap in the list are the following:
			- path: path to the element used to filter the documents
			- val: list of values to be checked
			- qual: the qualifier, 5 posibilities which determine the comparison:
				- 'eq': equal the value in path is EXACTLY ONE of the values in val e.g. val=["toxicity to bees", "toxicity to non-target arthropods"], path_value="toxicity to bees" OK
				- 'ne': not equal the value in path is NONE of the values in val e.g. val=["toxicity to bees"], path_value="toxicity to non-target arthropods" OK
				- 'lk': like the value in path CONTAINS at least one of the values in val e.g. val=["bee"], path_value="toxicity to bees" OK
				- 'nl': not like the value in path DOES NOT CONTAIN any of the values in val e.g. val=["bee"], path_value="toxicity to non-target arthropods" OK
				- 'hc': has content the value in the path is not empty (i.e. has_content check), no comparison with val is done
			- type: the data type, currently only three options possible to extract the value to be compared:
				- "text": text value as is
				- "picklist": the picklist text without description or remarks
				- "picklistMultiple": the ";" separated concatenation of individual picklist values without description or remarks
		ALL the rules need to be valid in order for a study to be selected i.e. the list of rules are joined by AND
	
	Returns:
		- filtStudyList: list of studies after applying the context rules
-->
<#function filterStudyListbyContext myStudyList context>

	<#if myStudyList?has_content && context?has_content>

		<#local filtStudyList = [] />

		<#--Loop over studies-->
		<#list myStudyList as study>

			<#-- Flag to set number of conditions met-->
			<#local nbCondMet=0/>

			<#--Loop over conditions-->
			<#list context as ctxt>

				<#local contextPath = "study." + ctxt["path"] />
				<#local contextVal = contextPath?eval/>

				<#-- Convert value depending on type -->
				<#if ctxt["type"]=="picklistMultiple">

					<#local contextVal2>
						<#compress>
							<#list contextVal as ctxtVal>
								<@com.picklist picklistValue=ctxtVal locale="en" printOtherPhrase=false printDescription=false printRemarks=false /><#if ctxt_has_next>;</#if>
							</#list>
						</#compress>
					</#local>

				<#elseif ctxt["type"]=="picklist">

					<#-- get the value of the picklist without remarks!!-->
					<#local contextVal2>
						<@com.picklist picklistValue=contextVal locale="en" printOtherPhrase=false printDescription=false printRemarks=false />
					</#local>
				
				<#elseif ctxt["type"]=="text">
					<#local contextVal2><@com.text contextVal/></#local>
				</#if>

				<#-- Evaluate condition and change flag -->
				<#if ctxt["qual"]=="eq">
					<#if ctxt["val"]?seq_contains(contextVal2)>
						<#local nbCondMet=nbCondMet+1/>
					</#if>

				<#elseif ctxt["qual"]=="ne">
					<#if !ctxt["val"]?seq_contains(contextVal2)>
						<#local nbCondMet=nbCondMet+1/>
					</#if>

				<#elseif ctxt["qual"] == "lk">
					<#list ctxt["val"] as val>
						<#if contextVal2?contains(val)>
							<#local nbCondMet=nbCondMet+1/>
							<#break>
						</#if>
					</#list>

				<#elseif ctxt["qual"] == "nl">
					<#local notfound=true/>
					<#list ctxt["val"] as val>
						<#if contextVal2?contains(val)>
							<#local notfound=false/>
						</#if>
					</#list>
					<#if notfound><#local nbCondMet=nbCondMet+1/></#if>

				<#elseif ctxt["qual"] == "hc">
					<#if contextVal?has_content>
						<#local nbCondMet=nbCondMet+1/>
					</#if>
				</#if>
			</#list>

			<#--Check if all conditions met and add study-->
			<#if nbCondMet==context?size>
				<#local filtStudyList = filtStudyList + [study] />
			</#if>
		</#list>
		<#return filtStudyList/>
	<#else>
		<#return myStudyList/>
	</#if>
</#function>

<!--
	filterStudyListbySection filters a list of studies by specific section numbers where the studies have been cross-referenced, 
	provided a global variable mapping documents to sections exists.

	Inputs:
	- studyList: list of study objects
	- section: string or list of strings indicating selected section numbers e.g. "5.1.1".
		NOTE: retrieval of the section is based on strict string matching at the beginning of a section numbering
			e.g. "5.1" would be matched by "5.1", "5.1.1", "5.1.2.1"..., but not by "4.5.1")
	- doc2sect: nested hashmap with document uuids as keys e.g.
		{'uuid1':{'name': X, 'endpoint': Y, 'section': [{'name': a, 'nb': b}, {'name': a2, 'nb': b2}, ..] }}
		NOTE: now it's by default the one used for analytical methods - not even passed as argument
	
	Returns:
	- secStudyList: list of filtered studies
-->
<#function filterStudyListbySection studyList section doc2sect>

	<#if studyList?has_content && section?has_content>

		<#local secStudyList = [] />

		<#if !(section?is_sequence)>
			<#local section=[section]/>
		</#if>

		<#list studyList as study>
			<#if _doc2SectHashMap?keys?seq_contains(study.documentKey.uuid)>
				<#local doc = _doc2SectHashMap[study.documentKey.uuid]/>
				<#local docSection = doc["section"]/>

				<#local docMainSection = docSection?sort_by("nb")[0]/>
				<#local docMainSectionNb = docMainSection["nb"]/>

				<#list section as sect>
					<#if docMainSectionNb?starts_with(sect)>
						<#local secStudyList  = secStudyList + [study]/>
						<#break>
					</#if>
				</#list>
			<#else>
				<#if section?seq_contains("NA")>
					<#local secStudyList  = secStudyList + [study]/>
				</#if>
			</#if>
		</#list>

		<#return secStudyList/>
	<#else>
		<#return studyList/>
	</#if>
</#function>

<#--
	getResultStudies filters a list of studies to get only those that are not waiving,
	by checking if they are data waiving using the isDataWaivingStudy function

	Inputs:
	- studyList: list of study objects
	
	Returns:
	- resultStudyList: list of filtered studies
-->
<#function getResultStudies studyList>
	<#local resultStudyList=[]/>
	<#if studyList?has_content>
		<#list studyList as study>
			<#if !isDataWaivingStudy(study)>
				<#local resultStudyList = resultStudyList + [study] />
			</#if>
		</#list>
	</#if>
	<#return resultStudyList/>
</#function>

<#--
	getDataWaivings filters a list of studies to get only those that are data waiving,
	by checking if they are data waiving using the isDataWaivingStudy function

	Inputs:
	- studyList: list of study objects
	
	Returns:
	- dataWaivingStudyList: list of filtered studies
-->
<#function getDataWaivings studyList>
	<#local dataWaivingStudyList=[]/>
	<#if studyList?has_content>
		<#list studyList as study>
			<#if isDataWaivingStudy(study)>
				<#local dataWaivingStudyList = dataWaivingStudyList + [study] />
			</#if>
	</#list>
	</#if>
	<#return dataWaivingStudyList/>
</#function>

<#--
	isDataWaivingStudy checks if a study is a data waiving, by checking two conditions:
	- AdministrativeData.DataWaiving has content
	- AdministrativeData.PurposeFlag is empty.
	If the field does not exist, it returns False.
	NOTE: this function could be improved.
	
	Inputs:
	- study: study object
	
	Returns: true or false
-->
<#function isDataWaivingStudy study>
	<#if !(study?has_content) || !(study.hasElement("AdministrativeData.DataWaiving"))>
		<#return false>
	</#if>
	<#local PurposeFlag = study.AdministrativeData.PurposeFlag />
	<#local DataWaiving = study.AdministrativeData.DataWaiving />
    <#return !(PurposeFlag?has_content) && DataWaiving?has_content />
</#function>

<#--
	getEntityName retrieves the name of a SUBSTANCE or MIXTURE ENTITY. 
	This corresponds to:
	- chemical name, in case of SUBSTANCE
	- mixture name, in case of MIXTURE
	NOTE: other entities are not covered
	
	Inputs:
	- entity: ENTITY object
	
	Returns: 
	- entityName: str with name
-->
<#function getEntityName entity>
	<#local entityName>
		<#compress>
			<#if entity.hasElement("ChemicalName")>
				<@com.text entity.ChemicalName ""/>
			<#elseif entity.hasElement("MixtureName")>
				<@com.text entity.MixtureName ""/>
			</#if>
		</#compress>
	</#local>
	<#return entityName>
</#function>

<#--
	getRequirementName gets the subtype of a document or list of documents and transforms its name to
	make it more readable i.e. adding spaces between words.
	If a specific name is provided, it overrides.

	Inputs:
	- name: str with required name
	- docSubTypes: str or list of str with document subtypes

	Returns:
	- name: str
-->
<#function getRequirementName name docSubTypes>

	<#if name=="">
		<#local name>
			<#compress>
				<#list docSubTypes as docSubType>
					${docSubType?replace("([A-Z]{1})", " $1", "r")?lower_case}<#if docSubType_has_next>, </#if>
				</#list>
			</#compress>
		</#local>
	</#if>

	<#return name>
</#function>
<#--
	getStudyReference gets the list of LITERATURE references in a study record
	e.g. ENDPOINT_STUDY_RECORD, sorted by literature type (GeneralInfo.LiteratureType).
	
	It uses the study.DataSource.Reference path.

	Inputs:
	- study: DOCUMENT object where references can be found in path DataSource.Reference
	- sorted_values: list of sorted values for the field GeneralInfo.LiteratureType in the LITERATURE entity

	Returns:
	- referenceList: list of LITERATURE entities, sorted by type 
-->
<#function getStudyReference study sorted_values = ["study report","publication","publication (copyright not owned for reproduction)","review article or handbook", "other company data", "other:"]>

	<#local referenceList=[]/>
	
	<#if study.hasElement("DataSource.Reference")>
		<#local referenceLinksList=study.DataSource.Reference/>
	<#elseif study.hasElement("AnalyticalInformation.DataSource.Reference")><#-- for AnalyticalInformation -->
		<#local referenceLinksList=study.AnalyticalInformation.DataSource.Reference/>
	</#if>
	
	<#if referenceLinksList?? &&  referenceLinksList?has_content>
		<#list referenceLinksList as referenceLink>
			<#local referenceEntry = iuclid.getDocumentForKey(referenceLink)/>
			<#local referenceList = referenceList + [referenceEntry]/>
		</#list>
	</#if>

	<#if referenceList?has_content && (referenceList?size > 1)>
		<#local referenceList = iuclid.sortByField(referenceList, "GeneralInfo.LiteratureType", sorted_values ) />
	</#if>
	
	<#return referenceList>
</#function>

<#--
	getReferenceElement extracts a specific element from a 
	list of LITERATURE entities. In case of more than one entity,
	the values or the elements are concatenated in the same string with a number
	between parenthesis.
	
	Inputs:
	- referenceList: list of LITERATURE entities
	- element: str with field path in LITERATURE entity
	- hyperlink: if true, retrieve element with hyperlink

	Returns:
	- referenceElement: str with the value for the element 
-->
<#function getReferenceElement referenceList element hyperlink=false>
	<#local referenceElement="">

	<#if referenceList?has_content>
		<#list referenceList as reference>
			
			<#-- get url -->
			<#local refUrl=iuclid.webUrl.documentView(reference.documentKey) />
			
			<#-- extract element -->
			<#local refElementPath = "reference." + element/>
			<#local refElement = refElementPath?eval/>
			<#local refValue><@com.value refElement ""/></#local>

			<#-- get the element with the right printing macro -->
			<#local refElement><#if (referenceList?size>1)>(${reference_index+1})</#if>
				<#if hyperlink><ulink url="${refUrl}"></#if>

				<#if refValue?has_content>${refValue}<#else>n.a.</#if>
				
				<#if hyperlink></ulink></#if>
			</#local>

			<#-- add to list -->
			<#local referenceElement = referenceElement + refElement/>

		</#list>
	</#if>
	<#return referenceElement/>
</#function>

<#--
	getNoSid extracts the NoS ids and related remarks from a list of LITERATURE entities.
	Finding a NoS id relies only on the selection of the right picklist value.
	The NoS id format is NOT checked for correcteness i.e. EFSA-YYYY-DDDDDDDD
	
	Inputs:
	- referenceList: LITERATURE entity or list of LITERATURE entities
	- element: str with field path in LITERATURE entity
	- hyperlink: if true, retrieve element with hyperlink

-->
<#function getNoSid referenceList>

	<#local NoSIds=[]/>

	<#-- if it's a single LITERATURE convert to list -->
	<#if !referenceList?is_sequence>
		<#local referenceList = [referenceList]/>
	</#if>

	<#-- iterate -->
	<#list referenceList as reference>

		<#-- check study identifiers -->
		<#if reference.GeneralInfo.StudyIdentifiers?has_content>

			<#list reference.GeneralInfo.StudyIdentifiers as studyId>

				<#-- get idType and check if it's NoS id-->
				<#local idType><@com.picklist studyId.StudyIDType/></#local>

				<#if idType=="Notification of Studies (NoS) ID">

					<#-- if Id has content, retrieve; else report n.a.-->
					<#local NoSId><@com.text studyId.StudyID ""/></#local>
					<#if !NoSId?has_content><#local NoSId="n.a."/></#if>

					<#if (referenceList?size>1)>
						<#local index>(${reference_index+1})</#local>
						<#local NoSId = index + NoSId />
					</#if>

					<#-- if remarks has content, retrieve and add to id -->
					<#local remark><#compress><@com.text studyId.Remarks ""/></#compress></#local>
					<#if remark?has_content>
						<#local NoSId = NoSId + ' (' + remark + ')'/>
					</#if>

					<#-- add id with remarks to the list -->
					<#local NoSIds = NoSIds + [NoSId]/>
					
				</#if>
			</#list>
		</#if>
	</#list>

	<#-- join all ids by ';' -->
	<#if NoSIds?has_content>
		<#return NoSIds?join("; ")/>
	<#else>
		<#return "">
	</#if>

</#function>

<#--
	getGuidelines retrieves information from the guidelines used in a study.
	
	It uses the macro guidelineList (of common_module_studies_and_summaries) to retrieve guidelines 
	from the MaterialsAndMethods section, and in addition, retrieves info from
	MethodNoGuideline, if it has content. 
	
	The only exception is the intermediate effects document, for which a different strategy is followed.
	
	Inputs:
	- study: document object where guidelines are included

	Returns:
	- guidelines: str with guideline information

-->
<#function getGuidelines study>

	<#local guidelines><#compress>
		<#if study.hasElement("MaterialsAndMethods") && study.documentSubType!="IntermediateEffects" && study.documentSubType!="AnalyticalProfileOfBatches">
			<@studyandsummaryCom.guidelineList guidelineRepeatableBlock=study.MaterialsAndMethods.Guideline includeVersion=true includeIndex=true separator=''/>
			<#if study.MaterialsAndMethods.MethodNoGuideline?has_content>
				<para>other method: <@com.text study.MaterialsAndMethods.MethodNoGuideline/></para>
			</#if>
		<#elseif study.documentSubType=="IntermediateEffects">
			<@com.picklist study.MaterialsAndMethods.MethodUsed.Qualifier/> <@com.picklist study.MaterialsAndMethods.MethodUsed.MethodUsed/>
			<para><@com.text study.MaterialsAndMethods.MethodUsed.PrincipleOfTheMethod/></para>
		<#else>
			n.a.
		</#if>
	</#compress></#local>

	<#return guidelines>
</#function>

<#--
	getGuidelinesDeviation retrieves information on deviation from guidelines. 
	
	It uses the macro guidelineList (of common_module_studies_and_summaries) to retrieve deviations
	from the MaterialsAndMethods section
	The strategy is common to all documents except Intermediate Effects.
	
	Inputs:
	- study: document object where guidelines are included

	Returns:
	- deviations: str with guideline deviation information
-->
<#function getGuidelinesDeviation study>

	<#local deviations><#compress>
		<#if study.hasElement("MaterialsAndMethods") && study.documentSubType!="IntermediateEffects" && study.documentSubType!="AnalyticalProfileOfBatches">
			<@studyandsummaryCom.guidelineList guidelineRepeatableBlock=study.MaterialsAndMethods.Guideline includeGuideline=false includeDeviation=true includeIndex=true separator=''/>
		<#elseif study.documentSubType=="IntermediateEffects">
			<@com.picklist study.MaterialsAndMethods.MethodUsed.Deviations/>
		<#else>
			n.a.
		</#if>
	</#compress></#local>

	<#return deviations>
</#function>	

<#--
	getChangeLogStatus checks if the study was previously used, by retrieving information
	from the change log: FLEXIBLE_RECORD.ChangeLog.
	
	By default, if the study (document UUID) is not in any change log, it returns "no" (it assumes
	the study was not previously used). If the document is found in a change log, then it returns
	a yes/no value (yes if the status is differnet than 'new') together with the status and remarks.
	If the document is listed more than once in all change logs, the statuses are concatenated.
	
	Inputs:
	- study: document object where guidelines are included
	- entity: ENTITY object where the change log is found

	Returns:
	- changeLogStatusMessage: str with the message on "previously used" to be displayed
-->
<#function getChangeLogStatus study entity>
	
	<#local changeLogEntryList=[]/>

	<#-- default flag for change log -->
	<#local changeLogFlag="no"/>

	<#-- retrieve change log documents -->
	<#local changeLogs = iuclid.getSectionDocumentsForParentKey(entity.documentKey, "FLEXIBLE_RECORD", "ChangeLog") />

	<#-- iterate over changelog dicuments and entries-->
	<#list changeLogs as changeLog>
		
		<#list changeLog.ChangeLog.ChangeLogEntries as changeLogEntry>
			
			<#-- get the document linked in the change log entry and check if it corresponds to current study doc (same uuid),
				If it does, then retrieve the status and add to the list
			-->
			<#local changeLogDoc=iuclid.getDocumentForKey(changeLogEntry.LinkToDocument)/>

			<#if changeLogDoc?has_content>

				<#if study.documentKey.uuid==changeLogDoc.documentKey.uuid>
					
					<#local changeLogStatus><@com.picklist changeLogEntry.Status/></#local>
					
					<#if changeLogStatus?has_content>

						<#-- if status is "new", then change flag to 'yes' -->
						<#if !(changeLogStatus?starts_with("new"))>
							<#local changeLogFlag="yes"/>
						</#if>

						<#-- add the change log entry to the list if status is set -->
						<#local changeLogEntryList = changeLogEntryList + [changeLogEntry]/>
					</#if>
				</#if>
			</#if>
		</#list>
	</#list>
	
	<#-- create the message to be displayed using:
		- the flag (yes or no)
		- the status with remarks (if multiple, concatenate separated by |)
	-->
	<#local changeLogStatusMessage>${changeLogFlag}
		<#if changeLogEntryList?has_content>
			<#list changeLogEntryList as clEntry>
				(<@com.picklist clEntry.Status/>)<#if clEntry.Remark?has_content> - <@com.text clEntry.Remark/></#if>
				<#if clEntry?has_next>|</#if>
			</#list>
		</#if>
	</#local>

	<#return changeLogStatusMessage>
</#function>

<#--
	getGLP retrieves GLP and related information from a study, by checking the fields:
		- GLPComplianceStatement
		- OtherQualityAssurance
		- ComplianceWithQualityStandards
	For intermediate effects a different path is used.  
	
	Inputs:
	- study: document object where guidelines are included

	Returns:
	- glpInfo: str with information on GLP
-->
<#function getGLP study>

	<#local glpInfo><#compress>

		<#-- This is the general path applicable to most documents -->
		<#if study.hasElement("MaterialsAndMethods") && study.documentSubType!="IntermediateEffects" && study.documentSubType!="AnalyticalProfileOfBatches">
			<@com.picklist study.MaterialsAndMethods.GLPComplianceStatement/>

			<#--NOTE: this path is not always present so better to check that it exists first-->
			<#if study.hasElement("MaterialsAndMethods.OtherQualityAssurance") && study.MaterialsAndMethods.OtherQualityAssurance?has_content>
				<?linebreak?>other quality assurance: <@com.picklist study.MaterialsAndMethods.OtherQualityAssurance/>
			</#if>
			<#if study.hasElement("MaterialsAndMethods.ComplianceWithQualityStandards") && study.MaterialsAndMethods.ComplianceWithQualityStandards?has_content>
				<?linebreak?>compliance with quality standards: <@com.picklist study.MaterialsAndMethods.ComplianceWithQualityStandards/>
			</#if>

		<#-- for Intermediate Effects -->
		<#elseif study.documentSubType=="IntermediateEffects">
			<@com.picklist study.MaterialsAndMethods.MethodUsed.GLPCompliance/>
			<?linebreak?>
			<#if study.MaterialsAndMethods.MethodUsed.OtherQualityFollowed?has_content>
				other quality assurance: <@com.picklist study.MaterialsAndMethods.MethodUsed.OtherQualityFollowed/>
			</#if>

		<#else>
			n.a.
		</#if>
	</#compress></#local>

	<#return glpInfo/>

</#function>

<#--
	sortByEndpoint sorts a list of studies so that studies with the same
	endpoint appear sequentially in the list.

	Input:
	- sortByEndpoint: list of document objects to be sorted

	Returns:
	- sortedStudies: sorted list of document objects
-->
<#function sortByEndpoint myStudyList>

	<#-- create a hashmap to sort by endpoint (same endpoints together)-->
	<#local studyHashMap={}/>
	<#list myStudyList as study>
		<#local endpoint><@com.picklist study.AdministrativeData.Endpoint/></#local>
		<#if studyHashMap[endpoint]??>
			<#local entry = studyHashMap[endpoint] + [study]/>
			<#local studyHashMap = studyHashMap + {endpoint : entry}/>
		<#else>
			<#local studyHashMap = studyHashMap + {endpoint:[study]}/>
		</#if>
	</#list>

	<#-- read hashmap and get ordered list-->
	<#local sortedStudies=[]/>
	<#list studyHashMap?keys as key>
		<#local sortedStudies = sortedStudies + studyHashMap[key]/>
	</#list>

	<#return sortedStudies/>
</#function>

<#-- 
	dataWaiver prints data waivers from a list of documents.
	
	This substitutes the macro in macros_common_studies_and_summaries.

	Inputs:
	- dwList: list of study objects being data waiver
	- endpoint: str with endpoint name to print
-->
<#macro dataWaiver dwList endpoint="" includeAnnotations=false>
	<#compress>
		<#if dwList?has_content>
			<#list dwList as study>
				<#if study?has_content>
	
					<sect4 xml:id="${study.documentKey.uuid!}" label="" role="NotInToc">
						
						<para><emphasis role="bold"><@com.text study.name ""/></emphasis>:</para>
						
						<para role="indent"><emphasis role="bold">Information requirement: </emphasis>
						 	<#if study.AdministrativeData.Endpoint?has_content>
						 		<@com.picklist study.AdministrativeData.Endpoint/>
						 	<#else>
						 		${endpoint!}
							</#if>
						</para>
	
						<para role="indent">
							<emphasis role="bold">Reason: </emphasis>
							<@com.picklist study.AdministrativeData.DataWaiving/>
						</para>
				
						<para role="indent">
							<emphasis role="bold">Justification:</emphasis>
							<@com.picklistMultiple study.AdministrativeData.DataWaivingJustification/>
						</para>

						<#if study.hasElement("ApplicantSummaryAndConclusion.Conclusions") && study.ApplicantSummaryAndConclusion.Conclusions?has_content> 
							<para role="indent">
								<emphasis role="bold">Conclusion:</emphasis>
								<@com.text study.ApplicantSummaryAndConclusion.Conclusions/>
							</para>
						</#if>

						<#if study.hasElement("ApplicantSummaryAndConclusion.ExecutiveSummary") && study.ApplicantSummaryAndConclusion.ExecutiveSummary?has_content>
							<para role="indent">
								<emphasis role="bold">Executive summary:</emphasis>
								<para style="background-color:#f7f7f7" role="indent">
									<@com.richText study.ApplicantSummaryAndConclusion.ExecutiveSummary/>
								</para>
							</para>
						</#if>

						<#if includeAnnotations>
							<#local annotationsList = getAnnotationsListFromDoc(study)/>
						</#if>
						<#if annotationsList?? && annotationsList?size gt 0>
							<@printAnnotationsforDoc study />
						<#else>
							<#local boxTitle = "Assessment and conclusion by RMS:">
							<#if workingContext?? && workingContext=="MRL">
								<#local boxTitle = "Assessment and conclusion by Evaluating Member State:">
							</#if>	
							<@com.printEmptyBox title=boxTitle color="#DAE9F7"/>
						</#if>	
					</sect4>
				</#if>
			</#list>
		</#if>
	</#compress>
</#macro>

<#--
	containsDoc checks if an entity contains a set of documents, with a specific
	context or cross-reference section, if provided.
	It returns true if at least ONE of the documents is present.

	Input:
	- subject: ENTITY object OR list of ENTITY objects
	- docTypesSubTypes: document type and subtype OR list of document types and subtypes e.g. ['ENDPOINT_STUDY_RECORD.Ph']
	- context: list of hashmaps containing the conditions to be used to select which document(s) to output
			The hashmaps are definied in the ftls of the individual reports that use the appendixEstudies macro
			E.g. one example containing one condition for analytical methods for risk assessment is
				[{"path": "AdministrativeData.Endpoint", "val" : ["risk"], "qual" : "lk", "type" : "picklist"}] 
			Fur further description see 'filterStudyListbyContext' function
	- includeMetabolites: bool; if true, studies of the selected type in metabolite datasets will also be searched
	- section: string or list of strings containing the section(s) where the document needs to be cross-referenced in order to be output
		This is only applicable to the analytical methods section.
		E.g. section="6" or section=["7.1", "7.3"]
	- includeImpurities: bool; if true, studies of the selected type in impurity datasets will also be searched	

	Returns: true or false
-->
<#function containsDoc subject docTypesSubTypes context=[] includeMetabolites=true section="" includeImpurities=true>

	<#-- convert to list if only one doc type and subtype provided as string -->
	<#if !docTypesSubTypes?is_sequence>
		<#local docTypesSubTypes=[docTypesSubTypes]/>
	</#if>
	
	<#-- create list of entities to loop for studies -->
	<#if !subject?is_sequence>
		<#local entities=[subject]/>
	<#else>
		<#local entities=subject/>
	</#if>

	<#-- add metabolites if required -->
	<#if includeMetabolites && _metabolites?? && _metabolites?has_content>
		<#local entities = entities + _metabolites/>
	</#if>

	<#-- add impurities if required -->
	<#if includeImpurities && _impurities?? && _impurities?has_content>
		<#local entities = entities + _impurities/>
	</#if>
	
	<#-- iterate through list of docs -->
	<#list docTypesSubTypes as docTypeSubType>

		<#local docTypeSubtypeList=docTypeSubType?split(".")/>
		
		<#local docList=[]/>
		<#list entities as entity>
			<#attempt>
				<#local docList = docList + iuclid.getSectionDocumentsForParentKey(entity.documentKey, docTypeSubtypeList[0], docTypeSubtypeList[1])/>
			<#recover>

			</#attempt>

		</#list>
		
		<#if context?has_content>
			<#local docList=filterStudyListbyContext(docList, context)/>
		</#if>
		
		<#if _doc2SectHashMap?? && section?has_content>
		    <#local docList = filterStudyListbySection(docList, section, _doc2SectHashMap)/>
        </#if>
		
		<#if docList?has_content>
			<#return true>
		</#if>
	</#list>

	<#return false>
</#function>

<#-- 
	extendSubTypes returns the list of all possible combinations of document types and subtypes

	Input:
	- docSubTypes: list of document subtypes to be extended
	- summaries: bool; if true, summary doc types are used, else the study ones

	Returns: list of extended document subtypes e.g. ['ENDPOINT_STUDY_RECORD.Ph']
-->
<#function extendSubTypes docSubTypes=[] summaries=true>
	<#if summaries>
		<#local docTypes = ["ENDPOINT_SUMMARY.","FLEXIBLE_SUMMARY."]>
	<#else>
		<#local docTypes = ["ENDPOINT_STUDY_RECORD.","FLEXIBLE_RECORD."]>
	</#if>

	<#local extendedDocSubTypes = []/>
	<#list docSubTypes as docSubType>
		<#list docTypes as docType>
			<#local extendedDocSubTypes += [docType + docSubType]>
		</#list>
	</#list>
	<#return extendedDocSubTypes/>	
</#function>

<#-- note: when merging, maybe append the title of the summary at the beginning of the paragraph-->
<#macro PPPsummary subject docSubTypes merge=false includeMetabolites=true includeAnnotations=false endpoint="" context='' includeImpurities=true includeImages=true entitiesHashKey="main">
	<#compress>

		<#--Get all documents, from same or different type-->
		<#if !docSubTypes?is_sequence>
			<#local docSubTypes=[docSubTypes]/>
		</#if>

		<#-- Convert subject always to a list -->
		<#if !subject?is_sequence>
			<#local subject=[subject]/>
		</#if>

		<#-- Create list of entities to loop for summaries, based on subject -->
		<#local entities = {}/>
		<#list subject as entity>
			<#local entities = com.addToHash(entities, entitiesHashKey, entity)/>
		</#list>

		<#-- Add metabolites to the list of entities to loop for summaries, if required -->
		<#if includeMetabolites && _metabolites?? && _metabolites?has_content>
			<#if _metabolites?is_sequence>
				<#list _metabolites as metabolite>
					<#local entities = com.addToHash(entities, "metab", metabolite)/>
				</#list>
			</#if>
		</#if>

		<#-- Add impurities to the list of entities to loop for summaries, if required -->
		<#if includeImpurities && _impurities?? && _impurities?has_content>
			<#if _impurities?is_sequence>
				<#list _impurities as impurity>
					<#local entities = com.addToHash(entities, "impur", impurity)/>
				</#list>
			</#if>
		</#if>

		<#-- Get all summaries for each entity into a hash -->
		<#local entity2summaryHash = {}/>
		<#list entities?keys as entityType> <#-- "main", "metab", "impur" -->
			<#local entitysummaryHash = {}/>
			<#list entities[entityType] as entity>
			
				<#local entitySummaryList=[]/>
				
				<#local flexibleSummarySubTypes = ["AquaticToxicityRacReporting", "NonDietaryExpo", "ToxRefValues", "EndocrineDisruptingPropertiesAssessmentPest", 
												   "ResiduesInLivestock", "MRLProposal", "ExpectedExposure", "DefinitionResidueFate", "SummaryEvaluation_EU_PPP",
												   "RelevantMetabolitesGroundWater", "InformationEcotoxicityMetabolites", "InformationToxicityMetabolites",
												   "AssessmentOfPotentialToxicity", "PathogenicityInfectivityHumans"]>
				<#list docSubTypes as docSubType>
					<#if docSubType?matches("EstConc.*") || flexibleSummarySubTypes?seq_contains(docSubType)>
						<#local summaryList = iuclid.getSectionDocumentsForParentKey(entity.documentKey, "FLEXIBLE_SUMMARY", docSubType) />
					<#else>
						<#local summaryList = iuclid.getSectionDocumentsForParentKey(entity.documentKey, "ENDPOINT_SUMMARY", docSubType) />
					</#if>

					<#-- filter list -->
					<#if context?has_content>
						<#local summaryList = filterStudyListbyContext(summaryList, context)/>
					</#if> 

					<#local entitySummaryList = entitySummaryList + summaryList/>
				</#list>
			
				<#if entitySummaryList?has_content>
					<#if entity.documentType=="MIXTURE">
						<#local entityName=entity.MixtureName/>
					<#elseif entity.documentType=="SUBSTANCE">
						<#local entityName=entity.ChemicalName/>
					</#if>
					<#local entitysummaryHash = entitysummaryHash + {entityName: {'summaries': entitySummaryList, 'entity': entity}}/>
				</#if>
			</#list>
			<#local entity2summaryHash = entity2summaryHash + {entityType: entitysummaryHash}/>
		</#list>	

		<#-- print "Summary" regardless if there is content or not -->
		<para><emphasis role="HEAD-KeyA">Summary</emphasis></para>

		<#-- check if there are summaries -->
		<#local summariesFound = false>
		<#list entity2summaryHash as entityType, entitysummaryHash>
			<#if entitysummaryHash?has_content>
				<#local summariesFound = true>
				<#break>
			</#if>
		</#list>

		<#-- Iterate through summaries and create section lists for each entity -->
		<#if !summariesFound>			
			<#--  <@com.emptyLine/>  -->
			<para>No summary information available for this section.</para>
			<#--  <@com.emptyLine/>  -->
		<#else>
			<#list entity2summaryHash as entityType, entitysummaryHash> 
				<#list entitysummaryHash as entityName, entityData>
				
					<#local allSummaryList=entityData['summaries']/>
					<#local entity=entityData['entity']/>

					<#-- create variables for the different sections, in order to be able to merge -->
					<#local keyInfoList=[]/>
					<#local linksList=[]/>
					<#local endpointsHash={}/>
					<#local higherTierList=[]/>
					<#local higherTierLinksList=[]/>
					<#local justificationList=[]/>
					<#local discussionList=[]/>
					<#local attachmentsList=[]/>

					<#if allSummaryList?has_content>

						<#-- print numbering for summaries if there is more than one for an entity -->
						<#local printSummaryIndex = allSummaryList?size gt 1 />

						<#-- print a header for metabolites -->
						<#if entityType == "metab">
							<@com.emptyLine/>
							<para><emphasis role="underline">----- Metabolite <emphasis role="bold">${entityName}</emphasis> -----</emphasis></para>
						</#if>

						<#-- print a header for impurities -->
						<#if entityType == "impur">
							<@com.emptyLine/>
							<para><emphasis role="underline">----- Relevant Impurity <emphasis role="bold">${entityName}</emphasis> -----</emphasis></para>
						</#if>

						<#-- define if to print summary titles for rich text fields when merge -->
						<#if merge && (allSummaryList?size>1)><#local printSummaryTitleWhenMerge=true/><#else><#local printSummaryTitleWhenMerge=false/></#if>

						<#-- iterate over summaries for the entity -->
						<#local summaryLinks = []/>
						<#list allSummaryList as summary>

							<#-- get url and link-->
							<#local summaryUrl=iuclid.webUrl.documentView(summary.documentKey) />
							<#local summaryName><@com.text summary.name ""/></#local>
							<#local summaryLink><ulink url="${summaryUrl}">${summaryName}</ulink></#local>
						
							<#if (!merge)>
								<#-- if not merge, print the summary title with the link -->
								<@com.emptyLine/>
								<para><emphasis role="bold"><#if printSummaryIndex>#${summary_index+1}: </#if>Title: ${summaryLink}</emphasis></para>
							<#else>
								<#-- if merge, make a list of links -->
								<#local summaryLinks = summaryLinks + [summaryLink]/>
							</#if>

							<#-- Key information 
								NOTE: in EstConc summaries the field is KeyInformation.field357; in ENDPOINT_SUMMARY.SupplementaryStudies is Keyinformation.KeyInformation
							-->
							<#if summary.documentSubType=="AnalyticalProfBatches" || summary.documentSubType=="InformationEcotoxicityMetabolites" || summary.documentSubType=="InformationToxicityMetabolites">
								<#local keyInfo = getSummaryRichTextSection(summary, ["DescriptionOfKeyInformation"], '', printSummaryTitleWhenMerge, true)/>
							<#else>
								<#local keyInfo = getSummaryRichTextSection(summary, ["KeyInformation.KeyInformation", "KeyInformation.field357", "Keyinformation.KeyInformation", "DescriptionKeyInfo.field436", "AssessmentOfPotentialToxicity.field4165", "AssessmentOfPotentialInfectivityAndPathogenicityOfTheMicroorganismToHumans.field9285"], '', printSummaryTitleWhenMerge)/>
							</#if>

							<#if summary.documentSubType=="InformationEcotoxicityMetabolites" || summary.documentSubType=="InformationToxicityMetabolites">
								<#local keyInfo = keyInfo + getSummaryRichTextSection(summary, ["DescriptionOfKeyInformation.Step5IsTheGenusOfTheStrainUnderEvaluationWellStudied"], '', printSummaryTitleWhenMerge, true, true)/>
							</#if>
							
							<#if keyInfo?has_content>
								<#if merge>
									<#local keyInfoList =  keyInfoList + [keyInfo]/>
								<#else>
									<para><emphasis role="bold">Key information: </emphasis></para>${keyInfo}
								</#if>
							</#if>
						
							<#-- Links, only if they are is not CSA (where they would appear)
									NOTE: in EstConc summaries RelevantSummaries.InputSummaries
						 	-->
							<#local CSApath = studyandsummaryCom.getObjectFromPathOptions(summary, ["KeyValueForChemicalSafetyAssessment", "KeyValueChemicalAssessment", "ResultsAndDiscussion"])/>
							<#if !CSApath?has_content>
								<#local links = studyandsummaryCom.getSummaryLinks(summary, ["LinkToRelevantStudyRecord.Link", "RelevantSummaries.InputSummaries"], 'indent')/>
								<#if links?has_content>
									<#if merge>
										<#-- add summary title if needed -->
										<#if printSummaryTitleWhenMerge><#local links><para>${summaryLink}</para>${links}</#local></#if>
										<#local linksList =  linksList + [links]/>
									<#else>
										<para><emphasis role="bold">Link to relevant study records: </emphasis></para>${links}
									</#if>
								</#if>
							</#if>

							<#--CSA value: depends on the summary or group of summaries -->
							<#if !merge>
								<#local csaTable><#compress>
									<#if _section??>
										<#if _section=="Ecotox" && keyEcotox??>
											<#if keyFate?? && summary.documentSubType=="EstConcWaterSed">
												<#--  for PECsw and sediment in Ecotox PPP RA Aquatic Organisms  -->
												<@keyFate.fateCSA summary=summary endpoint=endpoint bgcolor="#D3D3D3"/>
											<#else>
												<@keyEcotox.ecotoxCSA summaryList=summary bgcolor="#D3D3D3"/>
											</#if>
									
										<#elseif _section=="Fate" && keyFate??>
											<@keyFate.fateCSA summary=summary endpoint=endpoint bgcolor="#D3D3D3"/>
									
										<#elseif _section=="Biopropmicro" && keyBioPropMicro??>
											<#-- it's only one summary, and might change -->
											<#if summary.documentSubType=="ToxicityToOtherAboveGroundOrganisms">
												<@keyBioPropMicro.toxicityToOtherAboveGroundOrganismsSummaryTable summary/>
											</#if>

										<#elseif _section =="Physchem" && keyPhysChemSummary?? && CSApath?has_content>
											<@keyPhysChemSummary.physicalChemicalPropertiesTable _subject=entity selectedDocSubTypes=[summary.documentSubType] printTitle=false includeMetabolites=false includeImpurities=false bgcolor="#D3D3D3"/>

										<#elseif _section=="Tox" && keyTox??>
											<@keyTox.toxCSAtable summaryList=summary bgcolor="#D3D3D3"/>

										<#elseif _section=="Residues" && keyRes??>
											<@keyRes.residuesSummaryTable summary/>
										</#if>
									</#if>
								</#compress></#local>
							
								<#if csaTable?has_content>
									<#if _section?? && _section != "Residues">
										<para><emphasis role="bold">Key value for assessment: </emphasis></para>
									</#if>

									<para role='small'>${csaTable}</para>
								</#if>
							</#if>

							<#-- endocrine disrupting properties main summary -->
							<#if !merge && keyTox?? && summary.documentSubType=="EndocrineDisruptingPropertiesAssessmentPest">
								<@keyTox.endocrineDisruptingPropertiesTable summary=summary bgcolor="#D3D3D3"/>
							</#if>

							<#-- assessment of potential toxicity of the PPP -->
							<#if !merge && keyTox?? && (summary.documentSubType=="AssessmentOfPotentialToxicity" || summary.documentSubType=="PathogenicityInfectivityHumans")>
								<@keyTox.assessmentPotentialToxicity summary/>
							</#if>

							<#-- transformation products (for fate) -->
							<#if !merge && keyFate?? && summary.hasElement("InformationOnTransformationProducts") && summary.InformationOnTransformationProducts?has_content>
								<para><emphasis role="bold">Information on transformation products: </emphasis></para>
								<para role='small'><@keyFate.transformationProductsSummaryTable summaryList=summary bgcolor="#D3D3D3"/></para>
							</#if>

							<#-- relevant metabolites groundwater -->
							<#if !merge && keyFate?? && summary.documentSubType=="RelevantMetabolitesGroundWater">
								<@keyFate.relevantMetabolitesGroundWater summary=summary/>
							</#if>

							<#-- summary document of section "Other reports" in Summary and evaluation -->
							<#if !merge && keyAdm?? && summary.documentSubType=="SummaryEvaluation_EU_PPP">
								<@keyAdm.summaryEvaluationPPP summary=summary/>
							</#if>
							
							<#-- ecotoxicity of metabolites (for ecotox micro AS) -->
							<#if !merge 
								&& keyTox?? 
								&& (summary.documentSubType=="InformationEcotoxicityMetabolites" || summary.documentSubType=="InformationToxicityMetabolites") 
								&& summary.hasElement("DescriptionOfKeyInformation.Metabolites") 
								&& summary.DescriptionOfKeyInformation.Metabolites?has_content>
								<para><emphasis role="bold">Metabolites: </emphasis></para>
								<para role='small'><@keyTox.ecotoxicityMetabolitesTable summaryList=summary bgcolor="#D3D3D3"/></para>
							</#if>
		
							<#-- higher tier testing -->
							<#local higherTier = getSummaryRichTextSection(summary, ["KeyValueForChemicalSafetyAssessment.HigherTierTesting.KeyInformationFromHigherTierTesting"], '' , printSummaryTitleWhenMerge)/>
							<#local higherTierLinks = studyandsummaryCom.getSummaryLinks(summary, ["KeyValueForChemicalSafetyAssessment.HigherTierTesting.LinkToRelevantStudyRecord"], 'indent2')/>
							<#if higherTier?has_content || higherTierLinks?has_content>
								<#if merge>
									<#local higherTierList =  higherTierList + [higherTier]/>
									<#-- add summary title if needed -->
									<#if printSummaryTitleWhenMerge><#local higherTierLinks><para>${summaryLink}</para>${higherTierLinks}</#local></#if>
									<#local higherTierLinksList =  higherTierLinksList + [higherTierLinks]/>
								<#else>
									<para><emphasis role="bold">Higher tier testing: </emphasis></para>${higherTier}
									<#if higherTierLinks?has_content>
										<para><para role='indent' style="background-color:#f7f7f7">Link to relevant study records:${higherTierLinks}</para></para>
									</#if>
								</#if>
							</#if>

							<#-- Justification (Tox, Physchem) -->
							<#local justification = getSummaryRichTextSection(summary, ["Justification","JustificationForClassificationOrNonClassification"], '' , printSummaryTitleWhenMerge, true)/>
							<#if justification?has_content>
								<#--  <#local justification><@com.children path=justPath titleEmphasis=true style="background-color:#f7f7f7" printChildName=false printChildLabel=true/></#local>  -->
								<#if merge>
									<#local justificationList =  justificationList + [justification]/>
								<#else>
									<para><emphasis role="bold">Justification for classification or non-classification: </emphasis></para>${justification}
								</#if>
							</#if>

							<#-- Discussion -->
							<#local discussion = getSummaryRichTextSection(summary, ["AdditionalInformation.AdditionalInformation", "Discussion.Discussion"], '' , printSummaryTitleWhenMerge)/>
							<#if discussion?has_content>
								<#if merge>
									<#local discussionList =  discussionList + [discussion]/>
								<#else>
									<para><emphasis role="bold">Additional information: </emphasis></para>${discussion}
								</#if>
							</#if>

							<#--  Attachments  -->
							<#if includeImages>
								<#local attachments = getSummaryAttachments(summary, ["Discussion.AttachedBackgroundMaterial"], printSummaryTitleWhenMerge)/>
								<#if attachments?has_content>
									<#if merge>
										<#local attachmentsList =  attachmentsList + [attachments]/>
									<#else>
										<para><emphasis role="bold">Illustrations / pictures / graphs / attachments:</emphasis></para>${attachments}
									</#if>
								</#if>
							</#if>
							
							<#--  Annotations  -->
							<#if !merge>
								<#if includeAnnotations>
									<#local annotationsList = getAnnotationsListFromDoc(summary)/>
								</#if>
								<#if annotationsList?? && annotationsList?size gt 0>
									<@printAnnotationsforDoc summary />
								<#else>
									<#if workingContext?? && workingContext=="MRL">
										<#if _section?? && _section=="Anmeth">
										<#else>
											<@com.printEmptyBox title="Assessment and conclusion by Evaluating Member State:" color="#DAE9F7"/>
										</#if>
									<#else>
										<@com.printEmptyBox title="Assessment and conclusion by RMS:" color="#DAE9F7"/>
									</#if>			
								</#if>
							</#if>	
						</#list>

						<#if merge>

							<#-- print titles -->
							<@com.emptyLine/>
							<para><emphasis role="bold"><#if ( (summaryLinks?size) > 1) >Combined summary from: <#else>Title: </#if>${summaryLinks?join("; ")}</emphasis></para>

							<#-- key info -->
							<#if keyInfoList?has_content>
								<para><emphasis role="bold">Key information:</emphasis></para>
								${keyInfoList?join("")}
							</#if>

							<#if linksList?has_content>
								<para><emphasis role="bold">Link to relevant study records:</emphasis></para>
								${linksList?join("")}
							</#if>  

							<#-- csa table (only possible for new documents which can be merged)-->
							<#local csaTable><#compress>
								<#if _section??>
									<#if _section=="Ecotox" && keyEcotox??>
										<#if keyFate?? && allSummaryList[0].documentSubType=="EstConcWaterSed">
											<#--  for PECsw and sediment in Ecotox PPP RA Aquatic Organisms  -->
											<#list allSummaryList as summary>
												<@keyFate.fateCSA summary=summary endpoint=endpoint bgcolor="#D3D3D3"/>
											</#list>
										<#else>
											<@keyEcotox.ecotoxCSA summaryList=allSummaryList bgcolor="#D3D3D3"/>
										</#if>
									<#elseif _section=="Fate" && keyFate??>
										<@keyFate.fateCSA summary=allSummaryList bgcolor="#D3D3D3"/> <#--NOTE: this does not work -->	
									<#elseif _section =="Physchem" && keyPhysChemSummary??>
										<@keyPhysChemSummary.physicalChemicalPropertiesTable _subject=entity selectedDocSubTypes=docSubTypes printTitle=false includeMetabolites=false includeImpurities=false bgcolor="#D3D3D3"/>
									<#elseif _section=="Tox" && keyTox??>
										<@keyTox.toxCSAtable summaryList=allSummaryList bgcolor="#D3D3D3"/>
									<#elseif _section=="Residues" && keyRes??>
										<@keyRes.residuesSummaryTable allSummaryList/>
									</#if>
								</#if>
							</#compress></#local>
							<#if csaTable?has_content>
								<#if _section?? && _section != "Residues">
									<para><emphasis role="bold">Key value for assessment:</emphasis></para>
								</#if>
								<para role='small'>${csaTable}</para>
							</#if>

							<#-- endocrine disrupting properties main summary and assessment of potential toxicity of the PPP -->
							<#if keyTox??>
								<#list allSummaryList as summary>
									<#if summary.documentSubType=="EndocrineDisruptingPropertiesAssessmentPest">
										<@keyTox.endocrineDisruptingPropertiesTable summary=summary bgcolor="#D3D3D3"/>
									<#elseif summary.documentSubType=="AssessmentOfPotentialToxicity" || summary.documentSubType=="PathogenicityInfectivityHumans">
										<@keyTox.assessmentPotentialToxicity summary/>	
									</#if>
								</#list>
							</#if>

							<#-- transformation products (for fate)-->
							<#if _section=="Fate" && keyFate??>
								<#local trProdTable><@keyFate.transformationProductsSummaryTable summaryList=allSummaryList bgcolor="#D3D3D3"/></#local>
								<#if trProdTable?has_content>
									<para><emphasis role="bold">Information on transformation products:</emphasis></para>
									<para role='small'>${trProdTable}</para>
								</#if>
							</#if>

							<#-- relevant metabolites groundwater -->
							<#if keyFate??>
								<#list allSummaryList as summary>
									<#if summary.documentSubType=="RelevantMetabolitesGroundWater">
										<@keyFate.relevantMetabolitesGroundWater summary=summary/>
									</#if>
								</#list>
							</#if>

							<#-- summary document of section "Other reports" in Summary and evaluation -->
							<#if keyAdm??>
								<#list allSummaryList as summary>
									<#if summary.documentSubType=="SummaryEvaluation_EU_PPP">
										<@keyAdm.summaryEvaluationPPP summary=summary/>
									</#if>
								</#list>
							</#if>
							
							<#-- ecotoxicity of metabolites (for ecotox micro AS) -->
							<#if (_section=="Ecotox" || _section=="Tox") && keyTox??>
								<#local ecotoxMetabTable><@keyTox.ecotoxicityMetabolitesTable summaryList=allSummaryList bgcolor="#D3D3D3"/></#local>
								<#if ecotoxMetabTable?has_content>
									<para><emphasis role="bold">Metabolites: </emphasis></para>
									<para role='small'>${ecotoxMetabTable}</para>
								</#if>
							</#if>

							<#-- higher tier (for ecotox) -->
							<#if higherTierList?has_content || higherTierLinksList?has_content>
								<para><emphasis role="bold">Higher tier testing:</emphasis></para>
								${higherTierList?join("")}
								<#if higherTierLinksList?has_content>
									<para><para role='indent' style="background-color:#f7f7f7">Link to relevant study records:
									${higherTierLinksList?join("")}
									</para></para>
								</#if>
							</#if>

							<#-- justification (for tox, physchem) -->
							<#if justificationList?has_content>
								<para><emphasis role="bold">Justification for classification or non-classification:</emphasis></para>
								${justificationList?join("")}
							</#if>

							<#-- discussion -->
							<#if discussionList?has_content>
								<para><emphasis role="bold">Additional information:</emphasis></para>
								${discussionList?join("")}
							</#if>

							<#-- attachments -->
							<#if attachmentsList?has_content>
								<para><emphasis role="bold">Illustrations / pictures / graphs / attachments:</emphasis></para>
								${attachmentsList?join("")}
							</#if>

							<#--  Annotations  -->
							<#if includeAnnotations>
								<#local annotationsList = [] />
            					<#list allSummaryList as summary>
									<#local annotationsList = annotationsList + getAnnotationsListFromDoc(summary) />
            					</#list>
							</#if>
							<#if annotationsList?? && annotationsList?size gt 0>
								<@printAnnotationsforDoc allSummaryList true />
							<#else>
								<#if workingContext?? && workingContext=="MRL">
									<#if _section?? && _section=="Anmeth">
									<#else>
										<@com.printEmptyBox title="Assessment and conclusion by Evaluating Member State:" color="#DAE9F7"/>
									</#if>
								<#else>
									<@com.printEmptyBox title="Assessment and conclusion by RMS:" color="#DAE9F7"/>
								</#if>			
							</#if>
						</#if>
					</#if>
				</#list>
			</#list>	
		</#if>
	</#compress>
</#macro>

<#-- 
    This macro generates an HTML table displaying annotations information for a given document 
	or list of documents. It retrieves the annotations from IUCLID documents provided as 
	parameter and filters them out based on their type ("evaluation" annotations). The macro 
	supports handling a single document or a sequence (list) of documents.
-->
<#macro printAnnotationsforDoc docList merge=false>
	<#compress>
		<#--  Initialize variable that will hold list of annotations  -->
		<#local annotationsList = [] />

		<#--  Get list of annotations from list of documents  -->
		<#if docList?is_sequence>
            <#list docList as doc>
				<#local annotationsList = annotationsList + getAnnotationsListFromDoc(doc) />
            </#list>
        <#else>
			<#local annotationsList = getAnnotationsListFromDoc(docList) />
        </#if>

		<para><emphasis role="bold">
			<#if workingContext?? && workingContext=="MRL">
				Assessment and conclusion by Evaluating Member State
			<#else>
				Assessment and conclusion by RMS
			</#if>
		</emphasis></para>
		
		<#--  Check if list of annotations has content  -->
		<#if annotationsList?size gt 0>
			<#--  Iterate over list of annotations and create rows in the table  -->
			<#list annotationsList as annotation>
				<#--  Create annotations table  -->
				<table border="1">
					<#--  Set table title  -->
					<title><#if annotationsList?size gt 1>(${annotation?counter} of ${annotationsList?size})</#if></title>

					<#--  Define columns width  -->
					<col width="25%" />
					<col width="75%" />

					<thead valign="middle">
						<tr>
							<th colspan="2" align="center"><?dbfo bgcolor="#ccc" ?>
								<emphasis role="bold">${annotation.title}</emphasis>
							</th>
						</tr>
					</thead>

					<#--  Define table body  -->
					<tbody valign="middle">
						<#--  Iterate over annotation and print admin data  -->
						<#list annotation.admin as key, value>
							<tr>
								<td><emphasis role="bold">${key}</emphasis></td>
								<td>${value}</td>
							</tr>
						</#list>
						
						<#--  Iterate over annotation and print evaluation data  -->
						<#list annotation.evaluation as key, value>
							<tr>
								<td><emphasis role="bold">${key}</emphasis></td>
								<td>${value}</td>
							</tr>
						</#list>

						<#-- Executive summary --> 
						<tr>
							<td colspan="2"><emphasis role="bold">Executive summary:</emphasis>
								<#if annotation.executiveSummary?has_content>${annotation.executiveSummary}<#else>No executive summary available</#if>
							</td>
						</tr>

						<#--  Conclusions -->
						<tr>
							<td colspan="2"><?dbfo bgcolor="#FBCEB1" ?><emphasis role="bold">Conclusions:</emphasis>
								<#if annotation.conclusions?has_content>${annotation.conclusions}<#else>No conclusions available</#if>
							</td>
						</tr>
					</tbody>
				</table>
			</#list>
		</#if>
	</#compress>
</#macro>

<#--
    This function retrieves a list of annotations for a given document. It iterates over the 
	annotation keys of the document and filters them out based on their type ("evaluation" 
	annotations). For each matching annotation, it extracts relevant information such as the 
	annotation type, name, authority, and last update date.
    The function constructs a list of annotation entries, where each entry is represented by a 
	sublist containing the document link, annotation name, annotation type, and annotation 
	authority. The resulting list of annotation entries is returned by the function.
-->
<#function getAnnotationsListFromDoc doc>
	<#--  Initialize variable that will hold list of annotations for single document  -->
	<#local annotationsList = [] />

	<#--  Get document URL and link  -->
	<#local docUrl=iuclid.webUrl.documentView(doc.documentKey) />
	<#local docLink><ulink url="${docUrl}"><@com.text doc.name/></ulink></#local>

	<#--  Get list of annotation keys  -->
	<#local annotationKeys = doc.annotations />

	<#--  Iterate over list of annotation keys  -->
	<#list annotationKeys as key>
		<#--  Initialize hash that holds annotation data  -->
		<#local annotationHash = {} />

		<#--  Initialize hash that holds administrative data for an annotation  -->
		<#local adminHash = {} />

		<#--  Initialize hash that holds evaluation data for an annotation  -->
		<#local evalHash = {} />

		<#--  Get annotation  -->
		<#local annotation = iuclid.getDocumentForKey(key) />

		<#--  Get annotation type  -->
		<#local annotationType>
			<@com.picklist annotation.AdminInfo.AnnotationType />
		</#local>

		<#--  Get annotation status  -->
		<#local annotationStatus>
			<@com.picklist annotation.AdminInfo.AnnotationStatus />
		</#local>

		<#--  Check if annotation type is equal to "evaluation"  -->
		<#if annotationType?matches("evaluation") && annotationStatus?matches("final")>
			<#--  Get annotation name  -->
			<#local annotationName>
				<#if annotation.AdminInfo.Name?has_content>
					<ulink url="${docUrl}"><@com.text annotation.AdminInfo.Name/></ulink>
				</#if>
			</#local>
			<#--  Add annotation name to annotation hash  -->
			<#local annotationHash = annotationHash + {"title" : annotationName}>

			<#--  Get annotation authority  -->
			<#local annotationAuthority>
				<@com.text annotation.AdminInfo.Authority/>
			</#local>
			<#--  Add annotation authority to admin hash  -->
			<#local adminHash = adminHash + {"Authority/organization" : annotationAuthority!}>

			<#--  Get creation date  -->
			<#assign annotationDateCreated = getDateCreated(annotation) />
			<#--  Add creation date to admin hash  -->
			<#local adminHash = adminHash + {"Date created" : annotationDateCreated!}>

			<#--  Get annotation history  -->
			<#assign historyList = getFirstHistoryOnly(annotation)!"" />

			<#--  Get last modified date  -->
			<#if historyList?has_content>
				<#local annotationDateLastModified>
					${historyList.date}
				</#local>
			</#if>
			<#--  Add last modified date to admin hash  -->
			<#local adminHash = adminHash + {"Date last modified" : annotationDateLastModified!}>

			<#--  Get annotation remarks  -->
			<#local annotationRemarks>
				<@com.richText annotation.AdminInfo.Remarks />
			</#local>
			<#--  Add remarks to admin hash  -->
			<#if annotationRemarks?has_content>
				<#local adminHash = adminHash + {"Administrative Remarks" : annotationRemarks!}>
			</#if>

			<#--  Populate annotation hash  -->
			<#local annotationHash = annotationHash + {"admin" : adminHash}>

			<#--  Get annotation agreement  -->
			<#local annotationAgreement>
				<@com.picklist annotation.EvalInfo.AgreementWithApplicantsSummary/>
			</#local>
			<#--  Add agreement to eval hash  -->
			<#local evalHash = evalHash + {"Agreement with applicant's summary" : annotationAgreement!}>

			<#--  Get annotation data waiver  -->
			<#local annotationDataWaiver>
				<@com.picklist annotation.EvalInfo.DataWaiverAcceptable/>
			</#local>
			<#--  Add data waiver to eval hash  -->
			<#if annotationDataWaiver?has_content>
				<#local evalHash = evalHash + {"Data waiver acceptable" : annotationDataWaiver!}>
			</#if>

			<#--  Get annotation reliability  -->
			<#local annotationReliability>
				<@com.picklist annotation.EvalInfo.Reliability/>
			</#local>
			<#--  Add reliability to eval hash  -->
			<#if annotationReliability?has_content>
				<#local evalHash = evalHash + {"Reliability" : annotationReliability!}>
			</#if>

			<#--  Get annotation evaluation remarks  -->
			<#local annotationEvalRemarks>
				<@com.text annotation.EvalInfo.Remarks/>
			</#local>
			<#--  Add remarks to eval hash  -->
			<#if annotationEvalRemarks?has_content>
				<#local evalHash = evalHash + {"Evaluation Remarks" : annotationEvalRemarks!}>
			</#if>

			<#--  Get annotation cross reference  -->
			<#local annotationCrossReference>
				<@com.text annotation.EvalInfo.CrossReferenceToOtherStudy/>
			</#local>
			<#--  Add cross reference to eval hash  -->
			<#if annotationCrossReference?has_content>
				<#local evalHash = evalHash + {"Cross reference" : annotationCrossReference!}>
			</#if>

			<#--  Get annotation attached evaluation  -->
			<#local annotationAttachedEvaluation = "" />

			<#list annotation.AttachedRegulatoryAuthoritiesEvaluation.Evaluation as attachment>
				<#if attachment?has_content>
					<#local attachmentKey = iuclid.getMetadataForAttachment(attachment) />

					<#if attachmentKey?has_content>
						<#local annotationAttachedEvaluation = annotationAttachedEvaluation + "${attachmentKey.filename}" />
					</#if>

					<#if attachment?has_next><?linebreak?></#if>
				</#if>
			</#list>
			<#--  Add attached evaluation to eval hash  -->
			<#if annotationAttachedEvaluation?has_content>
				<#local evalHash = evalHash + {"Attached evaluation" : annotationAttachedEvaluation!}>
			</#if>

			<#--  Populate annotation hash with eval data  -->
			<#local annotationHash = annotationHash + {"evaluation" : evalHash}>

			<#--  Get annotation conclusions  -->
			<#local annotationConclusions>
				<@com.text annotation.EvalInfo.Conclusions ""/>
			</#local>
			<#--  Populate annotation hash with conclusions  -->
			<#local annotationHash = annotationHash + {"conclusions" : annotationConclusions!}>

			<#--  Get annotation executive summary  -->
			<#local annotationExecutiveSummary>
				<@com.richText annotation.EvalInfo.ExecutiveSummary />
			</#local>
			<#--  Populate annotation hash with executive summary  -->
			<#local annotationHash = annotationHash + {"executiveSummary" : annotationExecutiveSummary!}>

			<#--  Store annotation data in list of annotations  -->
			<#local annotationsList = annotationsList + [annotationHash] />
		</#if>
	</#list>

	<#--  Return list of annotations  -->
	<#return annotationsList />
</#function>

<#macro annotationAttachedEvaluationName annotation>
	<#if annotation?has_content>
		<#list annotation.AttachedRegulatoryAuthoritiesEvaluation.Evaluation as attachments>
			<#if attachments?has_content>
				<#assign attachment = iuclid.getMetadataForAttachment(attachments) />
				
				<#if attachment?has_content>			
					<para>${attachment.filename}</para>

					<para>(size: ${attachment.size} file type: ${attachment.mediaType})</para>

					<#if attachment.remarks?has_content>
						<emphasis role="bold">Remarks for attached Evaluation:</emphasis> ${attachment.remarks}
					</#if>
				</#if>

				<#if attachments_has_next??><?linebreak?></#if>
			</#if>
		</#list>
	</#if>
</#macro>

<#--  The "getDateCreated" function retrieves the creation date of the provided document  -->
<#function getDateCreated _subject>
	<#--  Get the history list for the provided document key  -->
	<#local historyList = iuclid.getHistoryForDocumentKey(_subject.documentKey) />

	<#-- Extract "date" property from history and store it in "dateCreated" -->
	<#if historyList?has_content>
		<#list historyList as history>
			<#if history?has_content>
				<#local dateCreated>${history.date}</#local>
			<#else/>
				<#local dateCreated = "" />
			</#if>
		</#list>
	<#else>
		<#local dateCreated = "" />
	</#if>

	<#return dateCreated />
</#function>

<#function getFirstHistoryOnly _subject>
	<#local historyList = iuclid.getHistoryForDocumentKey(_subject.documentKey) />
          
	<#if !(historyList)?has_content>
		<#return [] />
	</#if>
	
	<#if historyList?has_content>                 
		<#list historyList as history>          
			<#if !(history)?has_content>
				<#return [] />          
			<#else/>
				<#return historyList[0] />                                             
			</#if>
		</#list>
	</#if>
</#function>

<#-- getSummaryRichTextSection prints a formatted rich text section from a summary object.

    Inputs:
    - summary:      The summary document object to extract data from.
    - paths:        A list of string paths. The function searches these for available content in the summary.
    - role:         (Optional) Role attribute to apply to the para element. Default is empty string.
    - printSummaryTitle: (Optional, boolean) If true, prints the summary title with a link at the top. Default is true.
    - getChildren:  (Optional, boolean) If true, prints child elements using @com.children macro instead of rich text. Default is false.
-->

<#function getSummaryRichTextSection summary paths role='' printSummaryTitle=true getChildren=false printBlockLabel=false>

	<#-- style -->
	<#local backgroundStyle = "background-color:#f7f7f7"/>
	
	<#-- initialise output -->
	<#local pathValue = ''/>

	<#-- iterate over possible paths and create full path when found -->
	<#local pathObject = studyandsummaryCom.getObjectFromPathOptions(summary, paths)/>

	<#-- get summary Title -->
	<#local summaryUrl=iuclid.webUrl.documentView(summary.documentKey) />
	<#local summaryName><@com.text summary.name ""/></#local>
	<#local summaryLink><ulink url="${summaryUrl}">${summaryName}</ulink></#local>
	
	<#-- print path if found and has content-->
	<#if pathObject?has_content>
		<#local pathValue>
			<#compress>
			<#if printSummaryTitle><para>${summaryLink}</para></#if>
			<#if getChildren>
				<#if printBlockLabel>
					<@iuclid.label for=pathObject var="blockLabel"/>
					<#if blockLabel?? && blockLabel?has_content><para role="${role}" style="${backgroundStyle}">${blockLabel}</para></#if>
				</#if>
				<@com.children path=pathObject titleEmphasis=true style=backgroundStyle printChildLabel=true printChildPathIfNoLabel=false/>
			<#else>
				<para role="${role}" style="${backgroundStyle}"><@com.richText pathObject/></para>
			</#if>
			</#compress>
		</#local>
		
	</#if>

	<#return pathValue/>

</#function>

<#-- NOTE: DEPRECATED -->
<#function getDocCategory document>

	<#-- list needed cause some might belong to different categories "above ground'MAYBE TO BE CHANGED?-->
	<#local docCategories = []/>

	<#if document?has_content>
		<#-- if document is list, get the first one to identify category -->
		<#if document?is_sequence>
			<#local document = document[0]/>
		</#if>
		

		<#list _docCategoryMap as category, docNames>
			<#if docNames?seq_contains(document.documentSubType)>
				<#local docCategories = docCategories + [category]/>
			</#if>
		</#list>
	</#if>

	<#return docCategories/>

</#function>

<#-- printSummaryVersion prints the available version of a summary.

	Description: In April2023 EU PPP summaries (mostly Ecotox, some Fate and Tox) have been changed for their harmonised versions. 
	To ensure backwards compatibility, code will be kept to print the old summaries, 
    if they exist. Therefore, for each summary type there is a condition whereby:
        - if the new summary exists, it is printed using the macro in AppendixE
        - else, if the old summary exists, it is printed using the macro in the corresponding common module
		- else, a text message is printed saying there is no summary available

    If both new and old summaries exist, only the new version will be printed - but this situation should not happen. 
	NOTE: this could be changed to printing both summaries, if needed.
	
	New and old could be empty, for cases where a summary has been totally split into different ones (such as ToxicityTerrestrialArthropods_EU_PPP)

	Inputs:
	- new: string or list with new document type(s) and subtype(s); it can be empty 
			e.g. 'ENDPOINT_SUMMARY.ToxicityToBirds' or ['ENDPOINT_SUMMARY.ToxicityToBirds', 'ENDPOINT_SUMMARY.ToxicityToBees']
	- old: string or list with old document type(s) and subtype(s); it can be empty
			e.g. 'ENDPOINT_SUMMARY.ToxicityBirds_EU_PPP' or ['ENDPOINT_SUMMARY.ToxicityBirds_EU_PPP', 'ENDPOINT_SUMMARY.ToxicityToTerrestrialArthropods_EU_PPP']
	- merge: if true, summaries are condensed into one summary (each common section will be the merge across summaries)
	- includeMetabolites: bool; if true, summaries of the selected type in metabolite datasets will also be output
			This requires the global variable _metabolites to exist and have content.
			It is applicable for reports on the active substance (where metabolite studies should be also listed)
	- includeImpurities: bool; if true, summaries of the selected type in impurity datasets will also be output
			This requires the global variable _impurities to exist and have content.
			It is applicable for reports on the active substance (where impurity studies should be also listed)
-->
<#macro printSummaryVersion subject new old merge=false includeMetabolites=true includeImpurities=true includeAnnotations=false>
    
	<#if new?has_content && containsDoc(subject, new, "", includeMetabolites, "", includeImpurities)>
		
	   <#-- call the macro in this ftl
	   		NOTE: the document type(s) need to be removed (function 'removeTypeFromDocuments')
		-->
	   <@PPPsummary subject=subject docSubTypes=removeTypeFromDocuments(new) merge=merge includeMetabolites=includeMetabolites includeImpurities=includeImpurities includeAnnotations=includeAnnotations/>
	
	<#elseif old?has_content && containsDoc(subject, old, "", includeMetabolites, "", includeImpurities)>

		<#-- check that global variable with section name exists -->
		<#if _section?? && _section=="Ecotox" && keyEcotox??>
			<#-- call the macro in the corresponding ftl, based on section variable
				NOTE: the document type(s) need to be removed (function 'removeTypeFromDocuments')
			-->
			<@keyEcotox.ecotoxPPPsummary subject=subject docSubTypes=removeTypeFromDocuments(old) merge=merge includeMetabolites=includeMetabolites includeImpurities=includeImpurities/>
		<#else>
			<@PPPsummary subject=subject docSubTypes=removeTypeFromDocuments(old) merge=merge includeMetabolites=includeMetabolites includeImpurities=includeImpurities includeAnnotations=includeAnnotations/>
		</#if>
	
	<#else>
		<#-- if both new and old are empty, print message -->
		<#if new?has_content && old?has_content>
			<para><@com.emptyLine/><emphasis role="HEAD-WoutNo">Summary</emphasis></para>
			<@com.emptyLine/>
			<para>No summary information available for this section.</para>
		</#if>
	</#if>

</#macro>

<#-- removeTypeFromDocuments removes the document type from the a list of document types and subtypes

	Inputs:
	- docList: string or list of strings containing document type and subtype e.g. ['ENDPOINT_SUMMARY.ToxicityToBirds']

	Returns:
	- docListNoType: list of strings containing document subtypes only e.g. ['ToxicityToBirds']
-->
<#function removeTypeFromDocuments docList>

	<#-- convert to sequence -->
	<#if !docList?is_sequence>
		<#local docList = [docList]/>
	</#if>

	<#-- create new sequence with type removed -->
	<#local docListNoType = []/>
	<#list docList as doc>
        <#local docListNoType = docListNoType + [doc?split('.')[1]]>
    </#list>

	<#return docListNoType/>

</#function>

<#-- function to govern dinamically the printing of a letter according to the
	numeric value passed in i, then concatenated with ")" to create a), b)...
	sections of AppendixE
-->
<#function letterIndex i>
	<#return ("abcdefghijklmnopqrstuvwxyz")[i] + ")"/>
</#function>

<#-- 
    printContentFromSections prints all summaries (ENDPOINT or FLEXIBLE) and studies (ENDPOINT or FLEXIBLE) for each subject

    Inputs:
	    - subject: ENTITY or list of ENTITIES that summaries or studies belong to
        - mixture: MIXTURE where the subject was found
		- sectionNumbers: sequence of section numbers used to filter what to print
        - documentTypes: sequence of document types to be considered
		- sectLevel: DocBook tag name to open and close sections in report
		- titleRole: role of the title for sections in report
        - subjectType: category of entities to be printed. Default value is a generic "components"
        - ignoreRepetition: bool; if true do not check that an entity had already been described in the report before printing data
		- includeStudies: bool; if true studies are printed
		- sectionNb: text value to display the regulatory data point if known beforehand
		- includeAnnotations: bool; to include or not annotations created with IUCLID annotations tool
-->
<#macro printContentFromSections subject mixture="" sectionNumbers={} documentTypes={} sectLevel="sect3" titleRole="HEAD-4" subjectType="components" ignoreRepetition=true includeStudies=true sectionNb="" includeAnnotations=false>

	<#-- Identify report and set context -->
	<#if _docNameCode??>
		<#if _docNameCode == "C."> 
			<#assign rContext = "Vol4">
		<#elseif _docNameCode == "B.">
			<#assign rContext = "Vol3">
		</#if>
	<#elseif workingContext?? && workingContext == "MRL">
		<#assign rContext = "MRL">		 
	</#if>

    <#-- Convert subject always to a list -->
	<#if !subject?is_sequence>
		<#local subject=[subject]/>
	</#if>

    <#-- Create and fill hashmap containing information on document types and subtypes found in ToC for each subject -->
    <#local data2SectHashMap = retrieveDataFromToC(subject, sectionNumbers, documentTypes)/>

    <#-- Count how many entities have data -->
    <#local nEntitiesWithData = 0>
	<#list subject as entity>
		<#if data2SectHashMap[entity.documentKey]?has_content>
			<#local nEntitiesWithData += 1>
		</#if>
	</#list>		
    
	<#if rContext == "Vol3" && (subjectType == "relevant impurities" || subjectType?contains("metabolites"))>
		<para><emphasis role="HEAD-KeyA">${subjectType?cap_first}</emphasis></para>
	</#if>

    <#-- Print an introductory paragraph stating how many entities have been found with data -->
    <#local summaryText>There <#if subject?size gt 1>are<#else>is</#if> ${subject?size} <#if subject?size gt 1>${subjectType}<#else><#if subjectType == "non-relevant impurities">non-relevant impurity<#elseif subjectType == "co-formulants, safeners or synergists">co-formulant, safener or synergist<#elseif subjectType == "safeners and synergists">safener or synergist<#elseif subjectType == "relevant impurities">relevant impurity><#elseif subjectType?contains("metabolites")>metabolite<#else>component</#if></#if></#local>
	<#if nEntitiesWithData == 0>
		<#local summaryText>${summaryText}, but<#if subject?size gt 1> none have<#else> it does not have any </#if> data.</#local>		
	<#elseif nEntitiesWithData == subject?size>
        <#local summaryText>${summaryText},<#if subject?size gt 1> all have<#else> it has </#if> data and <#if subject?size gt 1>are<#else>is</#if> shown below<#if nEntitiesWithData==1>.<#else>:</#if></#local>
    <#else>    
		<#local summaryText>${summaryText}, of which ${nEntitiesWithData}<#if nEntitiesWithData == 1> has<#else> have</#if> data and <#if nEntitiesWithData == 1>is<#else>are</#if> shown below<#if nEntitiesWithData==1>.<#else>:</#if></#local> 
    </#if>
	<#if nEntitiesWithData gt 1>
		<#local summaryText>${summaryText}
			<#list subject as entity>
				<#if data2SectHashMap[entity.documentKey]?has_content>
					<para role="indent">
						<#if rContext == "Vol4">
							- <command linkend="${entity.documentKey!}_${rContext}:Data">${entity.name}</command>
						<#elseif rContext == "Vol3" || rContext == "MRL">
							- <command linkend="${entity.documentKey!}_${includeStudies?c}">${entity.name}</command>
						</#if>	
					</para>
				</#if>	
			</#list>
		</#local>	
	</#if>
    <para>${summaryText}</para>
    
    <@com.emptyLine/>

    <#-- Create the hashmap for the mapping of section numbers to values of the "_section" global variable:
         - For SUBSTANCE ToC is the same of the AS
         - For MIXTURE ToC is the one of "EU PPP Other Mixture" working context
    -->
    <#local sectNb2LabelHashMap = {"SUBSTANCE": {"2": "Physchem", 
                                                 "3": "",
                                                 "4": "Anmeth",
                                                 "5": "Tox",
                                                 "6": "Residues",
                                                 "7": "Fate",
                                                 "8": "Ecotox",
                                                 "10": "",
                                                 "11": ""
                                                },
                                   "MIXTURE":   {"2": "Physchem",
                                                 "3": "",
                                                 "4": "Anmeth",
                                                 "5": ["Application","Efficacy"],
                                                 "6": "Tox",
                                                 "7": "Fate",
                                                 "8": "Ecotox",
                                                 "10": "",
                                                 "11": ""
                                                }              
    }/>

    <#-- PRINT SUMMARIES AND STUDIES -->
    <#list subject as entity>
        <#if data2SectHashMap[entity.documentKey]?has_content>

			<#-- Increase index for metabolites or relevant impurities -->
			<#if rContext == "Vol3" || rContext == "MRL">
				<#global substanceEntityIndex += 1/>
			</#if>

			<#-- Create sections with relative xml:ids based on context -->
			<#if rContext == "Vol4">
            	<#if _describedComponents?? && !_describedComponents?seq_contains(entity.documentKey)> 
                	<${sectLevel} xml:id="${entity.documentKey!}_${rContext}:Data">
            	<#else>
                	<${sectLevel}>
            	</#if>
			<#elseif rContext == "Vol3" || rContext == "MRL">
				<${sectLevel} xml:id="${entity.documentKey!}_${includeStudies?c}" label="-${substanceEntityIndex}">
			</#if>	

                <#-- Get the component type(s) -->
                <#local compTypes = []>
				<#if rContext == "Vol4">
                	<#if subjectType == "non-relevant impurities">
                    	<#local compTypes = getImpurityType(mixture, entity.documentKey)>
                	<#elseif subjectType == "co-formulants, safeners or synergists">
                    	<#local compTypes = getCoformulantType(mixture, entity.documentKey)>
                	</#if>
				<#elseif rContext == "Vol3" || rContext == "MRL">
					<#if subjectType == "relevant impurities">
						<#local compTypes += ["relevant impurity"]>
					<#elseif subjectType?contains("metabolites")> 
						<#if subjectType?length gt 11>
							<#local metaboliteType = subjectType[0..9] + subjectType[subjectType?index_of(" ")..subjectType?length-1]>
							<#local compTypes += [metaboliteType]>	
						<#else>
							<#local compTypes += ["metabolite"]>
						</#if>
					<#elseif subjectType == "safeners and synergists">
						<#local fullCompTypes = getCoformulantType(mixture, entity.documentKey)>

						<#-- Discard co-formulant types from full list -->
						<#local compTypes = []>
						<#list fullCompTypes as compType>
							<#if ["safener","synergist"]?seq_contains(compType)>
								<#local compTypes += [compType]>
							</#if>
						</#list>		
					</#if>
				</#if>	        

                <title role="${titleRole}">${entity.name} <#if compTypes?has_content>(${compTypes?join(", ")})</#if></title>

                <#-- Link this section to Identity's -->
                <#if subjectType == "non-relevant impurities" && rContext == "Vol4">
                    <#local refSub = iuclid.getDocumentForKey(entity.ReferenceSubstance.ReferenceSubstance)/>
                    <#if refSub?has_content>
                        <#local refSubName><@com.value refSub.ReferenceSubstanceName ""/></#local>
                        <para>Link to impurity in Identity section: <emphasis role="underline"><command linkend="${refSub.documentKey!}">${refSubName}</command></emphasis></para>
                        <@com.emptyLine/>
                    </#if>    
                </#if>        

                <#if !ignoreRepetition && _describedComponents?? && rContext == "Vol4">
                    <#if !_describedComponents?seq_contains(entity.documentKey)>
                        <#global _describedComponents += [entity.documentKey]> <#-- append to list of components already described in order not to repeat -->
                        <@printContentBySection entity data2SectHashMap[entity.documentKey] sectNb2LabelHashMap sectLevel titleRole includeStudies sectionNb includeAnnotations/>
                    <#else>
                        <para><emphasis role="underline"><command linkend="${entity.documentKey!}_${rContext}:Data">${entity.name}</command></emphasis> is already described above.</para>
                    </#if> 
                <#else>
                    <#if _describedComponents?? && rContext == "Vol4">
                        <#global _describedComponents += [entity.documentKey]> <#-- append in any case -->
                    </#if>    
                    <@printContentBySection entity data2SectHashMap[entity.documentKey] sectNb2LabelHashMap sectLevel titleRole includeStudies sectionNb includeAnnotations/>   
                </#if>   
            </${sectLevel}>
            <@com.emptyLine/>        
        </#if>
    </#list>
</#macro>

<#-- 
    printContentBySection prints the subject summaries (ENDPOINT or FLEXIBLE) and studies (ENDPOINT or FLEXIBLE) by section number

    Inputs:
        - subject: ENTITY that the summaries and studies belong to
	    - dataHash: tocEntityData hashmap built by "section_tree_node" for an entity
        - sectNb2LabelHashMap: hash mapping section numbers to _global variable values 
		- sectLevel: DocBook tag name to open and close sections in report
		- titleRole: role of the title for sections in report
		- includeStudies: bool; if true studies are printed
		- sectionNb: text value to display the regulatory data point if known beforehand
		- includeAnnotations: bool; to include or not annotations created with IUCLID annotations tool
-->
<#macro printContentBySection subject dataHash sectNb2LabelHashMap sectLevel="sect3" titleRole="HEAD-4" includeStudies=true sectionNb="" includeAnnotations=false>

    <#local entityType = subject.documentType>

    <#local inSectLevelNumber = (sectLevel[4]?number+1)>
    <#if inSectLevelNumber gt 4><#local inSectLevelNumber = 4></#if> <#-- 5 is the lowest-level but text is too small so limit -->
    <#local inSectLevel = "sect" + inSectLevelNumber>

    <#local inTitleRoleNumber = (titleRole[5]?number+1)>
    <#if inTitleRoleNumber gt 5><#local inSectLevelNumber = 5></#if> <#-- same logic as above -->
    <#local inTitleRole = "HEAD-" + inTitleRoleNumber>
	
	<#local sectionIndex = 0>
    <#-- Loop over all sections and subsections with data -->
    <#list dataHash as sectNb, sectionData>

		<#local sectionIndex += 1>

        <#-- Set "_section" global variable value depending on sectNb -->
        <#if entityType == "SUBSTANCE">
            <#if sectNb == "11.3"> <#-- "RelevantMetabolitesGroundWater" -->
                <#global _section = "Fate"> 
            <#elseif sectNb == "11.4"> <#-- EndocrineDisruptingPropertiesAssessmentPest -->
                <#global _section = "Tox"> 
            <#elseif sectNb?starts_with("10") || sectNb?starts_with("11")>
                <#global _section = sectNb2LabelHashMap[entityType][sectNb[0..1]]>
            <#else>    
                <#global _section = sectNb2LabelHashMap[entityType][sectNb[0]]>
            </#if>
        <#elseif entityType == "MIXTURE">
            <#if sectNb?starts_with("5")>
                <#if sectNb == "5.1"> <#-- Efficacy --> 
                    <#global _section = sectNb2LabelHashMap[entityType][sectNb[0]][1]> 
                <#elseif sectNb == "5.2"> <#-- Application -->
                    <#global _section = sectNb2LabelHashMap[entityType][sectNb[0]][0]> 
                </#if>
            <#elseif sectNb?starts_with("10") || sectNb?starts_with("11")>
                <#global _section = sectNb2LabelHashMap[entityType][sectNb[0..1]]>    
            <#else>    
                <#global _section = sectNb2LabelHashMap[entityType][sectNb[0]]>    
            </#if>        
        </#if>        

        <${inSectLevel} role="NotInToc">
            <#local sectName = sectionData["sectName"]>
            <title role="${inTitleRole}">${sectName} (${sectNb}.)</title>

            <#if sectNb == "3" && entityType == "MIXTURE"> <#-- "ProtectionMeasures" -->
                <@keyAdm.protectionMeasures subject "" true/> 
            <#elseif sectNb == "10.1" && entityType == "SUBSTANCE" || sectNb == "10" && entityType == "MIXTURE"> <#-- "Ghs" -->
                <@keyCl.classificationAndLabellingTable subject "#d3d3d3"/>
            <#elseif sectNb == "10.2" && entityType == "SUBSTANCE"> <#-- "PbtAssessment" -->
                <@keyPbt.pbtandvPvBcriteriaAndJustification subject inSectLevel inTitleRole/>
            <#elseif sectNb == "11.1" && entityType == "SUBSTANCE"> <#-- "AssessmentOtherAuthorities" -->
                <@keyAdm.assessmentOtherAuthorities subject true "#d3d3d3"/>
            <#else> 
                <#-- Loop over different types and subtypes found for the subsection -->
                <#local subTypesHash = sectionData["subTypes"]>
                <#list subTypesHash as type, subTypes>
                    <#list subTypes as subType>
                        <#-- Invoke appendixE macros and functions -->
                        <#if type?ends_with("SUMMARY")>
                            <#if (subType?ends_with("EU_PPP") || subType == "EffectsBioMethodSewageTreatment") && (_section == "Ecotox" || _section == "Fate")>
                                <#local docTypeSubType>${type}.${subType}</#local>
                                <@printSummaryVersion subject=subject new="" old=docTypeSubType includeMetabolites=false includeImpurities=false/>
                            <#else>
                                <@PPPsummary subject=subject docSubTypes=subType includeMetabolites=false includeImpurities=false/>
                            </#if>
                        <#elseif type?ends_with("RECORD")>
							<#if rContext == "Vol4">
                            	<#local sectionCode = sectNb?string + ".">
							<#elseif rContext == "Vol3" || rContext == "MRL">
								<#-- If not hard-coded use section number from ToC, but this should not happen -->
								<#if !sectionNb?has_content><#local sectionNumber = sectNb><#else><#local sectionNumber = sectionNb></#if>
								<#local sectionCode = sectionNumber?string + "-" + substanceEntityIndex + "." + sectionIndex + "."/>
							</#if>	
                            <@appendixEstudies subject=subject docSubTypes=subType includeMetabolites=false includeStudies=includeStudies sectionNb=sectionCode includeImpurities=false includeAnnotations=includeAnnotations/>
                        </#if>
                    </#list>
                </#list>
            </#if>    
        </${inSectLevel}>
        <@com.emptyLine/>
    </#list>
</#macro>

<#-- Function that uses new ToC approach to check if subjects have data to be retrieved -->
<#function docsInToc subject sectionNumbers={} documentTypes={}>

	<#-- Convert subject always to a list -->
	<#if !subject?is_sequence>
		<#local subject=[subject]/>
	</#if>
	
	<#-- Create and fill hashmap containing information on document types and subtypes found in ToC for each subject -->
    <#local data2SectHashMap = retrieveDataFromToC(subject, sectionNumbers, documentTypes)/>

	<#-- Return true if at least one of the subjects has data -->
	<#list data2SectHashMap as entity, tocEntityData>
        <#if tocEntityData?has_content>
            <#return true>
        </#if>
    </#list>

	<#return false> 
</#function>

<#-- Function that iterates through ToC for each entity, saving sections having summaries or studies along with their document types and subtypes -->
<#function retrieveDataFromToC subject sectionNumbers={} documentTypes={}>

	<#-- Convert subject always to a list -->
	<#if !subject?is_sequence>
		<#local subject=[subject]/>
	</#if>

	<#-- Make sure that there are IUCLID sections to be searched -->
    <#if !sectionNumbers?has_content>
        <#assign sectionNumbers = {"SUBSTANCE": ["2","3","4","5","6","7","8","10","11"],
                                   "MIXTURE":   ["2","3","4","5","6","7","8","10","11"]
                                  }  
        />
    <#else>
        <#assign sectionNumbers = sectionNumbers>    
    </#if>

    <#-- Make sure that there are document types to be found -->
    <#if !documentTypes?has_content>
        <#assign documentTypes = {"SUBSTANCE": ["ENDPOINT_STUDY_RECORD","ENDPOINT_SUMMARY","FLEXIBLE_RECORD","FLEXIBLE_SUMMARY"],
                                  "MIXTURE":   ["ENDPOINT_STUDY_RECORD","ENDPOINT_SUMMARY","FLEXIBLE_RECORD","FLEXIBLE_SUMMARY"]
                                 }
        />
    <#else>
        <#assign documentTypes = documentTypes>    
    </#if>

	<#local data2SectHashMap={}/>

    <#-- Iterates through ToC and save sections that have summaries or studies to be printed along with the document types and subtypes -->
    <#list subject as entity>
        <#assign tocEntityData = {}>
        <#if entity.documentType=="SUBSTANCE" || entity.documentType=="MIXTURE">
            <#assign entityDocType = entity.documentType> <#-- save the document type for macro "section_tree_node" -->
            <#assign toc = iuclid.localizeTreeFor(entityDocType, entity.submissionType, entity.documentKey)/>
            <#recurse toc/> <#-- this calls automatically the macro "section_tree_node" through name recognition -->
        </#if>    

        <#local data2SectHashMap += {entity.documentKey: tocEntityData}>   
    </#list>

	<#return data2SectHashMap>
</#function>

<#-- Function that hard-codes the section number depending on the document subType for Legacy PPP records and summaries -->
<#function setLegacySectNumber docSubType>
    <#local sectNb = "">

    <#-- SUBSTANCE -->
    <#if docSubType == "AnalyticalInformation">
        <#local sectNb = "2.4">

    <#elseif docSubType == "RouteDegSoil_EU_PPP">
        <#local sectNb = "7.1.1">

    <#elseif docSubType == "EffectsBioMethodSewageTreatment">
        <#local sectNb = "8.8">

    <#-- MIXTURE -->
    <#elseif docSubType == "ToxicityPlants_EU_PPP"> 
        <#local sectNb = "8.2.1.4">

    <#-- SUBSTANCE AND MIXTURE -->
    <#elseif docSubType == "BiodegradationInSoil_EU_PPP">
        <#if entityDocType == "SUBSTANCE">
            <#local sectNb = "7.1.1">
        <#else>
            <#local sectNb = "7.1.1.1">
        </#if>

    <#elseif docSubType == "BiodegradationInWaterAndSedimentSimulationTests_EU_PPP" || docSubType == "RouteDegWaterSed_EU_PPP">
        <#if entityDocType == "SUBSTANCE">
            <#local sectNb = "7.2.2.2-4">
        <#else>
            <#local sectNb = "7.2.1">
        </#if>

    <#elseif docSubType == "ToxicityBirds_EU_PPP">
        <#local sectNb = "8.1.1">

    <#elseif docSubType == "ToxicityToOtherAboveGroundOrganisms_EU_PPP">
        <#if entityDocType == "SUBSTANCE">
            <#local sectNb = "8.1.4">
        <#else>
            <#local sectNb = "8.1.2">
        </#if>

    <#elseif docSubType == "Short-termToxicityToFish_EU_PPP">
        <#if entityDocType == "SUBSTANCE">
            <#local sectNb = "8.2.1">
        <#else>
            <#local sectNb = "8.2.1.1">
        </#if>

    <#elseif docSubType == "LongTermToxicityToFish_EU_PPP">
        <#if entityDocType == "SUBSTANCE">
            <#local sectNb = "8.2.2.1-2">
        <#else>
            <#local sectNb = "8.2.2.1">
        </#if>

    <#elseif docSubType == "BioaccumulationAquaticSediment_EU_PPP">
        <#if entityDocType == "SUBSTANCE">
            <#local sectNb = "8.2.2.3">
        <#else>
            <#local sectNb = "8.2.3">
        </#if>

    <#elseif docSubType == "ShortTermToxicityAquaInvert_EU_PPP">
        <#if entityDocType == "SUBSTANCE">
            <#local sectNb = "8.2.4">
        <#else>
            <#local sectNb = "8.2.1.2">
        </#if>

    <#elseif docSubType == "LongTermToxicityToAquaticInvertebrates_EU_PPP">
        <#if entityDocType == "SUBSTANCE">
            <#local sectNb = "8.2.5.1-3">
        <#else>
            <#local sectNb = "8.2.2.2">
        </#if>
        
    <#elseif docSubType == "SedimentToxicity_EU_PPP">
        <#if entityDocType == "SUBSTANCE">
            <#local sectNb = "8.2.5.4">
        <#else>
            <#local sectNb = "8.2.2.3">
        </#if>

    <#elseif docSubType == "ToxicityToAquaticAlgae_EU_PPP">
        <#if entityDocType == "SUBSTANCE">
            <#local sectNb = "8.2.6">
        <#else>
            <#local sectNb = "8.2.1.3">
        </#if>
        
    <#elseif docSubType == "ToxicityTerrestrialArthropods_EU_PPP">
        <#if entityDocType == "SUBSTANCE">
            <#local sectNb = "8.3.1">
        <#else>
            <#local sectNb = "8.3">
        </#if>

    <#elseif docSubType == "ToxicitySoilMacroorganisms_EU_PPP">
        <#local sectNb = "8.4.1">

    <#elseif docSubType == "ToxicityToSoilMicroorganisms_EU_PPP">
        <#local sectNb = "8.5">

    <#elseif docSubType == "ToxicityToTerrestrialPlants_EU_PPP">
        <#local sectNb = "8.6">
    </#if>

    <#return sectNb>         
</#function>

<#-- Function that sorts a sequence of Document keys by the linked Entity or Document name -->
<#function sortByName docs=[]>
    <#local names = []>
    <#list docs as doc>
        <#local names += [doc.name]>
    </#list>
    <#local sortedNames = names?sort>

    <#local sortedDocs = []>
    <#list sortedNames as name>
        <#list docs as doc>
            <#if doc.name == name>
                <#local sortedDocs = com.addDocumentToSequenceAsUnique(doc, sortedDocs)/>
            </#if>
        </#list>
    </#list>

    <#return sortedDocs>
</#function>

<#-- Function that returns the type(s) of the given impurity -->
<#function getImpurityType mixture impurityKey>
    <#local impTypes = []>

    <#local compositionList = iuclid.getSectionDocumentsForParentKey(mixture.documentKey, "FLEXIBLE_SUMMARY", "Impurities") />
    <#list compositionList as composition>
        <#local impurityList = composition.ListOfImpurities.Impurities/>
        <#list impurityList as impurity>
            <#if impurity.ImpurityLink?has_content>
                <#local substance = iuclid.getDocumentForKey(impurity.ImpurityLink)/>
                <#if substance.documentKey == impurityKey>
                    <#local impType><@com.value impurity.Type ""/></#local>
                    <#local impTypes += [impType]>
                </#if>    
            </#if>
        </#list>
    </#list>
    <#return impTypes>
</#function>

<#-- Function that returns the type(s) of the given co-formulant or any component in MIXTURE composition documents -->
<#function getCoformulantType mixture componentKey recursive=true parsedMixtures=[]>
    <#local compTypes = []>
    
	<#-- assign parsed mixtures -->
    <#local parsedMixtures =  parsedMixtures + [mixture.documentKey]/>

	<#-- search for component -->
    <#local compositionList = iuclid.getSectionDocumentsForParentKey(mixture.documentKey, "FLEXIBLE_RECORD", "MixtureComposition") />
    <#list compositionList as composition>
        <#local componentsList = composition.Components.Components/>
        <#list componentsList as component>
            <#if component.Reference?has_content>
                <#local reference = iuclid.getDocumentForKey(component.Reference)/>
                <#if reference.documentKey == componentKey>
                    <#local compType><@com.value component.Function ""/></#local>
					<#-- make sure type is one of the valid ones -->
                    <#if !['active substance','active substance (other, not to be assessed)','relevant impurity']?seq_contains(compType)>
						<#if !['safener','synergist']?seq_contains(compType)>
                        	<#local compType>co-formulant [${compType}]</#local>
                    	</#if>
                    	<#local compTypes += [compType]>
					</#if>	
                </#if>

				<#-- if mixture and recursive is true, call function again -->
				<#if reference.documentType=="MIXTURE" && recursive && !parsedMixtures?seq_contains(reference.documentKey)>
                    <#local compTypes = compTypes + getCoformulantType(reference, componentKey, recursive, parsedMixtures)/>
                </#if>    
            </#if>
        </#list>
    </#list>
	
	<#-- Remove type duplicates if any -->
	<#local uniqueCompTypes = []>
	<#list compTypes as compType>
		<#if !uniqueCompTypes?seq_contains(compType)>
			<#local uniqueCompTypes += [compType]>
		</#if>
	</#list>

    <#return uniqueCompTypes>
</#function>

<#--
    This macro iterates over the table of contents using the #recurse command, and it fills a hashmap containing information on
    document types and subtypes that exist for each IUCLID section

    Returns: tocEntityData
        Hashmap where keys are section numbers and values hashes storing the section name and another hash for document types and subtypes found:
            - <sectNb>: section number
                - sectName: section title
                - subTypes: hash of document subtypes found, structured like this:
                    - {"<docType>": list of subtypes having "ENDPOINT_STUDY_RECORD", "FLEXIBLE_RECORD", "ENDPOINT_SUMMARY" or "FLEXIBLE_SUMMARY" as document type}               
        E.g.:  
            {"4": 
                {"sectName": "Analytical methods", "subTypes": {"ENDPOINT_STUDY_RECORD": ["AnalyticalMethods"]}}
            ..}       
-->
<#macro "section_tree_node">

    <#local contents=(.node.content)!/>

    <#-- Go through all contents in node -->
    <#if contents?has_content>
        <#list contents as doc>
            <#-- Get document type -->
            <#local docType = doc.documentType />

            <#if documentTypes[entityDocType]?seq_contains(docType)>

                <#local docTypeSubType>${docType}.${doc.documentSubType}</#local>

                <#-- Get section number -->
                <#local sectNb = toc.nodeFor[docTypeSubType].number?string />

                <#-- Deal with section number if missing (Legacy PPP summaries and Legacy Spectra) -->
                <#if !sectNb?has_content>
                    <#local sectNb = setLegacySectNumber(doc.documentSubType)>
                </#if>

                <#-- Check if sectNb is among those to be searched -->
                <#local targetSection = false>
                <#list sectionNumbers[entityDocType] as sectionNumber>
                    <#if sectNb?starts_with(sectionNumber)>
                        <#local targetSection = true>
                        <#break>
                    </#if>       
                </#list>

                <#if targetSection>
                    <#-- Get section name -->
                    <#local sectName = toc.nodeFor[docTypeSubType].title />

                    <#-- Get document subtype -->
                    <#local docSubType = doc.documentSubType />

                    <#-- Reset data structures for tocEntityData reconstruction -->
                    <#local subTypesHash = {}>
                    <#local subTypesListForType = []> 

                    <#if !tocEntityData?keys?seq_contains(sectNb)> <#-- if section is new then add new entry -->
                        <#assign tocEntityData += {sectNb: {"sectName": sectName, "subTypes": {docType: [docSubType]}}}>
                    <#else>
                        <#if !tocEntityData[sectNb]["subTypes"]?keys?seq_contains(docType)> <#-- if document type is new then add key to subTypes hash -->
                            <#local subTypesHash = tocEntityData[sectNb]["subTypes"]>
                            <#local subTypesHash += {docType: [docSubType]}>
                        <#elseif !tocEntityData[sectNb]["subTypes"][docType]?seq_contains(docSubType)> <#-- if document subtype is new then add it to list -->
                            <#local subTypesListForType = tocEntityData[sectNb]["subTypes"][docType]>
                            <#local subTypesListForType += [docSubType]>
                        </#if>

                        <#-- Reconstruct tocEntityData if needed -->
                        <#if subTypesHash?has_content || subTypesListForType?has_content>
                            <#local tocEntityDataCopy = tocEntityData>
                            <#assign tocEntityData = {}>
                            <#list tocEntityDataCopy as sectNb2, sectNbHash2>
                                <#if sectNb2 != sectNb> <#-- copy section data as it is -->
                                    <#assign tocEntityData += {sectNb2: sectNbHash2}>
                                <#else>
                                    <#if subTypesListForType?has_content> <#-- if true need to rebuild subTypes hash first -->
                                        <#local subTypesHashCopy = sectNbHash2["subTypes"]>
                                        <#local subTypesHash = {}>
                                        <#list subTypesHashCopy as docType2, docSubTypes2>
                                            <#if docType2 != docType> <#-- copy document subtypes list as it is -->
                                                <#local subTypesHash += {docType2: docSubTypes2}>
                                            <#else>
                                                <#local subTypesHash += {docType: subTypesListForType}>
                                            </#if>
                                        </#list>
                                    </#if>
                                    <#assign tocEntityData += {sectNb: {"sectName": sectNbHash2["sectName"], "subTypes": subTypesHash}}>           
                                </#if>
                            </#list>
                        </#if>         
                    </#if>
                </#if> 
            </#if>       
        </#list>
    </#if>

    <#recurse/>
</#macro>