/*
  This file is part of t8code.
  t8code is a C library to manage a collection (a forest) of multiple
  connected adaptive space-trees of general element classes in parallel.

  Copyright (C) 2015 the developers

  t8code is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  t8code is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with t8code; if not, write to the Free Software Foundation, Inc.,
  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/** \file t8_cmesh_readmshfile.h
 * We define a function here that serves to open a mesh file generated by
 * GMSH and constructing a cmesh from it.
 */

#ifndef T8_CMESH_READMSHFILE_H
#define T8_CMESH_READMSHFILE_H

#include <t8.h>
#include <t8_eclass.h>
#include <t8_cmesh/t8_cmesh.h>

/** The supported .msh file versions.
 * Currently, we only support gmsh's file version 4 in ASCII format.
 */
#define T8_CMESH_N_SUPPORTED_MSH_FILE_VERSIONS 1

/** The gmsh file versions that are supported. Currently version 4 in ASCII format */
const int t8_cmesh_supported_msh_file_versions[T8_CMESH_N_SUPPORTED_MSH_FILE_VERSIONS] = { 4 };

T8_EXTERN_C_BEGIN ();

/* put declarations here */

/** Read a .msh file and create a cmesh from it.
 * \param [in]    fileprefix        The prefix of the mesh file.
 *                                  The file fileprefix.msh is read.
 * \param [in]    partition         If true the file is only opened on one process
 *                                  specified by the \a master argument and saved as
 *                                  a partitioned cmesh where each other process does not
 *                                  have any trees.
 * \param [in]    comm              The MPI communicator with which the cmesh is to be committed.
 * \param [in]    dim               The dimension to read from the .msh files. The .msh format
 *                                  can store several dimensions of the mesh and therefore the
 *                                  dimension to read has to be set manually.
 * \param [in]    master            If partition is true, a valid MPI rank that will
 *                                  read the file and store all the trees alone.
 * \param [in]    use_cad_geometry  Read the parameters of a parametric msh file and use the
 *                                  cad geometry.
 * \return        A committed cmesh holding the mesh of dimension \a dim in the
 *                specified .msh file.
 */
t8_cmesh_t
t8_cmesh_from_msh_file (const char *fileprefix, int partition, sc_MPI_Comm comm, int dim, int master,
                        int use_cad_geometry);

T8_EXTERN_C_END ();

#endif /* !T8_CMESH_READMSHFILE_H */
