/*
 * SPDX-FileCopyrightText: 2023 Helmholtz-Zentrum für Infektionsforschung GmbH (HZI) <PiaPost@helmholtz-hzi.de>
 *
 * SPDX-License-Identifier: AGPL-3.0-or-later
 */
import { differenceInDays, differenceInMonths, differenceInWeeks, differenceInYears, format, intervalToDuration, } from 'date-fns';
import { Injectable } from '@angular/core';
import * as i0 from "@angular/core";
import * as i1 from "@ngx-translate/core";
export class DateService {
    constructor(translateService) {
        this.translateService = translateService;
    }
    getDuration(interval) {
        const [start, end] = interval;
        return intervalToDuration({ start, end });
    }
    /**
     * Returns true if a duration is regular, e.g. if the duration is exactly 1 unit
     */
    isDurationRegular(duration) {
        let isRegular = Object.values(duration).filter((value) => value === 1).length === 1;
        if (!isRegular) {
            isRegular = duration.days === 7;
        }
        return isRegular;
    }
    getLabelForInterval(timeSeriesBeginning, interval) {
        const [start, end] = interval;
        const duration = this.getDuration(interval);
        const isDurationRegular = this.isDurationRegular(duration);
        if (duration.hours) {
            return format(start, 'dd.MM.yyyy HH:mm') + ' - ' + format(end, 'HH:mm');
        }
        if (isDurationRegular) {
            return this.getLabelForRegularInterval(timeSeriesBeginning, start, duration);
        }
        return this.getLabelForIrregularInterval(timeSeriesBeginning, start, duration);
    }
    getLabelForRegularInterval(timeSeriesBeginning, start, duration) {
        const { months, days, years, hours } = duration;
        if (hours) {
            return format(start, 'dd.MM.yyyy HH:mm');
        }
        const index = this.getLastIndexForInterval(timeSeriesBeginning, start, duration);
        if (months) {
            return `${this.translateService.instant('CHARTS.FEEDBACK_STATISTIC.MONTH')} ${index}`;
        }
        if (days === 1) {
            return `${this.translateService.instant('CHARTS.FEEDBACK_STATISTIC.DAY')} ${index}`;
        }
        if (days === 7) {
            return `${this.translateService.instant('CHARTS.FEEDBACK_STATISTIC.WEEK')} ${index}`;
        }
        if (years) {
            return `${this.translateService.instant('CHARTS.FEEDBACK_STATISTIC.YEAR')} ${index}`;
        }
        return '';
    }
    getLabelForIrregularInterval(timeSeriesBeginning, start, duration) {
        const { months, days, years } = duration;
        const lastIndex = this.getLastIndexForInterval(timeSeriesBeginning, start, duration);
        if (months) {
            return `${this.translateService.instant('CHARTS.FEEDBACK_STATISTIC.MONTHS')} ${this.returnIndexRangeString(months, lastIndex)}`;
        }
        if (days) {
            if (days % 7 === 0) {
                return `${this.translateService.instant('CHARTS.FEEDBACK_STATISTIC.WEEKS')} ${this.returnIndexRangeString(days / 7, lastIndex)}`;
            }
            return `${this.translateService.instant('CHARTS.FEEDBACK_STATISTIC.DAYS')} ${this.returnIndexRangeString(days, lastIndex)}`;
        }
        if (years) {
            return `${this.translateService.instant('CHARTS.FEEDBACK_STATISTIC.YEARS')} ${this.returnIndexRangeString(years, lastIndex)}`;
        }
        return '';
    }
    returnIndexRangeString(duration, lastIndex) {
        return `${lastIndex + 1 - duration}-${lastIndex}`;
    }
    getLastIndexForInterval(timeSeriesBeginning, start, duration) {
        const { months, days, years } = duration;
        if (years) {
            return differenceInYears(start, timeSeriesBeginning) + years;
        }
        if (months) {
            return differenceInMonths(start, timeSeriesBeginning) + months;
        }
        if (days) {
            if (days % 7 === 0) {
                return differenceInWeeks(start, timeSeriesBeginning) + days / 7;
            }
            return differenceInDays(start, timeSeriesBeginning) + days;
        }
        throw new Error(`Unsupported duration: ${JSON.stringify(duration)}`);
    }
    isIntervalWithDateObjects(p) {
        return p && p.start instanceof Date && p.end instanceof Date;
    }
    static { this.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "17.3.5", ngImport: i0, type: DateService, deps: [{ token: i1.TranslateService }], target: i0.ɵɵFactoryTarget.Injectable }); }
    static { this.ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "17.3.5", ngImport: i0, type: DateService, providedIn: 'root' }); }
}
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "17.3.5", ngImport: i0, type: DateService, decorators: [{
            type: Injectable,
            args: [{ providedIn: 'root' }]
        }], ctorParameters: () => [{ type: i1.TranslateService }] });
//# sourceMappingURL=data:application/json;base64,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