/*
 * SPDX-FileCopyrightText: 2021 Helmholtz-Zentrum für Infektionsforschung GmbH (HZI) <PiaPost@helmholtz-hzi.de>
 *
 * SPDX-License-Identifier: AGPL-3.0-or-later
 */

import { fakeAsync, TestBed, tick } from '@angular/core/testing';
import { PageManager } from './page-manager.service';
import { CurrentUser } from './current-user.service';
import { ComplianceManager } from './compliance-manager.service';
import { MockProvider } from 'ng-mocks';
import { Subject } from 'rxjs';
import { Role } from '../psa.app.core/models/user';
import SpyObj = jasmine.SpyObj;
import { FeedbackStatisticsService } from '../pages/feedback-statistics/feedback-statistics.service';

describe('PageManager', () => {
  let service: PageManager;

  let user: SpyObj<CurrentUser>;
  let complianceManager: SpyObj<ComplianceManager>;
  let complianceChangeSubject: Subject<void>;
  let feedbackStatisticsService: SpyObj<FeedbackStatisticsService>;

  beforeEach(() => {
    user = jasmine.createSpyObj<CurrentUser>('CurrentUser', [
      'hasRole',
      'isProband',
    ]);
    complianceChangeSubject = new Subject<void>();
    complianceManager = jasmine.createSpyObj<ComplianceManager>(
      'ComplianceManager',
      ['userHasCompliances', 'isInternalComplianceActive'],
      {
        complianceDataChangesObservable: complianceChangeSubject.asObservable(),
      }
    );

    feedbackStatisticsService = jasmine.createSpyObj<FeedbackStatisticsService>(
      'FeedbackStatisticsService',
      ['hasFeedbackStatisticsForProband']
    );
    feedbackStatisticsService.hasFeedbackStatisticsForProband.and.resolveTo(
      true
    );

    TestBed.configureTestingModule({
      providers: [
        PageManager,
        MockProvider(CurrentUser, user),
        MockProvider(ComplianceManager, complianceManager),
        MockProvider(FeedbackStatisticsService, feedbackStatisticsService),
      ],
    });
  });

  describe('navPagesObservable', () => {
    it('should emit a list of pages for Forscher role', fakeAsync(() => {
      // Arrange
      setRole('Forscher');
      const pagesCallback = jasmine.createSpy();
      service = TestBed.inject(PageManager);

      // Act
      service.navPages$.subscribe(pagesCallback);
      tick();

      // Assert
      expect(pagesCallback).toHaveBeenCalledTimes(1);
      expect(pagesCallback).toHaveBeenCalledWith([
        {
          name: 'SIDENAV.STUDY',
          path: ['study'],
          subpaths: ['study', 'study/'],
        },
        {
          name: 'SIDENAV.ADMINISTRATION',
          path: ['questionnaires/admin'],
          subpaths: [
            'questionnaires/',
            'questionnaire/',
            'questionnaire',
            'edit',
          ],
        },
        {
          name: 'SIDENAV.PROBANDS',
          path: ['probands'],
          subpaths: [
            'probands',
            'questionnaireInstances/',
            'laboratory-results;user_id=',
            'sample-management/',
            'view',
          ],
        },
        {
          name: 'SIDENAV.COMPLIANCE',
          path: ['compliance/setup'],
          subpaths: ['compliance/'],
        },
        {
          name: 'SIDENAV.FEEDBACK_STATISTICS',
          path: ['feedback-statistics'],
          subpaths: ['feedback-statistics/', 'feedback-statistics/edit'],
        },
      ]);
    }));

    it('should emit a list of pages for Untersuchungsteam role', fakeAsync(() => {
      // Arrange
      setRole('Untersuchungsteam');
      const pagesCallback = jasmine.createSpy();
      service = TestBed.inject(PageManager);

      // Act
      service.navPages$.subscribe(pagesCallback);
      tick();

      // Assert
      expect(pagesCallback).toHaveBeenCalledTimes(1);
      expect(pagesCallback).toHaveBeenCalledWith([
        {
          name: 'SIDENAV.STUDY',
          path: ['study'],
          subpaths: ['study', 'study/'],
        },
        {
          name: 'SIDENAV.PROBANDS',
          path: ['probands'],
          subpaths: [
            'probands',
            'probands/',
            'sample-management/',
            'questionnaires/user?user_id',
            'questionnaire/',
          ],
        },
        {
          name: 'SIDENAV.PLANNED_PROBANDS',
          path: ['planned-probands'],
          subpaths: ['planned-probands/'],
        },
        {
          name: 'SIDENAV.COMPLIANCE_MANAGEMENT',
          path: ['compliance/management'],
          subpaths: ['compliance/'],
        },
      ]);
    }));

    it('should emit a list of pages for SysAdmin role', fakeAsync(() => {
      // Arrange
      setRole('SysAdmin');
      const pagesCallback = jasmine.createSpy();
      service = TestBed.inject(PageManager);

      // Act
      service.navPages$.subscribe(pagesCallback);
      tick();

      // Assert
      expect(pagesCallback).toHaveBeenCalledTimes(1);
      expect(pagesCallback).toHaveBeenCalledWith([
        { name: 'SIDENAV.HOME', path: ['home'], subpaths: ['home'] },
        {
          name: 'SIDENAV.USER_ADMINISTRATION',
          path: ['internalUsers'],
          subpaths: ['internalUsers'],
        },
        {
          name: 'SIDENAV.STUDIES',
          path: ['studies'],
          subpaths: ['studies', 'studies/Evaluation/users'],
        },
        {
          name: 'SIDENAV.LOGS',
          path: ['deletelogs'],
          subpaths: ['deletelogs'],
        },
        {
          name: 'SIDENAV.PUBLIC_API',
          path: ['public-api'],
          subpaths: ['public-api'],
        },
      ]);
    }));

    it('should emit a list of pages for ProbandenManager role', fakeAsync(() => {
      // Arrange
      setRole('ProbandenManager');
      const pagesCallback = jasmine.createSpy();
      service = TestBed.inject(PageManager);

      // Act
      service.navPages$.subscribe(pagesCallback);
      tick();

      // Assert
      expect(pagesCallback).toHaveBeenCalledTimes(1);
      expect(pagesCallback).toHaveBeenCalledWith([
        {
          name: 'SIDENAV.STUDY',
          path: ['study'],
          subpaths: ['study', 'study/'],
        },
        {
          name: 'SIDENAV.PROBANDS',
          path: ['probands-personal-info'],
          subpaths: [
            'probands-personal-info',
            'probands-personal-info/',
            'questionnaireInstances/',
          ],
        },
        {
          name: 'SIDENAV.SAMPLE_MANAGEMENT',
          path: ['sample-management'],
          subpaths: ['sample-management', 'sample-management/'],
        },
        {
          name: 'SIDENAV.CONTACT_PROBAND',
          path: ['contact-proband'],
          subpaths: ['contact-proband'],
        },
        {
          name: 'SIDENAV.PROBANDS_TO_CONTACT',
          path: ['probands-to-contact'],
          subpaths: ['probands-to-contact', 'probands-to-contact/'],
        },
      ]);
    }));

    it('should emit a list of pages for EinwilligungsManager role', fakeAsync(() => {
      // Arrange
      setRole('EinwilligungsManager');
      const pagesCallback = jasmine.createSpy();
      service = TestBed.inject(PageManager);

      // Act
      service.navPages$.subscribe(pagesCallback);
      tick();

      // Assert
      expect(pagesCallback).toHaveBeenCalledTimes(1);
      expect(pagesCallback).toHaveBeenCalledWith([
        {
          name: 'SIDENAV.STUDY',
          path: ['study'],
          subpaths: ['study', 'study/'],
        },
        {
          name: 'SIDENAV.COMPLIANCE',
          path: ['compliance/view'],
          subpaths: ['compliance/'],
        },
      ]);
    }));

    it('should emit an empty list if role is not known', fakeAsync(() => {
      // Arrange
      user.hasRole.and.returnValue(false);
      user.isProband.and.returnValue(false);
      const pagesCallback = jasmine.createSpy();
      service = TestBed.inject(PageManager);

      // Act
      service.navPages$.subscribe(pagesCallback);
      tick();

      // Assert
      expect(pagesCallback).toHaveBeenCalledTimes(1);
      expect(pagesCallback).toHaveBeenCalledWith([]);
    }));
  });

  function setRole(role: Role): void {
    user.hasRole.and.callFake((checkRole) => role === checkRole);
  }
});
