# How To Add a New Configuration

If any service needs to receive a new configuration value, the following steps have to be taken to ensure that the
value is available in the service and that the service is able to use it:

1. Add the new configuration to the `src/config.ts` of the service implementation, e.g.:
   ```ts
   const conf = {
     // ...
     userPasswordLength: parseInt(
       ConfigUtils.getEnvVariable('USER_PASSWORD_LENGTH')
     ),
   };
   ```
2. Add the configuration to the `ConfigNames` under `k8s/src/pia/piaConfig.ts`, e.g.:
   ```ts
   const ConfigNames = {
     // ...
     userPasswordLength: null,
   };
   ```
3. Add the configuration to the `Configuration` chart under `k8s/src/configuration.ts`, e.g.:
   ```ts
   export class Configuration extends Chart {
     // ...
     public readonly variables = {
       // ...
       userPasswordLength: PiaConfig.getConfig(
         this.configSecret,
         'userPasswordLength'
       ),
     };
   }
   ```
4. Pass the configuration to the consuming Service under `k8s/src/pia/deployment`, e.g.:
   ```ts
   export class ExampleService extends NodeJSService {
     // ...
     constructor() {
       // ...
       super(scope, configuration, 'userservice', 'us', {
         USER_PASSWORD_LENGTH: configuration.variables.userPasswordLength,
       });
     }
   }
   ```
5. Add the configuration to the example config under `k8s/deployment/components/example-pia-config` to provide it for local development, e.g.:

   ```yaml
   # k8s/deployment/components/example-pia-config/kustomization.yaml
   apiVersion: kustomize.config.k8s.io/v1alpha1
   kind: Component

   secretGenerator:
     - name: pia-config
       options:
         disableNameSuffixHash: true
       type: kubernetes.io/opaque
       literals:
         # ...
         - userPasswordLength=8
   ```

6. Add the configuration to any overlay `kustomization.yaml` under `k8s/deployment/overlays/` in which you want to
   provide it. It works the same way as in the example config.
7. To also let the configuration be generated by the `k8s/utils/create-overlay.sh` script, also add an entry there and
   in the corresponding template, e.g.:

   ```sh
   # k8s/utils/create-overlay.sh
   read -p "Minimum valid length of user passwords and length of generated one-time passwords (number): " userPasswordLength
   ```

   ```yaml
   # k8s/utils/overlay-template.yaml

   # ...
   secretGenerator:
     - name: pia-config
       options:
       disableNameSuffixHash: true
       type: kubernetes.io/opaque
       literals:
         # ...
         - userPasswordLength={userPasswordLength}
   ```

8. Inform operators of the newly added configuration value and its purpose by adding an entry to `DISTRIBUTION_NOTES.md`, e.g.:

   ```md
   ## 1.25

   - the configuration variable `userPasswordLength` ("USER_PASSWORD_LENGTH") was added to allow configuration of the length of generated one-time passwords
   ```

9. Finally, also add an entry for the configuration to the [configuration documentation](docs/configuration.md)
