/****************************************************************************/
// Eclipse SUMO, Simulation of Urban MObility; see https://eclipse.dev/sumo
// Copyright (C) 2001-2026 German Aerospace Center (DLR) and others.
// This program and the accompanying materials are made available under the
// terms of the Eclipse Public License 2.0 which is available at
// https://www.eclipse.org/legal/epl-2.0/
// This Source Code may also be made available under the following Secondary
// Licenses when the conditions for such availability set forth in the Eclipse
// Public License 2.0 are satisfied: GNU General Public License, version 2
// or later which is available at
// https://www.gnu.org/licenses/old-licenses/gpl-2.0-standalone.html
// SPDX-License-Identifier: EPL-2.0 OR GPL-2.0-or-later
/****************************************************************************/
/// @file    GNEMeanDataFrame.cpp
/// @author  Pablo Alvarez Lopez
/// @date    Nov 2022
///
// The Widget for edit meanData elements
/****************************************************************************/

#include <netedit/changes/GNEChange_MeanData.h>
#include <netedit/elements/data/GNEMeanData.h>
#include <netedit/frames/GNEAttributesEditor.h>
#include <netedit/GNEApplicationWindow.h>
#include <netedit/GNENet.h>
#include <netedit/GNETagPropertiesDatabase.h>
#include <netedit/GNEUndoList.h>
#include <netedit/GNEViewNet.h>
#include <netedit/GNEViewParent.h>
#include <utils/gui/div/GUIDesigns.h>
#include <utils/gui/windows/GUIAppEnum.h>

#include "GNEMeanDataFrame.h"

// ===========================================================================
// FOX callback mapping
// ===========================================================================

FXDEFMAP(GNEMeanDataFrame::MeanDataTypeSelector) meanDataSelectorMap[] = {
    FXMAPFUNC(SEL_COMMAND,  MID_GNE_SET_TYPE,   GNEMeanDataFrame::MeanDataTypeSelector::onCmdSelectItem)
};

FXDEFMAP(GNEMeanDataFrame::MeanDataEditor) meanDataEditorMap[] = {
    FXMAPFUNC(SEL_COMMAND,  MID_GNE_CREATE,    GNEMeanDataFrame::MeanDataEditor::onCmdCreateMeanData),
    FXMAPFUNC(SEL_COMMAND,  MID_GNE_DELETE,    GNEMeanDataFrame::MeanDataEditor::onCmdDeletetMeanData),
    FXMAPFUNC(SEL_COMMAND,  MID_GNE_COPY,      GNEMeanDataFrame::MeanDataEditor::onCmdCopyMeanData)
};

FXDEFMAP(GNEMeanDataFrame::MeanDataSelector) meanDataTypeSelectorMap[] = {
    FXMAPFUNC(SEL_COMMAND,  MID_GNE_SET_TYPE,   GNEMeanDataFrame::MeanDataSelector::onCmdSelectItem)
};

// Object implementation
FXIMPLEMENT(GNEMeanDataFrame::MeanDataTypeSelector, GNEGroupBoxModule,  meanDataSelectorMap,        ARRAYNUMBER(meanDataSelectorMap))
FXIMPLEMENT(GNEMeanDataFrame::MeanDataEditor,       GNEGroupBoxModule,  meanDataEditorMap,          ARRAYNUMBER(meanDataEditorMap))
FXIMPLEMENT(GNEMeanDataFrame::MeanDataSelector,     GNEGroupBoxModule,  meanDataTypeSelectorMap,    ARRAYNUMBER(meanDataTypeSelectorMap))


// ===========================================================================
// method definitions
// ===========================================================================

// ---------------------------------------------------------------------------
// GNEMeanDataFrame::MeanDataTypeSelector - methods
// ---------------------------------------------------------------------------

GNEMeanDataFrame::MeanDataTypeSelector::MeanDataTypeSelector(GNEMeanDataFrame* meanDataFrameParent) :
    GNEGroupBoxModule(meanDataFrameParent, TL("MeanData Type")),
    myMeanDataFrameParent(meanDataFrameParent) {
    // Create MFXComboBoxIcon
    myTypeComboBox = new MFXComboBoxIcon(getCollapsableFrame(), meanDataFrameParent->getViewNet()->getViewParent()->getGNEAppWindows()->getStaticTooltipMenu(),
                                         false, GUIDesignComboBoxVisibleItems, this, MID_GNE_SET_TYPE, GUIDesignComboBox);
    // add mean data types
    const auto tagPropertiesMeanDatas = myMeanDataFrameParent->getViewNet()->getNet()->getTagPropertiesDatabase()->getTagPropertiesByType(GNETagProperties::Type::MEANDATA);
    for (const auto& meanDataType : tagPropertiesMeanDatas) {
        myTypeComboBox->appendIconItem(meanDataType->getTagStr().c_str(), GUIIconSubSys::getIcon(meanDataType->getGUIIcon()));
    }
    // set DEFAULT_VEHTYPE as default VType
    myCurrentMeanData = tagPropertiesMeanDatas.front();
    // MeanDataTypeSelector is always shown
    show();
}


GNEMeanDataFrame::MeanDataTypeSelector::~MeanDataTypeSelector() {}


const GNETagProperties*
GNEMeanDataFrame::MeanDataTypeSelector::getCurrentMeanData() const {
    return myCurrentMeanData;
}


void
GNEMeanDataFrame::MeanDataTypeSelector::refreshMeanDataTypeSelector() {
    // clear items
    myTypeComboBox->clearItems();
    // add mean data types
    const auto tagPropertiesMeanDatas = myMeanDataFrameParent->getViewNet()->getNet()->getTagPropertiesDatabase()->getTagPropertiesByType(GNETagProperties::Type::MEANDATA);
    for (const auto& meanDataType : tagPropertiesMeanDatas) {
        myTypeComboBox->appendIconItem(meanDataType->getTagStr().c_str(), GUIIconSubSys::getIcon(meanDataType->getGUIIcon()));
    }
    // make sure that tag is in myTypeMatchBox
    if (myCurrentMeanData != nullptr) {
        for (int i = 0; i < (int)myTypeComboBox->getNumItems(); i++) {
            if (myTypeComboBox->getItemText(i) == myCurrentMeanData->getTagStr()) {
                myTypeComboBox->setCurrentItem(i);
            }
        }
    } else {
        myCurrentMeanData = tagPropertiesMeanDatas.front();
    }
    // refresh meanData editor module
    myMeanDataFrameParent->myMeanDataEditor->refreshMeanDataEditorModule();
    // show modules
    myMeanDataFrameParent->myMeanDataEditor->showMeanDataEditorModule();
    myMeanDataFrameParent->myMeanDataSelector->showMeanDataSelector();
}


long
GNEMeanDataFrame::MeanDataTypeSelector::onCmdSelectItem(FXObject*, FXSelector, void*) {
    // add mean data types
    const auto tagPropertiesMeanDatas = myMeanDataFrameParent->getViewNet()->getNet()->getTagPropertiesDatabase()->getTagPropertiesByType(GNETagProperties::Type::MEANDATA);
    // Check if value of myTypeMatchBox correspond of an allowed additional tags
    for (const auto& meanDataType : tagPropertiesMeanDatas) {
        if (meanDataType->getTagStr() == myTypeComboBox->getText().text()) {
            // set pointer
            myCurrentMeanData = meanDataType;
            // set color of myTypeMatchBox to black (valid)
            myTypeComboBox->setTextColor(GUIDesignTextColorBlack);
            // refresh meanData editor module
            myMeanDataFrameParent->myMeanDataEditor->refreshMeanDataEditorModule();
            // show modules if selected item is valid
            myMeanDataFrameParent->myMeanDataEditor->showMeanDataEditorModule();
            myMeanDataFrameParent->myMeanDataSelector->showMeanDataSelector();
            return 1;
        }
    }
    myCurrentMeanData = nullptr;
    // refresh meanData editor module
    myMeanDataFrameParent->myMeanDataEditor->refreshMeanDataEditorModule();
    // hide all modules if selected item isn't valid
    myMeanDataFrameParent->myMeanDataEditor->hideMeanDataEditorModule();
    myMeanDataFrameParent->myMeanDataSelector->hideMeanDataSelector();
    // set color of myTypeMatchBox to red (invalid)
    myTypeComboBox->setTextColor(GUIDesignTextColorRed);
    return 1;
}

// ---------------------------------------------------------------------------
// GNEMeanDataFrame::MeanDataEditor - methods
// ---------------------------------------------------------------------------

GNEMeanDataFrame::MeanDataEditor::MeanDataEditor(GNEMeanDataFrame* meanDataFrameParent) :
    GNEGroupBoxModule(meanDataFrameParent, TL("MeanData Editor")),
    myMeanDataFrameParent(meanDataFrameParent) {
    // Create new meanData
    myCreateMeanDataButton = GUIDesigns::buildFXButton(getCollapsableFrame(), TL("Create MeanData"), "", "",
                             GUIIconSubSys::getIcon(GUIIcon::MODEMEANDATA), this, MID_GNE_CREATE, GUIDesignButton);
    // Create delete/reset meanData
    myDeleteMeanDataButton = GUIDesigns::buildFXButton(getCollapsableFrame(), TL("Delete MeanData"), "", "",
                             GUIIconSubSys::getIcon(GUIIcon::MODEDELETE), this, MID_GNE_DELETE, GUIDesignButton);
    // Create copy meanData
    myCopyMeanDataButton = GUIDesigns::buildFXButton(getCollapsableFrame(), TL("Copy MeanData"), "", "",
                           GUIIconSubSys::getIcon(GUIIcon::COPY), this, MID_GNE_COPY, GUIDesignButton);
}


GNEMeanDataFrame::MeanDataEditor::~MeanDataEditor() {}


void
GNEMeanDataFrame::MeanDataEditor::showMeanDataEditorModule() {
    refreshMeanDataEditorModule();
    show();
}


void
GNEMeanDataFrame::MeanDataEditor::hideMeanDataEditorModule() {
    hide();
}


void
GNEMeanDataFrame::MeanDataEditor::refreshMeanDataEditorModule() {
    // first check if selected VMeanData is valid
    if (myMeanDataFrameParent->myMeanDataSelector->getCurrentMeanData() == nullptr) {
        // disable buttons
        myDeleteMeanDataButton->disable();
        myCopyMeanDataButton->disable();
    } else {
        // enable buttons
        myDeleteMeanDataButton->enable();
        myCopyMeanDataButton->enable();
    }
    // update module
    recalc();
}


long
GNEMeanDataFrame::MeanDataEditor::onCmdCreateMeanData(FXObject*, FXSelector, void*) {
    auto net = myMeanDataFrameParent->myViewNet->getNet();
    // get current meanData type
    SumoXMLTag meanDataTag = myMeanDataFrameParent->myMeanDataTypeSelector->getCurrentMeanData()->getTag();
    // obtain a new valid MeanData ID
    const std::string typeID = myMeanDataFrameParent->myViewNet->getNet()->getAttributeCarriers()->generateMeanDataID(meanDataTag);
    // create new meanData
    GNEMeanData* meanData = new GNEMeanData(myMeanDataFrameParent->myMeanDataTypeSelector->getCurrentMeanData()->getTag(), typeID,
                                            net, net->getGNEApplicationWindow()->getFileBucketHandler()->getDefaultBucket(FileBucket::Type::MEANDATA));
    // add it using undoList (to allow undo-redo)
    net->getUndoList()->begin(meanData, "create meanData");
    net->getUndoList()->add(new GNEChange_MeanData(meanData, true), true);
    net->getUndoList()->end();
    // set created meanData in selector
    myMeanDataFrameParent->myMeanDataSelector->setCurrentMeanData(meanData);
    return 1;
}


long
GNEMeanDataFrame::MeanDataEditor::onCmdDeletetMeanData(FXObject*, FXSelector, void*) {
    // begin undo list operation
    myMeanDataFrameParent->myViewNet->getUndoList()->begin(GUIIcon::MODEDELETE, "delete meanData");
    // remove meanData (and all of their children)
    myMeanDataFrameParent->myViewNet->getNet()->deleteMeanData(myMeanDataFrameParent->myMeanDataSelector->getCurrentMeanData(),
            myMeanDataFrameParent->myViewNet->getUndoList());
    // end undo list operation
    myMeanDataFrameParent->myViewNet->getUndoList()->end();
    // set created meanData in selector
    myMeanDataFrameParent->myMeanDataSelector->refreshMeanDataSelector(false);
    return 1;
}


long
GNEMeanDataFrame::MeanDataEditor::onCmdCopyMeanData(FXObject*, FXSelector, void*) {
    // get current meanData type
    SumoXMLTag meanDataTag = myMeanDataFrameParent->myMeanDataTypeSelector->getCurrentMeanData()->getTag();
    // obtain a new valid MeanData ID
    const std::string typeID = myMeanDataFrameParent->myViewNet->getNet()->getAttributeCarriers()->generateMeanDataID(meanDataTag);
    // obtain meanData in which new MeanData will be based
    GNEMeanData* meanData = dynamic_cast<GNEMeanData*>(myMeanDataFrameParent->myMeanDataSelector->getCurrentMeanData());
    // check that meanData exist
    if (meanData) {
        // create a new MeanData based on the current selected meanData
        GNEMeanData* meanDataCopy = new GNEMeanData(meanData->getTagProperty()->getTag(), typeID, myMeanDataFrameParent->myViewNet->getNet(),
                meanData->getFileBucket());
        // begin undo list operation
        myMeanDataFrameParent->myViewNet->getUndoList()->begin(meanDataCopy, "copy meanData");
        // add it using undoList (to allow undo-redo)
        myMeanDataFrameParent->myViewNet->getUndoList()->add(new GNEChange_MeanData(meanDataCopy, true), true);
        // end undo list operation
        myMeanDataFrameParent->myViewNet->getUndoList()->end();
        // refresh MeanData Selector (to show the new VMeanData)
        myMeanDataFrameParent->myMeanDataSelector->refreshMeanDataSelector(false);
        // set created meanData in selector
        myMeanDataFrameParent->myMeanDataSelector->setCurrentMeanData(meanDataCopy);
        // refresh MeanData Editor Module
        myMeanDataFrameParent->myMeanDataEditor->refreshMeanDataEditorModule();
    }
    return 1;
}

// ---------------------------------------------------------------------------
// GNEMeanDataFrame::MeanDataSelector - methods
// ---------------------------------------------------------------------------

GNEMeanDataFrame::MeanDataSelector::MeanDataSelector(GNEMeanDataFrame* typeFrameParent) :
    GNEGroupBoxModule(typeFrameParent, TL("Current MeanData")),
    myMeanDataFrameParent(typeFrameParent),
    myCurrentMeanData(nullptr) {
    // get current meanData type
    SumoXMLTag meanDataTag = myMeanDataFrameParent->myMeanDataTypeSelector->getCurrentMeanData()->getTag();
    // Create MFXComboBoxIcon
    myMeanDataComboBox = new MFXComboBoxIcon(getCollapsableFrame(), typeFrameParent->getViewNet()->getViewParent()->getGNEAppWindows()->getStaticTooltipMenu(),
            false, GUIDesignComboBoxVisibleItems, this, MID_GNE_SET_TYPE, GUIDesignComboBox);
    // add meanDatas
    for (const auto& meanData : myMeanDataFrameParent->getViewNet()->getNet()->getAttributeCarriers()->getMeanDatas().at(meanDataTag)) {
        myMeanDataComboBox->appendIconItem(meanData.second->getID().c_str(), meanData.second->getACIcon());
    }
    if (myMeanDataFrameParent->getViewNet()->getNet()->getAttributeCarriers()->getMeanDatas().at(meanDataTag).size() > 0) {
        myCurrentMeanData = myMeanDataFrameParent->getViewNet()->getNet()->getAttributeCarriers()->getMeanDatas().at(meanDataTag).begin()->second;
    } else {
        myCurrentMeanData = nullptr;
    }
    // MeanDataSelector is always shown
    show();
}


GNEMeanDataFrame::MeanDataSelector::~MeanDataSelector() {}


void
GNEMeanDataFrame::MeanDataSelector::showMeanDataSelector() {
    // refresh mean data selector
    refreshMeanDataSelector(false);
    // show
    show();
}


void
GNEMeanDataFrame::MeanDataSelector::hideMeanDataSelector() {
    // hide attributes editor
    myMeanDataFrameParent->myMeanDataAttributesEditor->hideAttributesEditor();
    // hide
    hide();
}


GNEMeanData*
GNEMeanDataFrame::MeanDataSelector::getCurrentMeanData() const {
    return myCurrentMeanData;
}


void
GNEMeanDataFrame::MeanDataSelector::setCurrentMeanData(GNEMeanData* vMeanData) {
    myCurrentMeanData = vMeanData;
    refreshMeanDataSelector(false);
}


void
GNEMeanDataFrame::MeanDataSelector::refreshMeanDataSelector(bool afterChangingID) {
    // get current meanData type
    SumoXMLTag meanDataTag = myMeanDataFrameParent->myMeanDataTypeSelector->getCurrentMeanData()->getTag();
    // get mean datas sorted by ID
    std::map<std::string, GNEMeanData*> sortedMeanDatas;
    for (const auto& meanData : myMeanDataFrameParent->getViewNet()->getNet()->getAttributeCarriers()->getMeanDatas().at(meanDataTag)) {
        sortedMeanDatas[meanData.second->getID()] = meanData.second;
    }
    // clear items
    myMeanDataComboBox->clearItems();
    // fill myMeanDataMatchBox with meanDatas
    for (const auto& sortedMeanData : sortedMeanDatas) {
        myMeanDataComboBox->appendIconItem(sortedMeanData.first.c_str(), sortedMeanData.second->getACIcon());
    }
    // make sure that mean data exists
    if (myCurrentMeanData && myMeanDataFrameParent->getViewNet()->getNet()->getAttributeCarriers()->retrieveMeanData(myCurrentMeanData->getTagProperty()->getTag(), myCurrentMeanData->getID(), false)) {
        bool validMeanData = false;
        for (int i = 0; i < (int)myMeanDataComboBox->getNumItems(); i++) {
            if (myMeanDataComboBox->getItemText(i) == myCurrentMeanData->getID()) {
                myMeanDataComboBox->setCurrentItem(i);
                validMeanData = true;
            }
        }
        if (!validMeanData) {
            myCurrentMeanData = nullptr;
        }
    } else {
        myCurrentMeanData = nullptr;
    }
    // check if enable or disable comboBox
    if (sortedMeanDatas.size() > 0) {
        myMeanDataComboBox->enable();
        // check ifupdate myCurrentMeanData
        if (myCurrentMeanData == nullptr) {
            myCurrentMeanData = sortedMeanDatas.begin()->second;
        }
    } else {
        myMeanDataComboBox->disable();
    }
    // refresh meanData editor module
    myMeanDataFrameParent->myMeanDataEditor->refreshMeanDataEditorModule();
    // check if show attribute editor
    if (!afterChangingID) {
        if (myCurrentMeanData) {
            myMeanDataFrameParent->myMeanDataAttributesEditor->showAttributesEditor(myCurrentMeanData, true);
        } else {
            myMeanDataFrameParent->myMeanDataAttributesEditor->hideAttributesEditor();
        }
    }
}


void
GNEMeanDataFrame::MeanDataSelector::refreshMeanDataSelectorIDs() {
    if (myCurrentMeanData) {
        myMeanDataComboBox->updateIconItem(myMeanDataComboBox->getCurrentItem(), myCurrentMeanData->getID().c_str(), myCurrentMeanData->getACIcon());
    }
}


long
GNEMeanDataFrame::MeanDataSelector::onCmdSelectItem(FXObject*, FXSelector, void*) {
    // get current meanData type
    SumoXMLTag meanDataTag = myMeanDataFrameParent->myMeanDataTypeSelector->getCurrentMeanData()->getTag();
    // Check if value of myMeanDataMatchBox correspond of an allowed additional tags
    for (const auto& meanData : myMeanDataFrameParent->getViewNet()->getNet()->getAttributeCarriers()->getMeanDatas().at(meanDataTag)) {
        if (meanData.second->getID() == myMeanDataComboBox->getText().text()) {
            // set pointer
            myCurrentMeanData = meanData.second;
            // set color of myMeanDataMatchBox to black (valid)
            myMeanDataComboBox->setTextColor(GUIDesignTextColorBlack);
            // refresh meanData editor module
            myMeanDataFrameParent->myMeanDataEditor->refreshMeanDataEditorModule();
            // show modules if selected item is valid
            myMeanDataFrameParent->myMeanDataAttributesEditor->showAttributesEditor(myCurrentMeanData, true);
            // update viewNet
            myMeanDataFrameParent->getViewNet()->updateViewNet();
            return 1;
        }
    }
    myCurrentMeanData = nullptr;
    // refresh meanData editor module
    myMeanDataFrameParent->myMeanDataEditor->refreshMeanDataEditorModule();
    // hide all modules if selected item isn't valid
    myMeanDataFrameParent->myMeanDataAttributesEditor->hideAttributesEditor();
    // set color of myMeanDataMatchBox to red (invalid)
    myMeanDataComboBox->setTextColor(GUIDesignTextColorRed);
    // update viewNet
    myMeanDataFrameParent->getViewNet()->updateViewNet();
    return 1;
}

// ---------------------------------------------------------------------------
// GNEMeanDataFrame - methods
// ---------------------------------------------------------------------------

GNEMeanDataFrame::GNEMeanDataFrame(GNEViewParent* viewParent, GNEViewNet* viewNet) :
    GNEFrame(viewParent, viewNet, TL("MeanData")) {
    // build meanData type selector
    myMeanDataTypeSelector = new MeanDataTypeSelector(this);
    // build meanData editor
    myMeanDataEditor = new MeanDataEditor(this);
    // build meanData selector
    myMeanDataSelector = new MeanDataSelector(this);
    // build meanData attributes editor
    myMeanDataAttributesEditor = new GNEAttributesEditor(this, GNEAttributesEditorType::EditorType::EDITOR);
}


GNEMeanDataFrame::~GNEMeanDataFrame() {
}


void
GNEMeanDataFrame::show() {
    // refresh meanData type selector
    myMeanDataTypeSelector->refreshMeanDataTypeSelector();
    // show frame
    GNEFrame::show();
}


void
GNEMeanDataFrame::hide() {
    // hide frame
    GNEFrame::hide();
}


void
GNEMeanDataFrame::attributeUpdated(SumoXMLAttr attribute) {
    if (attribute == SUMO_ATTR_ID) {
        myMeanDataSelector->refreshMeanDataSelector(true);
    }
}


void
GNEMeanDataFrame::updateFrameAfterUndoRedo() {
}

/****************************************************************************/
