program rates
    !*****************************************************************************
    !     RATES integrates cross sections to obtain rate coefficients between
    !     100 and 10,000K. A Maxwellian velocity distribution is assumed. The 
    !     'alternative extended Simpson's rule' is used for the integration 
    !     (from Numerical recipes).  
    !
    !     ! This program assumes that energies and cross sections are
    !     tabulated respectively 
    !     EITHER in Rydbergs and Bohrs**2 (Default) 
    !     OR in eV and Angstroms^2.
    !
    !     ! Rates are converted from atomic units to cm**3 sec**-1 by
    !     multiplying by 6.126163*10**(-9).
    !
    !     ! Fortran95 rewrite by Keir Little
    ! 
    !*****************************************************************************
    implicit none
    ! housekeeping data
    character(11) :: moddat = '05-Dec-2016'
    integer :: i
    integer :: istatus
    character(3) :: elabel
    real :: aesimps
    ! input and output units can be overriden in namelist
    integer :: stderr = 0
    integer :: stdin = 5
    integer :: stdout = 6
    integer :: xsin = 9
    integer :: rout = 10
    integer :: rdef = 11
    integer :: ieunit = 1   ! 1 for energy in Rydbergs and cross section in bohrs^2
                            ! 2 for energy in eV and cross sections in Angstroms^2
    character(50) :: infile, outfile, deffile, name
    ! Filenames infile for energy/cross-section input,
    !           outfile for temperature/rate output,
    !           deffile for temperature/rate units.
    ! Name is a calculation title for output.
    namelist/ratesin/infile, outfile, deffile, stderr, stdin, stdout,&
        xsin, rout, rdef, name, ieunit

    ! input and output arrays and lengths
    integer, parameter :: nener = 20000
    integer, parameter :: ntemp = 505 ! hardcoded, like temp_range()
    integer :: n = 0
    real, dimension(nener) :: xsn = 0, xsnt, e
    real, dimension(ntemp) :: tempi

    ! constants
    real, parameter :: ev = 13.6057
    real, parameter :: pi = 2*acos(0.)
    real, parameter :: boltz = 3.16682981E-6    ! In Ht/K
    real, parameter :: factb = 8*pi*(1/(2*pi*boltz))**1.5
    real, parameter :: factun = 6.126163E-9

    ! calculation variables
    real :: temp    ! single temperature value
    real :: factbt  ! temperature-dependant factb
    real :: rate    ! single rate coefficient value
    real :: estep   ! difference in energy/x values
    
    ! Read basic data via namelist /RATESIN/
    read(stdin, ratesin)

    ! Date-stamp run and print title
    write(stdout, 10) moddat, name
    10 format('Program RATES (last modified ',A,')'//A/)

    ! Fill the array of required temperatures
    call temp_range(tempi)
    
    ! Read in energy and cross-section data
    open(unit=xsin, file=infile, status='old', action='read', iostat=istatus)
    if(istatus == 0) then
        readloop: do i = 1, nener
            read(xsin, *, iostat=istatus) e(i), xsn(i)
            if(istatus /= 0) exit ! either error or EOF
            n = n + 1
        end do readloop
    end if
    if(istatus > 0) then
        write(stderr,20) istatus
        20 format('Error opening input file: iostat = ', I0)
        stop
    end if
    close(xsin)

    ! Write out energy range depending on units, or quit if ieunit is invalid
    if (ieunit == 1) then
        elabel = 'Ryd'
    else if (ieunit == 2) then
        elabel = 'eV'
    else
        write(stderr,30) ieunit
        30 format('ERROR: ieunit = ',I0,'. It must be 1 or 2.')
        stop
    end if
    write(stdout,301) n, elabel, e(1), elabel, e(n)
    301 format('Number of energy values',I6,/,'Lowest energy in ',A,' ',&
            ES12.5,/,'Highest energy in ',A,' ',ES12.5)

    if (ieunit == 1) then   ! Convert energy from Rydbergs to Hartrees
        e = e/2.0
    else                    ! Convert energy from eV to Ht, cross sections
        e = e/(2.0*ev)      ! from Angstrom^2 to Bohr^2
        xsn = xsn/0.2800285608592
    end if
    ! Caculate difference between e values, which is delta x for the integral
    estep = e(2) - e(1)

    write(stdout,401) rout, outfile
    401 format(/,'Rate coefficients beween 100 and 10,000K are written on unit '&
        ,I0,' file ',A)

    ! write definition file
    open(unit=rdef, file=deffile, action='write', iostat=istatus)
    if (istatus == 0) then
        write(rdef,402) name
        write(rdef,403)
        402 FORMAT('#',A,/,'# **** ***********',/,'#')
        403 FORMAT('#   TEMP (K)',4X,'RATE COEFF. (cm^3/s)')
    else
        write(stderr,40) istatus
        40 format('Error opening definition file: iostat = ', I0)
        stop
    end if

    ! loop over temperatures and calculate ratecoefficient for each one
    open(unit=rout, file=outfile, action='write', iostat=istatus)
    if (istatus == 0) then
        do i = 1, ntemp
            temp = tempi(i)
            factbt = factb/temp**1.5
            xsnt = xsn*e*exp(-e/(boltz*temp))
            
            rate = aesimps(xsnt, n, estep)
            rate = rate * factbt * factun
            write(rout,501) temp, rate
            501 format(F10.2,ES18.5)
        end do
    else
        write(stderr,50) istatus
        50 format('Error opening output file: iostat = ', I0)
        stop
    end if
    close(rout)
    close(rdef)

    write(stdout,60)
    60 format(/,'*** Task has been successfully completed ***')
end program rates

real pure function aesimps(f, n, deltax) result(integral)
    ! "Alternative Extended Simpson's Rule": Numerical Recipes, p108
    implicit none
    real, dimension(*), intent(in) :: f ! array of function values to be integrated
    integer, intent(in) :: n            ! number of values
    real, intent(in) :: deltax
    integer :: i                        ! iterator

    integral=0.0
    integral=integral+(17./48.)*f(1)
    integral=integral+(59./48.)*f(2)
    integral=integral+(43./48.)*f(3)
    integral=integral+(49./48.)*f(4)
    do i = 5, n-4
        integral = integral + f(i)
    end do
    integral=integral+(49./48.)*f(n-3)
    integral=integral+(43./48.)*f(n-2)
    integral=integral+(59./48.)*f(n-1)
    integral=integral+(17./48.)*f(n)

    integral=integral*deltax
end function

subroutine temp_range(tempi)
    implicit none
    real, dimension(*), intent(out) :: tempi
    ! Hacky hardcoding of temperature steps copied from rates.f
    ! This could easily be modified to accept values from RATESIN
    real, parameter :: init1 = 100
    real, parameter :: step1 = 10
    real, parameter :: init2 = 6000
    real, parameter :: step2 = 1000
    integer, parameter :: n1 = 500
    integer, parameter :: n2 = 5
    integer :: i

    do i = 1, n1
        tempi(i) = init1+step1*(i-1)
    end do

    do i = 1, n2
        tempi(i+n1) = init2+step2*(i-1)
    end do
end subroutine
