! Copyright 2021
!
! For a comprehensive list of the developers that contributed to these codes
! see the UK-AMOR website.
!
! This file is part of UKRmol-out (UKRmol+ suite).
!
!     UKRmol-out is free software: you can redistribute it and/or modify
!     it under the terms of the GNU General Public License as published by
!     the Free Software Foundation, either version 3 of the License, or
!     (at your option) any later version.
!
!     UKRmol-out is distributed in the hope that it will be useful,
!     but WITHOUT ANY WARRANTY; without even the implied warranty of
!     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!     GNU General Public License for more details.
!
!     You should have received a copy of the GNU General Public License
!     along with  UKRmol-out (in source/COPYING). Alternatively, you can also visit
!     <https://www.gnu.org/licenses/>.
!
!> \brief  Photoionization delays
!> \author J Benda
!> \date   2021 - 2024
!>
!> Calculation of discrete photoionization delays from the partial wave dipoles
!> produced by RSOLVE (COMPAK). Two files are written: Standard one-photon delays
!> approximating the Wigner delay, and the molecular delays according to the
!> asymptotic theory of Baykusheva and Woerner (2017). The output files contain
!> photon energies above the first ionization threshold and the associated
!> delays in attoseconds. All numbers are for unoriented molecules.
!>
!> The input data are read from the namelist "pdel" on standard input. The parameters
!> are:
!>   - nsym: Number of partial dipole files to read
!>   - lupwd: List of unit numbers with partial dipoles
!>   - formpwd: FORMATTED (default) or UNFORMATTED access to the partial wave dipoles
!>   - iprint: Logging level (default: 1)
!>   - iwrite: Output unit (default: standard output)
!>   - states: List of (degenerate) states to include
!>   - couple: Consider dipole coupling to the listed states, too
!>   - first_IP: Used only for shifting the photon energies written to output files
!>   - omega: Energy of the IR photon in Hartree
!>   - smooth: Smoothing factor (default: 0.1)
!>   - cc: Model for continuum-continuum delays
!>       - 0: Dahlström et al. (2013) short-range
!>       - 1: Dahlström et al. (2013) long-range (default)
!>       - 2: Exact hydrogen atom cc delay in the s-p-s pathway (slow)
!>       - 3: Asymptotic hydrogen cc delay in the s-p-s pathway
!>
!> \warning This program requires that the photoelectron energies are sampled uniformly.
!>
program photodelays

#ifdef HAVE_NO_FINDLOC
    use algorithms,      only: findloc
#endif
    use iso_fortran_env, only: output_unit
    use precisn_gbl,     only: wp

    implicit none

    real(wp),    parameter :: tau = 24.188843266_wp     ! atomic unit of time (in as)
    real(wp),    parameter :: to_eV = 27.21138469_wp    ! atomic unit of energy (in eV)
    integer,     parameter :: niter = 10                ! number of smoothing iterations

    type :: partial_wave_dipoles
        character(len=80) :: title
        integer :: mtot, stot, gutot, lmaxp
        integer, allocatable :: starg(:), mtarg(:), gtarg(:), ichl(:), lchl(:), mchl(:), components(:)
        real(wp), allocatable :: echl(:), escat(:), ebound(:), re_d(:, :, :, :), im_d(:, :, :, :)
        real(wp) :: etarg
    contains
    end type partial_wave_dipoles


    call print_ukrmol_header(output_unit)

    print '(/,A)', 'Program PHOTODELAYS: Calculation of approximate RABITT delays'

    call photodelays_main

contains

    !> \brief  Main program
    !> \author J Benda
    !> \date   2021 - 2024
    !>
    !> Main driver of the program. Reads the partial wave dipoles from the RSOLVE-written
    !> files, conjugates them, multiplies in the Coulomb and partial wave phase factor
    !> and dispatches execution further.
    !>
    subroutine photodelays_main

        use multidip_io,     only: read_target_properties
        use multidip_params, only: imu
        use photo_outerio,   only: read_pw_dipoles

        real(wp), external :: cphaz

        character(len=11) :: formpwd, ccmodels(0:3)
        integer :: nset, iprint, iwrite, ifail, isym, l, ie, ibound, icomp, nsym, istate, ichan, nescat
        integer :: lutarg, lupwd(8), states(10), couple(10), cc
        real(wp) :: EkRy, re_d, im_d, sigma, eta, omega, first_IP, smooth
        complex(wp) :: d, factor

        type(partial_wave_dipoles), allocatable :: pwd(:)
        real(wp), allocatable :: prop(:,:,:), etarg(:)

        namelist /pdel/ nsym, formpwd, iprint, iwrite, lutarg, lupwd, states, couple, omega, first_IP, smooth, cc

        ccmodels(0) = 'DahlstromSR'
        ccmodels(1) = 'DahlstromLR'
        ccmodels(2) = 'HydrogenSPS'
        ccmodels(3) = 'CoulombFsHp'

        nsym = 0
        formpwd = 'FORMATTED'
        iprint = 1
        iwrite = output_unit
        lutarg = 24
        lupwd = 0
        states = 0
        couple = 0
        omega = 0.057  ! ~ 800 nm
        first_IP = 0
        smooth = 0.1  ! eV^-1
        cc = 1

        read (*, nml = pdel)

        cc = max(lbound(ccmodels, 1), min(cc, ubound(ccmodels, 1)))

        print '(/,A,/)', 'Input parameters'
        print '(1x,A,1x,*(1x,I0))', 'Pw dipole units:', lupwd(1:nsym)
        print '(1x,A,9x,F0.3,A)', 'First IP:', first_IP*to_eV, ' eV'
        print '(1x,A,8x,F0.3,A)', 'IR energy:', omega*to_eV, ' eV'
        print '(1x,A,1x,*(1x,I0))', 'Residual states:', states(1:count(states /= 0))
        print '(1x,A,2x,*(1x,I0))', 'Coupled states:', couple(1:count(couple /= 0))
        print '(1x,A,1x,F0.3,A)', 'Smoothing factor:', smooth, ' eV^-1'
        print '(1x,a,9x,i0,3a)', 'CC model:', cc, ' ("', trim(ccmodels(cc)), '")'

        ifail = 0

        allocate (pwd(nsym))

        print '(/,A)', 'Reading partial wave dipoles'

        do isym = 1, nsym

            print *

            ! read the partial wave dipoles file
            call read_pw_dipoles(lupwd(isym), nset, formpwd, &
                                 pwd(isym) % title, pwd(isym) % mtot, pwd(isym) % stot, pwd(isym) % gutot, &
                                 pwd(isym) % starg, pwd(isym) % mtarg, pwd(isym) % gtarg, &
                                 pwd(isym) % ichl, pwd(isym) % lchl, pwd(isym) % mchl, pwd(isym) % echl, pwd(isym) % escat, &
                                 pwd(isym) % lmaxp, pwd(isym) % components, pwd(isym) % ebound, pwd(isym) % etarg, &
                                 pwd(isym) % re_d, pwd(isym) % im_d, &
                                 iprint, iwrite, ifail)

            if (ifail /= 0) stop 1
            if (first_IP == 0) then
                first_IP = pwd(isym) % etarg - pwd(isym) % ebound(1)
                print '(1x,A,F0.3,A)', 'Using calculated first IP: ', first_IP*to_eV, ' eV'
            end if

            ! conjugate the matrix elements and multiply in the Coulomb phase
            nescat = size(pwd(isym) % escat)
            do l = 0, maxval(pwd(isym) % lchl)
                do ie = 1, nescat
                    do istate = 1, maxval(pwd(isym) % ichl)
                        ichan = findloc(pwd(isym) % ichl, istate, 1)
                        EkRy = 2*pwd(isym) % escat(ie) - pwd(isym) % echl(ichan)
                        if (EkRy < 0) then
                            factor = 0
                        else
                            eta = -1/sqrt(EkRy)
                            sigma = cphaz(l, eta, iwrite)
                            factor = imu**(-l) * cmplx(cos(sigma), sin(sigma), wp)
                        end if
                        do ibound = 1, size(pwd(isym) % ebound)
                            do icomp = 1, 3
                                if (pwd(isym) % components(icomp) == 1) then
                                    do ichan = 1, size(pwd(isym) % ichl)
                                        if (pwd(isym) % lchl(ichan) == l .and. pwd(isym) % ichl(ichan) == istate) then
                                            re_d = pwd(isym) % re_d(ibound, ichan, icomp, ie)
                                            im_d = pwd(isym) % im_d(ibound, ichan, icomp, ie)
                                            d = factor * cmplx(re_d, -im_d, wp)
                                            pwd(isym) % re_d(ibound, ichan, icomp, ie) = real(d, wp)
                                            pwd(isym) % im_d(ibound, ichan, icomp, ie) = aimag(d)
                                        end if
                                    end do
                                end if
                            end do
                        end do
                    end do
                end do
            end do

        end do

        if (count(couple > 0) /= 0) then
            call read_target_properties(lutarg, prop, etarg)
        end if

        call one_photon_delays(pwd, first_IP, omega, smooth, states)
        call molecular_delays(pwd, first_IP, omega, smooth, states, prop, couple, cc)
        call continuum_continuum_delays(pwd, first_IP, omega, states, cc)

        print '(/,A)', 'Done.'

    end subroutine photodelays_main


    !> \brief  Calculate first-order delays
    !> \author J Benda
    !> \date   2021 - 2024
    !>
    !> Evaluate the one-photon delay as a discrete derivative of the photoionization dipoles.
    !>
    subroutine one_photon_delays (pwd, first_IP, omega, smooth, states)

        type(partial_wave_dipoles), allocatable, intent(in) :: pwd(:)
        real(wp), intent(in) :: first_IP, omega, smooth
        integer, intent(in) :: states(:)
        complex(wp) :: qdip, d_p, d_m
        real(wp) :: re_d_p, im_d_p, re_d_m, im_d_m, dE
        integer :: offset, nescat, ie, isym, icomp, ichan, istate
        real(wp), allocatable :: dips(:,:)

        print '(/,A)', 'Calculating one-photon delays'

        dE = pwd(1) % escat(2) - pwd(1) % escat(1)
        offset = ceiling(2*omega / dE)
        nescat = size(pwd(1) % escat)
        allocate (dips(4, nescat))
        dips = 0

        print '(/,1x,A,I0)', 'Offset: ', offset

        do ie = 1 + offset/2, nescat - offset/2
            qdip = 0
            do isym = 1, size(pwd)
                do icomp = 1, size(pwd(isym) % components)
                    if (pwd(isym) % components(icomp) /= 0) then
                        do ichan = 1, size(pwd(isym) % ichl)
                            istate = pwd(isym) % ichl(ichan)
                            if (findloc(states, istate, 1) > 0) then
                                re_d_p = pwd(isym) % re_d(1, ichan, icomp, ie - offset/2)
                                im_d_p = pwd(isym) % im_d(1, ichan, icomp, ie - offset/2)
                                re_d_m = pwd(isym) % re_d(1, ichan, icomp, ie + offset/2)
                                im_d_m = pwd(isym) % im_d(1, ichan, icomp, ie + offset/2)
                                d_p = cmplx(re_d_p, im_d_p, wp)
                                d_m = cmplx(re_d_m, im_d_m, wp)
                                qdip = qdip + conjg(d_p)*d_m / 3
                            end if
                        end do
                    end if
                end do
            end do
            dips(1, ie) = real(qdip, wp)
            dips(2, ie) = aimag(qdip)
        end do

        call smooth_dipoles(dips, real(smooth*dE*to_eV, wp))
        call write_delays(dips, 'delays-1p.txt', pwd(1) % escat, omega, first_IP)

    end subroutine one_photon_delays


    !> \brief  Calculate molecular delays
    !> \author J Benda
    !> \date   2021 - 2024
    !>
    !> Evaluate orientation-averaged molecular delays according to the asymptotic theory
    !> of Woerner and Baykusheva (2017).
    !>
    subroutine molecular_delays (pwd, first_IP, omega, smooth, states, prop, couple, cc)

        use coupling_obj_gbl, only: couplings_type
        use multidip_params,  only: pi

        type(partial_wave_dipoles), allocatable, intent(in) :: pwd(:)
        real(wp),                   allocatable, intent(in) :: prop(:, :, :)
        real(wp),                                intent(in) :: first_IP, omega, smooth
        integer,                                 intent(in) :: states(:), couple(:), cc

        type(couplings_type)     :: ang
        complex(wp), allocatable :: Mpws(:,:,:,:,:), Mion(:,:,:,:,:,:), AkkMm(:,:,:,:), AkkMp(:,:,:,:)
        complex(wp)              :: d
        real(wp),    allocatable :: dips(:,:), dips1(:,:), dips2(:,:)
        real(wp)                 :: g, re_d, im_d, dE
        integer                  :: offset, nescat, l1, l2, maxl, maxpw, m1, m2, ipw, isym, jsym, a, b
        integer                  :: ie, ichan, jchan, istate, jstate, istat, jstat, nstat, ncouple
        character(len=200)       :: filename

        print '(/,A)', 'Calculating molecular delays'

        dE = pwd(1) % escat(2) - pwd(1) % escat(1)
        offset = ceiling(2*omega / dE)
        nstat = count(states /= 0)
        ncouple = count(couple /= 0)
        nescat = size(pwd(1) % escat)
        allocate (dips(4, nescat), dips1(4, nescat), dips2(4, nescat))

        print '(/,1x,A,I0)', 'Offset: ', offset

        ! get highest angular momentum
        maxl = 0
        do isym = 1, size(pwd)
            maxl = max(maxl, maxval(pwd(isym) % lchl))
        end do
        maxpw = (maxl + 1)**2
        allocate (Mpws(3, 3, maxpw, nescat, nstat), Mion(3, 3, maxpw, nescat, nstat, ncouple))
        Mpws = 0
        Mion = 0

        ! precompute complex Gaunt coefficients
        call ang % prec_cgaunt(maxl + 1)

        ! recouple partial waves by the IR photon
        do istat = 1, nstat
            do isym = 1, size(pwd)
                do a = 1, 3  ! y, z, x
                    if (pwd(isym) % components(a) == 0) cycle
                    do ichan = 1, size(pwd(isym) % ichl)
                        istate = pwd(isym) % ichl(ichan)
                        if (istate /= states(istat)) cycle  ! skip states not requested by user
                        l1 = pwd(isym) % lchl(ichan)
                        m1 = pwd(isym) % mchl(ichan)
                        do l2 = abs(l1 - 1), min(l1 + 1, maxl)
                            do m2 = max(-l2, m1 - 1), min(l2, m1 + 1)
                                ipw = l2*l2 + l2 + m2 + 1
                                do b = 1, 3  ! y, z, x
                                    g = ang % rgaunt(l1, l2, 1, m1, m2, b-2) * sqrt(4*pi/3)
                                    if (abs(g) < 1e-15_wp) cycle
                                    do ie = 1, nescat
                                        re_d = pwd(isym) % re_d(1, ichan, a, ie)
                                        im_d = pwd(isym) % im_d(1, ichan, a, ie)
                                        d = cmplx(re_d, im_d, wp)
                                        Mpws(a, b, ipw, ie, istat) = Mpws(a, b, ipw, ie, istat) + g*d
                                    end do
                                end do
                            end do
                        end do
                    end do
                end do
            end do
        end do

        ! calculate orientation-averaged, emission integrated quadratic matrix elements (partial wave coupling term)
        dips = 0
        do istat = 1, nstat
            call calculate_averaged_quadratic_dipoles(Mpws(:, :, :, :, istat), Mpws(:, :, :, :, istat), dips, offset)
        end do
        call smooth_dipoles(dips, real(smooth*dE*to_eV, wp))
        call write_delays(dips, 'delays-mol.txt', pwd(1) % escat, omega, first_IP)

        ! recouple residual ions by the IR photon
        do jstat = 1, ncouple
            do jsym = 1, size(pwd)
                do a = 1, 3  ! y, z, x
                    if (pwd(jsym) % components(a) == 0) cycle
                    do jchan = 1, size(pwd(jsym) % ichl)
                        jstate = pwd(jsym) % ichl(jchan)
                        if (jstate /= couple(jstat)) cycle  ! skip couplings not requested by user
                        l1 = pwd(jsym) % lchl(jchan)
                        m1 = pwd(jsym) % mchl(jchan)
                        ipw = l1*l1 + l1 + m1 + 1
                        do istat = 1, nstat
                            istate = states(istat)
                            do b = 1, 3  ! y, z, x
                                g = prop(istate, jstate, b)
                                if (abs(g) < 1e-15_wp) cycle
                                do ie = 1, nescat
                                    re_d = pwd(jsym) % re_d(1, jchan, a, ie)
                                    im_d = pwd(jsym) % im_d(1, jchan, a, ie)
                                    d = cmplx(re_d, im_d, wp)
                                    Mion(a, b, ipw, ie, jstat, istat) = Mion(a, b, ipw, ie, jstat, istat) + g*d
                                end do
                            end do
                        end do
                    end do
                end do
            end do
        end do

        print *

        ! add Akk prefactor (assume that it is the same for all degenerate components)
        dips = 0
        do istat = 1, nstat
            call add_akk_factors(pwd, Mpws(:, :, :, :, istat), AkkMp, states(istat), states(istat), +omega, 1, cc)
            call add_akk_factors(pwd, Mpws(:, :, :, :, istat), AkkMm, states(istat), states(istat), -omega, 1, cc)
            call calculate_averaged_quadratic_dipoles(AkkMp, AkkMm, dips, offset)
        end do

        ! calculate orientation-averaged, emission integrated quadratic matrix elements (ion coupling terms)
        do jstat = 1, ncouple

            print *, 'Pws+, Ion-'
            dips1 = 0
            do istat = 1, nstat
                call add_akk_factors(pwd, Mpws(:, :, :, :, istat),        AkkMp, states(istat), states(istat), +omega, 1, cc)
                call add_akk_factors(pwd, Mion(:, :, :, :, jstat, istat), AkkMm, couple(jstat), states(istat), -omega, 0, cc)
                call calculate_averaged_quadratic_dipoles(AkkMp, AkkMm, dips1, offset)
            end do

            print *, 'Ion+, Pws-'
            dips2 = 0
            do istat = 1, nstat
                call add_akk_factors(pwd, Mion(:, :, :, :, jstat, istat), AkkMp, couple(jstat), states(istat), +omega, 0, cc)
                call add_akk_factors(pwd, Mpws(:, :, :, :, istat),        AkkMm, states(istat), states(istat), -omega, 1, cc)
                call calculate_averaged_quadratic_dipoles(AkkMp, AkkMm, dips2, offset)
            end do

            ! only the imaginary part is new, the real stays from earlier
            dips(2, :) = dips1(2, :) + dips2(2, :)
            call smooth_dipoles(dips, real(smooth*dE*to_eV, wp))
            write (filename, '(A,I0,A)') 'delays-mol-coupl-', couple(jstat), '.txt'
            call write_delays(dips, trim(filename), pwd(1) % escat, omega, first_IP)

        end do

    end subroutine molecular_delays


    !> \brief   Calculate orientation averaged and emission integrated quadratic matrix elements
    !> \author  J Benda
    !> \date    2021 - 2024
    !>
    !> For each energy calculate the product (M*)M, integrated over emission directions and averaged over
    !> polarization directions. The first factor corresponds to the XUV+IR pathway, the second to XUV-IR.
    !> Sum over all residual ion states. Save real part to the first component of Q,
    !> imaginary part to the second.
    !>
    subroutine calculate_averaged_quadratic_dipoles (Mp, Mm, Q, offset)

        complex(wp),              intent(in)    :: Mp(:, :, :, :), Mm(:, :, :, :)
        real(wp),    allocatable, intent(inout) :: Q(:, :)
        integer,                  intent(in)    :: offset

        complex(wp) :: Mp1, Mm1, qdip
        real(wp)    :: Aabcd
        integer     :: nescat, npws, a, b, c, d, ie, ipw

        npws = size(Mp, 3)
        nescat = size(Mp, 4)

        do a = 1, 3  ! y, z, x
            do b = 1, 3  ! y, z, x
                do c = 1, 3  ! y, z, x
                    do d = 1, 3  ! y, z, x
                        Aabcd = 1/15._wp * (merge(1, 0, a == b .and. c == d) + &
                                            merge(1, 0, a == c .and. b == d) + &
                                            merge(1, 0, a == d .and. b == c))
                        if (Aabcd == 0) cycle
                        do ipw = 1, npws
                            do ie = 1 + offset/2, nescat - offset/2
                                Mp1 = Mp(a, b, ipw, ie - offset/2)
                                Mm1 = Mm(c, d, ipw, ie + offset/2)
                                qdip = Aabcd * conjg(Mp1) * Mm1
                                Q(1, ie) = Q(1, ie) + real(qdip, wp)
                                Q(2, ie) = Q(2, ie) + aimag(qdip)
                            end do
                        end do
                    end do
                end do
            end do
        end do

    end subroutine calculate_averaged_quadratic_dipoles


    !> \brief   Multiply Akk factors into the matrix elements
    !> \author  J Benda
    !> \date    2021 - 2024
    !>
    !> Add the target-independent (but final-projectile-energy-dependent) asymptotic factors
    !> to the two-photon matrix elements M.
    !>
    subroutine add_akk_factors (pwd, M, MM, nstate, fstate, omega, mode, cc)

        use multidip_asy, only: Akk

        type(partial_wave_dipoles), allocatable, intent(in)    :: pwd(:)
        complex(wp),                             intent(inout) :: M(:, :, :, :)
        complex(wp),                allocatable, intent(inout) :: MM(:, :, :, :)
        real(wp),                                intent(in)    :: omega
        integer,                                 intent(in)    :: fstate, nstate, mode, cc

        real(wp)    :: echlf, echln, EkfRy, kf, EknRy, kn
        integer     :: ie, ichanf, ichann

        if (.not. allocated(MM)) then
            allocate (MM, mold = M)
        end if

        ! get the first channel that is coupled to the requested states
        ichanf = findloc(pwd(1) % ichl, fstate, 1)
        ichann = findloc(pwd(1) % ichl, nstate, 1)

        ! get channel thresholds (in Ry) relative to the lowest channel threshold
        echlf = pwd(1) % echl(ichanf)
        echln = pwd(1) % echl(ichann)

        !$omp parallel do private(ie, EkfRy, EknRy, kf, kn) schedule(dynamic, 1)
        do ie = 1, size(M, 4)
            EkfRy = 2*pwd(1) % escat(ie) - echlf + 2*omega
            EknRy = 2*pwd(1) % escat(ie) - echln
            if (EkfRy > 0 .and. EknRy > 0) then
                kf = sqrt(EkfRy)
                kn = sqrt(EknRy)
                MM(:, :, :, ie) = Akk(kn, kf, mode, cc) * M(:, :, :, ie)
            else
                MM(:, :, :, ie) = 0
            end if
        end do

    end subroutine add_akk_factors


    !> \brief  Calculate CC delays
    !> \author J Benda
    !> \date   2021 - 2024
    !>
    !> Evaluate the requested continuum-continuum delay. For cc = 0 or cc = 1 the delay is obtained from the asymptotic
    !> radial integrals calculated by functions Akk. For cc = 2 the continuum-continuum delay is obtained directly as
    !> the difference between RABITT and 1-photon delays for ionization of H(1s) through the s-p-s angular momentum
    !> pathway.
    !>
    subroutine continuum_continuum_delays (pwd, first_IP, omega, states, cc)

        use multidip_asy, only: Akk

        type(partial_wave_dipoles), intent(in) :: pwd(:)
        real(wp),                   intent(in) :: first_IP, omega
        integer,                    intent(in) :: states(:), cc

        complex(wp)           :: Ap, Am, qdip
        real(wp), allocatable :: dips(:, :)
        real(wp)              :: Ekf, Ekp, Ekm, kf, kp, km, echl
        integer               :: ie, nescat, istat, istate, nstat, ichan
        character(len=200)    :: filename

        print '(/,A)', 'Calculating continuum-continuum delays'

        nstat = count(states /= 0)
        nescat = size(pwd(1) % escat)
        allocate (dips(4, nescat))
        dips = 0

        do istat = 1, nstat

            istate = states(istat)
            ichan = findloc(pwd(1) % ichl, istate, 1)
            echl = pwd(1) % echl(ichan)

            !$omp parallel do default(none) &
            !$omp& private(Ekf, Ekp, Ekm, kf, kp, km, Ap, Am, qdip) &
            !$omp& shared(pwd, omega, cc, dips, nescat, echl)
            do ie = 1, nescat

                Ekf = pwd(1) % escat(ie) - echl/2
                Ekp = Ekf - omega   ! Ekf = (XUV + IR) - IP
                Ekm = Ekf + omega   ! Ekf = (XUV - IR) - IP

                if (Ekp <= 0) cycle

                kf = sqrt(2*Ekf)
                kp = sqrt(2*Ekp)
                km = sqrt(2*Ekm)

                Ap = Akk(kp, kf, 1, cc)
                Am = Akk(km, kf, 1, cc)

                qdip = conjg(Ap) * Am

                dips(1, ie) = real(qdip, wp)
                dips(2, ie) = aimag(qdip)

            end do

            write (filename, '(A,I0,A)') 'delays-cc-', istate, '.txt'

            call write_delays(dips, trim(filename), pwd(1) % escat, omega, first_IP)

        end do

    end subroutine continuum_continuum_delays


    !> \brief  Apply a smoothing factor to the dipoles
    !> \author J Benda
    !> \date   2021
    !>
    !> Smooth using a distance-weighted gaussian convolution.
    !>
    subroutine smooth_dipoles (dips, smooth)

        real(wp), allocatable, intent(inout) :: dips(:, :)
        real(wp),              intent(in)    :: smooth

        real(wp), allocatable :: weights(:), dips0(:, :), dips1(:, :)
        real(wp)              :: weight, weight_sum, re_d, im_d
        integer               :: i, j, iter, n, d

        n = size(dips, 2)

        allocate (weights(n), dips0(2, n), dips1(2, n))

        ! set up the Gaussian weights
        do i = 1, n
            weights(i) = exp(-smooth*(i - 1)**2)
            dips0(:, i) = 0
        end do

        ! do the smoothing iterations
        do iter = 1, niter
            ! calculate entries of the smoothed dataset
            !$omp parallel do default(none) private(i, j, weight_sum, d, re_d, im_d, weight) shared(dips, dips1, dips0, weights, n)
            do i = 1, n
                weight_sum = 0
                dips1(:, i) = 0
                ! use all elements from the previous iteration
                do j = 1, n
                    d = 1 + abs(i - j)
                    re_d = dips(1, j) - dips0(1, j)
                    im_d = dips(2, j) - dips0(2, j)
                    weight = weights(d) / sqrt(1 + re_d*re_d + im_d*im_d)
                    dips1(1:2, i) = dips1(1:2, i) + dips(1:2, j) * weight
                    weight_sum = weight_sum + weight
                end do
                dips1(:, i) = dips1(:, i) / weight_sum
            end do
            !$omp end parallel do
            dips0 = dips1
        end do
        dips(3:4, 1:n) = dips0(1:2, 1:n)

    end subroutine smooth_dipoles


    !> \brief  Evaluate delays and write them to file
    !> \author J Benda
    !> \date   2021
    !>
    subroutine write_delays (dips, filename, escat, omega, first_IP)

        real(wp), allocatable, intent(in) :: dips(:,:), escat(:)
        real(wp),              intent(in) :: omega, first_IP
        character(len=*),      intent(in) :: filename

        integer  :: u, ie
        real(wp) :: Ek

        open (newunit = u, file = filename, action = 'write')
        write (u, '(A1,A24,6A25)') '#', 'Photon energy (eV)', 'Time delay (as)', 'Smoothed delay (as)', &
                                'Real d*.d', 'Imag d*.d', 'Smoothed Real d*.d', 'Smoothed Imag d*.d'

        do ie = 1, size(dips, 2)
            Ek = escat(ie)
            write (u, '(7E25.15)') (first_IP + Ek)*to_eV, &
                                    atan2(dips(2, ie), dips(1, ie)) / (2*omega) * tau, &
                                    atan2(dips(4, ie), dips(3, ie)) / (2*omega) * tau, &
                                    dips(1, ie), dips(2, ie), &
                                    dips(3, ie), dips(4, ie)
        end do

        close (u)

    end subroutine write_delays

end program photodelays
