!> \brief   Command-line interface to the OUTER library
!> \authors Jakub Benda
!> \date    2018
!>
!> This is a single program that calls selected subroutines from the outer library,
!> depending on the command line options. For example, to run "sw_interf", "r_solve"
!> and "bound_f" (in this order), use the following command:
!> \verbatim
!>     outer-run  swintf rsolve boundf
!> \endverbatim
!>
!> When no arguments are given, the program prints all possible keywords that it knows.
!>
program outer_run

    use const_gbl, only: stdout
    use mpi_gbl, only: mpi_mod_start
    use ukrmol_interface_gbl, only: finalize_mpi, destroy_ukrmolp

    implicit none

    ! interface that all sub-programs callable from this launcher must conform to
    abstract interface
        subroutine interfc (status)
            integer, intent(inout)  :: status
        end subroutine
    end interface

    ! length of the command line keyword (subroutine alias)
    integer, parameter :: name_len = 16

    ! association of function pointer to a specific command line keyword
    type :: routine
        character(len=name_len) :: keyword
        procedure(interfc), pointer, nopass :: ptr
    end type

    ! declaration of all sub-programs
    procedure(interfc) :: sw_interf
    procedure(interfc) :: r_solve
    procedure(interfc) :: eigen_p
    procedure(interfc) :: t_matrx
    procedure(interfc) :: i_xsecs
    procedure(interfc) :: reson_f
    procedure(interfc) :: bound_f
    procedure(interfc) :: timedel
    procedure(interfc) :: r_solve_compak
    procedure(interfc) :: mpi_r_solve

    logical, parameter :: master_writes_to_stdout = .false.
    logical, parameter :: use_shared_memory = .true.

    type(routine)     :: routines(10)
    integer           :: ifail = 0, narg, iarg, irtn
    character(len=name_len) :: argv

    ! set up a look-up table of procedure pointers
    routines(1 ) % keyword = 'swintf'     ; routines(1 ) % ptr => sw_interf
    routines(2 ) % keyword = 'old_rsolve' ; routines(2 ) % ptr => r_solve
    routines(3 ) % keyword = 'eigenp'     ; routines(3 ) % ptr => eigen_p
    routines(4 ) % keyword = 'tmatrx'     ; routines(4 ) % ptr => t_matrx
    routines(5 ) % keyword = 'ixsecs'     ; routines(5 ) % ptr => i_xsecs
    routines(6 ) % keyword = 'reson'      ; routines(6 ) % ptr => reson_f
    routines(7 ) % keyword = 'bound'      ; routines(7 ) % ptr => bound_f
    routines(8 ) % keyword = 'timedel'    ; routines(8 ) % ptr => timedel
    routines(9 ) % keyword = 'rsolve'     ; routines(9 ) % ptr => r_solve_compak
    routines(10) % keyword = 'mpi_rsolve' ; routines(10) % ptr => mpi_r_solve

    ! get number of sub-programs to execute
    narg = command_argument_count()

    ! if no argument is provided, print list of possible keywords
    if (narg == 0) then
        write(*, '(/,"Available OUTER sub-programs:",/)')
        do irtn = 1, size(routines)
            write(*, '(5x,A)') routines(irtn) % keyword
        end do
        write(*, *)
        stop
    end if

    ! print version information
    call print_ukrmol_header(stdout)

    ! execute all selected sub-programs
    call mpi_mod_start(master_writes_to_stdout, use_shared_memory)
    do iarg = 1, narg
        call get_command_argument(iarg, argv)
        do irtn = 1, size(routines)
            if (routines(irtn) % keyword == argv) exit
        end do
        if (irtn <= size(routines)) then
            call routines(irtn) % ptr(ifail)
        else
            write(*, '("Skipping unknown keyword """,A,""".")') argv
        end if
    end do
    call destroy_ukrmolp
    call finalize_mpi

end program outer_run
