! Copyright 2019
!
! For a comprehensive list of the developers that contributed to these codes
! see the UK-AMOR website.
!
! This file is part of UKRmol-out (UKRmol+ suite).
!
!     UKRmol-out is free software: you can redistribute it and/or modify
!     it under the terms of the GNU General Public License as published by
!     the Free Software Foundation, either version 3 of the License, or
!     (at your option) any later version.
!
!     UKRmol-out is distributed in the hope that it will be useful,
!     but WITHOUT ANY WARRANTY; without even the implied warranty of
!     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!     GNU General Public License for more details.
!
!     You should have received a copy of the GNU General Public License
!     along with  UKRmol-out (in source/COPYING). Alternatively, you can also visit
!     <https://www.gnu.org/licenses/>.
!
      program BORNCROS
!
!***********************************************************************
! this program calculates the Born correction for the elastic and
! excitation cross section.
!
!  BELXS = input namelist
!  SWBORN - switch
!         = 1 - calculate Born correction (default)
!         = 0 - do not calculate
!  SW - switch for the elastic/excitation Born correction
!     = 0 calculate elastic Born correction (default)
!     = 1 calculate excitation Born correction
!  LH = number of partial waves
!  DTM = dipole/transition moment in au
!       SW=0 - dominant component of the ground state dipole moment in au
!       SW=1 - transition moments in au
!  BROT = rotational constants (cm**-1)/excitation energy (eV)
!       SW=0 - dominant component of the rotational constants in cm**-1
!       SW=1 - excitation energy in e
!  ESTEP = scattering energies increment, default value is 0.025 eV
!  NESCAT = number of scattering energies, default value is 400
!  BORNEL - logical unit for output of Born correction
!         = 1 (default)
!  SWCS - switch to add the correction to the total cross section
!        = 1 - add Born correction to the total cross section (default)
!        = 0 - do not add
!  LUXSN - logical unit for total cross section without Born correction
!        = 57 default
!  LUXSNO - logical unit for total cross section with Born correction
!          = 77 default
!  SWX - column in LUXSN, holding cross section requiring correction
!      = column number 4 - elastic cross section - default
!  NC  = number of columns in  LUXSN
!  IWRITE - logical unit for printed output
!  NAME = title for any output
!
!  SWSUM - switch
!        = 1 - sum cross sections of different total symmetries,
!        = 0 - do not sum (default)
!  NSYM - number of cross sections of different total symmetries
!  NFTX - logical units holding cross sections of different total
!         symmetries (fort.30-30+NSYM)
!  ALL CROSS SECTIONS ARE in Bohr**2 or ANGSTROM**2, ENERGIES ARE in EV
!
! If the dominant component of the dipole moment is z-component, then
! we use zz-component of the rotational constant (max value)
!
!  CORRECT = Born correction (in Bohr**2) or ANSTROMS**2 depending on
!            ixsn
!
!  IXSN =  Units of the cross section (same variable name as ixsec)
!       1 = BOHR**2, 2=ANGSTOM**2
!  NTARG = 1 Number of target states.
!  QMOLN = FALSE If set to true, makes named files for each target
!          state.
!
!***********************************************************************
!
      implicit double precision (a-h,o-z)
      INTEGER SW,LH,NESCAT,BORNEL,LUXSN,LUXSNO,SWX,SWSUM,NSYM
      INTEGER SWBORN
      CHARACTER(50) NAME
      LOGICAL QMOLN
      NAMELIST/BELXS/ SW,LH,DTM,BROT,ESTART,ESTEP,NESCAT,BORNEL,LUXSN,
     *  LUXSNO,SWX,NC,SWCS,IWRITE,NAME,SWSUM,NSYM,SWBORN,IXSN,
     *  NTARG,QMOLN
      COMMON /BLOK9/GAMMA(51)
      COMMON/BLOK16/GAMAF(51),GAMAHF(51)
      DATA BORNEL/1/,IWRITE/6/,LUXSN/57/,LUXSNO/77/,SWBORN/1/
      DATA SWSUM/0/,SWCS/1/,SWX/4/,NESCAT/1499/,ESTEP/0.025D0/,
     &     ESTART/0.01d0/
      DATA NC/9/,SW/0/,IXSN/1/
!
! print version information
      CALL PRINT_UKRMOL_HEADER(6)
!
      NTARG=1
      QMOLN=.FALSE.
! read input data via namelist /BELXS/
      READ(5,BELXS)
      if (SWBORN.EQ.1) then
        if (SW.EQ.0) then
           WRITE(IWRITE,971) NAME,SW
        elseif (SW.EQ.1) then
           WRITE(IWRITE,972) NAME,SW
        endif
        if (IXSN .EQ. 1) then
           WRITE(IWRITE,973) LH,DTM,BROT,ESTEP,BORNEL,LUXSN,LUXSNO
        elseif (IXSN .EQ. 2) then
           WRITE(IWRITE,976) LH,DTM,BROT,ESTEP,BORNEL,LUXSN,LUXSNO
        endif
      elseif (SWBORN.EQ.0) then
       write(iwrite,975) SWBORN
      endif
!
!
! if swborn=1, calculate Born correction
      if (SWBORN.EQ.1) then
         call BORNCOR(SW, BROT, DTM, ESTART, ESTEP, NESCAT,
     &                LH, IXSN, BORNEL)
      end if
!
! call subroutine to sum cross sections of different total symmetries
      IF (SWSUM.EQ.1) THEN
         write(IWRITE,974) SWSUM,NSYM
         call SUMCROS(NC,NESCAT,LUXSN,NSYM,NTARG,QMOLN)
      END IF
!
! call subroutine to add Born correction to the total cross section
      IF (SWCS.EQ.1) THEN
         call BCROSEC(BORNEL,LUXSN,LUXSNO,SWX,NESCAT,NC,QMOLN,NTARG)
      END IF
!
! print
 975  FORMAT("You have chosen not to calculate the Born
     & correction. SWBORN = ", I2/)
 971  FORMAT("***",A50//
     1 "BORN correction for elastic cross section. SW =", I3//)
 972  FORMAT("***",A50//
     1 "BORN correction for excitation cross section. SW =", I3//)
 973  FORMAT("***********     Input data    ***************",//
     1 "Number of partial waves = ",I3/
     1 "Dipole moment/Transition moment = ",F14.12," a.u.",/
     1 "Rotational constant/Excitation energy = ",F11.9," cm**-1",/
     1 "Energy increment = ",F5.3/
     1 "The Born correction is output on unit ",I2," in Bohr**2",/
     1 "Cross sections are input on unit ",I3," in Bohr**2"/
     1 "Cross sections are output on unit ",I3," in Bohr**2"////)
 976  FORMAT("*********** Input data  ***********",//
     1 "Number of partial waves = ", I3/
     1 "Dipole moment/Transition moment = ", F14.12," a.u.",/
     1 "Rotational constant/Excitation energy = ", F11.9," cm**-1",/
     1 "Energy increment = ", F5.3/
     1 "The Born correction is output on unit ", I2, " in Angstrom**2",/
     1 "Cross sections are input on unit ", I3, " in Angstrom**2"/
     1 "Cross sections are output on unit ", I3, " in Angstrom**2"////)
 974  FORMAT("***  ",I2," Cross sections of different total
     & symmetries will be summed. SWSUM =", I2/)
!
      END
!
!++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
!
      SUBROUTINE BORNCOR(SW, BROT, DTM, ESTART, ESTEP, NESCAT,
     &                   LH,IXSN,BORNEL)
      implicit double precision(a-h, o-z)
      integer :: SW, BORNEL
      DATA PI/3.1415926540D0/

      toangsq= 0.529177d0**2
! decide, what correction to calculate
         IF (SW.EQ.0) THEN
            DELTA=(BROT*13.605d0)/109737.0d0
         ELSEIF (SW.EQ.1) THEN
            DELTA=BROT
         END IF
!
! calculate some parameters
         FUDGE=(8.D0*PI/3.0D0)
         DPS=DTM*DTM
!         EN=-0.015D0
         EN = ESTART - ESTEP
! calculate factorials
         CALL FACT
!
! calculate a correction and print it into a file
         DO NE=1,NESCAT
            EN = EN + ESTEP
            IF (EN.GT.DELTA) THEN
               XKI=SQRT(EN/13.605D0)
               XKF=SQRT((EN-DELTA)/13.605D0)
               XKS=XKI*XKI
               ZX=XKF/XKI
               Z=ZX*ZX
               XSUM=0.0D0
               DO LJ=1,LH
                  L=LJ-1
                  LMIN=L-1
                  LMAX=L+1
                  IF(L.EQ.0) LMAX=1
                  IF(L.EQ.0) LINC=1
                  IF(L.GT.0) LINC=2
                  DO LP=LMIN,LMAX,LINC
                     RESULT=XINT(L,LP,XKI,XKF,Z)
                     CG=WIG3J0(L,LP,1)
                     FACTOR=(2.0D0*L+1)*(2.0D0*LP+1)*CG*CG*RESULT*RESULT
                     XSUM=XSUM+FACTOR
                     BORL=((2.0*FUDGE*DPS*ZX)*XSUM)
                  END DO
               END DO
               BORN=((FUDGE*DPS/XKS)*DLOG((XKI+XKF)/(XKI-XKF)))
               CORRECT=BORN-BORL
            ELSE
               CORRECT=0.0D0
            ENDIF
            if (IXSN .EQ. 1) then
               write(BORNEL,600) EN,CORRECT
! convert the born correction to units of Angstroms^2 from Bohr^2
            elseif (IXSN .EQ. 2) then
               write(BORNEL,600) EN,CORRECT*toangsq
            endif
         END DO
      RETURN
 600  FORMAT (2x,D12.5,D16.8)
      END SUBROUTINE BORNCOR
!
!++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
!
      SUBROUTINE BCROSEC(BORNEL, LUXSN, LUXSNO, SWX, NESCAT, NC,
     &                   QMOLN, NTARG)
! add the Born correction to the total cross section
!
      implicit double precision (A-H,O-Z)
      INTEGER BORNEL,LUXSN,LUXSNO,SWX,NESCAT,NC, FIRST_BLOCK
      real, allocatable, dimension(:,:) :: CS
      integer, allocatable, dimension(:) :: EN
      real, allocatable, dimension(:) :: E,CORR
      real, allocatable, dimension(:) :: BORN
      logical :: QMOLN
! 
      FIRST_BLOCK=6
      MAX_STATES_PER_BLOCK=7
      if (NTARG.LT.FIRST_BLOCK) then
         NBLOCKS = 1
      elseif (MOD(NTARG-FIRST_BLOCK,MAX_STATES_PER_BLOCK).eq.0) then
         NBLOCKS = (NTARG - FIRST_BLOCK) / MAX_STATES_PER_BLOCK +1
      else
         NBLOCKS = (NTARG - FIRST_BLOCK)/ MAX_STATES_PER_BLOCK + 2
      end if
      write(6,*)"Number of scattering in Born correction:",NESCAT
      allocate(CS(NBLOCKS*(MAX_STATES_PER_BLOCK+1),NESCAT),CORR(NESCAT),
     *EN(NESCAT),E(NESCAT),BORN(NESCAT))
! open files holding cross sections
      rewind BORNEL
      OPEN(UNIT=LUXSN)
      rewind LUXSN
      OPEN(UNIT=LUXSNO,STATUS='UNKNOWN')
!
! read a file holding Born correction
      DO I=1,NESCAT
         READ (BORNEL,900) E(I),CORR(I)
      END DO
      close(BORNEL)
!
      ncol = NC-2
! read a file holding summed cross section
      DO L=1, NBLOCKS
         if (nblocks.gt.1.and.L.lt.nblocks) then
             ncol = 7
         elseif (nblocks.eq.1) then
             ncol = nc-2
         else if (mod(ntarg-first_block,max_states_per_block)
     &            .eq.0) then
             ncol = 7
         else
             ncol = mod(ntarg-first_block,max_states_per_block)
         end if
         nstart = L*(MAX_STATES_PER_BLOCK) - 6
         write(6,*) nstart, nstart+ncol-1
         DO I=1,NESCAT
            READ (LUXSN,950) EN(I),E(I),(CS(J,I),J=nstart,nstart+ncol-1)
            write(6,*) EN(I), E(I)
         END DO
         READ(LUXSN, *) ! Skip the empty line
      END DO
      close(LUXSN)
!
! column in array CS
      ncol = NC-2
      NX = SWX-2
! recalculate cross sections
      DO I=1,NESCAT
         if (qmoln) then
            BORN(I) = CS(NX, I) + CORR(I)
            WRITE(LUXSNO,951) EN(I), E(I), CS(NX,I), BORN(I)
         else
            CS(NX,I) = CS(NX,I) +  CORR(I)
            WRITE (LUXSNO,950) EN(I),E(I),(CS(K,I),K=1,ncol)
         end if
      END DO
!
      deallocate(CS,CORR)
      close(LUXSNO)
      return
!
 900  FORMAT (2x,D12.5,D16.8)
 950  FORMAT (1x,I4,2x,D12.5,7D16.8)
 951  FORMAT (1x,I4,2x,E12.5,2E16.8)
      END
!
!******************************************************
!
      SUBROUTINE SUMCROS(NC,NESCAT,LUXSN,NSYM,NTARG,QMOLN)
! sum cross sections of different total symmetries to produce
! a total cross section
! NSYM - number of cross sections of different total symmetries
! (number of logical units NFTX)
! EN are numbers (first column in NFTX),
! E are scattering energies (second column in NFTX).
!
      implicit double precision (A-H,O-Z)
      INTEGER NC,NESCAT,NSYM,LUXSN,ncol,MAX_STATES_PER_BLOCK,NBLOCKS,
     &        FIRST_BLOCK, IOS, O
      real, allocatable, dimension(:,:,:,:) :: CSP
      real, allocatable, dimension(:,:) :: E
      integer, allocatable, dimension(:) :: NFTX,EN,NSCATPOINTS,KINDEX
      logical :: QMOLN
      character(len=3) :: cinit
!
      FIRST_BLOCK=6
      MAX_STATES_PER_BLOCK=7
! Work out how many blocks of data there is relating to one target state
      if (NTARG.LT.FIRST_BLOCK) then
         NBLOCKS = 1
      elseif (MOD(NTARG-FIRST_BLOCK,MAX_STATES_PER_BLOCK).eq.0) then
         NBLOCKS = (NTARG-FIRST_BLOCK) / MAX_STATES_PER_BLOCK + 1
      else
         NBLOCKS = (NTARG-FIRST_BLOCK) / MAX_STATES_PER_BLOCK + 2
      end if

      allocate(CSP(NTARG,NBLOCKS*(MAX_STATES_PER_BLOCK+1),NESCAT,NSYM),
     *EN(NESCAT),E(NESCAT,NSYM),NFTX(NSYM), KINDEX(NSYM))
!
! define logical unit numbers
      NFTX(1) = 30
      DO I=2,NSYM
       NFTX(I)= NFTX(I-1) + 1
      END DO
      IOS = 0
! Read the actual number of scattering points from file. (Can be
! variable with multiplicity of final state.
      IF (QMOLN) then
         OPEN(unit=448, iostat=IOS, status="old",
     &       form="formatted", action="read") 
         IF (IOS .NE. 0) then
             WRITE(6,*) "There was a difficulty opening fort.448 for 
     & reading. Please make sure it exists."
         ELSE
             allocate(NSCATPOINTS(NSYM))
             DO I = 1, NSYM
                READ(448, '(i4)') NSCATPOINTS(I)
             END DO
         END IF
         CLOSE(448)
      END IF
!
! open and read cross section files
      DO K=1,NSYM ! Each symmetry is in a different unit file
         OPEN(NFTX(K))
         DO M=1,NTARG ! Loop over states
            DO L=1,NBLOCKS ! Loop over the number of blocks the data is spread over
! Skip the first 5 lines as they are a header
               DO I=1,5
                  READ(NFTX(K),*)
               ENDDO
! Get the number of columns of data are in the block
               if (nblocks.gt.1.and.L.lt.nblocks) then ! if there is more than one block and we're not doing the last block
                   ncol = 7
               elseif (nblocks.eq.1) then ! if there is only one block
                   ncol = nc-2
               else if (mod(ntarg-first_block,
     &                  max_states_per_block).eq.0) then ! If we are doing the last block and it's full
                   ncol = 7
               else
                   ncol = mod(ntarg-first_block, max_states_per_block) ! If we are doing the last block and it's not full
               end if
               nstart = L*(MAX_STATES_PER_BLOCK+1) - 7
               if (qmoln) then
                  DO I=1,NSCATPOINTS(K)
                     READ (NFTX(K),950) EN(I),E(I,K),(CSP(M,J,I,K),
     &                     J=nstart, nstart+ncol-1)
                  ENDDO
               else
                  DO I=1,NESCAT
                      READ (NFTX(K),950) EN(I),E(I,K),(CSP(M,J,I,K),
     &                      J=nstart ,nstart+ncol-1)
                  END DO ! end over energies
               end if
            ENDDO ! over blocks
        ENDDO  ! end over initial states
         close(NFTX(K))
      END DO
      if (qmoln) NESCAT = MINVAL(NSCATPOINTS)
!
! open output file
      OPEN (UNIT=LUXSN,STATUS='UNKNOWN')
!
! add up cross sections of different total symmetries
! BC - changed naming of local variable sum as there is an intrinsic
! function named sum and it's therefore good practise not to use it as
! a variable name.
      DO N=1,NTARG
         write(cinit,'(I3)') N
         cinit=adjustl(cinit)
         if (qmoln) then
            if (ntarg.gt.1) then
               open(unit=LUXSN+N,file='xsec_exc_'//trim(cinit)//'.dat',
     &              status='unknown')
            end if
            open(unit=LUXSN+NTARG+N,
     &           file='xsec_elastic_'//trim(cinit)//'.dat',
     &           status='unknown')
         end if
         DO L=1,NBLOCKS
            if (nblocks.gt.1.and.L.lt.nblocks) then
               ncol = 7
            elseif (nblocks.eq.1) then
               ncol = nc-2
            else if (mod(ntarg-first_block,
     &               max_states_per_block).eq.0) then
               ncol = 7
            else
               ncol = mod(ntarg-first_block,max_states_per_block)
            end if
            nstart = L*(MAX_STATES_PER_BLOCK+1) - 7
            DO K=1,NSYM
               KINDEX(K) = 0
            END DO
            DO I=1,NESCAT
               DO K=1,NSYM
                  KINDEX(K) = KINDEX(K) + 1
               END DO
               DO J=nstart,nstart+ncol-1
                  XSUM=0d0
                  DO K=1,NSYM
                     IF (E(KINDEX(K), K) .gt. E(KINDEX(1), 1)) then
                        DO O=1, K-1
                           KINDEX(O) = KINDEX(O) + 1
                        END DO
                     ELSE IF (E(KINDEX(K), K) .lt. E(KINDEX(1),1)) then
                        KINDEX(K) = KINDEX(K) + 1
                     END IF
                  END DO
                  DO K=1,NSYM
                     XSUM = XSUM + CSP(N,J,KINDEX(K),K)
                  END DO
                  CSP(N,J,KINDEX(1),1) = XSUM
               END DO
               WRITE(LUXSN,950) EN(I),E(KINDEX(1),1),
     &               (CSP(N,M,KINDEX(1),1),M=nstart,nstart+ncol-1)
               if (qmoln.and.ntarg.gt.1) then
                  write(LUXSN+N,950) EN(I),E(KINDEX(1), 1),
     &               (CSP(N,M,KINDEX(1),1),M=nstart,nstart+ncol-1)
               end if
            END DO
            WRITE(LUXSN,*)' '
            if (qmoln.and.ntarg.gt.1) then
               write(LUXSN+N,*) ' '
            end if
         END DO
         if (qmoln) then
            DO I=1,NESCAT
               write(LUXSN+NTARG+N,951) EN(I),E(I,1),CSP(N,N+1,I,1)
            END DO
            if (ntarg.gt.1) then
                close(LUXSN+N)
            end if
            close(LUXSN+NTARG+N)
         end if
      WRITE(LUXSN,*)' '
      WRITE(LUXSN,*)' '
      END DO
!
      deallocate(CSP,EN,E,NFTX)
      close(LUXSN)
! 
 950  FORMAT (1x,I4,2x,D12.5,7D16.8)
 951  FORMAT (1x,I4,2x,D12.5,D16.8)
      return
      END
!
C*******************************************
C
      FUNCTION XINT(L,LP,XKI,XKF,Z)
      implicit double precision (a-h,o-z)
      COMMON /BLOK9/GAMMA(51)
      COMMON/BLOK16/GAMAF(51),GAMAHF(51)
      DATA PI/3.1415926540D0/
      LDIF=LP-L
      LARG=(L+LP+1)/2
      XMU=L+0.5D0
      XNU=LP+0.5D0
      A=(XMU+XNU)/2.0D0
      B=(XNU-XMU)/2.0D0
      C=XNU+1.0D0
      XV=HYPGEO(A,B,C,Z)
      FAC1=SQRT(XKI*XKF)
      FAC2=(XKF/XKI)**XNU
      FAC3=(0.25D0*PI/FAC1)*FAC2
      FAC4=GAMAF(LARG)
      FAC5=GAMAHF(LP+2)
      IF(LDIF.EQ.1) FAC6=GAMAHF(1)
      IF(LDIF.EQ.-1) FAC6=GAMAHF(2)
      if ((FAC5*FAC6).NE.0)  THEN  ! avoid a divide by zero exception
          XINT=(FAC3*FAC4*XV)/(FAC5*FAC6)
      else
          XINT = 0  ! return a zero to represent infinity
      end if
      RETURN
      END
C
C++++++++++++++++++++++++++++++++++++++++


      FUNCTION HYPGEO(A,B,C,Z)
      implicit double precision (a-h,o-z)
      COMMON /BLOK9/GAMMA(51)
      COMMON/BLOK16/GAMAF(51),GAMAHF(51)
      DIMENSION X(51)
      DO J=1,30
         JP=J-1
         X(1)=1.0D0
         X(J+1)=((A+JP)*(B+JP)/(C+JP))*X(J)
      END DO
      XSUM=0.0D0
      DO JK=2,30
         JKK=JK-1
         XSUM=XSUM+X(JK)*(Z**JKK)/GAMAF(JKK+1)
      END DO
      HYPGEO=1.0D0+XSUM
      RETURN
      END
C ******************************************************
C
C
      FUNCTION WIG3J0 (J1,J2,J3)
      implicit double precision (a-h,o-z)
C
C        EVALUATION OF WIGNER 3-J SYMBOLS OF THE FORM
C                            ( J1 J2 J3 )
C                            (  0  0  0 )
C
C ******************************************************
C
      COMMON /BLOK9/GAMMA(51)
C
C        GAMMA CONTAINS LOGARITHMS OF FACTORIALS. SAME DIMENSION AS
C        IN THE MAIN.
C
      IF (J1.EQ.0) GO TO 1
      IF (J3.EQ.0) GO TO 2
      I1=(J1+J2+J3)/2
      FATT=(-1)**I1
      I1=I1+1
      I2=J1+J2-J3+1
      I3=J1-J2+J3+1
      I4=-J1+J2+J3+1
      I5=J1+J2+J3+2
      XNUM=(GAMMA(I2)+GAMMA(I3)+GAMMA(I4)-GAMMA(I5))*0.5E0
      I2=I1-J3
      I3=I1-J2
      I4=I1-J1
      XNAM=GAMMA(I1)-GAMMA(I2)-GAMMA(I3)-GAMMA(I4)
      WIG3J0=FATT*EXP(XNUM+XNAM)
      GO TO 3
    1 A=J3+J3+1
      WIG3J0=(-1)**J3/SQRT(A)
      GO TO 3
    2 A=J1+J1+1
      WIG3J0=(-1)**J1/SQRT(A)
    3 CONTINUE
      RETURN
      END
C
C*****************************************
C
       SUBROUTINE FACT
       implicit double precision (a-h,o-z)
C
C          THIS SUBROUTINE CALCULATES THE FOLLOWING FACTORIALS
C 
C              GAMA(N)         LOG OF GAMA(N)
C              GAMAF(N)        GAMA(N)
C              GAMAHF(N)       GAMA(N+0.5)
C 
       DIMENSION XID(51)
       COMMON/BLOK9/GAMA(51)
       COMMON/BLOK16/GAMAF(51),GAMAHF(51)
       GAMA(1)=0.0D0
       GAMA(2)=0.0D0
       DO I=3,26
          IX=I-1
          XI=IX
          GAMA(I)=GAMA(IX)+DLOG(XI)
       END DO
       GAMAF(1)=1.0D0
       GAMAF(2)=1.0D0
       DO I=2,30
          GAMAF(I+1)=GAMAF(I)*I
       END DO
       GAMAHF(1)=1.7724538510D0
       GAMAHF(2)=0.8862269255D0
       XID(1)=1.0
       DO I=2,26
         XI=I
         XID(I)=(2.0*XI-1.0)/2.0
         GAMAHF(I+1)=XID(I)*GAMAHF(I)
       END DO
       RETURN
       END
