! Copyright 2020
!
! For a comprehensive list of the developers that contributed to these codes
! see the UK-AMOR website.
!
! This file is part of UKRmol-out (UKRmol+ suite).
!
!     UKRmol-out is free software: you can redistribute it and/or modify
!     it under the terms of the GNU General Public License as published by
!     the Free Software Foundation, either version 3 of the License, or
!     (at your option) any later version.
!
!     UKRmol-out is distributed in the hope that it will be useful,
!     but WITHOUT ANY WARRANTY; without even the implied warranty of
!     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!     GNU General Public License for more details.
!
!     You should have received a copy of the GNU General Public License
!     along with  UKRmol-out (in source/COPYING). Alternatively, you can also visit
!     <https://www.gnu.org/licenses/>.
!
!> \brief   MULTIDIP unit tests
!> \author  J Benda
!> \date    2020
!>
!> This module contains a few sanity unit self-tests that can be executed using "multidip --test".
!>
module multidip_tests

    use precisn_gbl, only: wp

    implicit none

contains

    !> \brief   Numerical unit tests
    !> \author  J Benda
    !> \date    2020
    !>
    !> Run the low-level computational routines and compare their outputs to reference results.
    !>
    subroutine run_tests

        print '(/,A)', 'Running all tests'

        call test_sph
        call test_permutations
        call test_coul
        call test_levin
        call test_1p_eint
        call test_1p_cint
        call test_1p_gint
        call test_2p_cint
        call test_2p_gint
        call test_ang_dist

    end subroutine run_tests


    !> \brief   Test spherical harmonics
    !> \author  J Benda
    !> \date    2025
    !>
    !> Verifies DIPELM functions for evaluations of real and complex spherical harmonics.
    !>
    subroutine test_sph

        use precisn_gbl,              only: cfp
        use special_functions_gbl,    only: cfp_resh
        use dipelm_special_functions, only: a_legendre_p, a_sp_harm, a_re_sp_harm
        use multidip_params,          only: imu, pi

        integer,  parameter :: lmax = 3
        real(wp), parameter :: thetas(5) = [0*pi, pi/4, pi/2, 3*pi/4, pi]
        real(wp), parameter :: phis(5)   = [0*pi, pi/4, pi/2, 3*pi/4, pi]

        real(wp),    allocatable :: Plm(:)
        complex(wp), allocatable :: Ylm(:)

        integer     :: itheta, iphi, l, m, lm
        real(wp)    :: Plm_ref((lmax + 1)**2), Xlm, x, fac(0:2*lmax)
        real(cfp)   :: ex, ey, ez, SH(-lmax:lmax, 0:lmax)
        complex(wp) :: Ylm_ref

        fac(0) = 1
        do l = 1, ubound(fac, 1)
            fac(l) = l*fac(l - 1)
        end do

        print '(/,a,/)', 'Testing normalized associated Legendre polynomials...'
        print '(2x,a,2x,a,2x,a,4x,a,22x,a)', 'theta', 'l', 'm', 'Plm', 'reference'

        do itheta = 1, size(thetas)
            call a_legendre_p(lmax, thetas(itheta), Plm)
            x = cos(thetas(itheta))
            ! Plm from Wikipedia, Condon-Shortley factor included
            Plm_ref = [1._wp, sqrt(1-x*x)/2, x, -sqrt(1-x*x), (1-x*x)/8, x*sqrt(1-x*x)/2, (3*x*x-1)/2, -3*x*sqrt(1-x*x), &
                3*(1-x*x), (1-x*x)**1.5/48, x*(1-x*x)/8, -(1-5*x*x)*sqrt(1-x*x)/8, (5*x*x-3)*x/2, 3*(1-5*x*x)*sqrt(1-x*x)/2, &
                15*x*(1-x*x), -15*(1-x*x)**1.5]
            do l = 0, lmax
                do m = -l, l
                    lm = l*l + l + m + 1
                    Plm_ref(lm) = Plm_ref(lm) * sqrt(fac(l - m)/fac(l + m))
                    print '(f7.3,2i3,2e25.15,1x,a)', thetas(itheta), l, m, Plm(lm), Plm_ref(lm), &
                        merge('    ok', 'WRONG!', abs(Plm(lm) - Plm_ref(lm)) < 1e-10)
                end do
            end do
            deallocate (Plm)
        end do

        print '(/,a,/)', 'Testing complex spherical harmonics...'
        print '(2x,a,2x,a,4x,a,2x,a,4x,a,47x,a)', 'theta', 'phi', 'l', 'm', 'Ylm', 'reference'

        do itheta = 1, size(thetas)
            x = cos(thetas(itheta))
            ! Plm from Wikipedia, Condon-Shortley factor included
            Plm_ref = [1._wp, sqrt(1-x*x)/2, x, -sqrt(1-x*x), (1-x*x)/8, x*sqrt(1-x*x)/2, (3*x*x-1)/2, -3*x*sqrt(1-x*x), &
                3*(1-x*x), (1-x*x)**1.5/48, x*(1-x*x)/8, -(1-5*x*x)*sqrt(1-x*x)/8, (5*x*x-3)*x/2, 3*(1-5*x*x)*sqrt(1-x*x)/2, &
                15*x*(1-x*x), -15*(1-x*x)**1.5]
            do iphi = 1, size(phis)
                call a_sp_harm(lmax, thetas(itheta), phis(iphi), Ylm)
                do l = 0, lmax
                    do m = -l, l
                        lm = l*l + l + m + 1
                        Ylm_ref = sqrt((2*l+1)/(4*pi)*fac(l - m)/fac(l + m)) * Plm_ref(lm) * exp(imu*m*phis(iphi))
                        print '(2f7.3,2i3,4e25.15,1x,a)', thetas(itheta), phis(iphi), l, m, Ylm(lm), Ylm_ref, &
                            merge('    ok', 'WRONG!', abs(Ylm(lm) - Ylm_ref) <= 1e-10)
                    end do
                end do
                deallocate (Ylm)
            end do
        end do

        print '(/,a,/)', 'Testing real spherical harmonics...'
        print '(2x,a,2x,a,4x,a,2x,a,4x,a,22x,a)', 'theta', 'phi', 'l', 'm', 'Xlm', 'reference'

        do itheta = 1, size(thetas)
            do iphi = 1, size(phis)
                ex = sin(thetas(itheta))*cos(phis(iphi))
                ey = sin(thetas(itheta))*sin(phis(iphi))
                ez = cos(thetas(itheta))
                call a_re_sp_harm(lmax, thetas(itheta), phis(iphi), Ylm)
                call cfp_resh(SH, ex, ey, ez, lmax)
                do l = 0, lmax
                    do m = -l, l
                        lm = l*l + l + m + 1
                        Xlm = real(Ylm(lm))
                        print '(2f7.3,2i3,2e25.15,1x,a)', thetas(itheta), phis(iphi), l, m, Xlm, SH(m, l), &
                            merge('    ok', 'WRONG!', abs(Xlm - SH(m, l)) <= 1e-10)
                    end do
                end do
                deallocate (Ylm)
            end do
        end do

    end subroutine test_sph


    !> \brief   Permutations unit tests
    !> \author  J Benda
    !> \date    2020
    !>
    !> Calculate a few permutations using the custom implementation of next_permutation to test
    !> for correctness.
    !>
    subroutine test_permutations

        use multidip_special, only: next_permutation

        integer :: i, order(4)

        order = -1

        print '(/,A,/)', 'Testing permutations of 4 elements...'

        i = 0
        do while (next_permutation(order))
            i = i + 1
            print '(A,I2,A,*(1x,I0))', '  ', i, ':', order
        end do

    end subroutine test_permutations


    !> \brief   Negative-energy Coulomb function tests
    !> \author  J Benda
    !> \date    2020
    !>
    !> Compare negative-energy Coulomb (Whittaker) functions calculated by UKRmol-out and by GSL with
    !> the expected asymptotic behaviour (DLMF §13.19.3).
    !>
    subroutine test_coul

        use multidip_params,  only: nTermsAsy
        use multidip_special, only: coul_gsl, coul_ukrmol

        real(wp) :: r = 100, Z = 1, F, Fp, G1, G1p, G2, G2p, G3, G3p, k
        real(wp) :: Ek(4) = [ -1.000, -0.100, -0.010, -0.001 ]
        integer  :: ls(3) = [ 0, 1, 2 ], ipw, ie, n, l

        print '(/,A)', 'Testing Whittaker functions'
        print '(/,2x,A,3x,A,7x,A,20x,A,19x,A,17x,A,16x,A,20x,A)', 'l', 'Ek', 'GSL W', 'GSL W''', 'UKRmol W', 'UKRmol W''', &
            'asy W', 'asy W'''

        do ie = 1, size(Ek)
            k = sqrt(-2*Ek(ie))
            do ipw = 1, size(ls)
                l = ls(ipw)
                call coul_gsl(Z, l, Ek(ie), r, F, Fp, G1, G1p)
                call coul_ukrmol(Z, l, Ek(ie), r, F, Fp, G2, G2p)
                F = 1; G3 = 0; G3p = 0
                do n = 0, nTermsAsy
                    if (n > 0) F = F * (l + 1 - 1/k + n - 1) * (-l - 1/k + n - 1) / n / (-2*k*r)
                    G3 = G3 + F
                    G3p = G3p + (1/k - n)/r * F
                end do
                G3p = exp(-k*r) * (2*k*r)**(1/k) * (-k*G3 + G3p)
                G3 = exp(-k*r) * (2*k*r)**(1/k) * G3
                print '(I3,F8.3,6E25.15)', l, Ek(ie), G1, G1p, G2, G2p, G3, G3p
            end do
        end do

    end subroutine test_coul


    !> \brief   Tests of the Levin quadrature routine
    !> \author  J Benda
    !> \date    2021
    !>
    subroutine test_levin

        use multidip_romberg, only: nested_cgreen_correct_romberg
        use multidip_levin,   only: nested_cgreen_correct_levin

        real        :: t0, t1, t2
        real(wp)    :: ra = 15, rb = 100, c = 0, Z = 1
        integer     :: m, s1 = +1, s2, l1, l2
        complex(wp) :: k1, k2, romb, levn

        print '(/,A)', 'Testing Levin quadrature'
        print '(/,1x,A,2x,A,6x,A,4x,A,4x,A,4x,A,4x,A,2x,A,2x,A,2x,A,2x,A,4x,A,15x,A,15x,A,17x,A,15x,A,4x,A)', &
            'Ra', 'Rb', 'Re k1', 'Im k1', 'Re k2', 'Im k2', 'm', 's1', 's2', 'l1', 'l2', &
            're romberg', 'im romberg', 're levin', 'im levin', 'μs romberg', 'μs levin'

        k1 = (0.005, 0.000)
        k2 = (0.500, 0.000)

        do l1 = 0, 1
            do m = 0, 1
                do s2 = -1, +1, 2

                    l2 = l1 + 1

                    call cpu_time(t0)
                    romb = 0; call nested_cgreen_correct_romberg(Z, ra, rb, c, 1, s1, s2, [ m ], [ l1, l2 ], [ k1, k2 ], romb)
                    call cpu_time(t1)
                    levn = 0; call nested_cgreen_correct_levin  (Z, ra, rb, c, 1, s1, s2, [ m ], [ l1, l2 ], [ k1, k2 ], levn)
                    call cpu_time(t2)

                    print '(2(F4.0,1x),4(F8.3,1x),I4,SP,2I4,SS,2I4,4E25.15,2I12)', &
                        ra, rb, k1, k2, m, s1, s2, l1, l2, romb, levn, int(1e6*(t1 - t0)), int(1e6*(t2 - t1))

                end do
            end do
        end do

        k1 = (0.000, 0.005)
        k2 = (0.500, 0.000)

        do l1 = 0, 1
            do m = 0, 1
                do s2 = -1, +1, 2

                    l2 = l1 + 1

                    call cpu_time(t0)
                    romb = 0; call nested_cgreen_correct_romberg(Z, ra, rb, c, 1, s1, s2, [ m ], [ l1, l2 ], [ k1, k2 ], romb)
                    call cpu_time(t1)
                    levn = 0; call nested_cgreen_correct_levin  (Z, ra, rb, c, 1, s1, s2, [ m ], [ l1, l2 ], [ k1, k2 ], levn)
                    call cpu_time(t2)

                    print '(2(F4.0,1x),4(F8.3,1x),I4,SP,2I4,SS,2I4,4E25.15,2I12)', &
                        ra, rb, k1, k2, m, s1, s2, l1, l2, romb, levn, int(1e6*(t1 - t0)), int(1e6*(t2 - t1))

                end do
            end do
        end do

    end subroutine test_levin


    !> \brief   One-photon exponential integral unit tests
    !> \author  J Benda
    !> \date    2020
    !>
    !> Tests the function `nested_exp_integ` for several arguments.
    !>
    subroutine test_1p_eint

        use multidip_integ, only: nested_exp_integ

        real(wp) :: Z = 1, a = 150, c = 0
        real(wp) :: k1(2) = [ 0.011080185297_wp, 0.680747949322_wp ]
        real(wp) :: k2(2) = [ 1.069105413537_wp, 1.438602233160_wp ]

        integer  :: ie, m, s2, ms(1), ss(2)

        complex(wp) :: ks(2), integ
        complex(wp) :: reference(2,3,2) = reshape([(-4.772929e-01_wp, +8.885647e+01_wp), (-8.424650e-04_wp, +5.923864e-01_wp),  &
                                                   (+9.979744e-06_wp, +3.949186e-03_wp), (+1.307372e+00_wp, +3.240230e+02_wp),  &
                                                   (+3.965361e-02_wp, +2.158463e+00_wp), (+4.697322e-04_wp, +1.437268e-02_wp),  &
                                                   (-2.211202e-01_wp, +7.029786e+01_wp), (-9.899674e-06_wp, +4.686525e-01_wp),  &
                                                   (+9.695004e-06_wp, +3.124290e-03_wp), (-1.752969e+00_wp, +1.992839e+02_wp),  &
                                                   (+8.061387e-05_wp, +1.328557e+00_wp), (+7.894722e-05_wp, +8.855647e-03_wp)], &
                                                   [2, 3, 2])

        print '(/,A)', 'Testing one-photon exponential integrals'
        print '(/,2x,A,1x,A,1x,A,2x,A,12x,A,13x,A,2x,A,3x,A,4x,A)', 'm', 's2', 's1', 'k2', 'k1', &
            're calculated', 'im calculated', 're reference', 'im reference'

        do ie = 1, size(k1)
            do m = 0, 2
                do s2 = 1, 2

                    ms = [ 1 - m ]
                    ss = [ (-1)**(s2+1), +1 ]
                    ks = [ k1(ie), k2(ie) ]

                    integ = nested_exp_integ(Z, a, c, 1, ms, ss, ks)

                    print '(SP,3I3,SS,2F14.10,SP,4E16.7)', ms(1), ss(2), ss(1), real(ks(2),wp), real(ks(1),wp), &
                          real(integ, wp), aimag(integ), real(reference(s2,m+1,ie), wp), aimag(reference(s2,m+1,ie))
                end do
            end do
        end do

    end subroutine test_1p_eint


    !> \brief   One-photon Coulomb integral unit tests
    !> \author  J Benda
    !> \date    2020 - 2023
    !>
    !> Tests the function `nested_coul_integ` for several arguments.
    !>
    subroutine test_1p_cint

        use multidip_integ, only: nested_coul_integ

        real(wp) :: a = 150, c = 0, Z = 1
        real(wp) :: k1(2) = [ 0.011080185297_wp, 0.680747949322_wp ]
        real(wp) :: k2(2) = [ 1.069105413537_wp, 1.438602233160_wp ]

        integer  :: ie, ipw, s2, ms(1), ss(2), ls(2)
        integer  :: l1(8) = [ 1, 1, 3, 1, 3, 3, 3, 3 ]
        integer  :: l2(8) = [ 0, 2, 2, 2, 2, 4, 2, 4 ]

        complex(wp) :: ks(2), integ
        complex(wp) :: reference(2,8,2) = reshape([(-3.454030e+13_wp, +1.290212e+14_wp), (+4.518674e+14_wp, +1.770164e+14_wp),  &
                                                   (-1.054015e+14_wp, -8.206518e+13_wp), (-3.390989e+14_wp, +3.473927e+14_wp),  &
                                                   (-1.012103e+14_wp, -8.818320e+13_wp), (-3.209952e+14_wp, +3.673575e+14_wp),  &
                                                   (-1.054015e+14_wp, -8.206518e+13_wp), (-3.390989e+14_wp, +3.473927e+14_wp),  &
                                                   (-1.012103e+14_wp, -8.818320e+13_wp), (-3.209952e+14_wp, +3.673575e+14_wp),  &
                                                   (+1.307948e+14_wp, +3.038993e+13_wp), (+1.111367e+14_wp, -4.753884e+14_wp),  &
                                                   (-1.012103e+14_wp, -8.818320e+13_wp), (-3.209952e+14_wp, +3.673575e+14_wp),  &
                                                   (+1.307948e+14_wp, +3.038993e+13_wp), (+1.111367e+14_wp, -4.753884e+14_wp),  &
                                                   (-6.382646e+01_wp, +2.799838e+01_wp), (+7.071896e+01_wp, +1.844497e+02_wp),  &
                                                   (+1.584632e+01_wp, -6.787748e+01_wp), (-1.900621e+02_wp, -5.395706e+01_wp),  &
                                                   (+5.054597e+01_wp, +4.803432e+01_wp), (+4.962959e+01_wp, +1.911980e+02_wp),  &
                                                   (+1.584632e+01_wp, -6.787748e+01_wp), (-1.900621e+02_wp, -5.395706e+01_wp),  &
                                                   (+5.054597e+01_wp, +4.803432e+01_wp), (+4.962959e+01_wp, +1.911980e+02_wp),  &
                                                   (-6.443654e+01_wp, -2.668020e+01_wp), (-1.150300e+02_wp, -1.606738e+02_wp),  &
                                                   (+5.054597e+01_wp, +4.803432e+01_wp), (+4.962959e+01_wp, +1.911980e+02_wp),  &
                                                   (-6.443654e+01_wp, -2.668020e+01_wp), (-1.150300e+02_wp, -1.606738e+02_wp)], &
                                                   [2, 8, 2])

        print '(/,A)', 'Testing one-photon Coulomb integrals'
        print '(/,1x,A,1x,A,1x,A,2x,A,12x,A,13x,A,2x,A,3x,A,4x,A)', 's2', 'l2', 'l1', 'k2', 'k1', &
            're calculated', 'im calculated', 're reference', 'im reference'

        do ie = 1, size(k1)
            do ipw = 1, size(l1)
                do s2 = 1, 2

                    ms = [ 1 ]
                    ss = [ (-1)**(s2+1), +1 ]
                    ls = [ l1(ipw), l2(ipw) ]
                    ks = [ k1(ie), k2(ie) ]

                    integ = nested_coul_integ(Z, a, c, 1, ms, ss, ls, ks)

                    print '(SP,I3,SS,2I3,2F14.10,SP,4E16.7)', (-1)**(s2+1), l2(ipw), l1(ipw), k2(ie), k1(ie), &
                          real(integ, wp), aimag(integ), real(reference(s2,ipw,ie), wp), aimag(reference(s2,ipw,ie))
                end do
            end do
        end do

    end subroutine test_1p_cint


    !> \brief   One-photon Green integral unit tests
    !> \author  J Benda
    !> \date    2021 - 2023
    !>
    !> Tests the function `nested_cgreen_integ` for several arguments. Compares results obtained from asymptotic integration
    !> over interval (+50,+inf) to results obtained using a combination of numerical integration of exact Coulomb functions
    !> at interval (+50,+150) and asymptotic formulas on (+150,+inf).
    !>
    subroutine test_1p_gint

        use multidip_integ, only: nested_cgreen_integ

        real(wp) :: a = 50, b = 150, c = 0, Z = 1

        real(wp) :: k1(2) = [ 0.011080185297_wp, 0.680747949322_wp ]
        real(wp) :: k2(2) = [ 1.069105413537_wp, 1.438602233160_wp ]

        integer :: l1(8) = [ 1, 1, 3, 1, 3, 3, 3, 3 ]
        integer :: l2(8) = [ 0, 2, 2, 2, 2, 4, 2, 4 ]

        integer :: ie, ipw, s1, s2, sa, sb, ms(1), ls(2)

        complex(wp) :: ks(2), integ_ai, integ_bi, integ_ab

        print '(/,A)', 'Testing one-photon Green integrals (numerical integration)'
        print '(/,4(1x,A),2x,A,12x,A,12x,A,3x,A,3x,A,3x,A)', 's2', 's1', 'l2', 'l1', 'k2', 'k1', &
            're numerical ', 'im numerical ', 're asymptotic', 'im asymptotic'

        do ie = 1, size(k1)
            do ipw = 1, size(l1)
                do s2 = 1, 2
                    do s1 = 1, 2

                        ms = [ 1 ]
                        ls = [ l1(ipw), l2(ipw) ]
                        ks = [ k1(ie), k2(ie) ]

                        sa = (-1)**(s1 + 1)
                        sb = (-1)**(s2 + 1)

                        integ_ai = nested_cgreen_integ(Z, a, a, c, 1, sa, sb, ms, ls, ks)
                        integ_bi = nested_cgreen_integ(Z, b, b, c, 1, sa, sb, ms, ls, ks)
                        integ_ab = nested_cgreen_integ(Z, a, b, c, 1, sa, sb, ms, ls, ks)

                        print '(SP,2I3,SS,2I3,2F14.10,SP,8E16.7)', sb, sa, l2(ipw), l1(ipw), k2(ie), k1(ie), &
                            real(integ_ab - integ_bi, wp), aimag(integ_ab - integ_bi), &
                            real(integ_ai - integ_bi, wp), aimag(integ_ai - integ_bi)

                    end do
                end do
            end do
        end do

    end subroutine test_1p_gint


    !> \brief   Two-photon Coulomb integral unit tests
    !> \author  J Benda
    !> \date    2020 - 2023
    !>
    !> Tests the function `nested_coul_integ` for several arguments.
    !>
    subroutine test_2p_cint

        use multidip_integ, only: nested_coul_integ

        real(wp) :: a = 150, c = 0, Z = 1

        real(wp) :: k1(2) = [ 0.011080185297_wp, 0.680747949322_wp ]
        real(wp) :: k2(2) = [ 1.069105413537_wp, 1.438602233160_wp ]
        real(wp) :: k3(2) = [ 1.511902774652_wp, 1.917220644579_wp ]

        integer :: l1(8) = [ 1, 1, 3, 1, 3, 3, 3, 3 ]
        integer :: l2(8) = [ 0, 2, 2, 2, 2, 4, 2, 4 ]
        integer :: l3(8) = [ 1, 1, 1, 3, 3, 3, 3, 3 ]

        integer :: ie, ipw, s4, ms(2), ss(4), ls(4)

        complex(wp) :: ks(4), integ1, integ2, integ
        complex(wp) :: reference(2,8,2) = reshape([(-1.465883e+16_wp, -4.950701e+15_wp), (-3.842765e+16_wp,  8.620168e+16_wp),  &
                                                   (-1.466146e+16_wp, -4.951551e+15_wp), (-3.843793e+16_wp,  8.623263e+16_wp),  &
                                                   (-1.443664e+16_wp, -5.780723e+15_wp), (-3.378946e+16_wp,  8.865409e+16_wp),  &
                                                   ( 1.520251e+16_wp, -2.902864e+15_wp), (-8.964058e+15_wp, -9.400575e+16_wp),  &
                                                   ( 1.541476e+16_wp, -2.070261e+15_wp), (-1.420349e+16_wp, -9.382680e+16_wp),  &
                                                   ( 1.542117e+16_wp, -2.071208e+15_wp), (-1.422236e+16_wp, -9.390101e+16_wp),  &
                                                   ( 1.541476e+16_wp, -2.070261e+15_wp), (-1.420349e+16_wp, -9.382680e+16_wp),  &
                                                   ( 1.542117e+16_wp, -2.071208e+15_wp), (-1.422236e+16_wp, -9.390101e+16_wp),  &
                                                   (-4.322904e+03_wp, -4.890496e+03_wp), (-2.301503e+04_wp,  1.829823e+04_wp),  &
                                                   (-4.323345e+03_wp, -4.890988e+03_wp), (-2.301886e+04_wp,  1.830140e+04_wp),  &
                                                   ( 6.409175e+03_wp, -1.254518e+03_wp), ( 2.742317e+04_wp,  1.058896e+04_wp),  &
                                                   ( 5.914821e+03_wp,  2.763232e+03_wp), ( 1.382079e+04_wp, -2.596191e+04_wp),  &
                                                   (-5.378829e+03_wp,  3.705000e+03_wp), (-2.937519e+04_wp,  1.216512e+03_wp),  &
                                                   (-5.380092e+03_wp,  3.705885e+03_wp), (-2.938678e+04_wp,  1.217206e+03_wp),  &
                                                   (-5.378829e+03_wp,  3.705000e+03_wp), (-2.937519e+04_wp,  1.216512e+03_wp),  &
                                                   (-5.380092e+03_wp,  3.705885e+03_wp), (-2.938678e+04_wp,  1.217206e+03_wp)], &
                                                   [2, 8, 2])

        print '(/,A)', 'Testing two-photon Coulomb integrals'
        print '(/,4(1x,A),2x,A,12x,A,12x,A,12x,A,3x,A,3x,A,4x,A)', 's4', 'l3', 'l2', 'l1', 'k3', 'k2', 'k1', &
            're calculated', 'im calculated', 're reference', 'im reference'

        do ie = 1, size(k1)
            do ipw = 1, size(l1)
                do s4 = 1, 2

                    ms = [ 1, 1 ]

                    ss = [ (-1)**(s4+1), +1, -1, +1 ]
                    ls = [ l1(ipw), l2(ipw), l2(ipw), l3(ipw) ]
                    ks = [ k1(ie), k2(ie), k2(ie), k3(ie) ]

                    integ1 = nested_coul_integ(Z, a, c, 2, ms, ss, ls, ks)

                    ss = [ +1, +1, -1, (-1)**(s4+1) ]
                    ls = [ l3(ipw), l2(ipw), l2(ipw), l1(ipw) ]
                    ks = [ k3(ie), k2(ie), k2(ie), k1(ie) ]

                    integ2 = nested_coul_integ(Z, a, c, 2, ms, ss, ls, ks)

                    integ = integ1 + integ2

                    print '(SP,I3,SS,3I3,3F14.10,SP,4E16.7)', (-1)**(s4+1), l3(ie), l2(ipw), l1(ipw), k3(ie), k2(ie), k1(ie), &
                          real(integ, wp), aimag(integ), real(reference(s4,ipw,ie), wp), aimag(reference(s4,ipw,ie))
                end do
            end do
        end do

    end subroutine test_2p_cint


    !> \brief   Two-photon Green integral unit tests
    !> \author  J Benda
    !> \date    2020 - 2023
    !>
    !> Tests the function `nested_cgreen_integ` for several arguments.
    !>
    subroutine test_2p_gint

        use multidip_integ, only: nested_cgreen_integ

        real(wp) :: a = 150, c = 0, Z = 1

        real(wp) :: k1(2) = [ 0.011080185297_wp, 0.680747949322_wp ]
        real(wp) :: k2(2) = [ 1.069105413537_wp, 1.438602233160_wp ]
        real(wp) :: k3(2) = [ 1.511902774652_wp, 1.917220644579_wp ]

        integer :: l1(8) = [ 1, 1, 3, 1, 3, 3, 3, 3 ]
        integer :: l2(8) = [ 0, 2, 2, 2, 2, 4, 2, 4 ]
        integer :: l3(8) = [ 1, 1, 1, 3, 3, 3, 3, 3 ]

        integer :: ie, ipw, s4, ms(2), ls(3)

        complex(wp) :: ks(3), integ1, integ2
        complex(wp) :: reference(2,2,8,2) = reshape([(-8.310927e+15_wp,  7.523384e+15_wp), (-5.909915e+16_wp,  6.931934e+16_wp),  &
                                                     ( 5.679523e+16_wp,  4.672527e+16_wp), ( 8.225897e+24_wp,  1.848188e+24_wp),  &
                                                     (-6.506509e+15_wp,  2.066669e+16_wp), (-1.618385e+16_wp,  4.224605e+15_wp),  &
                                                     ( 1.050096e+17_wp,  4.554189e+16_wp), (-4.312479e+24_wp, -7.240598e+24_wp),  &
                                                     (-7.673705e+15_wp,  2.037601e+16_wp), (-1.647265e+16_wp,  3.342092e+15_wp),  &
                                                     ( 1.078882e+17_wp,  3.987695e+16_wp), (-4.728274e+24_wp, -7.026109e+24_wp),  &
                                                     (-4.503787e+15_wp, -2.119630e+16_wp), ( 1.619053e+16_wp,  4.287343e+15_wp),  &
                                                     (-1.138495e+17_wp,  1.201762e+16_wp), ( 2.556995e+23_wp,  8.432496e+24_wp),  &
                                                     (-3.344463e+15_wp, -2.151751e+16_wp), ( 1.600768e+16_wp,  5.199000e+15_wp),  &
                                                     (-1.135684e+17_wp,  1.836697e+16_wp), ( 7.238276e+23_wp,  8.446832e+24_wp),  &
                                                     (-8.612323e+15_wp, -1.723199e+16_wp), ( 3.535435e+16_wp,  4.008848e+16_wp),  &
                                                     (-9.807095e+16_wp,  3.755844e+16_wp), ( 7.275593e+24_wp,  4.341652e+24_wp),  &
                                                     (-3.344463e+15_wp, -2.151751e+16_wp), ( 1.600768e+16_wp,  5.199000e+15_wp),  &
                                                     (-1.135684e+17_wp,  1.836697e+16_wp), ( 7.238276e+23_wp,  8.446832e+24_wp),  &
                                                     (-8.612323e+15_wp, -1.723199e+16_wp), ( 3.535435e+16_wp,  4.008848e+16_wp),  &
                                                     (-9.807095e+16_wp,  3.755844e+16_wp), ( 7.275593e+24_wp,  4.341652e+24_wp),  &
                                                     (-1.247516e+03_wp,  3.118370e+03_wp), (-2.763140e+04_wp,  1.784564e+04_wp),  &
                                                     ( 1.332880e+04_wp,  9.920923e+03_wp), ( 6.345060e+03_wp,  4.578023e+04_wp),  &
                                                     (-3.896753e+03_wp,  9.080720e+02_wp), (-6.662204e+03_wp,  3.014237e+04_wp),  &
                                                     ( 6.717303e+03_wp,  1.712697e+04_wp), ( 4.397613e+04_wp, -1.193480e+04_wp),  &
                                                     ( 1.189054e+03_wp, -3.822275e+03_wp), (-2.263210e+04_wp, -2.099869e+04_wp),  &
                                                     ( 1.136951e+04_wp, -1.445428e+04_wp), (-3.253996e+04_wp, -3.189015e+04_wp),  &
                                                     ( 3.212357e+03_wp, -2.385969e+03_wp), ( 1.813020e+04_wp, -2.499891e+04_wp),  &
                                                     (-1.298806e+04_wp, -1.303291e+04_wp), (-4.510261e+04_wp, -6.582287e+03_wp),  &
                                                     ( 4.327121e+02_wp,  3.979850e+03_wp), ( 1.239242e+04_wp,  2.828936e+04_wp),  &
                                                     (-4.667920e+03_wp,  1.779027e+04_wp), ( 1.713954e+04_wp,  4.222912e+04_wp),  &
                                                     ( 1.147174e+03_wp,  5.355515e+03_wp), ( 1.429665e+04_wp,  1.753491e+04_wp),  &
                                                     (-5.010378e+03_wp,  2.217364e+04_wp), ( 3.781810e+04_wp,  1.923663e+04_wp),  &
                                                     ( 4.327121e+02_wp,  3.979850e+03_wp), ( 1.239242e+04_wp,  2.828936e+04_wp),  &
                                                     (-4.667920e+03_wp,  1.779027e+04_wp), ( 1.713954e+04_wp,  4.222912e+04_wp),  &
                                                     ( 1.147174e+03_wp,  5.355515e+03_wp), ( 1.429665e+04_wp,  1.753491e+04_wp),  &
                                                     (-5.010378e+03_wp,  2.217364e+04_wp), ( 3.781810e+04_wp,  1.923663e+04_wp)], &
                                                     [2,2,8,2])

        print '(/,A)', 'Testing two-photon Green integrals'
        print '(/,5(1x,A),2x,A,12x,A,12x,A,12x,A,3x,A,3x,A,4x,A)', 's4', 's1', 'l3', 'l2', 'l1', 'k3', 'k2', 'k1', &
            're calculated', 'im calculated', 're reference', 'im reference'

        do ie = 1, size(k1)
            do ipw = 1, size(l1)
                do s4 = 1, 2

                    ms = [ 1, 1 ]
                    ls = [ l1(ipw), l2(ipw), l3(ipw) ]
                    ks = [ k1(ie), k2(ie), k3(ie) ]

                    integ1 = nested_cgreen_integ(Z, a, a, c, 2, (-1)**(s4+1), +1, ms, ls, ks)
                    integ2 = nested_cgreen_integ(Z, a, a, c, 2, (-1)**(s4+1), -1, ms, ls, ks)

                    print '(SP,2I3,SS,3I3,3F14.10,SP,8E16.7)', (-1)**(s4+1), +1, l3(ipw), l2(ipw), l1(ipw), k3(ie), k2(ie), k1(ie),&
                          real(integ1, wp), aimag(integ1), real(reference(1,s4,ipw,ie), wp), aimag(reference(1,s4,ipw,ie))
                    print '(SP,2I3,SS,3I3,3F14.10,SP,8E16.7)', (-1)**(s4+1), -1, l3(ipw), l2(ipw), l1(ipw), k3(ie), k2(ie), k1(ie),&
                          real(integ2, wp), aimag(integ2), real(reference(2,s4,ipw,ie), wp), aimag(reference(2,s4,ipw,ie))
                end do
            end do
        end do
    end subroutine test_2p_gint


    !> \brief   Angular algebra unit tests
    !> \author  J Benda
    !> \date    2020
    !>
    !> Tests the function `beta_contraction_tensor` for several arguments.
    !>
    subroutine test_ang_dist

        use dipelm_special_functions, only: threej
        use multidip_params,          only: rone
        use multidip_special,         only: beta_contraction_tensor, beta_2p_demekhin, beta_2p_arb_pol_sph_harm

        integer  :: J = 0, li = 1, lj = 1, mi = 0, mj = 0, ki = 0, kj = 0, qi = 0, qj = 0, p = 0
        real(wp) :: beta, ref, beta_ap

        print '(/,A)', 'Testing one-photon angular integrals'
        print '(/,8A4,2A12)', 'J', 'p', 'li', 'lj', 'mi', 'mj', 'qi', 'qj', 'TJij', 'reference'

        do J = 0, 2
            do p = 0, +1
                do mi = -1, +1
                    do mj = -1, +1
                        do qi = -1, +1
                            do qj = -1, +1
                                beta = beta_contraction_tensor(J, 1, [ p ], li, mi, [ qi ], lj, mj, [ qj ])
                                ref = (-1)**(p + mi + qi) * (2*J + 1) * sqrt((2*li + rone)*(2*lj + rone))  &
                                    * threej(2*li, 0, 2*lj, 0, 2*J, 0)  &
                                    * threej(2*1, 2*p, 2*1, -2*p, 2*J, 0)  &
                                    * threej(2*li, 2*mi, 2*lj, -2*mj, 2*J, -2*(mi-mj))  &
                                    * threej(2*1, 2*qi, 2*1, -2*qj, 2*J, -2*(mi-mj))
                                if (abs(beta) > 1e-10 .or. abs(ref) > 1e-10) then
                                    print '(I4,SP,I4,SS,2I4,SP,4I4,2F12.7,A8)', &
                                        J, p, li, lj, mi, mj, qi, qj, beta, ref, &
                                        merge('    ok', 'WRONG!', abs(beta - ref) / (abs(beta) + abs(ref)) < 1e-5)
                                end if
                            end do
                        end do
                    end do
                end do
            end do
        end do

        print '(/,A)', 'Testing two-photon angular integrals'
        print '(/,10A4,3A12)', 'J', 'p', 'li', 'lj', 'mi', 'mj', 'ki', 'kj', 'qi', 'qj', 'TJij', 'TJij(ap)', 'reference'

        do J = 0, 2
            do p = 0, +1
                do mi = -1, +1
                    do mj = -1, +1
                        do ki = -1, +1
                            do kj = -1, +1
                                do qi = -1, +1
                                    do qj = -1, +1
                                        beta = beta_contraction_tensor(J, 2, [ p, p ], li, mi, [ ki, qi ], lj, mj, [ kj, qj ])
                                        beta_ap = beta_2p_arb_pol_sph_harm(J, 0, 2, [p,p], li, mi, [ki,qi], [p,p], lj, mj, [kj,qj])
                                        ref = beta_2p_demekhin(J, p, p, li, mi, ki, qi, lj, mj, kj, qj)
                                        if (abs(beta) > 1e-10 .or. abs(beta_ap) > 1e-10 .or. abs(ref) > 1e-10) then
                                            print '(I4,SP,I4,SS,2I4,SP,6I4,3F12.7,2A8)', &
                                                J, p, li, lj, mi, mj, ki, kj, qi, qj, beta, beta_ap, ref, &
                                                merge('    ok', 'WRONG!', abs(beta - ref) / (abs(beta) + abs(ref)) < 1e-5), &
                                                merge('    ok', 'WRONG!', abs(beta_ap - ref) / (abs(beta_ap) + abs(ref)) < 1e-5)
                                        end if
                                    end do
                                end do
                            end do
                        end do
                    end do
                end do
            end do
        end do

    end subroutine test_ang_dist

end module multidip_tests
