! Copyright 2024
!
! For a comprehensive list of the developers that contributed to these codes
! see the UK-AMOR website.
!
! This file is part of UKRmol-in (UKRmol+ suite).
!
!     UKRmol-in is free software: you can redistribute it and/or modify
!     it under the terms of the GNU General Public License as published by
!     the Free Software Foundation, either version 3 of the License, or
!     (at your option) any later version.
!
!     UKRmol-in is distributed in the hope that it will be useful,
!     but WITHOUT ANY WARRANTY; without even the implied warranty of
!     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!     GNU General Public License for more details.
!
!     You should have received a copy of the GNU General Public License
!     along with  UKRmol-in (in source/COPYING). Alternatively, you can also visit
!     <https://www.gnu.org/licenses/>.

!> \brief   Interfaces to C routines from linalg_cl.c
!> \authors J Benda
!> \date    2024 - 2025
!>
!> Interfaces in this module can be used to call C functions defined in linalg_cl.c from Fortran.
!>
module linalg_cl

    use iso_fortran_env, only: error_unit, output_unit

    implicit none

    integer :: u_err = error_unit       !< Unit for error outputs from this module (default to stderr)
    integer :: u_out = output_unit      !< Unit for info outputs from this module (default to stdout)

    ! Interfaces to C routines defined in "linalg_cl.c".

    interface
        function is_initialized_cl () bind(C, name='is_initialized_cl')
            use iso_c_binding, only: c_int
            integer(c_int) :: is_initialized_cl
        end function is_initialized_cl
    end interface

    interface
        subroutine initialize_cl (platform, device) bind(C, name='initialize_cl')
            use iso_c_binding, only: c_int
            integer(c_int), value :: platform, device
        end subroutine initialize_cl
    end interface

    interface
        subroutine finalize_cl () bind(C, name='finalize_cl')
        end subroutine finalize_cl
    end interface

    interface
        subroutine dgemm_cl (transa, transb, m, n, k, alpha, A, lda, B, ldb, beta, C, ldc) bind(C, name='dgemm_cl')
            use iso_c_binding, only: c_int, c_double
            integer(c_int), value, intent(in)    :: transa, transb, m, n, k, lda, ldb, ldc
            real(c_double), value, intent(in)    :: alpha, beta
            real(c_double),        intent(in)    :: A(*), B(*)
            real(c_double),        intent(inout) :: C(*)
        end subroutine dgemm_cl
    end interface

    interface
        subroutine zgemm_cl (transa, transb, m, n, k, alpha, A, lda, B, ldb, beta, C, ldc) bind(C, name='zgemm_cl')
            use iso_c_binding, only: c_int, c_double
            integer(c_int),    value, intent(in)    :: transa, transb, m, n, k, lda, ldb, ldc
            complex(c_double), value, intent(in)    :: alpha, beta
            complex(c_double),        intent(in)    :: A(*), B(*)
            complex(c_double),        intent(inout) :: C(*)
        end subroutine zgemm_cl
    end interface

    interface
        subroutine residr_cl (stage, nchan, nstat, compress, alpha, epole, etotr, wamp, ld, rmat) bind(C, name="residr_cl")
            use iso_c_binding, only: c_int, c_double
            integer(c_int), value :: stage, nchan, nstat, compress, ld
            real(c_double), value :: alpha, etotr
            real(c_double)        :: epole(*), wamp(*), rmat(*)
        end subroutine residr_cl
    end interface

contains

    !> \brief   Initialize this module
    !> \authors J Benda
    !> \date    2025
    !>
    !> Initialize OpenCL and redirect C outputs to given Fortran units.
    !>
    subroutine linalg_cl_init(platform, device, opt_u_out, opt_u_err)

        use iso_c_binding, only: c_int

        integer(c_int),    intent(in) :: platform, device
        integer, optional, intent(in) :: opt_u_out, opt_u_err

        if (present(opt_u_out)) u_out = opt_u_out
        if (present(opt_u_err)) u_err = opt_u_err

        call initialize_cl(platform, device)

    end subroutine


    !> \brief   Print information
    !> \authors J Benda
    !> \date    2025
    !>
    !> This subroutine is called from "linalg_cl.c" so that all text output is written from
    !> Fortran code. This is needed, because the Fortran and C runtime libraries may have
    !> separate I/O buffers, which messes up ordering of outputs if both are used simultaneously.
    !>
    subroutine f_print_info(string, length) bind(C, name='f_print_info')

        use iso_c_binding, only: c_char, c_int

        character(c_char), intent(in) :: string(*)
        integer(c_int),    intent(in) :: length

        if (length > 0) then
            write (u_out, '(*(a))') string(1:length)
        else
            write (u_out, '()')
        end if

    end subroutine f_print_info


    !> \brief   Print error
    !> \authors J Benda
    !> \date    2025
    !>
    !> This subroutine is called from "linalg_cl.c" so that all text output is written from
    !> Fortran code. This is needed, because the Fortran and C runtime libraries may have
    !> separate I/O buffers, which messes up ordering of outputs if both are used simultaneously.
    !>
    subroutine f_print_error(string, length) bind(C, name='f_print_error')

        use iso_c_binding, only: c_char, c_int

        character(c_char), intent(in) :: string(*)
        integer(c_int),    intent(in) :: length

        if (length > 0) then
            write (u_err, '(*(a))') string(1:length)
        else
            write (u_err, '()')
        end if

    end subroutine f_print_error

end module linalg_cl
