      module interpolate
      use bspline
      implicit none

      integer, parameter :: idp = kind(1d0) 

!     Types to hold knots and bspline coefficients

      type zspline
        integer :: kxord=5,kyord=5,kzord=5 !- order of the spline.  (input)
        integer :: nx,ny,nz,ndim
        real(kind=idp), allocatable, dimension(:) :: xknot, yknot,zknot  !- array of length ndata+korder containing the knot sequence.  (output)
        real(kind=idp), allocatable, dimension(:) :: bcoef_1D_re, bcoef_1D_im! - array of length ndata containing the b-spline coefficients. 
        real(kind=idp), allocatable, dimension(:,:) :: bcoef_2D_re, bcoef_2D_im
        real(kind=idp), allocatable, dimension(:,:,:) :: bcoef_3D_re, bcoef_3D_im
      end type zspline

      type dspline
        integer :: kxord=5,kyord=5,kzord=5 !- order of the spline.  (input)
        integer :: nx,ny,nz,ndim 
        real(kind=idp), allocatable, dimension(:) :: xknot, yknot,zknot  !- array of length ndata+korder containing the knot sequence.  (output)
        real(kind=idp), allocatable, dimension(:) :: bcoef_1D! - array of length ndata containing the b-spline coefficients. 
        real(kind=idp), allocatable, dimension(:,:) :: bcoef_2D
        real(kind=idp), allocatable, dimension(:,:,:) :: bcoef_3D
      end type dspline

      interface bspline_init
         module procedure zbspline_init_1D
         module procedure dbspline_init_1D
         module procedure zbspline_init_2D
         module procedure dbspline_init_2D
         module procedure zbspline_init_3D
!         module procedure rmat_bspline_init_1D
      end interface

      interface bspline_int
         module procedure zbspline_1D
         module procedure dbspline_1D
         module procedure zbspline_2D
         module procedure dbspline_2D
         module procedure zbspline_3D
      end interface

      contains

!     1D Interpolation
!     ******************************************************************

      subroutine zbspline_init_1D(xvec,xdata,s,nord)
      !Complex array
      type(zspline) ::s
      real(kind=idp), dimension(:) :: xvec
      complex(kind=idp), dimension(:) :: xdata
      integer,optional :: nord
      real(kind=idp), allocatable, dimension(:) :: xdata_re, xdata_im
      integer :: i

!     check allocation status of xdata_re, xdata_im

      s%nx=size(xvec)
      s%ndim=1
      if (present(nord)) then
         s%kxord=nord
      end if 

      allocate(xdata_re(s%nx), xdata_im(s%nx))
      allocate(s%xknot(s%nx+s%kxord),s%bcoef_1D_re(s%nx),s%bcoef_1D_im(s%nx))

      call dbsnak(s%nx,xvec,s%kxord,s%xknot)

      do i=1,s%nx
         xdata_re(i)=real(xdata(i))
         xdata_im(i)=aimag(xdata(i))
      end do

      call dbsint(s%nx,xvec,xdata_re,s%kxord,s%xknot,s%bcoef_1D_re)
      call dbsint(s%nx,xvec,xdata_im,s%kxord,s%xknot,s%bcoef_1D_im)

      deallocate(xdata_re, xdata_im)

      end subroutine zbspline_init_1D

!     ****************************************
      subroutine dbspline_init_1D(xvec,xdata,s,nord)
      !Double precision array
      type(dspline) ::s
      real(kind=idp), dimension(:) :: xvec
      real(kind=idp), dimension(:) :: xdata
      integer,optional :: nord
      integer :: i

!     check allocation status of xdata_re, xdata_im

      s%nx=size(xvec)
      s%ndim=1
      if (present(nord)) then
         s%kxord=nord
      end if 

      allocate(s%xknot(s%nx+s%kxord),s%bcoef_1D(s%nx))

      call dbsnak(s%nx,xvec,s%kxord,s%xknot)

      call dbsint(s%nx,xvec,xdata,s%kxord,s%xknot,s%bcoef_1D)


      end subroutine dbspline_init_1D

!     ****************************************
      complex(kind=idp) function zbspline_1D(x,s)
      real(kind=idp) :: x
      type(zspline) ::s

      if (s%ndim .ne. 1) then
         write(6,100)
         return
      end if

      zbspline_1D=cmplx(dbsval(x,s%kxord,s%xknot,s%nx,s%bcoef_1D_re),dbsval(x,s%kxord,s%xknot,s%nx,s%bcoef_1d_im),kind=idp)
 
      return
  100 format(/' 1D spline initialization required')
      end function zbspline_1D

!     ****************************************
      real(kind=idp) function dbspline_1D(x,s)
      real(kind=idp) :: x
      type(dspline) ::s

      if (s%ndim .ne. 1) then
         write(6,100)
         return
      end if

      dbspline_1D=dbsval(x,s%kxord,s%xknot,s%nx,s%bcoef_1D)
 
      return
  100 format(/' 1D spline initialization required')
      end function dbspline_1D

!     2D Interpolation
!     ******************************************************************

      subroutine zbspline_init_2D(xvec,yvec,xydata,s,nord)
      ! Complex array
      type(zspline) :: s
      integer :: ldf ! -the leading dimension of fdata exactly as specified in the dimension statement of the calling program. (input)
      real(kind=idp), dimension(:) :: xvec, yvec     !- array of length ndata containing the location of the data points.  (input)
      complex(kind=idp), dimension(:,:) :: xydata
      integer,optional :: nord
      real(kind=idp), allocatable, dimension(:,:) :: xydata_re, xydata_im
      integer :: i,j

      s%ndim=2
      s%nx=size(xvec); s%ny=size(yvec)
      ldf=s%nx

      if (present(nord)) then
         s%kxord=nord
         s%kyord=nord
      end if 

      allocate(xydata_re(s%nx,s%ny), xydata_im(s%nx,s%ny))

      allocate(s%xknot(s%nx+s%kxord), s%yknot(s%ny+s%kyord),s%bcoef_2D_re(s%nx,s%ny), s%bcoef_2D_im(s%nx,s%ny))


      call dbsnak(s%nx,xvec,s%kxord,s%xknot)
      call dbsnak(s%ny,yvec,s%kyord,s%yknot)


      do i=1, s%nx
        do j=1, s%ny
            xydata_re(i,j)=real(xydata(i,j))
            xydata_im(i,j)=aimag(xydata(i,j))
        end do
      end do

      call dbs2in(s%nx,xvec,s%ny,yvec,xydata_re,ldf,s%kxord,s%kyord,s%xknot,s%yknot,s%bcoef_2D_re)
      call dbs2in(s%nx,xvec,s%ny,yvec,xydata_im,ldf,s%kxord,s%kyord,s%xknot,s%yknot,s%bcoef_2D_im)

      deallocate(xydata_re, xydata_im)
 
      end subroutine zbspline_init_2D

!     ****************************************
      subroutine dbspline_init_2D(xvec,yvec,xydata,s,nord)
      ! Complex array
      type(dspline) :: s
      integer :: ldf ! -the leading dimension of fdata exactly as specified in the dimension statement of the calling program. (input)
      real(kind=idp), dimension(:) :: xvec, yvec     !- array of length ndata containing the location of the data points.  (input)
      real(kind=idp), dimension(:,:) :: xydata
      integer,optional :: nord
      integer :: i,j

      s%ndim=2
      s%nx=size(xvec); s%ny=size(yvec)
      ldf=s%nx

      if (present(nord)) then
         s%kxord=nord
         s%kyord=nord
      end if 


      allocate(s%xknot(s%nx+s%kxord), s%yknot(s%ny+s%kyord),s%bcoef_2D(s%nx,s%ny) )


      call dbsnak(s%nx,xvec,s%kxord,s%xknot)
      call dbsnak(s%ny,yvec,s%kyord,s%yknot)


      call dbs2in(s%nx,xvec,s%ny,yvec,xydata,ldf,s%kxord,s%kyord,s%xknot,s%yknot,s%bcoef_2D)

 
      end subroutine dbspline_init_2D


!     ****************************************
      complex(kind=idp) function zbspline_2D(x,y,s)
      type(zspline) :: s
      real(kind=idp) :: x,y

      if (s%ndim .ne. 2) then
         write(6,100)
         return
      end if

      zbspline_2D= cmplx(dbs2vl(x,y,s%kxord,s%kyord,s%xknot,s%yknot,s%nx,s%ny,s%bcoef_2D_re), &
                         dbs2vl(x,y,s%kxord,s%kyord,s%xknot,s%yknot,s%nx,s%ny,s%bcoef_2D_im))

      return
  100 format(/' 2D spline initialization required')
      end function zbspline_2D


!     ****************************************
      complex(kind=idp) function dbspline_2D(x,y,s)
      type(dspline) :: s
      real(kind=idp) :: x,y

      if (s%ndim .ne. 2) then
         write(6,100)
         return
      end if

      dbspline_2D= dbs2vl(x,y,s%kxord,s%kyord,s%xknot,s%yknot,s%nx,s%ny,s%bcoef_2D)

      return
  100 format(/' 2D spline initialization required')
      end function dbspline_2D



!     3D Interpolation
!     ******************************************************************

      subroutine zbspline_init_3D(xvec,yvec,zvec,xyzdata,s,nord)
      ! Complex array
      type(zspline) :: s
      integer :: ldf,ldm ! -the leading dimension of fdata exactly as specified in the dimension statement of the calling program. (input)
      real(kind=idp), dimension(:) :: xvec, yvec, zvec     !- array of length ndata containing the location of the data points.  (input)
      complex(kind=idp), dimension(:,:,:) :: xyzdata
      integer,optional :: nord
      real(kind=idp), allocatable, dimension(:,:,:) :: xyzdata_re, xyzdata_im
      integer :: i,j,k

      s%nx=size(xvec); s%ny=size(yvec); s%nz=size(zvec)
      s%ndim=3
      ldf=s%nx
      ldm=s%ny

      if (present(nord)) then
         s%kxord=nord
         s%kyord=nord
         s%kzord=nord
      end if 

      allocate(xyzdata_re(s%nx,s%ny,s%nz), xyzdata_im(s%nx,s%ny,s%nz))
      !Should check whether these array are already allocated
      allocate(s%xknot(s%nx+s%kxord), s%yknot(s%ny+s%kyord),s%zknot(s%nz+s%kzord),&
               s%bcoef_3D_re(s%nx,s%ny,s%nz), s%bcoef_3D_im(s%nx,s%ny,s%nz))


      call dbsnak(s%nx,xvec,s%kxord,s%xknot)
      call dbsnak(s%ny,yvec,s%kyord,s%yknot)
      call dbsnak(s%nz,zvec,s%kzord,s%zknot)


      do i=1, s%nx
        do j=1, s%ny
          do k=1, s%nz
            xyzdata_re(i,j,k)=real(xyzdata(i,j,k))
            xyzdata_im(i,j,k)=aimag(xyzdata(i,j,k))
          end do
        end do
      end do

      call dbs3in(s%nx,xvec,s%ny,yvec,s%nz,zvec,xyzdata_re,ldf,ldm,s%kxord,s%kyord,s%kzord, s%xknot,s%yknot,s%zknot,s%bcoef_3D_re)
      call dbs3in(s%nx,xvec,s%ny,yvec,s%nz,zvec,xyzdata_im,ldf,ldm,s%kxord,s%kyord,s%kzord, s%xknot,s%yknot,s%zknot,s%bcoef_3D_im)

      deallocate(xyzdata_re, xyzdata_im)

      end subroutine zbspline_init_3D

      complex(kind=idp) function zbspline_3D(x,y,z,s)
      type(zspline) :: s
      real(kind=idp) :: x,y,z

      if (s%ndim .ne. 3) then
         write(6,100)
         return
      end if

      zbspline_3D= cmplx(dbs3vl(x,y,z,s%kxord,s%kyord,s%kzord,s%xknot,s%yknot,s%zknot,s%nx,s%ny,s%nz,s%bcoef_3D_re),&
                         dbs3vl(x,y,z,s%kxord,s%kyord,s%kzord,s%xknot,s%yknot,s%zknot,s%nx,s%ny,s%nz,s%bcoef_3D_im))

      return
  100 format(/' 3D spline initialization required')
      end function zbspline_3D

      end module interpolate
