! Copyright 2019
!
! For a comprehensive list of the developers that contributed to these codes
! see the UK-AMOR website.
!
! This file is part of UKRmol-out (UKRmol+ suite).
!
!     UKRmol-out is free software: you can redistribute it and/or modify
!     it under the terms of the GNU General Public License as published by
!     the Free Software Foundation, either version 3 of the License, or
!     (at your option) any later version.
!
!     UKRmol-out is distributed in the hope that it will be useful,
!     but WITHOUT ANY WARRANTY; without even the implied warranty of
!     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!     GNU General Public License for more details.
!
!     You should have received a copy of the GNU General Public License
!     along with  UKRmol-out (in source/COPYING). Alternatively, you can also visit
!     <https://www.gnu.org/licenses/>.
!
      module gettimedel
!
! This module was required to supply data to the subroutine LSFUN1. This
! is used and specified  by a NAG routine. Now the NAG routine has been 
! replaced  by lmdif1 a driver routine for routine lmdif that is least-squares 
! solver and can be included in a specialised library or just appended to 
! this program. Module gettimedel can be retained to supply data to 
! subroutine fcn that is used by lmdif (NV-03).
!
      parameter (maxpnt=1000000)
      double precision :: q(maxpnt),qen(maxpnt)
      integer :: nrestofit
      end module gettimedel
!
!      module userdefkmat
! The variables used in the USER-defined K-matrix routines should 
! be defined in this module
!      end module userdefkmat
!
      subroutine timedel
! Main subroutine
      implicit none
      integer :: maxres,maxnopen,maxpnt,maxthresh,numeig
      parameter (maxres=1000,maxnopen=100)
      parameter (maxpnt=1000000,maxthresh=15)
      parameter (numeig=5) !This is the maximum number of eigenvalues considered
           !and should be sufficient
      double precision :: thresh(maxthresh),range(maxthresh)
      double precision :: tdvalue(maxpnt*numeig),tdenergy(maxpnt)
      double precision :: tdhold(maxnopen),dum1(maxpnt)
      double precision :: conven,crit,epsil,br(maxnopen)
      double precision :: gridinit,gridmin,grid,dele,delemax,dif
      double precision :: kmat1(maxnopen,maxnopen)
      double precision :: kmat2(maxnopen,maxnopen)
      double precision :: etarget(maxthresh),tmp(maxnopen)
      integer :: ithresh,lubr,lutd2,ntarget,ucount,neig,adj,n
      integer :: nes,nthresh,ilast,nopen,nopen2,nthreshols,lstop
      integer :: icount,i,j,ieunit,nres,nptperres,eigcount,io
      integer :: ithrecount,minarr(1),ios,lutd,lukmt,oadj,wthresh
      double precision :: epsab,epsbw,maxn,tmpa(maxpnt*numeig)
      double precision :: adjust,efinal,einit,energy,xthresh
      logical :: logindic,adapt,savekmt,writbr
      character(len=3) :: eunit(3)
     
      double precision :: zero,one,two,toev,pi
      parameter (zero=0.0D0,one=1.0D0,two=2.0D0,toev=1/0.073500d0)
      parameter (pi=3.14159265)
      external setupkmat, getkmat,zheev
!
! maxres      : Maximum number of resonances to be fitted
! maxnopen    : Maximum number of open channels
! maxpnt      : Maximum number of points at which time-delay evaluated
! maxthresh   : Maximum number of thresholds
! range       : Defines the borders for the energy ranges used
! tdvalue     : Holds the value of the time-delay
! tdenergy    : Holds the energy of the time-delay
! conven      : Conversion factor from input data to Rydbergs  
! kmat1       : K-matrix at E-dE
! kmat2       : K-matrix at E+dE
! adjust      : Factor involved in the grid spacing - allows the same
!               number of points per resonance to be calculated
! dele        : dE from the definition of the  Q-matrix
! delemax     : largest allowed value of dE
!DL13 neig    : neig is number of eigenvalues you want to print
! Input variables are described in the paper
!
      data ieunit/2/,gridinit/1.0D-1/,gridmin/1.0D-15/,delemax/1.0D-5/,
     1     crit/1.0D-0/,nes/20/,nthresh/0/,epsil/1.0D-02/,
     2     lutd/40/,lukmt/20/,adapt/.false./,savekmt/.true./,
     3     writbr/.false./,lubr/30/,eunit/'Ryd','eV','Eh'/,neig/3/,
     4     xthresh/0.1d0/,epsab/0.05/,epsbw/0.05/,maxn/10d0/,
     5     nptperres/500/,wthresh/0/
!
      namelist/time/einit,efinal,gridinit,gridmin,
     1     ieunit,delemax,nres,crit,nes,thresh,nthresh,epsil,
     2     nptperres,lutd,adapt,lukmt,savekmt,writbr,
     3     lubr,neig,xthresh,epsab,epsbw,maxn,wthresh

!     Read in namelist data:
      read (unit=5,nml=time,iostat=ios)
!     Convert energy units to Rydbergs:
      if (ieunit==2) conven=toev ! Units in eV
      if (ieunit==1) conven=1   ! Units in Rydberg
      if (ieunit==3) conven=0.5   ! Units in Hartrees 
      einit=einit/conven
      efinal=efinal/conven
!NV-03      grid=grid/conven  ! It's redundant and strictly wrong 
                              ! to be put in here
      nopen=zero
      energy=zero
      dele=zero
      gridmin=gridmin/conven
      gridinit=gridinit/conven
      delemax=delemax/conven
      thresh=thresh/conven
      epsab=epsab/conven
      epsbw=epsbw/conven
      lutd2=41
      if(neig.gt.numeig) then
        print*, 'Max value of neig is',numeig,', setting neig to',numeig
        neig=numeig
      endif
        open(2014,file='resonances')
        open(2015,file='branching_ratios')
        open(2016,file='resonance_X_ratios')
        open(2017,file='ground_state_widths')
! If using the adaptive grid with the user-defined K-matrix generator:
      if (adapt) then 

! Run subroutine to set up user K-matrix calculation and 
! find energy of thresholds
         call setupkmat(einit,efinal,maxthresh,etarget,ntarget)
! Set grid adjuster to give the correct number of points per resonance
         adjust=nptperres*gridinit/pi
         grid=gridinit
         dele=delemax
         tdhold=0
! Find the relevant thresholds within energy ranged asked for
! and put them into the array RANGE which denotes the
! energy borders. The first value
! is the inputted initial energy and the last is the inputted
! final energy:
       if(wthresh.ne.0) then
           if(wthresh.eq.1) then
            efinal=-2*(etarget(1)-etarget(2))
         endif
           if(wthresh.eq.2) then
             einit=-2*(etarget(1)-etarget(2))+0.01
             efinal=-2*(etarget(1)-etarget(4))
         endif
           if(wthresh.eq.3) then
             einit=-2*(etarget(1)-etarget(4))+0.01
           endif
         endif


         range(1)=einit ! we want the start of the first range to be einit
         nthresh=0
         do i=1,ntarget 
           if((-2*(etarget(1)-etarget(i)).gt.efinal)) exit ! check if threshold energy is larger than final 
           if((-2*(etarget(1)-etarget(i)).lt.einit)) cycle ! check if threshold is lower than einit (also rules out g/s) 
           if(abs(etarget(i)-etarget(i+1)).lt.1d-5) cycle ! check if threshold is degenerate
           nthresh=nthresh+1
           range(nthresh+1)=-2*(etarget(1)-etarget(i)) ! we want the n+1 to be the n threshold energy (first is einit)
         enddo
         write(6,*)'Calculation details:'
         write(6,*)'Initial energy:',einit*conven
         write(6,*)'Final energy:',efinal*conven
         write(6,*)'Maximum quantum defect n=',int(maxn)
         write(6,*)'No. eigenvalues considered for fitting=',neig
         write(6,*)'Available target energies:'
         do i=1,ntarget
            write(6,*)-2*(etarget(1)-etarget(i))*conven
         enddo
         if(nthresh.gt.0) then
         write(6,'(A22,I2,A29)') 'Calculation will cross ',nthresh,
     1       ' threshold(s) at energy(ies):'
         do i=1,nthresh
            write(6,*) i,': ',range(i+1)*toev,' eV'
         enddo
         else
           write(6,*) 'Calculation will not cross a threshold'
         endif
         range(nthresh+2)=efinal !final range value is efinal
!Range goes from 1 to nthresh+1 (to include efinal)

!DL         range(1)=einit-epsil
!DL         ithrecount=2
!NV         write (6,*)'Note that nthresh is',nthresh,
!NV     *              'and thresh are'thresh(1), thresh(2)
!DL        do i=1,nthresh
!DL            minarr=minloc(thresh(1:nthresh), mask = thresh .ne. 0.0D0)
!DL           if ((thresh(minarr(1))>einit).and.(thresh(minarr(1))<
!DL     1           efinal)) then
!DL               range(ithrecount)=thresh(minarr(1))
!DL               ithrecount=ithrecount+1
!DL            endif
!DL            thresh(minarr(1))=0.0D0
!DL         enddo
!DL       range(ithrecount)=efinal+epsil
!DL ITHRECOUNT now contains the number of borders (inclusive)
!DL
         ilast=0
! LOOP over ranges:
         do ithresh=1,nthresh+1

            if(ithresh.eq.1) then
            energy=range(ithresh) ! we want to start from einit
            else
            energy=range(ithresh)+epsab ! we want to start epsil away from the threshold energy
            endif

        do j=1,ntarget

           if(j.eq.ntarget) then
              if(energy.gt.-2*(etarget(1)-etarget(j))) 
     &                    print*, 'DEBUG: calc above final threshold'
              exit
              else if(-2*(etarget(1)-etarget(j)).lt.energy
     1          .and. -2*(etarget(1)-etarget(j+1)).gt.energy) then
                  exit

          endif

          enddo
            if(ithresh.ne.(nthresh+1)) then
                   efinal=-2*(etarget(1)-etarget(j+1)) - 1/maxn**2
            else if (j .ne. ntarget) then
            if( -2*(etarget(1)-etarget(j+1)) - 1/maxn**2 .lt.
     1          range(nthresh+2)) then
              print*,range(nthresh+2)
                print*,-2*(etarget(1)-etarget(j+1)) - 1/maxn**2 
                 efinal=-2*(etarget(1)-etarget(j+1))- 1/maxn**2
                else
                 efinal= range(nthresh+2)
                endif
            else 
                    print*,'here'
           write(6,*) 'efinal is set to the requested input value:', 
     & ' the program assumes there are no more thresholds below ',
     &  'this energy.' 
           write(6,*) 'If the results show unfittable resonant ', 
     & ' behaviour re-run the calculation with maxthresh increased, ', 
     &               ' to pick up the additional target states.' 
                efinal= range(nthresh+2)
            endif


          if(energy.gt.efinal) then 
          write(6,*) 'energy is greater than efinal'
          write(6,*) 're-run with higher maxn or lower einit'      
          write(6,*) 'energy:',energy*conven
          write(6,*) 'efinal:', efinal*conven
            write(6,*) 'skipping to next threshold if possible'
            cycle
          endif
!
            write(6,*) 'Energy range considered='
            write(6,*) energy*conven,'->',efinal*conven
!
            icount=0
          nopen2=0  !nopen2 is used to store previous nopen to check that it hasn't changed, this should not happen if range finding part is working correctly
! LOOP over energies within a range
            do while (energy<=efinal)
               icount=icount+1
               tdenergy(icount)=energy
! Calculate the K-matrices
             if(mod(icount,1000).eq.0) write(6,*) energy*conven
               call getkmat(maxnopen,energy-dele/2,kmat1,nopen)
               call getkmat(maxnopen,energy+dele/2,kmat2,nopen)
             if(icount.eq.1) then
             nopen2=nopen
             else
               if(nopen2.ne.nopen) then
              write(6,*) 'DEBUG MESSAGE: Number of channels has changed'
             endif
             nopen2=nopen
             endif

           if(icount.eq.1) then
           write(6,*) 'Number of open channels:', nopen
           endif
        

! Check to make sure that a threshold has not been crossed between
! these two points: i.e. nopen and nopen2 are the same
!DL13               if (nopen.ne.nopen2) then
!DL13                   write(6,*) 'Straddling threshold at energy=',
!DL13      1                 energy*conven
!DL13                   write(6,*) 'Point ignored'
!                  icount=icount-1
!                  exit
!               endif
! Save the calculated K-matrices if desired
               if (savekmt) then
                  write(lukmt,*) nopen,energy*conven,dele*conven
                  write(lukmt,*) kmat1(1:nopen,1:nopen)
                  write(lukmt,*) kmat2(1:nopen,1:nopen)
!NV-03                  write(lukmt,*) kmat2(1:nopen2,1:nopen2)
               endif
!
               call findtimedel(nopen,kmat1(1:nopen,1:nopen),
     1              kmat2(1:nopen,1:nopen),dele,tdhold,
!NV-03       1              kmat2(1:nopen2,1:nopen2),dele,tdvalue(icount),
     2              br(1:nopen))
            


             do i=0,nopen-1 ! reverses tdhold
               tmp(i+1) = tdhold(nopen-i)  
             enddo            
             tdhold(1:nopen)=tmp(1:nopen)
               do i=1,nopen
             write(1000,*) tdhold(1:nopen)
             enddo
               eigcount=1
             ucount=0 
               if(nopen.lt.numeig) then !don't go above numeig or nopen, stop at which ever is lower
             lstop=nopen
               else
               lstop=numeig
               endif
             if(nopen.lt.numeig) then
            adj=nopen
             else 
            adj=numeig
             endif
             if(icount.eq.1) then !store numeig or nopen eigenvalues in tdvalue depending on which one is lower, for case icount=1
               do i=icount,lstop !don't go above numeig or nopen, stop at which ever is lower
                         tdvalue(i)=tdhold(eigcount)
                 write(lutd+ucount,*) tdenergy(icount)*conven,tdvalue(i)
                  !write(1000,*) tdenergy(icount)*conven,tdvalue(i)
                  ucount=ucount+1
                  eigcount=eigcount+1
             enddo
             endif
             if(icount .ne. 1) then ! else store numeig or nopen eigenvalues in tdvalue depending on which one is lower
             do i=((icount-1)*adj+1),((icount-1)*adj+1)+lstop
                    tdvalue(i)=tdhold(eigcount)
                 write(lutd+ucount,*) tdenergy(icount)*conven,tdvalue(i)
                  !write(1000,*) tdenergy(icount)*conven,tdvalue(i)
                  ucount=ucount+1
                  eigcount=eigcount+1
                 enddo
               endif
!               write(6,*) tdenergy(icount)*conven,tdvalue(icount),grid
!
               if (energy>=efinal) then !brute force stop to energy going above efinal
               write(6,*) 'Energy is above efinal! energy=',
     1                     energy*conven,'efinal=',efinal*conven 
               write(6,*) 'Going to fitting...'
                  ilast=icount
                  exit
               endif             

               if (icount==maxpnt) then
                  write(6,*)'Reached maximum number of points,', maxpnt,
     1                 '. at energy ',energy*conven
                  write(6,*)'Will attempt to fit with the points already
     1                 calculated.'
                  exit
               endif
! Adjust grid spacing:
            ! if(nopen.le.3) then
           !  nptperres=30000
             !  else
              ! nptperres=500
             !  endif
             !  adjust=nptperres*gridinit/pi

             if(icount.eq.1.and.tdvalue(icount)>0) then
                  grid=max(gridmin,(min((gridinit/(adjust*
     1                 tdvalue(icount))),gridinit)))
             elseif(tdvalue((icount-1)*adj+1)>0) then
                  grid=max(gridmin,(min((gridinit/(adjust*
     1                 tdvalue((icount-1)*adj+1))),gridinit)))
               endif
! Adapt Delta E (dele) to the changing time-delay
               dele=min((grid/100),delemax)
               energy=energy+grid
            enddo               ! Over energy in this particular range

             if(neig.gt.nopen) then
             write(6,*) 'neig exceeds nopen,'
             write(6,'(7A,I2)') 'nopen =', nopen
             write(6,*) 'setting neig=nopen'
             neig=nopen
             endif

            do j=1,numeig
            do i=1,icount
            tmpa(i+(j-1)*icount)=tdvalue((i-1)*numeig+j)
            enddo
               enddo
         


      if(nopen.ge.numeig) then
         call eigsort(numeig,neig,icount,tdvalue,xthresh,adapt)  ! tracks resonances as they pass through multiple eigenvalues      
      else  
         call eigsort(nopen,neig,icount,tdvalue,xthresh,adapt)  ! tracks resonances as they pass through multiple eigenvalues      
      endif       
          do j=0,neig-1
                do i=1,icount
                  write(5000+j,*) tdenergy(i)*conven,tdvalue(i+j*icount)
               enddo
          enddo
        

            print*,'Starting fit for range:',ithresh

          do j=0,neig-1
                 write(6,*) 'Fitting resonance set number:',j+1
             call disconrm(icount,tdenergy(1:icount), !disconrm removes discontinuties that arise in eigsort
     1           tdvalue(1+j*icount:(j+1)*icount))
!               do i=1,icount
!                  write(6000+j,*) tdenergy(i)*conven,tdvalue(i+j*icount)
!               enddo                  
               call fitting(icount,tdenergy(1:icount),
     1           tdvalue(1+j*icount:(j+1)*icount),maxres,
     2           crit,conven,nes,maxnopen,adapt,eunit(ieunit),
     3             etarget,ntarget) 
             print*, 'Finished fit for range', ithresh
!            do i=1,icount
!            write(8787,*) tdenergy(i),tdvalue(i)
!            enddo
!              write(8787,*)
            enddo      
          
         enddo                  ! Over the different ranges
! 
! If reading in K-matrices:
      else 
         icount=0
         logindic=.true.
       io=0
         open (unit=20,status="unknown")
         do while (io .eq. 0)
            icount=icount+1
! Read in K-matrices each time
               read (lukmt,*,iostat=io) nopen,energy,dele
               read (lukmt,*)((kmat1(i,j),i=1,nopen),j=1,nopen)
               read (lukmt,*)((kmat2(i,j),i=1,nopen),j=1,nopen)
            energy=energy/conven
            dele=dele/conven
            tdenergy(icount)=energy

            call findtimedel(nopen,kmat1(1:nopen,1:nopen),
     1           kmat2(1:nopen,1:nopen),dele,tdvalue(icount),
     2           br(1:nopen))      
!
            write(lutd,*) tdenergy(icount)*conven,tdvalue(icount)
            write(6,*) tdenergy(icount)*conven,tdvalue(icount)
            if (writbr) then
               write(lubr,*) energy*conven,br(1:nopen)
            endif
!
            if (icount==maxpnt) then
               write(6,*)'Reached maximum number of points,', maxpnt,
     1              ', at energy ',energy*conven
               write(6,*)'Will attempt to fit with the points already
     1              calculated.'
               exit
            endif
         enddo
! 100     continue
      ntarget=0
      open(unit=1011,file="target")
      DO
      READ(1011, *, IOSTAT = IO )
      IF (IO < 0) EXIT
      ntarget = ntarget + 1
      END DO
      rewind(1011)

      do i=1,ntarget
      read(1011,*) etarget(i)
      enddo

      n=0
      open(unit=1010,status="unknown")
      DO
      READ(1010, *, IOSTAT = IO )
      IF (IO < 0) EXIT
      N = N + 1
      END DO
      rewind(1010)
     
      
      read(1010,*) neig 
      
      icount=(n-1)/neig

      do j=0,neig-1
      do i=1,icount
      read(1010,*) tdenergy(i),tdvalue(i+j*icount)
      enddo
      enddo

      tdenergy=tdenergy/conven


         call eigsort(numeig,neig,icount,tdvalue,xthresh,adapt)  ! tracks resonances as they pass through multiple eigenvalues      

          do j=0,neig-1
                do i=1,icount
                  write(5000+j,*) tdenergy(i)*conven,tdvalue(i+j*icount)
               enddo
          enddo
          do j=0,neig-1
                 write(6,*) 'Fitting resonance set number:',j+1
             call disconrm(icount,tdenergy(1:icount),
     1           tdvalue(1+j*icount:(j+1)*icount))
               call fitting(icount,tdenergy(1:icount),
     1           tdvalue(1+j*icount:(j+1)*icount),maxres,
     2           crit,conven,nes,maxnopen,adapt,eunit(ieunit),
     3             etarget,ntarget) 
      !       write(9999,*)
             print*, 'exiting fitting'
            enddo      

      endif
      end
!

      subroutine fitting(icount,tdenergy,tdvalue,maxres,crit,conven,
     1     nes,maxnopen,adapt,eunit,etarget,ntarget)
! Searches for maxima in the time-delay then sends the positions to FOUNDRES
      implicit none
      intent(in) icount,tdenergy,tdvalue,maxres,crit,conven,nes,
     1    maxnopen,adapt,eunit
      integer :: icount,nresmax,i,maxres,nes,maxnopen,j,ntarget
      double precision :: tdenergy(icount),tdvalue(icount),fd1,fd2,fd3
      double precision :: etarget(ntarget)
      logical :: logindic,adapt
      integer :: maxtd(maxres), k,flag
      double precision :: conven,crit,fd(4),efinal
      character(len=3) :: eunit
!      
! Error check
      if (icount<=5) then
         write(6,*)'Error: too few points available for fitting:',icount
         write(6,*)'Exiting...'
         return
      endif

      print*, 'entering fitting part'
      maxtd=0
 !       k=0


! Second part: fitting the time-delay: 
! So, time delay calculated on adaptive grid.
! Now find maximum positions but also check surrounding 
! first derivatives to be certain it's not just one rogue point:
      nresmax=0
      mainloop: do i=4,icount-3
         if (tdvalue(i).gt.tdvalue(i-1) .and.
     1       tdvalue(i).gt.tdvalue(i+1)) then
            logindic = .true.
            if (tdvalue(i).lt.100) logindic = .false. !cuts out noise in lower eigenvalues
            fd1 = (tdvalue(i) - tdvalue(i-1)) / 
     1            (tdenergy(i) - tdenergy(i-1))
            fd2 = (tdvalue(i+1) - tdvalue(i)) /
     1            (tdenergy(i+1) - tdenergy(i))
            fd3 = (tdvalue(i+2) - tdvalue(i+1)) /
     1            (tdenergy(i+2) - tdenergy(i+1))
            if (fd1*fd2.lt.0.and.fd3*fd2.lt.0) logindic = .false. ! ignores discontinuities from sorting
            do k = -100, 100
                if (tdvalue(i+k).gt.tdvalue(i) .or. 
     1             tdvalue(i-k).gt.tdvalue(i)) then ! checks not random peak
                  logindic=.false.
                  exit
               end if
            enddo
            if (logindic) then
               nresmax = nresmax + 1
               if (nresmax.eq.maxres) then 
                  print*, 'maxres reached,',
     1                    'calc may be too close to thresh'
                  exit mainloop
               end if
               maxtd(nresmax) = i
            end if
         end if
      end do mainloop
! nresmax holds the number of resonance maxima found. maxtd(i) holds the
! resonance positions



!!!!!!!!!!!!!NEED TO CHANGE MAX NO. OF POINTS



      if (nresmax.ne.0) then
         write(6,*)
         write(6,*)'Maxima found in this range'
!      print*, 'nmax',nresmax,'maxtd',maxtd,'npnt',icount
!      print*, 'tdvalue',tdvalue,'tdenergy',tdenergy
!      print*, 'crit',crit,'conven',conven
!      print*, 'nes',nes,'maxnopen',maxnopen
!      print*, 'adapt',adapt,'eunit',eunit
!      print*, 'maxtd(1:nresmax)',maxtd(1:nresmax)
         call foundres(nresmax,maxtd,icount,
     1        tdvalue,tdenergy,crit,conven,nes,
     2        maxnopen,adapt,eunit,
     3        etarget,ntarget)
      else
         write(6,*) 'No maxima found in this range'
         endif
      return
      end
!
      
      subroutine foundres(nmax,maxtd,icount,tdvalue,
     1     tdenergy,crit,conven,nes,maxnopen,adapt,eunit,
     2     etarget,ntarget)
      implicit none
! There are NMAX maxima (MAXTD(NMAX)).
! Taking the resonances in pairs, determines the number of
! resonances to be fitted together for each set: NRESTOG()
!      intent(in) nmax,maxtd,npnt,tdvalue,tdenergy,crit,conven,nes,
!     1     maxnopen,adapt,eunit
      integer :: nmax,maxnopen,nopen,kount,icount
      integer :: maxtd(nmax),tdmax(1),nes,npttofit,j,ifail,ntarget
      integer :: i,ncount,npos,init,ifinal,nrestog(nmax+1),k,l
      double precision :: tdvalue(icount),tdenergy(icount)
      double precision :: wid1,wid2,reson(2*nmax+1)
      double precision :: br(maxnopen),energy,dele
      double precision :: backgr,error,conven,crit,tdv(maxnopen)
      double precision :: kmat1(maxnopen,maxnopen)
      double precision :: kmat2(maxnopen,maxnopen)
      double precision :: fd1,fd2,fd3,fd4
      double precision :: etarget(ntarget),defect
      double precision :: fitdres(nmax),fitdwid(nmax),fitdbk(nmax)
      logical :: indicr(nmax),adapt,f1,f2
      double precision :: bigres(1),efinal
      character(len=3) :: eunit
      indicr=.false.
      nrestog=0
      reson=0
      kount=1
      nrestog(1)=1
      do i=1,(nmax-1)
         wid1=4/tdvalue(maxtd(i))
         wid2=4/tdvalue(maxtd(i+1))
         if ((wid1+wid2) > (crit*(tdenergy(maxtd(i+1))
     1        - tdenergy(maxtd(i))))) then
 !           indicr(i)=.true.
         endif
      enddo
! Nrestog determines how many
! resonances need to be fitted together
      ncount=0
      i=1
      do while (i.le.nmax)
         ncount=ncount+1
         nrestog(ncount)=1
         do while (indicr(i))
            nrestog(ncount)=nrestog(ncount)+1
            i=i+1
         enddo
         i=i+1
      enddo
      i=1
      npos=1
! Loop over the different set of resonances:
      do while (nrestog(i).ne.0)
       !  init=max(1,(maxtd(npos)-nes))
       !  ifinal=min(npnt,(maxtd(npos+nrestog(i)-1)+nes))
       !  npttofit=1+ifinal-init
        j=maxtd(npos)
        f1=.true.
        f2=.true.
        k=1
      do while(f1 .or. f2)
        if(j+k+2.eq.icount .and. f1) then
        ifinal=icount
        f1=.false.
        endif
        if(j-k-2.eq.1 .and. f2) then
        init=3
        f2=.false.
        endif
        fd1=(tdvalue(j+k+1)-tdvalue(j+k))/
     1  (tdenergy(j+k+1)-tdenergy(j+k))
        fd2=(tdvalue(j+k+2)-tdvalue(j+k+1))/
     1  (tdenergy(j+k+2)-tdenergy(j+k+1))
        fd3=(tdvalue(j-k)-tdvalue(j-k-1))/
     1  (tdenergy(j-k)-tdenergy(j-k-1))
        fd4=(tdvalue(j-k-1)-tdvalue(j-k-2))/
     1  (tdenergy(j-k-1)-tdenergy(j-k-2))
      if(fd1*fd2.lt.0 .and. f1) then
        f1=.false.
        ifinal=j+k
        endif
        if(fd3*fd4.lt.0 .and. f2) then
        f2=.false.
      init=j-k
        endif
        k=k+1
        enddo
         print*, init
         print*, ifinal
       print*,'exited do while'
       !init=j-20
      ! ifinal=j+20
       npttofit=1+ifinal-init
         write(6,*)
         write(6,200) i,nrestog(i)
!NV-03         write(6,*) 'Output directly from NAG routine (if any) follows:'
         tdmax(1) = maxtd(npos)-init  ! WARNING: other elements undefined when nrestog(i) > 1
         call fitlors(nrestog(i),npttofit,tdvalue(init:ifinal),
     1        tdenergy(init:ifinal),
     2        tdmax(1:nrestog(i)),
     3        reson((npos*2)-1:2*(npos+nrestog(i)-1)),backgr,error,
     4        ifail)
      if(reson(2*(npos-1)+1) .gt. tdenergy(ifinal)
     1  .or. reson(2*(npos-1)+1) .lt. tdenergy(init)) then
       write(6,*) 'Fitted resonance position outside of fitting limits'
       write(6,*) 'Position:', reson(2*(npos-1)+1)*conven
       write(6,*) 'Limits:',tdenergy(init)*conven,'->',
     1   tdenergy(ifinal)*conven
       write(6,*) 'Cycling...'
       npos=npos+nrestog(i)
         i=i+1
       cycle                     
      endif         !
!       bigres(1)=maxval(reson((npos*2)-1:2*(npos+nrestog(i)-1)))
!         error=sqrt(error)/(npttofit*bigres(1))
      print*, reson(2*(npos-1)+1)
      if (reson(2*(npos-1)+1) > (-2)*(etarget(1)-etarget(ntarget)))then
                write(6,*) 'Resonance position above highest'
               write(6,*) 'target state included in calculation.'
             write(6,*) 'Quantum defect cannot be calculated.'
               defect = 0
        else
        do k=1,ntarget - 1
            if(reson(2*(npos-1)+1).gt. -2*(etarget(1)-etarget(k))
     1  .and. reson(2*(npos-1)+1).lt. -2*(etarget(1)-etarget(k+1)))then
      defect=1/sqrt(-2*(etarget(1)-etarget(k+1))-reson(2*(npos-1)+1) )
            print*,'here'
                exit
             end if
          end do
      endif

      
         write(6,*)
       write(6,*) 'Fitting limits:',tdenergy(init)*conven,'->',
     1   tdenergy(ifinal)*conven
       write(6,211) 'Position (',eunit,')','Width (',eunit,')'  
! Loop over the different resonances in the set
! Note reson(odd) is energy position in Rydbergs
!  but reson(even) is time delay at resonance in atomic units
      !if(reson(2*(npos-1)+j).lt.efinal) then !cuts out printing of spurious resonances
         do j=1,(nrestog(i)*2-1),2
            write(6,210) i,1+j/2, reson(2*(npos-1)+j)*conven,
     1           conven*8/reson(2*(npos-1)+j+1)
        call stdev(npttofit,tdvalue(init:ifinal),tdenergy(init:ifinal),
     1       error,maxtd(npos)-init,reson(2*(npos-1)+j),
     2       8/reson(2*(npos-1)+j+1),backgr)   
         write(6,205) backgr,error
       write(6,230) defect
!     1           conven*4/reson(2*(npos-1)+j+1),eunit
!
! Find branching ratios at exact resonance positions
! if user-defined K-matrix routine available
            !adapt=.true.
            if (adapt) then
               energy=reson(2*(npos-1)+j)
               dele=(4/reson(2*(npos-1)+j+1))/1000
! Calculate the K-matrices
               call getkmat(maxnopen,energy-dele/2,kmat1,nopen)
               call getkmat(maxnopen,energy+dele/2,kmat2,nopen)  
               call findtimedel(nopen,kmat1(1:nopen,1:nopen),
     1              kmat2(1:nopen,1:nopen),dele,tdv,
     2              br(1:nopen))
               write(6,*) 'nopen=',nopen
               write (6,*) 'Branching ratios:',br(1:nopen)
           write(2014,'(2D20.13)') reson(2*(npos-1)+j)/2,
     1             4/reson(2*(npos-1)+j+1)
             write(2015,'(18D20.13)') br(1:nopen) ! format statement may need to be chagned if nopen exceeds 18
             write(2016,'(20D20.13)') reson(2*(npos-1)+j)/2, ! format statement may need to be chagned if nopen exceeds 18
     1             4/reson(2*(npos-1)+j+1),
     2        (br(l)*4/reson(2*(npos-1)+j+1), l=1,nopen)
             write(2017,'(5D20.13)') reson(2*(npos-1)+j)/2, !PRINTING HARTREES  
     1             4/reson(2*(npos-1)+j+1),
     2        (br(l)*4/reson(2*(npos-1)+j+1), l=1,3)
            endif
         enddo                  ! End loop over resonances
         npos=npos+nrestog(i)
         i=i+1
       ! endif
      enddo                     ! End loop over sets of resonances
 200  format('Attempting to fit resonance set ',i2,' as',i2,
     1     ' resonance(s)')
 205  format('Fitted with background=',D10.4,' and St. Dev.=',D10.4)
 210  format('Resonance:',i2,'.',i2,3X,D20.13,' ',
     1        D20.13)
 230  format('Quantum defect=',D20.13)
 211  format(19X,A10,A3,A1,7X,A7,A3,A1)
!      return
      end 
!
      subroutine fitlors(nresdum,npt,dum1,dum2,approx,reson,backgr,
     1     error,ifail)
! Fits a summation of NRESDUM Lorentzians to the timedelay to find the 
! resonance parameters, stored in RESON
      use gettimedel
      implicit none
      intent(in) nresdum,npt,dum1,dum2,approx
      intent(out) reson,backgr,error,ifail
      integer :: liw,ifail,i,lw,nresdum,npt
!NV-03 Attention, this must be revised. Temporaly I corrected the 
!      declaration for iw (it was a double precision array before!)
!      And as a result liw should be appropriate. It was set as 2 before!
      parameter (liw=2000)
      double precision :: error,backgr,fvec(npt),
     1     work(14*(nresdum+1)+4*(nresdum+1)**2 +4*npt*(nresdum+1)
     2     + 3*npt + (nresdum+1)*(2*(nresdum+1)-1))
      double precision :: dum1(npt),dum2(npt),reson(nresdum*2)
      double precision :: x(nresdum*2+1),tol
      integer :: approx(nresdum),info,iw(liw)
      external fcn
      print *,' fitlors '
      error=0.0D0
      ifail=0
      work=0.0D0
      x=0.0D0
      nrestofit=nresdum
! lw is the work space  required by NAG routine e04fdf
      lw=14*(nrestofit+1)+4*(nrestofit+1)**2 +4*npt*(nrestofit+1) + 
     1     3*npt + (nrestofit+1)*(2*(nrestofit+1)-1)
!
! The dummy variables are necessary as q, qen and nrestofit are 
! passed through the shared module gettimedel and therefore cannot
! be passed through the subroutine call.
      print*,'approx', approx
      q(1:npt)=dum1
      qen(1:npt)=dum2
      do i=1,nrestofit
         x(2*i-1)=qen(approx(i))
         x(2*i)=q(approx(i))
      enddo
!NV-03      ifail=-1
      tol=1.0D-08
!NV-03
!NV-03    
!NV-03      call e04fdf(npt,nrestofit*2+1,x,error,iw,liw,work,lw,ifail)
!NV-03
            print*,'npt',npt
                print*, nrestofit*2+1
                print*,'x'
                print*, x(1)*(1/0.073500d0)
                print*, x(2)
              call lmdif1 (fcn,npt,nrestofit*2+1,x,fvec,tol,info,iw,
     *              work,lw)            
      reson=x(1:nrestofit*2)
      backgr=x((nrestofit*2)+1)
!NV-03      if (ifail == 5) then
!NV-03         write(6,*) ' '
!NV-03         write(6,*) 'TIMEDEL Note: The above error message is due to'
!NV-03         write(6,*) 'over-sensitive error checking in the NAG routine.' 
!NV-03         write(6,*) 'It can safely be ignored.'
!NV-03      endif
        print *,'Fitlors done.INFO=',info
      return
      end
!
!NV-03      subroutine lsfun1(m,n,xc,fvecc)
!            
! Calculates the error between a guessed Lorentzian(s) form 
! and the actual time-delay. Used by lmdif1 routine.
!NV-03 Used by NAG routine E04FDF
!
      subroutine fcn(m,n,xc,fvecc,iflag)  
      use gettimedel
      implicit none
!NV-03      intent(in) m,n,xc
!NV-03      intent(out) fvecc
      integer :: m,n,i,j,iflag
      double precision :: xc(n),fvecc(m),td,en0,en,wid
!
      do i=1,m     
         td=0
         en=qen(i)/2
         do j=1,nrestofit
            en0=xc(2*j-1)/2
            wid=4/xc(2*j)
            td=xc(n)+td+wid/
     1           ((en-en0)**2+(wid**2)/4)
         enddo
         fvecc(i)=td-q(i)
      enddo
      return
      end
!
      subroutine findtimedel(nopen,kmat1,kmat2,dele,td,br)
! Given the two NOPENxNOPEN K-matrices (KMAT1 and KMAT2) an 
! energy DELE apart, convert the K-matrices into S-matrices (by calling 
! KTOSMAT) and then find the time-delay (td) and the corresponding 
! branching ratios (br) by calling TIMED.
      use precisn_gbl, only: wp
      implicit none
      intent(in) nopen,kmat1,kmat2,dele
      intent(out) td,br
      integer :: nopen
      double precision :: kmat1(nopen,nopen),kmat2(nopen,nopen)
      complex(wp) :: smat1(nopen,nopen),smat2(nopen,nopen)
      double precision :: dele,br(nopen)
      double precision :: td(nopen)
!
      call ktosmat(nopen,kmat1(1:nopen,1:nopen),  
     1     smat1(1:nopen,1:nopen))
      call ktosmat(nopen,kmat2(1:nopen,1:nopen), 
     1     smat2(1:nopen,1:nopen))
      call timed(nopen,dele,smat1(1:nopen,1:nopen),
     1     smat2(1:nopen,1:nopen),br,td)
      return
      end
!
      subroutine timed(nopen,dele,smat1,smat2,br,tdvalue)
! Given the NOPENxNOPEN S-matrices SMAT1 and SMAT2 an energy DELE
! apart, return the timedelay (TDVALUE) and branching ratios (BR)
      use precisn_gbl, only: wp
      implicit none
      intent(in) nopen,dele,smat1,smat2
      intent(out) br,tdvalue
      integer :: nopen,i,ifail
      complex(wp) :: smat1(nopen,nopen),smat2(nopen,nopen)
      double precision, allocatable :: rwork(:),eval(:)
      double precision :: dele, tdvalue(nopen),br(nopen)
      complex(wp), allocatable :: q(:,:),work(:)
      complex(wp), allocatable :: s(:,:),dSde(:,:)
      external zheev
      ifail=0
      allocate(q(nopen,nopen),
     *        work(nopen*nopen),s(nopen,nopen),dSde(nopen,nopen),
     *        rwork(3*nopen-2),eval(nopen))
!
! Find S from average of S1, S2
      S=(smat1+smat2)/2
!     dele is in Rydbergs, we want it in Hartrees, so appears as 2*dele
      dSde=cmplx(2/(1*dele),0)*(smat2-smat1)
      Q=(0,-1)*matmul(conjg(S),dSde)
! Q diagonals will be effectively real, but force it anyway:
      do i=1,nopen
         Q(i,i)=cmplx(real(Q(i,i)),0.0D0)
      enddo
! Extract eigenvectors/values from the Q matrix:
!NV-03      call f02haf('v','l',nopen,Q,nopen,eval,rwork,work,64*nopen,ifail)
! NAG routine returns eigenvectors in Q and eigenvalues
! in ascending  magnitude so final (nopen) eigenvalue is the one we want.
!            print*,'nopen',nopen
            call zheev('v','l',nopen,q,nopen,tdvalue,work,2*nopen-1,
     *                 rwork,ifail)
             if (ifail .ne. 0) write(6,*)'zheev timed! Ifail=', ifail
      br=abs(Q(:,nopen))**2
      deallocate(q,work,s,dSde,rwork,eval)
      return
      end   

      subroutine ktosmat(nopen,kmt,smat)
! Convert NOPENxNOPEN K-matrix (KMT) to S-matrix (SMAT)
      use precisn_gbl, only: wp
      implicit none
      integer, intent(in) :: nopen
      double precision, intent(in) :: kmt(nopen,nopen)
      complex(wp), intent(out) :: smat(nopen,nopen)
      integer :: i,ifail,info,AllocErr
      double precision :: wkspace(1000)
      complex(wp) ,allocatable :: Ident(:,:),A(:,:)
      integer,allocatable :: ipv(:)
      external :: zgesv

      AllocErr=0
! Set up identity matrix, IDENT:
        allocate (ident(nopen,nopen),a(nopen,nopen),ipv(nopen),
     *            STAT=AllocErr ) 
      if (allocerr.ne.0) then
       print *,'Allocation Error',AllocErr
      end if
      Ident(:,:)=(0.d0,0.d0)
      do i=1,nopen
         Ident(i,i)=(1,0)
      enddo
! Find the S-matrix by solving the equation:
! AS=B where A=(1-iK) and B=(1+iK),
! smat is input and output of zgsev
      A=Ident-(0,1)*kmt
      smat=Ident+(0,1)*kmt
!NV-03      B=Ident+(0,1)*kmt
        ifail=0
!NV-03      call f04adf(a,nopen,b,nopen,nopen,nopen,smat,nopen,wkspace,ifail)
        call zgesv(nopen,nopen,a,nopen,ipv,smat,nopen,ifail)
        if (ifail .ne. 0) write(6,*)'ZGESV! Ifail=', ifail
        deallocate ( ident,a,ipv)
        return
      end
!
!      subroutine setupkmat(einit,efinal)
! User-defined routine to initialize the calculation of K-matrices
!      use userdefkmat
!      intent(in) einit,efinal
!      double precision :: einit,efinal
!      return
!      end
!
!      subroutine getkmat(maxnopen,energy,kmt,nopen)
! User-defined routine to calculate a K-matrix at a given input energy. 
!      use userdefkmat
!      intent(in) maxnopen,energy
!     intent(out) kmt,nopen
!
!      integer :: maxnopen,nopen
!      double precision :: kmt(maxnopen,maxnopen)
!      double precision :: energy
!
!      return
!      end

!Subroutine eigsort tracks resonances as the pass through multiple
!eigenvalues by finding avoiding crossings. Avoided crossing are 
!detected as a minima in the difference between two eigenvalues
!with a ratio between the two timedel values gt than xthresh.

 
      subroutine eigsort(numeig,neig,icount,td,xthresh,adapt)
      integer :: numeig, neig, icount
      double precision :: td(numeig*icount), xthresh
      integer :: i,j,k,eigc,abv,blw,heig(neig),tmp, kl_m, kl_p
      double precision :: tmpa(numeig*icount),dif
      logical :: logindic,adapt
      integer :: klim_m, klim_p

!rearrange array so that it is in order eig1: td(1:icount),
!eig2: td(icount+1:2*icount) etc.

      klim_p = 10
      klim_m = 10 

      conven=1/0.0735
       print*, 'numeig=',numeig
      print*,adapt
      print*, icount

      if(adapt) then
      do j=1,numeig
      do i=1,icount
            tmpa(i+(j-1)*icount)=td((i-1)*numeig+j)
      enddo
      enddo
      td=tmpa
      endif
      tmpa=0d0
!array heig holds the order in which the eigenvalues
!are being recorded      
      
!initialise the array heig with default order (0,..,neig-1)
      do i=1,neig
      heig(i)=i-1
      enddo
!initialise tmpa array
      do i=1,2
        do eigc=0,neig-1
          tmpa(i+eigc*icount)=td(i+heig(eigc+1)*icount)
        enddo
      enddo

      mainloop : do i=3,icount-3 !loops over energy values
        kl_p = MIN(icount - i, klim_p)
        kl_m = MIN(i-1, klim_m)

      
        do eigc=0,neig-2 !loops over eigenvalues

          abv=i+eigc*icount
          blw=i+(eigc+1)*icount

        if(dif(td(abv),td(blw)).lt.dif(td(abv-1),td(blw-1)) .and. 
     1    dif(td(abv),td(blw)).lt.dif(td(abv+1),td(blw+1))) then !looks for minima with coarse search, may need to be improved for noisy data
             logindic=.true.
             if (logindic) then
                 do k = 2, kl_p
              if(dif(td(abv),td(blw)).gt.dif(td(abv+k),td(blw+k)))then
            logindic=.false.
                 exit
                endif
                enddo
             end if
             if (logindic) then
              do k = 2, kl_m
                if(dif(td(abv),td(blw)).gt.dif(td(abv-k),td(blw-k)))then
            logindic=.false.
                exit
                endif
               enddo
              end if
               ! if(td(blw)/td(abv).gt. xthresh) logindic=.true. !if ratio is over xthresh threshold then switch, default value should work for most cases (0.8)
            if(logindic) then
            do k=1,neig-1 !finds position of higher energy eigenvalue in heig
                  if(heig(k).eq.eigc) exit
            enddo
            do j=1,neig-1 !finds position of lower energy eigenvalue in heig
                  if(heig(j).eq.(eigc+1)) exit
            enddo
            !switches eigenvalues
            tmp=heig(k)
            heig(k)=heig(j)
            heig(j)=tmp
              endif
                logindic=.false.
         endif
      
      tmpa(i+eigc*icount)=td(i+heig(eigc+1)*icount) !tmpa holds sorted time delays
 !       write(5000+eigc,*) tdenergy(i)*conven,td(i+heig(eigc+1)*icount)

        enddo !eigc on exiting loop is final value +1

        eigc = neig - 1
      tmpa(i+eigc*icount)=td(i+heig(eigc+1)*icount)
   !     write(5000+eigc,*) tdenergy(i)*conven,td(i+heig(eigc+1)*icount)
      
      !write(60,*) tdenergy(i)*conven,heig(1)
      !write(61,*) tdenergy(i)*conven,heig(2)
      !write(62,*) tdenergy(i)*conven,heig(3)

      enddo mainloop
!finalise array
      do i=icount-2,icount
        do eigc=0,neig-1
          tmpa(i+eigc*icount)=td(i+heig(eigc+1)*icount)
        enddo
      enddo

      td=tmpa !subroutine returns sorted timedelays in original array
      return 
      end

      !disconrm finds discontinuites and replaces them with a
      !linear interpolant
      subroutine disconrm(icount,tden,td)
      integer :: icount
      double precision :: td(icount),tden(icount)
      
      integer :: i,j,k,i_v, j_m, j_p, flag
      double precision :: fd1,fd2,fd3,num,denom
      logical :: l1,l2
      flag=0 
      i_v = 1



      print*, icount

      mainloop : do i=2,icount-3
      
      if(flag.eq.1) then
        if (i <= i_v) cycle mainloop
         flag=0
      endif

     

      fd1=(td(i)-td(i-1))/(tden(i)-tden(i-1))
      fd2=(td(i+1)-td(i))/(tden(i+1)-tden(i))
      fd3=(td(i+2)-td(i+1))/(tden(i+2)-tden(i+1))
      l1=.false.
      l2=.false.
      if(fd1*fd2.lt.0 .and. fd3*fd2.lt.0) then !finds discontinuity by looking at change in sign of gradient
      j=1
      do 
      if(i-j-1 .eq. 1 .or. i+j+1 .eq.icount)  exit
        if(td(i+j).gt.td(i+j-1) .and. td(i+j).gt.td(i+j+1)) l1=.true.
      if(td(i-j).gt.td(i-j-1) .and. td(i-j).gt.td(i-j+1)) l2=.true.
      if(l1 .and. l2) then
      l1=.false.
      l2=.false.
        exit
      endif
        if(abs(td(i+j)-td(i-j)).lt.abs(td(i+j+1)-td(i-j-1))) exit !finds point at which linear interpolation should be performed between, 
        if((td(i+j)-td(i-j))*(td(i+j+1)-td(i-j-1)).lt.0) exit
        j=j+1
      enddo
      j=j+2

      j_m = MIN(j,(i-1))
      j_p = MIN(j,(icount-i))

      do k=-j_m, j_p
          num = tden(i+k) - tden(i-j_m)
          denom = tden(i+j_p) - tden(i-j_m) 
          td(i+k) = td(i-j_m) + (td(i+j_p) - td(i-j_m)) * (num/denom)
       enddo
! the energy values i+1, ..., i+j_p can be skipped as already linearized.
       i_v = i + j_p   
       flag=1
      endif    

      enddo mainloop      

      do i=1,icount
      write(9999,*) tden(i),td(i)      
      enddo

      return
      end

      function dif(a,b)
      double precision a,b,dif
      dif=a-b
      return
      end      

      subroutine stdev(icount,td,tdenergy,error,rpos,fpos,fwid,fbg)
      integer :: icount,rpos,i
      double precision :: td(icount),tdenergy(icount)
      double precision :: error,fpos,fwid,fbg,fpostmp,fwidtmp
      
      double precision :: sse,qe,st,tdenergytmp(icount)
      fpostmp=fpos/2
        fwidtmp=fwid/2
      tdenergytmp=tdenergy/2
      sse=0d0
      do i=1,icount
      if(tdenergytmp(i).lt.(fpostmp-fwidtmp)) cycle
        qe=fbg + fwidtmp/( ((tdenergytmp(i)-fpostmp)**2)
     1   + (fwidtmp/2)**2 ) 
      sse=sse+(td(i)-qe)**2
        if(tdenergytmp(i).ge.(fpostmp+fwidtmp)) exit
      enddo
      error=(sqrt(sse))/((i-1)*(4/fwidtmp))
      return
      end
      
      


