! Copyright 2019
!
! For a comprehensive list of the developers that contributed to these codes
! see the UK-AMOR website.
!
! This file is part of UKRmol-out (UKRmol+ suite).
!
!     UKRmol-out is free software: you can redistribute it and/or modify
!     it under the terms of the GNU General Public License as published by
!     the Free Software Foundation, either version 3 of the License, or
!     (at your option) any later version.
!
!     UKRmol-out is distributed in the hope that it will be useful,
!     but WITHOUT ANY WARRANTY; without even the implied warranty of
!     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!     GNU General Public License for more details.
!
!     You should have received a copy of the GNU General Public License
!     along with  UKRmol-out (in source/COPYING). Alternatively, you can also visit
!     <https://www.gnu.org/licenses/>.
!
      PROGRAM QB
c
c    Resonance analysis based on the QB method of
c    Quigley and Berrington (1996) J.Phys.B 29, 4529.
c    Department of Applied Maths, Queen's University, Belfast BT7 1NN
c    e-mail: k.berrington@qub.ac.uk
c    Coulomb functions from STGF and STGB, Seaton (1982) CPC 25,87.
c
c    Input files ...
c
c    H.DAT (unit 10, unformatted) 
c    - defined by Berrington, Eissner, Norrington (1995) CPC 92, 290.
c
c    QB.INP (unit 5, formatted) 
c    - standard input; here is a sample:
c-------------------------------------------------------------------
c    0, 0, 0                                    :IPRINT, IRAD, IPERT
c    1.E-6                                      :AC
c    1.                                         :RONE
c    1                                          :IOPT
c    0,2,1                                      :S,L,P or 0,2J,P
c    0.0, 10.0, 0.005                           :XMIN, XMAX, BDX
c-------------------------------------------------------------------
C     IPRINT= 0 normally
C     IRAD  = 0 normally
C     IPERT = 0 FOR NO PERTURBATION
C     IPERT = 1 FOR FIRST-ORDER PERTURBATION
C     AC    = ACCURACY REQUIRED FOR COULOMB FUNCTIONS
C     RONE  = 1. normally
C     IOPT  = number of energetically open states
c             (ie. resonance searching starts above state IOPT)
C     IS, IL, IP = 2S+1, L, parity (0=even,1=odd) (LS coupling)
C     IS, IL, IP = 0, 2J, parity (intermediate coupling)
C     XMIN, XMAX, BDX  = minimum, maximum, increment in effective n 
C          Set XMIN=0.0 to start scan at threshold of state IOPT.
c          (note: for neutral targets (Z=N) the program sets Z-N=1,
c           but will STOP anyway in subroutine ASY unless a suitable
c           asymptotic package is attached, eg Crees (1981) CPC 23, 181.)
c
c   Output files ...
c
c     QB.OUT (unit 6, formatted) 
c     - standard output with channel info, resonances, partial widths etc.
c
c     C (unit 17, formatted)
c     - summary of continuum states: one line per resonance containing
c       identification, energy, effective n, width (useful for archiving).
c
c     T (unit 18, formatted)
c     - summary of target core states: one line per state as in OP.
c
c     P (unit 14, formatted)
c     - energy, effective n, eigenphase sum, eigenphase derivative 
c       (useful for plotting via gnuplot etc).
c
c   Dimension parameters:
c     MZCHF = max number of channels
c     MZEST = max number of resonances to analyse
c     MZLMX = max number of multipoles in long range potential
c     MZLP1 = max channel angular momentum
c     MZMNP = max order of Hamiltonian matrix
c     MZPTS = max number of points in Coulomb functions
c     MZTAR = max number of target states
c     MZNRG = max terms in Buttle fit (eg. > NRANG2)
c     MZTET = max coefficients for theta.
C
      IMPLICIT REAL*8(A-H,O-Z)
C
      PARAMETER (MZCHF= 395, MZEST= 200, MZLMX=   4, MZLP1=  40,
     1           MZMNP=5700, MZPTS= 600, MZSLP=  80, MZTAR= 200,
     2           MZNRG=  40, MZTET=  50, BWARN=0.01)
      COMMON/CEN/MXE,NWT,NZ,ETOT
      COMMON/CACC/AC,ACNUM,ACJWBK,ACZP,LACC
      COMMON/CPOINT/KP0,KP1,KP2,RZERO,RONE,RTWO,H
      COMMON/CINPUT/
     1 LAMAX,LRANG2,LRGL2,MNP2,NAST,NCHAN,NELC,NPTY2,NSPN2,NZED,MORE2,
     2 ISAT(MZTAR),LAT(MZTAR),L2P(MZCHF),NCONAT(MZTAR),
     3 BSTO,RA,
     4 CF(MZCHF,MZCHF,MZLMX),COEFF(3,MZLP1),ENAT(MZTAR),
     5 WMAT(MZCHF,MZMNP),VALUE(MZMNP)
      COMMON/CHAN/ECH(MZCHF),LLCH(MZCHF),EPS(MZCHF),FKNU(MZCHF),
     1 CC(MZCHF),RINF(MZCHF),ITARG(MZCHF),NCHF,NCHOP,NCHOP1
      COMMON/CVECT/AAA(MZCHF,MZCHF),BBB(MZCHF,MZCHF),CCC(MZCHF,MZCHF),
     1 P(MZCHF,MZCHF),Q(MZCHF,MZCHF),XVECT(MZCHF)
      COMMON/CBODE/WBODE(MZPTS)
      COMMON/COULSC/FS(MZCHF,MZPTS),FSP(MZCHF),FC(MZCHF,MZPTS),
     1 FCP(MZCHF)
      COMMON/CPOT/LAMP(MZCHF,MZCHF),BW(MZCHF,MZCHF)
      COMMON/CALP/ASS(MZCHF,MZCHF),ASC(MZCHF,MZCHF),ACS(MZCHF,MZCHF),
     1 ACC(MZCHF,MZCHF)
      COMMON/CNTRL/IPRINT,IRAD,IPERT,KP2X
      COMMON/CDEGEN/NASTD,NASTR,NLEV(MZTAR),NCNATR(MZTAR),ENATR(MZTAR)
C
      LOGICAL EX,EX5
      DIMENSION XMINST(MZEST),XMAXST(MZEST),BDXST(MZEST)
      INTEGER MSLP(MZSLP),IPAT(MZTAR)
      DATA IPAT/MZTAR*9/, EV/13.605/
      CALL BLOCK
C
C  WRITE HEADER ON UNIT 6 = QB.OUT
C
      OPEN(6,FILE='QB.OUT',STATUS='UNKNOWN') 
      WRITE(6,6001)
      WRITE(6,6002) MZCHF,MZTAR,MZLMX,MZLP1,MZPTS,MZEST,MZMNP,MZSLP
     *,MZNRG,MZTET
C
C  READ DATA FROM UNIT 5 = QB.INP
C
      CALL READ5(IPRINT,IRAD,IPERT,AC,RONE,IOPT,MSLP,MZSLP,KSLP,
     *           XMINST,XMAXST,BDXST,MZEST,EX5)
      CALL ACSUB
C
C  READ DATA FROM UNIT 10 = H.DAT
C
      INQUIRE (FILE='H.DAT',EXIST=EX)
      IF (EX) THEN
        IF (.NOT.EX5) STOP
        OPEN(10,FILE='H.DAT',STATUS='OLD',FORM='UNFORMATTED')
      ELSE
        print *,'File H.DAT does not exist ... using CPC test data.'
c       call cpcrun
      ENDIF
C
C  READ R-MATRIX DATA INDEPENDENT OF SLPI
C
      if (EX) CALL READ1
      WRITE(6,6003) NZED,NELC,NAST,RA,BSTO
c
c  Warning for increment (different for LS and f-s resonances)
c
      IF(BDXST(1).LT.BWARN/10.OR.(ISAT(1).NE.0.AND.BDXST(1).LT.BWARN/2))
     *  print *,'Warning *** BDX increment may be too small'
      IF(BDXST(1).GT.BWARN.OR.(ISAT(1).EQ.0.AND.BDXST(1).GT.BWARN/2))
     *  print *,'Warning *** BDX increment may be too big'
C
C  SCALE TARGET DATA
C
      CALL SCALE1
      ZSQ=MAX(NZED-NELC,1)**2
C
C  LOOP OVER SLPI CASES, READ R-MATRIX FILE FOR EACH CASE
C
      KASE=1
      DO kount=1,MZSLP*10
        if (EX) CALL READ2
        WRITE(6,6004) NSPN2,LRGL2,NPTY2,NCHAN,MNP2
c
c  Define parity (IPAT) of target states for the T file
c
        i=0
        DO n=1,NAST
          i=i+NCONAT(n)
          IPAT(n)=MOD(L2P(i)+NPTY2,2)
        END DO
c
C  CHECK WHETHER THIS CASE REQUIRED
c
        ISLP=10000*NSPN2+100*LRGL2+NPTY2
        KK=0
        DO K=KASE,KSLP
          IF(ISLP.EQ.MSLP(K)) KK=K
        END DO
        IF(KK.NE.0) THEN
C
C  SLPI CASE FOUND. SCALE DATA, AND WRITE TARGET AND CHANNEL DATA
C
          CALL SCALE2
          WRITE(6,6005)
          I=0
          DO J=1,NAST
            ISL=100*ISAT(J)+10*LAT(J)+IPAT(J)
            tev=EV*ZSQ*ENAT(J)
            WRITE(6,6006)J,ISL,tev,ENAT(J),
     *                  (L,LLCH(L),L=I+1,I+NCONAT(J))
            I=I+NCONAT(J)
            IF (J.EQ.IOPT) WRITE(6,*)'                    ',
     *                  'here be resonances:',I,' open channels'
          END DO
C
C  SET XMIN TO THRESHOLD IF ZERO, AND DEFINE ECH1=FIRST CLOSED CHANNEL
C
          XMIN=XMINST(KK)
          ECH1=ENAT(1+IOPT)-ENAT(IOPT)
          XTEST=XMAXST(KK)
          IF (ECH1.GT.0.0) XTEST=1.0/SQRT(ECH1)
          IF (XMIN.EQ.0.0) XMIN=XTEST
          XMAX=XMAXST(KK)
          BDX=BDXST(KK)
          ECH1=ENAT(1+IOPT)
C
C  RE-ORDER STORED UNIT-5 DATA
C
          IF(KK.NE.KASE)THEN
            MSLP(KK)=MSLP(KASE)
            XMINST(KK)=XMINST(KASE)
            XMAXST(KK)=XMAXST(KASE)
            BDXST(KK)=BDXST(KASE)
          ENDIF
C
C  SCAN FOR RESONANCES
C
          CALL SCAN(XMIN,XMAX,BDX,ECH1)
C
C  SCAN FOR THIS CASE COMPLETE, CHECK WHETHER MORE DATA ON R-MATRIX FILE
C
          IF(KASE.EQ.KSLP)GOTO 3000
          KASE=KASE+1
          WRITE(6,*)
        ENDIF
        IF(MORE2.EQ.0)THEN
          WRITE(6,6007)
          DO I=KASE,KSLP
            WRITE(6,6008)MSLP(I)
          END DO
          WRITE(6,6009)
          GOTO 3000
        ENDIF
      END DO
C
C  TERMINATION
C  ***********
 3000 CONTINUE
C
C  Write T target file
C
      OPEN(18,FILE='T',FORM='FORMATTED',STATUS='UNKNOWN')
      WRITE(18,1001) NZED,NELC,'T'
      WRITE(18,1001) NAST
      IDNST1=0
      DO m=1,NAST
        WRITE(18,1013) ISAT(m),LAT(m),IPAT(m),ZSQ*ENAT(m),m
      END DO
      WRITE(18,1001) IDNST1
      CLOSE(18)
      CLOSE(17)
      WRITE(6,6010)
      STOP
C
C  FORMATS
C  *******
 1001 format(2i5,4x,a1)
 1013 format(3i5,1pe13.5,3x,'T',i3)
 6001 FORMAT(20X,'PROGRAM QB, VERSION 3, 1.1.98'/
     +       20X,'*****************************'/
     +       'QB method for resonance analysis by',
     +        ' Quigley & Berrington (1996)J.Phys.B29,4529')
 6002 FORMAT(' COMPILED FOR DIMENSIONS -'/
     + 15X,'CHANNELS                     MZ','CHF =',i6/
     + 15X,'TARGET STATES                MZ','TAR =',i6/
     + 15X,'MULTIPOLES                   MZ','LMX =',i6/
     + 15X,'SMALL L VALUES               MZ','LP1 =',i6/
     + 15X,'OUTER-REGION RADIAL POINTS   MZ','PTS =',i6/
     + 15X,'NUMBER BOUND-STATE ENERGIES  MZ','EST =',i6/
     + 15X,'R-MATRIX POLES               MZ','MNP =',i6/
     + 15X,'S, L, PI CASES               MZ','SLP =',i6/
     + 15X,'COEFFICIENTS FOR THETA       MZ','TET =',i6/
     + 15X,'TERMS IN BUTTLE FIT          MZ','NRG =',i6/)
 6003 FORMAT(/' NUCLEAR Z =',I3,'  TARGET',
     1   ' ELECTRONS =',I3,'  STATES =',I5,'  RA =',F9.4,' B=',F5.1)
 6004 FORMAT('2S+1 = ',I2,'  L =',I4,'  Parity=',I1,3X,
     *'CHANNELS =',I6,'  H-MATRIX =',I7)
 6005 FORMAT(28('-')/
     *'Target SLP     eV    Rydscaled  Channel: l ...')
 6006 FORMAT(I6,I4,1PE11.3,0PF9.5,
     *(I9,':',I2,I5,':',I2,I5,':',I2,I5,':',I2,I5,':',I2,I5,':',I2))
 6007 FORMAT(/10X,30('*')/
     1  10X,'NO DATA ON R-MATRIX FILE FOR'/
     2  10X,'10000*IS+100*IL+IP = '/)
 6008 FORMAT(10X,I10)
 6009 FORMAT(/10X,30(1H*)/)
 6010 FORMAT(40X,'END OF QB')
C
      END
c***********************************************************************
      SUBROUTINE ABG(E,L,AC,A,BG)
C
C  COMPUTES FUNCTION G(X,L) TO ACCURACY AC.
C
      IMPLICIT REAL*8 (A-H,O-Z)
      PARAMETER (MZLP1=40)
      DIMENSION temp(MZLP1)
C
      X=1./SQRT(-E)
C
C  CALCULATION OF A AND EAC=E*A*C
c     IF(L.GT.0)GOTO 2
      IF(L.LE.0) THEN
        A=1.
        EAC=0.
c     GOTO 20
c   2      IF(L.GT.1)GOTO 4
      ELSE IF(L.EQ.1) THEN
        A=1.+E
        EAC=E
c     GOTO 20
c   4      IF(X.LT.FLOAT(L+1)) GOTO 12
      ELSE IF(X.GE.FLOAT(L+1)) THEN
        C=0.
        A=1.
        A1=1
        A2=-E
        A3=2.*E
        DO I=1,L
          A2=A2+A3
          A1=A1+A2
          A=A*A1
          C=C+I/A1
        END DO
      EAC=E*A*C
c     GOTO 20
C  CASE OF X.LT.(L+1)
      ELSE
        A=1.
        A1=0.
c     DO 16  I=1,L
c     A=A*(1.+FLOAT(I*I)*E)
c     A2=I
c     DO 14 J=1,L
c     IF(J.EQ.I) GOTO 14
c     A2=A2*(1.+FLOAT(J*J)*E)
c  14 CONTINUE
c  16 A1=A1+A2
        do i=1,L
          temp(i)=1.+FLOAT(I*I)*E
        end do
        DO I=1,L
          A=A*temp(I)
          A2=I
          DO J=1,L
            IF(J.ne.I) A2=A2*temp(J)
          END DO
          A1=A1+A2
        END DO
        EAC=E*A1
C
C  COMPUTE A1=PI*BG/A-E*C=1/(2*X)+PSI(X)-LN(X)
      ENDIF
   20 A1=0.
C  TEST CONVERGENCE OF ASYMPTOTIC EXPANSION
      XN=(754.*AC)**(-.125)
c     IF(X.GT.XN)GOTO 40
      IF(X.LE.XN) THEN
C  USE RECURRENCE FORMULAE
        N=XN-X+1
        XN=X+N
        E=-1./(XN*XN)
        A1=A1-.5*(1./X+1./XN)+LOG(XN/X)
c       IF(N.LT.2)GOTO 40
        IF(N.GE.2) THEN
          N=N-1
          DO I=1,N
            A1=A1-1./(I+X)
          END DO
        ENDIF
      ENDIF
C  USE ASYMPTOTIC EXPANSION
   40 A1=A1+(((1.05*E+1.)*E+2.1)*E+21)*E*.003968253968
C
C  COMPLETE CALCULATION
      BG=(A*A1+EAC)*.318309886
      RETURN
      END      
c***********************************************************************
      SUBROUTINE ACSUB
C
C      $Id: acsub.f,v 1.1 1994/03/18 17:43:47 ferrous Exp $
C
C      $Log: acsub.f,v $
c Revision 1.1  1994/03/18  17:43:47  ferrous
c Initial revision
c
C
      IMPLICIT REAL*8(A-H,O-Z)
      COMMON/CACC/AC,ACNUM,ACJWBK,ACZP,LACC
C
      ACNUM=(24.*AC)**.1666666667
      ACJWBK=(6.*AC)**.2
      ACZP=16.*AC
      LACC=0
      IF(AC.LT.1.E-3)LACC=2
      IF(AC.LT.1.E-4)LACC=4
C
      RETURN
      END
c***********************************************************************
      SUBROUTINE ALPHA
C
C  CALCULATES ALPHA INTEGRALS
C
      IMPLICIT REAL*8(A-H,O-Y)
      IMPLICIT COMPLEX(Z)
C
      PARAMETER (MZCHF= 395,MZPTS= 600) 
      COMMON/CALP/ASS(MZCHF,MZCHF),ASC(MZCHF,MZCHF),ACS(MZCHF,MZCHF),
     1 ACC(MZCHF,MZCHF)
      COMMON/COULSC/FS(MZCHF,MZPTS),FSP(MZCHF),FC(MZCHF,MZPTS),
     1 FCP(MZCHF)
      COMMON/CPOINT/KP0,KP1,KP2,RZERO,RONE,RTWO,H
      COMMON/CPOT/LAMP(MZCHF,MZCHF),BW(MZCHF,MZCHF)
      COMMON/CBODE/WBODE(MZPTS)
      COMMON/CHAN/ECH(MZCHF),LLCH(MZCHF),EPS(MZCHF),FKNU(MZCHF),
     1 CC(MZCHF),RINF(MZCHF),ITARG(MZCHF),NCHF,NCHOP,NCHOP1      
      COMMON/CACC/AC,ACNUM,ACJWBK,ACZP,LACC
      COMMON/CNTRL/IPRINT,IRAD,IPERT,KP2X
C
C
C  INITIALISE ALPHA TO ZERO
      DO 10 I=1,NCHF
      DO 10 J=I,NCHF
      ASS(I,J)=0.
      ASC(I,J)=0.
      ACS(I,J)=0.
   10 ACC(I,J)=0.
C
C
C  CONTRIBUTION FROM RZERO TO RTWO
      R=RZERO-H
C  START LOOP ON RADIAL POINTS
      IF(KP2.EQ.1)GOTO 105
      DO 100 K=1,KP2
      W=WBODE(K)
      R=R+H
      X=1./R
C  OPEN-OPEN AND OPEN-CLOSED PARTS
      IF(NCHOP.EQ.0)GOTO 80
      DO 50 I=1,NCHOP
      DO 50 J=I,NCHF
      LIJ=LAMP(I,J)
      IF(LIJ.EQ.1)GOTO 50
      WIJ=W*BW(I,J)*X**LIJ
      ASS(I,J)=ASS(I,J)+FS(I,K)*WIJ*FS(J,K)
      ASC(I,J)=ASC(I,J)+FS(I,K)*WIJ*FC(J,K)
      ACS(I,J)=ACS(I,J)+FC(I,K)*WIJ*FS(J,K)
      ACC(I,J)=ACC(I,J)+FC(I,K)*WIJ*FC(J,K)
   50 CONTINUE
      IF(NCHOP.EQ.NCHF)GOTO 100
C  CLOSED-CLOSED PART
   80 DO 90 I=NCHOP1,NCHF
      DO 90 J=I,NCHF
      LIJ=LAMP(I,J)
      IF(LIJ.EQ.1)GOTO 90
      WIJ=W*BW(I,J)*X**LIJ
      ASS(I,J)=ASS(I,J)+FS(I,K)*WIJ*FS(J,K)
      ACS(I,J)=ACS(I,J)+FC(I,K)*WIJ*FS(J,K)
      ASC(I,J)=ASC(I,J)+FS(I,K)*WIJ*FC(J,K)
   90 CONTINUE
  100 CONTINUE
  105 CONTINUE
C
C
C
C  ASYMPTOTIC INTEGRALS FOR R=RTWO TO INFINITY
      IF(IPRINT.GT.1)WRITE(6,750)
  750 FORMAT(/' I,J AND ASS(I,J), ACS(I,J), ASC(I,J), ACC(I,J)'/)
C  - OPEN-OPEN PART
      IF(NCHOP.EQ.0)GOTO 180
      DO 150 I=1,NCHOP
      DO 150 J=I,NCHOP
      LIJ=LAMP(I,J)
      IF(LIJ.EQ.1) GO TO 150
      NLAG=2*LIJ+LACC
      NLEG=NLAG
      BIJ=BW(I,J)
C  P INTEGRALS
  110 ZP3=ZPLAG(I,J,NLAG,LIJ)*BIJ
  704     FORMAT(70X,2I3,' P=',2E12.4)
C  Q INTEGRALS
      ALP=RTWO*(FKNU(I)-FKNU(J))
      IF(ALP.LT.2.)GOTO 141
  120 ZQ3=ZQLAG(I,J,NLAG,LIJ)*BIJ
      GO TO 145
  141 ZQ3=ZQLEG(I,J,NLEG,LIJ)*BIJ
  705     FORMAT(70X,2I3,' Q=',2E12.4)
  145 ASS(I,J)=ASS(I,J)+.5*REAL(ZQ3-ZP3)
      ASC(I,J)=ASC(I,J)+.5*AIMAG(ZP3+ZQ3)
      ACS(I,J)=ACS(I,J)+.5*AIMAG(ZP3-ZQ3)
      ACC(I,J)=ACC(I,J)+.5*REAL(ZP3+ZQ3)
      IF(IPRINT.GT.1)WRITE(6,760)I,J,ASS(I,J),ACS(I,J),ASC(I,J),ACC(I,J)
  760 FORMAT(2I5,4E14.6)
  150 CONTINUE
C  - OPEN-CLOSED PART
      IF(NCHOP.EQ.NCHF)GOTO 300
      DO 170 I=1,NCHOP
      DO 170 J=NCHOP1,NCHF
      LIJ=LAMP(I,J)
      IF(LIJ.EQ.1) GO TO 170
      BIJ=BW(I,J)
      NLAG=2*LIJ+LACC
      CALL ZSLAG(I,J,NLAG,LIJ,ZS3,ZSD3)
      ZS3=ZS3*BIJ
      ZSD3=ZSD3*BIJ
  706     FORMAT(70X,2I3,' S=',2E12.4)
  707     FORMAT(70X,2I3,' SD=',2E12.4)
      ACC(I,J)=ACC(I,J)+REAL(ZSD3)
      ASC(I,J)=ASC(I,J)+AIMAG(ZSD3)
      ACS(I,J)=ACS(I,J)+REAL(ZS3)
      ASS(I,J)=ASS(I,J)+AIMAG(ZS3)
      IF(IPRINT.GT.1)WRITE(6,760)I,J,ASS(I,J),ACS(I,J)
  170 CONTINUE
C
C  CLOSED-CLOSED PART
  180 IF(NCHOP.EQ.NCHF)GOTO 300
      DO 200 I=NCHOP1,NCHF
      DO 200 J=I,NCHF
      LIJ=LAMP(I,J)
      IF(LIJ.EQ.1)GOTO 190
      NLAG=2*LIJ+LACC
      BIJ=BW(I,J)
      CALL TLAG(I,J,NLAG,LIJ,T1,T2,T3)
      ASS(I,J)=ASS(I,J)+T1*BIJ
      ACS(I,J)=ACS(I,J)+T2*BIJ
      ASC(I,J)=ASC(I,J)+T3*BIJ
      IF(IPRINT.GT.1)WRITE(6,760)I,J,ASS(I,J),ACS(I,J),ASC(I,J)
  190 CONTINUE
  200 CONTINUE
C
C
C  SYMMETRISE ALPHA
  300 IF(NCHF.EQ.1)RETURN
      DO 310 I=2,NCHF
      K=I-1
      DO 310 J=1,K
      ASS(I,J)=ASS(J,I)
      ACC(I,J)=ACC(J,I)
      ASC(I,J)=ACS(J,I)
  310 ACS(I,J)=ASC(J,I)
C
C
      RETURN
      END
      DOUBLE PRECISION FUNCTION ARGC(E,L,AC)
C
C  CALCULATES ARG(GAMMA(L+1-I/K)) -1/K -(1/K)*LN(K) - L*PI/2
C
      IMPLICIT REAL*8(A-H,O-Z)
C
      IF(E.GT.0)GOTO 10
      ARGC=-(L+.25)*3.141592654
      RETURN
C
   10 FK=SQRT(E)
      ET=1./FK
      IP=L+1
      P=IP
      PP=IP*IP
C
      IF(AC.LT.1.E-4)GOTO 100
      A1=10.*SQRT(ET)-ET*ET
      IF(A1.GT.PP)GOTO 20
      X=PP*E
      XP1=X+1.
      XH=P*FK
      A=-1.5707963327*(P+L-.5)
      GOTO 200
   20 L1=IP
      IP=1.+SQRT(A1)
      P=IP
      PP=IP*IP
      X=PP*E
      XP1=X+1.
      XH=P*FK
      A=-1.570796327*(P+L-.5)
      L2=IP-1
      DO 30 I=L1,L2
   30 A=A+ATAN(ET/I)
      GOTO 200
C
  100 A1=35.*ET**.25-ET*ET
      IF(A1.GT.PP)GOTO 120
      X=PP*E
      XP1=X+1.
      XH=P*FK
      A=-1.570796327*(P+L-.5)
      GOTO 140
  120 L1=IP
      IP=1.+SQRT(A1)
      P=IP
      PP=IP*IP
      X=PP*E
      XP1=X+1.
      XH=P*FK
      A=-1.570796327*(L+P-.5)
      L2=IP-1
      DO 130 I=L1,L2
  130 A=A+ATAN(ET/I)
  140 A=A+.000396825540*FK*E*(7.*(1.-3.*X)*XP1*XP1+
     C 2.*E*(1.-10.*X+5.*X*X))*XP1**(-5)
C
  200 A1=FK*X*X*.1666666667*PP
      IF(A1.GT.AC)GOTO 210
      A=A-FK*(2.-X)*.25*PP
      GOTO 220
  210 A=A-.5*ET*LOG(XP1)
  220 A2=(P-.5)*XH
      A1=A2*X*X
      IF(A1.GT.AC)GOTO 230
      A=A+A2*(1.-X*.3333333333)
      GOTO 240
  230 A=A+(P-.5)*ATAN(XH)
  240 ARGC=A+FK/(12.*(1.+X))
C
      RETURN
      END
c***********************************************************************
      SUBROUTINE ASY(IPERT,NCHF,NCHOP,EPS,CS,CSP,CC,CCP)
C
C CALCULATES ASYMPTOTIC SOLUTIONS AT RA:
C    CS=REGULAR FUNCTION FOR OPEN CHANNELS, DECAYING FUNCTION FOR CLOSED
C    CSP=DERIVATIVE WRT r
C    CC=IRREGULAR FUNCTION FOR OPEN CHANNELS
C    CCP=DERIVATIVE WRT r
C
      IMPLICIT REAL*8(A-H,O-Z)
      PARAMETER (MZCHF= 395, MZLMX=   4, MZLP1=  40, MZMNP=5700,
     1 MZTAR= 200)      
C
      COMMON/CINPUT/
     1 LAMAX,LRANG2,LRGL2,MNP2,NAST,NCHAN,NELC,NPTY2,NSPN2,NZED,MORE2,
     2 ISAT(MZTAR),LAT(MZTAR),L2P(MZCHF),NCONAT(MZTAR),
     3 BSTO,RA,
     4 CF(MZCHF,MZCHF,MZLMX),COEFF(3,MZLP1),ENAT(MZTAR),
     5 WMAT(MZCHF,MZMNP),VALUE(MZMNP)
C     
c     DIMENSION CS(MZCHF,MZCHF),CSP(MZCHF,MZCHF),CC(MZCHF,MZCHF),
c    1 CCP(MZCHF,MZCHF),EPS(MZCHF)
C
      print *,' *** Neutral target: attach a suitable asymptotic code'
     *,'  eg. Crees (1981) CPC 23, 181.'
      STOP
      END
c***********************************************************************
      SUBROUTINE BAKSUB(N,X,B,U,V,W)
C
C      SOLVES FOR X THE VECTOR EQUATION UPP X=B
C      WHERE UPP IS AN UPPER TRIANGULAR MATRIX WITH ONLY THREE NON-ZERO
C      DIAGONALS, X AND B ARE COLUMN VECTORS. THE ARRAYS U, V AND W ARE
C      AS DESCRIBED IN SUBROUTINE VECTOR.
C
      IMPLICIT REAL*8(A-H,O-Z)
      DIMENSION X(N),B(N),U(N),V(N),W(N)
C
      X(N) = B(N)/U(N)
      X(N-1) = (B(N-1)-V(N-1)*X(N))/U(N-1)
      IF (N.NE.2) THEN
        N1 = N - 1
        DO 10 I = 2,N1
          X(N-I) = (B(N-I)-V(N-I)*X(N-I+1)-W(N-I)*X(N-I+2))/U(N-I)
   10   CONTINUE
      ENDIF
C
      END                  
c***********************************************************************
      SUBROUTINE BLOCK
C
C      $Id: block.f,v 1.1 1994/03/18 17:43:47 ferrous Exp $
C
C      $Log: block.f,v $
c Revision 1.1  1994/03/18  17:43:47  ferrous
c Initial revision
c
C
C
C  PROVIDES
C     BLOCK DATA
C  CALLED AS SUBROUTINE TO AVOID LINKAGE PROBLEMS WITH LIBRARIES
C
C  DATA FOR QUADRATURES -
C  LAGUERRE AND LEGENDRE QUADRATURES WITH NUMBERS OF POINTS
C  N = 2, 4, 6, 8 AND 10
C
      IMPLICIT REAL*8(A-H,O-Z)
      COMMON/CBLK/XLAG(30),WLAG(30),XLEG(15),WLEG(15)
C
      DATA XLAG/
     1 .58578644,3.4142136,
     2 .32254769,1.7457611,4.5366203,9.3950709,
     3 .22284660,1.1889321,2.9927363,5.7751436,9.8374674,
     4 15.982874,
     5 .17027963,.90370178,2.2510866,4.26670017,7.0459054,
     6 10.758516,15.7406786,22.8631317,
     7 .13779347,.72945455,1.8083429,3.4014337,5.5524961,
     8 8.3301527,11.8437858,16.279258,21.996586,29.920697/
      DATA WLAG/
     1 .85355339,.14644661,
     2 .60315410,.35741869,.38887909E-1,.53929471E-3,
     3 .45896467,.41700083,.11337338,.10399197E-1,
     4 .26101720E-3,.89854791E-6,
     5 .36918859,.41878678,.17579499,3.3343492E-2,2.7945362E-3,
     6 9.0765088E-5,8.4857467E-7,1.0480012E-9,
     7 .30844112,.40111993,.21806829,6.2087456E-2,9.5015170E-3,
     8 7.5300839E-4,2.8259233E-5,4.2493140E-7,1.8395648E-9,
     9 9.9118272E-13/
      DATA XLEG,WLEG/.577350269,
     1 .339981044,.861136312,
     2 .238619186,.661209386,.932469514,
     3 .183434642,.525532410,.796666477,.960289856,
     4 .148874339,.433395394,.679409568,.865063367,.973906529,
     5 1.,
     6 .652145159,.347854845,
     7 .467913935,.360761573,.171324492,
     8 .362683783,.313706646,.222381034,.101228536,
     9 .295524225,.269266719,.219086363,.149451349,.066671344/
C
      RETURN
      END
c***********************************************************************
      SUBROUTINE BODE
C
C      $Id: bode.f,v 1.1 1994/03/18 17:43:47 ferrous Exp $
C
C      $Log: bode.f,v $
c Revision 1.1  1994/03/18  17:43:47  ferrous
c Initial revision
c
C
C
      IMPLICIT REAL*8(A-H,O-Z)
      PARAMETER (MZPTS= 600)
      COMMON/CPOINT/KP0,KP1,KP2,RZERO,RONE,RTWO,H
      COMMON/CBODE/WBODE(MZPTS)
C
      W=.3111111111*H
      WBODE(1)=W
      WBODE(KP2)=W
      W=1.422222222*H
      M=KP2-1
      DO 10 K=2,M,2
   10 WBODE(K)=W
      W=.5333333333*H
      M=KP2-2
      DO 20 K=3,M,4
   20 WBODE(K)=W
      IF(KP2.EQ.5)RETURN
      W=.6222222222*H
      M=KP2-4
      DO 30 K=5,M,4
   30 WBODE(K)=W
C
      RETURN
      END
c***********************************************************************
      SUBROUTINE CBUT0(NBUT,U,B,C)
C
C      $Id: cbut0.f,v 1.1 1994/03/18 17:43:47 ferrous Exp $
C
C      $Log: cbut0.f,v $
c Revision 1.1  1994/03/18  17:43:47  ferrous
c Initial revision
c
C
      IMPLICIT REAL*8(A-H,O-Z)
      PARAMETER (MZNRG=  40)
      COMMON/CBUT/FKN(0:MZNRG),UKN(0:MZNRG)
C
      LOGICAL POLE
C
C
      B=0.
      C=0.
C
C  CASE OF U.GT.0.04
      IF(U.GT..04)THEN
        FK=SQRT(U)
        POLE=.FALSE.
        DO 10 N=0,NBUT
          IF(ABS(FK-FKN(N)).GT..3)THEN
            A=1./(U-UKN(N))
            B=B+A
            C=C-A**2
          ELSE
            POLE=.TRUE.
            D1=FK-FKN(N)
          ENDIF
   10   CONTINUE
        IF(POLE)THEN
          D2=D1**2
          D=.3333333333*D1*(1.+.06666666667*D2*(1.+.095238095*D2))
          A=1./(2.*FK-D1)
          BB=(D+A)/FK
          D=.3333333333*(1.+D2*(.2+.031746032*D2))
          C=2.*C+.5*(D-A**2-BB)/U
          B=2.*B+BB
        ELSE
          T=TAN(FK)
          TK=T/FK
          B=2.*B+TK
          C=2.*C+.5*(1.+T**2-TK)/U
        ENDIF
C
C  SUM FOR U.LT..04
      ELSE
        DO 20 N=0,NBUT
          A=1./(U-UKN(N))
          B=B+A
   20     C=C-A**2
C
C  CASE OF U.LT..04 AND U.GT.-.04
        IF(U.GT.-.04)THEN
          B=2.*B+1.+.3333333333*U*(1.+.4*U)
          C=2.*C+.3333333333*(1.+U*(.8+.48571429*U))
C
C  CASE OF U.LT.-.04
        ELSE
          FK=SQRT(-U)
          T=TANH(FK)
          TK=T/FK
          B=2.*B+TK
          C=2.*C+.5*(1.-T**2-TK)/U
        ENDIF
C
      ENDIF
C
      RETURN
      END
c***********************************************************************
      SUBROUTINE COUL1(IPERT,NCHF,NCHOP,BSTO,CS,CSP,CC,CCP)
C
C CALCULATES COULOMB FUNCTIONS AT RA
C (WITHOUT (IPERT=0) OR WITH (IPERT=1) PERTURBATION):
C    CS=REGULAR FUNCTION FOR OPEN CHANNELS, DECAYING FUNCTION FOR CLOSED
C    CSP=DERIVATIVE WRT r
C    CC=IRREGULAR FUNCTION FOR OPEN CHANNELS
C    CCP=DERIVATIVE WRT r
C
      IMPLICIT REAL*8(A-H,O-Z)
      PARAMETER (MZCHF= 395, MZPTS= 600)
C
      COMMON/COULSC/FS(MZCHF,MZPTS),FSP(MZCHF),FC(MZCHF,MZPTS),
     1 FCP(MZCHF)
      COMMON/CALP/ASS(MZCHF,MZCHF),ASC(MZCHF,MZCHF),ACS(MZCHF,MZCHF),
     1 ACC(MZCHF,MZCHF)
C     
      DIMENSION CS(MZCHF,MZCHF),CSP(MZCHF,MZCHF),CC(MZCHF,MZCHF),
     1 CCP(MZCHF,MZCHF),S(MZCHF),SP(MZCHF),C(MZCHF),CP(MZCHF)
C
C  CALCULATE CHANNEL ENERGIES, NUMBER OF OPEN CHANNELS AND TABULAR POINTS
C
      CALL POINTS(0,0)
C
C  CALCULATE COULOMB FUNCTIONS AT RZERO
C
C       FS(I) IS REGULAR FUNCTION FOR OPEN CHANNELS, DECAYING
C       FUNCTION THETA FOR CLOSED CHANNELS.
C       FC(I) IS IRREGULAR FUNCTION FOR OPEN CHANNELS,
C       THETAD = ENERGY DERIVATIVE OF DECAYING FUNCTIONS FOR
C       CLOSED CHANNELS.
C
      CALL COUL
      DO I=1,NCHF
        S(I)=FS(I,1)
        SP(I)=FSP(I)
        IF(I.LE.NCHOP) THEN
          C(I)=FC(I,1)
          CP(I)=FCP(I)-BSTO*FS(I,1)
        ELSE
	  C(I)=FS(I,1)
	  CP(I)=FSP(I)-BSTO*FC(I,1)
        ENDIF
      END DO
C..CASE OF IPERT.GT.0
      IF(IPERT.GT.0)THEN
        CALL ALPHA
C  CALCULATE PERTURBED FUNCTIONS
C     OPEN-OPEN
        IF(NCHOP.GT.0) THEN
          DO J=1,NCHOP
            DO I=1,NCHOP
              CS(I,J)=S(I)*ACS(I,J)-C(I)*ASS(I,J)
              CSP(I,J)=SP(I)*ACS(I,J)-CP(I)*ASS(I,J)
              CC(I,J)=S(I)*ACC(I,J)-C(I)*ASC(I,J)
              CCP(I,J)=SP(I)*ACC(I,J)-CP(I)*ASC(I,J)
            END DO
          END DO
          DO I=1,NCHOP
            CS(I,I)=CS(I,I)+S(I)
            CSP(I,I)=CSP(I,I)+SP(I)
            CC(I,I)=CC(I,I)+C(I)
            CCP(I,I)=CCP(I,I)+CP(I)
          END DO
        ENDIF
C     CLOSED-OPEN
        IF(NCHOP.GT.0.AND.NCHOP.LT.NCHF) THEN
          DO J=1,NCHOP
            DO I=NCHOP+1,NCHF
              CS(I,J)=C(I)*ASS(I,J)-S(I)*ACS(I,J)
              CSP(I,J)=CP(I)*ASS(I,J)-SP(I)*ACS(I,J)
              CC(I,J)=C(I)*ASC(I,J)-S(I)*ACC(I,J)
              CCP(I,J)=CP(I)*ASC(I,J)-SP(I)*ACC(I,J)
            END DO
          END DO
C     OPEN-CLOSED
          DO J=NCHOP+1,NCHF
            DO I=1,NCHOP
              CC(I,J)=S(I)*ACS(I,J)-C(I)*ASS(I,J)
              CCP(I,J)=SP(I)*ACS(I,J)-CP(I)*ASS(I,J)
            END DO
          END DO
        ENDIF
C     CLOSED-   CLOSED
        IF(NCHOP.LT.NCHF) THEN
          DO J=NCHOP+1,NCHF
            DO I=NCHOP+1,NCHF
              CC(I,J)=C(I)*ASS(I,J)-S(I)*ACS(I,J)
              CCP(I,J)=CP(I)*ASS(I,J)-SP(I)*ACS(I,J)
            END DO
          END DO
          DO I=NCHOP+1,NCHF
            CC(I,I)=CC(I,I)+S(I)
            CCP(I,I)=CCP(I,I)+SP(I)
          END DO
        ENDIF
C
      ELSE
C..CASE OF IPERT.LE.0
        DO I=1,NCHF
          CS(I,I)=S(I)
          CSP(I,I)=SP(I)
          CC(I,I)=C(I)
          CCP(I,I)=CP(I)
        END DO
      ENDIF
      RETURN
      END
c***********************************************************************
      SUBROUTINE COUL
C
C  MODIFIED 21.11.85
C  MODIFIED 22.5.86 FOR CASE OF IRAD.NE.0
C
C  CALCULATION OF COULOMB FUNCTIONS
C
      IMPLICIT REAL*8(A-H,O-Z)
C
      LOGICAL QDT
C
      PARAMETER (MZCHF= 395,MZPTS= 600)
      COMMON/CQDT/R2ST(MZCHF),QDT,NQ
      COMMON/CACC/AC,ACNUM,ACJWBK,ACZP,LACC
      COMMON/COULSC/FS(MZCHF,MZPTS),FSP(MZCHF),FC(MZCHF,MZPTS),
     1 FCP(MZCHF)
      COMMON/CPOINT/KP0,KP1,KP2,RZERO,RONE,RTWO,H
      COMMON/CHAN/ECH(MZCHF),LLCH(MZCHF),EPS(MZCHF),FKNU(MZCHF),
     1 CC(MZCHF),RINF(MZCHF),ITARG(MZCHF),NCHF,NCHOP,NCHOP1      
      COMMON/CNTRL/IPRINT,IRAD,IPERT,KP2X
      COMMON/CEN/IE,NWT,NZ,ETOT
C
      DIMENSION FST(MZPTS)
C
C
C  CALCULATE OPEN-CHANNEL SOLUTIONS AT RTWO
C  AND PERFORM INWARDS INTERGATIONS TO RZERO
C  (FOR CASE OF RINF.GT.RZERO, THE REGULAR SOLUTION IS CALCULATED
C   AT RZERO USING SERIES AND INTEGRATED OUTWARDS).
      IF(NCHOP.GT.0)THEN
        HM=-H
        J=0
        DO 190 I=1,NCHOP
C...    CASE OF R2ST(I).LE.RTWO, PERTURBATION MAY BE USED
          IF(R2ST(I).LE.RTWO)THEN
            CALL INJWBK(EPS(I),LLCH(I),J)
            CALL JWBK(RTWO,J,FS(I,KP2),FSP(I),FC(I,KP2),FCP(I))
            IF(KP2.GT.1)THEN
              IF(RINF(I).LE.RZERO)THEN
                CALL NUMSC(EPS(I),CC(I),RTWO,HM,KP2,I)
              ELSE
                IF(IPERT.GT.0)THEN
                  IPERT=-IPERT
                  IF(IPRINT.GT.0)WRITE(6,720)
                ENDIF
                CALL COULS(LLCH(I),EPS(I),RZERO,S,SP)
                FSP(I)=SP
                CALL NUMS(EPS(I),CC(I),RZERO,H,1,KP2,S,SP,FST)
                W2=FC(I,KP2)*SP-FCP(I)*S
                DO 170 K=1,KP2
  170             FS(I,K)=FST(K)
                C=FC(I,KP2)
                CP=FCP(I)
                CALL NUMS(EPS(I),CC(I),RTWO,H,KP2,1,C,CP,FST)
                W0=C*FSP(I)-CP*FS(I,1)
                IF(IPRINT.GT.1)WRITE(6,600)I,RINF(I),W0,W2
                FCP(I)=CP
                DO 180 K=1,KP2
  180             FC(I,K)=FST(K)
              ENDIF
            ENDIF
C...    CASE OF R2ST(I).GT.RTWO, PERTURBATION CANNOT BE USED
          ELSE
            IPERT=0
            CALL SC(EPS(I),LLCH(I),RZERO,AC,
     +              FS(I,1),FSP(I),FC(I,1),FCP(I))
          ENDIF
  190     J=J+15
      ENDIF
C
C  CALCULATE QDT SOLUTIONS AT RZERO
      IF(QDT)THEN
        DO 200 I=NCHOP+1,NQ
          CALL SC(EPS(I),LLCH(I),RZERO,AC,
     +            FS(I,1),FSP(I),FC(I,1),FCP(I))
  200   CONTINUE
      ENDIF
C
C  CALCULATE CLOSED CHANNEL SOLUTIONS AT RTWO
C  AND PERFORM INWARDS INTEGRATIONS TO RZERO
c
      DO 210 I=NCHOP1,NCHF
C...  CASE OF R2ST(I).LE.RTWO, PERTURBATION MAY BE USED
c       IF(R2ST(I).LE.RTWO)THEN
          CALL THETA(RTWO,I,FS(I,KP2),FSP(I),FC(I,KP2),FCP(I),ICONV)
c         IF(ICONV.EQ.0)THEN
            CALL NUMT(EPS(I),CC(I),RTWO,H,KP2,KP0,I)
c         ELSE
c           IPERT=0
c         ENDIF
C...  CASE OF R2ST(I).GT.RTWO, PERTUBATION CANNOT BE USED
c       ELSE
c         IPERT=0
c         CALL SC(EPS(I),LLCH(I),RZERO,AC,FSA,FSPA,FCA,FCPA)
c         SINF=SIN(3.141592654*FKNU(I))
c         COSF=COS(3.141592654*FKNU(I))
c         FS(I,1)=FCA*SINF-FSA*COSF
c         FSP(I)=FCPA*SINF-FSPA*COSF
c       ENDIF
  210 CONTINUE
C
      IF(IPRINT.GT.1)THEN
        WRITE(6,701)ETOT
        WRITE(6,705)RTWO,KP2,H
        WRITE(6,700)
        DO 230 J=1,NCHF
  230     WRITE(6,710)J,FS(J,1),FSP(J),FC(J,1),FCP(J)
      ENDIF
C
      RETURN
C
  600 FORMAT(24X,'I =',I3,', RINF =',F7.2,', W0 =',F9.6,', W2 =',F9.6/)
  620 FORMAT(///10X,30(1H*)//10X,'FOR ETOT =',E15.6/
     + 10X,'(R2ST(',I2,')=',F8.2,').GT.(RTWO =',F9.2,')  AND'/
     + 10X,'(RINF(',I2,') =',F9.2,').GT.(RZERO=',F8.2,')'/
     + 10X,'KP2 =',I5,', (AMPERSAND)PTS = MZPTS'/
     + 10X,'CANNOT CALCULATE RADIATIVE DATA FOR THIS CASE'/
     + 10X,'TRY LARGER VALUE OF (AMPERSAND)PTS'///)
  630 FORMAT(///10X,30(1H*)//10X,'FOR ETOT =',E15.6/
     + 10X,'CHANNEL ',I2,' HAS'/
     + 10X,'CHANNEL ANGULAR MOMENTUM QUANTUM NUMBER =',I3/
     + 10X,'CHANNEL EFFECTIVE QUANTUM NUMBER =',F9.2/
     + 10X,'CANNOT CALCULATE RADIATIVE DATA FOR THIS CASE WITH ',
     + '(AMPERSAND)TET=MZTET'/10X,'TRY SMALLER VALUE OF QNMAX OR LARGER', *VALUE!
     + ' VALUE OF (AMPERSAND)TET'///)
  700 FORMAT(//10X,'COULOMB FUNCTIONS S,SP,C AND CP'/)
  701 FORMAT(//11X,'E =',F11.6/11X,14('-'))
  705 FORMAT(//'   RTWO =',F11.6,',  KP2 =',I5,',  H =',F11.6)
  710 FORMAT(I5,4E15.6)
  720 FORMAT(5X,5(1H*),' REGULAR COULOMB FUNCTION FROM SERIES ',5(1H*)/)
C
      END
c***********************************************************************
      SUBROUTINE COULFG(LL,EPS,RHO,ACC,F,FP,G,GP,K,IERR,ACTACC)
C
C  CALCULATES COULOMB FUNCTIONS F AND G AND THEIR DERIVATIVES
C
C  ORIGINAL VERSION PUBLISHED IN COMP. PHYS. COMM.25, 87, 1982.
C  PRESENT VERSION MODIFIED TO AVOID UNDERFLOW AND OVERFLOW
C  CONDITIONS IN THE SUMMATIONS OVER N OF
C         U(N)=A(N)*RHO**(N+L+1)
C     AND V(N)=D(N)*RHO**(N+L+1)
C  U(N) AND V(N) ARE CALCULATED RECURSIVELY.
C
C  A FURTHER MODIFICATION IS THAT THESE SUMMATIONS ARE
C  DONE IN DOUBLE PRECISION EVEN WHEN SINGLE PRECISION
C  IS USED ELSEWHERE.
C
C
C  INPUT -
C        LL=ANGULAR MOMENTUM QUANTUM NUMBER
C        EPS=Z-SCALED ENERGY IN RYDBERGS
C        RHO=Z-SCALED RADIAL VARIABLE IN ATOMIC UNITS
C        ACC=ACCURACY REQUIRED
C
C  OUTPUT -
C        F=REGULAR FUNCTION
C        FP=DERIVATIVE OF F
C        G=IRREGULAR FUNCTION
C        GP=DERIVATIVE OF G
C        K=NUMBER OF TERMS NEEDED IN EXPANSION
C        IERR=ERROR CODE
C        ACTACC=ACCURACY ACTUALLY ACHIEVED
C
C  CONVERGENCE CRITERION -
C        VALUE OF WRONSKIAN CONVERGED TO ACCURACY OF 0.5*ACC
C
C  ERROR CODES -
C        IERR=0, CONVERGED WITH ACTACC.LT.ACC
C        IERR=1, CONVERGED WITH ACTACC.GT.ACC
C        IERR=2, NOT CONVERGED WITH 101 TERMS IN MAIN SUMMATION
C
C  INITIALIZATION
C
C
      IMPLICIT REAL*8 (A-H,O-Z)
C
      DOUBLE PRECISION DUBU0,DUBU1,DUBU2,DUBV0,DUBV1,DUBV2,
     +  DUBQ1,DUBF,DUBFP,DUBS,DUBSP
C
C  NOTE - IMPLICIT REAL*(A-H,O-Z) IS OPTIONAL BUT IT IS
C         RECOMMENDED THAT THE EXPLICIT DOUBLE PRECISION
C         STATEMENT SHOULD ALWAYS BE RETAINED.
C
      DATA R2PI,PS0/.159154943,-.154431330/
      IERR=0
      LP1=LL+1
      L2=2*LL
      L2P1=L2+1
      FL=LL
      FLP1=LP1
      FL2P1=L2P1
      E2=.5*EPS
      R2=2.*RHO
      ACC2=2.*ACC
C
C     INITIALIZE FA=FACTORIAL(2*LL+1)
C     AND PS=PSI(2*LL+2)+PSI(1)
C
      FA=1.
      PS=PS0
C
C
C  CALCULATE ALPHA(N) AND BETA(N) AND INITIALIZE S AND SP
C  CONTINUE CALCULATION OF FA AND PS
C
C     S AND SP FOR N=0
      X3=-L2
      X2=L2P1
      X1=-2.*R2**(-LP1)
      SP=X3*X1
      X1=R2*X1
      S=X1
C
C     INITIALIZE FOR COEFFICIENTS IN RECURSION FORMULAE
      P1=FL*E2
      P2=P1
      Q1=-E2
C
C     INITIALIZE ALPHA AND BETA
      ALP1=1.
      ALP2=1.+P2
      BET1=0.
      BET2=Q1
C
c     IF(LL.EQ.0)GOTO 20
      IF(LL.NE.0) THEN
C
C     S AND SP FOR N=1
        X3=X3+2.
        X2=X2-1.
        X1=X1/X2
        SP=SP+X3*X1
        X1=R2*X1
        S=S+X1
C
C     LOOP FOR N=2 TO 2*LL
        DO 10 N=2,L2
C
C     CONTINUE CALCULATION OF FA AND PSI
          FN=N
          FA=FN*FA
          PS=PS+1./FN
C
C     CONTINUE CALCULATION OF S AND SP
          X3=X3+2.
          X2=X2-1.
          X1=X1/(X2*FN)
          SP=SP+X3*X1*ALP2
          X1=R2*X1
          S=S+X1*ALP2
C
C     COMPUTE COEFFICIENTS IN RECURSION FORMULAE
          P1=P1-E2
          P2=P2+P1
          Q1=Q1-E2
C     NOW HAVE P2=-N*(N-2*LL-1)*EPS/4
C     AND Q1=-N*EPS/2
C
C     NEW ALPHA AND BETA
          ALP0=ALP1
          ALP1=ALP2
          ALP2=ALP1+P2*ALP0
          BET0=BET1
          BET1=BET2
   10     BET2=BET1+P2*BET0+Q1*ALP0
C
C     NORMALIZE S AND SP, COMPLETE CALCULATION OF FA AND PS
        S=S*FA
        SP=SP*FA
        FA=FL2P1*FA
        PS=PS+1./FL2P1
C
C     COMPLETE CALCULATION OF ALPHA AND BETA
        P1=P1-E2
        P2=P2+P1
        Q1=Q1-E2
        ALP0=ALP1
        ALP1=ALP2
        BET0=BET1
        BET1=BET2
        BET2=BET1+P2*BET0+Q1*ALP0
      ENDIF
C
c  20 CONTINUE
C     NOW HAVE ALP1=ALPHA(2*LL+1)
C     AND BET1=BETA(2*LL+1), BET2=BETA(2*LL+2)
C
C     VALUE OF A=A(EPS,LL)
      A=ALP1
      A4=4.*A
      CL=2.*A*LOG(ABS(R2))
CD    FOR SINGLE PRECISION REPLACE DLOG BY ALOG AND DABS BY ABS
      CLP=2.*A/RHO
C
C  CALCULATE F AND FP AND CONTINUE CALCULATION OF S AND SP
C
C     CALCULATE A0,A1,D0,D1
      A0=(2.**LP1)/FA
      A1=-A0/FLP1
      PS=2.*PS*A
      D0=(BET1-PS)*A0
      D1=(BET2-PS-(2.+1./FLP1)*A)*A1
C
C     INITIALIZE F,FP, CONTINUE CALCULATION OF S,SP
C          -VALUES FOR N=0
C           U0 AND V0
      FNPLP1=FLP1
      C1=RHO**LL
      U0=A0*C1
      V0=D0*C1
      FP=FNPLP1*U0
      SP=SP+FNPLP1*V0
      U0=U0*RHO
      V0=V0*RHO
      F=U0
      S=S+V0
      W1=F*(CLP*F+SP)-FP*S
      NNN=0
C
C          - VALUES FOR N=1
C            U1 AND V1
      FNPLP1=FNPLP1+1.
      C1=C1*RHO
      U1=A1*C1
      V1=D1*C1
      FP=FP+FNPLP1*U1
      SP=SP+FNPLP1*V1
      U1=U1*RHO
      V1=V1*RHO
      F=F+U1
      S=S+V1
      W2=F*(CLP*F+SP)-FP*S
      DW2=ABS(W2-W1)
C
C     INITIALIZE FOR COEFFICIENTS IN RECURSION FORMULAE
      P1=-2.*FLP1
      P2=P1
      Q1=A4+2.*A*FL2P1
      REPS=RHO*EPS
C
C  CONVERT TO DOUBLE
                          DUBU0=U0
                          DUBU1=U1
                          DUBV0=V0
                          DUBV1=V1
                          DUBQ1=Q1
                          DUBS=S
                          DUBSP=SP
                          DUBF=F
                          DUBFP=FP
C     LOOP FOR N=2 TO 100
      DO 40 N=2,100
C
C     COMPUTE COEFFICIENTS IN RECURSION FORMULAE
        P1=P1-2.
        P2=P2+P1
        DUBQ1=DUBQ1+DBLE(A4)
C     NOW HAVE P2=-N*(N+2*LL+1)
C     AND DUBQ1=2*A*(2*N+2*LL+1)
C
C      COMPUTE DUBU2=U(N) AND DUBV2=V(N)
        DUBU2=(2.*DUBU1+DBLE(REPS)*DUBU0)/DBLE(P2)
        DUBV2=(2.*DUBV1+DBLE(REPS)*DUBV0+DUBQ1*DUBU2)/DBLE(P2)
C
C     INCREMENT DUBFP AND DUBSP
        FNPLP1=FNPLP1+1.
        DUBFP=DUBFP+DBLE(FNPLP1)*DUBU2
        DUBSP=DUBSP+DBLE(FNPLP1)*DUBV2
C
C     INCREMENT DUBF AND DUBS
        DUBU2=DUBU2*DBLE(RHO)
        DUBV2=DUBV2*DBLE(RHO)
        DUBF=DUBF+DUBU2
        DUBS=DUBS+DUBV2
C
C     CALCULATE WRONSKIAN
        W1=W2
        DW1=DW2
        W2=DUBF*(CLP*DUBF+DUBSP)-DUBFP*DUBS
        DW2=ABS(W2-W1)
C
C     CONVERGENCE TEST
        K=N+1
c     IF(DW1.GT.ACC2)GOTO 30
c     IF(DW2.GT.ACC2)GOTO 30
c     GOTO 50
        if (DW1.le.ACC2.and.DW1.le.ACC2) go to 50
C
C  NEW DUBU0,DUBU1,DUBV0,DUBV1
c  30   CONTINUE
        DUBU0=DUBU1
        DUBU1=DUBU2
        DUBV0=DUBV1
        DUBV1=DUBV2
C
   40 CONTINUE
C
C  NOT CONVERGED
C
      IERR=2
      ACTACC=ABS(0.25*W2-1.)
      GOTO 60
C
C  CONVERGED
C
   50 ACTACC=ABS(0.25*W2-1.)
      IF(ACTACC.GT.ACC)IERR=1
C
C  COMPLETE CALCULATION OF G AND GP
C
   60                    S=DUBS
                         SP=DUBSP
                         F=DUBF
                         FP=DUBFP
      G=(S+CL*F)*R2PI
      GP=(SP+CL*FP+CLP*F)*R2PI
C
      RETURN
C
      END
c***********************************************************************
      SUBROUTINE COULS(LL,EPS,RHO,S,SP)
C
C  INSERTED 21.11.85
C
C  CALCULATES COULOMB FUNCTION S AND ITS DERIVATIVE SP
C  FROM POWER-SERIES EXPANSION.
C
C  THE STATEMENT "IMPLICIT REAL*8(A-H,O-Z)" CAN BE DELETED FOR
C  SINGLE PRECISION BUT IT IS RECOMMENDED TO KEEP THE
C  EXPLICIT DOUBLE PRECISION STATEMENT.  THE SUMMATION
C  FOR THE COULOMB FUNCTIONS IS THEN CARRIED OUT IN
C  DOUBLE PRECISION.
      IMPLICIT REAL*8 (A-H,O-Z)
      DOUBLE PRECISION C1,D0,D1,D2,DM,DEPS,DRHO,F,FLP1,FNPLP1,FP,
     +  P1,P2,REPS,U0,U1,U2,UM
C
      COMMON/CACC/AC,ACNUM,ACJWBK,ACZP,LACC
      COMMON/CNTRL/IPRINT,IRAD,IPERT,KP2X
      LOGICAL T(0:2),TP(0:2)
C
C  INITIALISATIONS
      DRHO=RHO
      DEPS=EPS
      FLP1=LL+1
      ACC10=.1*AC
c     DO 10 I=0,1
c       T(I)=.FALSE.
c  10   TP(I)=.FALSE.
      T(0)=.FALSE.
      T(1)=.FALSE.
      TP(0)=.FALSE.
      TP(1)=.FALSE.
C
C  POWER-SERIES EXPANSION
C  **********************
C
C   VALUES FOR N=0
      FNPLP1=FLP1
      C1=DRHO**LL
      U0=C1
      D0=FNPLP1*U0
      DM=ABS(D0)
      FP=D0
      U0=U0*DRHO
      UM=ABS(U0)
      F=U0
C
C   VALUES FOR N=1
      FNPLP1=FNPLP1+1.
      C1=C1*DRHO
      U1=-C1/FLP1
      D1=FNPLP1*U1
      DM=MAX(ABS(D1),DM)
      FP=FP+D1
      U1=U1*DRHO
      UM=MAX(ABS(U1),UM)
      F=F+U1
C
C   INITIALIZE FOR COEFFICIENTS IN RECURSION FORMULAE
      P1=-2.*FLP1
      P2=P1
      REPS=DRHO*DEPS
C
C   LOOP FOR N=2 TO 100
      DO 40 N=2,100
C     COMPUTE COEFFICIENTS IN RECURSION FORMULAE
        P1=P1-2.
        P2=P2+P1
C     NOW HAVE P2=-N*(N+2*LL+1)
C     COMPUTE U2 AND INCREMENT FP
        FNPLP1=FNPLP1+1.
        U2=(2.*U1+REPS*U0)/P2
        D2=FNPLP1*U2
        DM=MAX(ABS(D2),DM)
        FP=FP+D2
C     MODIFY U2 AND INCREMENT F
        U2=U2*DRHO
        UM=MAX(ABS(U2),UM)
        F=F+U2
C     TEST CONVERGENCE -- 2 FOR 10 LINES, WE'92SEP19:
        T(2)=ABS(U2).LT.ABS(F)*ACC10
        TP(2)=ABS(D2).LT.ABS(FP)*ACC10
c       DO 20 I=0,2
c  20   IF(.NOT.T(I))GOTO 21
        IF(T(0).AND.T(1).AND.T(2)) THEN
          IF(IPRINT.GT.1)THEN
            UM=UM/ABS(F)
            WRITE(6,610)LL,EPS,UM
          ENDIF
          GOTO 50
        ENDIF
c  21   DO 22 I=0,2
c  22     IF(.NOT.TP(I))GOTO 23
        IF(TP(0).AND.TP(1).AND.TP(2)) THEN
          IF(IPRINT.GT.1)THEN
            DM=DM/ABS(FP)
            WRITE(6,620)LL,EPS,DM
          ENDIF
          GOTO 50
        ENDIF
C     NEW U0,U1,T AND TP
   23   U0=U1
        U1=U2
c       DO 30 I=0,1
c       T(I)=T(I+1)
c  30   TP(I)=TP(I+1)
        T(0)=T(1)
        T(1)=T(2)
        TP(0)=TP(1)
        TP(1)=TP(2)
   40 CONTINUE
C
C  SERIES NOT CONVERGED
      WRITE(6,600)LL,EPS,RHO
      STOP
C
C  NORMALISATION
C  *************
C
C   NORMALISE FOR FUNCTIONS .5*F AND .5*FP
C
   50 S=F
      SP=FP
      DO 60 K=1,LL
        C=1./(K*(2*K+1))
        S=S*C
   60   SP=SP*C
C
C   CALCULATE CAP B AND FUNCTIONS S AND SP
      IF(EPS.GT.0)THEN
        A=1.
        IF(LL.GT.0)THEN
          A1=1.
          A2=-EPS
          A3=EPS+EPS
          DO 70 I=1,LL
            A2=A2+A3
            A1=A1+A2
   70       A=A*A1
        ENDIF
        IF(EPS.LT.0.01)THEN
          B=A
        ELSE
          B=A/(1.-EXP(-6.28318531/SQRT(EPS)))
        ENDIF
        C=2.50662828*SQRT(B)
      ELSE
        C=2.50662828
      ENDIF
      S=S*C
      SP=SP*C
C
      RETURN
C
  600 FORMAT(//10X,60('*')//10X,'SERIES IN COULS NOT CONVERGED'
     + /10X,' LL =',I3,',  EPS =',1P,E15.5,',  RHO =',
     +  E15.5//10X,60('*')//)
  610 FORMAT(/5X,'SUBROUTINE COULS, LL =',I3,',  EPS =',
     +  1P,E13.5,',  UM =',E11.2)
  620 FORMAT(/5X,'SUBROUTINE COULS, LL =',I3,',  EPS =',
     +  1P,E13.5,',  DM =',E11.2)
      END
c***********************************************************************
      SUBROUTINE EIGEN(N,EIG,EPSI,P,R,POLY,BETA)
C
C      ACCEPTS THE ARRAYS R AND P OF MAIN AND SUPER DIAGONAL ELEMENTS
C      RESPECTIVELY. USING THE STURM SEQUENCE PROPERTY
C      A BISECTION METHOD IS APPLIED TO DETERMINE THE EIGENVALUES
C      (STORED IN THE ARRAY EIG ON RETURN) TO AN ACCURACY
C      SPECIFIED BY EPSI. N IS AS DEFINED IN THE SUBROUTINE HSLDR.
C
      IMPLICIT REAL*8(A-H,O-Z)
      DIMENSION EIG(N),P(N),R(N),POLY(N),BETA(N)
      DATA ZERO,HALF,ONE/0.0D0,0.5D0,1.0D0/,
     A TINY/1.0D-5/
C
C      CALCULATE THE AVERAGE OF THE GREATEST AND SMALLEST MAIN
C      DIAGONAL ELEMENTS STORING THE RESULT IN AMID.
C
      ASMALL = R(1)
      ALARG = R(1)
      DO 10 I = 2,N
        IF (R(I).GT.ALARG) ALARG = R(I)
        IF (R(I).LT.ASMALL) ASMALL = R(I)
   10 CONTINUE
      AMID = (ALARG+ASMALL)*HALF
C
C      REDUCE EACH MAIN DIAGONAL ELEMENT BY AMID AND CALCUATE, USING
C      THE STURM SEQUENCE PROPERTY, THE EIGENVALUES OF THE CORRESPONDING
C      REDUCED TRI-DIAGONAL MATRIX.
C
      DO 20 I = 1,N
        R(I) = R(I) - AMID
   20 CONTINUE
C
C      CALCULATE THE MAXIMUM INFINITY NORM G OF THE MATRIX. THE
C      EIGENVALUES LIE IN THE RANGE -G TO +G.
C
      G = ZERO
      DO 30 I = 1,N
        G1 = ABS(R(I))
        IF (I.GT.1) G1 = G1 + ABS(P(I-1))
        IF (I.LT.N) G1 = G1 + ABS(P(I))
        IF (G.LT.G1) G = G1
   30 CONTINUE
C
C      CALCULATE THE SQUARES OF THE SUPER DIAGONAL ELEMENTS AND STORE
C      THESE IN THE ARRAY BETA.
C
      N1 = N - 1
      DO 40 I = 1,N1
        BETA(I) = P(I)*P(I)
   40 CONTINUE
C
C      THIS LOOP DETERMINES THE EIGENVALUES ONE AT A TIME IN ORDER OF
C      ALGEBRAIC SIZE DOWNWARDS.
C
      DO 100 K = 1,N
        AL = -G
        BL = G
C
C      ONCE THROUGH THIS LOOP IS ONE BISECTION OF THE RANGE. CL1 IS THE
C      CURRENT ESTIMATE, CL THE IMMEDIATELY PREVIOUS ESTIMATE OF THE
C      EIGENVALUE.
C
        DO 80 J = 1,100
          CL1 = (AL+BL)*HALF
          IF (J.NE.1) THEN
C
C      IF THE EIGENVALUE HAS BEEN DETERMINED TO A SPECIFIED ACCURACY
C      EPSI, THE CALCULATION IS COMPLETE.
C
            IF (ABS(CL1-CL).LT.EPSI) GOTO 90
C
C      LSUM STORES THE NUMBER OF AGREEMENTS IN SIGN IN THE STURM
C      SEQUENCE.
C
          ENDIF
          LSUM = 0
          DO 60 I = 1,N
            POLY(I) = R(I) - CL1
   60     CONTINUE
          X = POLY(1)
          IF (POLY(1).GT.ZERO) LSUM = 1
C
C      THIS LOOP CALCULATES ALL THE REMAINING MEMBERS OF THE STURM
C      SEQUENCE. THE NUMBER OF AGREEMENTS IN SIGN IS ALSO DETERMINED.
C
          DO 70 I = 2,N
            IF (X.EQ.ZERO) THEN
              X = POLY(I) - ABS(P(I-1))/ (ONE+TINY)
C
            ELSE
              X = POLY(I) - BETA(I-1)/X
            ENDIF
C
            IF (X.GT.ZERO) LSUM = LSUM + 1
   70     CONTINUE
          CL = CL1
C
C      THE NEW RANGE FOR THE EIGENVALUE (DEPENDENT ON THE VALUE OF LSUM)
C      IS DETERMINED.
C
          IF (LSUM.LT.K) THEN
            BL = CL1
C
          ELSE
            AL = CL1
          ENDIF
C
   80   CONTINUE
C
C      THE EIGENVALUE IS STORED IN THE ARRAY EIG.
C
   90   CONTINUE
        EIG(K) = CL1
C
C      RETURN TO CALCULATE THE NEXT EIGENVALUE.
C
  100 CONTINUE
C
C      THE ELEMENTS OF THE ORIGINAL TRI-DIAGONAL MATRIX ARE REGAINED
C      AND ITS EIGENVALUES FOUND.
C
      DO 110 I = 1,N
        R(I) = R(I) + AMID
        EIG(I) = EIG(I) + AMID
  110 CONTINUE
C
      END
c***********************************************************************
      SUBROUTINE EIGVEC(N,A,LENGTH,X,P)
C
C      TAKES THE EIGENVECTOR OF THE TRI-DIAGONAL MATRIX
C      STORED IN X AND DETAILS OF THE MATRICES USED IN TRANSFORMING
C      THE ORIGINAL MATRIX TO TRI-DIAGONAL FORM, STORED IN A,
C      AND OBTAINS THE CORRESPONDING EIGENVECTOR OF THE ORIGINAL
C      MATRIX. N AND LENGTH ARE AS DEFINED IN THE SUBROUTINE HSLDR.
C
      IMPLICIT REAL*8(A-H,O-Z)
      DIMENSION A(LENGTH),X(N),P(N)
      DATA ZERO/0.0D0/
C
      N2 = N - 2
      N22 = (N* (N+1))/2
C
C      N2=N-2 TRANSFORMATIONS TO OBTAIN EACH EIGENVECTOR.
C
      DO 30 K = 1,N2
C
C      K1 IS THE NUMBER OF ELEMENTS IN THE FIRST (K-1) ROWS OF THE
C      UPPER TRIANGLE STORED IN A.
C
        K1 = N22 - ((K+2)* (K+3))/2 + 1
        SOP = ZERO
        KP1 = K + 1
        NK1 = N - K - 1
        DO 10 I = 1,KP1
          SOP = SOP + A(K1+I)*X(NK1+I)
   10   CONTINUE
        IF (SOP.EQ.ZERO) GOTO 30
C
C      FROM INFORMATION STORED IN THE ARRAY A BKR IS DETERMINED AS IN
C      SUBROUTINE HOUSE.
C
        BKR = - (P(NK1)*A(K1+1))
        SOP = SOP/BKR
        DO 20 J = 1,KP1
          X(NK1+J) = X(NK1+J) - A(K1+J)*SOP
   20   CONTINUE
   30 CONTINUE
C
C      THE EIGENVECTOR OF THE ORIGINAL MATRIX IS NORMALISED.
C
      CALL NORM(N,X)
C
      END       
      DOUBLE PRECISION FUNCTION FCHI(E,L,AC)
C
C  CALCULATES REAL PART OF PSI(L+1+I*GAM) - LN(GAM)
C  WHERE E = 1/(GAM**2).
C  THIS IS REQUIRED FOR CALCULATION OF SCRIPT G.
C
      IMPLICIT REAL*8(A-H,O-Y)
      IMPLICIT COMPLEX(Z)
C
      FCHI=0.
      IF(E.EQ.0)RETURN
C
      AC1=(20.*AC)**.333
C
c     IF(E.GT.AC1)GOTO 100
      IF(E.LE.AC1) THEN
        C=0.
c       IF(L.EQ.0)GOTO 20
        IF(L.NE.0) THEN
          A1=1.
          A2=-E
          A3=E+E
          DO 10 I=1,L
            A2=A2+A3
            A1=A1+A2
   10       C=C+I/A1
        ENDIF
   20   FCHI=E*((((1.05*E+1.)*E+2.1)*E+21.)*.003968253968+C)
        RETURN
      ENDIF
C
  100 AC1=1./SQRT(AC1)
      FL=L+1
c     IF(FL.GT.AC1)GOTO 300
      IF(FL.LE.AC1) THEN
        N=AC1
        FL=N+1
        L1=L+1
        DO 210 I=L1,N
          FI=I
  210     FCHI=FCHI+FI/(1.+E*FI*FI)
        FCHI=-FCHI*E
      ENDIF
C
  300 X1=FL*E
      X=1.+X1*FL
      ZE=CMPLX(FL,1./SQRT(E))
      ZE=-1./(ZE*ZE)
      FCHI=FCHI+.5*(LOG(X)-(X1/X))+REAL((((1.05*ZE+1.)*ZE
     C +2.1)*ZE+21.)*ZE)*.003968253968
C
      RETURN
      END
c***********************************************************************
      SUBROUTINE HOUSE(N,A,LENGTH,P,R,ARRAY1)
C
C      ACCEPTS THE UPPER TRIANGLE OF ELEMENTS OF A SYMMETRIC MATRIX,
C      STORED IN THE LINEAR ARRAY A, AND USING THE HOUSEHOLDER METHOD
C      REDUCES THIS TO TRI-DIAGONAL FORM, STORING THE NEW MAIN
C      DIAGONAL ELEMENTS IN POSITION IN A AND ALSO IN THE ARRAY R,
C      AND THE SUPER-DIAGONAL ELEMENTS IN THE ARRAY P. DETAILS
C      OF THE TRANSFORMING MATRICES ARE OVERWRITTEN IN THE REDUNDANT
C      SPACE OF A. N AND LENGTH ARE AS DEFINED IN SUBROUTINE HSLDR.
C
      IMPLICIT REAL*8(A-H,O-Z)
      DIMENSION A(LENGTH),P(N),R(N),ARRAY1(N)
      DATA EPSI/1.0D-9/,ZERO,HALF/0.0D0,0.5D0/
C
      N1 = N - 1
      N2 = N - 2
C
C      EACH TIME THROUGH THIS LOOP ONE MORE ROW OF THE MATRIX IS
C      TRANSFORMED TO TRI-DIAGONAL FORM
C
      DO 150 K = 1,N2
        DO 10 J = K,N
          P(J) = 0.0D0
   10   CONTINUE
C
C      KB=KC-1 IS THE NUMBER OF ELEMENTS IN THE FIRST (K-1) ROWS OF THE
C      UPPER TRIANGLE
C
        NJ = N - K
        KC = ((K-1)* (NJ+N+2))/2 + 1
C
C      THE SQUARE ROOT OF THE SUM OF THE SQUARES OF THE REMAINING
C      OFF DIAGONAL ELEMENTS IN ROW K IS FOUND AND STORED IN SUM
C
        SUM = 0.0D0
        DO 20 J = 1,NJ
          SUM = A(KC+J)*A(KC+J) + SUM
   20   CONTINUE
        SUM = SQRT(SUM)
        IF (SUM.LT.EPSI) GOTO 130
C
C      SUM IS GIVEN THE SAME SIGN AS THE SUPER DIAGONAL ELEMENT IN ROW K
C
        IF (A(KC+1).LT.ZERO) SUM = -SUM
        BKR = SUM* (SUM+A(KC+1))
C
C      THE FIRST ELEMENT OF THE VECTOR FROM WHICH THE TRANSFORMING
C      MATRIX IS DERIVED IS OVERWRITTEN ON THE OLD SUPER DIAGONAL
C      ELEMENT IN ROW K. THE REMAINING ELEMENTS ARE ALREADY IN POSITION
C      IN ROW K.
C
        A(KC+1) = A(KC+1) + SUM
C
C      THE SUPER DIAGONAL ELEMENT IN ROW K OF THE NEW TRI-DIAGONAL
C      MATRIX IS STORED IN ARRAY P
C
        P(K) = -SUM
C
C      KD IS THE NUMBER OF ELEMENTS IN THE FIRST K ROWS OF THE UPPER
C      TRIANGLE
C
        KD = (K* (NJ+N+1))/2
C
C      THE TRANSFORMATION DERIVED FROM THE NJ=(N-K) VECTOR ELEMENTS
C      STORED IN A IS NOW APPLIED TO THE LAST NJ ROWS OF THE MATRIX
C      (THE LAST NJ SPACES OF THE ARRAY P ARE SUCCESSIVELY
C       OVERWRITTEN IN THE PROCESS).
C
        LOL = KC - K + 1
        DO 70 M = K,N1
          NM = N - M
          MO = ((NM+N+1)*M)/2
          M1 = M - 1
          LO = LOL + M1
          SUM = P(1+M)
          DO 30 L = 1,NM
            SUM = SUM + A(MO+L)*A(LO+L)
   30     CONTINUE
          IF (M.EQ.K) GOTO 60
          JM1 = KD + M - K + 1
          DO 40 L = K,M1
            ARRAY1(L) = A(JM1)
            JM1 = JM1 + N1 - L
   40     CONTINUE
          DO 50 L = K,M1
            SUM = SUM + ARRAY1(L)*A(LOL+L)
   50     CONTINUE
   60     P(1+M) = SUM/BKR
   70   CONTINUE
C
        SUM = 0.0D0
        DO 80 L = 1,NJ
          SUM = SUM + A(KC+L)*P(K+L)
   80   CONTINUE
        SUM = SUM/BKR
        DO 90 L = 1,NJ
          P(K+L) = P(K+L) - A(KC+L)*SUM*HALF
   90   CONTINUE
        DO 120 I = 1,NJ
          AIM = A(KC+I)
          PIN = P(K+I)
          DO 100 J = I,NJ
            ARRAY1(J) = -AIM*P(K+J) - PIN*A(KC+J)
  100     CONTINUE
          KE = KD + (I-1)*NJ - (I* (I-1))/2
          DO 110 J = I,NJ
            A(KE+J) = A(KE+J) + ARRAY1(J)
  110     CONTINUE
  120   CONTINUE
        GOTO 150
C
  130   CONTINUE
        P(K) = -EPSI
        DO 140 I = 1,NJ
          A(I+KC) = 0.0D0
  140   CONTINUE
  150 CONTINUE
C
C      THE LAST SUPER DIAGONAL ELEMENT IS ENTERED INTO THE ARRAY P.
C
      ILK = ((N+1)*N)/2 - 1
      P(N1) = A(ILK)
C
C     THE MAIN DIAGONAL ELEMENTS PICKED OUT FROM THE ARRAY A ARE
C     STORED IN THE ARRAY R.
C
      DO 160 I = 1,N
        ILK = ((I-1)* (2*N-I+2))/2 + 1
        R(I) = A(ILK)
  160 CONTINUE
C
      END
      SUBROUTINE HSLDR(N,A,LENGTH,EPSI,EIG,X,NO,P,MSV)
C***********************************************************************
C
C                   HSLDR
C
C***********************************************************************
C
C      THIS SUBROUTINE ACCEPTS THE UPPER TRIANGLE OF AN N*N SYMMETRIC
C      MATRIX AND ON THE FIRST CALL DETERMINES ALL THE EIGENVALUES AND
C      THE FIRST EIGENVECTOR.  ON EACH FURTHER CALL ONE MORE OF THE
C      REMAINING EIGENVECTORS IS CALCULATED.
C
C      DEFINITION OF THE ARGUMENTS.
C      N........... THE DEGREE OF THE SYMMETRIC MATRIX TO BE
C                   DIAGONALISED.
C      A........... THE LINEAR ARRAY CONTAINING THE UPPER TRIANGLE OF
C                   THE ORIGINAL MATRIX, OVERWRITTEN ON RETURN BY THE
C                   MAIN DIAGONAL ELEMENTS OF THE TRI-DIAGONAL MATRIX
C                   AND DETAILS OF THE TRANSFORMING MATRICES.
C      LENGTH...... =(N*(N+1))/2 , THE SIZE OF THE ARRAY A.
C      EPSI........ THE ACCURACY TO WHICH THE EIGENVALUES ARE TO BE
C                   DETERMINED.
C      EIG......... THIS ARRAY CONTAINS THE EIGENVALUES ON RETURN.
C      X........... THIS ARRAY CONTAINS ONE EIGENVECTOR ON RETURN.
C      NO.......... THIS RUNS FROM 1 TO N AND SPECIFIES WHICH
C                   EIGENVECTOR IS STORED IN X ON RETURN
C
      IMPLICIT REAL*8(A-H,O-Z)
      DIMENSION A(LENGTH),EIG(N),X(N),P(MSV,9)
C
C      THE SIZE OF THE MATRIX WAS LIMITED BY THE SIZE OF THE INTERNAL
C      ARRAYS IN THE SAVE1, SAVE2 AND SAVE3 COMMON BLOCKS.
C
      IF (N.LE.1) THEN
        X(1) = 1.0D0
        EIG(1) = A(1)
        P(1,2) = A(1)
        GOTO 60
C
      ELSE IF (N.EQ.2) THEN
        P(1,1) = A(2)
        P(1,2) = A(1)
        P(2,2) = A(3)
        GOTO 10
C
      ENDIF
C
C      IF THE FIRST EIGENVECTOR HAS ALREADY BEEN FOUND THE
C      TRI-DIAGONALISING AND EIGENVALUE SUBROUTINES ARE SKIPPED ROUND.
C
      IF (NO.NE.1) GOTO 20
C
C      THE TRI-DIAGONALISING SUBROUTINE IS ENTERED.
C
      CALL HOUSE(N,A,LENGTH,P(1,1),P(1,2),P(1,9))
C
C      THE ELEMENTS OF THE TRI-DIAGONAL MATRIX ARE USED TO DETERMINE
C      THE EIGENVALUES.
C
   10 CONTINUE
      CALL EIGEN(N,EIG,EPSI,P(1,1),P(1,2),P(1,3),P(1,4))
C
C      THE EIGENVECTOR OF THE TRI-DIAGONAL MATRIX CORRESPONDING TO A
C      PARTICULAR EIGENVALUE IS DETERMINED.
C
   20 CONTINUE
      CALL VECTOR(N,EIG,X,NO,P(1,1),P(1,2),P(1,3),P(1,4),P(1,5),P(1,6),
     A            P(1,7),P(1,8),P(1,9))
      IF (N.EQ.2) GOTO 30
C
C      THE CORRESPONDING EIGENVECTOR OF THE ORIGINAL MATRIX IS FOUND.
C
      CALL EIGVEC(N,A,LENGTH,X,P)
C
C      NORMALIZING THE EIGENVECTOR.
C
   30 CONTINUE
      APP = 0.0D0
      DO 40 I = 1,N
        APP = APP + X(I)*X(I)
   40 CONTINUE
      APP = SQRT(APP)
      DO 50 I = 1,N
        X(I) = X(I)/APP
   50 CONTINUE
C
   60 CONTINUE
C
      END       
c***********************************************************************
      SUBROUTINE INJWBK(E,L,J)
C
C  COMPUTES ARRAY D WHICH IS HELD IN COMMON/CJWBK/ AND
C  USED FOR CALCULATION OF JWBK FUNCTIONS.
C
      IMPLICIT REAL*8(A-H,O-Z)
      PARAMETER (MZCHF= 395,MX15N=15*MZCHF)
      COMMON/CJWBK/D(MX15N)
      COMMON/CACC/AC,ACNUM,ACJWBK,ACZP,LACC
C
      D(J+1)=E
C
c     IF(L.GT.0)GOTO 10
      IF(L.LE.0) THEN
C
C  CASE OF L.EQ.0
        D(J+4)=0.
        IF(E.EQ.0)GOTO 30
C  CASE OF L.EQ.0 AND E.GT.0
        FK=SQRT(E)
        D(J+2)=FK
        D(J+3)=1./FK
c       GOTO 30
c  10      IF(E.GT.0)GOTO 20
      ELSE IF(E.LE.0) THEN
C
C  CASE OF L.GT.0 AND E.EQ.0
        C=L*(L+1)
        D(J+4)=C
        SC=SQRT(C)
        D(J+5)=SC
        D(J+6)=(C+.125)/SC
        D(J+13)=6.*C
        D(J+14)=-C*C
c       GOTO 30
c  20   CONTINUE
      ELSE
C
C  CASE OF L.GT.0 AND E.GT.0
        FK=SQRT(E)
        D(J+2)=FK
        D(J+3)=1./FK
        C=L*(L+1)
        D(J+4)=C
        SC=SQRT(C)
        D(J+5)=SC
        D(J+6)=(C+.125)/SC
        A=1.+E*C
        D(J+7)=A
        A=3.*A
        D(J+8)=A-1.
        D(J+9)=A+1.
        D(J+10)=FK*C
        D(J+11)=-4.*E
        D(J+12)=-9.+2.*A
        D(J+13)=6.*C
        D(J+14)=-C*C
      ENDIF
C
C  TERM IN ARG GAMMA ETC
   30 D(J+15)=ARGC(E,L,AC)
C
      RETURN
      END
c***********************************************************************
      SUBROUTINE INV(A,N)
C
C  MATRIX INVERSION WITH FULL PIVOTING
C
      IMPLICIT REAL*8(A-H,O-Z)
      PARAMETER (MZCHF= 395)
      DIMENSION IPIVOT(MZCHF),A(MZCHF,MZCHF),INDEX(MZCHF,2),PIVOT(MZCHF)
C
C
C  INITIALIZATION
   10 DETERM=1.
   15 DO 20 J=1,N
   20 IPIVOT(J)=0
   30 DO 550 I=1,N
C  SEARCH FOR PIVOT ELEMENT
   40 AMAX=0.
   45 DO 105 J=1,N
   50 IF(IPIVOT(J)-1)60,105,60
   60 DO 100 K=1,N
   70 IF(IPIVOT(K)-1)80,100,740
   80 IF(ABS(AMAX)-ABS(A(J,K)))85,100,100
   85 IROW=J
   90 ICOLUM=K
   95 AMAX=A(J,K)
  100 CONTINUE
  105 CONTINUE
  110 IPIVOT(ICOLUM)=IPIVOT(ICOLUM)+1
C  INTERCHANGE ROWS TO PUT PIVOT ELEMENT ON DIAGONAL
  130 IF(IROW-ICOLUM)140,260,140
  140 DETERM=-DETERM
  150 DO 200 L=1,N
  160 SWAP=A(IROW,L)
  170 A(IROW,L)=A(ICOLUM,L)
  200 A(ICOLUM,L)=SWAP
  260 INDEX(I,1)=IROW
  270 INDEX(I,2)=ICOLUM
  310 PIVOT(I)=A(ICOLUM,ICOLUM)
C  DIVIDE PIVOT ROW BY PIVOT ELEMENT
  330 A(ICOLUM,ICOLUM)=1.
  340 DO 350 L=1,N
  350 A(ICOLUM,L)=A(ICOLUM,L)/PIVOT(I)
C  REDUCE NON-PIVOT ROWS
  380 DO 550 L1=1,N
  390 IF(L1-ICOLUM)400,550,400
  400 T=A(L1,ICOLUM)
  420 A(L1,ICOLUM)=0.
  430 DO 450 L=1,N
  450 A(L1,L)=A(L1,L)-A(ICOLUM,L)*T
  550 CONTINUE
C  INTERCHANGE COLUMNS
  600 DO 710 I=1,N
  610 L=N+1-I
  620 IF(INDEX(L,1)-INDEX(L,2))630,710,630
  630 JROW=INDEX(L,1)
  640 JCOLUM=INDEX(L,2)
  650 DO 700 K=1,N
  660 SWAP=A(K,JROW)
  670 A(K,JROW)=A(K,JCOLUM)
  700 A(K,JCOLUM)=SWAP
  710 CONTINUE
  740 RETURN
      END      
c***********************************************************************
      SUBROUTINE JWBK(R,J,S,SP,C,CP)
C
C  COMPUTES FUNCTIONS S AND C AND THEIR DERIVATIVES SP AND
C  CP USING IJWBK METHOD.
C  USES DATA IN ARRAY D WHICH IS HELD IN COMMON/CJWBK/
C  AND SHOULD HAVE BEEN COMPUTED IN SUBROUTINE INJWBK.
C
      IMPLICIT REAL*8(A-H,O-Z)
      PARAMETER (MZCHF= 395,MX15N=15*MZCHF)
      COMMON/CJWBK/D(MX15N)
      COMMON/CACC/AC,ACNUM,ACJWBK,ACZP,LACC
C
      E=D(J+1)
      C=D(J+4)
      X=1./R
C
c     IF(C.EQ.0)GOTO 30
      IF(C.NE.0) THEN
        IF(E.EQ.0)GOTO 70
C
C  CASE OF C.GT.0 AND E.GT.0
        W=E+X*(2.-C*X)
        WH=SQRT(W)
        Z=R*WH
        FK=D(J+2)
        RK=R*FK
        RMC=R-C
        ALP=Z+RK
        CK=D(J+10)
C  COMPUTE PHASE
        P=Z+D(J+15)
C  LOG TERM
        B=FK*ALP
c       IF(B.GT.ACJWBK)GOTO 10
        IF(B.LE.ACJWBK) THEN
          B=-B
          P=P+ALP*((((.2*B+.25)*B+.33333333)*B+.5)*B+1.)
c       GOTO 20
c  10   CONTINUE
        ELSE
          P=P+D(J+3)*LOG(1.+B)
        ENDIF
C  ARCTAN TERM
   20   S=D(J+5)*(Z-FK*RMC)
        G=CK*Z+RMC
        P=P+D(J+6)*ATAN2(S,G)
C  CAP. PHI TERM
        P=P+((5.*RMC/(Z*Z))-(Z*D(J+9)+RK*D(J+8)+CK)/
     C (ALP*D(J+7)))/(24.*Z)
C  COMPUTE AMPLITUDE
        A1=.0625*(X/W)**3
        CC=A1*(((D(J+14)*X+D(J+13))*X+D(J+12))*X+D(J+11))
        BB=A1*(((6.*D(J+14)*X+5.*D(J+13))*X+4.*D(J+12))
     1   *X+3.*D(J+11))
        GOTO 100
      ENDIF
C
c  30 IF(E.EQ.0)GOTO 60
      IF(E.NE.0) THEN
C
C  CASE OF C.EQ.0 AND E.GT.0
        W=2.*X+E
        WH=SQRT(W)
        Z=R*WH
        FK=D(J+2)
        RK=R*FK
        ALP=Z+RK
C  COMPUTE PHASE
        P=Z+D(J+15)
        B=FK*ALP
c       IF(B.GT.ACJWBK)GOTO 40
        IF(B.LE.ACJWBK) THEN
          B=-B
          P=P+ALP*((((.2*B+.25)*B+.3333333333)*B+.5)*B+1.)
c       GOTO 50
c  40   CONTINUE
        ELSE
          P=P+D(J+3)*LOG(1.+B)
        ENDIF
   50   P=P+1/(4.*ALP)+(5.*R/(Z*Z)-2.*(Z+ALP)/ALP)/(24.*Z)
C  COMPUTE AMPLITUDE
        A1=.0625*(X/W)**3
        CC=A1*(-4.*E-3.*X)
        BB=-12.*A1*(E+X)
        GOTO 100
      ENDIF
C
C  CASE OF C.EQ.0 AND E.EQ.0
   60 W=2.*X
      WH=SQRT(W)
      Z=R*WH
      P=2.*Z*(1.+.046875*X)+D(J+15)
      WMQ=1./SQRT(WH)
      ET=(1.+.0234375*X)*WMQ
      ZET=(1.-.046875*X)*WH
      ETP=.25*(1.-.0703125*X)*X*WMQ
      GOTO 110
C
C  CASE OF E.EQ.0 AND C.GT.0
   70 W=X*(2.-C*X)
      WH=SQRT(W)
      Z=R*WH
      RMC=R-C
C  COMPUTE PHASE
      P=2.*Z+D(J+15)
      S=D(J+5)*Z
      P=P+D(J+6)*ATAN2(S,RMC)
      P=P-(3.*R+C)/(24.*(RMC+R)*Z)
C  COMPUTE AMPLITUDE
      A1=.0625*(X/W)**3
      CC=((D(J+14)*X+D(J+13))*X-3.)*X*A1
      BB=((6.*D(J+14)*X+5*D(J+13))*X-12.)*X*A1
C
C  COMPLETE CALCULATION OF S,SP,C AND CP
 100  WMQ=1./SQRT(WH)
      ET=(1.-CC)*WMQ
      ETP=(.5*(X*X/W)*(1.-C*X)*(1.-13.*CC)+X*BB)*WMQ
      ZET=(1.+2.*CC)*WH
  110 SI=SIN(P)
      CO=COS(P)
      S=ET*SI
      C=ET*CO
      SP=ETP*SI+C*ZET
      CP=ETP*CO-S*ZET
C
      RETURN
      END
c***********************************************************************
      SUBROUTINE KDIAG(ITER,IPERT,ETOT,EIG,EIGSUM,EIGSUD,X1,XX)
C
c  Calculate at energy ETOT: K-matrix(RK), eigenvalues(EIG), vectors(XX),
c  eigenphase sum(EIGSUM), derivative(EIGSUD), partial width and ID (X1).
C
      IMPLICIT REAL*8(A-H,O-Z)
C
      PARAMETER (MZCHF= 395,MZLMX=   4,MZLP1=  40,MZMNP=5700,MZPTS= 600,
     1 MZTAR= 200,MXA=(MZCHF*(MZCHF+1))/2,MZCHF2=MZCHF*MZCHF)      
      PARAMETER (EPSI=1.0d-9,PI=3.141592654)
C
      COMMON/CHAN/ECH(MZCHF),LLCH(MZCHF),EPS(MZCHF),FKNU(MZCHF),
     1 CCT(MZCHF),RINF(MZCHF),ITARG(MZCHF),NCHF,NCHOP,NCHOP1
      COMMON/CINPUT/
     1 LAMAX,LRANG2,LRGL2,MNP2,NAST,NCHAN,NELC,NPTY2,NSPN2,NZED,MORE2,
     2 ISAT(MZTAR),LAT(MZTAR),L2P(MZCHF),NCONAT(MZTAR),
     3 BSTO,RA,
     4 CF(MZCHF,MZCHF,MZLMX),COEFF(3,MZLP1),ENAT(MZTAR),
     5 WMAT(MZCHF,MZMNP),VALUE(MZMNP)
      COMMON/CRMAT/RMAT(MZCHF,MZCHF),RMATD(MZCHF,MZCHF),
     1 UMAT(MZCHF,MZCHF),GAMMA
C     
      DIMENSION UPPER(MXA),EIG(MZCHF),EIGD(MZCHF,MZCHF),EIGDDD(MZCHF),
     1 X1(MZCHF),DUMMY(MZCHF,9),XX(MZCHF,MZCHF),RK(MZCHF,MZCHF),
     2 DK(MZCHF,MZCHF),A(MZCHF,MZCHF),B(MZCHF,MZCHF),S(MZCHF),SP(MZCHF),
     3 C(MZCHF),CP(MZCHF)    
      DIMENSION Q(MZCHF,MZCHF),X(MZCHF),DUMX(MZCHF),DUMY(MZCHF)
      DIMENSION CS(MZCHF,MZCHF),CSP(MZCHF,MZCHF),CC(MZCHF,MZCHF),
     1 CCP(MZCHF,MZCHF)
      DIMENSION SD(MZCHF),SPD(MZCHF),CD(MZCHF),CPD(MZCHF)
      DIMENSION S1(MZCHF),SP1(MZCHF),C1(MZCHF),CP1(MZCHF)
      DIMENSION CSD(MZCHF,MZCHF),CSPD(MZCHF,MZCHF),CCD(MZCHF,MZCHF),
     1 CCPD(MZCHF,MZCHF)
      DIMENSION CS1(MZCHF,MZCHF),CSP1(MZCHF,MZCHF),CC1(MZCHF,MZCHF),
     1 CCP1(MZCHF,MZCHF)
C
      DATA CS1/MZCHF2*0.0/,CSP1/MZCHF2*0.0/,CSD/MZCHF2*0.0/,
     1 CSPD/MZCHF2*0.0/
      DATA CC1/MZCHF2*0.0/,CCP1/MZCHF2*0.0/,CCD/MZCHF2*0.0/,
     1 CCPD/MZCHF2*0.0/
      DATA S1/MZCHF*0.0/,SP1/MZCHF*0.0/,SD/MZCHF*0.0/,SPD/MZCHF*0.0/
      DATA C1/MZCHF*0.0/,CP1/MZCHF*0.0/,CD/MZCHF*0.0/,CPD/MZCHF*0.0/
      DATA E1/0.0/
C
      SAVE S1,SP1,SD,SPD,C1,CP1,CD,CPD,E1
      SAVE CS1,CSP1,CSD,CSPD,CC1,CCP1,CCD,CCPD
C
C WHERE: S1=REGULAR FUNCTION FOR OPEN CHANNELS, DECAYING FUNCTION FOR CLOSED
C        SD=ENERGY DERIVATIVE OF S1
C        SP1=DERIVATIVE WRT r
C        SPD=ENERGY DERIVATIVE OF SP1 
C        C1=IRREGULAR FUNCTION FOR OPEN CHANNELS
C        CD=ENERGY DERIVATIVE OF C1
C        CP1=DERIVATIVE WRT r
C        CPD=ENERGY DERIVATIVE OF CP1
C
C  CALCULATE CHANNEL ENERGIES EPS, NUMBER OF OPEN CHANNELS NCHOP
C
      NCHOP=0
      DO I=1,NCHF
        EPS(I)=ETOT-ECH(I)
        IF(EPS(I).GE.0.0) NCHOP=NCHOP+1
      END DO
      NCHOP1=NCHOP+1
C
C  CALCULATE R-MATRIX AND ITS ENERGY DERIVATIVE
C
      CALL RMATB2
      IF (ITER.LE.1) THEN
C
C  CALCULATE ASYMPTOTIC SOLUTIONS AT RA.
C
        IF (NZED.NE.NELC) THEN
          CALL COUL1(IPERT,NCHF,NCHOP,BSTO,CS,CSP,CC,CCP)
        ELSE
          CALL ASY(IPERT,NCHF,NCHOP,EPS,CS,CSP,CC,CCP)
        ENDIF
C
C  WHEN NO PERTURBATION, COPY DIAGONAL INTO S,C AND SP,CP
C
        IF(IPERT.EQ.0) THEN
          DO I=1,NCHF
            S(I)=CS(I,I)
            SP(I)=CSP(I,I)
            C(I)=CC(I,I)
            CP(I)=CCP(I,I)
          END DO
        ENDIF
C
C CALCULATE d/dE OF S,C AND SP,CP FROM ADJACENT VALUES - linear
C
        IF (ITER.GE.0) THEN
          denom=1.0/(E1-ETOT)
          DO J=1,NCHF
            IF (IPERT.GT.0) THEN
              DO I=1,NCHF
                CSD(I,J)=(CS1(I,J)-CS(I,J))*denom
                CCD(I,J)=(CC1(I,J)-CC(I,J))*denom
                CSPD(I,J)=(CSP1(I,J)-CSP(I,J))*denom
                CCPD(I,J)=(CCP1(I,J)-CCP(I,J))*denom
              END DO
            ELSE
              SD(J)=(S1(J)-S(J))*denom
              CD(J)=(C1(J)-C(J))*denom
              SPD(J)=(SP1(J)-SP(J))*denom
              CPD(J)=(CP1(J)-CP(J))*denom
            ENDIF
          END DO
C
C NORMALISE closed channel d/dE
C
          DO J=NCHOP+1,NCHF
            IF(IPERT.GT.0) THEN
              DO I=NCHOP+1,NCHF
                sn=1.0/(CS(I,J)*CSPD(I,J)-CSP(I,J)*CSD(I,J))
                cn=1.0/(CC(I,J)*CCPD(I,J)-CCP(I,J)*CCD(I,J))
                CSD(I,J)=CSD(I,J)*sn
                CCD(I,J)=CCD(I,J)*cn
                CSPD(I,J)=CSPD(I,J)*sn
                CCPD(I,J)=CCPD(I,J)*cn
                CS1(I,J)=CS(I,J)
                CC1(I,J)=CC(I,J)
                CSP1(I,J)=CSP(I,J)
                CCP1(I,J)=CCP(I,J)
              END DO
            ELSE
              sn=1.0/(S(J)*SPD(J)-SP(J)*SD(J))
              cn=1.0/(C(J)*CPD(J)-CP(J)*CD(J))
              SD(J)=SD(J)*sn
              CD(J)=CD(J)*cn
              SPD(J)=SPD(J)*sn
              CPD(J)=CPD(J)*cn
              S1(J)=S(J)
              C1(J)=C(J)
              SP1(J)=SP(J)
              CP1(J)=CP(J)
            ENDIF
          END DO
        ENDIF
C
C STORE CURRENT VALUES OF S,C AND SP,CP FOR NEXT ENERGY
C
        DO J=1,NCHF
          IF(IPERT.GT.0) THEN
            DO I=1,NCHF
              CS1(I,J)=CS(I,J)
              CC1(I,J)=CC(I,J)
              CSP1(I,J)=CSP(I,J)
              CCP1(I,J)=CCP(I,J)
            END DO
          ELSE
            S1(J)=S(J)
            C1(J)=C(J)
            SP1(J)=SP(J)
            CP1(J)=CP(J)
          ENDIF
        END DO
        E1=ETOT
C
C  CALCULATE ASYMPTOTIC SOLUTIONS FROM LAST VALUE ASSUMING SAME d/dE
C
      ELSE
        edif=E1-ETOT
        DO I=1,NCHF
          IF(IPERT.GT.0)THEN
            DO J=1,NCHF
              CS(I,J)=CS1(I,J)-CSD(I,J)*edif
              CC(I,J)=CC1(I,J)-CCD(I,J)*edif
              CSP(I,J)=CSP1(I,J)-CSPD(I,J)*edif
              CCP(I,J)=CCP1(I,J)-CCPD(I,J)*edif
            END DO
          ELSE
            S(I)=S1(I)-SD(I)*edif
            C(I)=C1(I)-CD(I)*edif
            SP(I)=SP1(I)-SPD(I)*edif
            CP(I)=CP1(I)-CPD(I)*edif
          ENDIF
        END DO
      ENDIF
C
C..CASE OF IPERT.GT.0
C  CALCULATE MATRICES A AND B
      IF(IPERT.GT.0)THEN
        DO I=1,NCHF
          DO K=1,NCHF
            DUMX(K)=RMAT(I,K)
          END DO
          DO J=1,NCHF
            A(I,J)=CC(I,J)
            IF (J.LE.NCHOP) THEN
              B(I,J)=CS(I,J)
              DO K=1,NCHF
                A(I,J)=A(I,J)-DUMX(K)*CCP(K,J)
                B(I,J)=B(I,J)-DUMX(K)*CSP(K,J)
              END DO
            ELSE
              DO K=1,NCHF
                A(I,J)=A(I,J)-DUMX(K)*CCP(K,J)
              END DO
            ENDIF
          END DO
        END DO
      ELSE
C..CASE OF IPERT.LE.0
C  CALCULATE MATRICES A AND B
        DO J=1,NCHOP
          DO I=1,NCHF
            A(I,J)=-RMAT(I,J)*CP(J)
            B(I,J)=-RMAT(I,J)*SP(J)
          END DO
          A(J,J)=A(J,J)+C(J)
          B(J,J)=B(J,J)+S(J)
        END DO
        DO J=NCHOP1,NCHF
          DO I=1,NCHF
            A(I,J)=-RMAT(I,J)*SP(J)
          END DO
          A(J,J)=A(J,J)+S(J)
        END DO
      ENDIF
C
C  COMPLETE CALCULATION OF REACTANCE MATRIX
C
      CALL INV(A,NCHF)
      DO I=1,NCHF
        DO K=1,NCHF
          DUMX(K)=A(I,K)
        END DO
        DO J=1,NCHOP
          RK(I,J)=0.
          DO K=1,NCHF
            RK(I,J)=RK(I,J)-DUMX(K)*B(K,J)
          END DO
        END DO
      END DO
C
C  DIAGONALISE K-MATRIX
C
      LENGTH=0
      DO I=1,NCHOP
        DO J=I,NCHOP
          LENGTH=LENGTH+1
          UPPER(LENGTH)=RK(I,J)
        END DO
      END DO
      EIGSUM=0.
      DO I=1,NCHOP
        NO=I
        CALL HSLDR(NCHOP,UPPER,LENGTH,EPSI,EIG,X1,NO,DUMMY,MZCHF)
        DO J=1,NCHOP
          XX(J,I)=X1(J)
        END DO
C        print *,'Upper K'
C        print *,UPPER
C        print *,'diagonalised K EIG(',I,')=',EIG(I)
        EIGSUM=EIGSUM+ATAN(EIG(i))
        IF(EIGSUM.LT.0.) EIGSUM=EIGSUM+PI
        IF(EIGSUM.GT.PI) EIGSUM=EIGSUM-PI
      END DO
      IF (ITER.LT.0) RETURN
C
C CALCULATE THE Q-MATRIX
C
      IF(IPERT.GT.0)THEN
        DO I=1,NCHF
          DO K=1,NCHF
            DUMX(K)=RMATD(I,K)
            DUMY(K)=RMAT(I,K)
          END DO
          DO J=1,NCHF
            B(I,J)=-CCD(I,J)
            Q(I,J)=-CSD(I,J)
            DO K=1,NCHF
              B(I,J)=B(I,J)+DUMX(K)*CCP(K,J)+DUMY(K)*CCPD(K,J)
              Q(I,J)=Q(I,J)+DUMX(K)*CSP(K,J)+DUMY(K)*CSPD(K,J)
            END DO
          END DO
        END DO
      ELSE 
        DO J=1,NCHF
          DO I=1,NCHF
            B(I,J)=RMATD(I,J)*CP(J)+RMAT(I,J)*CPD(J)
            Q(I,J)=RMATD(I,J)*SP(J)+RMAT(I,J)*SPD(J)
          END DO
          B(J,J)=B(J,J)-CD(J)
          Q(J,J)=Q(J,J)-SD(J)
        END DO
      ENDIF
C
C COMPLETE THE Q-MATRIX
C
      DO I=1,NCHF
        DO K=1,NCHF
          DUMX(K)=B(I,K)
        END DO
        DO J=1,NCHOP
          DO K=1,NCHF
            Q(I,J)=Q(I,J)+DUMX(K)*RK(K,J)
          END DO
        END DO
      END DO

C
C SOLVE THE MATRIX EQUATION BK'=Q WHERE K'=DK(I,J)
C
      DO I=1,NCHF
        DO K=1,NCHF
          DUMY(K)=A(I,K)
        END DO
        DO J=1,NCHOP
          DK(I,J)=0.0
          DO K=1,NCHF
            DK(I,J)=DK(I,J)+DUMY(K)*Q(K,J)
          END DO
        END DO
      END DO
C
C CALCULATE THE DERIVATIVE OF THE EIGENVECTORS
C
      IF (IPERT.EQ.0) THEN
        DO I=1,NCHOP
          ED=0.
          DO L=1,NCHOP
            XLI=XX(L,I)
            DO K=1,NCHOP
              ED=ED+XX(K,I)*DK(K,L)*XLI
            END DO
          END DO
          EIGD(I,I)=ED
        END DO
      ELSE
C
        DO I=1,NCHOP
          DO J=1,NCHOP
            DUMX(J)=0.
            DO K=1,NCHOP
              DUMX(J)=DUMX(J)+XX(K,I)*DK(K,J)
            END DO
          END DO
          DO J=1,NCHOP
            EIGD(I,J)=0.
            DO K=1,NCHOP
              EIGD(I,J)=EIGD(I,J)+DUMX(K)*XX(K,J)
            END DO
          END DO
        END DO
      ENDIF
C
C CALCULATE THE DERIVATIVE OF THE EIGENPHASE SUM
C
      EIGSUD=0.
      DO I=1,NCHOP
        EIGDDD(I)=EIGD(I,I)/(1.0+EIG(I)*EIG(I))
        EIGSUD=EIGSUD+EIGDDD(I)
      END DO
      IF (ITER.LE.0) RETURN
c
c  Normalize partial widths X1(i),i=1,NCHOP
c
      DO I=1,NCHOP
        X1(I)=0.
        DO J=1,NCHOP
          X1(I)=X1(I)+XX(I,J)*XX(I,J)*EIGDDD(J)
        END DO
        X1(I)=X1(I)/EIGSUD
      END DO
C
C RESONANCE IDENTIFICATION
C
C OUTER REGION NORMALISATION:-
C
      DO J=1,NCHF
        DUMX(J)=0.
        IF(IPERT.GT.0) THEN
          DO L=1,NCHOP
            IF(J.LE.NCHOP) DUMX(J)=DUMX(J)-CS(J,L)*X1(L)
            DO K=1,NCHOP
              DUMX(J)=DUMX(J)+RMAT(J,K)*CSP(K,L)*X1(L)
            END DO
          END DO
        ELSE
          IF(J.LE.NCHOP) DUMX(J)=-S(J)*X1(J)
          DO K=1,NCHOP
            DUMX(J)=DUMX(J)+RMAT(J,K)*SP(K)*X1(K)
          END DO
        ENDIF 
      END DO
C 
      XSQ=0.0
      ASQ=0.0
      DO I=1,NCHF
        X(I)=0.0
        DO J=1,NCHF
          X(I)=X(I)+A(I,J)*DUMX(J)
        END DO
        IF(I.GT.NCHOP) XSQ=XSQ+X(I)**2
C
C INNER REGION:-
C
        DUMY(I)=0.0
        IF(IPERT.GT.0) THEN
          DO L=1,NCHOP
            IF(I.LE.NCHOP) DUMY(I)=DUMY(I)+CSP(I,L)*X1(L)
            DUMY(I)=DUMY(I)+CCP(I,L)*X(L)
          END DO
        ELSE
          IF(I.LE.NCHOP) DUMY(I)=SP(I)*X1(I)
          DUMY(I)=DUMY(I)+CP(I)*X(I)
        ENDIF  
      END DO
      DO J=1,NCHF
        DO I=1,NCHF
          ASQ=ASQ+DUMY(I)*RMATD(I,J)*DUMY(J)
        END DO
      END DO
C
C  Normalise closed channel weights X1(i),i=1,NCHOP+1,NCHF
C
      IF (ASQ.LE.0.0) ASQ=0.
      XNORM=1.0/SQRT(ASQ+XSQ)
      DO I=NCHOP+1,NCHF
        X1(I)=X(I)*XNORM
      END DO
C   
      RETURN
      END      
c***********************************************************************
      SUBROUTINE NORM(N,X)
C
C      NORMALISES THE VECTOR X OF DIMENSION N
C      SUCH THAT THE LARGEST COMPONENT IS UNITY.
C
      IMPLICIT REAL*8(A-H,O-Z)
      DIMENSION X(N)
C
      G = ABS(X(1))
      DO 10 I = 2,N
        GA = ABS(X(I))
        IF (GA.GT.G) G = GA
   10 CONTINUE
      DO 20 I = 1,N
        X(I) = X(I)/G
   20 CONTINUE
C
      END       
c***********************************************************************
      SUBROUTINE NUMS(E,C,R1,HP,N1,N2,F,FP,FST)
C
C  INSERTED 21.11.85
C
C  NUMEROV INTEGRATION OF COULOMB FUNCTION F.
C  INTEGRATION FROM POINT N1 TO N2.
C  INTERVAL HP IS POSITIVE.
C  INPUT FUNCTIONS F,FP AT N1 WHERE FP IS R-DERIVATIVES
C  FUNCTIONS AT INTERMEDIATE POINTS STORED IN FST, FDST
C  OUTPUT FUNCTIONS F,FP,FD,FDP AT N2.
C
      IMPLICIT REAL*8(A-H,O-Z)
C
      PARAMETER (MZPTS= 600)
      DIMENSION FST(MZPTS)
C
      V(X)=EQ+X*(Q2-X*CQ)
C
      N21=N2-N1
C
      F1=F
      F1P=FP
      FST(N1)=F1
      IF(N21.EQ.0)RETURN
C
      IP=IABS(N21)
      IS=N21/IP
      H=HP*IS
      K=N1
      IP=IP-1
C
C
C  FUNCTIONS AT K=(N1+IS)
      Q=H*H
      EQ=E*Q
      Q2=2.*Q
      CQ=C*Q
C
      X1=1./R1
      CX=C*X1
      HX=H*X1
      A1=-2.*HX*HX*H
      U1P=A1*(1.-CX)
      A1=-A1*HX
      U1PP=A1*(2.-3.*CX)
      A1=-A1*HX*6.
      U1PPP=A1*(1.-2.*CX)
      U1=V(X1)
C
      R2=R1+H
      X2=1./R2
      U2=V(X2)
C
      A2=1.+U2*.03333333333
      B1=1.+U1*(.025*U1-.4666666667)-.1333333333*U1P-.025*U1PP
     1 +.002777777778*(4.*U1*U1P-U1PPP)
      C1=H*(1.+U1*(.002777777778*U1-.1333333333)
     1 -.05*U1P-.008333333333*U1PP)
      P2=.03333333333*Q
C
      F2=(B1*F1+C1*F1P)/A2
      K=K+IS
      FST(K)=F2
C
C
      U3=U2
      U2=U1
      F3=F2
      F2=F1
c     IF(IP.EQ.0)GOTO 20
      IF(IP.NE.0) THEN
C
C  CONTINUE INTEGRATION
        EQ=EQ*.08333333333
        Q2=Q2*.08333333333
        CQ=CQ*.08333333333
        U2=U2*.08333333333
        U3=U3*.08333333333
        Q=Q*.083333333333
        R3=R2
C
        DO 10 M=1,IP
          U1=U2
          U2=U3
          F1=F2
          F2=F3
          R3=R3+H
          X3=1./R3
          U3=V(X3)
          D3=1./(1.+U3)
          D2=(2.-10.*U2)*D3
          D1=(1.+U1)*D3
          F3=D2*F2-D1*F1
          K=K+IS
   10     FST(K)=F3
C
        U2=12.*U2
        U3=12.*U3
        EQ=12.*EQ
        Q2=12.*Q2
        CQ=12.*CQ
        Q=12.*Q
      ENDIF
C
C  CALCULATE FINAL DERIVATIVE
   20 H=-H
      CX=C*X3
      HX=H*X3
      A1=-2.*HX*HX*H
      U3P=A1*(1.-CX)
      A1=-A1*HX
      U3PP=A1*(2.-3.*CX)
      A1=-A1*HX*6.
      U3PPP=A1*(1.-2.*CX)
      A2=1.+U2*.03333333333
      B3=1.+U3*(.025*U3-.4666666667)-.1333333333*U3P-.025*U3PP
     1 +.002777777778*(4.*U3*U3P-U3PPP)
      C3=H*(1.+U3*(.002777777778*U3-.1333333333)-.05*U3P
     1 -.008333333333*U3PP)
      P2=.03333333333*Q
      F3P=(A2*F2-B3*F3)/C3
C
      F=F3
      FP=F3P
C
      RETURN
      END 
c***********************************************************************
      SUBROUTINE NUMSC(E,C,R1,H,N,I)
C
C  NUMEROV INTEGRATION OF COULOMB FUNCTIONS S AND C.
C  INTERVAL H, NUMBER OF POINTS N.
C  FUNCTIONS IN ARRAYS FS,FC
C  INWARDS INTEGRATION, FUNCTIONS CALCULATED FOR
C  K=N,(N-1),...,1.
C
C  MODIFIED TO PERFORM BOTH INWARD AND OUTWARD INTEGRATION DEPENDING
C  ON THE SIGN OF H: (H NEGATIVE)=INWARD; (H POSITIVE)=OUTWARD  1.5.86
C
      IMPLICIT REAL*8(A-H,O-Z)
      PARAMETER (MZCHF= 395,MZPTS= 600)
      COMMON/COULSC/FS(MZCHF,MZPTS),FSP(MZCHF),FC(MZCHF,MZPTS),
     1 FCP(MZCHF)
C
      V(X)=EQ+X*(Q2-X*CQ)
C
C  DETERMINE THE DIRECTION OF INTEGRATION AND SET INCREMENT
      IF(H.GT.0.0) THEN
        INC=1
        K=1
      ELSE
        INC=-1
        K=N
      ENDIF
      F1=FS(I,K)
      F1P=FSP(I)
      G1=FC(I,K)
      G1P=FCP(I)
      IF(N.EQ.1)RETURN
C
C  FUNCTIONS AT K=(K+INC)
      Q=H*H
      EQ=E*Q
      Q2=2.*Q
      CQ=C*Q
C
      X1=1./R1
      CX=C*X1
      HX=H*X1
      A1=-2.*HX*HX*H
      U1P=A1*(1.-CX)
      A1=-A1*HX
      U1PP=A1*(2.-3.*CX)
      A1=-A1*HX*6.
      U1PPP=A1*(1.-2.*CX)
      U1=V(X1)
C
      R2=R1+H
      X2=1./R2
      U2=V(X2)
C
      A2=1.+U2*.03333333333
      B1=1.+U1*(.025*U1-.4666666667)-.1333333333*U1P-.025*U1PP
     1 +.002777777778*(4.*U1*U1P-U1PPP)
      C1=H*(1.+U1*(.002777777778*U1-.1333333333)
     1 -.05*U1P-.008333333333*U1PP)
C
      F2=(B1*F1+C1*F1P)/A2
      G2=(B1*G1+C1*G1P)/A2
      K=K+INC
      FS(I,K)=F2
      FC(I,K)=G2
C
      U3=U2
      U2=U1
      F3=F2
      F2=F1
      G3=G2
      G2=G1
c     IF(N.EQ.2)GOTO 20
      IF(N.NE.2) THEN
C
C  CONTINUE INTEGRATION
        EQ=EQ*.08333333333
        Q2=Q2*.08333333333
        CQ=CQ*.08333333333
        U2=U2*.08333333333
        U3=U3*.08333333333
        R3=R2
C
        DO 10 M=3,N
          U1=U2
          U2=U3
          F1=F2
          F2=F3
          G1=G2
          G2=G3
          R3=R3+H
          X3=1./R3
          U3=V(X3)
          D3=1./(1.+U3)
          D2=(2.-10.*U2)*D3
          D1=(1.+U1)*D3
          F3=D2*F2-D1*F1
          G3=D2*G2-D1*G1
          K=K+INC
          FS(I,K)=F3
   10     FC(I,K)=G3
C
        U2=12.*U2
        U3=12.*U3
        EQ=12.*EQ
        Q2=12.*Q2
        CQ=12.*CQ
      ENDIF
C 
C  CALCULATE FINAL DERIVATIVE
   20 IF(INC.EQ.1) RETURN
      H=-H
      CX=C*X3
      HX=H*X3
      A1=-2.*HX*HX*H
      U3P=A1*(1.-CX)
      A1=-A1*HX
      U3PP=A1*(2.-3.*CX)
      A1=-A1*HX*6.
      U3PPP=A1*(1.-2.*CX)
      A2=1.+U2*.03333333333
      B3=1.+U3*(.025*U3-.4666666667)-.1333333333*U3P-.025*U3PP
     1 +.002777777778*(4.*U3*U3P-U3PPP)
      C3=H*(1.+U3*(.002777777778*U3-.1333333333)-.05*U3P
     1 -.008333333333*U3PP)
      FSP(I)=(A2*F2-B3*F3)/C3
      FCP(I)=(A2*G2-B3*G3)/C3
      H=-H
C
      RETURN
      END
c***********************************************************************
      SUBROUTINE NUMT(E,C,R1,HP,N1,N2,I)
C
C  NUMEROV INTEGRATION OF COULOMB FUNCTIONS.
C  THE INTERVAL HP IS POSITIVE.
C  INTEGRATION FROM TABULAR POINT N1 TO TABULAR POINT N2.
C  FUNCTIONS THETA,THETP STORED IN FS,FSP
C  FUNCTIONS THETAD,THETADP STORED IN FC,FCP
C  STARTS WITH FUNCTIONS AND DERIVATIVES AT N1 STORED IN FS, FSP
C  CALCULATES FUNCTIONS AT ALL POINT TO N2 AND DERIVATIVE AT
C  THE POINT N2.
C
      IMPLICIT REAL*8(A-H,O-Z)
      PARAMETER (MZCHF= 395,MZPTS= 600,MZTET=  50)
      COMMON/COULSC/FS(MZCHF,MZPTS),FSP(MZCHF),FC(MZCHF,MZPTS),
     1 FCP(MZCHF)
      COMMON/CTHET/BB(MZCHF,MZTET),BG(MZCHF,MZTET),MSUM(MZCHF)
C
      V(X)=EQ+X*(Q2-X*CQ)
C
      N21=N2-N1
C
C  RENORMALISE FOR CASE OF N2.EQ.N1
c     IF(N21.NE.0)GOTO 5
      IF(N21.EQ.0) THEN
        W=FS(I,N1)*FCP(I)-FSP(I)*FC(I,N1)
        W1=1./SQRT(W)
        FS(I,N1)=FS(I,N1)*W1
        FC(I,N1)=FC(I,N1)*W1
        FSP(I)=FSP(I)*W1
        FCP(I)=FCP(I)*W1
        BB(I,2)=BB(I,2)*W1
        NM=MSUM(I)
        DO 2 M=3,NM
          BB(I,M)=BB(I,M)*W1
    2     BG(I,M)=BG(I,M)*W1
        RETURN
      ENDIF
C
C  INTEGRATIONS FOR N2.NE.N1
   5  IP=IABS(N21)
      IS=N21/IP
      H=HP*IS
      K=N1
      IP=IP-1
C
      F1=FS(I,N1)
      F1P=FSP(I)
      G1=FC(I,N1)
      G1P=FCP(I)
C
C  FUNCTIONS AT K=(N1+IS)
      Q=H*H
      EQ=E*Q
      Q2=2.*Q
      CQ=C*Q
C
      X1=1./R1
      CX=C*X1
      HX=H*X1
      A1=-2.*HX*HX*H
      U1P=A1*(1.-CX)
      A1=-A1*HX
      U1PP=A1*(2.-3.*CX)
      A1=-A1*HX*6.
      U1PPP=A1*(1.-2.*CX)
      U1=V(X1)
C
      R2=R1+H
      X2=1./R2
      U2=V(X2)
C
      A2=1.+U2*.03333333333
      B1=1.+U1*(.025*U1-.4666666667)-.1333333333*U1P-.025*U1PP
     1 +.002777777778*(4.*U1*U1P-U1PPP)
      C1=H*(1.+U1*(.002777777778*U1-.1333333333)
     1 -.05*U1P-.008333333333*U1PP)
      P2=.03333333333*Q
      D1=Q*(-.4666666667+.025*U1+.025*U1+.002777777778*U1P)
      E1=H*Q*(-.1333333333+.002777777778*U1)
C
      F2=(B1*F1+C1*F1P)/A2
      G2=(B1*G1+C1*G1P+D1*F1+E1*F1P-P2*F2)/A2
      K=K+IS
      FS(I,K)=F2
      FC(I,K)=G2
C
C
      U3=U2
      U2=U1
      F3=F2
      F2=F1
      G3=G2
      G2=G1
c     IF(IP.EQ.0)GOTO 20
      IF(IP.NE.0) THEN
C
C  CONTINUE INTEGRATION
        EQ=EQ*.08333333333
        Q2=Q2*.08333333333
        CQ=CQ*.08333333333
        U2=U2*.08333333333
        U3=U3*.08333333333
        Q=Q*.083333333333
        R3=R2
C
        DO 10 M=1,IP
          U1=U2
          U2=U3
          F1=F2
          F2=F3
          G1=G2
          G2=G3
          R3=R3+H
          X3=1./R3
          U3=V(X3)
          D3=1./(1.+U3)
          D2=(2.-10.*U2)*D3
          D1=(1.+U1)*D3
          F3=D2*F2-D1*F1
          G3=D2*G2-D1*G1-Q*D3*(F3+10.*F2+F1)
          K=K+IS
          FC(I,K)=G3
   10     FS(I,K)=F3
C
        U2=12.*U2
        U3=12.*U3
        EQ=12.*EQ
        Q2=12.*Q2
        CQ=12.*CQ
        Q=12.*Q
      ENDIF
C
C  CALCULATE FINAL DERIVATIVE
   20 H=-H
      CX=C*X3
      HX=H*X3
      A1=-2.*HX*HX*H
      U3P=A1*(1.-CX)
      A1=-A1*HX
      U3PP=A1*(2.-3.*CX)
      A1=-A1*HX*6.
      U3PPP=A1*(1.-2.*CX)
      A2=1.+U2*.03333333333
      B3=1.+U3*(.025*U3-.4666666667)-.1333333333*U3P-.025*U3PP
     1 +.002777777778*(4.*U3*U3P-U3PPP)
      C3=H*(1.+U3*(.002777777778*U3-.1333333333)-.05*U3P
     1 -.008333333333*U3PP)
      P2=.03333333333*Q
      D3=Q*(-.4666666667+.025*U3+.002777777778*U3P)
      E3=H*Q*(-.1333333333+.002777777777*U3)
      F3P=(A2*F2-B3*F3)/C3
      G3P=(A2*G2-B3*G3+P2*F2-D3*F3-E3*F3P)/C3
C
C  RE-NORMALISE CLOSED-CHANNEL FUNCTIONS
C ==  MODIFICATION FOR VAX CALCULATION OF
C ==  W1=1./SQRT(F3*G3P-F3P*G3)
      AMAX=1./MAX(ABS(F3),ABS(G3),ABS(F3P),ABS(G3P))
      AF3=F3*AMAX
      AG3=G3*AMAX
      AF3P=F3P*AMAX
      AG3P=G3P*AMAX
      W1=AMAX/SQRT(AF3*AG3P-AF3P*AG3)
C == END MODIFICATION
      FSP(I)=F3P*W1
      FCP(I)=G3P*W1
      IP=IP+2
      DO 30 J=1,IP
        FS(I,J)=FS(I,J)*W1
   30   FC(I,J)=FC(I,J)*W1
C  RE-NORMALISE COEFFICIENTS
      BB(I,2)=BB(I,2)*W1
      NM=MSUM(I)
      DO 40 M=3,NM
        BB(I,M)=BB(I,M)*W1
   40   BG(I,M)=BG(I,M)*W1
C
      RETURN
      END
c***********************************************************************
      SUBROUTINE OUTRES(KOUNT,RES,WIDTH,IDECAY,IWT,ITARG,NCHOPX,ECH,BDX)
c
c  Output C-file for RES(k),WIDTH(k) = position,width of 
c  resonances k=1,KOUNT converging to channel NCHOPX threshold.
c
C
      IMPLICIT REAL*8(A-H,O-Z)
      PARAMETER (MZCHF= 395,MZEST= 200,MZLMX=   4,MZLP1=  40,
     1 MZMNP=5700,MZTAR= 200)
      PARAMETER (PI=3.141592654,EV=13.605)
      PARAMETER (SMALLN=0.1)
      LOGICAL OK,OK2,km(MZCHF)
      CHARACTER*1 COLON(MZEST)
      INTEGER IDECAY(KOUNT),IWT(2,KOUNT),ITARG(MZCHF)
      INTEGER NEFF(MZCHF),KCHAN(MZEST)
      DIMENSION RES(KOUNT),WIDTH(KOUNT),ECH(MZCHF)
      DIMENSION EFLAST(MZCHF)
      COMMON/CINPUT/
     1 LAMAX,LRANG2,LRGL2,MNP2,NAST,NCHAN,NELC,NPTY2,NSPN2,NZED,MORE2,
     2 ISAT(MZTAR),LAT(MZTAR),L2P(MZCHF),NCONAT(MZTAR),
     3 BSTO,RA,
     4 CF(MZCHF,MZCHF,MZLMX),COEFF(3,MZLP1),ENAT(MZTAR),
     5 WMAT(MZCHF,MZMNP),VALUE(MZMNP)
c
c  Initializing. NMIN=min(n) for rydberg electron (nl).
c
      ZSQ=MAX(NZED-NELC,1)**2
      NMIN=(NELC+45)/18
      IF(NELC.LE.1) NMIN=SQRT(0.25+2*NAST)+0.499
c
c  initialise channel data
c
      DO i=1,NCHAN
        NEFF(I)=MAX(NMIN,L2P(i)+1)
        EFLAST(i)=0.0
        ECH(I)=ZSQ*ENAT(ITARG(I))
c
c  check digit (km) for channel l duplication in fine-structure
c
        km(i)=.false.
        if(NSPN2.eq.0.and.i.gt.1) then
          do ii=1,i-1
          if(ITARG(i).eq.ITARG(ii).and.L2P(i).eq.L2P(ii)) km(i)=.true.
          end do
        endif
      END DO
      ECH1=ECH(NCHOPX)
c
c  Run backwards through a rydberg resonance series - the highest 
c  resonances should have the least ambiguous channel assignment based
c  on the external region weights (IWT).
c  KCHAN(i),EFLAST(i)=channel assignment,last effective n for channel i.
c  NMINL=min(n) for rydberg electron (nl) in each channel.
c
      DO k=KOUNT,1,-1
c
c  Give preference to channel corresponding to dominant weight
c
        ICHAN=ABS(IWT(1,k))
        EFFN=MAX(NZED-NELC,1)/SQRT(ECH(ICHAN)-RES(k))
        COLON(k)=' '
c
c  If dominant weight is rather small (IWT<0), check neff against
c  that of next biggest weight
c
        IF (IWT(1,k).LT.0.AND.NCHOPX.NE.NCHAN) THEN
          DIF=ABS(EFLAST(ICHAN)-EFFN-1)
          NMINL=MAX(NMIN,L2P(ICHAN)+1)
          ICHAN2=ABS(IWT(2,k))
          EFFN2=MAX(NZED-NELC,1)/SQRT(ECH(ICHAN2)-RES(k))
          DIF2=ABS(EFLAST(ICHAN2)-EFFN2-1)
          NMIN2=MAX(NMIN,L2P(ICHAN2)+1)
          OK=EFLAST(ICHAN).GT.0.0
          OK2=EFLAST(ICHAN2).GT.0.0
c
c  If next biggest weight appears better, use it and set colon
c
          IF ((OK.AND.OK2.AND.DIF.GT.DIF2).OR.
     *        (NINT(EFFN2+0.5).GE.NMIN2.AND.
     *         ((IWT(2,k).GT.0.AND.
     *         ((OK.AND.DIF.GT.SMALLN).OR.(OK2.AND.DIF2.LE.SMALLN)))
     *     .OR.NINT(EFFN+0.5).LT.NMINL))) THEN
            ICHAN=ICHAN2
            EFFN=EFFN2
            NMINL=NMIN2
            COLON(k)=':'
          END IF
        END IF
c
c  Store channel and neff for this resonance
c
        EFLAST(ICHAN)=EFFN
        NEFF(ICHAN)=NEFF(ICHAN)-1
        IF(L2P(ICHAN).EQ.0) EFFN=EFFN+0.5
        IF(NEFF(ICHAN).LT.NMINL) NEFF(ICHAN)=MAX(NINT(EFFN),NMINL)
        KCHAN(k)=ICHAN
      END DO
c
c  Check that the n assigned to lowest member of series is not too low
c
      DO i=1,NCHAN
        NMINL=MAX(NMIN,L2P(i)+1)
        NEFF(I)=MAX(NEFF(I),NMINL)
      END DO
c
c  Output resonance data in OP-style format to C file
c
      OPEN(17,FILE='C',FORM='FORMATTED',STATUS='UNKNOWN')
      WRITE(17,1001) NZED,NELC,'C'
      WRITE(17,1010) NSPN2,LRGL2,NPTY2,KOUNT
      ii=ITARG(NCHOPX)
      write(17,1005) ii,ECH1,NAST,BDX
c
c  Find Rydberg electron (nl) and target state (it) for each resonance.
c
      DO k=1,KOUNT
        I=KCHAN(k)
        EFF=MAX(NZED-NELC,1)/SQRT(ECH1-RES(k))
        EFFII=MAX(NZED-NELC,1)/SQRT(ECH(I)-RES(k))
c       if(ii.lt.NAST)EFF1=MAX(NZED-NELC,1)/SQRT(ZSQ*ENAT(ii+1)-RES(k))
        if(km(i)) colon(k)='+'
        if(ii.lt.NAST) THEN
        WRITE(17,1003) k,'T',ITARG(I),NEFF(I),L2P(I),COLON(k),RES(k),
     *             EFF,WIDTH(k),IDECAY(k),IWT(1,k),EFFII
        ELSE
        WRITE(17,1003) k,'T',ITARG(I),NEFF(I),L2P(I),COLON(k),RES(k),
     *             EFF,WIDTH(k),IDECAY(k)
        ENDIF
        NEFF(I)=NEFF(I)+1
      END DO
      WRITE(6,*)'C file written: ',KOUNT,' records.'
c
c  Output to C file finished.  Here are the Formats ...
c
      RETURN
 1001 format(2i5,4x,a1)
 1003 format(i5,2x,a1,2i3,i2,a1,1pe13.5,0pf10.5,1pe10.2,i4,i6,0pf10.5)
 1005 format(i5,e14.6,i6,'T',f6.4)
 1010 format(4i5)
      END
c***********************************************************************
      SUBROUTINE POINTS(IOPT1,IMESH)
C
C  LAST MODIFIED 26.2.86 & '93MAY09
C
C  CALCULATES CHANNELS ENERGIES, NUMBER OF OPEN CHANNELS
C  AND TABULAR POINTS
      IMPLICIT REAL*8(A-H,O-Z)
C
      PARAMETER(TINY=-1.E-6)
      LOGICAL QDT
C
      PARAMETER (MZCHF= 395,MZLMX=   4,MZLP1=  40,MZMNP=5700,MZTAR= 200)
      PARAMETER (MZPTS= 600)
      COMMON/CQDT/R2ST(MZCHF),QDT,NQ
      COMMON/CACC/AC,ACNUM,ACJWBK,ACZP,LACC
      COMMON/CPOINT/KP0,KP1,KP2,RZERO,RONE,RTWO,H
      COMMON/CHAN/ECH(MZCHF),LLCH(MZCHF),EPS(MZCHF),FKNU(MZCHF),
     1 CC(MZCHF),RINF(MZCHF),ITARG(MZCHF),NCHF,NCHOP,NCHOP1
      COMMON/CEN/MXE,NWT,NZ,ETOT
      COMMON/CINPUT/
     1 LAMAX,LRANG2,LRGL2,MNP2,NAST,NCHAN,NELC,NPTY2,NSPN2,NZED,MORE2,
     2 ISAT(MZTAR),LAT(MZTAR),L2P(MZCHF),NCONAT(MZTAR),
     3 BSTO,RA,
     4 CF(MZCHF,MZCHF,MZLMX),COEFF(3,MZLP1),ENAT(MZTAR),
     5 WMAT(MZCHF,MZMNP),VALUE(MZMNP)
      COMMON/CNTRL/IPRINT,IRAD,IPERT,KP2X      
C
      KP0=1
      QDT=.FALSE.
      IQ=0
C
C  CHANNEL ENERGIES EPS AND NUMBER OF OPEN CHANNELS NCHOP
      NCHOP=0
      DO 100 I=1,NCHF
       E=ETOT-ECH(I)
       IF(E.LE.TINY) GO TO 90
       IF(E.GE.0.) GO TO 70
       E=0.
       IF(IOPT1.LE.9) GO TO 70
C       HES'93MAY5 --> WE'MAY8-9:
        QDT=IMESH.EQ.2
        NQ=NCHOP+1
   70  FKNU(I)=SQRT(E)
       NCHOP=NCHOP+1
       GO TO 100
   90  FKNU(I)=1./SQRT(-E)
  100  EPS(I)=E
      NCHOP1=NCHOP+1
C
C  SET QDT FOR CASE OF IMESH.EQ.2
C  ASSUMES  LL IN ASCENDING ORDER
C  NO QDT CHANNEL FOR FNU.LT.LL+0.5
C
      IF(IMESH.GE.2.AND.NCHOP.LT.NCHF)THEN
C----
C  LOOP OVER ALL DEGENERATE CHANNELS HERE (KAB,JAN94)
        DO 120 I=NCHOP+1,NCHF
         IF(ABS(EPS(I)-EPS(NCHOP+1)).GT.ABS(TINY)) GO TO 121
C----
         xxll=LLCH(I)+5
         IF(FKNU(I)+0.00002.LE.MAX(QNMAX,xxll))GO TO 121
         QDT=.TRUE.
  120    NQ=I
  121   IF(QDT)THEN
          NCHOP1=NQ+1
          IPERT=0
          IF(IPRINT.GT.0)WRITE(6,630)NCHOP,NQ
        ENDIF
      ENDIF
C
C
C  CALCULATION OF RTWO
C  INCLUDES CALCULATION OF INNER POINTS OF INFLECTION RINF
      RTWO=RZERO
C
C  (1) OPEN CHANNELS
C  FOR OPEN CHANNELS RTWO IS DEFINED BY CONVERGENCE CRITERION
C  FOR THE JWBK METHOD.
      DO 139 I=1,NCHOP
      E=EPS(I)
      C=CC(I)
      L=LLCH(I)
      EC=E*C
      IF(EC.GT.AC)THEN
        RINF(I)=(SQRT(1.+EC)-1.)/E
      ELSE
        RINF(I)=.5*C
      ENDIF
      IF(L-1)131,132,133
  131 IF(AC.GE.1.E-3)THEN
        CONST=12.
      ELSE
         CONST=56.
      ENDIF
      CE=CONST*E
      IF(CE.LT..1)THEN
        R2=.5*CONST
      ELSE
        R2=(SQRT(1.+CE)-1.)/E
      ENDIF
      GOTO 138
  132 IF(AC.GE.1.E-3)THEN
        R2=3.9*RINF(I)
      ELSE
        R2=16.*RINF(I)
      ENDIF
      GOTO 138
  133 IF(AC.GE.1.E-3)THEN
        R2=RINF(I)*(1.2+5.7/L)
      ELSE
        R2=RINF(I)*(1.4+9.8/L)
      ENDIF
  138 R2ST(I)=R2
      IF(RTWO.LT.R2)RTWO=R2
  139 CONTINUE
C
C  (2) CLOSED CHANNELS (STRONGLY CLOSED FOR QDT.EQ..TRUE.)
C  FOR CLOSED CHANNELS RTWO IS EQUAL TO THE OUTER POINT OF
C  INFLECTION, EXCEPT FOR THE CASE OF FNU.LT.(LL+1)
      DO 150 I=NCHOP1,NCHF
      FNU=FKNU(I)
      FLP1=LLCH(I)+1
      IF(FNU.LT.FLP1)THEN
        RINF(I)=0.
        R2=RZERO
      ELSE
        A1=SQRT(FNU*FNU-CC(I))
        R2=FNU*(FNU+A1)
        RINF(I)=FNU*(FNU-A1)
      ENDIF
      IF(RTWO.LT.R2)RTWO=R2
  150 R2ST(I)=R2
      IF(RONE.GT.RTWO)RTWO=RONE
C
C  FIND INTERVAL AND TABULAR POINTS BETWEEN RZERO AND RTWO
      IF(RTWO.LE.RZERO)THEN
      KP2=1
      H=0.
      RETURN
      ENDIF
      WM=0.
      DO 170 I=1,NCHF
      C=CC(I)
      E=EPS(I)
      X=1./RZERO
      W=ABS(E+X*(2.-C*X))
      IF(W.GT.WM)WM=W
      X=1./RTWO
      W=ABS(E+X*(2.-C*X))
      IF(W.GT.WM)WM=W
      IF(C.LT.RZERO.OR.C.GT.RTWO)GOTO 170
      W=ABS(E+1./C)
      IF(W.GT.WM)WM=W
  170 CONTINUE
C
      H=ACNUM/SQRT(WM)
C.....            MODIFICATION 2.2.87
                  IF(ABS(RTWO-RZERO).LT.(2.*H))THEN
                        RTWO=RZERO
                        KP2=1
                        H=0.
                        RETURN
                  ENDIF
C.....            END MODIFICATION
      N=(RTWO-RZERO)/H
      N=4*((N-1)/4)+5
      KP2=N
C
C  CHECK DIMENSIONS FOR NUMBER OF OUTER-REGION POINTS
      IF(KP2.GT.MZPTS)THEN
        KP2O=KP2
        RTWOO=RTWO
        IPERTO=IPERT
        KP2=4*((MZPTS-1)/4)+1
        RTWO=(KP2-1)*H+RZERO
        IPERT=0
        IF(IPRINT.GT.0.AND.IPERTO.NE.0)THEN
         WRITE(6,610)RTWOO,KP2O,RTWO,KP2
         DO 200 I=1,NCHF
          IF(R2ST(I).GT.RTWO)WRITE(6,620)I,R2ST(I)
  200     CONTINUE
        ENDIF
      ENDIF
C
      H=(RTWO-RZERO)/(KP2-1)
C
C  WEIGHTS FOR BODE RULE INTEGRATION
      CALL BODE
C
C
  610 FORMAT(/10X,'USE OF PERTURBATION REQUIRES RTWO =',
     *  F9.2,', KP2 =',I5/10X,'WHICH IS LARGER THAN MAXIMUM OF ',
     +  'MZPTS ALLOWED BY DIMENSIONS'/10X,'SET IPERT = 0, RTWO =',
     *  F9.2,', KP2 =',I4)
  620 FORMAT(10X,'CHANNEL',I4,' REQUIRES RTWO =',F9.2,
     * ' FOR USE OF PERTURBATION')
  630 FORMAT(/'0***** QDT USED'/7X,'NCHOP =',I4/7X,'NQ    =',I4/)
C
      RETURN
      END                                            
c***********************************************************************
      SUBROUTINE READ1
C
C      $Id: read1.f,v 1.1 1994/03/18 17:43:47 ferrous Exp $
C
C      $Log: read1.f,v $
c Revision 1.1  1994/03/18  17:43:47  ferrous
c Initial revision
c
C
C
C  READS DATA INDEPENDENT OF SLPI FROM R-MATRIX FILE, FILA
C
C  THE FOLLOWING DATA ARE READ _
C NZ = NUCLEAR CHARGE
C NELC = NUMBER OF ELECTRONS IN TARGET
C NAST = NUMBER OF TARGET STATES
C LRANG2 = TOTAL NUMBER OF SMALL L VALUES
C LAMAX = MAXIMUM LAMBDA FOR MULTIPOLE POTENTIALS
C RA = R-MATRIX RADIUS
C BSTO = LOGARITHMIC DERIVATIVE
C FOR I = 1,NAST -
C  ENAT(I) = TARGET ENERGIES
C  LAT(I) = TARGET ORBITAL ANGULAR MOMENTA
C  ISAT(I) = VALUES OF (2*S+1) FOR TARGET STATES
C FOR I = 1,3 AND L = 1,LRANG2 -
C  COEFF(I,L) = BUTTLE CORRECTION
C
      IMPLICIT REAL*8(A-H,O-Z)
C
      PARAMETER (MZCHF= 395,MZLMX=   4,MZLP1=  40,MZMNP=5700,MZTAR= 200)
      COMMON/CINPUT/
     1 LAMAX,LRANG2,LRGL2,MNP2,NAST,NCHAN,NELC,NPTY2,NSPN2,NZ,MORE2,
     2 ISAT(MZTAR),LAT(MZTAR),L2P(MZCHF),NCONAT(MZTAR),
     3 BSTO,RA,
     4 CF(MZCHF,MZCHF,MZLMX),COEFF(3,MZLP1),ENAT(MZTAR),
     5 WMAT(MZCHF,MZMNP),VALUE(MZMNP)
C
      REWIND(10)
      READ(10)NELC,NZ,LRANG2,LAMAX,NAST,RA,BSTO
      READ(10)(ENAT(I),I=1,NAST)
      READ(10)(LAT(I),I=1,NAST)
      READ(10)(ISAT(I),I=1,NAST)
      READ(10)((COEFF(I,L),I=1,3),L=1,LRANG2)
C
C  DIMENSION CHECKS
      IF(NAST.GT.MZTAR)THEN
      WRITE(6,610)NAST,MZTAR
      STOP
      ENDIF
      IF(LRANG2.GT.MZLP1)THEN
      WRITE(6,620)LRANG2,MZLP1
      STOP
      ENDIF
C
  610 FORMAT(/'TOO MANY TARGET STATES',
     1 10X,'VALUE READ FOR NAST IS',I10/
     2 10X,'MAXIMUM ALLOWED BY DIMENSIONS IS MZ','TAR =',I10)
  620 FORMAT(/'TOO MANY BUTTLE COEFFICIENTS',
     1 10X,'VALUE READ FOR LRANG2 IS',I10/
     2 10X,'MAXIMUM VALUE ALLOWED BY DIMENSIONS IS MZ','LP1 =',I10)
C
      RETURN
      END
c***********************************************************************
      SUBROUTINE READ2
C
C      $Id: read2.f,v 1.1 1994/03/18 17:43:47 ferrous Exp $
C
C      $Log: read2.f,v $
c Revision 1.1  1994/03/18  17:43:47  ferrous
c Initial revision
c
C
C
C  READS R-MATRIX DATA FOR ONE SLPI CASE, FROM FILA
C
C  THE FOLLOWING DATA ARE READ -
C	LRGL2 = TOTAL ORBITAL ANGULAR MOMENTUM
C	NSPN2 = TOTAL (2*S+1)
C	NPTY2 = TOTAL PARITY
C	NCHAN = NUMBER OF CHANNELS
C	MNP2 = NUMBER OF R-MATRIX POLES
C	MORE2 = ZERO TO TERMINATE SLPI CASES
C	FOR I = 1,NAST -
C		NCONAT(I) = NUMBER OF CHANNELS FOR TARGET STATE I
C	FOR I = 1,NCHAN -
C		L2P(I) = SMALL L FOR CHANNEL I
C	FOR I = 1,NCHAN AND N = 1,NCHAN AND M = 1,LAMAX -
C		CF(I,N,M) = COEFFICIENTS IN MULTIPOLE POTENTIALS
C	FOR I = 1,MNP2 -
C		VALUE(I) = R-MATRIX POLE ENERGIES
C	FOR K = 1,NCHAN AND I = 1,MNP2 -
C		WMAT(K,I) = R-MATRIX AMPLITUDES
C
      IMPLICIT REAL*8(A-H,O-Z)
C
      PARAMETER (MZCHF= 395,MZLMX=   4,MZLP1=  40,MZMNP=5700,MZTAR= 200)
      COMMON/CINPUT/
     1 LAMAX,LRANG2,LRGL2,MNP2,NAST,NCHAN,NELC,NPTY2,NSPN2,NZ,MORE2,
     2 ISAT(MZTAR),LAT(MZTAR),L2P(MZCHF),NCONAT(MZTAR),
     3 BSTO,RA,
     4 CF(MZCHF,MZCHF,MZLMX),COEFF(3,MZLP1),ENAT(MZTAR),
     5 WMAT(MZCHF,MZMNP),VALUE(MZMNP)
C
      COMMON/CDEGEN/NASTD,NASTR,NLEV(MZTAR),NCNATR(MZTAR),ENATR(MZTAR)
C
      READ(10)LRGL2,NSPN2,NPTY2,NCHAN,MNP2,MORE2
      READ(10)(NCONAT(I),I=1,NASTR)
      READ(10)(L2P(I),I=1,NCHAN)
C      READ(10)(((CF(I,N,M),I=1,NCHAN),N=1,NCHAN),M=1,LAMAX)
C      print *,'got here4'
C      print *,(((CF(I,N,M),I=1,NCHAN),N=1,NCHAN),M=1,LAMAX)
      READ(10)(VALUE(I),I=1,MNP2)
      READ(10)((WMAT(K,I),K=1,NCHAN),I=1,MNP2)
C
C  DIMENSION CHECKS
      IF(NCHAN.GT.MZCHF)THEN
      WRITE(6,600)NSPN2,LRGL2,NPTY2,NCHAN,MZCHF
      STOP
      ENDIF
      IF(MNP2.GT.MZMNP)THEN
      WRITE(6,610)NSPN2,LRGL2,NPTY2,MNP2,MZMNP
      STOP
      ENDIF
      IF(LAMAX.GT.MZLMX)THEN
      WRITE(6,620) LAMAX,MZLMX
      STOP
      ENDIF
C
C  GROUP TOGETHER CHANNELS BELONGING TO DEGENERATE LEVELS
C  AND PRESERVE THE READ VALUES OF NCONAT IN /CDEGEN/
      IF(NASTD.GT.0)THEN
        DO 41 I=1,NASTR
        NCNATR(I)=NCONAT(I)
   41   CONTINUE
        N1=1
        DO 42 I=1,NASTD
        NCON=0
        N2=NLEV(I)+N1-1
        DO 43 IN=N1,N2
        NCON=NCON+NCNATR(IN)
   43   CONTINUE
        NCONAT(I)=NCON
        N1=N2+1
   42   CONTINUE
      END IF
C
  600 FORMAT(/'TOO MANY CHANNELS FOR (IS, IL, IP) = (',
     1  3I3,')',10X,'VALUE READ FOR NCHAN IS',I10/
     2  10X,'MAXIMUM ALLOWED BY DIMENSIONS IS MZ','CHF =',I10)
  610 FORMAT(/'TOO MANY R-MATRIX STATES FOR (IS, IL, IP) = (',
     * 3I3,')',10X,'VALUE READ FOR MNP2 IS',I10/
     3 10X,'MAXIMUM ALLOWED BY DIMENSIONS IS MZ','MNP =',I10)
  620 FORMAT(/'TOO MANY MULTIPOLES',
     1 10X,'VALUE READ FOR LAMAX IS',I10/
     2 10X,'MAXIMUM ALLOWED BY DIMENSIONS IS MZ','LMX =',I10)
C
      RETURN
      END
c***********************************************************************
      SUBROUTINE READ5(IPRINT,IRAD,IPERT,AC,RONE,IOPT,MSLP,MZSLP,KSLP,
     *                 XMINST,XMAXST,BDXST,MZEST,EX5)
C
C  READ DATA FROM UNIT 5 = QB.INP
C
      IMPLICIT REAL*8(A-H,O-Z)
      LOGICAL EX5
      DIMENSION XMINST(MZEST),XMAXST(MZEST),BDXST(MZEST)
      INTEGER MSLP(MZSLP)
C
C  Open QB.INP ... if it doesn't exist, open it anyway
C                     and fill it with CPC test run data.
C
      INQUIRE (FILE='QB.INP',EXIST=EX5)
      IF (EX5) THEN
        OPEN(UNIT=5,FILE='QB.INP',STATUS='OLD')
        print *,' Data read from QB.INP on unit 5'
      ELSE
        print *,'File QB.INP does not exist ... creating CPC test'
        OPEN(UNIT=5,FILE='QB.INP',STATUS='NEW')
        WRITE(5,600)0,0,0,1.0e-6,1.0,1
        WRITE(5,603)0,2,1
        WRITE(5,604)0.0,10.2,0.005
        REWIND 5
      ENDIF
      READ(5,*)IPRINT,IRAD,IPERT
      print *,IPRINT,IRAD,IPERT
      READ(5,*) AC
      print *,AC
      READ(5,*) RONE
      print *,RONE
      READ(5,*) IOPT
      print *,IOPT
      WRITE(6,600)IPRINT,IRAD,IPERT,AC,RONE,IOPT
      IF (IOPT.LE.0) THEN
        print *,'Cannot deal with this case. Reset IOPT.'
        STOP
      ENDIF
      KSLP=0
      DO kount=1,MZSLP
        READ(5,*,END=502)IS,IL,IP
        WRITE(6,603) IS,IL,IP
        IF(IL.EQ.-1) GOTO 502
        KSLP=KSLP+1
        MSLP(KSLP)=10000*IS+100*IL+IP
        READ(5,*) XMINST(KSLP),XMAXST(KSLP),BDXST(KSLP)
        WRITE(6,604)XMINST(KSLP),XMAXST(KSLP),BDXST(KSLP)
      END DO
C
  502 RETURN
  600 FORMAT(
     1  3I3,18X,   ':IPRINT,IRAD,IPERT'/
     2  1PE9.1,18X,':AC'/
     3  0PF5.3,22X,':RONE'/
     4  I5,22X,    ':IOPT')
  603 FORMAT(I2,I4,I3,18X,':2S+1,L,Parity')
  604 FORMAT(F8.4,F9.4,F8.5,2X,':XMIN,XMAX,BDX')
      END
c***********************************************************************
      SUBROUTINE RMATB2
C
C      $Id: rmatb2.f,v 1.1 1994/03/18 17:43:47 ferrous Exp $
C
C      $Log: rmatb2.f,v $
c Revision 1.1  1994/03/18  17:43:47  ferrous
c Initial revision
c
C
C
C  THE R-MATRIX IS
C        R = T + S/DE
C  WHERE
C        DE = VALUE(KP) - ETOT
C  THIS SUBROUTINE CALCULATES T IN RMAT AND ENERGY DERIVATIVE
C  OF T IN RMATD.
C
      IMPLICIT REAL*8(A-H,O-Z)
C
      LOGICAL NEWBUT
C
      PARAMETER (MZCHF= 395,MZLMX=   4,MZLP1=  40,MZMNP=5700,MZTAR= 200)
      COMMON/CEN/MXE,NWT,NZ,ETOT
      COMMON/CINPUT/
     1 LAMAX,LRANG2,LRGL2,MNP2,NAST,NCHAN,NELC,NPTY2,NSPN2,NZED,MORE2,
     2 ISAT(MZTAR),LAT(MZTAR),L2P(MZCHF),NCONAT(MZTAR),
     3 BSTO,RA,
     4 CF(MZCHF,MZCHF,MZLMX),COEFF(3,MZLP1),ENAT(MZTAR),
     5 WMAT(MZCHF,MZMNP),VALUE(MZMNP)
      COMMON/CHAN/ECH(MZCHF),LLCH(MZCHF),EPS(MZCHF),FKNU(MZCHF),
     1 CC(MZCHF),RINF(MZCHF),ITARG(MZCHF),NCHF,NCHOP,NCHOP1
      COMMON/CRMAT/RMAT(MZCHF,MZCHF),RMATD(MZCHF,MZCHF),
     1 UMAT(MZCHF,MZCHF),GAMMA
      COMMON/CPOINT/KP0,KP1,KP2,RZERO,RONE,RTWO,H
      COMMON/CLOGB/NEWBUT
      DIMENSION TEMP(MZMNP)
C
c      print *,'calling rmatb2'
C
      DO K=1,MNP2
        TEMP(K)=1.0/(0.5*(VALUE(K)-ETOT))
      END DO
      RA2=RZERO*RZERO
C
      DO I=1,NCHF
        DO J=I,NCHF
          RMATD(J,I)=0.
          RMAT(J,I)=0.
        END DO
C
        L=LLCH(I)+1
        IF(NEWBUT)THEN
          print *,'probably not called'
          NBUT=COEFF(3,I)
          U=COEFF(2,I)+RA2*EPS(I)
          CALL CBUT0(NBUT,U,B,C)
          B=B*COEFF(1,I)
          C=C*COEFF(1,I)*RA2
          RMATD(I,I)=C
          RMAT(I,I)=B
        ELSE
            E=EPS(I)
C           E=0.5*EPS(I)
C           write *,'E=',E
C
C          Connor 1998
C
          RMATD(I,I)=COEFF(2,I)+2.*E*COEFF(3,I)
   15     RMAT(I,I)=COEFF(1,I)+E*(COEFF(2,I)+E*COEFF(3,I))
        ENDIF
C
C
        DO K=1,MNP2
          VI=TEMP(K)*WMAT(I,K)
          DO J=I,NCHF
            VIJ=VI*WMAT(J,K)
            RMAT(J,I)=RMAT(J,I)+VIJ
            RMATD(J,I)=RMATD(J,I)+VIJ*TEMP(K)
          END DO
        END DO
C
        DO J=I,NCHF
          RMATD(I,J)=RMATD(J,I)
          RMAT(I,J)=RMAT(J,I)
        END DO
      END DO
C       print *,'final R-matrix'
C       print *,((RMAT(I,J),I=1,NCHF),J=1,NCHF)
C
      RETURN
      END
c***********************************************************************
      SUBROUTINE SC(E,L,R,AC,S,SP,C,CP)
C
C  COULOMB FUNCTIONS S AND C FROM
C  POWER SERIES EXPANSIONS.
C
      IMPLICIT REAL*8(A-H,O-Z)
C
C  POWER SERIES FOR F AND G
      CALL COULFG(L,E,R,AC,F,FP,G,GP,K,IERR,ACT)
      IF(IERR.EQ.2)THEN
        WRITE(6,600)L,E
        STOP
      ENDIF
      IF(IERR.EQ.1)WRITE(6,610)L,E,K,ACT
C
C  CASE OF E.GE.0
      IF(E.GE.0)THEN
C  CALCULATE CAP. A
        A=1.
        IF(L.GT.0)THEN
          A1=1.
          A2=-E
          A3=E+E
          DO 10 I=1,L
            A2=A2+A3
            A1=A1+A2
   10       A=A*A1
        ENDIF
C  CALCULATE SCRIPT G AND COULOMB FUNCTION H AND DERIVATIVE
        SG=A*FCHI(E,L,AC)*.318309886
        H=-G-SG*F
        HP=-GP-SG*FP
C  CALCULATE CAP B
        IF(E.LT.0.01)THEN
          B=A
        ELSE
          B=A/(1.-EXP(-6.28318531/SQRT(E)))
        ENDIF
C
C  CASE OF E.LT.0
      ELSE
        CALL ABG(E,L,AC,A,BG)
        H=-(G+BG*F)
        HP=-(GP+BG*FP)
        B=A
      ENDIF
C
C  COMPLETE CALCULATION OF S AND C
      C1=1.25331414*SQRT(B)
      S=C1*F
      SP=C1*FP
      C1=C1/B
      C=H*C1
      CP=HP*C1
      RETURN
  600 FORMAT(///10X,'SERIES IN COULFG NOT CONVERGED'/
     + 10X,'SMALL L = ',I5,', EPS =',1P,E13.4//)
  610 FORMAT(//5X,'*** FUNCTIONS FROM COULFG INACCURATE ***'/5X,
     + 'SMALL L=',I5,', EPS =',1P,E13.4,', K =',I5,', ACTACC =',E13.4//)
C
      END                 
c***********************************************************************
      SUBROUTINE SCALE1
C
C      $Id: scale1.f,v 1.1 1994/03/18 17:43:47 ferrous Exp $
C
C      $Log: scale1.f,v $
c Revision 1.1  1994/03/18  17:43:47  ferrous
c Initial revision
c
C
C
C   CONVERTS R-MATRIX TARGET DATA TO Z-SCALED FORM
C
      IMPLICIT REAL*8(A-H,O-Z)
C
C
      LOGICAL NEWBUT
C
      PARAMETER (MZCHF= 395,MZLMX=   4,MZLP1=  40,MZMNP=5700,MZNRG=  40,
     1 MZTAR= 200)
C      
C  COMMON BLOCKS FROM ASYMPTOTIC ROUTINE
      COMMON/CEN/MXE,NWT,NZ,ETOT
      COMMON/CHAN/ECH(MZCHF),LLCH(MZCHF),EPS(MZCHF),FKNU(MZCHF),
     1 CC(MZCHF),RINF(MZCHF),ITARG(MZCHF),NCHF,NCHOP,NCHOP1
      COMMON/CPOINT/KP0,KP1,KP2,RZERO,RONE,RTWO,H
      COMMON/CPOT/LAMP(MZCHF,MZCHF),BW(MZCHF,MZCHF)
      COMMON/CENAT1/ENAT1
      COMMON/CACC/AC,ACNUM,ACJWBK,ACZP,LACC
      COMMON/CLOGB/NEWBUT
      COMMON/CBUT/FKN(0:MZNRG),UKN(0:MZNRG)
C
C  COMMON BLOCK FROM SUBROUTINE READ
C  NOTE USE OF NZED IN PLACE OF NZ
      COMMON/CINPUT/
     1 LAMAX,LRANG2,LRGL2,MNP2,NAST,NCHAN,NELC,NPTY2,NSPN2,NZED,MORE2,
     2 ISAT(MZTAR),LAT(MZTAR),L2P(MZCHF),NCONAT(MZTAR),
     3 BSTO,RA,
     4 CF(MZCHF,MZCHF,MZLMX),COEFF(3,MZLP1),ENAT(MZTAR),
     5 WMAT(MZCHF,MZMNP),VALUE(MZMNP)
C
C
C  COMMON BLOCK FOR DEGENERATE TARGET LEVELS
      COMMON/CDEGEN/NASTD,NASTR,NLEV(MZTAR),NCNATR(MZTAR),ENATR(MZTAR)
C
      DIMENSION INDEX(MZTAR),AVENGY(MZTAR)
C  NUCLEAR CHARGE
      AZ=MAX(NZED-NELC,1)
      AZ1=1./AZ
      AZ2=AZ1*AZ1
      TAZ2=2.*AZ2
      AZAZ=AZ*AZ
C
C
C  Z-SCALED TARGET ENERGIES, RELATIVE TO ZERO FOR
C  TARGET GROUND STATE
      ENAT1=ENAT(1)
      IF(NAST.EQ.1)GOTO 20
      DO 10 I=2,NAST
   10 ENAT(I)=TAZ2*(ENAT(I)-ENAT1)
  
C
C  TEST FOR DEGENERACY AND FORM AVERAGED TARGET ENERGY LEVELS
      IF (NASTD .EQ. 0) GO TO 20
      NASTR=0
      DO 13 N=1,NASTD
      NASTR=NASTR+NLEV(N)
   13 CONTINUE
      IF(NASTR.NE.NAST)THEN
        WRITE(6,640)NASTR,NAST,(NLEV(N),N=1,NASTD)
        STOP
      END IF
  640 FORMAT(///5X,'*****INCORRECT DATA   ',/
     1          5X,'EXPECT ',I3, 'LEVELS FROM NLEV DATA BUT NAST IS',
     2          I3,/5X,'NLEV=',20I3)
      AVENGY(1)=0.0
      N1=NLEV(1)+1
      INDEX(1)=NLEV(1)
      DO 14 J=2,NASTD
      N=NLEV(J)
      N2=N+N1-1
      INDEX(J)=N2
      ENSUM=0.0
      DO 15 IN=N1,N2
      ENSUM=ENAT(IN)+ENSUM
   15 CONTINUE
      AVENGY(J)=ENSUM/N
      N1=N2+1
   14 CONTINUE
   20 ENAT(1)=0.
      NASTR=NAST
C
C  RA AND BSTO
      RZERO=RA*AZ
      BSTO=BSTO/RZERO
C
C  BUTTLE CORRECTION
      IF(COEFF(3,1).GT.-10000.)THEN
        NEWBUT=.FALSE.
        AA=RZERO*AZ2
        DO 80 M=1,3
        AA=AA*AZAZ
        AA=1.0
        DO 80 L=1,LRANG2
   80   COEFF(M,L)=COEFF(M,L)*AA
      ELSE
        NEWBUT=.TRUE.
        DO 90 L=1,LRANG2
        NBUT=-INT(COEFF(3,L))/10000
        IF(NBUT.GT.MZNRG)THEN
          WRITE(6,699)NBUT,MZNRG
          STOP
        ENDIF
   90    COEFF(3,L)=NBUT
C
C       molecular adaption
C
C   90   COEFF(1,L)=RZERO*COEFF(1,L)
C       INITIALISE FKN AND UKN
        G=-1.5707963
        DO 100 I=0,MZNRG
        G=G+3.141592654
        FKN(I)=G
  100   UKN(I)=G*G
      ENDIF
C
C  IF THERE ARE DEGENERATE LEVELS PRINT THE AVERAGED ENERGIES
C  PUT THEM IN /CINPUT/ AND PRESERVE THE ENERGIES READ FROM 
C  THE H FILE IN /CDEGEN/
C
      IF(NASTD.NE.0)THEN
        DO 3 I=1,NAST
        ENATR(I)=ENAT(I)
    3   CONTINUE
        INR1=1
        DO 4 IND=1,NASTD
        N=NLEV(IND)
        INR2=INR1+N-1
        IF(N.GT.1)THEN
          WRITE(6,664)INR1,INR2
        END IF
        INR1=INR2+1
    4   CONTINUE
        WRITE(6,665)
        DO 5 J=1,NASTD
        ENAT(J)=AVENGY(J)
        WRITE(6,666)INDEX(J),J,ENAT(J)
    5   CONTINUE
        NAST=NASTD
      END IF
  664 FORMAT(/' LEVELS ',I3,' TO ',I3,' ARE COMBINED' )
  665 FORMAT(/15X,'EQUIVALENT TARGET STATES -'/15X,26('*')/
     1       12X,'OLD INDEX',5X,'NEW INDEX',7X,'SCALED ENERGY'/)
  666 FORMAT(3X,2I14,9X,F12.6)
  699 FORMAT(/'TOO MANY CONTINUUM TERMS FOR BUTTLE',
     1 10X,'VALUE READ FOR NBUT IS',I10/
     2 10X,'MAXIMUM ALLOWED BY DIMENSIONS IS MZ','NRG =',I10)
      RETURN
      END
c***********************************************************************
      SUBROUTINE SCALE2
C
C      $Id: scale2.f,v 1.1 1994/03/18 17:43:47 ferrous Exp $
C
C      $Log: scale2.f,v $
c Revision 1.1  1994/03/18  17:43:47  ferrous
c Initial revision
c
C
C
C   CONVERTS R-MATRIX DATA TO Z-SCALED FORM
C
      IMPLICIT REAL*8(A-H,O-Z)
C
      PARAMETER (MZCHF= 395,MZLMX=   4,MZLP1=  40,MZMNP=5700,MZTAR= 200)
C      
C  COMMON BLOCKS FROM ASYMPTOTIC ROUTINE
      COMMON/CEN/MXE,NWT,NZ,ETOT
      COMMON/CHAN/ECH(MZCHF),LLCH(MZCHF),EPS(MZCHF),FKNU(MZCHF),
     1 CC(MZCHF),RINF(MZCHF),ITARG(MZCHF),NCHF,NCHOP,NCHOP1
      COMMON/CPOINT/KP0,KP1,KP2,RZERO,RONE,RTWO,H
      COMMON/CPOT/LAMP(MZCHF,MZCHF),BW(MZCHF,MZCHF)
C
C  COMMON BLOCK FROM SUBROUTINE READ
C  NOTE USE OF NZED IN PLACE OF NZ
      COMMON/CINPUT/
     1 LAMAX,LRANG2,LRGL2,MNP2,NAST,NCHAN,NELC,NPTY2,NSPN2,NZED,MORE2,
     2 ISAT(MZTAR),LAT(MZTAR),L2P(MZCHF),NCONAT(MZTAR),
     3 BSTO,RA,
     4 CF(MZCHF,MZCHF,MZLMX),COEFF(3,MZLP1),ENAT(MZTAR),
     5 WMAT(MZCHF,MZMNP),VALUE(MZMNP)
      COMMON/CENAT1/ENAT1
      COMMON/CNTRL/IPRINT,IRAD,IPERT,KP2X
C
C  STATISTICAL WEIGHT * 2
      NWT=2*(2*LRGL2+1)*NSPN2
      IF(NSPN2.EQ.0) NWT=2*(LRGL2+1)
C
C  NUCLEAR CHARGE
      AZ=MAX(NZED-NELC,1)
      AZ1=1./AZ
      AZ2=AZ1*AZ1
      TAZ2=2.*AZ2
      AZHR=1./SQRT(AZ)
      AZAZ=AZ*AZ
C
C
C  CHANNELS
      NCHF=NCHAN
      DO 40 I=1,NCHF
      LL=L2P(I)
      CC(I)=FLOAT(LL*(LL+1))
   40 LLCH(I)=LL
      I=0
      DO 50 J=1,NAST
      K=NCONAT(J)
      IF(K.EQ.0)GOTO 50
      DO 45 L=1,K
      I=I+1
      ITARG(I)=J
      ECH(I)=ENAT(J)
   45 CONTINUE
   50 CONTINUE
C
C  R-MATRIX
C  VALUE AND WMAT
      DO 60 N=1,MNP2
      DO 60 I=1,NCHF
   60 WMAT(I,N)=WMAT(I,N)*AZHR
      IF(IPRINT.GT.2)WRITE(6,635)
  635 FORMAT(//' N, VALUE(N) AND WMAT(I,N)'/)
C
C      connor MARCH 1998
C
      DO 70 N=1,MNP2
      VALUE(N)=TAZ2*(VALUE(N)-ENAT1)
      IF(IPRINT.GT.2)WRITE(6,640)N,VALUE(N),(WMAT(I,N),I=1,NCHF)
   70 CONTINUE
  640 FORMAT(I5,E12.3,5X,9E12.3,(/5X,9E12.3))
C
C  COEFFICIENTS IN POTENTIAL
      LMX=LAMAX
      IF(IPERT.NE.1)GOTO 160
      IF(IPRINT.GT.2)WRITE(6,655)
  655 FORMAT(/' PERTURBATION P FOR MULTIPOLE POTENTIAL'//
     1 '  - DIPOLE PART'/)
C  LAMP(I,J) AND BW(I,J)
      DO 120 I=1,NCHF
      DO 120 J=1,NCHF
  120 LAMP(I,J)=1
      IF(LMX.EQ.0)GOTO 160
      A1=1./RZERO
      A2=.5*A1
      IF(NCHF.EQ.1)GOTO 140
      DO 130 I=1,NCHF
      I1=I+1
      DO 130 J=I1,NCHF
      BU1=-CF(I,J,1)
      IF(BU1.EQ.0.)GOTO 130
      LAMP(I,J)=2
      BW(I,J)=BU1
      P=A2*BU1
      IF(IPRINT.GT.2)WRITE(6,660)I,J,P
  660 FORMAT(2I5,E12.4)
  130 CONTINUE
      IF(LMX.EQ.1)GOTO 160
  140 CONTINUE
      IF(IPRINT.GT.2)WRITE(6,665)
  665 FORMAT(/'  - QUADRUPOLE PART'/)
      A2=A2*A1
      DO 150 I=1,NCHF
      DO 150 J=I,NCHF
      BU2=-CF(I,J,2)
      IF(BU2.EQ.0.)GOTO 150
      LAMP(I,J)=3
      BU2=BU2*AZ
      BW(I,J)=BU2
      CF(I,J,2)=AZ*CF(I,J,2)
      P=BU2*A2
      IF(IPRINT.GT.2)WRITE(6,660)I,J,P
  150 CONTINUE
  160 CONTINUE
C
      RETURN
      END
c***********************************************************************
      SUBROUTINE SCAN(X0,X1,BDX,ECH1)
C
C   Scan over effective n below threshold energy ECH1
C
      IMPLICIT REAL*8(A-H,O-Z)
      PARAMETER (MZCHF= 395,MZEST= 200,MZLMX=   4,MZLP1=  40,
     1 MZMNP=5700,MZTAR= 200)
      PARAMETER (PI=3.141592654, EV=13.605)
      PARAMETER (MSCALE=2, nprint=15, quad=0.001, ACC=0.0001)
C            
      CHARACTER*1 COLON,LR
      CHARACTER*8 strang
      INTEGER ISORT(MZCHF),IWT(2,MZEST),IDECAY(MZEST)
      LOGICAL LEFT,RITE,missed
      COMMON/CEN/MXE,NWT,NZ,ETOT
      COMMON/CNTRL/IPRINT,IRAD,IPERT,KP2X
      COMMON/CHAN/ECH(MZCHF),LLCH(MZCHF),EPS(MZCHF),FKNU(MZCHF),
     1 CC(MZCHF),RINF(MZCHF),ITARG(MZCHF),NCHF,NCHOP,NCHOP1
      COMMON/CINPUT/
     1 LAMAX,LRANG2,LRGL2,MNP2,NAST,NCHAN,NELC,NPTY2,NSPN2,NZED,MORE2,
     2 ISAT(MZTAR),LAT(MZTAR),L2P(MZCHF),NCONAT(MZTAR),
     3 BSTO,RA,
     4 CF(MZCHF,MZCHF,MZLMX),COEFF(3,MZLP1),ENAT(MZTAR),
     5 WMAT(MZCHF,MZMNP),VALUE(MZMNP)
C     
      DIMENSION EIG(MZCHF),Y(MZCHF),yb(MZCHF),XX(MZCHF,MZCHF)
      DIMENSION RES(MZEST),WIDTH(MZEST),cprint(nprint,2),test(5)
      DATA PD1,PD2,PD3,PD4,E1,E2,E3,W1,W2/9*0.0/, PDINF/1.0e+12/
C
      IF (BDX.LE.0.0.OR.X1.LE.X0) RETURN
      OPEN(14,FILE='P',STATUS='UNKNOWN')
C
C  INITIAL WRITES TO FILE EIGOUT
C
      WRITE(6,6000)
C
C  INITIALISE FOR NUMBER OF POINTS AND INTERVAL
C
      N=1+MSCALE*(X1-X0)/BDX
      IF (N.LT.5) THEN
        N=5
        BDX=(X1-X0)/FLOAT(N-1)
      END IF
      ZSQ=MAX(NZED-NELC,1)**2
      KOUNT=0
      XA=X0-BDX/MSCALE
      xtest=1.0
      just=0
C
C SCAN FROM X0 TO X1.
c Interval BDX is multiplied here by a factor 
c which varies from 1/MSCALE near integer XA to 1 near half-integer XA.
C
      DO 6 I=1,N
        if (I.GT.4) xtest=1.0+(MSCALE-1)*2*ABS(NINT(XA)-XA)
        XA=XA+BDX*xtest/float(MSCALE)
        IF (XA.GT.X1) GO TO 10
        ETOT=ECH1-1./XA**2
        ITER=0
        IF (I.LE.2) ITER=-1
c
c  Find eigenphase sum derivative etc for this energy in the scan
c
        CALL KDIAG(ITER,IPERT,ETOT,EIG,EIGSUM,EIGSUD,Y,XX)
        PDMIN=XA**3
        if(XA.lt.1.0) PDMIN=1.
c
c Serious overlapping is denoted by current resonance on LEFT or RITE of
c perturbing resonance: allow for maximimum not being reached.
c
        PD12=PD1-PD2
        PD21=PD2-PD1
        LEFT=PD12.LE.PD2-PD3.AND.PD12.LT.EIGSUD-PD1.AND.PD12.GE.0.0
     *       .AND.EIGSUD-PD1.GE.PD2-PD3.AND.PD2.GE.2.0*PDMIN
     *       .AND.PD2.LE.PDINF.AND.PD3.GE.PD4
        RITE=PD21.LE.PD1-EIGSUD.AND.PD21.LT.PD3-PD2.AND.PD21.GE.0.0
     *       .AND.PD3-PD2.GE.PD1-EIGSUD.AND.PD1.GE.2.0*PDMIN
     *       .AND.PD2.LE.PDINF.AND.PD3.LE.PD4
c
c  Store energies and eigenphase derivatives so that current values
c  are in (E1,PD1), previous in (E2,PD2), previous previous in (E3,PD3),
c  previous previous previous in (E4,PD4).
c
        EIGDIF=EIGSUM-W2
        W2=W1
        W1=EIGSUM
        PD4=PD3
        PD3=PD2
        PD2=PD1
        PD1=EIGSUD
        E4=E3
        E3=E2
        E2=E1
        E1=ETOT
        xxxb=1.0/SQRT(ECH1-E2)
        just=just+1
        IF (I.LE.3) GO TO 6
c
c  Check if PD2 is greater than adjacent values
c
        IF (((PD2.GT.PD3.AND.PD2.GT.PD1.AND.PD2.GT.PDMIN.AND.
     *        PD2.LE.PDINF.AND.(PD3.GT.PD4.OR.just.LE.3))
     *       .OR.LEFT.OR.RITE).AND.ABS(EIGDIF).GT.ACC) THEN
          just=0
c
c  Seems to be a maximum here.
c  initialise 3 points for the iterations, with 2nd as peak
c
          EE1=E1
          EE2=E2
          EE3=E3
          DD1=PD1
          DD2=PD2
          DD3=PD3
          dbest=0.0
          LR=' '
          IF (RITE) LR='R'
          IF (LEFT) THEN
            LR='L'
            EE1=E2
            EE2=E3
            EE3=E4
            DD1=PD2
            DD2=PD3
            DD3=PD4
          ENDIF
c
c   Define end-points and limits for iterations
c
          END1=EE1
          END3=EE3
          DEND1=DD1
          DEND3=DD3
          test(2)=DD2
          IF (LEFT.OR.RITE) THEN
            test(2)=0.0
            IF (END1.NE.END3) test(2)=DD2-
     *       ((DEND1-DEND3)*EE2+DEND3*END1-DEND1*END3)/(END1-END3)
          ENDIF
          exceed=test(2)
          PDMAX=MAX(DEND1,DEND3)
          PDMAX=MAX(PDMAX,DD2)
          missed=.TRUE.
          IF (IPRINT.GT.0) print *,'Next iteration ...'
c
c  Bisect both intervals E1,E2 and E2,E3 to get close enough for quadratic
c
    3     DO IBSECT=1,10
            ITER=IBSECT
            IF (IPRINT.GT.0)
     *        WRITE(6,6009)ITER,EE3*ZSQ,EE2*ZSQ,EE1*ZSQ,DD3,DD2,DD1
            IF (ABS(DD2-0.5*(DD1+DD3)).LT.quad*DD2) GO TO 4
            missed=.FALSE.
c
c  Bisecting entails expanding the 3 points to 5 by inserting 2 extra
c
            EE5=EE3
            EE3=EE2
            EE4=(EE3+EE5)*0.5
            EE2=(EE1+EE3)*0.5
            DD5=DD3
            DD3=DD2
            test(3)=test(2)
            ETOT=EE4
            DO IHALF=4,2,-2
c
c  Find eigenphase sum derivative in each half interval
c
              CALL KDIAG(ITER,IPERT,ETOT,EIG,EIGSUM,test(IHALF),Y,XX)
c
c  Store at the biggest eigenphase sum derivative
c
              IF (EIGSUD.GT.dbest)THEN
                dbest=EIGSUD
                ebest=ETOT
                pbest=EIGSUM
                DO J=1,NCHF
                  yb(J)=Y(J)
                END DO
              ENDIF
              ETOT=EE2
            END DO
            DD2=test(2)
            DD4=test(4)
c
c  For LEFT or RITE, maximise displacement from line joining end-points
c
            IF (LEFT.OR.RITE) THEN
              IF (DD2.LE.MAX(DD1,DD3).AND.DD3.LE.MAX(DD2,DD4).AND.
     *            DD4.LE.MAX(DD3,DD5)) THEN
                ETOT=EE4
                DO IHALF=4,2,-2
                  dtest=test(IHALF)
                  test(IHALF)=0.0
                  IF (END1.NE.END3) test(IHALF)=dtest-
     *       ((DEND1-DEND3)*ETOT+DEND3*END1-DEND1*END3)/(END1-END3)
                  ETOT=EE2
                END DO
                IF (test(3).GT.MAX(test(2),test(4))) ETOT=EE3
                IF (test(4).GT.MAX(test(2),test(3))) ETOT=EE4
              ELSE
c
c  Treat as a normal maximum if LEFT or RITE rises above end-points
c
                LEFT=.FALSE.
                RITE=.FALSE.
                test(3)=DD3
                LR=':'
              ENDIF
	      IF (IPRINT.GT.0) print *,LR,test(4),test(3),test(2)
              IF (LR.NE.':') GO TO 5
            ENDIF
c
c  Re-create 3 points from 5, such that 2nd point is peak
c
	    IF (test(2).LT.test(3).AND.test(4).LT.test(3)) THEN
              EE1=EE2
              EE2=EE3
              EE3=EE4
              DD1=DD2
              DD2=DD3
              DD3=DD4
              test(2)=test(3)
            ELSE IF (test(4).GT.test(3)) THEN
              EE1=EE3
              EE2=EE4
              EE3=EE5
              DD1=DD3
              DD2=DD4
              DD3=DD5
              test(2)=test(4)
            ENDIF
          END DO
c
c  Abort if insufficient rise to make a resonance
c
   4      IF (ITER.GT.1.AND.test(2).LT.PDMAX.AND.LR.EQ.' ') GO TO 6
c
c   Maximum found in eigenphase sum. Fit the 3 points to quadratic.
c
          CALL SQUAD(EE3,EE2,EE1,DD3,DD2,DD1,EDELTA,DDELTA)
          ETOT=EE2+EDELTA
          EIGSUD=DD2+DDELTA
          IF (((ETOT.LT.END3.OR.ETOT.GT.END1).AND.LR.EQ.' ')
     *          .OR.EIGSUD.LE.PDMIN.OR.EIGSUD.GT.PDINF) GO TO 6
c
c  Find eigenphase sum derivative etc for energy at peak
c
   5      CALL KDIAG(ITER,IPERT,ETOT,EIG,EIGSUM,EIGSUD,Y,XX)
          IF (IPRINT.GT.0)
     *      WRITE(6,6009)ITER,EE3*ZSQ,ETOT*ZSQ,EE1*ZSQ,DD3,EIGSUD,DD1
          IF (.NOT.LEFT.AND..NOT.RITE) THEN
c
c  Re-define the 3 points taking into account new values of ETOT,EIGSUD
c
            IF (EIGSUD.GT.DD2) THEN
              IF (ETOT.LT.EE2) THEN
                EE1=EE2
                DD1=DD2
              ELSE
                EE3=EE2
                DD3=DD2
              ENDIF
              EE2=ETOT
              DD2=EIGSUD
c
c  If peak too high, jump back and bisect
c
              IF (missed.AND.ABS(DD2-0.5*(DD1+DD3)).GT.quad*DD2) THEN
                missed=.FALSE.
                GO TO 3
              ENDIF
c
c  Final fit of 3 points to a quadratic
c
              CALL SQUAD(EE3,EE2,EE1,DD3,DD2,DD1,EDELTA,DDELTA)
              IF (ETOT+EDELTA.GE.END3.AND.ETOT+EDELTA.LE.END1
     *              .AND.DDELTA.GT.0.0) THEN
                ETOT=EE2+EDELTA
                EIGSUD=DD2+DDELTA
              ENDIF
            ENDIF
c
c  Iteration finished -  retrieve the biggest eigenphase sum derivative
c
            IF (dbest.GT.EIGSUD) THEN
              EIGSUD=dbest
              EIGSUM=pbest
              ETOT=ebest
              DO J=1,NCHF
                Y(I)=yb(J)
              END DO
            ENDIF
          ENDIF
          E=ETOT*ZSQ
          GAMMA=(2./EIGSUD)*ZSQ
          xxxa=1.0/SQRT(ECH1-ETOT)
c
c   Write to P file for resonance peak
c
c           ta=2.*tan(EIGSUM)**2/(1.+tan(EIGSUM)**2)
c           tb=sqk=2.*tan(W2)**2/(1.+tan(W2)**2)
          td=PD2/ZSQ
          te=EIGSUD/ZSQ
          IF (ETOT.LT.E2) THEN
            WRITE(14,800) E,xxxa,EIGSUM,te
            WRITE(14,800) E2*ZSQ,xxxb,W2,td
          ELSE
            WRITE(14,800) E2*ZSQ,xxxb,W2,td
            WRITE(14,800) E,xxxa,EIGSUM,te
          END IF
c
c  Final check on whether this is a resonance
c
          IF (EIGSUD.GT.PDMAX.AND.EIGSUD.LE.PDINF.AND.
     *        ((ETOT.GE.E3.AND.ETOT.LE.E1).OR.LR.NE.' ')) THEN
c     gam3=0.0
c     gam33=(h1+h3)*EIGSUD-h1*PD3-h3*PD1
c     if(gam33.ne.0.0) gam3=8*h1*h3*(h1+h3)/gam33
c     if(gam3.gt.0.0) gam3=(gam3**0.3333333)*ZSQ
c
c  Write to QB.OUT file
c
            iiie=109737*E
            NC=MIN(NCHF-NCHOP,nprint)
            DO J=1,NC
              cprint(j,1)=100.0*Y(NCHOP+J)**2
              cprint(j,2)=1.0/SQRT(ECH(NCHOP+J)-ETOT)
            END DO
            strang='        '
            IF (LEFT.OR.RITE) strang=' STRANGE'
            WRITE(6,6001)xxxa,E,ev*E,EIGSUD,GAMMA,ev*GAMMA,strang
            IF (NCHOP.GT.1) WRITE(6,6002) (100.0*Y(J),J=1,NCHOP)
            WRITE(6,6003) (cprint(j,1),j=1,NC)
            IF (ECH(NCHOP+1).NE.ECH(NCHF))
     *        WRITE(6,6004) (cprint(j,2),j=1,NC)
c
c   KDECAY=IDECAY(KOUNT) = open channel with biggest partial width.
c
            KDECAY=1
            DO j=1,NCHOP
              IF (Y(j).GT.Y(KDECAY)) KDECAY=j
            END DO
c
c  increment KOUNT for each resonance and store info for routine OUTRES.
c  IWT(1,KOUNT),IWT(2,KOUNT)=closed channel with biggest,next big weight,
c    sign of IWT is reversed if weight is rather small.
c
            KOUNT=KOUNT+1
            IF (KOUNT.LE.MZEST) THEN
              IDECAY(KOUNT)=KDECAY
              RES(KOUNT)=E
              WIDTH(KOUNT)=GAMMA
              BIGEST=999.
              SMALL=0.6
              DO k=1,2
                BIG=0.0
                DO j=NCHOP+1,NCHF
                  B=ABS(Y(j))
                  IF (B.GT.BIG.AND.B.LT.BIGEST) THEN
                    BIG=B
                    IWT(k,KOUNT)=j
                  END IF
                END DO
                BIGEST=BIG
                IF(BIG.LT.SMALL) IWT(k,KOUNT)=-IWT(k,KOUNT)
                SMALL=0.25*BIG
              END DO
            END IF
          END IF
        END IF
c
c   Write to P file
c
c       tb=2.*tan(W2)**2/(1.+tan(W2)**2)
        td=PD2/ZSQ
        IF (ITER.EQ.0) WRITE(14,800)E2*ZSQ,xxxb,W2,td
    6 CONTINUE
      IF(KOUNT.GT.MZEST) PRINT *,'** Dimension problem (MZEST)',KOUNT
c
c  Write to C file
c
   10 NCHOPX=NCHOP+1
      CALL OUTRES(KOUNT,RES,WIDTH,IDECAY,IWT,ITARG,NCHOPX,ECH,BDX)
      IF (IPRINT.EQ.0) RETURN
C
C FOR QUANTUM DEFECTS ABOVE THRESHOLDS:
C
      ITER=-1
      DO jump=1,2
        IF (NCHOP.LT.NCHF) THEN
          IF (jump.GT.1.AND.ETOT.EQ.ECH(NCHOP+1)) then
            IF (ETOT.EQ.ECH(NCHF)) RETURN
            NCHOP=NCHF-1
          ENDIF
          ETOT=ECH(NCHOP+1)
          CALL KDIAG(ITER,IPERT,ETOT,EIG,EIGSUM,EIGSUD,Y,XX)
          CALL SORDER(XX,NCHOP,ISORT,IERR)
          WRITE(6,6005) ITARG(NCHOP)
          DO i=1,NCHOP
            yb(i)=ATAN(EIG(i))/PI
            COLON=' '
            IF (ISORT(i).EQ.IERR) COLON=':'
            IF (ISORT(i).GE.NCHOPX)
     *        WRITE(6,6006)yb(i),ISORT(i),COLON,(XX(j,i),j=NCHOPX,NCHOP)
          END DO
        END IF
      END DO
C
 6000 FORMAT(/'Effective n',7X,'Ryd--Position--eV',7X,'EIGSUMD',5X,
     *       'Ryd--Width--eV')
 6001 FORMAT(F11.5,1PE16.6,E12.4,E10.1,E12.3,E10.3,A8)
 6002 FORMAT('o%',15F5.1)
 6003 FORMAT('c%',15F5.1)
 6004 FORMAT('n ',15F5.2)
 6005 FORMAT(/'QUANTUM DEFECTS ABOVE NEXT THRESHOLD',I6)
 6006 FORMAT(F8.4,I5,A1,(10f6.2))
 6009 FORMAT(I2,1P6E13.6)
  800 FORMAT(1PE13.6,0P2F11.6,1P2E11.3)
C
      RETURN
      END
c***********************************************************************
c     DOUBLE PRECISION FUNCTION SDIS(E3,E2,E1,D3,D2,D1)
c
c   Find perpendicular displacement of point (E2,D2) from the line
c   joining (E1,D1) and (E3,D3).
c
c     SDIS=0.
c     temp=0.
c     S12=(D1-D2)**2-(E1-E2)**2
c     S13=(D1-D3)**2-(E1-E3)**2
c     S23=(D2-D3)**2-(E2-E3)**2
c     IF (S13.GT.0.0) temp=S23-((S23-S12+S13)**2)/(4.0*S13)
c     IF (temp.GT.0.0) SDIS=SQRT(temp)
c     RETURN
c     END
c***********************************************************************
      SUBROUTINE SQUAD(E3,E2,E1,D3,D2,D1,EDELTA,DDELTA)
      IMPLICIT REAL*8(A-H,O-Z)
c
c  Fit points (E3,D3), (E2,D2), (E1,D1) to a quadratic
c  to find values at maximum (E+EDELTA,D+DDELTA)
c
      EDELTA=0.0
      DDELTA=0.0
      H1=E2-E1
      H3=E3-E2
      B2=0.5*(D1*H3**2+D2*(H1**2-H3**2)-D3*H1**2)
      B1=     D1*H3   -D2*(H1+H3)      +D3*H1
      IF (H1.NE.0.0.AND.H3.NE.0.0.AND.B1.NE.0.0) THEN
        EDELTA=B2/B1
        DDELTA=-EDELTA*B2/(H1*H3*(H1+H3))
      ENDIF
      RETURN
      END
c***********************************************************************
      SUBROUTINE SORDER(XX,NCHOP,ISORT,IERR)
c
c  ISORT(i)=position of dominant element (or reasonably dominant
c           consistent with uniqueness) in the i-th eigenvector XX(j,i).
c  The uniqueness of these positions is such that no two eigenvectors
c  will be assigned the same value of ISORT.
c
      IMPLICIT REAL*8(A-H,O-Z)
      PARAMETER (MZCHF= 395)
      DIMENSION XX(MZCHF,MZCHF)
      INTEGER ISORT(MZCHF)
      INTEGER IBIG(MZCHF),JBIG(MZCHF)
      LOGICAL LI,LJ
c
c  Find the biggest element in the eigenvector matrix XX, make a note
c  of the row and column (jbig,ibig), and assign the first ISORT.
c  Then, excluding that row and column in subsequent searches,
c  find the next biggest element, and so on.
c
      DO k=1,NCHOP
        BIG=0.0
        DO i=1,NCHOP
          LI=.TRUE.
          LJ=.TRUE.
          IF (k.GT.1) THEN
            DO kk=1,k-1
              IF (I.EQ.IBIG(kk)) LI=.FALSE.
              IF (I.EQ.JBIG(kk)) LJ=.FALSE.
            END DO
          END IF
          IF (k.EQ.NCHOP) THEN
            IF (LI) IBIG(k)=i
            IF (LJ) JBIG(k)=i
            LI=.FALSE.
          END IF
          IF (LI) THEN
            DO j=1,NCHOP
              IF (ABS(XX(j,i)).GT.BIG) THEN
                LJ=.TRUE.
                IF(k.GT.1) THEN
                  DO kk=1,k-1
                    IF(j.EQ.JBIG(kk)) LJ=.FALSE.
                  END DO
                END IF
                IF (LJ) THEN
                  IBIG(k)=i
                  JBIG(k)=j
                  BIG=ABS(XX(j,i))
                END IF
              END IF
            END DO
          END IF
        END DO
        ISORT(IBIG(k))=JBIG(k)
      END DO
      IERR=0
      IF (NCHOP.GT.1) IERR=IBIG(NCHOP)
      RETURN
      END
c***********************************************************************
      SUBROUTINE TANDTD(R,I,TA,TDA,TP)
C
C  CALCULATES THETA AND THETAD =THETA DOT FOR  R REAL
C     THETA = TA*EXP(TP)
C     THETAD = TDA*EXP(TP)
C     TP = FNU*LOG(R) - R/FNU
C
      IMPLICIT REAL*8(A-H,O-Z)
      PARAMETER (MZCHF= 395,MZTET=  50)
      COMMON/CTHET/BB(MZCHF,MZTET),BG(MZCHF,MZTET),MSUM(MZCHF)
C
      MI=MSUM(I)
      S=BB(I,2)
      CX=0.
      E=BG(I,1)
      F2=BG(I,2)
      FNU=BB(I,1)
C
      X=2.*R/FNU
      Y=1./X
      AS=1.
      DO 10 L=3,MI
        AS=AS*Y
        S=S+BB(I,L)*AS
   10   CX=CX+BG(I,L)*AS
C
      DLR=LOG(R)
      TP=-.5*X+FNU*DLR
      TA=S
      TDA=E*((DLR+R*F2)*S+CX)
C
      RETURN
      END
c***********************************************************************
      SUBROUTINE THETA(R,I,T,TP,TD,TDP,ICONV)
C
      IMPLICIT REAL*8(A-H,O-Z)
C
      PARAMETER (MZCHF= 395,MZTET=  50)
      COMMON/CHAN/ECH(MZCHF),LLCH(MZCHF),EPS(MZCHF),FKNU(MZCHF),
     1 CC(MZCHF),RINF(MZCHF),ITARG(MZCHF),NCHF,NCHOP,NCHOP1
      COMMON/CPOT/LAMP(MZCHF,MZCHF),BW(MZCHF,MZCHF)
      COMMON/CEN/MXE,NWT,NZ,ETOT
      COMMON/CACC/AX,ACNUM,ACJWBK,ACZP,LACC
      COMMON/CTHET/BB(MZCHF,MZTET),BG(MZCHF,MZTET),MSUM(MZCHF)
      COMMON/CNTRL/IPRINT,IRAD,IPERT,KP2X
      COMMON/CPOINT/KP0,KP1,KP2,RZERO,RONE,RTWO,H
C
C     CHANGES INSERTED TO FORCE CONVERGENCE WHEN NU<LL
C     THESE INVOLVE RESETTING AC AND M
      AC=AX
      FNU=FKNU(I)
      LL=LLCH(I)
      M=FNU+LL+12
      CNTAC=0
c     IF(M.LT.MZTET)GOTO 10
      IF(M.GE.MZTET) THEN
        IF(IPRINT.GT.0)WRITE(6,700)I,M
        GOTO 100
      ENDIF
   10 F1=1./FNU
      F2=F1*F1
      X=2.*R*F1
      Y=1./X
      FL=LL
C
      R1=1./R
      A=FNU*R1-F1
      B=LOG(R)+R*F2
      C=R1+F2
      D=-R1
      E=.5*FNU**3
C
      BB(I,1)=FNU
      BB(I,2)=1.
      BG(I,1)=E
      BG(I,2)=F2
C
      YN=1.
      A1=FL-FNU
      A2=FL+FNU+1.
      BN=-2.*FNU-1.
C
      BET=1.
      GAM=0.
      W0=C
C
      S=1.
      U=0.
      CX=0.
      CY=0.
C
C
      N=0
      DO 20 MN=3,M
        N=N+1
        A1=A1+1.
        A2=A2-1.
        BN=BN+2.
        CN=A1*A2
        DN=BN+F1*CN
        GAM=CN*GAM+DN*BET
        BET=CN*BET
        YN=YN*Y
        U=U+BET*YN
        CY=CY+GAM*YN
        AN=1./N
        GAM=GAM*AN
        BET=BET*AN
        S=S+BET*YN
        CX=CX+GAM*YN
        W1=C*S*S+D*(S*CY-U*CX)
        BB(I,MN)=BET
        BG(I,MN)=GAM
        IF(ABS((W1-W0)/W1).LT.AC)GOTO 30
        WSAVE=W0
   20   W0=W1
C
C HES: PROBLEMS WITH LARGE L AND NU.LT.L+1. START FIX
      IF(FNU.LT.FL+0.01) THEN
        AC=AC*10.
C       M=M-1
        CNTAC=CNTAC+1
        IF(CNTAC.LT.2)GO TO 10
        WRITE(6,111) R,ETOT,W1,WSAVE,LL,AC
  111 FORMAT(' SR: THETA, NO CONV. FOR R,E,W,W,L,AC',4E12.4,I2,E8.1)
        WRITE(6,'('' SET RONE.GT.R AND TRY AGAIN'')')
        STOP
      ENDIF
C HES END FIX
C  NOT CONVERGED
      IF(IPRINT.GT.0)WRITE(6,610)N
      GOTO 100
C
C  SUMMATIONS CONVERGED
C  30 P=EXP(-R*F1)*R**FNU
C
C+++++++++++
C  MODIFICATION TO COPE WITH VAX EXPONENT RANGE
C
C  STATEMENT DELETED -
C  30 P=EXP(-R*F1)*R**FNU
C  RELACED BY -
   30 P=EXP(-.5*R*F1)*R**(.5*FNU)
      CFACT=1./P
      BB(I,2)=BB(I,2)*CFACT
      DO 40 J=3,N+2
        BB(I,J)=BB(I,J)*CFACT
   40   BG(I,J)=BG(I,J)*CFACT
C
C  END MODIFICATION
C+++++++++++
      T=P*S
      TP=P*(A*S+D*U)
      TD=P*E*(B*S+CX)
      TDP=P*E*((A*B+C)*S+B*D*U+A*CX+D*CY)
      N2=N+2
      MSUM(I)=N2
      ICONV=0
      RETURN
C
C  USE SUBROUTINE SC AND SET IPERT = 0
  100 CALL SC(EPS(I),LLCH(I),RZERO,AC,FSA,FSPA,FCA,FCPA)
      SINF=SIN(3.141592654*FKNU(I))
      COSF=COS(3.141592654*FKNU(I))
      T=FCA*SINF-FSA*COSF
      TP=FCPA*SINF-FSPA*COSF
CVKL               FSAP WRONG! 19/03/88
      IPERT=0
      ICONV=1
      RETURN
C
  700 FORMAT(//10X,30(1H*)/10X,'SUBROUTINE THETA'/
     1 10X,'FOR I=',I3,' REQUIRE M=',I3/
     2 10X,'WHICH IS LARGER THAN MAXIMUM VALUE OF MZTET ALLOWED BY '
     3 ,'DIMENSIONS'/10X,'USING SUBROUTINE SC WITH IPERT = 0'/10X,
     4 30(1H*))
  610 FORMAT(//10X,30(1H*)//10X,'SUBROUTINE THETA'//
     * 10X,'SUMMATIONS NOT CONVERGED WITH ',I3,' TERMS'/
     2 10X,'USING SUBROUTINE SC WITH IPERT = 1'/10X,30(1H*))
      END
c***********************************************************************
      SUBROUTINE TLAG(I,J,NLAG,LIJ,T1,T2,T3)
C
C  CALCULATES T INTEGRALS FOR CHANNELS I,J USING
C  LAGUERRE QUADRATURE WITH NLAG POINTS
C  THE INTEGRALS ARE -
C    T1 FOR (THETAI,THETAJ)
C    T2 FOR (THETADI,THETAJ)
C    T3 FOR (THETAI,THETADJ)
C
      IMPLICIT REAL*8(A-H,O-Z)
C
      PARAMETER (MZCHF= 395,MZTET=  50)
      COMMON/CTHET/BB(MZCHF,MZTET),BG(MZCHF,MZTET),MSUM(MZCHF)
      COMMON/CBLK/XLAG(30),WLAG(30),XLEG(15),WLEG(15)
      COMMON/CPOINT/KP0,KP1,KP2,RZERO,RONE,RTWO,H
C
C  CAP K
      FNUI=BB(I,1)
      FNUJ=BB(J,1)
      FK=1./FNUI+1./FNUJ
C  INITIALISE FOR QUADRATURE
      NS=NLAG/2
      M=NS*(NS-1)
      N1=M+1
      N2=M+NLAG
      T1=0.
      T2=0.
      T3=0.
C  START QUADRATURE
      DO 40 N=N1,N2
        U=XLAG(N)
        R=RTWO+U/FK
C  CALCULATE THETA FUNCTIONS
        CALL TANDTD(R,I,TI,TDI,TPI)
        CALL TANDTD(R,J,TJ,TDJ,TPJ)
C  ADD TO SUM
C++ VAX MOD
C       A1=(R**(-LIJ))*WLAG(N)*EXP(U+TPI+TPJ)
        A1=(R**(-LIJ))*WLAG(N)
        U2=.5*U
        AI=EXP(U2+TPI)
        AJ=EXP(U2+TPJ)
        TI=TI*AI
        TDI=TDI*AI
        TJ=TJ*AJ
        TDJ=TDJ*AJ
C++ END MOD
        T1=T1+TI*A1*TJ
        T2=T2+TDI*A1*TJ
   40   T3=T3+TI*A1*TDJ
      F1=1./FK
      T1=T1*F1
      T2=T2*F1
      T3=T3*F1
C
      RETURN
      END
c***********************************************************************
      SUBROUTINE VECTOR(N,EIG,X,NO,P,R,RN,XM,LXCH,B,U,V,W)
C
C      TAKES ARRAYS R OF MAIN DIAGONAL ELEMENTS, P OF SUPER DIAGONAL
C      ELEMENTS, EIG OF EIGENVALUES, OF THE TRI-DIAGONAL MATRIX
C      AND BY MEANS OF INVERSE ITERATIONS DETERMINES
C      AN EIGENVECTOR OF THE TRI-DIAGONAL MATRIX.
C
      IMPLICIT REAL*8(A-H,O-Z)
      LOGICAL ITER
      DIMENSION EIG(N),X(N),P(N),R(N),RN(N),XM(N),LXCH(N),B(N),U(N),
     A          V(N),W(N)
      DATA EPSI/1.0D-9/
C
C      THE ARRAY RN HOLDS THE MAIN DIAGONAL ELEMENTS OF A NEW
C      TRI-DIAGONAL MATRIX.
C
      DO 10 K = 1,N
        RN(K) = R(K) - EIG(NO)
   10 CONTINUE
C
C      BY MEANS OF GAUSSIAN ELIMINATION THE NEW TRI-DIAGONAL MATRIX
C      IS TRANSFORMED INTO UPPER TRIANGULAR FORM. THE ROW MULTIPLIERS
C      ARE STORED IN ARRAY XM. IF ROW I IS INTERCHANGED WITH ROW I+1
C      LXCH(I)=1  THE MAIN DIAGONAL ELEMENTS OF THE UPPER TRIANGULAR
C      MATRIX ARE STORED IN THE ARRAY U, THE NEXT DIAGONAL IN THE
C      ARRAY V AND THE LAST DIAGONAL IN THE ARRAY W.
C
      PA = RN(1)
      QA = P(1)
      N1 = N - 1
      DO 50 I = 1,N1
C
C      DETERMINE IF A ROW INTERCHANGE IS NECESSARY.
C
        GA = ABS(P(I))
        PPA = ABS(PA)
c       IF (GA.LE.EPSI) GOTO 20
        IF (GA.GT.EPSI) THEN
          IF (GA.GT.PPA) GOTO 30
        ENDIF
C
C      NO INTERCHANGE.
C
   20   CONTINUE
        U(I) = PA
        V(I) = QA
        W(I) = 0.0D0
        HA = P(I)
        PA = RN(I+1)
        QA = P(I+1)
        LXCH(I) = 0
        GOTO 40
C
C      INTERCHANGE.
C
   30   CONTINUE
        U(I) = P(I)
        V(I) = RN(I+1)
        W(I) = P(I+1)
        HA = PA
        PA = QA
        QA = 0.0D0
        LXCH(I) = 1
C
C      THE ROW MULTIPLIER IS DETERMINED.
C
   40   CONTINUE
        XM(I) = HA/U(I)
C
C      ROW I IS MULTIPLIED BY XM(I) AND SUBTRACTED FROM ROW I+1.
C
        HA = 0.0D0
        PA = PA - XM(I)*V(I)
        QA = QA - XM(I)*W(I)
   50 CONTINUE
C
C      THE SINGLE ELEMENT IN THE LAST ROW IS PLACED IN U(N)
C
      U(N) = PA
      IF (ABS(U(N)).LT.EPSI) U(N) = EPSI
C
C      THE ARRAY X IS RESERVED FOR THE CURRENT ESTIMATE OF THE
C      EIGENVECTOR. THE SUBROUTINE BAKSUB IS CALLED AND AN ESTIMATE OF
C      THE EIGENVECTOR OBTAINED.
C
      DO 70 I = 1,N
        B(I) = 1.0D0
   70 CONTINUE
      CALL BAKSUB(N,X,B,U,V,W)
C
C      THE EIGENVECTOR STORED IN X IS NORMALISED.
C
      CALL NORM(N,X)
C
C      TO SAVE COMPUTING TIME, A FURTHER ITERATION TO YIELD A BETTER
C      ESTIMATE OF THE VECTOR CAN BE SUPPRESSED BY SETTING ITER=.TRUE.
C
      ITER = .FALSE.
      IF (ITER) GOTO 130
C
C      A NEW COLUMN VECTOR RELATED TO X BY EXACTLY THE SAME ROW
C      INTERCHANGES AND MULTIPLICATIONS BY WHICH THE UPPER TRIANGULAR
C      MATRIX WAS OBTAINED FROM THE NEW TRI-DIAGONAL MATRIX, IS
C      CALCULATED AND STORED IN B.
C
      DO 100 I = 1,N
        B(I) = X(I)
  100 CONTINUE
      N1 = N - 1
      DO 120 I = 1,N1
c       IF (LXCH(I).EQ.0) GOTO 110
        IF (LXCH(I).NE.0) THEN
          AC = B(I)
          BC = B(I+1)
          B(I) = BC
          B(I+1) = AC
        ENDIF
  110   B(I+1) = B(I+1) - XM(I)*B(I)
  120 CONTINUE
C
C      THE NEW VECTOR STORED IN B IS NORMALISED.
C
      CALL NORM(N,B)
C
C      THE SUBROUTINE BAKSUB IS CALLED AND A NEW ESTIMATE OF THE
C      EIGENVECTOR IS OBTAINED.
C
      CALL BAKSUB(N,X,B,U,V,W)
C
C      THE NEW ESTIMATE OF THE EIGENVECTOR IS NORMALISED.
C
      CALL NORM(N,X)
  130 CONTINUE
C
      END      
c***********************************************************************
      SUBROUTINE ZPHI(I,ZR,ZAI,ZPI)
C
C  COMPUTES AMPLITUDE ZAI AND PHASE ZPI OF COULOMB FUNCTION ZPHI
C  FOR COMPLEX RADIAL CO-ORDINATE ZR.
C  USES DATA IN ARRAY D WHICH IS HELD IN COMMON/CJWBK/
C  AND SHOULD HAVE BEEN COMPUTED IN SUBROUTINE INJWBK.
C  THE STRUCTURE OF ZPHI IS SIMILAR TO THAT OF SUBROUTINE JWBK.
C
C
      IMPLICIT REAL*8(A-H,O-Y)
      IMPLICIT COMPLEX(Z)
      PARAMETER (MZCHF= 395,MX15N=15*MZCHF)
      COMMON/CJWBK/D(MX15N)
      COMMON/CACC/AC,ACNUM,ACJWBK,ACZP,LACC
C
C
      J=(I-1)*15
      E=D(J+1)
      C=D(J+4)
      ZX=1./ZR
C
      IF(C.EQ.0)GOTO 30
      IF(E.EQ.0)GOTO 70
C
C  CASE OF C.GT.0 AND E.GT.0
      ZW=E+ZX*(2.-C*ZX)
      ZWH=SQRT(ZW)
      Z=ZR*ZWH
      FK=D(J+2)
      ZRK=ZR*FK
      ZRMC=ZR-C
      ZALP=Z+ZRK
      CK=D(J+10)
C  COMPUTE PHASE
      ZP=Z+D(J+15)
C  LOG TERM
      ZB=FK*ZALP
      IF(ABS(ZB).GT.ACJWBK)GOTO 10
      ZB=-ZB
      ZP=ZP+ZALP*((((.2*ZB+.25)*ZB+.3333333333)*ZB+.5)*ZB+1.)
      GOTO 20
   10 ZP=ZP+D(J+3)*LOG(1.+ZB)
C  ARCTAN TERM
  20  ZA1=1./(ZR*D(J+7))
      ZS=D(J+5)*(Z-FK*ZRMC)*ZA1
      ZG=(CK*Z+ZRMC)*ZA1
      ZP=ZP+D(J+6)*(0.,-1.)*LOG(ZG+(0.,1.)*ZS)
C  CAP PHI TERM
      ZP=ZP+((5.*ZRMC/(Z*Z))-(Z*D(J+9)+ZRK*D(J+8)+CK)/
     C (ZALP*D(J+7)))/(24.*Z)
C  COMPLETE CALCULATION OF ZPHI
      ZA1=.0625*(ZX/ZW)**3
      ZCC=ZA1*(((D(J+14)*ZX+D(J+13))*ZX+D(J+12))*ZX+D(J+11))
      ZPI=ZP
      ZAI=(1.-ZCC)/SQRT(ZWH)
      RETURN
C
   30 IF(E.EQ.0)GOTO 60
C
C  CASE OF C.EQ.0 AND E.GT.0
      ZW=2.*ZX+E
      ZWH=SQRT(ZW)
      Z=ZR*ZWH
      FK=D(J+2)
      ZRK=ZR*FK
      ZALP=Z+ZRK
C  COMPUTE PHASE
      ZP=Z+D(J+15)
      ZB=FK*ZALP
      IF(ABS(ZB).GT.ACJWBK)GOTO 40
      ZB=-ZB
      ZP=ZP+ZALP*((((.2*ZB+.25)*ZB+.3333333333)*ZB+.5)*ZB+1.)
      GOTO 50
   40 ZP=ZP+D(J+3)*LOG(1.+ZB)
   50 ZP=ZP+1/(4.*ZALP)+(5.*ZR/(Z*Z)-2.*(Z+ZALP)/ZALP)/(24.*Z)
C  COMPLETE CALCULATION OF ZPHI
      ZA1=.0625*(ZX/ZW)**3
      ZCC=ZA1*(-4.*E-3.*ZX)
      ZPI=ZP
      ZAI=(1.-ZCC)/SQRT(ZWH)
      RETURN
C
C  CASE OF C.EQ.0 AND E.EQ.0
   60 ZW=2.*ZX
      ZWH=SQRT(ZW)
      Z=ZR*ZWH
      ZP=2.*Z*(1.+.046875*ZX)+D(J+15)
      ZWMQ=1./SQRT(ZWH)
      ZET=(1.+.0234375*ZX)*ZWMQ
      ZAI=ZET
      ZPI=ZP
      RETURN
C
C  CASE OF E.EQ.0 AND C.GT.0
   70 ZW=ZX*(2.-C*ZX)
      ZWH=SQRT(ZW)
      Z=ZR*ZWH
      ZRMC=ZR-C
C  COMPUTE PHASE
      ZP=2.*Z+D(J+15)
      ZA1=1./ZR
      ZS=D(J+5)*Z*ZA1
      ZG=ZRMC*ZA1
      ZP=ZP+D(J+6)*(0.,-1.)*LOG(ZG+(0.,1.)*ZS)
      ZP=ZP-(3.*ZR+C)/(24.*(ZRMC+ZR)*Z)
C  COMPLETE CALCULATION OF ZPHI
      ZA1=.0625*(ZX/ZW)**3
      ZCC=((D(J+14)*ZX+D(J+13))*ZX-3.)*ZX*ZA1
      ZAI=(1.-ZCC)/SQRT(ZWH)
      ZPI=ZP
C
      RETURN
      END
      COMPLEX FUNCTION ZPLAG(I,J,NLAG,LIJ)
C
C  CALCULATES P INTEGRALS USING LAGUERRE QUADRATURE
C
      IMPLICIT REAL*8(A-H,O-Y)
      IMPLICIT COMPLEX(Z)
C
      PARAMETER (MZCHF= 395)
      COMMON/CBLK/XLAG(30),WLAG(30),XLEG(15),WLEG(15)
      COMMON/CPOINT/KP0,KP1,KP2,RZERO,RONE,RTWO,H
      COMMON/CHAN/ECH(MZCHF),LLCH(MZCHF),EPS(MZCHF),FKNU(MZCHF),
     1 CC(MZCHF),RINF(MZCHF),ITARG(MZCHF),NCHF,NCHOP,NCHOP1
      COMMON/CACC/AC,ACNUM,ACJWBK,ACZP,LACC
C
      FK=FKNU(I)+FKNU(J)
      X=RTWO
      B=SQRT(8.*X)
      G=FK*.125*B
      IF(FK.GT.0)THEN
      GM=1./G
      G2=1.+G
      G2=G2*G2
      ENDIF
      ZB=(0.,1.)/B
C
      NS=NLAG/2
      M=NS*(NS-1)
      N1=M+1
      N2=M+NLAG
C
      ZPLAG=0.
      DO 30 N=N1,N2
      U=XLAG(N)
      A1=FK*U
      IF(A1.LE.ACZP)THEN
      ZET=1.+.5*ZB*U
      ELSE
      ZET=(SQRT(G2+ZB*G*U)-1.)*GM
      ENDIF
      ZMU=-8.*ZB*(G+1./ZET)
      ZET=ZET*ZET
      ZR=ZET*X
      CALL ZPHI(I,ZR,ZAI,ZPI)
      CALL ZPHI(J,ZR,ZAJ,ZPJ)
   30 ZPLAG=ZPLAG+ZAI*ZAJ*EXP((0.,1.)*(ZPI+ZPJ)+U)*
     1 (ZR**(-LIJ))*WLAG(N)/ZMU
C
      RETURN
      END
      COMPLEX FUNCTION ZQLAG(I,J,NLAG,LIJ)
C
C  CALCULATES Q INTEGRALS USING LAGUERRE QUADRATURE
C
      IMPLICIT REAL*8(A-H,O-Y)
      IMPLICIT COMPLEX(Z)
C
      PARAMETER (MZCHF= 395)
      COMMON/CBLK/XLAG(30),WLAG(30),XLEG(15),WLEG(15)
      COMMON/CPOINT/KP0,KP1,KP2,RZERO,RONE,RTWO,H
      COMMON/CHAN/ECH(MZCHF),LLCH(MZCHF),EPS(MZCHF),FKNU(MZCHF),
     1 CC(MZCHF),RINF(MZCHF),ITARG(MZCHF),NCHF,NCHOP,NCHOP1
C
      FK=FKNU(I)-FKNU(J)
      X=RTWO
      ZMUM=(0.,1.)/FK
C
      NS=NLAG/2
      M=NS*(NS-1)
      N1=M+1
      N2=M+NLAG
C
      ZQLAG=0.
      DO 30 N=N1,N2
      U=XLAG(N)
      ZR=X+U*ZMUM
      CALL ZPHI(I,ZR,ZAI,ZPI)
      CALL ZPHI(J,ZR,ZAJ,ZPJ)
   30 ZQLAG=ZQLAG+ZAI*ZAJ*EXP((0.,1.)*(ZPI-ZPJ)+U)*
     1 (ZR**(-LIJ))*WLAG(N)
      ZQLAG=ZQLAG*ZMUM
C
      RETURN
      END
      COMPLEX FUNCTION ZQLEG(I,J,NLEG,LIJ)
C
C  CALCULATES Q INTEGRALS USING LEGENDRE QUADRATURE
C
      IMPLICIT REAL*8(A-H,O-Y)
      IMPLICIT COMPLEX(Z)
C
      PARAMETER (MZCHF= 395)
      COMMON/CBLK/XLAG(30),WLAG(30),XLEG(15),WLEG(15)
      COMMON/CPOINT/KP0,KP1,KP2,RZERO,RONE,RTWO,H
      COMMON/CHAN/ECH(MZCHF),LLCH(MZCHF),EPS(MZCHF),FKNU(MZCHF),
     1 CC(MZCHF),RINF(MZCHF),ITARG(MZCHF),NCHF,NCHOP,NCHOP1
C
      FK=FKNU(I)-FKNU(J)
      X=RTWO
      IF(FK.EQ.0.)THEN
        ZA=1.
      ELSE
        ZA=(0.,1.)/(1.+FK*X)
      ENDIF
C
      NS=NLEG/2
      J1=(NS*(NS-1))/2
      J2=J1+NS
      J1=J1+1
C
      ZQLEG=0.
      DO 10 JJ=J1,J2
      V=XLEG(JJ)
      ZR1=X*(1.+ZA*(1.-V)/(1.+V))
      ZR2=X*(1.+ZA*(1.+V)/(1.-V))
      CALL ZPHI(I,ZR1,ZAI,ZPI)
      CALL ZPHI(J,ZR1,ZAJ,ZPJ)
      ZF=ZAI*ZAJ*EXP((0.,1.)*(ZPI-ZPJ))*
     1 (ZR1**(-LIJ))/(1.+V)**2
      CALL ZPHI(I,ZR2,ZAI,ZPI)
      CALL ZPHI(J,ZR2,ZAJ,ZPJ)
      ZF=ZF+ZAI*ZAJ*EXP((0.,1.)*(ZPI-ZPJ))*
     1 (ZR2**(-LIJ))/(1.-V)**2
   10 ZQLEG=ZQLEG+ZF*WLEG(JJ)
      ZQLEG=2.*X*ZA*ZQLEG
C
      RETURN
      END
c***********************************************************************
      SUBROUTINE ZSLAG(I,J,NLAG,LIJ,ZS3,ZSD3)
C
C  CALCULATES S INTEGRALS USING LAGUERRE QUADRATURE
C
      IMPLICIT REAL*8(A-H,O-Y)
      IMPLICIT COMPLEX(Z)
C
      PARAMETER (MZCHF= 395)
      COMMON/CBLK/XLAG(30),WLAG(30),XLEG(15),WLEG(15)
      COMMON/CPOINT/KP0,KP1,KP2,RZERO,RONE,RTWO,H
      COMMON/CHAN/ECH(MZCHF),LLCH(MZCHF),EPS(MZCHF),FKNU(MZCHF),
     1 CC(MZCHF),RINF(MZCHF),ITARG(MZCHF),NCHF,NCHOP,NCHOP1
C
      ZFK=CMPLX(FKNU(I),1./FKNU(J))
      X=RTWO
      B=SQRT(2.*X)
      ZB=(0.,1.)/B
      ZG=.5*ZFK*B
C
      ZA1=1./ZG
      ZA2=1.+ZG
      ZA2=ZA2*ZA2
      ZA3=ZB*ZG
C
      NS=NLAG/2
      M=NS*(NS-1)
      N1=M+1
      N2=M+NLAG
C
      ZS3=0.
      ZSD3=0.
      DO 10 N=N1,N2
      U=XLAG(N)
      ZET=ZA1*(SQRT(ZA2+ZA3*U)-1.)
      ZMUM=-.5*ZET/(ZB*(1.+ZET*ZG))
      ZET=ZET*ZET
      ZR=ZET*X
      CALL ZPHI(I,ZR,ZAI,ZPI)
      CALL ZTHETA(J,ZR,ZTAJ,ZTDAJ,ZTPJ)
      ZB1=(ZR**(-LIJ))*WLAG(N)*ZMUM*
     1 EXP((0.,1.)*ZPI+ZTPJ+U)*ZAI
      ZS3=ZS3+ZB1*ZTAJ
   10 ZSD3=ZSD3+ZB1*ZTDAJ
C
      RETURN
      END
c***********************************************************************
      SUBROUTINE ZTHETA(I,ZR,ZTA,ZTDA,ZTP)
C
C  CALCULATES THETA AND THETAD FOR CHANNEL I AND COMPLEX ZR
C     THETA = ZTA*CEXP(ZTP)
C     THETAD = ZTDA**CEXP(ZTP)
C     ZTP = FNUI*LOG(ZR) - ZR/FNUI
C
      IMPLICIT REAL*8(A-H,O-Y)
      IMPLICIT COMPLEX(Z)
C
      PARAMETER (MZCHF= 395,MZTET=  50)
      COMMON/CTHET/BB(MZCHF,MZTET),BG(MZCHF,MZTET),MSUM(MZCHF)
C
      MI=MSUM(I)
      FNUI=BB(I,1)
      E=BG(I,1)
      F2=BG(I,2)
      Z=2.*ZR/FNUI
      ZY=1./Z
      ZAS=1.
      ZS=BB(I,2)
      ZX=0.
      DO 10 M=3,MI
      ZAS=ZAS*ZY
      ZX=ZX+BG(I,M)*ZAS
   10 ZS=ZS+BB(I,M)*ZAS
C
      ZDLR=LOG(ZR)
      ZTP=(FNUI*ZDLR-.5*Z)
      ZTA=ZS
      ZTDA=E*((ZDLR+ZR*F2)*ZS+ZX)
C
      RETURN
      END
