! Copyright 2019
!
! For a comprehensive list of the developers that contributed to these codes
! see the UK-AMOR website.
!
! This file is part of UKRmol-out (UKRmol+ suite).
!
!     UKRmol-out is free software: you can redistribute it and/or modify
!     it under the terms of the GNU General Public License as published by
!     the Free Software Foundation, either version 3 of the License, or
!     (at your option) any later version.
!
!     UKRmol-out is distributed in the hope that it will be useful,
!     but WITHOUT ANY WARRANTY; without even the implied warranty of
!     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!     GNU General Public License for more details.
!
!     You should have received a copy of the GNU General Public License
!     along with  UKRmol-out (in source/COPYING). Alternatively, you can also visit
!     <https://www.gnu.org/licenses/>.
!
      subroutine ROTIONS_LIN(IFAIL)
C
C***********************************************************************
C
C     ROTIONS calculates rotational cross sections of linear molecular
c      ions by electron impact. It uses  T-matrices
C      and the Coulomb-Born approximation.
C      It is intended to be a self contained module which can be
C      run independently from the main scattering calculation.
C
C***********************************************************************
C
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      integer anr_op,cb_op,grid
c
      PARAMETER (maxene=10000,MAXERN=10)
      parameter (maxnsym=12)
      parameter (ev=27.2114d0,angst2=.280028560859d0,
     * pi=3.1415926535898d0)
c
      DIMENSION EINC(2,MAXERN),NESCAT(MAXERN),IPRNT(6)
      dimension lutmtv(maxnsym),luchanv(maxnsym)
      dimension nchan(maxnsym),maxchi(maxnsym),maxchf(maxnsym),
     1          mgvn(maxnsym)
      integer, allocatable :: ivchl(:),lvchl(:),mvchl(:)
      dimension maxvi(maxene,maxnsym),maxvj(maxene,maxnsym),
     1          ndopen(maxene,maxnsym),ntset(maxnsym)
      dimension num_pw(maxnsym)
      double precision, allocatable :: tmr(:,:),tmi(:,:),enscat(:),
     * evchl(:),sigma(:),acbsigma(:),tcbsigma(:)
c
      common /binomium/binom(100,100)  
C! this matrix is used in calculating 3j-symbols

      CHARACTER*80 NAME,name_s
      CHARACTER*4 CEUNIT(3)
      CHARACTER*9 FORM
      CHARACTER*1 ITFORM
      CHARACTER*11 TFORM,MODDAT
      INTEGER STOT,SYMTYP
C
C***********************************************************************
C
C     Basic data is input via namelist /XSECIN/
c      anr_op   = Switch to activate the ANR correction in the cross
c                 sections
c                 0= no correction, 1= correction applied (default)
c      be       = rotational constant (in cm-1)
c      cb_op    = Switch to activate the use of Coulomb-Born approx
c                 0= do NOT perform CB calculations (only T-matrix 
c                    cross sections)
c                 1= perform T-matrix plus CB calculations
c                 2= Only perform CB calculations (not T-matrices are
c                    required)
C      EMIN     = MINIMUM REQUIRED SCATTERING ENERGY (IN UNITS AS
C                 SPECIFIED BY IEUNIT)
C      EMAX     = MAXIMUM REQUIRED SCATTERING ENERGY
c      grid     = Kind of grid for energies when cb_op=2: 1=linear, 
c                 2=logarithmic
C      ITFORM   = 'F' if T-matrix dataset is formatted, else 'U'
C      IEUNIT   = UNITS of the  INPUT and output SCATTERING ENERGIES
C                 1= eV, 2= Hartree, 3= Ryd
c      IPRNT    = IPRNT(1)=1: print out all the input data. 
c                 IPRNT(4)=1,2,3,4: print out output data 
c                               (1 - short output, 4 -detailed output)     
C      IWRITE   = UNIT FOR console output
C      IXSN     = Units to be used for cross-section output
C                 2=ANGSTROM**2, 3=PI*BOHR**2
c      ixsout   = Logical unit for cross sections output 
c      luchanv  = vector containing the logical units for channel input 
c                 files
c      lutmtv   = vector containing the logical units for t-matrix input
c                 files
C      MAXI     = LABEL OF HIGHEST INITIAL STATE FOR WHICH CROSS-
C                 SECTIONS ARE REQUIRED
C      MAXF     = LABEL OF HIGHEST FINAL STATE FOR WHICH CROSS-SECTIONS
C                 ARE REQUIRED
C      NAME     = TITLE FOR ANY OUTPUT
C      NTSET    = SET NUMBER OF T-MATRIX INPUT
c      nsym     = Number of molecular symmetries to consider
C      numener  = Number of energies to be read (the T-matrix files may 
c                 have T-matrix for a number of energies and this allows 
c                 not to read all of them but just "numener".) 
c                 If numener<1 (default), all T-matrices will be read.
c      q1       = Permanent dipole of the target (in Debye)
c      lmax     = Maximum value of l (partial waves) to add in the 
c                 computation of the partial CB cross section.
c      lmaxq    = Number of partial waves to use in the computation of
c                 the total quadrupole CB cross section
c      q2       = Permanent quadrupole of the target (in atomic units)
c      aq       = Lower limit for the numerical integral of the quadrupoles
c                 M02 and M20
c      bq       = Upper limit for the numerical integral of the quadrupoles
c                 M02 and M20

      NAMELIST/RXSECIN/IXSN,NVXPRT,IEUNIT,IWRITE,MAXI,MAXF,NAME,
     1LUCHANv,EMIN,EMAX,LUTMTv,NTSET,ITFORM,IPRNT,R,nsym,ixsout,
     2numener,j,jp,mjj,be,anr_op,cb_op,q1,lmax,lmaxq,q2,aq,bq,ion,grid
C
C***********************************************************************
c  Input variable Defaults

      data nsym/1/,numener/0/,j,jp/0,1/,mjj/6/
      DATA IXSN/1/,NVXPRT/0/,IWRITE/6/,VBIG/1.D+8/,R/0.D0/
     3,LUCHANv,LUTMTv/maxnsym*10,maxnsym*12/,NTSET/maxnsym*1/
     4,CEUNIT/' eV ',' E_h',' RYD'/
     4,IEUNIT/1/,IPRNT/6*0/,MAXI,MAXF/1,0/
     5,anr_op/1/,ixsout/8/,cb_op/0/,lmax/5/,q1/0.d0/,lmaxq/20/,k/0/
     6,q2/0.d0/,aq,bq/1.d-1,2.d4/,ion/1/,grid/1/

      DATA ZERO/0.D0/,RYD/0.073500D0/
      DATA TFORM,FORM/2*'FORMATTED'/,ITFORM/'U'/
      DATA MODDAT/'13-Jun-2001'/

      IFAIL = 0
      NEXT = 1
      EMIN = ZERO
      EMAX = VBIG
C
C---- Read basic data via namelist /RXSECIN/
      READ(5,RXSECIN)
      IF(ITFORM.EQ.'U') TFORM='UN'//FORM
C
C---- Date-stamp run and print title
*      CALL DATEST(DAYTIM)
*      NAME(61:) = DAYTIM
      name_s = name
      WRITE(IWRITE,100)MODDAT,NAME
 100  FORMAT(' Program ROTIONS  (last modified ',A,' )'//A/)

c ---------------------------------------------------------------------
c be ... rotational constant. It's used to calculate the rotational thresholds
c        with the formula E_j=2*be*j(j+1). It is read in 1/cm and here is
c        converted to au.

      be=be*4.5563352672d-6

c q1 is the permanent dipole moment of the target. Here is converted
c from Debye to a.u.

      q1=q1/2.54158d0
c ---------------------------------------------------------------------

      write(iwrite,495)ion
 495  format(/,' The charge of the target is ',i2)

      if (anr_op.eq.1) write(iwrite,500)
 500  format(/,' The cross sections will include the ANR correction.')

      if (cb_op.eq.1.or.cb_op.eq.2) write(iwrite,505)
 505  format(/,' Coulomb-Born cross sections will be obtained.') 
      if (cb_op.eq.1.or.cb_op.eq.2) then
         if (abs(jp-j).eq.1.and.q1.eq.0.d0) stop 'CB_op=1 or 2 and 
     & |jp-j| =1 requires a non-zero value for q1.'
         if (abs(jp-j).eq.2.and.q2.eq.0.d0) stop 'CB_op=1 or 2 and 
     & |jp-j| =2 requires a non-zero value for q2.'
      end if
c      
      if (cb_op.eq.2) go to 5811
c
C----- Find required T-matrix set and read dimension information
      call GETDIM(nsym,TFORM,NTSET,LUTMTv,maxtgt,maxch,maxmi,maxmf,
     1            netot,IPRNT(2),IFAIL)
c
      allocate (ivchl(maxch),lvchl(maxch),mvchl(maxch),evchl(maxch))
      LTMT = MAXmi*MAXmf*netot 
      allocate (enscat(netot),tmr(ltmt,nsym),tmi(ltmt,nsym),
     * sigma(netot))

      do js=1,nsym   
c! js is the index referring to total molecular symmetries

         write(iwrite,510)js
 510     format(/' ----------------------------------------------------
     &T-matrix ',i2)

         lutmt=lutmtv(js)
         luchan=luchanv(js)

         CALL READTH(LUTMT,NAME,NTSET(js),NCHAN(js),NVIB,NDIS,NTARG,
     *               MAXCHI(js),MAXCHF(js),MGVN(js),STOT,symtyp,NE,
     *               NERANG,NESCAT,EINC,IVCHL,
     *               LVCHL,MVCHL,EVCHL,TFORM,IWRITE,IPRNT(1),IFAIL)
         IF(IFAIL.NE.0) stop 'FAIL reading T-matrix header'
         NETOT = NE

         if (numener.gt.maxene.or.netot.gt.maxene) then
            write(iwrite,512) 
            numener=maxene
         end if
 512     format(/,' Parameter MAXENE is smaller than the number of energ
     &ies to be read. NUMENER reset to MAXENE.')

         if (numener.gt.0) then
            write(iwrite,515)ne,numener
            netot=numener
         end if
 515     format(/,' The number of T-matrices in the file is ',i5,' bu
     &t only the first ',i5,' will be read')

c The T-matrices files may content information for transitions from several
c initial states to several final states. For the moment, this program will
c only cope with rotational transitions within the same electronic 
c (and vibrational, if the T-matrix is vibrationally resolved) state.
c To do this, it is necessary to know the number of partial waves in the
c T-matrix file, and to get this number we use the next lines:  

         if (symtyp.eq.0) then
            num_pw(js)=lmax-lvchl(1)+1
         else if (symtyp.eq.1) then
            if (mod(lmax,2).eq.0) then
               num_pw(js)=(lmax-lvchl(1)+1)/2+1
            else 
               num_pw(js)=(lmax-lvchl(1)+1)/2
            endif
         else if (symtyp.eq.-1) then
            if (mod(lmax,2).eq.0) then
               num_pw(js)=(lmax-lvchl(1)+1)/2
            else 
               num_pw(js)=(lmax-lvchl(1)+1)/2+1
            endif 
         endif
         
         write(iwrite,520)num_pw(js),lvchl(1),lmax
 520     format(/' The number of partial waves is ',i2,
     &           ' (l=',i2,' to ',i2,')')

c save values about energy range to compare with other files to read

         if (js.gt.1) then
            ii=lutmtv(js)
            ih=lutmtv(js-1)
            if (nvib_old.ne.nvib) then; write(iwrite,525)ii,ih; endif
 525        format(' NVIB not the same in ',i2,' and ',i2)
            if (ndis_old.ne.ndis) then; write(iwrite,526)ii,ih; endif
 526        format(' NDIS not the same in ',i2,' and ',i2)
            if (ntarg_old.ne.ntarg)then; write(iwrite,527)ii,ih; endif
 527        format(' NTARG not the same in ',i2,' and ',i2)
            if (stot_old.ne.stot) then; write(iwrite,528)ii,ih; endif
 528        format(' STOT not the same in ',i2,' and ',i2)
            if (symtyp_old.ne.symtyp)then;write(iwrite,529)ii,ih; endif
 529        format(' GUTOT not the same in ',i2,' and ',i2)
            if (ne_old.ne.ne) then; write(iwrite,530)ii,ih; endif
 530        format(' NE not the same in ',i2,' and ',i2)
            if (nerang_old.ne.nerang) then;write(iwrite,531)ii,ih;endif
 531        format(' NERANG not the same in ',i2,' and ',i2)
*            if (nescat_old.ne.nescat) then;write(iwrite,532)ii,ih;endif
* 532       format(' NESCAT not the same in ',i2,' and ',i2)
*            if (einc_old.ne.einc) then; write(iwrite,533)ii,ih; endif
* 533       format(' EINC not the same in ',i2,' and ',i2)
         end if

         nvib_old=nvib
         ndis_old=ndis
         ntarg_old=ntarg
         stot_old=stot
         symtyp_old=symtyp
         ne_old=ne
         nerang_old=nerang         
*         nescat_old=nescat
*         einc_old=einc

C
C---- Recompute energy parameters NERANG,NESCAT and EINC for energy
C     range [EMIN,EMAX] (this just need to be done once, but it doesn't 
c     take long to do it for every file read)
         emin0=emin
         emax0=emax
         IF(IEUNIT.EQ.1) THEN
            EMIN0 = RYD*EMIN
            EMAX0 = RYD*EMAX
         ENDIF
         if (ieunit.eq.2) then
            emin0 = 2.d0*emin
            emax0 = 2.d0*emax
         end if

c  get information about the energy range to explore.

         CALL NEWE(EMIN0,EMAX0,NE,NERANG,NESCAT,EINC)

         EMINP = EMIN0
         EMAXP = EMAX0
         IF(IEUNIT.EQ.1) THEN
            EMINP = EMIN0/RYD
            EMAXP = EMAX0/RYD
         End if
         if (ieunit.eq.2) then            
            EMINP = 0.5d0*EMIN0
            EMAXP = 0.5d0*EMAX0
         ENDIF
         WRITE(IWRITE,104) NE,EMINP,EMAXP,CEUNIT(IEUNIT)

C
C----- Assign storage for energy independant data
 
         write(iwrite,540)js
 540     format(/,' ---> T-matrix ',i2)
c
         IF(NVIB.EQ.0) NVIB=NTARG
         IF(MAXF.EQ.0) MAXF=NVIB
         NVIBD  = max(NVIB+NDIS,ntarg)
c
C----- ALLOCATE SPACE FOR T-MATRICES AND CROSS-SECTIONS
c T-matrices for more than one energy will be read (in general), so 
c these pointers have to allow for that.
cC
C---- read "netot" T-matrices
         net = netot
         CALL READT(net,MAXCHI(js),MAXCHF(js),NDIS,MAXVI(1,js),
     *              MAXVJ(1,js),
     *              NDOPEN(1,js),TMR(1,js),TMI(1,js),enscat)

         write(iwrite,550)js,net
 550   format(/,' T-matrices in symmetry ',i2,' have been read for ',i5,
     &         ' energies',/)

c
c end do loop on symmetries (js)

      end do
C
C---- Calculate rotational cross sections
c
      call TROTSEC(nsym,maxene,ltmt,maxchi,maxchf,maxvi,maxvj,ndopen,
     1     mgvn,enscat,tmr,tmi,num_pw,symtyp,def,
     1     j,jp,mjj,be,net,anr_op,sigma,iprnt(4),iwrite)


c Calculate Coulomb-Born cross sections: total and partial 
c dipole allowed transitions...

      if (ion.le.0.and.cb_op.eq.1) then 
         write(iwrite,560)ion
 560     format(' **Coulomb-Born approximation cannot be used because 
     &ION=',i1)
         cb_op=0
      end if

      if (abs(jp-j).gt.2.and.cb_op.eq.1) then
         write(iwrite,570)jp-j
 570     format(' **Coulomb-Born approximation only implemented for |j-jp|
     &<=2, but not =',i2)
         cb_op=0
      end if

 5811 if (cb_op.eq.1.or.cb_op.eq.2) then

c create energy grid in case of no T-matrices are given

         if (cb_op.eq.2) then
            if (emin.le.0.d0) stop 'EMIN should be > 0'
            if (emax.le.0.d0) stop 'EMAX should bd > 0'

            if (ieunit.eq.1) then   ! Energy in T-matrix files is in RYD. 
               emin=emin*RYD        ! As we use the same variable to store
               emax=emax*RYD        ! the energy grid, this is set up in RYD.
            end if
            if (ieunit.eq.2) then
               emin=2.d0*emin
               emax=2.d0*emax
            end if

            if (grid.eq.1) then ! linear grid 
               if (numener.le.1) then
                  paso=0.d0
                  numener=1
               else
                  paso=(emax-emin)/dble(numener-1)
               end if
               write(iwrite,582)emin,emax,numener
 582           format(/,' Linear grid in energy (RYD) with EMIN=',
     &                e9.2,' EMAX=',e9.2,' NUMENER=',i5)
               allocate (enscat(numener))
               do ie=1,numener
                  enscat(ie)=emin+paso*dble(ie-1)
               end do
            end if
            if (grid.eq.2) then ! logarithmic grid
               if (numener.le.1) then
                  paso=0.d0
                  numener=1
               else
                  paso=(dlog(emax)-dlog(emin))/dble(numener-1)
               end if
               write(iwrite,5831)emin,emax,numener
 5831          format(/,'Logarithmic grid in energy (RYD) with EMIN=',
     &                e9.2,' EMAX=',e9.2,' NUMENER=',i5) 
               allocate (enscat(numener))
               do ie=1,numener
                  enscat(ie)=exp(log(emin)+paso*dble(ie-1))
               end do
            end if
            net=numener
            allocate (sigma(net))
         end if

         allocate (tcbsigma(net),acbsigma(net))

c call routine to initialize binom, neccessary for the threej function
         call pascal(binom,100)

         write(iwrite,580)
 580     format(/,' **** Coulomb-Born rotational cross sections.',//,
     &          '   ie ',6x,'ei (eV)',7x,'ef (eV)',7x,'TCB (A^2)') 
c
         do ie=1,net
            ei=enscat(ie)/2.d0  ! energy is converted from Ryd to au
            def=be*dble(jp*(jp+1)-j*(j+1)) ! be should be in au!
            ef=ei-def
            if (ef.lt.0.d0) goto 600
            if (abs(jp-j).eq.1) then
               if (lmax.gt.40) write(iwrite,581)
 581     format(' Warning! LMAX is very large and there is risk of
     &overflow.')
c
c Dipole CB cross sections...
c
              call DCB(ei,ef,j,jp,lmax,q1,ion,dxs,pdxs,iwrite,iprnt(4))
c
              tcbsigma(ie)=dxs  ! total CB-cross sections
              acbsigma(ie)=pdxs ! sum of the first lmax partial-CB XS
c
            end if
c
c Quadrupole CB cross sections...
c check k, qa, qb, npq
c
            if (abs(jp-j).eq.2) then
               if (lmax.gt.lmaxq) then
                  write(iwrite,583)
                  lmax=lmaxq
               end if
 583           format(' Warning!! lmax should not be > lmaxq',/,
     &                '           lmax has been reset to lmax=lmaxq')
               call QCB(ei,ef,j,jp,k,lmax,lmaxq,q2,ion,aq,bq,
     &                   qxs,pqxs,iwrite,iprnt(4))
               tcbsigma(ie)=qxs  ! total CB-cross sections
               acbsigma(ie)=pqxs ! sum of the first lmax partial-CB XS
            end if
c
c Threshold correction to the cross-sections
c
            if (anr_op.eq.1) then
               anr=sqrt(ef/ei)
               tcbsigma(ie)=tcbsigma(ie)*anr
               acbsigma(ie)=acbsigma(ie)*anr
            end if
c
            ei=ei*ev  ! energy coverted from au to eV for output in iwrite
            ef=ef*ev
c
            write(iwrite,590)ie,ei,ef,tcbsigma(ie)*angst2
 590        format(i5,2(2x,f12.6),2x,g16.6)
c
 600     end do                 ! ie
      end if

c Print results in file

      itype=1 ! by default, this is an "R-matrix" rotatitonal xs
      if (cb_op.eq.1) itype=2  ! Total CB xs included
      if (cb_op.eq.1.and.lmax.gt.0) itype=3  ! TCB and partial CB xs
      if (cb_op.eq.2) itype=4 ! only Total CB xs calculated 
      if (cb_op.eq.2.and.lmax.gt.0) itype=5 ! only partial and Total CB 
                                            !xs calculated
c
      def=def*2.d0  !def is converted from Hartree to Ryd and passed to wrotxs
      call WROTXS(name_s,itype,j,jp,lmax,lmaxq,net,enscat,def,
     &     symtyp,sigma,tcbsigma,acbsigma,ixsn,ieunit,ixsout)
C
C---- Close files and return to main routine
      do js=1,nsym
         CLOSE(UNIT=LUCHANv(js),STATUS='KEEP')
         CLOSE(UNIT=LUTMTv(js),STATUS='KEEP')
      end do
      if (cb_op.eq.0.or.cb_op.eq.1) then
         deallocate (enscat,tmr,tmi,sigma,ivchl,lvchl,mvchl,evchl)
      else if (cb_op.eq.1.or.cb_op.eq.2) then
         deallocate(enscat,tcbsigma,acbsigma,sigma)
      endif
      WRITE(IWRITE,16)
      return
C
 16   FORMAT(//' *** Task has been successfully completed ***')
 92   FORMAT(/' *** DATA ON T-MATRIX FILE IS INCOMPATIBLE WITH CHANNEL D
     1ATA ***')
 104  FORMAT(/' Cross-sections will be computed for ',I5,' energies in t
     1he T-matrix file',/,' from',g12.5,' to',g12.5,A4)
 105  FORMAT(10A8)
 107  FORMAT(/' Number of target states       =',I4/' Number of scatteri
     1ng channels =',I4)
      END
c
      subroutine WROTXS(name,itype,j,jp,lmax,lmaxq,net,energy,def,
     &     symtyp,sigma,tcb,acb,ixsn,ieunit,iout)
      IMPLICIT double precision (A-H,O-Z)
      integer symtyp
      CHARACTER*80 NAME
      character*23 type(5)
      character*4 eunit(3)
      CHARACTER*8 XUNIT(3)
      parameter (ev=27.2114,angst2=0.2800285608592d0,
     1           pi=3.1415926535897932D+00)
      dimension energy(net),sigma(net),tcb(net),acb(net)
      data eunit/' eV ',' E_h',' Ryd'/,
     &     XUNIT/'BOHR**2','ANGS**2','PI*A0**2'/     
      data type/'R-matrix XS            ',
     &          'R-matrix & Total CB XS ',
     &          'R-matrix & TCB & ACB XS',
     &          'Total CB XS            ',
     &          'Partial and total CB XS'/

c ***********************************************************************
C      NAME     = TITLE FOR ANY OUTPUT
c     itype     = Flag to indicate the type of XS  
c                 1 - R-matrix XS
c                 2 - R-matrix and total Coulomb-Born XS
c                 3 - R-matrix, Total CB and added CB up to lmax partial
c                      waves
c                 4 - Total CB rotational cross sections
c                 5 - Partial and total CB rotational cross sections
c     j and jp  = Initial and final rotational numbers of the target
c     lmax      = Number of partial waves included in "added CB" (ACB) XS
c     lmaxq     = Number of partial waves included in the total CB XS 
c                 (quadrupole)
c     net       = Number of energies
c     energy    = Vector containing the energies
c     def       = Threshold energy
c     symtyp    = g/u symmetry of the molecule if applicable
c     sigma     = Vector containing the XS
c     ixsn      = Flag to select units for XS 
c                 1 - a0^2
c                 2 - Angstrom^2
c                 3 - pi*a0^2
c     ieunit    = Flag to select units for the energy
c                 1 - eV
c                 2 - Hartree
c                 3 - Ryd
c     iout      = Unit to write the data
c ***********************************************************************

c calculate conversion units factors. ---------------------------------
      if (ieunit.eq.1)  energy_units=ev*0.5d0   ! Ryd --> eV
      if (ieunit.eq.2)  energy_units=0.5d0      ! Ryd --> Hartree
      if (ieunit.eq.3)  energy_units=1.d0       ! Ryd --> Ryd

      if (ixsn.eq.1) xs_units=1.d0              ! a0^2 --> a0^2
      if (ixsn.eq.2) xs_units=angst2            ! a0^2 --> Angtroms^2
      if (ixsn.eq.3) xs_units=pi                ! a0^2 --> pi*a0^2
c ---------------------------------------------------------------------
c
      write(iout,100)name
 100  format(/'# ',a)
      write(iout,110)itype,type(itype),net
 110  format('#',/,'# key= ',i2,2x,a,/,'# Number of energies=',i5)
      write(iout,120)j,jp
 120  format('#',/,'# Rotational transition: j=',i2,'-->',i2)
      if (symtyp.eq.1) then
         write(iout,125)
 125  format('#',/,'# The target is homonuclear. Total symmetry is g.')
      else if (symtyp.eq.-1) then
         write(iout,126)
 126  format('#',/,'# The target is homonuclear. Total symmetry is u.')
      end if
      write(iout,130)eunit(ieunit),xunit(ixsn)
 130  format('#',/,'# Energy in ',a4,'; Cross sections in ',a8,/,'#')

      def=def*energy_units
      write(iout,135)def,eunit(ieunit)
 135  format('# Threshold energy',e17.8,' ',a4,/,'#')

      if (itype.eq.1) write(iout,140)
      if ((itype.eq.2.or.itype.eq.3.or.itype.eq.4.or.itype.eq.5)
     &     .and.abs(j-jp).eq.2) write(iout,145)lmaxq
      if (itype.eq.2) write(iout,150)
      if (itype.eq.3) write(iout,160)lmax
      if (itype.eq.4) write(iout,170)
      if (itype.eq.5) write(iout,180)lmax
 140  format('#',4x,'Energy',10x,'TM CS  ',/,'#')
 145  format('# Number of partial waves in TCB (lmaxq)=',i2,/,'#')
 150  format('#',4x,'Energy',10x,'TM CS  ',9x,'TCB CS  ',/,'#')
 160  format('# Number of partial waves in PCB (lmax) =',i2,/,'#',/,
     &'#',4x,'Energy',6x,'TM+TCB-PCB',8x,'TM CS  ',8x,'TCB CS  ',
     &8x,'PCB CS  ')
 170  format('#',4x,'Energy',10x,'TCB CS  ',/,'#')
 180  format('# Number of partial waves in ACB (lmax) =',i2,/,'#',/,
     &'#',4x,'Energy',10x,'TCB CS  ',8x,'ACB CS  ',/,'#')
c
      do ie=1,net
         energy(ie)=energy(ie)*energy_units
         sigma(ie) =sigma(ie)*xs_units        
      end do
      if (itype.eq.2.or.itype.eq.3.or.itype.eq.4.or.itype.eq.5) then
         do ie=1,net
            tcb(ie)=tcb(ie)*xs_units
            acb(ie)=acb(ie)*xs_units
         end do
      end if
      if (itype.eq.1) then
         do ie=1,net
            write(iout,190)energy(ie),sigma(ie)
         end do
      else if (itype.eq.2) then
         do ie=1,net
            write(iout,190)energy(ie),sigma(ie),tcb(ie)
         end do
      else if (itype.eq.3) then
         do ie=1,net
            write(iout,190)energy(ie),
     &           sigma(ie)+tcb(ie)-acb(ie),
     &           sigma(ie),tcb(ie),acb(ie)
         end do
      else if (itype.eq.4) then
         do ie=1,net
            write(iout,190)energy(ie),tcb(ie)
         end do
      else if (itype.eq.5) then
         do ie=1,net
            write(iout,190)energy(ie),tcb(ie),acb(ie)
         end do
      end if
c
 190  format(f12.8,5(2x,f14.6))
      return
      end
ccc
c Program to calculate rotational cross sections from  BF T matrices
c
      subroutine TROTSEC(nsym,maxern,ltmt,maxchi,maxchf,maxvi,maxvj,
     *                   ndopen,mgvn,enscat,tmr,tmi,num_pw,symtyp,dej,
     1     j,jp,mjj,be,net,anr_op,sigma,iprint,iout)

      implicit double precision (a-h,o-z)
      double precision kj,kjp
      integer anr_op,symtyp
      logical lamok

      parameter (pi=3.1415926535897932D+00,angst2=0.2800285608592d0,
     &           ev=27.2114)
      parameter (maxli=10,maxlf=30,maxjj=10)

      dimension tmr(ltmt,nsym),tmi(ltmt,nsym)
      dimension maxvi(maxern,nsym),maxvj(maxern,nsym),
     1          ndopen(maxern,nsym),sigma(*)
      dimension maxchi(nsym),maxchf(nsym)
      dimension mgvn(nsym),enscat(net)
      dimension tr(nsym,maxli,maxlf),ti(nsym,maxli,maxlf)
      dimension num_pw(nsym)
      dimension i_count(nsym)
      dimension tt(0:maxjj),ttr(0:maxjj),tti(0:maxjj),xs(0:maxjj)
      dimension binom(100,100)
c
c nsym... number of symmetries (of T matrices) to consider (to read)
c j...... initial rotational quantum number
c jp..... final rotational quantum number
c mjj.... Maximum value for J (total angular momentum quantum number)
c mgvn.... vector containing the lambda values of the T-matrices.


c be ... rotational constant. It's used to calculate the rotational thresholds
c        with the formula E_j=2*be*j(j+1)

      write(iout,190)
 190  format(' **** Subroutine TROTSEC. Calculation of rotational cross s
     &ections from T-matrices',/)

c initialize Clebsh-Gordon coeffs
        call pascal(binom,100)

c initialise accumulative pointers used in the T-matrix extraction
      
c      do js=1,nsym
c         i_count(js)=0
c      end do

c do loop on energy

      write(iout,351)
 351  format('  I  ','  energy (eV) ','  sigma (A^2)',
     &'   Col. strength')
c
      do js=1,nsym
        i_count(js) = 0
      end do
      do ienergy=1,net
         energy=enscat(ienergy) ! we assume that the energies are the
                                ! same in all T-matrix files. 
         energy=energy/2.d0     ! convert Ryd to au 

c extract T-matrices 

         do js=1,nsym
            kk = i_count(js)
            do lf=1,maxchf(js)
               do li=1,maxchi(js)
                  kk = kk+1
                  if (lf.le.maxvi(ienergy,js).and.
     &                li.le.(maxvj(ienergy,js)+ndopen(ienergy,js))) then
c
                     tr(js,li,lf)=tmr(kk,js)
                     ti(js,li,lf)=tmi(kk,js)
c                    
                  end if
               end do
            end do
            i_count(js) = kk
         end do
c
c the value of li and lf for each symmetry is not the real one. For example,
c in the symmetry Delta, there are no partial waves with l<2.
c The value \lambda for each symmetry is used to correct this storage-meaning
c discrepancy.
c
c check T-matrices if debugging
         if (iprint.gt.0) then
            write(iout,210)
 210        format(/,' First elements of the T-matrix',/)
            do js=1,nsym
               write(iout,220)mgvn(js),tr(js,1,1),tr(js,2,1),ti(js,1,1),
     &             ti(js,2,1)
            end do
 220        format(2x,i2,4(1x,g20.13))
c 
            write(iout,230)energy
 230        format(/,'::::::::::::::::::::::::: Energy= ',f12.5)
         end if

c calculate momenta to correct the ANR cross sections

         ej=be*dble(j*(j+1))
         ejp=be*dble(jp*(jp+1))
         dej=ejp-ej ! dej is the energy the molecule will take from the electron
         if (dej.gt.energy) goto 9999

         kj=dsqrt(2.d0*energy)
         kjp=dsqrt(2.d0*(energy-dej))

c do loop on J (total angular momentum), which is jj in this code.

      do jj=0,mjj

         if (iprint.gt.0) write(iout,240)jj
 240     format(/,'=======================================  J=',i2)

c do loop on li and lf

         tt(jj)=0.d0

         increment=1
         if (abs(symtyp).eq.1) increment=2
         do l=abs(JJ-j),jj+j,increment
            do lp=abs(jj-jp),jj+jp,increment

               if (iprint.gt.1) write(iout,250)l,lp
 250     format(/,' ------------------------------ l=',i2,' lp=',i2)

c do loop on lambda (the symmetry of the system)

               lpar=-1
               if (mod(l,2).eq.mod(lp,2)) lpar=mod(l,2) ! l and lp are both even or odd
               
               ttr(jj)=0.d0
               tti(jj)=0.d0
               do lambda=-lp,lp

                  if (symtyp.eq.1.and.lpar.ne.0) go to 20 ! when gerade, only l,lp even are needed
                  if (symtyp.eq.-1.and.lpar.ne.1) go to 20 ! when ungerade, only l,lp odd are needed

                  if (iprint.gt.1)write(iout,260)lambda
 260              format(/,' .............................. lambda=',i2)

                  lam=abs(lambda)

c if l is smaller that lam, then there won't be a T-matrix element
c for this case
                  if (l.lt.lam) then
                     if (iprint.gt.1) write(iout,270)l,lam
 270                 format(' l=',i2,' < lam=',i2,' ....skipping')
                     go to 20
                  end if

c check that there is a T-matrix for that lambda

                  lamok=.false.
                  do js=1,nsym
                     if (lam.eq.mgvn(js)) then 
                        lamok=.true.
                        itsym=js
                     end if
                  end do

                  if (lamok) then

                     a=dsqrt(dble(2*j+1))*(-1)**(j-l+lambda)
                     ap=dsqrt(dble(2*jp+1))*(-1)**(jp-lp+lambda)

                     cg=threej(j,l,JJ,0,lambda,-lambda,binom)
                     a=a*cg
            if (iprint.gt.2) write(iout,280)j,l,jj,lambda,-lambda,cg

                     cg=threej(jp,lp,JJ,0,lambda,-lambda,binom)
                     ap=ap*cg
            if (iprint.gt.2) write(iout,280)jp,lp,jj,lambda,-lambda,cg

 280       format('CG(',i2,',',i2,',',i2,';0,',i2,',',i2,')=',f12.5)

c translate the values of l and lp, so they correspond to the values
c used for storage in the matrices tr and ti

                     if (symtyp.eq.1) lcorr=0     
                     if (symtyp.eq.-1) lcorr=1

                     if (symtyp.eq.0) then
                        lstor=l-lam+1
                        lpstor=lp-lam+1
                     else
                        lstor=int((l-lam)/2+1)
                        lpstor=int((lp-lam)/2+1)
                     end if

          if (iprint.gt.1) write(iout,290)lambda,l,lp,itsym,lstor,lpstor
 290      format(/,2x,'T(',2(i2,','),i2,') <-- T(',2(i2,','),i2,')')

                     if (lstor.gt.num_pw(itsym)) then 
                        if (iprint.gt.1)
     &                   write(iout,300)l,itsym
 300  format(2x,'l=',i2,' for sym=',i2,2x,'***not available***')
                        go to 20
                     end if
                     if (lpstor.gt.num_pw(itsym)) then 
                        if (iprint.gt.1) write(iout,301)lp,itsym
 301  format(2x,'lp=',i2,' for sym=',i2,2x,'***not available***')
                        go to 20
                     end if

                     ttr(jj)=ttr(jj)+a*ap*tr(itsym,lstor,lpstor)
                     tti(jj)=tti(jj)+a*ap*ti(itsym,lstor,lpstor)

                     if (iprint.gt.3) then ! detailed debugging
                        write(iout,310)tr(itsym,lstor,lpstor),
     &                                 ti(itsym,lstor,lpstor)
 310                    format(2x,'tr()=',g20.13,' ti()=',g20.13)

                        write(iout,320)jj,ttr(jj)
                        write(iout,321)jj,tti(jj)
 320                    format(2x,'ttr(',i2,')=',g20.13)
 321                    format(2x,'tti(',i2,')=',g20.13)
                     end if

                  else
                     if (iprint.gt.1)
     &                write(iout,*)' There is no T-mat for lambda=',lam
                  end if

 20               continue
               end do

               tt(jj)=tt(jj)+(2*JJ+1)*(ttr(jj)*ttr(jj)+tti(jj)*tti(jj))

               if (iprint.gt.0) write(iout,330)jj,tt(jj)
 330           format('tt(',i2,')=',f14.7)
               
            end do
         end do

* this assumes that energy is in Hartree
         xs(jj)=pi/(2.d0*energy*dble(2*j+1)) * tt(jj)

      end do

c final sum
      sigt=0.d0
      do jj=0,mjj
         sigt=sigt+xs(jj)
      end do

      if (iprint.gt.0) write(iout,340)(xs(jj),jj=0,mjj)
 340  format(' xs(jj): ',6(1x,f12.5))

c calculate correction ANR and cross section

      anr=kjp/kj
      if (anr_op.eq.1) sigt=anr*sigt
      sigma(ienergy)=sigt
c
      if (iprint.gt.0) write(iout,*)'ANR correction =',anr
c
c calculate collision strengths (energy in Hartree)
c
      omega=sigt*2.d0*energy*dble(2*j+1)/pi
c
c output results

      write(iout,350)ienergy,energy*ev,sigt*angst2,omega
 350  format(i5,f12.5,f12.3,f12.3)

 9999 continue

c end do loop on energies

      end do

c end program

      return
      end
ccc
      subroutine DCB(ei,ef,j,jp,lmax,q1,ion,dxs,pdxs,iwrite,iprint)
c
c Program to calculate total Coulomb-Born cross section for electron
c molecular-ion scattering. It is based on the paper
c   Chu and Dalgarno, Phys. Rev. A, 10, (1974) 788
c It is calculated both total (eq. 21) and a truncated sum (eq. 17)
c
c To calculate the necessary integrals between Coulomb functions we follow 
c the paper by Regemorter (MON.NOT.R.ASTR.SOC., 121 (1960) 213-31)
c 
c Both papers take many equations from 
c Alder et al, Rev. Mod. Phys. 28, (1956) 432
c
c This implementation is used in Rabadan, Sarpal and Tennyson, J. Phys. B. 31 (1998)
c
      implicit double precision (a-h,o-z)
      double precision ki,kf,mpoles
      parameter (maxl=40)
      dimension mpoles(0:maxl+1),dpoles_t(0:maxl),dpoles_b(0:maxl)
      external fE1
      common /charge/iz
      common /binomium/binom(100,100)

      iz=ion

c get electron momentum 

      ki=sqrt(2.d0*ei)
      kf=sqrt(2.d0*ef)

* in Regemorter's paper these numbers are defined positive. Here we follow the
* definition by Alder et al, Rev. Mod. Phys. 28, (1956) 432.

      etai=-dble(ion)/ki
      etaf=-dble(ion)/kf
      xim=etaf-etai

c Calculates right hand of eq. 18 in Chu and Dalgarno
c  The number is 9/(64*pi*pi)

      eq18rh=0.0142482914497d0/(ki*kf)*fE1(etai,xim)
c
c Calculate monopole integrals
      call MONOPOLE(ki,kf,lmax+1,mpoles)
c Obtain, from monopoles, the dipole integrals
      call dipole(ki,kf,lmax,mpoles,dpoles_t,dpoles_b)
c Do the sum of these integrals with the algebraic (3j symbols) weight
      call sumatorio_d(lmax,dpoles_t,dpoles_b,sum,iwrite,iprint)

*** factor that does not depend on J (eq. 17)
* the number is 16*pi/3
c
      factor=16.755160819146*q1*q1*kf/ki

c calculate the number that depends on J (eq. 17)

      thrj=threej(j,jp,1,0,0,0,binom)
      factor_j=dble(2*jp+1)*thrj*thrj

c calculate the cross sections

      dxs=factor*factor_j*eq18rh  ! total CB XS
      pdxs=factor*factor_j*sum    ! Partial XS, contribution from 0 to lmax.

      return
      end

***
* Function fE1, eq 8 in Rabadan, Sarpal and Tennyson, JPB, 31 (1998)
* and eq 22 in Chu and Dalgarno, pra, 10, (1974) 788
*
      function fE1(etai,zeta)
      implicit double precision (a-h,o-z)
      double precision ichi
      double complex hypgeo,gammln,tempx,aa,bb,cc
      double complex cichi,gam1,gam2,gam3,f1,f2,f3,f4,f5,f6,fe0
      double complex ephi1,ephi2
      parameter (fac1=110.24453930773,pi=3.1415926535898d0)
      data zero,one,two/0.d0,1.d0,2.d0/
      external hypgeo,gammln
c
      etaf=zeta+etai
      chi=-4.d0*etai*etaf/(zeta*zeta)
      ichi=one/chi
      cichi=cmplx(ichi,zero) 
c
c hypergeometric functions
c
      aa = cmplx(zero,etai)
      bb = cmplx(zero,etai)
      cc = cmplx(one,-zeta)
      f1=hypgeo(aa,bb,cc,cichi)
      aa = cmplx(one,-etai)
      bb = cmplx(zero,-etai)
      cc = cmplx(one, zeta)
      f2=hypgeo(aa,bb,cc,cichi)
      aa = cmplx(one,-etaf)
      bb = cmplx(zero,-etaf)
      cc = cmplx(one,-zeta)
      f3=hypgeo(aa,bb,cc,cichi)
      aa = cmplx(zero, etaf)
      bb = cmplx(zero,etaf)
      cc = cmplx(one, zeta)
      f4=hypgeo(aa,bb,cc,cichi)
      f5=f3
      f6=f2
c
c arguments of gamma function products
c
      tempx = cmplx(zero,zeta)
      gam1=gammln(tempx)
      tempx = cmplx(zero,etai)
      gam2=gammln(tempx)
      tempx = cmplx(zero,etaf)
      gam3=gammln(tempx)
c
      arg1=aimag(gam1+gam2-gam3)
c
      tempx = cmplx(zero,-zeta)
      gam1=gammln(tempx)
c
      arg2=aimag(gam1+gam3-gam2)  ! Actually, I think that  arg2=-arg1
c
c angles
c
      phi1=two*arg1+zeta*log(dabs(chi))
      phi2=two*arg2-zeta*log(dabs(chi))
      ephi1=exp(cmplx(zero,phi1))
      ephi2=exp(cmplx(zero,phi2))

c factor

      factor=-fac1*etai*etaf/zeta/(exp(two*pi*zeta)-one)

c fe1 function

      fe0=one/etai*f1*(f2+ephi1*f3)+one/etaf*f4*(f5+ephi2*f6)
      fe1=factor*aimag(fe0)

      return
      end
***
* Subroutine qcb 
      subroutine qcb(ei,ef,j,jp,k,lmax0,lmax,q2,ion,a,b,
     &                qxs,pqxs,iwrite,iprint)
c  
c  Program to calculate total quadrupole induced Coulomb-Born cross section
c  for electron-molecular_ion scattering. It is based on the paper
c   Chu, Phys. Rev. A 12 (1975) 396
c
c  The same theory is in 
c   Alder et al, Rev. Mod. Phys. 28, (1956) 432
c
c  It is also cover in Rabadan, Sarpal and Tennyson, JPB 31 (1998)

      implicit double precision (a-h,o-z)
      double precision multipole,ki,kf,mpoles
      dimension mpoles(0:100),qpoles_t(0:100),qpoles_b(0:100),
     1     qpoles_d(0:100)
      common /charge/iz
      common /binomium/binom(100,100)
      external multipole,func,aa,f

      iz=ion

c get electron momentum

      ki=dsqrt(2.d0*ei)
      kf=dsqrt(2.d0*ef)

c calculate monopole integrals

      call MONOPOLE(ki,kf,lmax+1,mpoles)

c print them out for debugging...

      if (iprint.gt.0) then

         write(iwrite,100)
 100     format('# monopoles:',/,'#  l',8x,'M_ll')

         do l=0,lmax
            write(iwrite,110)l,mpoles(l)
         end do
 110  format(2x,i2,3(2x,g16.6))

      end if

c call function multipole to calculate the M20 and M02  quadrupoles

      qpoles_t(0)=multipole(ki,kf,0,2,2,a,b,0)
      if (iprint.gt.0) write(iwrite,125)qpoles_t(0)
 125  format('# Integrated quadrupole M02=',g16.6)

      qpoles_b(0)=multipole(ki,kf,2,0,2,a,b,0)
      if (iprint.gt.0) write(iwrite,126)qpoles_b(0)
 126  format('# Integrated quadrupole M20=',g16.6)

c call subroutines quadrupole to calculate quadrupoles
** off-diagonal elements
      call quadrupole_od(ki,kf,lmax,mpoles,qpoles_t,qpoles_b)
** diagonal elements
      call quadrupole_d(ki,kf,lmax,qpoles_t,qpoles_b,qpoles_d)

c print them out

      if (iprint.gt.0) then

         write(iwrite,130)
 130  format('# Quadrupoles:',/,'#  l',8x,'M_l,l+2',11x,'M_l+2,l',
     &       10x,'M_l,l')

         do l=0,lmax
            write(iwrite,110)l,qpoles_t(l),qpoles_b(l),qpoles_d(l)
         end do

      end if

c calculate sumatory of Coulomb integrals...

      call sumatorio_q(lmax0,lmax,qpoles_t,qpoles_b,qpoles_d,
     &     sum0,sum,iwrite,iprint)

c calculate the factor that does not depend on J or K (eq. 31)
c (the number is 16*pi/5)

      factor=10.053096491487*q2*q2*kf/ki

c calculate the number that depends on J and K (eq. 31)

      thrj=threej(j,jp,2,k,-k,0,binom)
      factor_jk=dble(2*jp+1)*thrj*thrj

c calculate the cross section

      qxs=factor*factor_jk*sum
      pqxs=factor*factor_jk*sum0

      return
      end
***
* function multipole
*
      function multipole(ki,kf,li,lf,gam,aq,bq,np)
*
* function to calculate numerically integrals between radial
* Coulomb functions of angular momenta li and lf, the coupling factor 
* being r^(-gam-1).
*   
* ki .... initial kinetic momentum of the electron
* kf .... final ""
* gam ... multipole interaction (1=dipole, 2=quadrupole, ...)
* aq,bq ... range in which the electronic coordinate will be integrated
* np .... number of points at which the function to integrate will be
*         evaluated in order to be printed out. 
*         If np=0, it won't be printed out.
*
      implicit double precision (a-h,o-z)
      integer gam,gamma
      double precision multipole,ki,kf
      parameter (lw=4000,liw=1000) ! these are required by D01AKF
      dimension w(lw),iw(liw)      ! and these
      common /commqp/xzi,xzf,gamma,kount             
      common /transition/xki,xkf,xli,xlf
      common /charge/iz
      external func
 
c initialize counter of the number of calls to func
      kount = 0                                                   

      xli=dble(li) ! in routine coulfg these index are real*8. This is
      xlf=dble(lf) ! the reason for this transformation here

      xki=ki ! These values come in the argument of the function, but 
      xkf=kf ! need to be pass to the function func by using a common.

c correction of the exponent for the interaction potential

      gamma=gam+1

c Obtain kinetic momentum and scaled charge

      xzi=-dble(iz)/ki
      xzf=-dble(iz)/kf

c This is to plot the function which is going to be integrated

      if (np.gt.0) then
         a1=aq
         if (aq.lt.1.d1) then
            a1=1.d1
            stepr=(dlog(a1)-dlog(aq))/dble(np/10)
            do ir=1,np/10
               r=dexp(dlog(aq)+stepr*dble(ir-1))
               f=func(r)
               write(75,1000)r,f
            end do
         end if

         stepr=(bq-aq)/dble(np)
         do ir=1,np
            r=aq+stepr*dble(ir-1)
            f=func(r)
            write(75,1000)r,f
         end do
      end if
 1000 format(2(2x,g12.6))

c integrate the function func using a NAG subroutine.

      EPSABS = 0.0D0
      EPSREL = 1.0D-05

      ifail=0
C    The following line is a warning that should be removed once D01AKF
C    is replaced.
      WRITE(6,*) 'NAG routine not replaced. This part of the
     1 program does not work in this version of the code'
!NV03     call d01akf(func,aq,bq,epsabs,epsrel,xm,abserr,w,lw,iw,liw,ifail)
      if (ifail.ne.0) print*,' d01akf ifail=',ifail

      multipole=xm

c divide the multipole by the kinetic factor ki*kf

      multipole=multipole/(ki*kf)

      return
      end

***************************************************************************
* Declare function func. It is the product of the radial Coulomb functions
*                        times the potential to be integrated.
*
      function func(r)
      implicit double precision (a-h,o-z)
      integer gamma
      parameter (maxl=4)
      dimension fci(maxl),fcf(maxl),gc(1),fcp(1),gcp(1)
      common /commqp/xzi,xzf,gamma,kount             
      common /transition/xki,xkf,xli,xlf

c counter
         kount=kount+1

c prepare variables
         ri=xki*r
         rf=xkf*r

c call the Coulomb routine for entrance channel

         ifail=0
         ifail2=0
         call coulfg(ri,xzi,xli,xli,fci,gc,fcp,gcp,3,0,ifail)
         if (ifail.ne.0) print*,' Warning coulfg (i): ifail=',ifail

c call the Coulomb routine for exit channel

         ifail=0
         ifail2=0
         call coulfg(rf,xzf,xlf,xlf,fcf,gc,fcp,gcp,3,0,ifail)
         if (ifail.ne.0) print*,' Warning coulfg (f): ifail=',ifail

c Obtain the value of the function

         func=fcf(int(xlf+1))*fci(int(xli+1))/(r**gamma)

         return
         end
***
* subroutine monopole
*
      subroutine MONOPOLE(ki,kf,lmax,mpoles)
*
* Calculate the necessary integrals between Coulomb functions analytically.
* we follow the paper by Regemorter (MON.NOT.R.ASTR.SOC., 121 (1960) 213-31)
*
      implicit double precision (a-h,o-z)
      double precision  mpoles,lo,ki,kf
      double complex gammln,hypgeo,tempx
      double complex gammai,gammaf,gammaif,f,q,a,b,c,aa,bb,cc,ct
      double complex lp1etai,lp1etaf,letai,letaf
      dimension mpoles(0:lmax)
      common /charge/iz
      parameter (halfpi=1.5707963267949d0)
c
      etai=-dble(iz)/ki
      etaf=-dble(iz)/kf
      setai=etai*etai
      setaf=etaf*etaf
      etaif=etai*etaf
c
      xim=etaf-etai
      xip=etaf+etai
      x=-4.d0*etaif/(xim*xim)
      t=1.d0/(1.d0-x)
      ct=cmplx(t,0.d0)
c
      aux1=(ki-kf)**(-2)
      aux2=exp(-halfpi*xim)/etaif

c calculation of the first two monopole radial matrices

      do l=0,1

         lo=dble(l+1)

c  complex gamma functions
         tempx = cmplx(lo,etai)    
         gammai=gammln(tempx)
         tempx = cmplx(lo,etaf)
         gammaf=gammln(tempx)
         gammaif=exp(gammai+gammaf)

c  hipergeometric function, eq 5'
         aa = cmplx(lo,-etai)
         bb = cmplx(lo,etaf)
         cc = cmplx(1.d0,xim)
         f=HYPGEO(aa,bb,cc,ct)
c
         tempx = cmplx(0.d0,-xim)
         a=gammln(tempx)
         tempx = cmplx(lo,-etaf)
         b=gammln(tempx)
         tempx = cmplx(lo,etai)
         c=gammln(tempx)

         q=2.d0*exp(a-b-c)*t**cmplx(lo,xim*0.5d0)*f
**         q=2.d0*a/(b*c)*t**cmplx(lo,xim/2.d0)*f  ! to use when a,b,c=gamma()

c  M_l (monopole terms)

         mpoles(l)=aux1*aux2*(-x)**l*abs(gammaif)*dble(q)
*      mpoles(l)=aux1*aux2*(-x)**l*abs(gammai)*abs(gammaf)*dble(q)

c divide the monopoles by ki*kf 

         mpoles(l)=mpoles(l)/(ki*kf)

      end do
      
c Calculates the rest of the  monopoles from the first two

      do l=1,lmax-1

         lo=dble(l+1)
         twol=2.d0*dble(l)

         lp1etai=cmplx(lo,etai)
         lp1etaf=cmplx(lo,etaf)
         letai=cmplx(dble(l),etai)
         letaf=cmplx(dble(l),etaf)

         y1=twol*abs(lp1etai)*abs(lp1etaf)
         y2=-(twol+1.d0)*((setai+setaf)/etaif*dble(l)*lo +
     &                                              2.d0*etaif)
         y3=2.d0*lo*abs(letai)*abs(letaf)

         mpoles(l+1)=-1.d0/y1*(y2*mpoles(l)+y3*mpoles(l-1))

      end do

      return
      end
***
* subroutine to calculate the dipole terms
*
      subroutine dipole(ki,kf,lmax,mpoles,dpoles_t,dpoles_b)
      implicit double precision (a-h,o-z)
      double precision ki,kf,lo,mpoles
      double complex lp1etai,lp1etaf
      dimension mpoles(0:lmax+1),dpoles_t(0:lmax),dpoles_b(0:lmax)
      common /charge/iz

      etai=-dble(iz)/ki
      etaf=-dble(iz)/kf

      do l=0,lmax

         lo=dble(l+1)

         lp1etai=cmplx(lo,etai)
         lp1etaf=cmplx(lo,etaf)

         dpoles_t(l)=kf/lo*abs(lp1etaf)*mpoles(l) - 
     &               ki/lo*abs(lp1etai)*mpoles(l+1)

         dpoles_b(l)=ki/lo*abs(lp1etai)*mpoles(l) - 
     &               kf/lo*abs(lp1etaf)*mpoles(l+1)

      end do

      return
      end

****
* subroutine to calculate by recurrence relations
* off-diagonal matrix integrals for 
* the quadrupole  interaction from monopole integrals
* 
*     eq. 36 in Chu75 , PRA 12 (1975) 396
*     eq. 80 in ABH56 , RMP 28 (1956) 432
*     eq. A3 in RST98 , JBP 31 (1998)

      subroutine quadrupole_od(ki,kf,lmax,mpoles,qpoles_t,qpoles_b)
      implicit double precision (a-h,o-z)
      double precision ki,kf,mpoles
      dimension mpoles(0:lmax+1),qpoles_t(0:lmax),qpoles_b(0:lmax)
      
      do l=1,lmax
         qpoles_t(l)=f(ki,kf,l)*qpoles_t(l-1)/f(ki,kf,l-1)+
     &               Aa(ki,kf,l-1,mpoles)
         qpoles_b(l)=f(kf,ki,l)*qpoles_b(l-1)/f(kf,ki,l-1)+
     &               Aa(kf,ki,l-1,mpoles)
      end do

      return
      end
***
*  function A    eq. 38 in Chu75 , PRA 12 (1975) 396
*                eq. 80 in ABH56 , RMP 28 (1956) 432
*                eq. a5 in RST98 , JPB 31 (1998)
* 
      function AA(ki,kf,l,mpoles)
      implicit double precision (a-h,o-z)
      double precision mpoles,ki,kf
      double complex lp1etai,lp1etaf,lp2etaf,lp3etaf
      dimension mpoles(0:*)
      common /charge/iz

      etai=-dble(iz)/ki
      etaf=-dble(iz)/kf

      setai=etai*etai
      setaf=etaf*etaf

c  complex numbers

      lp1etai=cmplx(dble(l+1),etai)
      lp1etaf=cmplx(dble(l+1),etaf)
      lp2etaf=cmplx(dble(l+2),etaf)
      lp3etaf=cmplx(dble(l+3),etaf)

c  calculate the function A
*  terms

      first=(ki*ki-kf*kf)/
     &       (dble(4*(l+1))*abs(lp2etaf)*abs(lp3etaf)*setai)

      second=2.d0*setai*setaf+setaf*dble((l+1)*(2*l+3))-
     &        setai*dble(l+1)

      third=2.d0*etai*etaf*abs(lp1etai)*abs(lp1etaf)

*  the function

      Aa=first*(second*mpoles(l+1)-third*mpoles(l))
   
      return
      end
****
* function f    eq. 37 in Chu75 , PRA 12 (1975) 396
*               eq. 80 in ABH56 , RMP 28 (1956) 432
*               eq. A4 in RST98 , JPB 31 (1998) 
* 
      function f(ki,kf,l)
      implicit double precision (a-h,o-z)
      double precision ki,kf
      double complex gamrat,gammln,lp1etai,lp3etaf
      common /charge/iz

      etai=-dble(iz)/ki
      etaf=-dble(iz)/kf

c complex numbers

      lp1etai=cmplx(dble(l+1),etai)
      lp3etaf=cmplx(dble(l+3),etaf)

c calculate the function f

      gamrat=exp(gammln(lp1etai)-gammln(lp3etaf))

      f=(etaf/etai)**l*abs(gamrat)

      return
      end
****
* subroutine to calculate analytically diagonal matrix integrals for 
* the quadrupole  interaction from off-diagonal quadrupole integrals
*
*   eq 32 in Chu75, PRA 12 (1975) 396
*   eq 72 in ABH56, RMP 28 (1956) 432
*   eq A6 in RST98, JPB 31 (1998)
*
      subroutine quadrupole_d(ki,kf,lmax,qpoles_t,qpoles_b,qpoles_d)
      implicit double precision (a-h,o-z)
      double precision ki,kf
      double complex lp1etai,lp1etaf,lp2etai,lp2etaf,letai,letaf
      dimension qpoles_t(0:lmax),qpoles_b(0:lmax),qpoles_d(0:lmax)
      common /charge/iz

      etai=-dble(iz)/ki
      etaf=-dble(iz)/kf

      etaif=etai*etaf
      setai=etai*etai
      setaf=etaf*etaf
   
      do l=1,lmax

         factor=dble(2*l+3)/dble(2*l+1)

c  complex numbers

         lp1etai=cmplx(dble(l+1),etai)
         lp1etaf=cmplx(dble(l+1),etaf)
         lp2etai=cmplx(dble(l+2),etai)
         lp2etaf=cmplx(dble(l+2),etaf)
         letai  =cmplx(dble(l),etai)
         letaf  =cmplx(dble(l),etaf)

c y factors

         y = dble(l*(l+1))*(setaf-setai)/3.d0
         y1=-setai*abs(lp1etaf)*abs(lp2etaf)
         y2= etaif*factor*abs(letai)*abs(lp1etaf)
         y3= setaf*abs(lp1etai)*abs(lp2etai)
         y4=-etaif*factor*abs(letaf)*abs(lp1etai)

c quadrupoles...

         qpoles_d(l)=(y1*qpoles_t(l)+y2*qpoles_t(l-1)+
     &                y3*qpoles_b(l)+y4*qpoles_b(l-1))/y

      end do

      return
      end
***
* subroutine sumatorio_d
*   calculates the sum in eq.17 of ChD74 PRA, 10 (1974) 788
*
      subroutine sumatorio_d(lmax,dpoles_t,dpoles_b,sum,iwrite,iprint)
      implicit double precision (a-h,o-z)
      dimension dpoles_t(0:lmax),dpoles_b(0:lmax)
      common /binomium/binom(100,100)

      sum=0.d0
      sum_old=0.d0
      sum_t=0.d0
      sum_b=0.d0

      if (iprint.gt.0) write(iwrite,100)
 100  format('# Dipole sumatory convergence...',/,
     & '#  l',7x,'Sum ',13x,'% error')
      
      do l=0,lmax

         f=dble(4*l*l+8*l+3)

         thrj=threej(l,l+1,1,0,0,0,binom)
         sum_t=sum_t+f*thrj*thrj*dpoles_t(l)*dpoles_t(l)

         thrj=threej(l+1,l,1,0,0,0,binom)
         sum_b=sum_b+f*thrj*thrj*dpoles_b(l)*dpoles_b(l)

         sum=sum_t+sum_b
         error=(sum-sum_old)/sum*100.d0
         if (iprint.gt.0) write(iwrite,101)l,sum,error
 101     format(2x,i2,3(2x,g16.6))

         sum_old=sum

      end do

      return
      end
***
* subroutine sumatorio
*     calculates the sum in eq.31: Chu, PRA 12 396 (1975)
*
      subroutine sumatorio_q(lmax0,lmax,qpoles_t,qpoles_b,qpoles_d,
     &                        sum0,sum,iwrite,iprint)
      implicit double precision (a-h,o-z)
      dimension qpoles_t(0:lmax),qpoles_b(0:lmax),qpoles_d(0:lmax)
      common /binomium/binom(100,100)

      sum0 =0.d0
      sum_old=0.d0
      sum_t=0.d0
      sum_b=0.d0
      sum_d=0.d0

      if (iprint.gt.0) write(iwrite,100)
 100  format('# Quadrupole sumatory convergence...',/,
     & '#  l',7x,'Sum ',13x,'% error')

      do l=0,lmax
         
         f_od=dble(4*l*l + 12*l + 5)
         f_d =dble(4*l*l +  4*l + 1)

         thrj=threej(l,l+2,2,0,0,0,binom)
         sum_t=sum_t+f_od*thrj*thrj*qpoles_t(l)*qpoles_t(l)

         thrj=threej(l+2,l,2,0,0,0,binom)
         sum_b=sum_b+f_od*thrj*thrj*qpoles_b(l)*qpoles_t(l)

         thrj=threej(l,l,2,0,0,0,binom)
         sum_d=sum_d+f_d*thrj*thrj*qpoles_d(l)*qpoles_d(l)

         sum=sum_t+sum_b+sum_d
         error=(sum-sum_old)/sum*100.d0
         if (iprint.gt.0) write(iwrite,101)l,sum,error
 101     format(2x,i2,3(2x,g16.6))
         
         sum_old=sum

         if (l.eq.lmax0) sum0=sum

      end do

      return
      end
c
****
* function gammln. Calculates the log(gamma(x))
* It is a conversion to complex argument version of 
* the Numerical Recepies by William. H. Press et al.
*
      FUNCTION GAMMLN(XX)
      double precision COF(6),STP,HALF,FPF
      double complex x,xx,gammln,tmp,ser,one
      DATA COF,STP/76.18009173D0,-86.50532033D0,24.01409822D0,
     *    -1.231739516D0,.120858003D-2,-.536382D-5,2.50662827465D0/
      DATA HALF,ONE,FPF/0.5D0,(1.0D0,0.d0),5.5D0/
      X=XX-ONE
      TMP=X+FPF
      TMP=(X+HALF)*log(TMP)-TMP
      SER=ONE
      DO 11 J=1,6
        X=X+ONE
        SER=SER+COF(J)/X
11    CONTINUE
      GAMMLN=TMP+log(STP*SER)
      RETURN
      END
c
      SUBROUTINE PASCAL(BINOM,NBIN)
C
C     USES PACAL'S TRIANGLE TO INITIALIZE AN ARRAY OF BINOMAL COEFFICIEN
C
      IMPLICIT double precision (A-H,O-Z)
      DIMENSION BINOM(NBIN,nbin)
      DATA Z1/1.0D0/
      BINOM(1,1) = Z1
      DO 1 I=2,NBIN
      BINOM(I,1) = Z1
      BINOM(I,I) = Z1
      IF (I .LE. 2) GOTO 1
      I1 = I - 1
      DO 2 J=2,I1
      BINOM(I,J) = BINOM(I1,J-1) + BINOM(I1,J)
    2 CONTINUE
    1 CONTINUE
c
      RETURN
      END
      FUNCTION THREEJ(J1,J2,J3,M1,M2,M3,BINOM)
      IMPLICIT double precision (A-H,O-Z)
      DIMENSION BINOM(100,100)
      DATA ZERO,ONE/0.0D0,1.0D0/
c
      THREEJ = ZERO
      IF (M1+M2+M3.ne.0) go to 999
      I1 = -J1+J2+J3+1
      IF (I1.le.0) go to 999
      I2 =  J1-J2+J3+1
      IF (I2.le.0) go to 999
      I3 =  J1+J2-J3+1
      IF (I3.le.0) go to 999
      K1 =  J1+M1+1
      IF (K1.le.0) go to 999
      K2 =  J2+M2+1
      IF (K2.le.0) go to 999
      K3 =  J3+M3+1
      IF (K3.le.0) go to 999
      L1 =  J1-M1+1
      IF (L1.le.0) go to 999
      L2 =  J2-M2+1
      IF (L2.le.0) go to 999
      L3 =  J3-M3+1
      IF (L3.le.0) go to 999
      N1 = -J1-M2+J3
      N2 =  M1-J2+J3
      N3 =  J1-J2+M3
      IMIN = MAX(-N1,-N2,0)+1
      IMAX = MIN(L1,K2,I3)
      IF (IMIN .GT. IMAX) RETURN
      SIGN = ONE
      DO 20 I=IMIN,IMAX
      SIGN = -SIGN
      THREEJ = THREEJ + SIGN*BINOM(I1,N1+I)*BINOM(I2,N2+I)*BINOM(I3,I)
   20 CONTINUE
      THREEJ = THREEJ * DSQRT(BINOM(J2+J2+1,I3)*BINOM(J1+J1+1,I2)
     1       / (BINOM(J1+J2+J3+2,I3)*DBLE(J3+J3+1)
     2       * BINOM(J1+J1+1,L1)*BINOM(J2+J2+1,L2)*BINOM(J3+J3+1,L3)))
      IF (MOD(N3+IMIN,2) .NE. 0) THREEJ = - THREEJ
  999 RETURN
      END
      FUNCTION HYPGEO(a,b,c,z)
      PARAMETER (NMAX=50,KMAXX=200)
      DOUBLE COMPLEX hypgeo,a,b,c,z
      DOUBLE PRECISION EPS,xp(KMAXX),yp(NMAX,KMAXX),dxsav,yy(4)
      PARAMETER (EPS=1.d-6)
CU    USES bsstep,hypdrv,hypser,odeint
      INTEGER kmax,nbad,nok
      EXTERNAL bsstep,hypdrv
      DOUBLE COMPLEX z0,dz,aa,bb,cc,y(2)
      COMMON /hypg/ aa,bb,cc,z0,dz
      COMMON /path/ kmax,kount,dxsav,xp,yp
c
      kmax=0
      if (dble(z)**2+aimag(z)**2.le.0.25d0) then
        call hypser(a,b,c,z,hypgeo,y(2))
        return
      else if (dble(z).lt.0.d0) then
        z0=cmplx(-0.5d0,0.d0)
      else if (dble(z).le.1.0d0) then
        z0=cmplx(0.5d0,0.d0)
      else
        z0=cmplx(0.d0,sign(0.5d0,aimag(z)))
      endif
      aa=a
      bb=b
      cc=c
      dz=z-z0
      call HYPSER(aa,bb,cc,z0,y(1),y(2))
      yy(1) = dble(y(1))
      yy(2) = aimag(y(1))
      yy(3) = dble(y(2))
      yy(3) = aimag(y(2))
      call ODEINT(yy,4,0.d0,1.d0,EPS,.1d0,.0001d0,nok,nbad,hypdrv,
     *            bsstep)
      hypgeo=cmplx(yy(1),yy(2))
      return
      END
C  (C) Copr. 1986-92 Numerical Recipes Software 'k'1k30m,t+W.
      SUBROUTINE BSSTEP(y,dydx,nv,x,htry,eps,yscal,hdid,hnext,derivs)
      INTEGER nv,NMAX,KMAXX,IMAX
      double precision  eps,hdid,hnext,htry,x,dydx(nv),y(nv),yscal(nv),
     * SAFE1,SAFE2,REDMAX,REDMIN,TINY,SCALMX
      PARAMETER (NMAX=50,KMAXX=8,IMAX=KMAXX+1,SAFE1=.25d0,SAFE2=.7d0,
     *REDMAX=1.d-5,REDMIN=.7d0,TINY=1.d-30,SCALMX=.1d0)
CU    USES derivs,mmid,pzextr
      INTEGER i,iq,k,kk,km,kmax,kopt,nseq(IMAX)
      double precision  eps1,epsold,errmax,fact,h,red,scale,work, 
     * wrkmin,xest,xnew,a(IMAX),alf(KMAXX,KMAXX),err(KMAXX),yerr(NMAX),
     * ysav(NMAX),yseq(NMAX)
      LOGICAL first,reduct
      SAVE a,alf,epsold,first,kmax,kopt,nseq,xnew
      EXTERNAL derivs
      DATA first/.true./,epsold/-1.d0/
      DATA nseq /2,4,6,8,10,12,14,16,18/
c
      if(eps.ne.epsold)then
        hnext=-1.d29
        xnew=-1.d29
        eps1=SAFE1*eps
        a(1)=nseq(1)+1
        do 11 k=1,KMAXX
          a(k+1)=a(k)+nseq(k+1)
11      continue
        do 13 iq=2,KMAXX
          do 12 k=1,iq-1
            alf(k,iq)=eps1**((a(k+1)-a(iq+1))/((a(iq+1)-a(1)+1.d0)*(2*k+
     *1)))
12        continue
13      continue
        epsold=eps
        do 14 kopt=2,KMAXX-1
          if(a(kopt+1).gt.a(kopt)*alf(kopt-1,kopt))goto 1
14      continue
1       kmax=kopt
      endif
      h=htry
      do 15 i=1,nv
        ysav(i)=y(i)
15    continue
      if(h.ne.hnext.or.x.ne.xnew)then
        first=.true.
        kopt=kmax
      endif
      reduct=.false.
2     do 17 k=1,kmax
        xnew=x+h
        if(xnew.eq.x) stop 'step size underflow in bsstep'
        call MMID(ysav,dydx,nv,x,h,nseq(k),yseq,derivs)
        xest=(h/nseq(k))**2
        call PZEXTR(k,xest,yseq,y,yerr,nv)
        if(k.ne.1)then
          errmax=TINY
          do 16 i=1,nv
            errmax=max(errmax,dabs(yerr(i)/yscal(i)))
16        continue
          errmax=errmax/eps
          km=k-1
          err(km)=(errmax/SAFE1)**(1.d0/(2*km+1))
        endif
        if(k.ne.1.and.(k.ge.kopt-1.or.first))then
          if(errmax.lt.1.d0)goto 4
          if(k.eq.kmax.or.k.eq.kopt+1)then
            red=SAFE2/err(km)
            goto 3
          else if(k.eq.kopt)then
            if(alf(kopt-1,kopt).lt.err(km))then
              red=1.d0/err(km)
              goto 3
            endif
          else if(kopt.eq.kmax)then
            if(alf(km,kmax-1).lt.err(km))then
              red=alf(km,kmax-1)*SAFE2/err(km)
              goto 3
            endif
          else if(alf(km,kopt).lt.err(km))then
            red=alf(km,kopt-1)/err(km)
            goto 3
          endif
        endif
17    continue
3     red=min(red,REDMIN)
      red=max(red,REDMAX)
      h=h*red
      reduct=.true.
      goto 2
4     x=xnew
      hdid=h
      first=.false.
      wrkmin=1.d35
      do 18 kk=1,km
        fact=max(err(kk),SCALMX)
        work=fact*a(kk+1)
        if(work.lt.wrkmin)then
          scale=fact
          wrkmin=work
          kopt=kk+1
        endif
18    continue
      hnext=h/scale
      if(kopt.ge.k.and.kopt.ne.kmax.and..not.reduct)then
        fact=max(scale/alf(kopt-1,kopt),SCALMX)
        if(a(kopt+1)*fact.le.wrkmin)then
          hnext=h/fact
          kopt=kopt+1
        endif
      endif
      return
      END
C  (C) Copr. 1986-92 Numerical Recipes Software 'k'1k30m,t+W.
      SUBROUTINE HYPDRV(s,y,dyds)
      DOUBLE PRECISION s
      DOUBLE COMPLEX y(2),dyds(2),aa,bb,cc,z0,dz,z
      COMMON /hypg/ aa,bb,cc,z0,dz
c
      z=z0+s*dz
      dyds(1)=y(2)*dz
      dyds(2)=(aa*bb*y(1)-(cc-(aa+bb+1.d0)*z)*y(2))*dz/(z*(1.d0-z))
      return
      END
C  (C) Copr. 1986-92 Numerical Recipes Software 'k'1k30m,t+W.
      SUBROUTINE hypser(a,b,c,z,series,deriv)
      INTEGER n
      DOUBLE COMPLEX a,b,c,z,series,deriv,aa,bb,cc,fac,temp
c
      deriv=cmplx(0.d0,0.d0)
      fac=cmplx(1.d0,0.d0)
      temp=fac
      aa=a
      bb=b
      cc=c
      do 11 n=1,1000
        fac=fac*aa*bb/cc
        deriv=deriv+fac
        fac=fac*z/n
        series=temp+fac
        if (series.eq.temp) return
        temp=series
        aa=aa+1.d0
        bb=bb+1.d0
        cc=cc+1.d0
11    continue
      stop 'convergence failure in hypser'
      END
C  (C) Copr. 1986-92 Numerical Recipes Software 'k'1k30m,t+W.
      SUBROUTINE ODEINT(ystart,nvar,x1,x2,eps,h1,hmin,nok,nbad,derivs,
     * rkqs)
      INTEGER nbad,nok,nvar,KMAXX,MAXSTP,NMAX
      double precision eps,h1,hmin,x1,x2,ystart(nvar),TINY
      EXTERNAL derivs,rkqs
      PARAMETER (MAXSTP=10000,NMAX=50,KMAXX=200,TINY=1.d-30)
      INTEGER i,kmax,kount,nstp
      double precision dxsav,h,hdid,hnext,x,xsav,dydx(NMAX),xp(KMAXX),
     * y(NMAX),yp(NMAX,KMAXX),yscal(NMAX)
      COMMON /path/ kmax,kount,dxsav,xp,yp
c
      x=x1
      h=sign(h1,x2-x1)
      nok=0
      nbad=0
      kount=0
      do 11 i=1,nvar
        y(i)=ystart(i)
11    continue
      if (kmax.gt.0) xsav=x-2.d0*dxsav
      do 16 nstp=1,MAXSTP
        call derivs(x,y,dydx)
        do 12 i=1,nvar
          yscal(i)=dabs(y(i))+dabs(h*dydx(i))+TINY
12      continue
        if(kmax.gt.0)then
          if(abs(x-xsav).gt.abs(dxsav)) then
            if(kount.lt.kmax-1)then
              kount=kount+1
              xp(kount)=x
              do 13 i=1,nvar
                yp(i,kount)=y(i)
13            continue
              xsav=x
            endif
          endif
        endif
        if((x+h-x2)*(x+h-x1).gt.0.d0) h=x2-x
        call RKQS(y,dydx,nvar,x,h,eps,yscal,hdid,hnext,derivs)
        if(hdid.eq.h)then
          nok=nok+1
        else
          nbad=nbad+1
        endif
        if((x-x2)*(x2-x1).ge.0.d0)then
          do 14 i=1,nvar
            ystart(i)=y(i)
14        continue
          if(kmax.ne.0)then
            kount=kount+1
            xp(kount)=x
            do 15 i=1,nvar
              yp(i,kount)=y(i)
15          continue
          endif
          return
        endif
        if(abs(hnext).lt.hmin) stop
     *  'stepsize smaller than minimum in odeint'
        h=hnext
16    continue
      stop 'too many steps in odeint'
      return
      END
C  (C) Copr. 1986-92 Numerical Recipes Software 'k'1k30m,t+W.
      SUBROUTINE MMID(y,dydx,nvar,xs,htot,nstep,yout,derivs)
      INTEGER nstep,nvar,NMAX
      DOUBLE PRECISION htot,xs,dydx(nvar),y(nvar),yout(nvar)
      EXTERNAL derivs
      PARAMETER (NMAX=50)
      INTEGER i,n
      DOUBLE PRECISION h,h2,swap,x,ym(NMAX),yn(NMAX)
c
      h=htot/nstep
      do 11 i=1,nvar
        ym(i)=y(i)
        yn(i)=y(i)+h*dydx(i)
11    continue
      x=xs+h
      call DERIVS(x,yn,yout)
      h2=2.d0*h
      do 13 n=2,nstep
        do 12 i=1,nvar
          swap=ym(i)+h2*yout(i)
          ym(i)=yn(i)
          yn(i)=swap
12      continue
        x=x+h
        call DERIVS(x,yn,yout)
13    continue
      do 14 i=1,nvar
        yout(i)=0.5d0*(ym(i)+yn(i)+h*yout(i))
14    continue
      return
      END
C  (C) Copr. 1986-92 Numerical Recipes Software 'k'1k30m,t+W.
      SUBROUTINE PZEXTR(iest,xest,yest,yz,dy,nv)
      INTEGER iest,nv,IMAX,NMAX
      DOUBLE PRECISION xest,dy(nv),yest(nv),yz(nv)
      PARAMETER (IMAX=13,NMAX=50)
      INTEGER j,k1
      DOUBLE PRECISION delta,f1,f2,q,d(NMAX),qcol(NMAX,IMAX),x(IMAX)
      SAVE qcol,x
c
      x(iest)=xest
      do 11 j=1,nv
        dy(j)=yest(j)
        yz(j)=yest(j)
11    continue
      if(iest.eq.1) then
        do 12 j=1,nv
          qcol(j,1)=yest(j)
12      continue
      else
        do 13 j=1,nv
          d(j)=yest(j)
13      continue
        do 15 k1=1,iest-1
          delta=1.d0/(x(iest-k1)-xest)
          f1=xest*delta
          f2=x(iest-k1)*delta
          do 14 j=1,nv
            q=qcol(j,k1)
            qcol(j,k1)=dy(j)
            delta=d(j)-q
            dy(j)=f1*delta
            d(j)=f2*delta
            yz(j)=yz(j)+dy(j)
14        continue
15      continue
        do 16 j=1,nv
          qcol(j,iest)=dy(j)
16      continue
      endif
      return
      END
C  (C) Copr. 1986-92 Numerical Recipes Software 'k'1k30m,t+W.
      SUBROUTINE GETDIM(nsym,TFORM,NTSET,LUTMT,maxtg,maxch,maxmi,maxmf,
     1 netot,IPRNT,IFAIL)
C
C     GETDIM reads T-matrix headers to determine maximum dimensions
C
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      PARAMETER (MAXESC=10,maxmax=1000)
      DIMENSION NTSET(nsym),NEREP(MAXESC+1),EINC(2*MAXESC),ICHL(maxmax),
     2 LCHL(maxmax),MCHL(maxmax),ECHL(maxmax),lutmt(nsym)
      INTEGER GUTOT,STOT
      CHARACTER*11 TFORM
      CHARACTER*80 TITLE
      DATA IWRITE/6/
C
 501  FORMAT(/' INVALID VALUE OF NTSET ',I4,' FOR REQUEST NUMBER',I4)
 502  format(/' Symmetry ',i2,' has different number of energies (',
     * i4,') from first symmetry (',i4,')')
 503  FORMAT(/' Maximum dimensions required: NCHAN=',I3,3x,'MAXCHI=',i3,
     13x,'MAXCHF=',i3,3x,'MAXTGT=',i3)
C
      maxtg = 0
      maxch = 0
      maxmi = 0
      maxmf = 0
      netot = 0
      do 1 isym=1,nsym
      IF(NTSET(isym).GT.0) THEN
        CALL READTH(LUTMT(isym),TITLE,NTSET(isym),NCHAN,NVIB,NDIS,
     1  NTARG,MAXCHI,MAXCHF,MGVN,STOT,GUTOT,NE,MAXNE,NEREP,EINC,ICHL,
     2  LCHL,MCHL,ECHL,TFORM,IWRITE,IPRNT-1,IFAIL)
        IF(IFAIL.NE.0) RETURN
        maxtg = max(maxtg,ntarg,nvib)
        maxch = max(maxch,nchan)
        maxmi = max(maxmi,maxchi)
        maxmf = max(maxmf,maxchf)
        if(isym.eq.1) then
          netot = ne
        else
          if(ne.ne.netot) then
            write(iwrite,502) isym,ne,netot
            stop
          endif
        endif
      ELSE
        WRITE(IWRITE,501) NTSET(isym),isym
        IFAIL = 1
        RETURN
      ENDIF
 1    continue
      write(iwrite,503) maxch,maxmi,maxmf,maxtg
c
      RETURN
      END
