! Copyright 2019
!
! For a comprehensive list of the developers that contributed to these codes
! see the UK-AMOR website.
!
! This file is part of UKRmol-out (UKRmol+ suite).
!
!     UKRmol-out is free software: you can redistribute it and/or modify
!     it under the terms of the GNU General Public License as published by
!     the Free Software Foundation, either version 3 of the License, or
!     (at your option) any later version.
!
!     UKRmol-out is distributed in the hope that it will be useful,
!     but WITHOUT ANY WARRANTY; without even the implied warranty of
!     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!     GNU General Public License for more details.
!
!     You should have received a copy of the GNU General Public License
!     along with  UKRmol-out (in source/COPYING). Alternatively, you can also visit
!     <https://www.gnu.org/licenses/>.
!
      subroutine ROT_LIN_S(IFAIL)
C***********************************************************************
C
C     ROT_LIN calculates rotational cross sections of linear neutral
c     molecules by electron impact. It uses T-matrices and the Born
c     approximation.  It is intended to be a self contained module which
c     can be run independently from the main scattering calculation.  It
c     has been adapted from rotions_lin and, therefore, some parameter
c     names are not consistent with a neutral target (eg CB for
c     Coulomb-Born) 
c     AF 04/04/2007
C
C***********************************************************************
C
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      integer anr_op,cb_op,grid
c
      PARAMETER (maxene=10000,MAXERN=10)
      parameter (maxnsym=12)
      parameter (ev=2.d0/0.0735d0,angst2=.280028560859d0,
     * pi=3.1415926535898d0)
c
      DIMENSION EINC(2,MAXERN),NESCAT(MAXERN),IPRNT(6)
      dimension lutmtv(maxnsym),luchanv(maxnsym)
      dimension nchan(maxnsym),maxchi(maxnsym),maxchf(maxnsym),
     1          mgvn(maxnsym)
      integer, allocatable :: ivchl(:),lvchl(:),mvchl(:)
      dimension maxvi(maxene,maxnsym),maxvj(maxene,maxnsym),
     1          ndopen(maxene,maxnsym),ntset(maxnsym)
      dimension num_pw(maxnsym)
      double precision, allocatable :: tmr(:,:),tmi(:,:),enscat(:),
     * evchl(:),sigma(:),acbsigma(:),tcbsigma(:)
c
      common /binomium/binom(100,100)  
C! this matrix is used in calculating 3j-symbols

      CHARACTER*80 NAME,name_s
      CHARACTER*4 CEUNIT(3)
      CHARACTER*9 FORM
      CHARACTER*1 ITFORM
      CHARACTER*11 TFORM,MODDAT
      INTEGER STOT,SYMTYP,n,np
      DOUBLE PRECISION trgspin,j,jp		
     
C
C***********************************************************************
C
C     Basic data is input via namelist /XSECIN/    
c      anr_op   = Switch to activate the ANR correction in the cross
c                 sections
c                 0= no correction, 1= correction applied (default)
c      be       = rotational constant (in cm-1)
c      cb_op    = Switch to activate the use of Coulomb-Born approx
c                 0= do NOT perform CB calculations (only T-matrix 
c                    cross sections)
c                 1= perform T-matrix plus CB calculations
c                 2= Only perform CB calculations (not T-matrices are
c                    required)
C      EMIN     = MINIMUM REQUIRED SCATTERING ENERGY (IN UNITS AS
C                 SPECIFIED BY IEUNIT)
C      EMAX     = MAXIMUM REQUIRED SCATTERING ENERGY
c      grid     = Kind of grid for energies when cb_op=2: 1=linear, 
c                 2=logarithmic
C      ITFORM   = 'F' if T-matrix dataset is formatted, else 'U'
C      IEUNIT   = UNITS of the output SCATTERING ENERGIES
C                 input ENERGIES are necessarily in Rydbergs 
C                 1= eV, 2= Hartree, 3= Ryd
c      IPRNT    = IPRNT(1)=1: print out all the input data. 
c                 IPRNT(4)=1,2,3,4: print out output data 
c                               (1 - short output, 4 -detailed output)     
C      IWRITE   = UNIT FOR console output
C      IXSN     = Units to be used for cross-section output
C                 2=ANGSTROM**2, 3=PI*BOHR**2
C      j	= j quantum number, only required for spin-rotation xsecs
C      jp	= j' quantum number, only required for spin-rotation xsecs
c      ixsout   = Logical unit for cross sections output 
c      luchanv  = vector containing the logical units for channel input 
c                 files
c      lutmtv   = vector containing the logical units for t-matrix input
c                 files
C      MAXI     = LABEL OF HIGHEST INITIAL STATE FOR WHICH CROSS-
C                 SECTIONS ARE REQUIRED
C      MAXF     = LABEL OF HIGHEST FINAL STATE FOR WHICH CROSS-SECTIONS
C                 ARE REQUIRED
C      N        = INITIAL ROTATIONAL QUANTUM NUMBER
C      NP	= FINAL ROTATIOANL QUANTUM NUMBER
C      NAME     = TITLE FOR ANY OUTPUT
C      NTSET    = SET NUMBER OF T-MATRIX INPUT
c      nsym     = Number of molecular symmetries to consider
C      numener  = Number of energies to be read (the T-matrix files may 
c                 have T-matrix for a number of energies and this allows 
c                 not to read all of them but just "numener".) 
c                 If numener<1 (default), all T-matrices will be read.
c      q1       = Permanent dipole of the target (in Debye)
c      lmax     = Maximum value of l (partial waves) to add in the 
c                 computation of the partial CB cross section.
c      lmaxq    = Number of partial waves to use in the computation of
c                 the total quadrupole CB cross section
c      q2       = Permanent quadrupole of the target (in atomic units)
c      trgspin = The spin value of the target molecule ONLY, only define
C		  this variable if you want spin-rotation xsecs

      NAMELIST/RXSECIN/IXSN,NVXPRT,IEUNIT,IWRITE,MAXI,MAXF,NAME,
     1LUCHANv,EMIN,EMAX,LUTMTv,NTSET,ITFORM,IPRNT,R,nsym,ixsout,
     2numener,n,np,mjj,be,anr_op,cb_op,q1,lmax,lmaxq,q2,ion,grid,trgspin
     &,j,jp 
C
C***********************************************************************
c  Input variable Defaults

      data nsym/1/,numener/0/,n,np/0,1/,mjj/6/
      DATA IXSN/1/,NVXPRT/0/,IWRITE/6/,VBIG/1.D+8/,R/0.D0/
     3,LUCHANv,LUTMTv/maxnsym*10,maxnsym*12/,NTSET/maxnsym*1/
     4,CEUNIT/' eV ',' E_h',' RYD'/
     4,IEUNIT/1/,IPRNT/6*0/,MAXI,MAXF/1,0/
     5,anr_op/1/,ixsout/8/,cb_op/0/,lmax/5/,q1/0.d0/,lmaxq/20/,k/0/
     6,q2/0.d0/,ion/0/,grid/1/,trgspin/-1.0/,j,jp/-1.0,-1.0/

      DATA ZERO/0.D0/,RYD/0.073500D0/
      DATA TFORM,FORM/2*'FORMATTED'/,ITFORM/'U'/
      DATA MODDAT/'17-Feb-2012'/

      IFAIL = 0
      NEXT = 1
      EMIN = ZERO
      EMAX = VBIG
C
C---- Read basic data via namelist /RXSECIN/
      READ(5,RXSECIN)
      IF(ITFORM.EQ.'U') TFORM='UN'//FORM
C
C---- Date-stamp run and print title
*      CALL DATEST(DAYTIM)
*      NAME(61:) = DAYTIM
      name_s = name
      WRITE(IWRITE,100)MODDAT,NAME
 100  FORMAT(' Program ROTLIN_S  (last modified ',A,' )'//A/)

c ---------------------------------------------------------------------
c be ... rotational constant. It's used to calculate the rotational thresholds
c        with the formula E_j=2*be*j(j+1). It is read in 1/cm and here is
c        converted to au.

      be=be*4.5563352672d-6

c q1 is the permanent dipole moment of the target. Here is converted
c from Debye to a.u.

      q1=q1/2.54158d0

c Force override cb_op as it must =1 when doing spin-coupled transitions
c needed to calc 0-1 rotational Born correction
      IF (trgspin.ne.-1.0) THEN
         cb_op = 1
      ENDIF
c ----------------------------------------------------------------------
      write(iwrite,495)ion
 495  format(/,' The charge of the target is ',i2)

      if (anr_op.eq.1) write(iwrite,500)
 500  format(/,' The cross sections will include the ANR correction.')

      if (trgspin.ne.-1) GOTO 678
       
      if (cb_op.eq.1.or.cb_op.eq.2) write(iwrite,505)
 505  format(/,' Born cross sections will be obtained.') 
      if (cb_op.eq.1.or.cb_op.eq.2) then
         if (abs(np-n).eq.1.and.q1.eq.0.d0) stop 'CB_op=1 or 2 and 
     & |np-n| =1 requires a non-zero value for q1.'
         if (abs(np-n).eq.2.and.q2.eq.0.d0) stop 'CB_op=1 or 2 and 
     & |np-n| =2 requires a non-zero value for q2.'
      end if

c      
 678  if (cb_op.eq.2) go to 5811
c
C----- Find required T-matrix set and read dimension information
      call GETDIM(nsym,TFORM,NTSET,LUTMTv,maxtgt,maxch,maxmi,maxmf,
     1            netot,IPRNT(2),IFAIL)
c
      allocate (ivchl(maxch),lvchl(maxch),mvchl(maxch),evchl(maxch))
      LTMT = MAXmi*MAXmf*netot 
      allocate (enscat(netot),tmr(ltmt,nsym),tmi(ltmt,nsym),
     * sigma(netot))

      do js=1,nsym   
c! js is the index referring to total molecular symmetries

         write(iwrite,510)js
 510     format(/' ----------------------------------------------------
     &T-matrix ',i2)

         lutmt=lutmtv(js)
         luchan=luchanv(js)

         CALL READTH(LUTMT,NAME,NTSET(js),NCHAN(js),NVIB,NDIS,NTARG,
     *               MAXCHI(js),MAXCHF(js),MGVN(js),STOT,symtyp,NE,
     *               NERANG,NESCAT,EINC,IVCHL,
     *               LVCHL,MVCHL,EVCHL,TFORM,IWRITE,IPRNT(1),IFAIL)
         IF(IFAIL.NE.0) stop 'FAIL reading T-matrix header'
         NETOT = NE

         if (numener.gt.maxene.or.netot.gt.maxene) then
            write(iwrite,512) 
            numener=maxene
         end if
 512     format(/,' Parameter MAXENE is smaller than the number of energ
     &ies to be read. NUMENER reset to MAXENE.')

         if (numener.gt.0) then
            write(iwrite,515)ne,numener
            netot=numener
         end if
 515     format(/,' The number of T-matrices in the file is ',i5,' but
     & only the first ',i5,' will be read')

c The T-matrices files may content information for transitions from several
c initial states to several final states. For the moment, this program will
c only cope with rotational transitions within the same electronic 
c (and vibrational, if the T-matrix is vibrationally resolved) state.
c To do this, it is necessary to know the number of partial waves in the
c T-matrix file, and to get this number we use the next lines:  

         if (symtyp.eq.0) then
            num_pw(js)=lmax-lvchl(1)+1
         else if (symtyp.eq.1) then
            if (mod(lmax,2).eq.0) then
               num_pw(js)=(lmax-lvchl(1)+1)/2+1
            else 
               num_pw(js)=(lmax-lvchl(1)+1)/2
            endif
         else if (symtyp.eq.-1) then
            if (mod(lmax,2).eq.0) then
               num_pw(js)=(lmax-lvchl(1)+1)/2
            else 
               num_pw(js)=(lmax-lvchl(1)+1)/2+1
            endif 
         endif
         
         write(iwrite,520)num_pw(js),lvchl(1),lmax
 520     format(/' The number of partial waves is ',i2,
     &           ' (l=',i2,' to ',i2,')')

c save values about energy range to compare with other files to read

         if (js.gt.1) then
            ii=lutmtv(js)
            ih=lutmtv(js-1)
            if (nvib_old.ne.nvib) then; write(iwrite,525)ii,ih; endif
 525        format(' NVIB not the same in ',i2,' and ',i2)
            if (ndis_old.ne.ndis) then; write(iwrite,526)ii,ih; endif
 526        format(' NDIS not the same in ',i2,' and ',i2)
            if (ntarg_old.ne.ntarg)then; write(iwrite,527)ii,ih; endif
 527        format(' NTARG not the same in ',i2,' and ',i2)
            if (stot_old.ne.stot) then; write(iwrite,528)ii,ih; endif
 528        format(' STOT not the same in ',i2,' and ',i2)
            if (symtyp_old.ne.symtyp)then;write(iwrite,529)ii,ih; endif
 529        format(' GUTOT not the same in ',i2,' and ',i2)
            if (ne_old.ne.ne) then; write(iwrite,530)ii,ih; endif
 530        format(' NE not the same in ',i2,' and ',i2)
            if (nerang_old.ne.nerang) then;write(iwrite,531)ii,ih;endif
 531        format(' NERANG not the same in ',i2,' and ',i2)
*            if (nescat_old.ne.nescat) then;write(iwrite,532)ii,ih;endif
* 532       format(' NESCAT not the same in ',i2,' and ',i2)
*            if (einc_old.ne.einc) then; write(iwrite,533)ii,ih; endif
* 533       format(' EINC not the same in ',i2,' and ',i2)
         end if

         nvib_old=nvib
         ndis_old=ndis
         ntarg_old=ntarg
         stot_old=stot
         symtyp_old=symtyp
         ne_old=ne
         nerang_old=nerang         
*         nescat_old=nescat
*         einc_old=einc

C
C---- Recompute energy parameters NERANG,NESCAT and EINC for energy
C     range [EMIN,EMAX] (this just need to be done once, but it doesn't 
c     take long to do it for every file read)
         emin0=emin
         emax0=emax
         IF(IEUNIT.EQ.1) THEN
            EMIN0 = RYD*EMIN
            EMAX0 = RYD*EMAX
         ENDIF
         if (ieunit.eq.2) then
            emin0 = 2.d0*emin
            emax0 = 2.d0*emax
         end if

c  get information about the energy range to explore.

         CALL NEWE(EMIN0,EMAX0,NE,NERANG,NESCAT,EINC)

         EMINP = EMIN0
         EMAXP = EMAX0
         IF(IEUNIT.EQ.1) THEN
            EMINP = EMIN0/RYD
            EMAXP = EMAX0/RYD
         End if
         if (ieunit.eq.2) then            
            EMINP = 0.5d0*EMIN0
            EMAXP = 0.5d0*EMAX0
         ENDIF
         WRITE(IWRITE,104) NE,EMINP,EMAXP,CEUNIT(IEUNIT)

C
C----- Assign storage for energy independant data
 
         write(iwrite,540)js
 540     format(/,' ---> T-matrix ',i2)
c
         IF(NVIB.EQ.0) NVIB=NTARG
         IF(MAXF.EQ.0) MAXF=NVIB
         NVIBD  = max(NVIB+NDIS,ntarg)
c
C----- ALLOCATE SPACE FOR T-MATRICES AND CROSS-SECTIONS
c T-matrices for more than one energy will be read (in general), so 
c these pointers have to allow for that.
cC
C---- read "netot" T-matrices
         net = netot
         CALL READT(net,MAXCHI(js),MAXCHF(js),NDIS,MAXVI(1,js),
     *              MAXVJ(1,js),
     *              NDOPEN(1,js),TMR(1,js),TMI(1,js),enscat)

         write(iwrite,550)js,net
 550   format(/,' T-matrices in symmetry ',i2,' have been read for ',i5,
     &         ' energies',/)

c
c end do loop on symmetries (js)

      end do
C
C---- Calculate rotational cross sections (spin inclusive or exclusive)
c
	IF (trgspin.eq.-1.0) THEN

      call TROTSEC(nsym,maxene,ltmt,maxchi,maxchf,maxvi,maxvj,ndopen,
     1     mgvn,enscat,tmr,tmi,num_pw,symtyp,def,
     1     n,np,mjj,be,net,anr_op,sigma,iprnt(4),iwrite)

	ELSE
C Need to perform checks here to make sure j and jp are acceptable
C (obviously doesn't matter what they are if targspin not defined
C as won't be used anywhere and TROTSEC above will run fine) 

	IF (j.eq.-1.0 .or. jp.eq.-1.0) then
	  WRITE(*,*)'ERROR:j and/or jp not defined. Ending program'
	  GOTO 874
	ELSE IF (j.lt.(abs(n-trgspin)) .or. j.gt.((n+trgspin))) then
	  WRITE(*,*)'ERROR:j not within correct bounds. Ending program'
	  GOTO 874
	ELSE IF (jp.lt.(abs(np-trgspin)) .or. jp.gt.(np+trgspin)) then
	  WRITE(*,*)'ERROR:jp not within correct bounds. Ending program'
	  GOTO 874
	ELSE
	  GOTO 234
	END IF

234    WRITE(*,*) "Target spin defined as S=",trgspin
	WRITE(*,*) "Spin-Rotation cross-sections will be computed"
      call TROTSEC3(nsym,maxene,ltmt,maxchi,maxchf,maxvi,maxvj,ndopen,
     1     mgvn,enscat,tmr,tmi,num_pw,symtyp,def,
     1     n,np,mjj,be,net,anr_op,sigma,iprnt(4),iwrite,trgspin,j,jp)

	END IF
	


c Calculate Born cross sections: total and partial 
c dipole allowed transitions...

      if (abs(np-n).gt.2.and.cb_op.eq.1) then
         write(iwrite,570)np-n
 570     format(' **Born approximation only implemented for |j-jp|
     &<=2, but not =',i2)
         cb_op=0
      end if

c Again need to force override cb_op as it must =1 when doing spin-coupled transitions
c needed to calc 0-1 rotational Born correction
      IF (trgspin.ne.-1.0) THEN
         cb_op = 1
      ENDIF

 5811 if (cb_op.eq.1.or.cb_op.eq.2) then

c create energy grid in case of no T-matrices are given

         if (cb_op.eq.2) then
            if (emin.le.0.d0) stop 'EMIN should be > 0'
            if (emax.le.0.d0) stop 'EMAX should bd > 0'

            if (ieunit.eq.1) then   ! Energy in T-matrix files is in RYD. 
               emin=emin*RYD        ! As we use the same variable to store
               emax=emax*RYD        ! the energy grid, this is set up in RYD.
            end if
            if (ieunit.eq.2) then
               emin=2.d0*emin
               emax=2.d0*emax
            end if

            if (grid.eq.1) then ! linear grid 
               if (numener.le.1) then
                  paso=0.d0
                  numener=1
               else
                  paso=(emax-emin)/dble(numener-1)
               end if
               write(iwrite,582)emin,emax,numener
 582           format(/,' Linear grid in energy (RYD) with EMIN=',
     &                e9.2,' EMAX=',e9.2,' NUMENER=',i5)
               allocate (enscat(numener))
               do ie=1,numener
                  enscat(ie)=emin+paso*dble(ie-1)
               end do
            end if
            if (grid.eq.2) then ! logarithmic grid
               if (numener.le.1) then
                  paso=0.d0
                  numener=1
               else
                  paso=(dlog(emax)-dlog(emin))/dble(numener-1)
               end if
               write(iwrite,5831)emin,emax,numener
 5831          format(/,'Logarithmic grid in energy (RYD) with EMIN=',
     &                e9.2,' EMAX=',e9.2,' NUMENER=',i5) 
               allocate (enscat(numener))
               do ie=1,numener
                  enscat(ie)=exp(log(emin)+paso*dble(ie-1))
               end do
            end if
            net=numener
            allocate (sigma(net))
         end if

         allocate (tcbsigma(net),acbsigma(net))

c call routine to initialize binom, neccessary for the threej function
         call pascal(binom,100)

         write(iwrite,580)
 580     format(/,' **** Born rotational cross sections.',//,
     &          '   ie ',6x,'ei (eV)',7x,'ef (eV)',7x,'TCB (A^2)') 
c
         do ie=1,net
            ei=enscat(ie)/2.d0  ! energy is converted from Ryd to au
            def=be*dble(np*(np+1)-n*(n+1)) ! be should be in au!
            ef=ei-def
            if (ef.lt.0.d0) goto 600
            if (abs(np-n).eq.1) then
               if (lmax.gt.40) write(iwrite,581)
 581     format(' Warning! LMAX is very large and there is risk of
     &overflow.')
c
c Dipole CB cross sections...Depends on if spin-coupling required
c


	IF (trgspin.eq.-1.0) THEN
              call DCB(ei,ef,n,np,lmax,q1,dxs,pdxs,iwrite,iprnt(4))
	ELSE
              call DCBSC(ei,ef,n,np,lmax,q1,dxs,pdxs,iwrite,iprnt(4),j,
     &                      jp,trgspin)
	END IF		
c
              tcbsigma(ie)=dxs  ! total CB-cross sections
              acbsigma(ie)=pdxs ! sum of the first lmax partial-CB XS
c
            end if
c
c Quadrupole CB cross sections... CURRENTLY NOT IMPLEMENTED (AF 04/04/2007)
c check k, qa, qb, npq
c
c            if (abs(jp-j).eq.2) then
c               if (lmax.gt.lmaxq) then
c                  write(iwrite,583)
c                  lmax=lmaxq
c               end if
c 583           format(' Warning!! lmax should not be > lmaxq',/,
c     &                '           lmax has been reset to lmax=lmaxq')
c               call QCB(ei,ef,j,jp,k,lmax,lmaxq,q2,ion,aq,bq,
c     &                   qxs,pqxs,iwrite,iprnt(4))
c               tcbsigma(ie)=qxs  ! total CB-cross sections
c               acbsigma(ie)=pqxs ! sum of the first lmax partial-CB XS
c            end if

c
c     Threshold correction to the cross-sections
c
            if (anr_op.eq.1) then
c     neutral correction based on Morrison treatment
               anr=sqrt(ef/ei)
               acbsigma(ie)=acbsigma(ie)*anr
            end if
c
            ei=ei*ev  ! energy coverted from au to eV for output in iwrite
            ef=ef*ev
c
            write(iwrite,590)ie,ei,ef,tcbsigma(ie)*angst2
c 590        format(i5,2(2x,f12.6),2x,g16.6)
 590        format(i5,2(2x,f12.4),2x,g16.6)
c
 600     end do                 ! ie
      end if

c Print results in file

      itype=1 ! by default, this is an "R-matrix" rotatitonal xs
      if (cb_op.eq.1) itype=2  ! Total CB xs included
      if (cb_op.eq.1.and.lmax.gt.0) itype=3  ! TCB and partial CB xs
      if (cb_op.eq.2) itype=4 ! only Total CB xs calculated 
      if (cb_op.eq.2.and.lmax.gt.0) itype=5 ! only partial and Total CB 
                                            !xs calculated

c
      def=def*2.d0  !def is converted from Hartree to Ryd and passed to wrotxs
      call WROTXS(name_s,itype,n,np,lmax,lmaxq,net,enscat,def,
     &     symtyp,sigma,tcbsigma,acbsigma,ixsn,ieunit,ixsout)
C
C---- Close files and return to main routine
      do js=1,nsym
         CLOSE(UNIT=LUCHANv(js),STATUS='KEEP')
         CLOSE(UNIT=LUTMTv(js),STATUS='KEEP')
      end do
      if (cb_op.eq.0.or.cb_op.eq.1) then
         deallocate (enscat,tmr,tmi,sigma,ivchl,lvchl,mvchl,evchl)
      else if (cb_op.eq.1.or.cb_op.eq.2) then
         deallocate(enscat,tcbsigma,acbsigma,sigma)
      endif
      WRITE(IWRITE,16)
      return
C
 16   FORMAT(//' *** Task has been successfully completed ***')
 92   FORMAT(/' *** DATA ON T-MATRIX FILE IS INCOMPATIBLE WITH CHANNEL D
     1ATA ***')
 104  FORMAT(/' Cross-sections will be computed for ',I5,' energies in t
     1he T-matrix file',/,' from',g12.5,' to',g12.5,A4)
 105  FORMAT(10A8)
 107  FORMAT(/' Number of target states       =',I4/' Number of scatteri
     1ng channels =',I4)
 874  END
c
c ************************************************************************************************************

      subroutine WROTXS(name,itype,j,jp,lmax,lmaxq,net,energy,def,
     &     symtyp,sigma,tcb,acb,ixsn,ieunit,iout)
      IMPLICIT double precision (A-H,O-Z)
      integer symtyp
      CHARACTER*80 NAME
      character*23 type(5)
      character*4 eunit(3)
      CHARACTER*8 XUNIT(3)
      parameter (ev=2.d0/0.0735d0,angst2=0.2800285608592d0,
     1           pi=3.1415926535897932D+00)
      dimension energy(net),sigma(net),tcb(net),acb(net)
      data eunit/' eV ',' E_h',' Ryd'/,
     &     XUNIT/'BOHR**2','ANGS**2','PI*A0**2'/     
      data type/'R-matrix XS            ',
     &          'R-matrix & Total CB XS ',
     &          'R-matrix & TCB & ACB XS',
     &          'Total CB XS            ',
     &          'Partial and total CB XS'/

c ***********************************************************************
C      NAME     = TITLE FOR ANY OUTPUT
c     itype     = Flag to indicate the type of XS  
c                 1 - R-matrix XS
c                 2 - R-matrix and total Born XS
c                 3 - R-matrix, Total CB and added CB up to lmax partial
c                      waves
c                 4 - Total CB rotational cross sections
c                 5 - Partial and total CB rotational cross sections
c     j and jp  = Initial and final rotational numbers of the target
c     lmax      = Number of partial waves included in "added CB" (ACB) XS
c     lmaxq     = Number of partial waves included in the total CB XS 
c                 (quadrupole)
c     net       = Number of energies
c     energy    = Vector containing the energies
c     def       = Threshold energy
c     symtyp    = g/u symmetry of the molecule if applicable
c     sigma     = Vector containing the XS
c     ixsn      = Flag to select units for XS 
c                 1 - a0^2
c                 2 - Angstrom^2
c                 3 - pi*a0^2
c     ieunit    = Flag to select units for the energy
c                 1 - eV
c                 2 - Hartree
c                 3 - Ryd
c     iout      = Unit to write the data
c ***********************************************************************

c calculate conversion units factors. ---------------------------------
      if (ieunit.eq.1)  energy_units=ev*0.5d0   ! Ryd --> eV
      if (ieunit.eq.2)  energy_units=0.5d0      ! Ryd --> Hartree
      if (ieunit.eq.3)  energy_units=1.d0       ! Ryd --> Ryd

      if (ixsn.eq.1) xs_units=1.d0              ! a0^2 --> a0^2
      if (ixsn.eq.2) xs_units=angst2            ! a0^2 --> Angtroms^2
      if (ixsn.eq.3) xs_units=pi                ! a0^2 --> pi*a0^2
c ---------------------------------------------------------------------
c
c      write(iout,100)name
c 100  format(/'# ',a)
c      write(iout,110)itype,type(itype),net
c 110  format('#',/,'# key= ',i2,2x,a,/,'# Number of energies=',i5)
c      write(iout,120)j,jp
c 120  format('#',/,'# Rotational transition: j=',i2,'-->',i2)
c      if (symtyp.eq.1) then
c         write(iout,125)
c 125  format('#',/,'# The target is homonuclear. Total symmetry is g.')
c      else if (symtyp.eq.-1) then
c         write(iout,126)
c 126  format('#',/,'# The target is homonuclear. Total symmetry is u.')
c      end if
c      write(iout,130)eunit(ieunit),xunit(ixsn)
c 130  format('#',/,'# Energy in ',a4,'; Cross sections in ',a8,/,'#')

      def=def*energy_units
c      write(iout,135)def,eunit(ieunit)
c 135  format('# Threshold energy',e17.8,' ',a4,/,'#')

      write(iout,'(2i3,(2x,f10.8))') j,jp,def

      if (itype.eq.1) write(iout,140)
c     if ((itype.eq.2.or.itype.eq.3.or.itype.eq.4.or.itype.eq.5)
c     &     .and.abs(j-jp).eq.2) write(iout,145)lmaxq
c     &     .and.abs(j-jp).eq.2) write(iout,161)
      if (itype.eq.2) write(iout,150)
c      if (itype.eq.3) write(iout,160)lmax
      if (itype.eq.3) write(iout,161)
      if (itype.eq.4) write(iout,170)
      if (itype.eq.5) write(iout,180)lmax
 140  format('#',4x,'Energy',10x,'TM CS  ',/,'#')
 145  format('# Number of partial waves in TCB (lmaxq)=',i2,/,'#')
 150  format('#',4x,'Energy',10x,'TM CS  ',9x,'TCB CS  ',/,'#')
 160  format('# Number of partial waves in PCB (lmax) =',i2,/,'#',/,
     &     '#',4x,'Energy',6x,'TM+TCB-PCB',8x,'TM CS  ',8x,'TCB CS  ',
     &     8x,'PCB CS  ')
 161  format('#',4x,'Energy',6x,'TM+TCB-PCB',8x,'TM CS  ',8x,'TCB CS  ',
     &     8x,'PCB CS  ',/,'#')
 170  format('#',4x,'Energy',10x,'TCB CS  ',/,'#')
 180  format('# Number of partial waves in ACB (lmax) =',i2,/,'#',/,
     &     '#',4x,'Energy',10x,'TCB CS  ',8x,'ACB CS  ',/,'#')
c     
      do ie=1,net
         energy(ie)=energy(ie)*energy_units
         sigma(ie) =sigma(ie)*xs_units        
      end do
      if (itype.eq.2.or.itype.eq.3.or.itype.eq.4.or.itype.eq.5) then
         do ie=1,net
            tcb(ie)=tcb(ie)*xs_units
            acb(ie)=acb(ie)*xs_units
         end do
      end if
      if (itype.eq.1) then
         do ie=1,net
            if (sigma(ie).gt.1e-10) then
               write(iout,190)energy(ie),sigma(ie)
            endif
         end do
      else if (itype.eq.2) then
         do ie=1,net
            write(iout,190)energy(ie),sigma(ie),tcb(ie)
         end do
      else if (itype.eq.3) then
         do ie=1,net
            if (sigma(ie).gt.1e-10) then
               write(iout,190)energy(ie),
     &              sigma(ie)+tcb(ie)-acb(ie),
     &              sigma(ie),tcb(ie),acb(ie)
            endif
         end do
      else if (itype.eq.4) then
         do ie=1,net
            write(iout,190)energy(ie),tcb(ie)
         end do
      else if (itype.eq.5) then
         do ie=1,net
            write(iout,190)energy(ie),tcb(ie),acb(ie)
         end do
      end if
c
c 190  format(f12.8,5(2x,f14.6))
 190  format(f12.4,5(2x,e14.6))
      return
      end
ccc
c ************************************************************************************************************
c Program to calculate rotational cross sections from  BF T matrices
c
      subroutine TROTSEC(nsym,maxern,ltmt,maxchi,maxchf,maxvi,maxvj,
     *                   ndopen,mgvn,enscat,tmr,tmi,num_pw,symtyp,dej,
     1     j,jp,mjj,be,net,anr_op,sigma,iprint,iout)

      implicit double precision (a-h,o-z)
      double precision kj,kjp
      integer anr_op,symtyp
      logical lamok

      parameter (pi=3.1415926535897932D+00,angst2=0.2800285608592d0,
     &           ev=2.d0/0.0735d0)
      parameter (maxli=50,maxlf=50,maxjj=50)

      dimension tmr(ltmt,nsym),tmi(ltmt,nsym)
      dimension maxvi(maxern,nsym),maxvj(maxern,nsym),
     1          ndopen(maxern,nsym),sigma(*)
      dimension maxchi(nsym),maxchf(nsym)
      dimension mgvn(nsym),enscat(net)
      dimension tr(nsym,maxli,maxlf),ti(nsym,maxli,maxlf)
      dimension num_pw(nsym)
      dimension i_count(nsym)
      dimension tt(0:maxjj),ttr(0:maxjj),tti(0:maxjj),xs(0:maxjj)
      dimension binom(100,100)
c
c nsym... number of symmetries (of T matrices) to consider (to read)
c j...... initial rotational quantum number
c jp..... final rotational quantum number
c mjj.... Maximum value for J (total angular momentum quantum number)
c mgvn.... vector containing the lambda values of the T-matrices.


c be ... rotational constant. It's used to calculate the rotational thresholds
c        with the formula E_j=2*be*j(j+1)

      write(iout,190)
 190  format(' **** Subroutine TROTSEC. Calculation of rotational cross s
     &ections from T-matrices',/)

c initialize Clebsh-Gordon coeffs
        call pascal(binom,100)

c initialise accumulative pointers used in the T-matrix extraction
      
c      do js=1,nsym
c         i_count(js)=0
c      end do

c do loop on energy

      write(iout,351)
 351  format('  I  ','  energy (eV) ','  sigma (A^2)',
     &'   Col. strength')
c
      do js=1,nsym
        i_count(js) = 0
      end do
      do ienergy=1,net
         energy=enscat(ienergy) ! we assume that the energies are the
                                ! same in all T-matrix files. 
         energy=energy/2.d0     ! convert Ryd to au 

c extract T-matrices 

         do js=1,nsym
            kk = i_count(js)
            do lf=1,maxchf(js)
               do li=1,maxchi(js)
                  kk = kk+1
                  if (lf.le.maxvi(ienergy,js).and.
     &                li.le.(maxvj(ienergy,js)+ndopen(ienergy,js))) then
c
                     tr(js,li,lf)=tmr(kk,js)
                     ti(js,li,lf)=tmi(kk,js)
c                    
                  end if
               end do
            end do
            i_count(js) = kk
         end do
c
c the value of li and lf for each symmetry is not the real one. For example,
c in the symmetry Delta, there are no partial waves with l<2.
c The value \lambda for each symmetry is used to correct this storage-meaning
c discrepancy.
c
c check T-matrices if debugging
         if (iprint.gt.0) then
            write(iout,210)
 210        format(/,' First elements of the T-matrix',/)
            do js=1,nsym
               write(iout,220)mgvn(js),tr(js,1,1),tr(js,2,1),ti(js,1,1),
     &             ti(js,2,1)
            end do
 220        format(2x,i2,4(1x,g20.13))
c 
            write(iout,230)energy
 230        format(/,'::::::::::::::::::::::::: Energy= ',f12.5)
         end if

c calculate momenta to correct the ANR cross sections

         ej=be*dble(j*(j+1))
         ejp=be*dble(jp*(jp+1))
         dej=ejp-ej ! dej is the energy the molecule will take from the electron
         if (dej.gt.energy) goto 9999

         kj=dsqrt(2.d0*energy)
         kjp=dsqrt(2.d0*(energy-dej))

c do loop on J (total angular momentum), which is jj in this code.

      do jj=0,mjj

         if (iprint.gt.0) write(iout,240)jj
 240     format(/,'=======================================  J=',i2)

c do loop on li and lf

         tt(jj)=0.d0

         increment=1
         if (abs(symtyp).eq.1) increment=2
         do l=abs(JJ-j),jj+j,increment
            do lp=abs(jj-jp),jj+jp,increment

               if (iprint.gt.1) write(iout,250)l,lp
 250     format(/,' ------------------------------ l=',i2,' lp=',i2)

c do loop on lambda (the symmetry of the system)

               lpar=-1
               if (mod(l,2).eq.mod(lp,2)) lpar=mod(l,2) ! l and lp are both even or odd
               
               ttr(jj)=0.d0
               tti(jj)=0.d0
               do lambda=-lp,lp

                  if (symtyp.eq.1.and.lpar.ne.0) go to 20 ! when gerade, only l,lp even are needed
                  if (symtyp.eq.-1.and.lpar.ne.1) go to 20 ! when ungerade, only l,lp odd are needed

                  if (iprint.gt.1)write(iout,260)lambda
 260              format(/,' .............................. lambda=',i2)

                  lam=abs(lambda)

c if l is smaller that lam, then there won't be a T-matrix element
c for this case
                  if (l.lt.lam) then
                     if (iprint.gt.1) write(iout,270)l,lam
 270                 format(' l=',i2,' < lam=',i2,' ....skipping')
                     go to 20
                  end if

c check that there is a T-matrix for that lambda

                  lamok=.false.
                  do js=1,nsym
                     if (lam.eq.mgvn(js)) then 
                        lamok=.true.
                        itsym=js
                     end if
                  end do

                  if (lamok) then

                     a=dsqrt(dble(2*j+1))*(-1)**(j-l+lambda)
                     ap=dsqrt(dble(2*jp+1))*(-1)**(jp-lp+lambda)

                     cg=threej(j,l,JJ,0,lambda,-lambda,binom)
                     a=a*cg
            if (iprint.gt.2) write(iout,280)j,l,jj,lambda,-lambda,cg

                     cg=threej(jp,lp,JJ,0,lambda,-lambda,binom)
                     ap=ap*cg
            if (iprint.gt.2) write(iout,280)jp,lp,jj,lambda,-lambda,cg

 280       format('CG(',i2,',',i2,',',i2,';0,',i2,',',i2,')=',f12.5)

c translate the values of l and lp, so they correspond to the values
c used for storage in the matrices tr and ti

                     if (symtyp.eq.1) lcorr=0     
                     if (symtyp.eq.-1) lcorr=1

                     if (symtyp.eq.0) then
                        lstor=l-lam+1
                        lpstor=lp-lam+1
                     else
                        lstor=int((l-lam)/2+1)
                        lpstor=int((lp-lam)/2+1)
                     end if

          if (iprint.gt.1) write(iout,290)lambda,l,lp,itsym,lstor,lpstor
 290      format(/,2x,'T(',2(i2,','),i2,') <-- T(',2(i2,','),i2,')')

                     if (lstor.gt.num_pw(itsym)) then 
                        if (iprint.gt.1)
     &                   write(iout,300)l,itsym
 300  format(2x,'l=',i2,' for sym=',i2,2x,'***not available***')
                        go to 20
                     end if
                     if (lpstor.gt.num_pw(itsym)) then 
                        if (iprint.gt.1) write(iout,301)lp,itsym
 301  format(2x,'lp=',i2,' for sym=',i2,2x,'***not available***')
                        go to 20
                     end if

                     ttr(jj)=ttr(jj)+a*ap*tr(itsym,lstor,lpstor)
                     tti(jj)=tti(jj)+a*ap*ti(itsym,lstor,lpstor)

                     if (iprint.gt.3) then ! detailed debugging
                        write(iout,310)tr(itsym,lstor,lpstor),
     &                                 ti(itsym,lstor,lpstor)
 310                    format(2x,'tr()=',g20.13,' ti()=',g20.13)

                        write(iout,320)jj,ttr(jj)
                        write(iout,321)jj,tti(jj)
 320                    format(2x,'ttr(',i2,')=',g20.13)
 321                    format(2x,'tti(',i2,')=',g20.13)
                     end if

                  else
                     if (iprint.gt.1)
     &                write(iout,*)' There is no T-mat for lambda=',lam
                  end if

 20               continue
               end do

               tt(jj)=tt(jj)+(2*JJ+1)*(ttr(jj)*ttr(jj)+tti(jj)*tti(jj))

               if (iprint.gt.0) write(iout,330)jj,tt(jj)
 330           format('tt(',i2,')=',f14.7)
               
            end do
         end do

* this assumes that energy is in Hartree
         xs(jj)=pi/(2.d0*energy*dble(2*j+1)) * tt(jj)

      end do

c final sum
      sigt=0.d0
      do jj=0,mjj
         sigt=sigt+xs(jj)
      end do

      if (iprint.gt.0) write(iout,340)(xs(jj),jj=0,mjj)
 340  format(' xs(jj): ',6(1x,f12.5))

c calculate correction ANR and cross section

      anr=kjp/kj

      if (anr_op.eq.1) sigt=anr*sigt
      sigma(ienergy)=sigt
c
      if (iprint.gt.0) write(iout,*)'ANR correction =',anr
c
c calculate collision strengths (energy in Hartree)
c
      omega=sigt*2.d0*energy*dble(2*j+1)/pi
c
c output results
      write(iout,350)ienergy,energy*ev,sigt*angst2,omega
c 350  format(i5,f12.5,f12.3,f12.3)
 350  format(i5,f12.4,f12.3,f12.3)

 9999 continue

c end do loop on energies

      end do

c end program

      return
      end
c
c ************************************************************************************************************
c     Subroutine TROTSEC2 is equivalent to TROTSEC. It is just implemented
c     in a different way, following Itikawa Theor Chem Acc (2000) 105 123.
c     It was written in preparation to DCS implementation but it was finally
c     decided not to implement the DCS calculation in the present code
c     AF 05/04/2007

      subroutine TROTSEC2(nsym,maxern,ltmt,maxchi,maxchf,maxvi,maxvj,
     *     ndopen,mgvn,enscat,tmr,tmi,num_pw,symtyp,dej,
     1     j,jp,mjj,be,net,anr_op,sigma,iprint,iout)
      
      implicit double precision (a-h,o-z)
      double precision kj,kjp
      integer anr_op,symtyp
      logical lamok

      parameter (pi=3.1415926535897932D+00,angst2=0.2800285608592d0,
     &           ev=27.2114)
      parameter (maxli=10,maxlf=30,maxjj=10)

      dimension tmr(ltmt,nsym),tmi(ltmt,nsym)
      dimension maxvi(maxern,nsym),maxvj(maxern,nsym),
     1          ndopen(maxern,nsym),sigma(*)
      dimension maxchi(nsym),maxchf(nsym)
      dimension mgvn(nsym),enscat(net)
      dimension tr(nsym,maxli,maxlf),ti(nsym,maxli,maxlf)
      dimension num_pw(nsym)
      dimension i_count(nsym)
      dimension tt(0:maxjj),ttr(0:maxjj),tti(0:maxjj),xs(0:maxjj)
      dimension Q(0:maxjj)
      dimension binom(100,100)
c
c nsym... number of symmetries (of T matrices) to consider (to read)
c j...... initial rotational quantum number
c jp..... final rotational quantum number
c mjj.... Maximum value for J (total angular momentum quantum number)
c mgvn.... vector containing the lambda values of the T-matrices.


c be ... rotational constant. It's used to calculate the rotational thresholds
c        with the formula E_j=2*be*j(j+1)

      write(iout,190)
 190  format(' **** Subroutine TROTSEC2. Calculation of rotational cross s
     &ections from T-matrices',/)

c initialize Clebsh-Gordon coeffs
        call pascal(binom,100)

c initialise accumulative pointers used in the T-matrix extraction
      
c      do js=1,nsym
c         i_count(js)=0
c      end do

c do loop on energy

      write(iout,351)
 351  format('  I  ','  energy (eV) ','  sigma (A^2)',
     &'   Col. strength')
c
      do js=1,nsym
        i_count(js) = 0
      end do
      do ienergy=1,net
         energy=enscat(ienergy) ! we assume that the energies are the
                                ! same in all T-matrix files. 
         energy=energy/2.d0     ! convert Ryd to au 

c extract T-matrices 

         do js=1,nsym
            kk = i_count(js)
            do lf=1,maxchf(js)
               do li=1,maxchi(js)
                  kk = kk+1
                  if (lf.le.maxvi(ienergy,js).and.
     &                li.le.(maxvj(ienergy,js)+ndopen(ienergy,js))) then
c
                     tr(js,li,lf)=tmr(kk,js)
                     ti(js,li,lf)=tmi(kk,js)
c                    
                  end if
               end do
            end do
            i_count(js) = kk
         end do
c
c the value of li and lf for each symmetry is not the real one. For example,
c in the symmetry Delta, there are no partial waves with l<2.
c The value \lambda for each symmetry is used to correct this storage-meaning
c discrepancy.
c
c check T-matrices if debugging
         if (iprint.gt.0) then
            write(iout,210)
 210        format(/,' First elements of the T-matrix',/)
            do js=1,nsym
               write(iout,220)mgvn(js),tr(js,1,1),tr(js,2,1),ti(js,1,1),
     &             ti(js,2,1)
            end do
 220        format(2x,i2,4(1x,g20.13))
c 
            write(iout,230)energy
 230        format(/,'::::::::::::::::::::::::: Energy= ',f12.5)
         end if

c calculate momenta to correct the ANR cross sections

         ej=be*dble(j*(j+1))
         ejp=be*dble(jp*(jp+1))
         dej=ejp-ej ! dej is the energy the molecule will take from the electron
         if (dej.gt.energy) goto 9999

         kj=dsqrt(2.d0*energy)
         kjp=dsqrt(2.d0*(energy-dej))

c
c     here begins the frame transformation implementation
c

c
c     lma1 is lmax  (mjj=lmax+j)
c     
         lma1 = mjj-j
         
         
c do loop on J (total angular momentum), which is jj in this code.

         do jj=0,mjj

         if (iprint.gt.0) write(iout,240)jj
 240     format(/,'=======================================  J=',i2)

c do loop on li and lf

         tt(jj)=0.d0
         Q(jj)=0.d0

         increment=1
         if (abs(symtyp).eq.1) increment=2
         do l=0, lma1, increment
            do lp=0, lma1, increment
               
               if (iprint.gt.1) write(iout,250)l,lp
 250     format(/,' ------------------------------ l=',i2,' lp=',i2)

c do loop on lambda (the symmetry of the system)

               lpar=-1
               if (mod(l,2).eq.mod(lp,2)) lpar=mod(l,2) ! l and lp are both even or odd
               
               ttr(jj)=0.d0
               tti(jj)=0.d0
               do lambda=-lp,lp
                  
                  if (symtyp.eq.1.and.lpar.ne.0) go to 20 ! when gerade, only l,lp even are needed
                  if (symtyp.eq.-1.and.lpar.ne.1) go to 20 ! when ungerade, only l,lp odd are needed
                  
                  if (iprint.gt.1)write(iout,260)lambda
 260              format(/,' .............................. lambda=',i2)
                  
                  lam=abs(lambda)

c if l is smaller that lam, then there won't be a T-matrix element
c for this case
                  if (l.lt.lam) then
                     if (iprint.gt.1) write(iout,270)l,lam
 270                 format(' l=',i2,' < lam=',i2,' ....skipping')
                     go to 20
                  end if

c check that there is a T-matrix for that lambda

                  lamok=.false.
                  do js=1,nsym
                     if (lam.eq.mgvn(js)) then 
                        lamok=.true.
                        itsym=js
                     end if
                  end do

                  if (lamok) then

                     a=(-1)**lambda

                     cg=threej(lp,l,jj,lambda,-lambda,0,binom)
                     a=a*cg
            if (iprint.gt.2) write(iout,280) lp,l,jj,lambda,-lambda,cg

 280        format('CG(',i2,',',i2,',',i2,',',i2,';0,',')=',
     &           f12.5)

c translate the values of l and lp, so they correspond to the values
c used for storage in the matrices tr and ti

                     if (symtyp.eq.1) lcorr=0     
                     if (symtyp.eq.-1) lcorr=1

                     if (symtyp.eq.0) then
                        lstor=l-lam+1
                        lpstor=lp-lam+1
                     else
                        lstor=int((l-lam)/2+1)
                        lpstor=int((lp-lam)/2+1)
                     end if

          if (iprint.gt.1) write(iout,290)lambda,l,lp,itsym,lstor,lpstor
 290      format(/,2x,'T(',2(i2,','),i2,') <-- T(',2(i2,','),i2,')')

                     if (lstor.gt.num_pw(itsym)) then 
                        if (iprint.gt.1)
     &                   write(iout,300)l,itsym
 300  format(2x,'l=',i2,' for sym=',i2,2x,'***not available***')
                        go to 20
                     end if
                     if (lpstor.gt.num_pw(itsym)) then 
                        if (iprint.gt.1) write(iout,301)lp,itsym
 301  format(2x,'lp=',i2,' for sym=',i2,2x,'***not available***')
                        go to 20
                     end if

                     ttr(jj)=ttr(jj)+a*tr(itsym,lstor,lpstor)
                     tti(jj)=tti(jj)+a*ti(itsym,lstor,lpstor)

                     if (iprint.gt.3) then ! detailed debugging
                        write(iout,310)tr(itsym,lstor,lpstor),
     &                       ti(itsym,lstor,lpstor)
 310                    format(2x,'tr()=',g20.13,' ti()=',g20.13)
                        
                        write(iout,320)jj,ttr(jj)
                        write(iout,321)jj,tti(jj)
 320                    format(2x,'ttr(',i2,')=',g20.13)
 321                    format(2x,'tti(',i2,')=',g20.13)
                     end if

                  else
                     if (iprint.gt.1)
     &                write(iout,*)' There is no T-mat for lambda=',lam
                  end if
                  
 20               continue
               end do

               tt(jj)=tt(jj)+(ttr(jj)*ttr(jj)+tti(jj)*tti(jj))

               if (iprint.gt.0) write(iout,330)jj,tt(jj)
 330           format('tt(',i2,')=',f14.7)
               
            end do
         end do
         
* this assumes that energy is in Hartree
c 
c     The formula below corresponds to equations 14 and 15 of Shimoi and
c     Itikawa JPB 32 65 (1999). 
c
         Q(jj) = Q(jj) + pi/(2.d0*energy)*(2*JJ+1)*tt(jj)
         xs(jj)=(2*jp+1)*threej(jp,jj,j,0,0,0,binom)**2*Q(jj)
         

      end do
      
c     final sum
      sigt=0.d0
      do jj=0,mjj
         sigt=sigt+xs(jj)
      end do
      
      if (iprint.gt.0) write(iout,340)(xs(jj),jj=0,mjj)
 340  format(' xs(jj): ',6(1x,f12.5))
      
c calculate correction ANR and cross section
      
      anr=kjp/kj
      if (anr_op.eq.1) sigt=anr*sigt
      sigma(ienergy)=sigt
c
      if (iprint.gt.0) write(iout,*)'ANR correction =',anr
c
c calculate collision strengths (energy in Hartree)
c
      omega=sigt*2.d0*energy*dble(2*j+1)/pi
c
c output results

      write(iout,350)ienergy,energy*ev,sigt*angst2,omega
 350  format(i5,f12.5,f12.3,f12.3)

 9999 continue

c end do loop on energies

      end do

c end program

      return
      end

c ************************************************************************************************************
c ************************************************************************************************************
c Program to calculate spin-inclusive rotational cross sections from  BF T matrices
      subroutine TROTSEC3(nsym,maxern,ltmt,maxchi,maxchf,maxvi,maxvj,
     *                   ndopen,mgvn,enscat,tmr,tmi,num_pw,symtyp,dej,
     1     n,np,mjj,be,net,anr_op,sigma,iprint,iout,S,j,jp)

      implicit double precision (a-h,o-z)
      double precision kn,knp,s,tj,sj,finalsigt,j,jp,dn,dnp,dl,nsigt(99)
      double precision sigt(99)
      integer anr_op,symtyp,n,np,lamlim
      logical lamok

      parameter (pi=3.1415926535897932D+00,angst2=0.2800285608592d0,
     &           ev=2.d0/0.0735d0)
      parameter (maxli=50,maxlf=50,maxjj=50)

      dimension tmr(ltmt,nsym),tmi(ltmt,nsym)
      dimension maxvi(maxern,nsym),maxvj(maxern,nsym),
     1          ndopen(maxern,nsym),sigma(*)
      dimension maxchi(nsym),maxchf(nsym)
      dimension mgvn(nsym),enscat(net)
      dimension tr(nsym,maxli,maxlf),ti(nsym,maxli,maxlf)
      dimension num_pw(nsym)
      dimension i_count(nsym)
      dimension tt(0:maxjj),ttr(0:maxjj),tti(0:maxjj),xs(0:maxjj)
      dimension binom(100,100)
      dimension partsum(100)

c
c nsym... number of symmetries (of T matrices) to consider (to read)
c n...... initial rotational quantum number
c np..... final rotational quantum number
c j...... user specified value of j (between |n-s| and n+s)
c jp..... user specified value of jp (between |np-s| and np+s)
c mjj.... Maximum value for J (total angular momentum quantum number)
c mgvn... vector containing the lambda values of the T-matrices.
c s...... spin value of the target molecule only

c be ... rotational constant. It's used to calculate the rotational thresholds
c        with the formula E_j=2*be*j(j+1)

      write(iout,190)
 190  format(' **** Subroutine TROTSEC3. Calculation of spin-rotation cross s
     &ections from T-matrices',/)

c initialize Clebsh-Gordon coeffs
        call pascal(binom,100)

c initialise accumulative pointers used in the T-matrix extraction
      
c      do js=1,nsym
c         i_count(js)=0
c      end do

c do loop on energy

      write(iout,351)
 351  format('  I  ','  energy (eV) ','  sigma (A^2)',
     &'   Col. strength')
c
      do js=1,nsym
        i_count(js) = 0
      end do
      do ienergy=1,net
         energy=enscat(ienergy) ! we assume that the energies are the
                                ! same in all T-matrix files. 
         energy=energy/2.d0     ! convert Ryd to au 

c extract T-matrices 

         do js=1,nsym
            kk = i_count(js)
            do lf=1,maxchf(js)
               do li=1,maxchi(js)
                  kk = kk+1
                  if (lf.le.maxvi(ienergy,js).and.
     &                li.le.(maxvj(ienergy,js)+ndopen(ienergy,js))) then
c
                     tr(js,li,lf)=tmr(kk,js)
                     ti(js,li,lf)=tmi(kk,js)
c                    
                  end if
               end do
            end do
            i_count(js) = kk
         end do
c
c the value of li and lf for each symmetry is not the real one. For example,
c in the symmetry Delta, there are no partial waves with l<2.
c The value \lambda for each symmetry is used to correct this storage-meaning
c discrepancy.
c
c check T-matrices if debugging
         if (iprint.gt.0) then
            write(iout,210)
 210        format(/,' First elements of the T-matrix',/)
            do js=1,nsym
               write(iout,220)mgvn(js),tr(js,1,1),tr(js,2,1),ti(js,1,1),
     &             ti(js,2,1)
            end do
 220        format(2x,i2,4(1x,g20.13))
c 
            write(iout,230)energy
 230        format(/,'::::::::::::::::::::::::: Energy= ',f12.5)
         end if

c Need to do the loop over the 0->Lam x-secs here.
c Note for this loop I replace n with 0, np with i
c As the x-secs wanted are 0->Lamda ones.
c Lam goes up to n+np
         lamlim = n+np
         do i=0,lamlim



c do loop on J (total angular momentum), which is jj in this code.

      do jj=0,mjj

         if (iprint.gt.0) write(iout,240)jj
 240     format(/,'=======================================  J=',i2)

c do loop on li and lf

         tt(jj)=0.d0

         increment=1
         if (abs(symtyp).eq.1) increment=2
         do l=abs(JJ-0),jj+0,increment
            do lp=abs(jj-i),jj+i,increment

               if (iprint.gt.1) write(iout,250)l,lp
 250     format(/,' ------------------------------ l=',i2,' lp=',i2)

c do loop on lambda (the symmetry of the system)

               lpar=-1
               if (mod(l,2).eq.mod(lp,2)) lpar=mod(l,2) ! l and lp are both even or odd
               
               ttr(jj)=0.d0
               tti(jj)=0.d0
               do lambda=-lp,lp

                  if (symtyp.eq.1.and.lpar.ne.0) go to 20 ! when gerade, only l,lp even are needed
                  if (symtyp.eq.-1.and.lpar.ne.1) go to 20 ! when ungerade, only l,lp odd are needed

                  if (iprint.gt.1)write(iout,260)lambda
 260              format(/,' .............................. lambda=',i2)

                  lam=abs(lambda)

c if l is smaller that lam, then there won't be a T-matrix element
c for this case
                  if (l.lt.lam) then
                     if (iprint.gt.1) write(iout,270)l,lam
 270                 format(' l=',i2,' < lam=',i2,' ....skipping')
                     go to 20
                  end if

c check that there is a T-matrix for that lambda

                  lamok=.false.
                  do js=1,nsym
                     if (lam.eq.mgvn(js)) then 
                        lamok=.true.
                        itsym=js
                     end if
                  end do

                  if (lamok) then

                     a=dsqrt(dble(2*0+1))*(-1)**(0-l+lambda)
                     ap=dsqrt(dble(2*i+1))*(-1)**(i-lp+lambda)

                     cg=threej(0,l,JJ,0,lambda,-lambda,binom)
                     a=a*cg
            if (iprint.gt.2) write(iout,280)0,l,jj,lambda,-lambda,cg

                     cg=threej(i,lp,JJ,0,lambda,-lambda,binom)
                     ap=ap*cg
            if (iprint.gt.2) write(iout,280)i,lp,jj,lambda,-lambda,cg

 280       format('CG(',i2,',',i2,',',i2,';0,',i2,',',i2,')=',f12.5)

c translate the values of l and lp, so they correspond to the values
c used for storage in the matrices tr and ti

                     if (symtyp.eq.1) lcorr=0     
                     if (symtyp.eq.-1) lcorr=1

                     if (symtyp.eq.0) then
                        lstor=l-lam+1
                        lpstor=lp-lam+1
                     else
                        lstor=int((l-lam)/2+1)
                        lpstor=int((lp-lam)/2+1)
                     end if

          if (iprint.gt.1) write(iout,290)lambda,l,lp,itsym,lstor,lpstor
 290      format(/,2x,'T(',2(i2,','),i2,') <-- T(',2(i2,','),i2,')')

                     if (lstor.gt.num_pw(itsym)) then 
                        if (iprint.gt.1)
     &                   write(iout,300)l,itsym
 300  format(2x,'l=',i2,' for sym=',i2,2x,'***not available***')
                        go to 20
                     end if
                     if (lpstor.gt.num_pw(itsym)) then 
                        if (iprint.gt.1) write(iout,301)lp,itsym
 301  format(2x,'lp=',i2,' for sym=',i2,2x,'***not available***')
                        go to 20
                     end if

                     ttr(jj)=ttr(jj)+a*ap*tr(itsym,lstor,lpstor)
                     tti(jj)=tti(jj)+a*ap*ti(itsym,lstor,lpstor)

                     if (iprint.gt.3) then ! detailed debugging
                        write(iout,310)tr(itsym,lstor,lpstor),
     &                                 ti(itsym,lstor,lpstor)
 310                    format(2x,'tr()=',g20.13,' ti()=',g20.13)

                        write(iout,320)jj,ttr(jj)
                        write(iout,321)jj,tti(jj)
 320                    format(2x,'ttr(',i2,')=',g20.13)
 321                    format(2x,'tti(',i2,')=',g20.13)
                     end if

                  else
                     if (iprint.gt.1)
     &                write(iout,*)' There is no T-mat for lambda=',lam
                  end if

 20               continue
               end do

               tt(jj)=tt(jj)+(2*JJ+1)*(ttr(jj)*ttr(jj)+tti(jj)*tti(jj))

               if (iprint.gt.0) write(iout,330)jj,tt(jj)
 330           format('tt(',i2,')=',f14.7)
               
            end do
         end do

* this assumes that energy is in Hartree
         xs(jj)=pi/(2.d0*energy*dble(2*0+1)) * tt(jj)

      end do

c final sum
      sigt(i)=0.d0
      do jj=0,mjj
         sigt(i)=sigt(i)+xs(jj)
      end do

c end do loop on the cycle through the 0->lambda x-sec creation
      end do

      
c Applying IOS approx to get spin-rot cross-sections
c eq. 4.11 of Corey & McCourt, J. Phys. Chem, 87, 15, 1983
	do lambda=0,lamlim
		tj = threej(np,n,lambda,0,0,0,binom)
c need to convert lambda, n and np to doubles for use in 6j routine
		dl = DBLE(lambda)
		dn = DBLE(n)
		dnp= DBLE(np)
		CALL sixj(dl,j,jp,s,dnp,dn,sj)

c include this bit for de-excitation x-secs, needed for detailed balance.
c This only works when np = 0.
                if (np.lt.n) then
                   if (np.eq.0) then
                      nsigt(lambda)=(2.d0*lambda+1.d0)*sigt(lambda)
                   else
                      write(iout,*)'Error'
                      write(iout,*)'np > 0 de-exc not implemented'
                      write(iout,*)'Ending program'
                      GOTO 99999
                   end if   
                else
                   nsigt(lambda)=sigt(lambda)
                end if   

		partsum(lambda) = (2.d0*n+1.d0)*(2.d0*np+1.d0)*(2.d0*jp+1.d0)*
     &                              (tj*tj)*(sj*sj)*nsigt(lambda)

	end do

	finalsigt = 0.0
	do ii=0,lamlim
		finalsigt = finalsigt + partsum(ii)
	end do

c calculate correction ANR and cross section
      ej=be*dble(n*(n+1))
      ejp=be*dble(n*(np+1))
      dej=ejp-ej ! dej is the energy the molecule will take from the electron
      if (dej.gt.energy) goto 9999

      kn=dsqrt(2.d0*energy)
      knp=dsqrt(2.d0*(energy-dej))

      anr=knp/kn
      if (anr_op.eq.1) finalsigt=anr*finalsigt
      sigma(ienergy)=finalsigt
c
      if (iprint.gt.0) write(iout,*)'ANR correction =',anr
c
c calculate collision strengths (energy in Hartree)
c
      omega=finalsigt*2.d0*energy*dble(2*n+1)/pi
c
c output results
      write(iout,350)ienergy,energy*ev,finalsigt*angst2,omega
c 350  format(i5,f12.5,f12.3,f12.3)
 350  format(i5,f12.4,f12.3,f12.3)

 9999 continue

c end do loop on energies

      end do

c end program

99999 return
      end
c
c
c ************************************************************************************************************
      subroutine DCB(ei,ef,j,jp,lmax,q1,dxs,pdxs,iwrite,iprint)
c
c     Program to calculate integral (total and partial) Born cross
c     section for electron molecule scattering. It is based on the paper
c     Itikawa, Theor Chem Acc 105 13 (2000)
c
      implicit double precision (a-h,o-z)
      double precision ki,kf
      parameter (maxl=40)
      common /binomium/binom(100,100)
      parameter (ev=2.d0/0.0735d0,angst2=.280028560859d0,
     *     pi=3.1415926535898d0)
      double precision kn,knp
      parameter (maxli=50,maxlf=50)
c
      dimension tbr(0:maxli,0:maxlf),tbi(0:maxli,0:maxlf)
      double precision Ib(0:maxli,0:maxlf)
     

c     get electron momentum

      ki=sqrt(2.d0*ei)
      kf=sqrt(2.d0*ef)

      factor = 8.d0*pi/3.d0*q1**2/ki**2*dlog(abs((ki+kf)/(ki-kf)))

c     calculate the number that depends on J 

      factor_j=dble(max(j,jp))/dble(2*j+1)

c     calculate the cross sections

      dxs=factor*factor_j         ! total Born XS

c
c     here begins the partial Born calculation for dipole correction
c     
      energy = ei
c
c     here begins the partial implementation
c
c
c     lma1 is lmax 
c     
      lma1 = lmax
        
c     do loop on li and lf
         
      ttdb=0.d0
      Qdb=0.d0
         
      do l=0, lma1
         do lp=0, lma1
            
            ttdbr=0.d0
            ttdbi=0.d0
            
            if (l.ne.lp) then
               
               Ib(l,lp)=dsin((pi*(l-lp))/2.d0)/
     &              (ki*dble(l-lp)*dble(l+lp+1))
               
               tbi(l,lp)=-4.d0/3.d0*q1*ki*
     &              sqrt(dble(2*l+1)*dble(2*lp+1))*
     &              threej(l,lp,1,0,0,0,binom)*
     &              Ib(l,lp)
               
               ttdbi=tbi(l,lp)
               
               ttdb=ttdb+(ttdbr*ttdbr+ttdbi*ttdbi)
               
            endif
            
         end do
      end do
      
*     this assumes that energy is in Hartree
c     
c     The formula below corresponds to equations 14 and 15 of Shimoi and
c     Itikawa JPB 32 65 (1999). 
c     
      Qdb = pi/(2.d0*energy)*(2*1+1)*ttdb
      pdxs =(2*jp+1)*threej(jp,1,j,0,0,0,binom)**2*Qdb
      
      
c     calculate collision strengths (energy in Hartree)
c     
      omega=pdxs*2.d0*energy*dble(2*j+1)/pi
c     
c     output results
      
      return
      end
c ************************************************************************************************************
      subroutine DCBSC(ei,ef,n,np,lmax,q1,finaldxs,finalpdxs,iwrite,
     &                      iprint,j,jp,s)
c
c     Program to calculate integral (total and partial) Born cross
c     section for electron molecule scattering. It is based on the paper
c     Itikawa, Theor Chem Acc 105 13 (2000)
c
      implicit double precision (a-h,o-z)
      double precision ki,kf,j,jp,s,npdxs,ndxs,tj,sj
      parameter (maxl=40)
      common /binomium/binom(100,100)
      parameter (ev=2.d0/0.0735d0,angst2=.280028560859d0,
     *     pi=3.1415926535898d0)
      double precision kn,knp
      parameter (maxli=50,maxlf=50)
c
      dimension tbr(0:maxli,0:maxlf),tbi(0:maxli,0:maxlf)
      double precision Ib(0:maxli,0:maxlf)
      dimension partsum(100)

c     get electron momentum

      ki=sqrt(2.d0*ei)
      kf=sqrt(2.d0*ef)

      factor = 8.d0*pi/3.d0*q1**2/ki**2*dlog(abs((ki+kf)/(ki-kf)))

c     calculate the number that depends on J 

      factor_n=dble(max(0,1))/dble(2*1+1)

c     calculate the cross sections

      dxs=factor*factor_n         ! total Born XS

      
c Believe spin-coupling should be implemented here.
c Applying IOS approx to get spin-rot cross-sections
c eq. 4.11 of Corey & McCourt, J. Phys. Chem, 87, 15, 1983

C Lambda only needs to go up to 1 here as only 0-1 has
C a Born correction of all 0->N' transitions
	do lambda=1,1
		tj = threej(np,n,lambda,0,0,0,binom)
c need to convert lambda, n and np to doubles for use in 6j routine
		dl = DBLE(lambda)
		dn = DBLE(n)
		dnp= DBLE(np)
		CALL sixj(dl,j,jp,s,dnp,dn,sj)

c include this bit for de-excitation x-secs, needed for detailed balance.
c This only works when np = 0.
                if (np.lt.n) then
                   if (np.eq.0) then
                      dxs=(2.d0*lambda+1.d0)*dxs
                   end if   
                else
                   ndxs=dxs
                end if   

		partsum(lambda) = (2.d0*n+1.d0)*(2.d0*np+1.d0)*(2.d0*jp+1.d0)*
     &                              (tj*tj)*(sj*sj)*ndxs

	end do

	finaldxs = 0.0
	do i=1,1
		finaldxs = finaldxs + partsum(i)
	end do


c **************
c     here begins the partial Born calculation for dipole correction
c     
      energy = ei
c
c     here begins the partial implementation
c
c
c     lma1 is lmax 
c     
      lma1 = lmax
        
c     do loop on li and lf
         
      ttdb=0.d0
      Qdb=0.d0
         
      do l=0, lma1
         do lp=0, lma1
            
            ttdbr=0.d0
            ttdbi=0.d0
            
            if (l.ne.lp) then
               
               Ib(l,lp)=dsin((pi*(l-lp))/2.d0)/
     &              (ki*dble(l-lp)*dble(l+lp+1))
               
               tbi(l,lp)=-4.d0/3.d0*q1*ki*
     &              sqrt(dble(2*l+1)*dble(2*lp+1))*
     &              threej(l,lp,1,0,0,0,binom)*
     &              Ib(l,lp)
               
               ttdbi=tbi(l,lp)
               
               ttdb=ttdb+(ttdbr*ttdbr+ttdbi*ttdbi)
               
            endif
            
         end do
      end do
      
*     this assumes that energy is in Hartree
c     
c     The formula below corresponds to equations 14 and 15 of Shimoi and
c     Itikawa JPB 32 65 (1999). 
c     
      Qdb = pi/(2.d0*energy)*(2*1+1)*ttdb
      pdxs =(2*1+1)*threej(1,1,0,0,0,0,binom)**2*Qdb

      
c Believe spin-coupling should be implemented here.
c Applying IOS approx to get spin-rot cross-sections
c eq. 4.11 of Corey & McCourt, J. Phys. Chem, 87, 15, 1983

C This upper limit on lambda is arbitrary...needs testing
C to find out at which value the results converge
	do lambda=1,1
		tj = threej(np,n,lambda,0,0,0,binom)
c need to convert lambda, n and np to doubles for use in 6j routine
		dl = DBLE(lambda)
		dn = DBLE(n)
		dnp= DBLE(np)
		CALL sixj(dl,j,jp,s,dnp,dn,sj)

c include this bit for de-excitation x-secs, needed for detailed balance.
c This only works when np = 0.
                if (np.lt.n) then
                   if (np.eq.0) then
                      pdxs=(2.d0*lambda+1.d0)*pdxs
                   end if   
                else
                   npdxs=pdxs
                end if   

		partsum(lambda) = (2.d0*n+1.d0)*(2.d0*np+1.d0)*(2.d0*jp+1.d0)*
     &                              (tj*tj)*(sj*sj)*npdxs

	end do

	finalpdxs = 0.0
	do i=1,1
		finalpdxs = finalpdxs + partsum(i)
	end do



      
c     calculate collision strengths (energy in Hartree)
c     
      omega=finalpdxs*2.d0*energy*dble(2*1+1)/pi
c     
c     output results
      
      return
      end

***
c ************************************************************************************************************
* Subroutine qcb 
c      subroutine QCB(ei,ef,j,jp,k,lmax0,lmax,q2,ion,a,b,
c     &                qxs,pqxs,iwrite,iprint)
c  
c  Program to calculate total quadrupole induced Born cross section
c  for electron-molecule scattering. It is based on the paper
c    Theor Chem Acc 105 13 (2000) 
c

c      implicit double precision (a-h,o-z)
c      double precision multipole,ki,kf
c      common /binomium/binom(100,100)
c get electron momentum

c      ki=dsqrt(2.d0*ei)
c      kf=dsqrt(2.d0*ef)

c calculate the cross section

c      qxs=factor*factor_jk*sum
c      pqxs=factor*factor_jk*sum0

c      return
c      end
c
c ************************************************************************************************************
      SUBROUTINE PASCAL(BINOM,NBIN)
C
C     USES PACAL'S TRIANGLE TO INITIALIZE AN ARRAY OF BINOMAL COEFFICIEN
C
      IMPLICIT double precision (A-H,O-Z)
      DIMENSION BINOM(NBIN,nbin)
      DATA Z1/1.0D0/
      BINOM(1,1) = Z1
      DO 1 I=2,NBIN
      BINOM(I,1) = Z1
      BINOM(I,I) = Z1
      IF (I .LE. 2) GOTO 1
      I1 = I - 1
      DO 2 J=2,I1
      BINOM(I,J) = BINOM(I1,J-1) + BINOM(I1,J)
    2 CONTINUE
    1 CONTINUE
c
      RETURN
      END
c ************************************************************************************************************
      FUNCTION THREEJ(J1,J2,J3,M1,M2,M3,BINOM)
      IMPLICIT double precision (A-H,O-Z)
      DIMENSION BINOM(100,100)
      DATA ZERO,ONE/0.0D0,1.0D0/
c
      THREEJ = ZERO
      IF (M1+M2+M3.ne.0) go to 999
      I1 = -J1+J2+J3+1
      IF (I1.le.0) go to 999
      I2 =  J1-J2+J3+1
      IF (I2.le.0) go to 999
      I3 =  J1+J2-J3+1
      IF (I3.le.0) go to 999
      K1 =  J1+M1+1
      IF (K1.le.0) go to 999
      K2 =  J2+M2+1
      IF (K2.le.0) go to 999
      K3 =  J3+M3+1
      IF (K3.le.0) go to 999
      L1 =  J1-M1+1
      IF (L1.le.0) go to 999
      L2 =  J2-M2+1
      IF (L2.le.0) go to 999
      L3 =  J3-M3+1
      IF (L3.le.0) go to 999
      N1 = -J1-M2+J3
      N2 =  M1-J2+J3
      N3 =  J1-J2+M3
      IMIN = MAX(-N1,-N2,0)+1
      IMAX = MIN(L1,K2,I3)
      IF (IMIN .GT. IMAX) RETURN
      SIGN = ONE
      DO 20 I=IMIN,IMAX
      SIGN = -SIGN
      THREEJ = THREEJ + SIGN*BINOM(I1,N1+I)*BINOM(I2,N2+I)*BINOM(I3,I)
   20 CONTINUE
      THREEJ = THREEJ * DSQRT(BINOM(J2+J2+1,I3)*BINOM(J1+J1+1,I2)
     1       / (BINOM(J1+J2+J3+2,I3)*DBLE(J3+J3+1)
     2       * BINOM(J1+J1+1,L1)*BINOM(J2+J2+1,L2)*BINOM(J3+J3+1,L3)))
      IF (MOD(N3+IMIN,2) .NE. 0) THREEJ = - THREEJ
  999 RETURN
      END
C  (C) Copr. 1986-92 Numerical Recipes Software 'k'1k30m,t+W.
c ************************************************************************************************************
      SUBROUTINE GETDIM(nsym,TFORM,NTSET,LUTMT,maxtg,maxch,maxmi,maxmf,
     1 netot,IPRNT,IFAIL)
C
C     GETDIM reads T-matrix headers to determine maximum dimensions
C
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      PARAMETER (MAXESC=10,maxmax=1000)
      DIMENSION NTSET(nsym),NEREP(MAXESC+1),EINC(2*MAXESC),ICHL(maxmax),
     2 LCHL(maxmax),MCHL(maxmax),ECHL(maxmax),lutmt(nsym)
      INTEGER GUTOT,STOT
      CHARACTER*11 TFORM
      CHARACTER*80 TITLE
      DATA IWRITE/6/
C
 501  FORMAT(/' INVALID VALUE OF NTSET ',I4,' FOR REQUEST NUMBER',I4)
 502  format(/' Symmetry ',i2,' has different number of energies (',
     * i4,') from first symmetry (',i4,')')
 503  FORMAT(/' Maximum dimensions required: NCHAN=',I3,3x,'MAXCHI=',i3,
     13x,'MAXCHF=',i3,3x,'MAXTGT=',i3)
C
      maxtg = 0
      maxch = 0
      maxmi = 0
      maxmf = 0
      netot = 0
      do 1 isym=1,nsym
      IF(NTSET(isym).GT.0) THEN
        CALL READTH(LUTMT(isym),TITLE,NTSET(isym),NCHAN,NVIB,NDIS,
     1  NTARG,MAXCHI,MAXCHF,MGVN,STOT,GUTOT,NE,MAXNE,NEREP,EINC,ICHL,
     2  LCHL,MCHL,ECHL,TFORM,IWRITE,IPRNT-1,IFAIL)
        IF(IFAIL.NE.0) RETURN
        maxtg = max(maxtg,ntarg,nvib)
        maxch = max(maxch,nchan)
        maxmi = max(maxmi,maxchi)
        maxmf = max(maxmf,maxchf)
        if(isym.eq.1) then
          netot = ne
        else
          if(ne.ne.netot) then
            write(iwrite,502) isym,ne,netot
            stop
          endif
        endif
      ELSE
        WRITE(IWRITE,501) NTSET(isym),isym
        IFAIL = 1
        RETURN
      ENDIF
 1    continue
      write(iwrite,503) maxch,maxmi,maxmf,maxtg
c
      RETURN
      END

c*******************************************************************************
c****************************************************************************
c Based hugely on the copyrighted function of Paul Stevenson
c p.stevenson@surrey.ac.uk, where I have changed it into a subroutine.
c details can be found on http://personal.ph.surrey.ac.uk/~phs3ps/cleb.html,
c where the original (very useful) library, anglib.f90 is avaliable for download
      
      subroutine sixj(da,db,dc,dd,de,df,sj)
      implicit none
      integer :: a,b,c,d,e,f
      real(8) :: sj,da,db,dc,dd,de,df
      integer :: nlo, nhi, n
      real(8) :: outfactors, sum, sumterm, angdelta, binom

c slight adjustment on the original to accept half integer inputs, and then
c converts them to integer values of 2x true value in order to run the program itself

      a = INT(2*da) 
      b = INT(2*db)
      c = INT(2*dc)
      d = INT(2*dd)
      e = INT(2*de)
      f = INT(2*df)

c calculates a Wigner 6-j symbol. Argument a-f are integer and are
c twice the true value of the 6-j's arguments, in the form
c { a b c }
c { d e f }
c Calculated using binomial coefficients to allow for (reasonably) high
c arguments.

c First check for consistency of arguments:
      sj=0.0
      if(mod(a+b,2)/=mod(c,2)) return
      if(mod(c+d,2)/=mod(e,2)) return
      if(mod(a+e,2)/=mod(f,2)) return
      if(mod(b+d,2)/=mod(f,2)) return
      if(abs(a-b)>c .or. a+b<c) return
      if(abs(c-d)>e .or. c+d<e) return
      if(abs(a-e)>f .or. a+e<f) return
      if(abs(b-d)>f .or. b+d<f) return

      outfactors = angdelta(a,e,f)/angdelta(a,b,c)
      outfactors = outfactors * angdelta(b,d,f)*angdelta(c,d,e)

      nlo = max( (a+b+c)/2, (c+d+e)/2, (b+d+f)/2, (a+e+f)/2 )
      nhi = min( (a+b+d+e)/2, (b+c+e+f)/2, (a+c+d+f)/2)

      sum=0.0
      do n=nlo,nhi
       sumterm = (-1)**n
       sumterm = sumterm * binom(n+1,n-(a+b+c)/2)
       sumterm = sumterm * binom((a+b-c)/2,n-(c+d+e)/2)
       sumterm = sumterm * binom((a-b+c)/2,n-(b+d+f)/2)
       sumterm = sumterm * binom((b-a+c)/2,n-(a+e+f)/2)
       sum=sum+sumterm
      end do

      sj = sum * outfactors
      
      return
      end

c**************************************************************************
      function angdelta(a,b,c)
      implicit none
      integer :: a,b,c
      real(8)    :: angdelta, scr1, factorial
c calculate the function delta as defined in varshalovich et al. for
c use in 6-j symbol:
      scr1= factorial((a+b-c)/2)
      scr1=scr1/factorial((a+b+c)/2+1)
      scr1=scr1*factorial((a-b+c)/2)
      scr1=scr1*factorial((-a+b+c)/2)
      angdelta=sqrt(scr1)
      end function angdelta

c**************************************************************************
      recursive function factorial(n) result(res)
      implicit none
      integer :: n
      real(8) :: res

      if (n==0 .or. n==1) then
       res=1.0
      else
       res=n*factorial(n-1)
      end if
      end function factorial
 
c**************************************************************************
      recursive function binom(n,r) result(res)
      integer ::  n,r
      real(8) ::  res

      if(n==r .or. r==0) then
       res = 1.0
      else if (r==1) then
       res = real(n,8)
      else
       res = real(n,8)/real(n-r,8)*binom(n-1,r)
      end if
      end function binom     
c**************************************************************************

