! Copyright 2019
!
! For a comprehensive list of the developers that contributed to these codes
! see the UK-AMOR website.
!
! This file is part of UKRmol-out (UKRmol+ suite).
!
!     UKRmol-out is free software: you can redistribute it and/or modify
!     it under the terms of the GNU General Public License as published by
!     the Free Software Foundation, either version 3 of the License, or
!     (at your option) any later version.
!
!     UKRmol-out is distributed in the hope that it will be useful,
!     but WITHOUT ANY WARRANTY; without even the implied warranty of
!     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!     GNU General Public License for more details.
!
!     You should have received a copy of the GNU General Public License
!     along with  UKRmol-out (in source/COPYING). Alternatively, you can also visit
!     <https://www.gnu.org/licenses/>.
!
!> \brief   Output interface from UKRmol+ to RMT
!> \authors Zdeněk Mašín
!> \date    2017
!>
!> This module is used by swinterf and rmt_interface. The purpose is to generate the file molecular_data containing all channel
!> and amplitude data for use in RMT. The target moments and the inner dipoles must be supplied in the DENPROP and CDENPROP output
!> files. The inner dipoles must be produced by CDENPROP_TARGET for all pairs of N+1 symmetries.
!>
module rmt_molecular_interface

   use precisn_gbl
   use utils_gbl, only: xermsg

   implicit none

   !> Integer kind used to compile the RMT codes. This is the kind that will be used to store molecular data.
   integer, parameter :: rmt_int = shortint

   private

   public copy_wamps, copy_channel_data, print_all_data, generate_data_for_rmt, rmt_int, properties_file_obj

   type :: channel_amplitude_data_obj
      !channel + target data from swinterf
      integer :: MGVN = -1, STOT = -1, NCHAN = -1, NTARG = -1, NSTAT0 = -1, ISMAX = -1, NFDM = -1, nelc = -1, nz = -1
      integer, allocatable :: ICHL(:),LCHL(:),MCHL(:),IRRCHL(:),STARG(:),MTARG(:),GTARG(:)
      real(kind=wp), allocatable :: ECHL(:), ETARG(:), FNMC(:), WAMP2(:,:,:), EPOLE(:), R_POINTS(:)
      !dipole data from cdenprop
   contains
      procedure :: get_amplitude_data
      procedure :: get_channel_data
      procedure :: print_data
   end type channel_amplitude_data_obj

   !> Parameters used by properties_file_obj.
   integer, parameter :: n_keys = 8, maxl = 8

   !> Maximum number of N+1 symmetries.
   integer, parameter :: max_symmetries = 8

   type :: properties_file_obj
      integer, private :: n_lines = 0, nstat = 0, lutarg = 0
      character(len=132), allocatable, private :: file_lines(:)
      character(len=1), private :: str_key(n_keys)
      integer, allocatable, private :: key_start_end(:,:), absolute_to_relative(:,:)
      real(kind=wp), allocatable, private :: state_energies(:)
      integer :: sym_index(max_symmetries)
      logical, private :: prop_avail((maxl+1)**2)
      logical, private :: initialized = .false.
   contains
      procedure :: load_file
      procedure :: get_properties
      procedure :: get_single_property
      procedure :: final
   end type properties_file_obj

   !> Actual number of N+1 symmetries.
   integer :: n_np1_symmetries = 0

   !> Channel, target and dipoles data for all N+1 symmetries
   type(channel_amplitude_data_obj) :: channel_amplitude_data(max_symmetries)

contains

   !> \brief   Write the RMT molecular input file
   !> \authors Zdeněk Mašín
   !> \date    2017
   !>
   !> This routine is called from the program generate_data_for_rmt to generate the data required for RMT. It produces the file
   !> molecular_data which will be read-in from RMT. The input parameters are: delta_r,nfdm,lutarg,lunp1. Additionally, channel
   !> data for all symmetries had to been transfered from swinterf using the routines copy_channel_data and copy_wamps.
   !> At the moment it is required that all three components (x,y,z) of the N+1 dipoles are available on the unit lunp1.
   !> The target dipoles for all three components are required too but that is usually not a problem.
   !>
   subroutine generate_data_for_rmt(delta_r,nfdm_inp,lutarg,lunp1)

      real(kind=wp), intent(in) :: delta_r
      integer, intent(in) :: nfdm_inp
      integer, intent(in) :: lunp1, lutarg
      !Generic data:
      integer(kind=rmt_int) :: nelc,nz,lrang2,lamax,ntarg,inast,nchmx,nstmx,lmaxp1,intr,n_rg
      integer(kind=rmt_int) :: nfdm
      real(kind=wp) :: rmatr,bbloch
      real(kind=wp), allocatable :: etarg(:)
      !Symmetry-specific data:
      integer(kind=rmt_int), allocatable :: nconat(:), l2p(:), m2p(:)
      integer(kind=rmt_int) :: lrgl, nspn, npty, nchan, mnp1
      integer(kind=rmt_int), allocatable :: iidip(:), ifdip(:), ltarg(:),starg(:), lm_rg(:,:), ichl(:)
      real(kind=wp), allocatable :: eig(:), wmat(:,:), cf(:,:,:), dipsto(:,:,:), rg(:)

      real(kind=wp), allocatable :: crlv(:,:)

      integer(kind=rmt_int) :: j, k, l, ch_1, ch_2, no_cpl, ind, ii, jj, lu, irri, irrf
      integer(kind=rmt_int) :: lmax
      integer :: i, err, m
      integer, allocatable :: absolute_to_relative(:,:)
      real(kind=wp), allocatable :: dipoles(:,:,:), energies(:)
      type(properties_file_obj) :: properties_file
      integer(kind=rmt_int) :: s, s1, s2
      character(len=*), parameter :: path = 'molecular_data'

         write(*,'(/,10X,"generate_data_for_rmt: start")')
         write(*,'(  10X,"============================",/)')

         open(newunit=lu,file=path,status='replace',form='unformatted',access='stream')

         !Read-in the inner dipoles
         call properties_file%load_file(lunp1)

         do m=-1,1 !-1: y, 0: z, 1: x
            !get property (dipole) with (l,m) = (1,m)
            call properties_file%get_single_property(1,m,dipsto,iidip,ifdip,intr,energies,absolute_to_relative)

            s = size(dipsto,3)
            s1 = size(dipsto,1)
            s2 = size(dipsto,2)
            write(lu) s, s1, s2
            write(lu) iidip, ifdip
            write(lu) dipsto
         enddo !m

         call properties_file%final
         
         ntarg = channel_amplitude_data(1)%ntarg
         allocate(crlv(ntarg,ntarg),stat=err)
         if (err .ne. 0) call xermsg('rmt_molecular_interface','generate_data_for_rmt','Memory allocation 0 failed.',err,1)

         !Read-in the target dipoles
         call properties_file%load_file(lutarg)

         write(lu) ntarg

         do m=-1,1 !-1: y, 0: z, 1: x
            !get property (dipole) with (l,m) = (1,m)
            call properties_file%get_single_property(1,m,dipsto,iidip,ifdip,intr,energies,absolute_to_relative)

            !Transform dipsto into the array crlv which contains a single dipole for EACH pair of target states.
            !Take into account that DENPROP writes only one of <j|D|k> / <k|D|j>.
            crlv = 0.0_wp
            do i=1,intr
               do j=1,ntarg
                  do k=1,ntarg
                     irri = absolute_to_relative(2,j); ii = absolute_to_relative(1,j)  ! symmetry and rel. index of target state j
                     irrf = absolute_to_relative(2,k); jj = absolute_to_relative(1,k)  ! symmetry and rel. index of target state k
                     ! use <j|D|k> if nonzero
                     if (irri == iidip(i) .and. irrf == ifdip(i) .and. dipsto(ii,jj,i) /= 0) then
                        crlv(j,k) = dipsto(ii,jj,i)
                        crlv(k,j) = dipsto(ii,jj,i)
                     end if
                     ! use <k|D|j> if nonzero
                     if (irrf == iidip(i) .and. irri == ifdip(i) .and. dipsto(jj,ii,i) /= 0) then
                        crlv(j,k) = dipsto(jj,ii,i)
                        crlv(k,j) = dipsto(jj,ii,i)
                     end if
                  enddo !k
               enddo !j
            enddo !i

            write(lu) crlv

         enddo !m

         call properties_file%final

         !Generate the coupling coefficients needed in RMT to construct the
         !laser-target and laser-electron asymptotic potentials.
         lmax = 0
         do i=1,n_np1_symmetries
            lmax = max(int(lmax),maxval(channel_amplitude_data(i)%lchl))
         enddo !i
         call generate_couplings(lmax,n_rg,rg,lm_rg)

         write(lu) n_rg
         write(lu) rg,lm_rg

         nfdm = nfdm_inp !Convert default integer type to RMT type

         inast = n_np1_symmetries
         bbloch = 0.0_wp
         lrang2 = 0
         nchmx = 0
         nstmx = 0
         ntarg = channel_amplitude_data(1)%ntarg
         rmatr = maxval(channel_amplitude_data(1)%r_points)
         nelc = channel_amplitude_data(1)%nelc
         nz = channel_amplitude_data(1)%nz
         do i=1,n_np1_symmetries
            if (channel_amplitude_data(i) % nfdm /= nfdm) then
               call xermsg('rmt_molecular_interface', 'generate_data_for_rmt', &
                           'Inconsistency in NFDM between the N+1 symmetries.', i, 1)
            end if
            lrang2 = max(int(lrang2),maxval(channel_amplitude_data(i)%lchl(:)))
            nchmx = max(int(nchmx),channel_amplitude_data(i)%nchan)
            nstmx = max(int(nstmx),channel_amplitude_data(i)%nstat0)
            if (ntarg /= channel_amplitude_data(i) % ntarg) then
               call xermsg('rmt_molecular_interface', 'generate_data_for_rmt', &
                           'Inconsistency in the number of target states between the N+1 symmetries.', 1, 1)
            end if
            if (rmatr /= maxval(channel_amplitude_data(1) % r_points)) then
               call xermsg('rmt_molecular_interface', 'generate_data_for_rmt', &
                           'Inconsistency in the R-matrix radius between the N+1 symmetries.', 2, 1)
            end if
            if (nelc /= channel_amplitude_data(i) % nelc) then
               call xermsg('rmt_molecular_interface', 'generate_data_for_rmt', &
                           'Inconsistency in the number of electrons between the N+1 symmetries.', 3, 1)
            end if
            if (nz /= channel_amplitude_data(i) % nz) then
               call xermsg('rmt_molecular_interface', 'generate_data_for_rmt', &
                           'Inconsistency in the residual charge between the N+1 symmetries.', 4, 1)
            end if
         enddo !i
         lrang2 = lrang2 + 1
         lmaxp1 = lrang2

         lamax = maxval(channel_amplitude_data(1:n_np1_symmetries)%ismax)

         allocate(etarg(ntarg),ltarg(ntarg),starg(ntarg),stat=err)
         if (err .ne. 0) call xermsg('rmt_molecular_interface','generate_data_for_rmt','Memory allocation 1 failed.',err,1)
  
         etarg(1:ntarg) = channel_amplitude_data(1)%etarg(1:ntarg)
         starg(1:ntarg) = channel_amplitude_data(1)%starg(1:ntarg)
         !IRR of the states in the range 1 to 8:
         ltarg(1:ntarg) = channel_amplitude_data(1)%mtarg(1:ntarg)+1

         write(lu) nelc,nz,lrang2,lamax,ntarg,inast,nchmx,nstmx,lmaxp1
         write(lu) rmatr,bbloch
         write(lu) etarg,ltarg,starg

         write(lu) nfdm,delta_r
         write(lu) channel_amplitude_data(1)%r_points(1:nfdm+1)

         allocate(nconat(ntarg), l2p(nchmx), m2p(nchmx), eig(nstmx), wmat(nchmx,nstmx), &
                  cf(nchmx,nchmx,lamax), ichl(nchmx), stat=err)
         if (err .ne. 0) call xermsg('rmt_molecular_interface','generate_data_for_rmt','Memory allocation 2 failed.',err,1)

         do i=1,inast
            ! initialize:
            l2p = -1
            m2p = -1
            lrgl = -1
            nspn = 0
            npty = 0
            nchan = 0
            mnp1 = 0
            nconat = 0
            eig = 0.0_wp
            wmat = 0.0_wp
            cf = 0.0_wp
            ichl = 0
            !
            lrgl = channel_amplitude_data(i)%mgvn+1
            nspn = channel_amplitude_data(i)%stot
            npty = 0
            nchan = channel_amplitude_data(i)%nchan
            mnp1 = channel_amplitude_data(i)%nstat0

            do j=1,ntarg
               nconat(j) = 0
               do k=1,channel_amplitude_data(i)%nchan
                  if (channel_amplitude_data(i)%ichl(k) .eq. j) nconat(j) = nconat(j)+1
               enddo !k
            enddo !j
            l2p(1:nchan) = channel_amplitude_data(i)%lchl(1:nchan)
            m2p(1:nchan) = channel_amplitude_data(i)%mchl(1:nchan)
            ichl(1:nchan) = channel_amplitude_data(i)%ichl(1:nchan)

            !Unpack the triangular coupling potential matrix
            no_cpl = nchan*(nchan+1)/2
            do ch_1=1,nchan
               do ch_2=1,nchan
                  do l=1,channel_amplitude_data(i)%ismax
                     j = max(ch_1,ch_2)
                     k = min(ch_1,ch_2)
                     ind = j*(j-1)/2+k + (l-1)*no_cpl
                     cf(ch_1,ch_2,l) = channel_amplitude_data(i)%fnmc(ind)
                  enddo !l
               enddo !ch_2
            enddo !ch_1

            eig(1:mnp1) = channel_amplitude_data(i)%epole(1:mnp1)

            !wmat should contain the amplitudes on the R-matrix sphere
            !The R-matrix amplitudes are the last ones: (nfdm+1)
            wmat(1:nchan,1:mnp1) = channel_amplitude_data(i)%wamp2(1:nchan,1:mnp1,nfdm+1)

            write(lu) lrgl, nspn, npty, nchan, mnp1, nconat, l2p, m2p
            write(lu) eig, wmat, cf
            write(lu) ichl

            s1 = size(channel_amplitude_data(i)%wamp2,1)
            s2 = size(channel_amplitude_data(i)%wamp2,2)
            write(lu) s1, s2
            write(lu) channel_amplitude_data(i)%wamp2(1:s1,1:s2,1:nfdm)
         enddo !i

         write(*,'(/,10X,"generate_data_for_rmt: finished")')

   end subroutine generate_data_for_rmt

   subroutine generate_couplings(maxl,n_rg,rg,lm_rg)
      use coupling_obj_gbl

      integer(kind=rmt_int), intent(in) :: maxl
      integer(kind=rmt_int), intent(out) :: n_rg
      integer(kind=rmt_int), allocatable :: lm_rg(:,:)
      real(kind=wp), allocatable :: rg(:)

      type(couplings_type) :: cpl
      integer :: l1,l2,l3,m1,m2,m3,err
      real(kind=wp) :: tmp

         call cpl%prec_cgaunt(int(maxl))

         l3 = 1

         !How many non-zero dipole couplings there are
         n_rg = 0
         do l1=0,maxl
            do m1=-l1,l1
               do l2=0,maxl
                  do m2=-l2,l2
                     do m3=-l3,l3
                        tmp = cpl%rgaunt(l1,l2,l3,m1,m2,m3)
                        if (tmp .ne. 0.0_wp) n_rg = n_rg+1
                     enddo !m3
                  enddo !m2
               enddo !l2
            enddo !m1
         enddo !l1

         if (allocated(rg)) deallocate(rg)
         if (allocated(lm_rg)) deallocate(lm_rg)
         allocate(rg(n_rg),lm_rg(6,n_rg),stat=err)
         rg = 0.0_wp
         lm_rg = -2

         !Save them
         n_rg = 0
         do l1=0,maxl
            do m1=-l1,l1
               do l2=0,maxl
                  do m2=-l2,l2
                     do m3=-l3,l3
                        tmp = cpl%rgaunt(l1,l2,l3,m1,m2,m3)
                        if (tmp .ne. 0.0_wp) then
                           n_rg = n_rg+1
                           rg(n_rg) = tmp
                           lm_rg(1:6,n_rg) = (/l1,m1,l2,m2,l3,m3/)
                           !write(*,'(e25.15,6(i2,1X))') tmp,l1,m1,l2,m2,l3,m3
                        endif
                     enddo !m3
                  enddo !m2
               enddo !l2
            enddo !m1
         enddo !l1

   end subroutine generate_couplings

   function matching_symmetry(mgvn,stot)

      integer, intent(in) :: mgvn,stot
      integer :: matching_symmetry

      integer :: i

         !Find the data structure for the present symmetry mgvn,stot
         matching_symmetry = -1
         do i=1,n_np1_symmetries
            if (channel_amplitude_data(i)%mgvn .eq. mgvn .and. channel_amplitude_data(i)%stot .eq. stot) then
               matching_symmetry = i
               !todo transform this lame check into a rule that n-th symmetry is
               !in the n-th channel_amplitude_data.
               if (i /= mgvn+1) then
                  call xermsg('rmt_molecular_interface', 'matching_symmetry', &
                              'The symmetries on input must be ordered according to their symmetry value mgvn+1.', 1, 1)
               end if
               exit
            endif
         enddo

   end function matching_symmetry

   subroutine copy_wamps(wamp2,mgvn,stot,nchan,nstat0,i_radial_pt,nfdm,radius)

      integer, intent(in) :: mgvn,stot,nchan,nstat0,i_radial_pt,nfdm
      real(kind=wp), intent(in) :: wamp2(nchan,nstat0), radius

      integer :: i, i_match

         !Find the data structure for the present symmetry mgvn,stot
         i_match = matching_symmetry(mgvn,stot)

         if (i_match .eq. -1) then !Data for a new symmetry
            n_np1_symmetries = n_np1_symmetries + 1
            if (n_np1_symmetries > max_symmetries) then
               call xermsg('rmt_molecular_interface', 'copy_wamps', &
                           'The value of max_symmetries is too small: increase it and recompile.', 1, 1)
            end if
            i_match = n_np1_symmetries
         endif

         write(*,'(/,10X,"Copying amplitudes for symmetry (MGVN,STOT): (",i3,",",i3,") and radial point: ",e25.15)') &
            mgvn, stot, radius

         call channel_amplitude_data(i_match)%get_amplitude_data(wamp2,mgvn,stot,nchan,nstat0,i_radial_pt,nfdm,radius)

         write(*,'(10X,"Saved into data set with sequence number: ",i4,/)') i_match

   end subroutine copy_wamps

   !> \brief   Evaluate wave-function amplitudes
   !> \authors Zdeněk Mašín
   !> \date    2017
   !>
   !> IMPORTANT: The amplitudes generated by SWINTERF contain the factor 1/sqrt(2) so that the expression for the R-matrix reads
   !> \f[
   !>     a R_{ij} = \sum_k w_{ik} w_{jk} \,.
   !> \f]
   !> This means that the factor 1/2 is absorbed into the amplitudes. However, this convention is probably not followedin RMT,
   !> so we get rid of the 1/sqrt(2) factor here.
   !>
   subroutine get_amplitude_data(this,wamp2,mgvn,stot,nchan,nstat0,i_radial_pt,nfdm,radius)

      class(channel_amplitude_data_obj) :: this
      integer, intent(in) :: mgvn,stot,nchan,nstat0,i_radial_pt,nfdm
      real(kind=wp), intent(in) :: wamp2(nchan,nstat0), radius

      integer :: error

         this%mgvn = mgvn
         this%stot = stot

         if (allocated(this%wamp2)) then
            error = 0
            if (this%nchan .ne. nchan) error = 1
            if (this%nstat0 .ne. nstat0) error = 2
            if (this%nfdm .ne. nfdm) error = 3
            if (error /= 0) then
               call xermsg('rmt_molecular_interface', 'copy_wamps', 'Inconsistency in input nchan,nstat0,nfdm values.', error, 1)
            end if
         else
            if (allocated(this%r_points)) deallocate(this%r_points)
            allocate(this%wamp2(nchan,nstat0,nfdm+1),this%r_points(nfdm+1),stat=error)
            if (error .ne. 0) call xermsg('rmt_molecular_interface','copy_wamps','Memory allocation failed.',error,1)
            this%nchan = nchan
            this%nstat0 = nstat0
            this%nfdm = nfdm
         endif

         this%r_points(i_radial_pt) = radius
         this%wamp2(1:nchan,1:nstat0,i_radial_pt) = wamp2(1:nchan,1:nstat0)*sqrt(2.0_wp)

   end subroutine get_amplitude_data

   subroutine copy_channel_data(neltot, nz, MGVN, STOT, NCHAN, NSTAT0, ICHL, LCHL, MCHL, ECHL, IRRCHL, NTARG, STARG, &
                                MTARG, GTARG, ETARG, EPOLE, FNMC, ISMAX)

      integer, intent(in) :: neltot, nz, MGVN, STOT, NCHAN, NSTAT0, ICHL(NCHAN), LCHL(NCHAN), MCHL(NCHAN), IRRCHL(NCHAN), &
                             NTARG, STARG(NCHAN), MTARG(NCHAN), GTARG(NCHAN), ISMAX
      real(kind=wp), intent(in) :: ECHL(NCHAN),ETARG(NCHAN),EPOLE(NSTAT0),FNMC(:)
      integer :: i_match

         i_match = matching_symmetry(mgvn,stot)

         if (i_match .eq. -1) then !Data for a new symmetry
            n_np1_symmetries = n_np1_symmetries + 1
            if (n_np1_symmetries > max_symmetries) then
               call xermsg('rmt_molecular_interface', 'copy_channel_data', &
                           'The value of max_symmetries is too small: increase it and recompile.', 1, 1)
            end if
            i_match = n_np1_symmetries
         endif

         write(*,'(/,10X,"Copying channel data for symmetry (MGVN,STOT): (",i3,",",i3,")")') mgvn, stot

         call channel_amplitude_data(i_match) % get_channel_data(neltot, nz, MGVN, STOT, NCHAN, NSTAT0, ICHL, LCHL, MCHL, ECHL, &
                                                                 IRRCHL, NTARG, STARG, MTARG, GTARG, ETARG, EPOLE, FNMC, ISMAX)

         write(*,'(10X,"Saved into data set with sequence number: ",i4,/)') i_match

   end subroutine copy_channel_data

   subroutine get_channel_data(this, neltot, nz, MGVN, STOT, NCHAN, NSTAT0, ICHL, LCHL, MCHL, ECHL, IRRCHL, NTARG, STARG, MTARG, &
                               GTARG, ETARG, EPOLE, FNMC, ISMAX)

      class(channel_amplitude_data_obj) :: this
      integer, intent(in) :: neltot, nz, MGVN, STOT, NCHAN, NSTAT0, ICHL(NCHAN), LCHL(NCHAN), MCHL(NCHAN), IRRCHL(NCHAN), &
                             NTARG, STARG(NTARG), MTARG(NTARG), GTARG(NTARG), ISMAX
      real(kind=wp), intent(in) :: ECHL(NCHAN), ETARG(NTARG), EPOLE(NSTAT0), FNMC(:)

      integer :: err, n

         this%mgvn = mgvn
         this%stot = stot
         this%nchan = nchan
         this%nstat0 = nstat0
         this%ntarg = ntarg
         this%ismax = ismax
         this%nelc = neltot-1
         this%nz = nz

         if (allocated(this%ichl)) deallocate(this%ichl)
         if (allocated(this%lchl)) deallocate(this%lchl)
         if (allocated(this%mchl)) deallocate(this%mchl)
         if (allocated(this%irrchl)) deallocate(this%irrchl)
         if (allocated(this%starg)) deallocate(this%starg)
         if (allocated(this%mtarg)) deallocate(this%mtarg)
         if (allocated(this%gtarg)) deallocate(this%gtarg)
         if (allocated(this%echl)) deallocate(this%echl)
         if (allocated(this%etarg)) deallocate(this%etarg)
         if (allocated(this%epole)) deallocate(this%epole)
         if (allocated(this%fnmc)) deallocate(this%fnmc)

         n = size(fnmc)
         allocate(this % ichl(nchan), this % lchl(nchan), this % mchl(nchan), this % irrchl(nchan), this % starg(ntarg), &
                  this % mtarg(ntarg), this % gtarg(ntarg), this % echl(nchan), this % etarg(ntarg), this % epole(nstat0), &
                  this % fnmc(n), stat = err)

         this%ichl(1:nchan) = ichl(1:nchan)
         this%lchl(1:nchan) = lchl(1:nchan)
         this%mchl(1:nchan) = mchl(1:nchan)
         this%irrchl(1:nchan) = irrchl(1:nchan)
         this%starg(1:ntarg) = starg(1:ntarg)
         this%mtarg(1:ntarg) = mtarg(1:ntarg)
         this%gtarg(1:ntarg) = gtarg(1:ntarg)
         this%echl(1:nchan) = echl(1:nchan)
         this%etarg(1:ntarg) = etarg(1:ntarg)
         this%epole(1:nstat0) = epole(1:nstat0)
         this%fnmc(1:n) = fnmc(1:n)

   end subroutine get_channel_data

   subroutine print_data(this)

      class(channel_amplitude_data_obj) :: this

      integer :: i, ind

         write(*,'(/,10X,"Channel data for symmetry (MGVN,STOT): (",i3,",",i3,")")') this%mgvn, this%stot
         write(*,'(10X,"Number of electrons in the target: ",i10)') this%nelc
         write(*,'(10X,"Total nuclear charge: ",i10)') this%nz
         write(*,'(10X,"Number of channels: ",i10)') this%nchan
         write(*,'(10X,"Number of target states: ",i10)') this%ntarg
         write(*,'(10X,"Number of N+1 states: ",i10)') this%nstat0
         write(*,'(10X,"Largest lambda in e-mol potential: ",i10)') this%ismax
         write(*,'(10X,"Number of FD points: ",i10)') this%nfdm

         write(*,'(10X,"Points for which amplitudes have been evaluated:")') 
         do i=1,this%nfdm+1
            write(*,'(10X,i4,e25.15)') i, this%r_points(i)
         enddo
         write(*,'(/)')

         do i=1,this%nchan
            ind = this%ichl(i)
            write(*,'(4i4,e25.15,3i4,e25.15)') &
                this % ichl(i), this % lchl(i), this % mchl(i), this % irrchl(i), this % etarg(ind), &
                this % starg(ind), this % mtarg(ind), this % gtarg(ind), this % echl(i)
         enddo !i

   end subroutine print_data

   subroutine print_all_data

      integer :: i

         do i=1,n_np1_symmetries
            call channel_amplitude_data(i)%print_data
         enddo !i

   end subroutine print_all_data

   subroutine load_file(this,lutarg)

      class(properties_file_obj) :: this
      integer, intent(in) :: lutarg

      logical :: op
      integer :: err, i, j, irr, state_no, inx(8)
      real(kind=wp) :: dnx
      character(len=26) :: head
      character(len=132) :: one_line

        write(*,'(/,10X,"load_file")')
        write(*,'(  10X,"====================")')

        write(*,'(/,5X,"Input unit: ",i10)') lutarg

        this%lutarg = lutarg

        ! open the file (if not already connected)
        inquire (unit = this % lutarg, opened = op)
        if (.not. op) then
           open(unit = this % lutarg, form = 'FORMATTED', action = 'read', iostat = err, iomsg = one_line)
           if (err /= 0) then
              call xermsg('rmt_molecular_interface', 'load_file', &
                          'Error opening the properties file: ' // one_line, 2, 1)
           end if
        end if

        !Count the number of lines of the properties file:
        rewind this%lutarg
        this%n_lines = 0
        do
           read(this%lutarg,'(a)',end=100) one_line
           this%n_lines = this%n_lines + 1
        enddo
    100 rewind this%lutarg

        write(*,'(/,5X,"Number of lines on the input file: ",i15)') this%n_lines

        if (allocated(this%file_lines)) deallocate(this%file_lines)
        if (allocated(this%key_start_end)) deallocate(this%key_start_end)
        allocate(this%file_lines(this%n_lines),this%key_start_end(2,n_keys),stat=err)
        if (err .ne. 0) call xermsg('rmt_molecular_interface','load_file','Memory allocation 1 failed.',err,1)

        do j=1,n_keys
           write(this%str_key(j),'(i1)') j
        enddo

        !Read the whole file into the array setting the ranges of the key positions
        this%key_start_end = 0
        do i=1,this%n_lines
           read(this%lutarg,'(a)') one_line
           this%file_lines(i) = one_line

           do j=1,n_keys
              if (one_line(1:1) .eq. this%str_key(j)) then
                 if (this%key_start_end(1,j) .eq. 0) this%key_start_end(1,j) = i
                 this%key_start_end(2,j) = i
              endif
           enddo
        enddo !i

        close(this%lutarg)

        write(*,'(/,5X,"Keys found on the input unit and their line number ranges:")')
        do j=1,n_keys
           if (this % key_start_end(1,j) > 0) then
             write(*, '(5X,"Key ",i2," start and end: ",2i15)') j, this % key_start_end(1:2,j)
           end if
        enddo

        if (this % key_start_end(1,5) == 0) then
           call xermsg('rmt_molecular_interface', 'load_file', 'Energy data not present on the properties file.', 1, 1)
        end if

        if (this % key_start_end(1,1) == 0) then
           call xermsg('rmt_molecular_interface', 'load_file', 'Property data not present on the properties file.', 2, 1)
        end if

        this%nstat = this%key_start_end(2,5)-this%key_start_end(1,5)+1
        write(*,'(/,5X,"Total number of states: ",i15)') this%nstat

        !Read the energy data (key = 5) for the states and map them to their order within each IRR
        if (allocated(this%absolute_to_relative)) deallocate(this%absolute_to_relative)
        if (allocated(this%state_energies)) deallocate(this%state_energies)
        allocate(this%absolute_to_relative(2,this%nstat),this%state_energies(this%nstat),stat=err)
        if (err .ne. 0) call xermsg('rmt_molecular_interface','load_file','Memory allocation 2 failed.',err,1)

        this%absolute_to_relative = 0
        this%sym_index = 0
        do i=this%key_start_end(1,5),this%key_start_end(2,5)
           one_line = this%file_lines(i)
           read(one_line,*) (inx(j),j=1,8),dnx,head
           if (inx(1) /= 5) then
              call xermsg('rmt_molecular_interface', 'load_file', 'Unexpected key number: error parsing the properties file.', 3, 1)
           end if
           irr = inx(5)+1
           state_no = inx(2)
           this%sym_index(irr) = this%sym_index(irr) + 1
           this%absolute_to_relative(1,state_no) = this%sym_index(irr)
           this%absolute_to_relative(2,state_no) = irr
           this%state_energies(state_no) = dnx
        enddo !i

        write(*,'(/,5X,"Number of states per symmetry: ",8(i10,","))') (this%sym_index(irr),irr=1,max_symmetries)

        this%initialized = .true.

        write(*,'(/,10X,"load_file: finished")')

   end subroutine load_file

   subroutine final(this)

      class(properties_file_obj) :: this

        write(*,'(/,10X,"final")')
        write(*,'(  10X,"====================")')

        if (allocated(this%file_lines)) deallocate(this%file_lines)
        if (allocated(this%key_start_end)) deallocate(this%key_start_end)
        this%initialized = .false. 

        write(*,'(/,10X,"final: finished")')

   end subroutine final

   !> All properties with L <= lmaxprop will be read-in.
   subroutine get_properties(this,lmaxprop,prop,state_energies,absolute_to_relative)

      class(properties_file_obj) :: this
      integer, intent(in) :: lmaxprop
      real(kind=wp), allocatable :: prop(:,:,:), state_energies(:)
      integer, allocatable :: absolute_to_relative(:,:)

      character(len=132) :: one_line
      character(len=1) :: str_X
      integer :: err, i, j, inx(8), it1, it2, lop, mop, iq, block, state_no, irr, n_prop, m, l
      real(kind=wp) :: dnx
      logical :: prop_avail((maxl+1)**2)

        write(*,'(/,10X,"get_properties")')
        write(*,'(  10X,"====================")')

        write(*,'(  5X,"Properties to read: all properties with L <= ",i5)') lmaxprop

        if (lmaxprop < 0) call xermsg('rmt_molecular_interface','get_properties','On input lmaxprop < 0.',1,1)

        if (.not. this % initialized) then
           call xermsg('rmt_molecular_interface', 'get_properties', &
                       'The properties file has not been loaded into memory: call this%load_file first.', 2, 1)
        end if

        if (allocated(absolute_to_relative)) deallocate(absolute_to_relative)
        if (allocated(state_energies)) deallocate(state_energies)
        if (allocated(prop)) deallocate(prop)

        allocate(absolute_to_relative,source=this%absolute_to_relative,stat=err)
        if (err .ne. 0) call xermsg('rmt_molecular_interface','get_properties','Memory allocation 2 failed.',err,1)
        allocate(state_energies(this%nstat),source=this%state_energies,stat=err)
        if (err .ne. 0) call xermsg('rmt_molecular_interface','get_properties','Memory allocation 3 failed.',err,1)

        !Read the property data (key = 1)
        iq = (lmaxprop+1)**2
        allocate(prop(this%nstat,this%nstat,iq),stat=err)
        if (err .ne. 0) call xermsg('rmt_molecular_interface','get_properties','Memory allocation 4 failed.',err,1)

        n_prop = 0
        prop = 0.0_wp
        prop_avail = .false.
        do i=this%key_start_end(1,1),this%key_start_end(2,1)
           one_line = this%file_lines(i)
           read(one_line,*) (inx(j),j=1,8),dnx
           it1=inx(2)
           it2=inx(4)
           lop=inx(7)
           mop=inx(8)
           if (lop .le. maxl) then
              iq = lop*lop+lop+mop+1
              prop_avail(iq) = .true.
           else
              write(*,'("Maximum L of available property exceeds the size of the prop_avail array.")')
           endif
           if (lop < 1 .or. lop > lmaxprop) cycle
           iq = lop*lop+lop+mop+1
           prop(it1,it2,iq) = dnx
           prop(it2,it1,iq) = prop(it1,it2,iq)
           n_prop = n_prop+1
        enddo !i

        write(*,'(/,5X,"Properties (m,l) available on the file: R marks those that have been read-in")')
        i = 0
        do l=0,maxl
           str_X = ''
           if (l .le. lmaxprop) str_X = 'R'
           do m=-l,l
              i = i + 1
              if (prop_avail(i)) write(*,'(5X,i2,1X,i2,1X,a2)') m,l,str_X
           enddo
        enddo 

        write(*,'(/,5X,"Total number of properties that have been read-in: ",i15)') n_prop
        write(*,'(/,10X,"get_properties: finished")')

   end subroutine get_properties

   !> \brief   Retrieve a single property from the line store
   !> \authors Zdeněk Mašín
   !> \date    2017
   !>
   !> A single property with the given L,M will be read from the line storage (read from a DENPROP or CDENPROP properties file).
   !> The output data have the structure required by RMT.
   !>
   subroutine get_single_property(this,lp,mp,dipsto,iidip,ifdip,intr,state_energies,absolute_to_relative)

      class(properties_file_obj) :: this
      integer, intent(in) :: lp, mp
      integer(kind=rmt_int), intent(out) :: intr
      real(kind=wp), allocatable :: dipsto(:,:,:), state_energies(:)
      integer, allocatable :: absolute_to_relative(:,:)
      integer(kind=rmt_int), allocatable :: iidip(:), ifdip(:)

      character(len=132) :: one_line
      character(len=1) :: str_X
      integer :: err, i, j, inx(8), it1, it2, lop, mop, irr1, irr2, block, state_no, irr, n_prop, m, l, mxstat
      real(kind=wp) :: dnx
      integer :: irr_map(max_symmetries,max_symmetries)

        write(*,'(/,10X,"get_single_property")')
        write(*,'(  10X,"====================")')

        write(*,'(  5X,"Properties to read: a single property with L,M: ",2i5)') lp,mp

        if (lp < 0 .or. abs(mp) > lp) then
           call xermsg('rmt_molecular_interface', 'get_single_property', 'On input lp,mp were out of range.', 1, 1)
        end if

        if (.not. this % initialized) then
           call xermsg('rmt_molecular_interface', 'get_single_property', &
                       'The properties file has not been loaded into memory: call this%load_file first.', 2, 1)
        end if

        if (allocated(absolute_to_relative)) deallocate(absolute_to_relative)
        if (allocated(state_energies)) deallocate(state_energies)
        allocate(absolute_to_relative,source=this%absolute_to_relative,stat=err)
        if (err .ne. 0) call xermsg('rmt_molecular_interface','get_single_property','Memory allocation 2 failed.',err,1)
        allocate(state_energies,source=this%state_energies,stat=err)
        if (err .ne. 0) call xermsg('rmt_molecular_interface','get_single_property','Memory allocation 3 failed.',err,1)

        mxstat = maxval(this%sym_index)
        if (mxstat .le. 0) call xermsg('rmt_molecular_interface','get_single_property','There are no states on the file!',4,1)

        !Read the property data (key = 1): determine the number of different symmetry blocks
        irr_map = 0
        block = 0
        do i=this%key_start_end(1,1),this%key_start_end(2,1)
           one_line = this%file_lines(i)
           read(one_line,*) (inx(j),j=1,8),dnx
           it1=inx(2)
           it2=inx(4)
           lop=inx(7)
           mop=inx(8)
           if (lop .ne. lp .or. mop .ne. mp) cycle
           irr1 = inx(3)+1
           irr2 = inx(5)+1

           if (absolute_to_relative(2,it1) .ne. irr1 .or. absolute_to_relative(2,it2) .ne. irr2) then
              print *,it1,irr1,absolute_to_relative(2,it1)
              print *,it2,irr2,absolute_to_relative(2,it2)
              call xermsg('rmt_molecular_interface', 'get_single_property', &
                          'State data and property data are not compatible...corrupted property file?', 5, 1)
           endif

           if (irr_map(irr1,irr2) .ne. 0) cycle

           block = block + 1
           irr_map(irr1,irr2) = block
        enddo !i

        intr = block

        if (block .eq. 0) write(*,'(5X,"The selected property is zero for all pairs of states.")')
        i = max(block,1)

        if (allocated(dipsto)) deallocate(dipsto)
        if (allocated(iidip)) deallocate(iidip)
        if (allocated(ifdip)) deallocate(ifdip)
        allocate(dipsto(mxstat,mxstat,i),iidip(i),ifdip(i),stat=err)
        if (err .ne. 0) call xermsg('rmt_molecular_interface','get_single_property','Memory allocation 3 failed.',err,1)

        !Read the property data (key = 1): save the dipoles into the symmetry blocks
        n_prop = 0
        dipsto = 0.0_wp
        iidip = -1
        ifdip = -1
        do i=this%key_start_end(1,1),this%key_start_end(2,1)
           one_line = this%file_lines(i)
           read(one_line,*) (inx(j),j=1,8),dnx
           it1=inx(2)
           it2=inx(4)
           lop=inx(7)
           mop=inx(8)
           if (lop .ne. lp .or. mop .ne. mp) cycle
           irr1 = inx(3)+1
           irr2 = inx(5)+1

           block = irr_map(irr1,irr2)
           iidip(block) = irr1
           ifdip(block) = irr2

           it1 = absolute_to_relative(1,it1)
           it2 = absolute_to_relative(1,it2)

           dipsto(it1,it2,block) = dnx
           if (irr1 .eq. irr2) dipsto(it2,it1,block) = dipsto(it1,it2,block)
           !write(*,'(2i7,i2,e25.15)') it1,it2,block,dnx

           n_prop = n_prop+1
        enddo !i

        write(*,'(/,5X,"Number of different symmetry blocks:",i2)') intr
        write(*,'(/,5X,"IRRs of the wavefunctions in the symmetry blocks:")')
        do block=1,intr
           write(*,'(5X,i2,": ",i1,1X,i1)') block,iidip(block),ifdip(block)
        enddo

        write(*,'(/,5X,"Total number of properties that have been read-in: ",i15)') n_prop
        write(*,'(/,10X,"get_single_property: finished")')

   end subroutine get_single_property

end module rmt_molecular_interface
