! Copyright 2019
!
! For a comprehensive list of the developers that contributed to these codes
! see the UK-AMOR website.
!
! This file is part of UKRmol-out (UKRmol+ suite).
!
!     UKRmol-out is free software: you can redistribute it and/or modify
!     it under the terms of the GNU General Public License as published by
!     the Free Software Foundation, either version 3 of the License, or
!     (at your option) any later version.
!
!     UKRmol-out is distributed in the hope that it will be useful,
!     but WITHOUT ANY WARRANTY; without even the implied warranty of
!     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!     GNU General Public License for more details.
!
!     You should have received a copy of the GNU General Public License
!     along with  UKRmol-out (in source/COPYING). Alternatively, you can also visit
!     <https://www.gnu.org/licenses/>.
!
      SUBROUTINE READRH(LURMT,NRSET,RFORM,MGVN,STOT,GUTOT,NCHAN,NVIB,
     1      NDIS,NTARG,ION,RR,RMASS,RMATR,IBUT,ISMAX,nstat,NOCSF,NPOLE,
     2      ezero,iex,IWRIT0,IPRNT0,IFAIL)
C
C***********************************************************************
C
C     READRH reads data required to assemble R-matrices from set number
C     NRSET on unit LURMT
C
C     The entry point READRH reads the header record to obtain dimension
C     information.  This enables the calling routine to dynamically
C     allocate space for the arrays which are subsequently read by
C     calling READRM
C
C     Input
C      LURMT  = Logical unit holding R-matrix data
C      NRSET  = Set number
C      RFORM  = 'FORMATTED' or 'UNFORMATTED'
C      RR     = internuclear distance
C      IWRITE = logical unit for printed output
C      IPRNT  = print switch
C
C     Output
C      MGVN   = Overall M symmetry of system
C      STOT   = Spin multiplicity
C      GUTOT  = g/u symmetry
C      NTARG  = number of target electronic configurations
C      NVIB   = number of target vibrational levels
C      NDIS   = number of dissociating states
C      NCHAN  = total number of scattering channels
C      ION    = residual charge on system
C      RMASS  = reduced mass of system
C      RMATR  = R-matrix boundary radius
C      IBUT   = Buttle correction flag (0=no Buttle)
C      ISMAX  = maximum multipole in expansion of asymptotic potential
C      nstat  = number of R-matrix poles passed
C      NOCSF  = dimension of Hamiltonian matrix
C      NSTAT  = number of eigenvalues/vectors passed
C      NPOLE  = number of CI vectors saved
C      ezero  = provided by scatci, averaged energy of the poles omitted
C      IEX    = extra states used correction of partitioned R-matrix
C      IFAIL  = success/failure flag
C
C***********************************************************************
C
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      CHARACTER(LEN=11) RFORM
      CHARACTER(LEN=80) HEADER
      DIMENSION CF(NCHAN*(NCHAN+1)/2,*),EIG(nstat),WAMP(NCHAN,nstat),
     1BCOEF(3,NCHAN),VEC(NOCSF,*),sfac(nchan),ecex(iex),rcex(nchan,iex)
      INTEGER STOT,GUTOT
      SAVE
      DATA KEYRM/11/,TINY/1.D-8/
C
C---- FIND REQUIRED SET
      IPRNT = IPRNT0
      IWRITE = IWRIT0
      NSET = NRSET
      CALL GETSET(LURMT,NSET,KEYRM,RFORM,IFAIL)
      IF(IFAIL.NE.0) GO TO 99
C
      IF(RFORM.EQ.'FORMATTED') THEN
        READ(LURMT,100) KEYRM,NSET,NREC,NINFO,NDATA
        READ(LURMT,11) HEADER
        READ(LURMT,10) NTARG,NVIB,NDIS,NCHAN
        READ(LURMT,12) MGVN,STOT,GUTOT,ION,R,RMASS
        READ(LURMT,12) ISMAX,nstat,NPOLE,IBUT,RMATR
        if (abs(ibut).gt.1) read(lurmt,33) nocsf,ezero,iex
      ELSE
        READ(LURMT) KEYRM,NSET,NREC,NINFO,NDATA
        READ(LURMT) HEADER
        READ(LURMT) NTARG,NVIB,NDIS,NCHAN
        READ(LURMT) MGVN,STOT,GUTOT,ION,R,RMASS
        READ(LURMT) ISMAX,nstat,NPOLE,IBUT,RMATR
        if (abs(ibut).gt.1) read(lurmt) nocsf,ezero,iex
      ENDIF
      if (abs(ibut).le.1) then
         nocsf=nstat
         iex=0
      endif
C
C---- CHECK GEOMETRY DATA
      IF(ABS(RR).GT.TINY.AND.ABS(RR-R).GT.TINY) THEN
        WRITE(IWRITE,96) RR,NSET,R
        IFAIL = 1
      ELSE
        WRITE(IWRITE,16) NSET,LURMT,HEADER
        IF(IPRNT.GE.0) WRITE(IWRITE,13)NOCSF,NPOLE,ISMAX,RMATR
        RR = R
      ENDIF
      IF(IPRNT.GT.0) THEN
        WRITE(IWRITE,17)
        WRITE(IWRITE,110) KEYRM,NSET,NREC,NINFO,NDATA
        WRITE(IWRITE,111) HEADER
        WRITE(IWRITE,110) NTARG,NVIB,NDIS,NCHAN
        WRITE(IWRITE,112) MGVN,STOT,GUTOT,ION,R,RMASS
        WRITE(IWRITE,112) ISMAX,nstat,NPOLE,IBUT,RMATR
        if (abs(ibut).gt.1) write(iwrite,133) nocsf,ezero,iex
      ENDIF
C
      RETURN
C
      ENTRY READRM(LURMT,RFORM,NCHAN,nstat,NOCSF,ISMX,ISM,NPOLE,NKEEP,
     1             IBUT,CF,EIG,WAMP,VEC,BCOEF,sfac,iex,ecex,rcex,
     2             IFAIL)
C
C***********************************************************************
C
C     READRM reads R-matrix and asymptotic potential data
C
C     Input
C       LURMT,RFORM,NCHAN,NOCSF,NPOLE,IBUT, as defined above
C       NKEEP = Number of CI vectors to be used in current run
C       ISMX  = ISMAX, see above
C       ISM   = maximum multipole to be used in current run
C
C     Output
C       EIG   = Eigenvalues of Hamiltonian matrix (in hartrees)
C       WAMP  = R-matrix boundary amplitudes
C       VEC   = First NKEEP eigenvectors of Hamiltonian matrix
C       BCOEF = Coefficients of quadratic fit to Buttle correction
C       CF    = Coefficients of multipole expansion of asymptotic
C               potential stored as ISM lower triangles
c       sfac  = approximate amplitudes for partitioned R-matrix method
c       ecex  = poles for partitioned R-matrix correction
c       rcex  = amplitutes for partitioned R-matrix correction
C       IFAIL = 0 for success, =1 for failure in reading data
C
C***********************************************************************
C
C     The R-matrix is constructed from the above data as follows
C
C     RMATR*RMATRX(i,j) = sum(k=1,NSTAT)(WAMP(i,k)*WAMP(j,k)/(EIG(k)-E))
C                         + delta(i,j)(BCOEF(1,i)+KSQ(i)*BCOEF(2,i)+
C                                       KSQ(i)*KSQ(i)*BCOEF(3,i))
C                         + correction term if NSTAT < NOCSF 
C     where delta(i,j) denotes a Kroneker delta function
C           E is the total energy of the system in hartrees
C           KSQ(i) is the scattering energy in channel i in rydbergs
C           KSQ(i) = 2*(E-ETARG(i))
C
C***********************************************************************
C
      NCHSQ = NCHAN*(NCHAN+1)/2
      ISDIF = ISMX-ISM
      IF(RFORM.EQ.'FORMATTED') THEN
        IF(ISMX.GT.0) READ(LURMT,14) ((CF(I,K),I=1,NCHSQ),K=1,ISM),
     1  ((dum,I=1,NCHSQ),K=1,ISDIF)
        READ(LURMT,14) (EIG(I),I=1,nstat)
        READ(LURMT,14) ((WAMP(I,J),I=1,NCHAN),J=1,nstat)
        IF(NPOLE.GT.0) READ(LURMT,14) ((VEC(I,J),I=1,NOCSF),J=1,NKEEP),
     1  ((dum,I=1,NOCSF),J=NKEEP+1,NPOLE)
        IF(IBUT.GT.0) READ(LURMT,14) ((BCOEF(I,J),I=1,3),J=1,NCHAN)
        IF(abs(IBUT).GT.1) then
           READ(LURMT,14) (sfac(J),J=1,NCHAN)
           READ(LURMT,14) (ecex(J),J=1,iex)
           READ(LURMT,14) ((rcex(I,J),I=1,NCHAN),J=1,iex)
        endif
      ELSE
        IF(ISMX.GT.0) READ(LURMT) ((CF(I,K),I=1,NCHSQ),K=1,ISM),
     1  ((dum,I=1,NCHSQ),K=1,ISDIF)
        READ(LURMT) (EIG(I),I=1,nstat)
        READ(LURMT) ((WAMP(I,J),I=1,NCHAN),J=1,nstat)
        IF(NPOLE.GT.0) READ(LURMT) ((VEC(I,J),I=1,NOCSF),J=1,NKEEP),
     1  ((dum,I=1,NOCSF),J=NKEEP+1,NPOLE)
        IF(IBUT.GT.0) READ(LURMT) ((BCOEF(I,J),I=1,3),J=1,NCHAN)
        IF(abs(IBUT).GT.1) then
           READ(LURMT) (sfac(J),J=1,NCHAN)
           READ(LURMT) (ecex(J),J=1,iex)
           READ(LURMT) ((rcex(I,J),I=1,NCHAN),J=1,iex)
        endif
      ENDIF
C
      IF(IPRNT.GT.0) THEN
        WRITE(IWRITE,170)
        IF(ISM.GT.0) WRITE(IWRITE,114) ((CF(I,K),I=1,NCHSQ),K=1,ISM)
        WRITE(IWRITE,114) (EIG(I),I=1,nstat)
        WRITE(IWRITE,114) ((WAMP(I,J),I=1,NCHAN),J=1,nstat)
        IF(NPOLE.GT.0) WRITE(IWRITE,114)((VEC(I,J),I=1,NOCSF),J=1,NKEEP)
        IF(IBUT.GT.0) WRITE(IWRITE,114) ((BCOEF(I,J),I=1,3),J=1,NCHAN)
        IF(abs(IBUT).GT.1) then
           WRITE(IWRITE,114) (sfac(J),J=1,NCHAN)
           WRITE(IWRITE,114) (ecex(J),J=1,iex)
           WRITE(IWRITE,114) ((rcex(I,J),I=1,NCHAN),J=1,iex)
        endif
      ENDIF
C
      RETURN
C
 99   WRITE(IWRITE,98) NRSET,KEYRM,LURMT
      IFAIL = 1
      RETURN
 10   FORMAT(16I5)
 11   FORMAT(A80)
 12   FORMAT(4I5,2D20.13)
 33   format(i10,d20.13,i5)
 13   FORMAT(/' Dimension of Hamiltonian matrix           =',I10/
     2        ' Number of CI vectors stored on file       =',I5/
     1        ' Maximum multipole in asymptotic potential =',I5/
     3        ' R-matrix radius                           =',F10.4)
 14   FORMAT(4D20.13)
 16   FORMAT(/' R-matrix data has been read from set number',
     1I3,' on unit number',I3/' Header :',A80)
 17   FORMAT(/ ' Data read by READRH :')
 170  FORMAT(/ ' Data read by READRM :')
 96   FORMAT(/' Inconsistent geometry input R =',F7.4,' but on NRSET =',
     1I2,' R =',F7.4)
 98   FORMAT(/' Unable to find R-matrix dataset number',I3,'  key =',I3,
     1' on unit',I3)
 100  FORMAT(10I7)
 110  FORMAT(1X,16I5)
 111  FORMAT(1X,A80)
 112  FORMAT(1X,4I5,2D20.13)
 133  format(1x,i5,d20.13,i5)
 114  FORMAT(1X,4D20.13)
      END
      SUBROUTINE SKIPRM(LURMT,RFORM,NCHAN,NSTAT,NOCSF,ISMX,ISM,NPOLE,
     1                  NKEEP,IBUT,IFAIL)
      IMPLICIT DOUBLE PRECISION(A-H,O-Z)
      CHARACTER(LEN=11) RFORM
      INTENT(IN) LURMT,NCHAN,NSTAT,NOCSF,ISMX,ISM,NPOLE,NKEEP,IBUT,RFORM
      INTENT(OUT) IFAIL
C
C***********************************************************************
C
C     SKIPRM skips the records that READRM would read
C
C***********************************************************************
C
      IF(RFORM.EQ.'FORMATTED') THEN
        IF(ISMX.GT.0) READ(LURMT,14)
        READ(LURMT,14)
        READ(LURMT,14)
        IF(NPOLE.GT.0) READ(LURMT,14)
        IF(IBUT.GT.0) READ(LURMT,14)
        IF(ABS(IBUT).GT.1) THEN
           READ(LURMT,14)
           READ(LURMT,14)
           READ(LURMT,14)
        ENDIF
      ELSE
        IF(ISMX.GT.0) READ(LURMT)
        READ(LURMT)
        READ(LURMT)
        IF(NPOLE.GT.0) READ(LURMT)
        IF(IBUT.GT.0) READ(LURMT)
        IF(ABS(IBUT).GT.1) THEN
           READ(LURMT)
           READ(LURMT)
           READ(LURMT)
        ENDIF
      ENDIF
      IFAIL=0
      RETURN
 14   FORMAT(4D20.13)
      END
      SUBROUTINE READKH(LUKMT0,NKSET,MGVN,STOT,GUTOT,NCHAN,NVIB,NDIS,
     1 NTARG,ION,MAXNE,NEREP,EINC,RR,NAPPR,KFORM0,IWRIT0,IPRNT0,IFAIL)
      IMPLICIT DOUBLE PRECISION(A-H,O-Z)
C
C***********************************************************************
C
C     READKH locates K-matrix set number NKSET on unit LUKMT and reads 
C     its header
C
C     Input:
C      LUKMT  = Unit number of input device
C      NKSET  = Set number
C      KFORM  = Formatted/unformatted flag
C      RR     = Bond length if fixed nuclei data, else zero
C      IWRITE = Unit number for printed output
C
C***********************************************************************
C
      DIMENSION NEREP(*),EINC(2,*),AKMAT(*)
      INTEGER STOT,GUTOT
      CHARACTER(LEN=11) KFORM,KFORM0
      CHARACTER(LEN=80) TITLE
      DATA ZERO/0.D0/,TOL/1.D-6/,KKEY/11/
      SAVE
C
      LUKMT = LUKMT0
      KFORM = KFORM0
      IPRNT = IPRNT0
      IWRITE= IWRIT0
C
      NSET = NKSET
      CALL GETSET(LUKMT,NSET,KKEY,KFORM,IFAIL)
      IF(IFAIL.NE.0) GO TO 97
C
      IF(KFORM.EQ.'FORMATTED') THEN
        READ(LUKMT,100)KEY,NKSET,NREC,NINFO,NDATA
        READ(LUKMT,1)TITLE
        READ(LUKMT,*) MGVN,STOT,GUTOT,ION,R,RMASS
        READ(LUKMT,*) NTARG,NVIB,NDIS,NCHAN,MAXNE
        DO I=1,MAXNE
          READ(LUKMT,*) K,NEREP(K),(EINC(J,K),J=1,2)
        END DO
      ELSE
        READ(LUKMT)KEY,NKSET,NREC,NINFO,NDATA
        READ(LUKMT)TITLE
        READ(LUKMT) MGVN,STOT,GUTOT,ION,R,RMASS
        READ(LUKMT) NTARG,NVIB,NDIS,NCHAN,MAXNE
        DO I=1,MAXNE
          READ(LUKMT) K,NEREP(K),(EINC(J,K),J=1,2)
        END DO
      ENDIF
      IF(RR.NE.ZERO.AND.ABS(R-RR).GT.TOL) GO TO 99
      WRITE(IWRITE,104) NKSET,LUKMT
      IF(IPRNT.GE.0) WRITE(IWRITE,105)MGVN,STOT,GUTOT
      IF(R.NE.ZERO) THEN
        NAPPR = 0
        IF(IPRNT.GE.0) WRITE(IWRITE,106) R
        RR = R
      ELSE
        NAPPR = 1
        IF(IPRNT.GE.0) WRITE(IWRITE,108)
      ENDIF
      IF(IPRNT.GE.0) THEN
        WRITE(IWRITE,107) NTARG,NCHAN
        WRITE(IWRITE,14) (NEREP(I),I=1,MAXNE)
        WRITE(IWRITE,15) (EINC(1,I),I=1,MAXNE)
        WRITE(IWRITE,16) (EINC(2,I),I=1,MAXNE)
      ENDIF
C
      RETURN
C
      ENTRY READKM(NOPEN,NDOPEN,NCHAN2,EN,AKMAT)
C
C***********************************************************************
C
C     READKM reads K-matrices from a file which has previously been
C     positioned by a call to READKH.  They are stored as the lower
C     triangle of a symmetric matrix of dimension NOPEN.  The linear
C     dimension is NCHAN2=NOPEN*(NOPEN+1)/2. The corresponding 
C     scattering energy EN is in Rydbergs.  NDOPEN is the number of
C     dissociation channels.
C
C***********************************************************************
C
      nopen = 0
      IF(KFORM.EQ.'FORMATTED') THEN
        READ(LUKMT,*,end=2) NOPEN,NDOPEN,NCHAN2,EN
        READ(LUKMT,12) (AKMAT(I),I=1,NCHAN2)
      ELSE
        READ(LUKMT,end=2) NOPEN,NDOPEN,NCHAN2,EN,(AKMAT(I),I=1,NCHAN2)
      ENDIF
      IF(IPRNT.NE.0) WRITE(IWRITE,10) NOPEN,NDOPEN,NCHAN2,EN,
     1 (AKMAT(I),I=1,NCHAN2)
C
 2    RETURN
C
 1    FORMAT(A)
camar  100  FORMAT (10I7)
 100  FORMAT (2I3,I12,I3,I12)
 10   FORMAT(3I5,F20.13/(1X,4D20.13))
 12   FORMAT(4D20.13)
 104  FORMAT(/' K-matrix input, set',I3,' on unit',I3)
 105  FORMAT(/' Symmetry data  MGVN =',I2,3X,'STOT =',I2,3X,'GUTOT =',I2
     1)
 106  FORMAT(/' Fixed nuclei data for R =',F6.3)
 107  FORMAT(/' Number of target states       =',I4/' Number of scatteri
     1ng channels =',I4)
 108  FORMAT(/' K-matrices are vibrationally resolved')
 14   FORMAT(/' Input Energy grid (Ryd) '/' Number of points',10I10)
 15   FORMAT(' Initial values  ',10F10.5)
 16   FORMAT(' Increments      ',10F10.5)
C
 99   WRITE(IWRITE,98) NKSET,R,RR
 98   FORMAT(/' INCOMPATIBLE GEOMETRY DATA FOR NKSET =',I2,2D15.6)
      IFAIL = 1
      RETURN
 97   WRITE(IWRITE,96) NKSET,LUKMT
 96   FORMAT(/' UNABLE TO LOCATE K-MATRIX SET NUMBER',I4,'  ON UNIT',I3)
      IFAIL = 1
      RETURN
      END
      SUBROUTINE READTH(LUTMT0,TITLE,NTSET,NCHAN,NVIB,NDIS,NTARG,MAXCHI,
     1MAXCHF,MGVN,STOT,GUTOT,NETOT,MAXNE,NEREP,EINC,ICHL,LCHL,MCHL,ECHL,
     2TFORM0,IWRIT0,IPRNT0,IFAIL)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
C***********************************************************************
C
C     READTH locates T-matrix set number NTSET on unit LUTMT and reads
C     its header
C
C***********************************************************************
C
      INTEGER STOT,GUTOT
      CHARACTER(LEN=11) TFORM,TFORM0
      CHARACTER(LEN=80) TITLE
      DIMENSION TR(MAXCHI,MAXCHF+NDIS,*),TI(MAXCHI,MAXCHF+NDIS,*),
     1 NEREP(*),EINC(2,*),EN(*),MAXVI(*),MAXVJ(*),MAXVD(*),
     2 ICHL(*),LCHL(*),MCHL(*),ECHL(*)
      SAVE
      DATA zero/0.D0/,KEYT/12/
C
C---- Locate set number NTSET on unit LUTMT
      TFORM = TFORM0
      LUTMT = LUTMT0
      IPRNT = IPRNT0
      IWRITE = IWRIT0
      NSET = NTSET
      CALL GETSET(LUTMT,NSET,KEYT,TFORM,IFAIL)
      IF(IFAIL.NE.0) GO TO 99
C
C---- Read header
      IF(TFORM.EQ.'FORMATTED') THEN
        READ(LUTMT,100) KEY,NSET,NREC,NINFO,NDATA
        IF(KEY.NE.KEYT) GO TO 99
        READ(LUTMT,13) TITLE
        READ(LUTMT,10) MGVN,STOT,GUTOT,NCHAN,NAPPR,MAXNE,NVIB,
     1  NDIS,NTARG,MAXCHI,MAXCHF,R
        DO I=1,MAXCHF
          READ(LUTMT,*) ICHL(I),LCHL(I),MCHL(I),ECHL(I)
        END DO
        DO IE=1,MAXNE
          READ(LUTMT,*) Indx,NEREP(IE),(EINC(J,IE),J=1,2)
        END DO
      ELSE
        READ(LUTMT) KEY,NSET,NREC,NINFO,NDATA
        IF(KEY.NE.KEYT) GO TO 99
        READ(LUTMT) TITLE
        READ(LUTMT) MGVN,STOT,GUTOT,NCHAN,NAPPR,MAXNE,NVIB,NDIS,
     1  NTARG,MAXCHI,MAXCHF,R
        DO I=1,MAXCHF
          READ(LUTMT) ICHL(I),LCHL(I),MCHL(I),ECHL(I)
        END DO
        DO IE=1,MAXNE
          READ(LUTMT) Indx,NEREP(IE),(EINC(J,IE),J=1,2)
        END DO
      ENDIF
C
C---- Print header information
      WRITE(IWRITE,101) NSET,LUTMT,TITLE
      IF(IPRNT.GE.0) WRITE(IWRITE,105) MGVN,STOT,GUTOT
      IF(NAPPR.EQ.0) THEN
        IF(IPRNT.GE.0) WRITE(IWRITE,106) R
      ELSE IF(NAPPR.EQ.1) THEN
        IF(IPRNT.GE.0) WRITE(IWRITE,108)
      ELSE
        IF(IPRNT.GE.0) WRITE(IWRITE,109)
      ENDIF
C
C---- CALCULATE TOTAL NUMBER OF SCATTERING ENERGIES
      NETOT = 0
      DO I=1,MAXNE
        NETOT = NETOT+NEREP(I)
      END DO
C
      IF(IPRNT.GE.0) THEN
        WRITE(IWRITE,107) NTARG,NCHAN
        WRITE(IWRITE,102)
        DO I=1,MAXCHF
          WRITE(IWRITE,103) I,ICHL(I),LCHL(I),MCHL(I),ECHL(I)
        END DO
        WRITE(IWRITE,14) (NEREP(I),I=1,MAXNE)
        WRITE(IWRITE,15) (EINC(1,I),I=1,MAXNE)
        WRITE(IWRITE,16) (EINC(2,I),I=1,MAXNE)
      ENDIF
C
      RETURN
C
      ENTRY READT(NE,MAXCHI,MAXCHF,NDIS,MAXVI,MAXVJ,MAXVD,TR,TI,EN)
C
C***********************************************************************
C
C     READT reads T-matrices from unit LUTMT which has previously been
C     positioned correctly via a call to READTH
C
C     NE input,  number of T-matrices required
C        output, number of T-matrices found
C
C***********************************************************************
C
      ien = 0
      IF(TFORM.EQ.'FORMATTED') THEN
        DO 1 IE=1,NE
        READ(LUTMT,12,end=4) MVI,MVJ,MVD,EN(IE)
        ien = ie
        MAXVI(IE) = MVI
        MAXVJ(IE) = MVJ
        MAXVD(IE) = MVD
        READ(LUTMT,11) ((TR(I,J,IE),I=1,MVI),J=1,MVJ+MVD)
        READ(LUTMT,11) ((TI(I,J,IE),I=1,MVI),J=1,MVJ+MVD)
 1      CONTINUE
      ELSE
        DO 2 IE=1,NE
        READ(LUTMT,end=4) MVI,MVJ,MVD,EN(IE)
        ien = ie
        MAXVI(IE) = MVI
        MAXVJ(IE) = MVJ
        MAXVD(IE) = MVD
        READ(LUTMT) ((TR(I,J,IE),I=1,MVI),J=1,MVJ+MVD)
        READ(LUTMT) ((TI(I,J,IE),I=1,MVI),J=1,MVJ+MVD)
 2      CONTINUE
      ENDIF
 4    if(ien.lt.ne) write(iwrite,121) ien
      ne = ien
      IF(IPRNT.GT.0) THEN
        WRITE(IWRITE,17)
        DO 3 IE=1,NE
        MVI = MAXVI(IE)
        MVJ = MAXVJ(IE)+MAXVD(IE)
        WRITE(IWRITE,120) MAXVI(IE),MAXVJ(IE),MAXVD(IE),EN(IE)
        WRITE(IWRITE,110) ((TR(I,J,IE),I=1,MVI),J=1,MVJ+MVD)
        WRITE(IWRITE,110) ((TI(I,J,IE),I=1,MVI),J=1,MVJ+MVD)
 3      CONTINUE
      ENDIF
C
      RETURN
 99   WRITE(IWRITE,98) NTSET,LUTMT
      IFAIL = 1
      RETURN
C
 100  FORMAT(10I20)
 10   FORMAT(11I5,F15.8)
 11   FORMAT(4E20.13)
 12   FORMAT(3I5,F10.6)
 13   FORMAT(A80)
 14   FORMAT(/' Input Energy grid (Ryd) '/' Number of points',10I10)
 15   FORMAT(' Initial values  ',10F10.5)
 16   FORMAT(' Increments      ',10F10.5)
 17   FORMAT(/' Energy dependant data read from LUTMT')
 101  FORMAT(/' T-matrices will be read from set',I3,' on unit',I3/'  ('
     1,A80,' )')
 102  FORMAT(' Channel data corresponding to stored T-matrix elements'/
     1'  Element  Target   L    M    Energy')
 103  FORMAT(I5,I10,I6,I5,F10.5)
 105  FORMAT(/' Symmetry data  MGVN =',I2,3X,'STOT =',I2,3X,'GUTOT =',I2
     1)
 106  FORMAT(/' Fixed nuclei data for R =',F6.3)
 107  FORMAT(/' Number of target states       =',I4/' Number of scatteri
     1ng channels =',I4)
 108  FORMAT(/' T-matrices have been adiabatically averaged')
 109  FORMAT(/' T-matrices have been calculated from non-adiabatic K-mat
     1rices')
 110  FORMAT(1X,4E20.13)
 120  FORMAT(1X,3I5,F10.6)
 121  FORMAT(/' Fewer T-matrices found than were requested, continuing w
     1ith ',i4,' energies')
 98   FORMAT(/' UNABLE TO FIND T-MATRIX SET',I3,' ON UNIT',I3)
      END
      SUBROUTINE WRVCMT(A,B,N,M,NN,MM,NCOL,IWRITE)
C
C***********************************************************************
C
C     WRVCMT PRINTS OUT A N*M MATRIX STORED IN A NN * MM ARRAY
C     together with a header vector B
C
C***********************************************************************
C
      IMPLICIT DOUBLE PRECISION(A-H,O-Z)
      CHARACTER(LEN=1) BLANK
      DIMENSION A(NN,MM),B(NN)
      DATA BLANK/' '/
C
 1000 FORMAT(10D12.4)
 1010 FORMAT(A)
C
      NTIM=M/NCOL
      NF=0
      IF (NTIM.EQ.0) GO TO 30
C
      DO 20 I=1,NTIM
      NI=NF+1
      NF=NF+NCOL
      write(iwrite,1000) (b(k),k=ni,nf)
      WRITE(IWRITE,1010) BLANK
      DO 10 J=1,N
      WRITE(IWRITE,1000) (A(J,K),K=NI,NF)
   10 CONTINUE
      WRITE(IWRITE,1010) BLANK
   20 CONTINUE
C
   30 NI=NF+1
      IF(NI.GT.M) RETURN
      write(iwrite,1000) (b(k),k=ni,m)
      WRITE(IWRITE,1010) BLANK
      DO 40 J=1,N
      WRITE(IWRITE,1000) (A(J,K),K=NI,M)
   40 CONTINUE
C
      RETURN
      END
C    The following routine does not seem to be called by any outer region module
C    (JDG 21/02/03)
      SUBROUTINE READTDH(LUTRD0,NTDSET,NDATA,TITLE,NBOUND1,NBOUND2,
     1MGVN1,MGVN2,STOT1,STOT2,GUTOT1,GUTOT2,NSTAT1,NSTAT2,RR1,RR2,
     1TDFORM0,IPRNT0,IWRIT0,IFAIL)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
C***********************************************************************
C
C     READTDH locates and reads set NTDSET on unit LUTRD
C
C***********************************************************************
C
      INTEGER STOT1,STOT2,GUTOT1,GUTOT2
      CHARACTER(LEN=11) TDFORM,TDFORM0
      CHARACTER(LEN=80) TITLE
      DIMENSION EN1(*),EN2(*),TRDIP(*),OSCST(*),SMOOTH(*)
      DATA KEYBC/11/
      SAVE
C
      TDFORM = TDFORM0
      LUTRD  = LUTRD0
      IPRNT  = IPRNT0
      IWRITE = IWRIT0
C
C---- Locate set number NTDSET on unit LUTRD
      NSET = NTDSET
      CALL GETSET(LUTRD,NSET,KEYBC,TDFORM,IFAIL)
      IF(IFAIL.NE.0) GO TO 99
C
C---- Read header
      IF(TDFORM.EQ.'FORMATTED') THEN
        READ(LUTRD,10) KEYBC,NSET,NREC,NINFO,NDATA
        READ(LUTRD,13) TITLE
        READ(LUTRD,10) NBOUND1,MGVN1,STOT1,GUTOT1,NSTAT1
        READ(LUTRD,20) RR1
        READ(LUTRD,10) NBOUND2,MGVN2,STOT2,GUTOT2,NSTAT2
        READ(LUTRD,20) RR2
      ELSE
        READ(LUTRD) KEYBC,NSET,NREC,NINFO,NDATA
        READ(LUTRD) TITLE
        READ(LUTRD) NBOUND1,MGVN1,STOT1,GUTOT1,NSTAT1
        READ(LUTRD) RR1
        READ(LUTRD) NBOUND2,MGVN2,STOT2,GUTOT2,NSTAT2
        READ(LUTRD) RR2
      ENDIF
C
C---- Print header information
      IF(IPRNT.NE.0)  THEN
        WRITE(IWRITE,14)
        WRITE(IWRITE,100) KEYBC,NSET,NREC,NINFO,NDATA
        WRITE(IWRITE,130) TITLE
        WRITE(IWRITE,100) NBOUND1,MGVN1,STOT1,GUTOT1,NSTAT1
        WRITE(IWRITE,101) RR1
        WRITE(IWRITE,100) NBOUND2,MGVN2,STOT2,GUTOT2,NSTAT2
        WRITE(IWRITE,101) RR2
      ENDIF
C
      RETURN
C
      ENTRY READTD(EN1,EN2,TRDIP,OSCST,SMOOTH,NDATA)
C
      DO 146 I=1,NDATA
        IF(TDFORM.EQ.'FORMATTED') THEN
          READ(LUTRD,11) EN1(I),EN2(I),TRDIP(I),OSCST(I),SMOOTH(I)
        ELSE
          READ(LUTRD) EN1(I),EN2(I),TRDIP(I),OSCST(I),SMOOTH(I)
        ENDIF
        IF(IPRNT.NE.0) WRITE(IWRITE,110) EN1(I),EN2(I),TRDIP(I),OSCST(I)
     1,SMOOTH(I)
 146  CONTINUE
C
      RETURN
C
 99   WRITE(IWRITE,98) NTDSET,LUTRD
      IFAIL = 1
      RETURN
C
 11   FORMAT(10F20.13)
 10   FORMAT(10I5)
 20   FORMAT(10F20.6)
 13   FORMAT(A80)
 14   FORMAT(/' Header on LUTRD')
 100  FORMAT(1X,10I5)
 101  FORMAT(1X,10F20.6)
 110  FORMAT(1X,10E20.13)
 130  FORMAT(1X,A80)
 98   FORMAT(/' UNABLE TO FIND TRANSITION DIPOLE SET',I3,' ON UNIT',I3)
      END
      SUBROUTINE WRECMT(A,N,M,NN,MM,NCOL,IWRITE)
C
C***********************************************************************
C
C     WRECMT PRINTS OUT A N*M MATRIX STORED IN A NN * MM ARRAY
C
C***********************************************************************
C
      IMPLICIT DOUBLE PRECISION(A-H,O-Z)
      CHARACTER(LEN=1) BLANK
      DIMENSION A(NN,MM)
      DATA BLANK/' '/
C
 1000 FORMAT(10D12.4)
 1010 FORMAT(A)
C
      NTIM=M/NCOL
      NF=0
      IF (NTIM.EQ.0) GO TO 30
C
      DO 20 I=1,NTIM
      NI=NF+1
      NF=NF+NCOL
      DO 10 J=1,N
      WRITE(IWRITE,1000) (A(J,K),K=NI,NF)
   10 CONTINUE
      WRITE(IWRITE,1010) BLANK
   20 CONTINUE
C
   30 NI=NF+1
      IF(NI.GT.M) RETURN
      DO 40 J=1,N
      WRITE(IWRITE,1000) (A(J,K),K=NI,M)
   40 CONTINUE
C
      RETURN
      END
C    The following routine does not seem to be called by any outer region module
C    (JDG 21/02/03)
      SUBROUTINE READWH(LUWAFN,NWSET,WFORMI,MGVN,STOT,GUTOT,ION,R,
     2 RMASS,NCHAN,NVIB,NDIS,NTARG,MAXNE,NEREP,EINC,IPRNT,IWRITE,IFAIL)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
C***********************************************************************
C
C     READWH locates wave-function file set number NWSET on unit LUWAFN
C            and reads header.
C
C***********************************************************************
C
      INTEGER STOT,GUTOT
      CHARACTER(LEN=11) WFORM,WFORMI
      CHARACTER(LEN=80) TITLE
      DIMENSION NEREP(*),EINC(2,*),FX(NCHAN,NCHAN,2),FXP(NCHAN,NCHAN,2)
      SAVE
      DATA KEY/13/
C
C----- Locate set number NWSET on unit luwafn
      LUWFN = LUWAFN
      WFORM = WFORMI
C
      NSET = NWSET
      CALL GETSET(LUWFN,NSET,KEY,WFORM,IFAIL)
      IF(IFAIL.NE.0) WRITE(IWRITE,20) NSET,LUWFN
      IF(IFAIL.NE.0) RETURN
      IF(NWSET.NE.1) NSET = NSET+1
      NWSET = NSET
C
C---- Read header
      IF(WFORM.EQ.'FORMATTED') THEN
        READ(LUWFN,10) KEY,NSET,NREC,NINFO,NDATA
        READ(LUWFN,13) TITLE
        READ(LUWFN,12) MGVN,STOT,GUTOT,ION,R,RMASS
        READ(LUWFN,10) NTARG,NVIB,NDIS,NCHAN,MAXNE
        DO I=1,MAXNE
          READ(LUWFN,15) K,NEREP(I),(EINC(J,I),J=1,2)
        END DO
      ELSE
        READ(LUWFN) KEY,NSET,NREC,NINFO,NDATA
        READ(LUWFN) TITLE
        READ(LUWFN) MGVN,STOT,GUTOT,ION,R,RMASS
        READ(LUWFN) NTARG,NVIB,NDIS,NCHAN,MAXNE
        DO I=1,MAXNE
          READ(LUWFN) K,NEREP(I),(EINC(J,I),J=1,2)
        END DO
      ENDIF
C
C---- Print header information
      WRITE(IWRITE,16) NSET,LUWFN,TITLE
      IF(IPRNT.GT.0) THEN
        WRITE(IWRITE,112) MGVN,STOT,GUTOT
        WRITE(IWRITE,110) NTARG,NVIB,NDIS,NCHAN,MAXNE,ION,R,RMASS
        DO I=1,MAXNE
          WRITE(IWRITE,111) I,NEREP(I),(EINC(J,I),J=1,2)
        END DO
      ENDIF
C
      RETURN
 10   FORMAT(10I5)
 11   FORMAT(3I10,D20.13)
 12   FORMAT(4I5,2D15.6)
 13   FORMAT(A80)
 15   FORMAT(2I10,2D20.13)
 16   FORMAT(/' WAVEFUNCTIONS READ FROM SET',I3,' ON UNIT',I3/2X,A80)
 17   FORMAT((4D20.13))
 20   FORMAT(2X,'FAILED TO READ SET=',I3,3X,'FROM UNIT=',I3)
 110  FORMAT(/1X,'NTARG=',I3,2X,'NVIB=',I3,2X,'NDIS=',I3,2X,'NCHAN=',I3
     1,2X,'MAXNE=',I3,2X,'ION=',I3,2X/10X,'R=',D15.6,5X,'RMASS=',D15.6)
 111  FORMAT(1X,2I10,2D20.13)
 112  FORMAT(/1X,' SYMMETRY DATA: MGVN=',I3,3X,'STOT=',I3,3X,
     1 'GUTOT=',I3)
C
C***********************************************************************
C
      ENTRY  READWF(NCHAN,nopen,en,fx,fxp)
C
C***********************************************************************
C
C     READWF READS   WAVEFUNCTIONS AND DERIVATIVES FROM UNIT
C            LUWFN which has previously been positioned correctly via a
C            call to READWH
C
C***********************************************************************
C
      IF(WFORM.EQ.'FORMATTED') THEN
        READ(LUWFN,11) NCHANF,NOPEN,EN
        READ(LUWFN,17)  (((FX(I,J,K),I=1,NCHANF),J=1,NCHANF),K=1,2),
     2                  (((FXP(I,J,K),I=1,NCHANF),J=1,NCHANF),K=1,2)
      ELSE
        READ(LUWFN) NCHANF,NOPEN,EN,
     2                  (((FX(I,J,K),I=1,NCHANF),J=1,NCHANF),K=1,2),
     3                  (((FXP(I,J,K),I=1,NCHANF),J=1,NCHANF),K=1,2)
      ENDIF
C
      RETURN
      END
C   I don't think this routine is used either (it's only called in rsolve and timedel
C   if LUWFN is different from 0. And this parameter is always 0!!
C   JDG 27/02/03 (I modify the format in the write anyway)
      SUBROUTINE WRITWH(LUWAFN,NWSET,WFORM0,TITLE,MGVN,STOT,GUTOT,ION,R,
     1 RMASS,NCHAN,NVIB,NDIS,NTARG,MAXNE,NEREP,EINC,NETOT,IPRT,IWRT,
     1 IFAIL)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
C***********************************************************************
C
C     WRITWH writes header on wave-function file on unit LUWFN
C      
C     If NWSET = 1 on input then R-matrices and wavefunctions are 
C                  written as the first set 
C              = 0 then they are written at end-of-information
C     On output NWSET holds the actual set number
C
C***********************************************************************
C
      INTEGER STOT,GUTOT
      CHARACTER(LEN=11) WFORM,WFORM0
      CHARACTER(LEN=80) TITLE
      DIMENSION NEREP(MAXNE),EINC(2,MAXNE),FX(NCHAN,NCHAN,2),
     1 FXP(NCHAN,NCHAN,2)
      SAVE
      DATA zero/0.D0/,KEY/13/
C
C----- Position file at end of information (or end of set number NTSET)
      NSET = NWSET
      LUWFN = LUWAFN
      WFORM = WFORM0
      IPRNT  = IPRT
      IWRITE  = IWRT
      CALL GETSET(LUWFN,NSET,KEY,WFORM,IFAIL)
      IF(IFAIL.NE.0) RETURN
      IF(NWSET.NE.1) NSET = NSET+1
      WRITE(IWRITE,16) NSET,LUWFN
      NWSET = NSET
C
C---- Calculate number of records to be written to K-matrix set.
C     This is only accurate if LUWFN is unformatted or all channels are
C     open, and the loop over scattering energies terminates normally.
      IF(WFORM.EQ.'FORMATTED') THEN
        NDATA = 2+MAXNE+NETOT*(NCHAN+NCHAN+1)
      ELSE
        NDATA = 2+MAXNE+NETOT
      ENDIF
      NINFO = 1
      NREC = NDATA+NINFO
C
C---- Write header
      IF(WFORM.EQ.'FORMATTED') THEN
        WRITE(LUWFN,210) KEY,NSET,NREC,NINFO,NDATA
        WRITE(LUWFN,13) TITLE
        WRITE(LUWFN,12) MGVN,STOT,GUTOT,ION,R,RMASS
        WRITE(LUWFN,10) NTARG,NVIB,NDIS,NCHAN,MAXNE
        DO I=1,MAXNE
          WRITE(LUWFN,15) I,NEREP(I),(EINC(J,I),J=1,2)
        END DO
      ELSE
        WRITE(LUWFN) KEY,NSET,NREC,NINFO,NDATA
        WRITE(LUWFN) TITLE
        WRITE(LUWFN) MGVN,STOT,GUTOT,ION,R,RMASS
        WRITE(LUWFN) NTARG,NVIB,NDIS,NCHAN,MAXNE
        DO I=1,MAXNE
          WRITE(LUWFN) I,NEREP(I),(EINC(J,I),J=1,2)
        END DO
      ENDIF
C
C---- PRINT HEADER
      IF(IPRNT.GT.0) THEN
        WRITE(IWRITE,14)        
        WRITE(IWRITE,110) KEY,NSET,NREC,NINFO,NDATA
        WRITE(IWRITE,113) TITLE
        WRITE(IWRITE,112) MGVN,STOT,GUTOT,ION,R,RMASS
        WRITE(IWRITE,110) NTARG,NVIB,NDIS,NCHAN,MAXNE
        DO I=1,MAXNE
          WRITE(IWRITE,111) I,NEREP(I),(EINC(J,I),J=1,2)
        END DO
      ENDIF
C
      RETURN
C
      ENTRY WRITWF(NCHAN,NOPEN,EN,FX,FXP)
C
C***********************************************************************
C
C     WRITWF writes R-matrices, wavefunctions and derivatives to unit 
C            LUWFN which has previously been positioned correctly via a 
C            call to WRITWH
C
C***********************************************************************
C
      IF(WFORM.EQ.'FORMATTED') THEN
        WRITE(LUWFN,11) NCHAN,NOPEN,EN
        WRITE(LUWFN,17) (((FX(I,J,K),I=1,NCHAN),J=1,NCHAN),K=1,2),
     2                  (((FXP(I,J,K),I=1,NCHAN),J=1,NCHAN),K=1,2)
      ELSE
        WRITE(LUWFN) NCHAN,NOPEN,EN,
     2                  (((FX(I,J,K),I=1,NCHAN),J=1,NCHAN),K=1,2),
     3                  (((FXP(I,J,K),I=1,NCHAN),J=1,NCHAN),K=1,2)
      ENDIF
C
C---- PRINT R-MATRIX AND WAVEFUNCTIONS
      IF(IPRNT.GT.0) THEN
        WRITE(IWRITE,18)
        WRITE(IWRITE,114) NCHAN,NOPEN,EN
        WRITE(IWRITE,115) (((FX(I,J,K),I=1,NCHAN),J=1,NCHAN),K=1,2),
     2                  (((FXP(I,J,K),I=1,NCHAN),J=1,NCHAN),K=1,2)
      ENDIF
C
      RETURN
 10   FORMAT(10I5)
 11   FORMAT(2I10,D20.13)
 12   FORMAT(4I5,2D15.6)
 13   FORMAT(A80)
 14   FORMAT(/' Header on LUWFN')
 15   FORMAT(2I10,2D20.13)
 16   FORMAT(/' Wavefunctions will be written to set',I3,' on unit',I3)
 17   FORMAT((4D20.13))
 18   FORMAT(/' Data written to LUWFN')
 110  FORMAT(1X,10I5)
 111  FORMAT(1X,2I10,2D20.13)
 112  FORMAT(1X,4I5,2D15.6)
 113  FORMAT(1X,A80)
 114  FORMAT(1X,3I10,D20.13)
 115  FORMAT((1X,4D20.13))
 210  FORMAT (10I7)
      END
      SUBROUTINE WRITCH(LUCHAN,NCSET,CFORM,R,RMASS,ICHL,LCHL,MCHL,ECHL,
     1NTARG,STARG,MTARG,GUTARG,ETARG,IVTARG,IV,HEADER,IPRNT,IFAIL)
C
C***********************************************************************
C
C     WRITCH WRITES HEADER,TARGET AND CHANNEL DATA TO SET NUMBER NCSET
C      ON UNIT LUCHAN
C
C      This data is
C        NCHAN = Number of scattering channels
C        ICHL  = POINTERS FROM SCATTERING CHANNELS TO TARGET ELECTRONIC
C                STATES
C        LCHL  = ORBITAL ANGULAR MOMENTUM IN EACH CHANNEL
C        MCHL  = M QUANTUM NUMBER IN EACH CHANNEL
C        ECHL  = CHANNEL THRESHOLDS IN RYD RELATIVE TO LOWEST
C                 TARGET STATE
C        NTARG  = number of target electronic states
C        MTARG  = SYMMETRIES OF TARGET STATES
C        STARG  = SPIN MULTIPLICITIES OF TARGET STATES
C        GUTARG = G/U SYMMETRIES OF TARGET STATES
C        ETARG  = ENERGIES OF TARGET STATES IN HARTREES
C
C***********************************************************************
C
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      CHARACTER(LEN=1) D
      CHARACTER(LEN=11) CFORM
      CHARACTER(LEN=80) HEADER
      COMMON/INTFS/ NCHAN,ION,NVIB,NDIS,IWRITE,MGVN,STOT,GUTOT
      DIMENSION ICHL(NCHAN),ECHL(NCHAN),ETARG(NTARG),STARG(NTARG),
     1MTARG(NTARG),GUTARG(NTARG),LCHL(NCHAN),MCHL(NCHAN),IVTARG(*),IV(*)
      INTEGER STARG,GUTARG,STOT,GUTOT
      DATA KEYCH/10/,TWO/2.D0/,D/'D'/
C
C---- FIND REQUIRED SET
      NSET = NCSET
      CALL GETSET(LUCHAN,NSET,KEYCH,CFORM,IFAIL)
      IF(IFAIL.NE.0) GO TO 99
      IF(NCSET.NE.1) NSET = NSET+1
      WRITE(IWRITE,16) CFORM,NSET,LUCHAN,HEADER
C
C---- WRITE TARGET AND CHANNEL DATA
C
      NVD = NVIB+NDIS
      NREC = 3+NTARG+NCHAN+NVD
      NINFO = 1
      NDATA = NREC-NINFO
      IF(CFORM.EQ.'FORMATTED') THEN
        WRITE(LUCHAN,10) KEYCH,NSET,NREC,NINFO,NDATA
        WRITE(LUCHAN,11) HEADER
        WRITE(LUCHAN,10) NTARG,NVIB,NDIS,NCHAN
        WRITE(LUCHAN,12) MGVN,STOT,GUTOT,ION,R,RMASS
        DO I=1,NTARG
          WRITE(LUCHAN,12) I,MTARG(I),STARG(I),GUTARG(I),ETARG(I)
        END DO
        DO I=1,NCHAN
          WRITE(LUCHAN,12) I,ICHL(I),LCHL(I),MCHL(I),ECHL(I)
        END DO
        IF(NVD.GT.0) THEN
          DO I=1,NVD
            WRITE(LUCHAN,10) I,IVTARG(I),IV(I)
          END DO
        ENDIF
      ELSE
        WRITE(LUCHAN) KEYCH,NSET,NREC,NINFO,NDATA
        WRITE(LUCHAN) HEADER
        WRITE(LUCHAN) NTARG,NVIB,NDIS,NCHAN
        WRITE(LUCHAN) MGVN,STOT,GUTOT,ION,R,RMASS
        DO I=1,NTARG
          WRITE(LUCHAN) I,MTARG(I),STARG(I),GUTARG(I),ETARG(I)
        END DO
        DO I=1,NCHAN
          WRITE(LUCHAN) I,ICHL(I),LCHL(I),MCHL(I),ECHL(I)
        END DO
        IF(NVD.GT.0) THEN
          DO I=1,NVD
            WRITE(LUCHAN) I,IVTARG(I),IV(I)
          END DO
        ENDIF
      ENDIF
C
C----- PRINT OUT TARGET AND CHANNEL DATA
C
      TWOM = TWO*RMASS
      WRITE(IWRITE,21) (I,STARG(I),MTARG(I),GUTARG(I),ETARG(I),
     1 I=1,NTARG)
      WRITE(IWRITE,18) ION
      WRITE(IWRITE,23) (I,ICHL(I),LCHL(I),MCHL(I),ECHL(I),I=1,NCHAN
     1 -NDIS)
      IF(NDIS.GT.0) WRITE(IWRITE,121) (I,D,LCHL(I),MCHL(I),ECHL(I)/TWOM
     1 ,I=NCHAN-NDIS+1,NCHAN)
      IF(NVD.GT.0) WRITE(IWRITE,24) NVIB,NDIS,(I,IVTARG(I),IV(I),I=1,
     1NVIB)
      IF(IPRNT.GT.0) THEN
        WRITE(IWRITE,13)
        WRITE(IWRITE,10) KEYCH,NSET,NREC,NINFO,NDATA
        WRITE(IWRITE,11) HEADER
        WRITE(IWRITE,10) NTARG,NVIB,NDIS,NCHAN
        WRITE(IWRITE,12) MGVN,STOT,GUTOT,ION,R,RMASS
        DO I=1,NTARG
          WRITE(IWRITE,12) I,MTARG(I),STARG(I),GUTARG(I),ETARG(I)
        END DO
        DO I=1,NCHAN-NDIS
          WRITE(IWRITE,12) I,ICHL(I),LCHL(I),MCHL(I),ECHL(I)
        END DO
        IF(NVD.GT.0) THEN
          DO I=1,NVD
            WRITE(IWRITE,10) I,IVTARG(I),IV(I)
          END DO
        ENDIF
      ENDIF
C
      RETURN
C
 99   WRITE(IWRITE,98) NCSET,KEYCH,LUCHAN
      IFAIL = 1
      RETURN
C
 10   FORMAT(16I5)
 11   FORMAT(A80)
 12   FORMAT(4I5,2D20.12)
 121  FORMAT(I5,12X,A1,6X,I2,2X,I2,F12.6)
 13   FORMAT(/' Data written to LUCHAN :'/)
 16   FORMAT(/' Channel data will be written ',A11,' to set number',
     1I3,' on unit number',I3/' Header :',A80)
 18   FORMAT(/' Residual charge on system =',I2)
 21   FORMAT(/' Target state    Spin    M    G/U  Energy (au)'/
     1(2I10,2I6,F12.6))
 23   FORMAT(/' Channel   Target state  L   M  Threshold (Ryd)'/
     1(I5,5X,I8,6X,I2,2X,I2,F12.6))
 24   FORMAT(/' Number of vibrational levels =',I3,5X,'Number of dissoci
     1ating states =',I2/' Level  Electronic state  Vib. q. no.'/
     2(1X,I3,9X,I2,9X,I2))
 98   FORMAT(/' Unable to find channel dataset number',I3,'  key =',I3,
     1' on unit',I3)
      END
      SUBROUTINE WRITKH(LUKMAT,NKSET,KFORM0,TITLE,MGVN,STOT,GUTOT,ION,R,
     1 RMASS,NCHAN,NVIB,NDIS,NTARG,MAXNE,NEREP,EINC,NETOT,IPRNT,IWRITE,
     1 IFAIL)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
C***********************************************************************
C
C     WRITKH writes header on K-matrix file on unit LUKMT
C      
C     If NKSET = 1 on input then K-matrices are written as the first se
C              = 0 then they are written at end-of-information
C     On output NKSET holds the actual set number
C
C***********************************************************************
C
      INTEGER STOT,GUTOT
      CHARACTER(LEN=11) KFORM,KFORM0
      CHARACTER(LEN=80) TITLE
      DIMENSION AKMAT(NOPEN,NOPEN),NEREP(MAXNE),EINC(2,MAXNE)
      SAVE
      DATA zero/0.D0/,KEY/11/
C
C----- Position file at end of information (or end of set number NTSET)
      NSET = NKSET
      LUKMT = LUKMAT
      KFORM = KFORM0
      CALL GETSET(LUKMT,NSET,KEY,KFORM,IFAIL)
      IF(IFAIL.NE.0) RETURN
      IF(NKSET.NE.1) NSET = NSET+1
      WRITE(IWRITE,16) NSET,LUKMT
      NKSET = NSET
C
C---- Calculate number of records to be written to K-matrix set.
C     This is only accurate if LUKMT is unformatted or all channels are
C     open, and the loop over scattering energies terminates normally.
      IF(KFORM.EQ.'FORMATTED') THEN
        NDIM = NCHAN*(NCHAN+1)/2
        NDATA = 2+MAXNE+NETOT*(NDIM+3)/4
      ELSE
        NDATA = 2+MAXNE+NETOT
      ENDIF
      NINFO = 1
      NREC = NDATA+NINFO
C
C---- Write header
      IF(KFORM.EQ.'FORMATTED') THEN
        WRITE(LUKMT,210) KEY,NSET,NREC,NINFO,NDATA
        WRITE(LUKMT,13) TITLE
        WRITE(LUKMT,12) MGVN,STOT,GUTOT,ION,R,RMASS
        WRITE(LUKMT,10) NTARG,NVIB,NDIS,NCHAN,MAXNE
        DO I=1,MAXNE
          WRITE(LUKMT,15) I,NEREP(I),(EINC(J,I),J=1,2)
        END DO
      ELSE
        WRITE(LUKMT) KEY,NSET,NREC,NINFO,NDATA
        WRITE(LUKMT) TITLE
        WRITE(LUKMT) MGVN,STOT,GUTOT,ION,R,RMASS
        WRITE(LUKMT) NTARG,NVIB,NDIS,NCHAN,MAXNE
        DO I=1,MAXNE
          WRITE(LUKMT) I,NEREP(I),(EINC(J,I),J=1,2)
        END DO
      ENDIF
C
C---- PRINT HEADER
      IF(IPRNT.GT.0) THEN
        WRITE(IWRITE,14)        
        WRITE(IWRITE,110) KEY,NSET,NREC,NINFO,NDATA
        WRITE(IWRITE,113) TITLE
        WRITE(IWRITE,112) MGVN,STOT,GUTOT,ION,R,RMASS
        WRITE(IWRITE,110) NTARG,NVIB,NDIS,NCHAN,MAXNE
        DO I=1,MAXNE
          WRITE(IWRITE,111) I,NEREP(I),(EINC(J,I),J=1,2)
        END DO
      ENDIF
C
      RETURN
C
      ENTRY WRITKM(NOPEN,NDOPEN,EN,AKMAT)
C
C***********************************************************************
C
C     WRITKM writes K-matrices to unit LUKMT which has previously been
C     positioned correctly via a call to WRITKH
C
C***********************************************************************
C
      NCHSQ = NOPEN*(NOPEN+1)/2
      IF(KFORM.EQ.'FORMATTED') THEN
        WRITE(LUKMT,11) NOPEN,NDOPEN,NCHSQ,EN,
     1                  ((AKMAT(I,J),I=1,J),J=1,NOPEN)
      ELSE
        WRITE(LUKMT) NOPEN,NDOPEN,NCHSQ,EN,
     1               ((AKMAT(I,J),I=1,J),J=1,NOPEN)
      ENDIF
C
      RETURN
 10   FORMAT(10I5)
 11   FORMAT(3I10,D20.13/(4D20.13))
 12   FORMAT(4I5,2D15.6)
 13   FORMAT(A80)
 14   FORMAT(/' Header on LUKMT')
 15   FORMAT(2I10,2D20.13)
 16   FORMAT(/' K-matrices will be written to set ',I0,' on unit ',I0)
 110  FORMAT(1X,10I5)
 111  FORMAT(1X,2I10,2D20.13)
 112  FORMAT(1X,4I5,2D15.6)
 113  FORMAT(1X,A80)
camar 210  FORMAT(10I7)
 210  FORMAT (2I3,I12,I3,I12)
      END
      SUBROUTINE READTC(LUCHAN,NCSET,NCHAN,NVIB,NDIS,NTARG,ION,IVTARG,
     1IV,ITCHL,LCHL,MCHL,ECHL,STARG,MTARG,GUTARG,ETARG,R,RMASS,CFORM,
     2IWRITE,IPRNT,IFAIL)
C
C***********************************************************************
C
C     READTC reads target and channel data from set number NCSET on unit
C      LUCHAN
C
C      Input
C        NTARG  = number of target electronic states
C        NVIB   = number of target vibrational levels ( 0 if no 
C                 vibration) 
C        NDIS   = number of dissociating states
C        NCHAN  = number of scattering channels
C        LUCHAN = logical unit holding data
C        NCSET  = set number
C        CFORM  = 'FORMATTED' or 'UNFORMATTED'
C        R      = bond length
C        IWRITE = logical unit for printed output
C        IPRNT  = print flag
C
C      Output
C        MTARG  = symmetries of target states
C        STARG  = spin multiplicities of target states
C        GUTARG = G/U symmetries of target states
C        ETARG  = energies of target states in hartrees
C        ITCHL  = pointers from scattering channels to vibronic levels
C        LCHL   = orbital angular momentum in each channel
C        MCHL   = M quantum number in each channel
C        ECHL   = channel thresholds, in rydbergs, relative to lowest
C                 vibronic level
C        IVTARG = pointers from vibrational/dissociating states to
C                 parent target electronic configuration
C        IV     = vibrational quantum numbers of vibronic levels
C        ION    = residual charge on target
C        RMASS  = reduced mass of target
C        IFAIL  = 0 for success, =1 for failure
C
C***********************************************************************
C
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      CHARACTER(LEN=1) D
      CHARACTER(LEN=11) CFORM
      CHARACTER(LEN=80) HEADER
      INTEGER :: IVTARG(*),ITCHL(*),IV(*),STARG(*),MTARG(*),GUTARG(*),
     1           LCHL(*),MCHL(*)
      DOUBLE PRECISION :: ETARG(*), ECHL(*)
C      DIMENSION ITCHL(NCHAN),ECHL(NCHAN),ETARG(NTARG)
C     1,STARG(NTARG),MTARG(NTARG),GUTARG(NTARG),LCHL(NCHAN),MCHL(NCHAN)
      INTEGER STOT,GUTOT
      DATA KEYCH/10/,TINY/1.D-6/,D/'D'/,TWO/2.D0/
C
C---- FIND REQUIRED SET
      NSET = NCSET
      CALL GETSET(LUCHAN,NSET,KEYCH,CFORM,IFAIL)
      IF(IFAIL.NE.0) GO TO 99
      NCSET = NSET
C
C---- READ TARGET AND CHANNEL DATA
C
      NVDIN = NVIB+NDIS
      IF(CFORM.EQ.'FORMATTED') THEN
        READ(LUCHAN,*) KEY,NSET,NREC,NINFO,NDATA
        IF(KEY.NE.KEYCH) GO TO 99
        READ(LUCHAN,9) HEADER
        READ(LUCHAN,*) NTARG,NVIB,NDIS0,NCHAN0
        NVD = NVIB+NDIS0
        READ(LUCHAN,*) MGVN,STOT,GUTOT,ION,RR,RMASS
        DO J=1,NTARG
          READ(LUCHAN,*) I,MTARG(I),STARG(I),GUTARG(I),ETARG(I)
        END DO
        DO J=1,NCHAN
          READ(LUCHAN,*) I,ITCHL(I),LCHL(I),MCHL(I),ECHL(I)
        END DO
        DO J=NCHAN+1,NCHAN0
          READ(LUCHAN,*) I
        END DO
        IF(NVD.GT.0) THEN
          DO J=1,NVDIN
            READ(LUCHAN,*) I,IVTARG(I),IV(I)
          END DO
          DO J=NVDIN+1,NVD
            READ(LUCHAN,*)
          END DO
        ENDIF
      ELSE
        READ(LUCHAN) KEY,NSET,NREC,NINFO,NDATA
        IF(KEY.NE.KEYCH) GO TO 99
        READ(LUCHAN) HEADER
        READ(LUCHAN) NTARG,NVIB,NDIS0,NCHAN0
        NVD = NVIB+NDIS0
        READ(LUCHAN) MGVN,STOT,GUTOT,ION,RR,RMASS
        DO J=1,NTARG
          READ(LUCHAN) I,MTARG(I),STARG(I),GUTARG(I),ETARG(I)
        END DO
        DO J=1,NCHAN
          READ(LUCHAN) I,ITCHL(I),LCHL(I),MCHL(I),ECHL(I)
        END DO
        DO J=NCHAN+1,NCHAN0
          READ(LUCHAN) I
        END DO
        IF(NVD.GT.0) THEN
          DO J=1,NVDIN
            READ(LUCHAN) I,IVTARG(I),IV(I)
          END DO
          DO J=NVDIN+1,NVD
            READ(LUCHAN)
          END DO
        ENDIF
      ENDIF
C
C---- CHECK GEOMETRY DATA
      IF(ABS(R).GT.TINY.AND.ABS(R-RR).GT.TINY) THEN
        WRITE(IWRITE,96) R,NCSET,RR
        IFAIL = 1
      ELSE
C
C----- PRINT OUT TARGET AND CHANNEL DATA
C
        TWOM = TWO*RMASS
        WRITE(IWRITE,10) NCSET,LUCHAN
        IF(ABS(R).GT.TINY) WRITE(IWRITE,14) R
        IF(IPRNT.GE.0) THEN
          WRITE(IWRITE,11) (I,STARG(I),MTARG(I),GUTARG(I),ETARG(I),
     1    I=1,NTARG)
          IF(NVD.GT.0) WRITE(IWRITE,12) (I,IVTARG(I),IV(I),I=1,NVDIN-
     1                 NDIS)
          WRITE(IWRITE,13) (I,ITCHL(I),LCHL(I),MCHL(I),ECHL(I),
     1                      I=1,NCHAN-NDIS)
          IF(NDIS.GT.0) WRITE(IWRITE,131) (I,D,LCHL(I),MCHL(I),
     1    ECHL(I)/TWOM,I=NCHAN-NDIS+1,NCHAN)
        ENDIF
        R = RR
      ENDIF
C
      RETURN
C
 99   WRITE(IWRITE,98) NCSET,KEYCH,LUCHAN
      IFAIL = 1
      RETURN
C
 9    FORMAT(A80)
 10   FORMAT(/' Target and channel data is read from set',I3,' on unit'
     1,I3)
 11   FORMAT(/' Target state    Spin    M    G/U  Energy (au)'/(2I10,
     12I6,F12.6))
 12   FORMAT(/' Vibronic level    Electronic state    v'/(I10,8X,I10,2X
     1,I10))
 13   FORMAT(/' Channel   Vibronic level   L   M  Threshold (Ryd)'/
     1(I5,5X,I10,7X,I2,2X,I2,F12.6))
 131  FORMAT(I5,14X,A1,7X,I2,2X,I2,F12.6)
 14   FORMAT(/' Bond length',F7.4)
 96   FORMAT(/' Inconsistent geometry input R=',F7.4,' but on NCSET =',
     1I2,' R =',F7.4)
 98   FORMAT(/' Unable to find channel dataset number',I3,'  key =',I3,
     1' on unit',I3)
      END
      SUBROUTINE WRITTH(LUTMT,TITLE,NTSET,MAXCHI,MAXCHF,NETOT,MAXNE,
     1NEREP,EINC,NAPPR,NTARG,IVCHL,LVCHL,MVCHL,EVCHL,R,TFORM0,IPRNT0,
     2IFAIL)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
C***********************************************************************
C
C     WRITTH writes header on T-matrix file on unit LUTMT
C
C***********************************************************************
C
      INTEGER STOT,GUTOT
      CHARACTER(LEN=11) TFORM,TFORM0
      CHARACTER(LEN=80) TITLE
      COMMON/INTFS/ NCHAN,ION,NVIB,NDIS,IWRITE,MGVN,STOT,GUTOT
      DIMENSION TR(MAXCHI,MAXCHF+NDIS,*),TI(MAXCHI,MAXCHF+NDIS,*),
     1 MAXVI(*),MAXVJ(*),MAXVD(*),EN(*),NEREP(MAXNE),EINC(2,MAXNE),
     2 IVCHL(NCHAN),LVCHL(NCHAN),MVCHL(NCHAN),EVCHL(NCHAN)
      DATA zero/0.D0/,KEY/12/
      save
C
C----- Position file at end of information (or end of set number NTSET)
      TFORM = TFORM0
      IPRNT = IPRNT0
      NSET = NTSET
      CALL GETSET(LUTMT,NSET,KEY,TFORM,IFAIL)
      IF(IFAIL.NE.0) RETURN
      IF(NTSET.NE.1) NSET = NSET+1
      WRITE(IWRITE,16) NSET,LUTMT
C
      WRITE(IWRITE,14) MAXCHI,MAXCHF
C
C---- Calculate number of records to be written to T-matrix set.
      IF(TFORM.EQ.'FORMATTED') THEN
        NDATA = NETOT*(2*((MAXCHI*MAXCHF+3)/4)+1)+1+MAXCHF+MAXNE
      ELSE
        NDATA = 3*NETOT+1+MAXCHF+MAXNE
      ENDIF
      NINFO = 1
      NREC = NDATA+NINFO
C
C---- Write header
      IF(TFORM.EQ.'FORMATTED') THEN
        WRITE(LUTMT,210) KEY,NSET,NREC,NINFO,NDATA
        WRITE(LUTMT,13) TITLE
        WRITE(LUTMT,10) MGVN,STOT,GUTOT,NCHAN,NAPPR,MAXNE,NVIB,
     1  NDIS,NTARG,MAXCHI,MAXCHF,R
        DO I=1,MAXCHF
          WRITE(LUTMT,12) IVCHL(I),LVCHL(I),MVCHL(I),EVCHL(I)
        END DO
        DO I=1,MAXNE
          WRITE(LUTMT,19) I,NEREP(I),(EINC(J,I),J=1,2)
        END DO
      ELSE
        WRITE(LUTMT) KEY,NSET,NREC,NINFO,NDATA
        WRITE(LUTMT) TITLE
        WRITE(LUTMT) MGVN,STOT,GUTOT,NCHAN,NAPPR,MAXNE,NVIB,NDIS,
     1  NTARG,MAXCHI,MAXCHF,R
        DO I=1,MAXCHF
          WRITE(LUTMT) IVCHL(I),LVCHL(I),MVCHL(I),EVCHL(I)
        END DO
        DO I=1,MAXNE
          WRITE(LUTMT) I,NEREP(I),(EINC(J,I),J=1,2)
        END DO
      ENDIF
      IF(IPRNT.NE.0) THEN
        WRITE(IWRITE,17)
        WRITE(IWRITE,110) KEY,NSET,NREC,NINFO,NDATA
        WRITE(IWRITE,113) TITLE
        WRITE(IWRITE,110) MGVN,STOT,GUTOT,NCHAN,NAPPR,MAXNE,NVIB,
     1  NDIS,NTARG,MAXCHI,MAXCHF,R
        DO I=1,MAXCHF
          WRITE(IWRITE,112) IVCHL(I),LVCHL(I),MVCHL(I),EVCHL(I)
        END DO
        DO I=1,MAXNE
          WRITE(IWRITE,119) I,NEREP(I),(EINC(J,I),J=1,2)
        END DO
      ENDIF
C
      RETURN
C     
      ENTRY WRITET(LUTMT,NE,MAXCHI,MAXCHF,MAXVI,MAXVJ,MAXVD,TR,TI,EN)
C
C***********************************************************************
C
C     WRITET writes T-matrices to unit LUTMT which has previously been
C     positioned correctly via a call to WRITTH
C
C***********************************************************************
C
      IF(TFORM.EQ.'FORMATTED') THEN
        DO 3 IE=1,NE
        MI = MAXVI(IE)
        MJ = MAXVJ(IE)
        MD = MAXVD(IE)
        WRITE(LUTMT,12) MI,MJ,MD,EN(IE)
        WRITE(LUTMT,11) ((TR(I,J,IE),I=1,MI),J=1,MJ+MD)
        WRITE(LUTMT,11) ((TI(I,J,IE),I=1,MI),J=1,MJ+MD)
 3      CONTINUE
      ELSE
        DO 2 IE=1,NE
        MI = MAXVI(IE)
        MJ = MAXVJ(IE)
        MD = MAXVD(IE)
        WRITE(LUTMT) MI,MJ,MD,EN(IE)
        WRITE(LUTMT) ((TR(I,J,IE),I=1,MI),J=1,MJ+MD)
        WRITE(LUTMT) ((TI(I,J,IE),I=1,MI),J=1,MJ+MD)
 2      CONTINUE
      ENDIF
      IF(IPRNT.NE.0) THEN
        WRITE(IWRITE,18)
        DO 8 IE=1,NE
        MI = MAXVI(IE)
        MJ = MAXVJ(IE)
        MD = MAXVD(IE)
        WRITE(IWRITE,112) MI,MJ,MD,EN(IE)
        WRITE(IWRITE,111) ((TR(I,J,IE),I=1,MI),J=1,MJ+MD)
        WRITE(IWRITE,111) ((TI(I,J,IE),I=1,MI),J=1,MJ+MD)
 8      CONTINUE
      ENDIF
C
      RETURN
 210  FORMAT (10I20)
 10   FORMAT(11I5,F15.8)
 11   FORMAT(4E20.13)
 12   FORMAT(3I5,F10.6)
 13   FORMAT(A60,A20)
 110  FORMAT(1X,11I5,F15.8)
 111  FORMAT(1X,4E20.13)
 112  FORMAT(1X,3I5,F10.6)
 113  FORMAT(1X,A60,A20)
 14   FORMAT(/' Subset of T-matrix which will be saved is',
     * '  (( T(i,j), i = 1,',I3,' ) ,j = 1,',I3,' )')
 15   FORMAT(10F8.5)
 16   FORMAT(/' T-matrices will be written to set',I3,' on unit',I3)
 17   FORMAT(/' Header data written to LUTMT')
 18   FORMAT(/' Energy dependant data written to LUTMT')
 19   FORMAT(2I5,2F15.8)
 119  FORMAT(1X,2I5,2F15.8)
      END
      SUBROUTINE GETSET(LUNIT,NSET,INKEY,FORM,IFAIL)
C
C***********************************************************************
C
C     GETSET locates set number NSET on unit LUNIT
C     If NSET = 0 file is positioned at end-of-information
C             = 1 the file is opened
C             = n file is positioned at the beginning of set number n
C     On return NSET = sequence number of current set 
C
C***********************************************************************
C
      LOGICAL OP
      CHARACTER(LEN=11) FORM
      CHARACTER(LEN=80) LINE
C
      IFAIL=0
      INQUIRE(UNIT=LUNIT,OPENED=OP)
      IF(.NOT.OP) OPEN(UNIT=LUNIT,ERR=99,FORM=FORM,status='unknown')
      IF(NSET.EQ.1) THEN
        REWIND(UNIT=LUNIT,ERR=100)
 100    RETURN
      ELSE
        REWIND LUNIT
        INSET = NSET
        ITIME = 1
C
C------ LOCATE SET NUMBER INSET
 5      IF(FORM.EQ.'FORMATTED') THEN
 15       READ(LUNIT,*,END=9) KEY,NSET,NREC
          IF(NSET.EQ.INSET.AND.KEY.EQ.INKEY) THEN
            BACKSPACE LUNIT
            RETURN
          ELSE
            DO I=1,NREC
              READ(LUNIT,*,END=99)
            END DO
          ENDIF
          GO TO 15
        ELSE
 25       READ(LUNIT,END=9) KEY,NSET,NREC
          IF(NSET.EQ.INSET.AND.KEY.EQ.INKEY) THEN
            BACKSPACE LUNIT
            RETURN
          ELSE
            DO I=1,NREC
              READ(LUNIT,END=99)
            END DO
          ENDIF
        GO TO 25
        ENDIF
C
C------ END OF FILE HAS BEEN REACHED
 9      IF(INSET.EQ.0) THEN
          BACKSPACE LUNIT
          RETURN
        ELSE IF(ITIME.EQ.1) THEN
          ITIME = 2
          REWIND LUNIT
          GO TO 5
        ENDIF
      ENDIF
C
 99   IFAIL = 1
      RETURN
      END
      SUBROUTINE WRITRM(LURMT,NRSET,RFORM,nstat,NOCSF,ISMAX,NPOLE,CF,
     1   EIG,WAMP,VEC,BCOEF,NTARG,R,RMASS,RMATR,IBUT,ezero,sfac,iex,
     2   ecex,rcex,HEADER,IPRNT,IFAIL)
C
C***********************************************************************
C
C     WRITRD writes data required to assemble R-matrices to set number
C     NRSET on unit LURMT
C
C***********************************************************************
C
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      CHARACTER(LEN=11) RFORM
      CHARACTER(LEN=80) HEADER
      COMMON/INTFS/ NCHAN,ION,NVIB,NDIS,IWRITE,MGVN,STOT,GUTOT
      DIMENSION CF(NCHAN*(NCHAN+1)/2,*),EIG(nstat),WAMP(NCHAN,nstat),
     1BCOEF(3,*),VEC(NOCSF,*),sfac(nchan),ecex(iex),rcex(nchan,iex)
      INTEGER STOT,GUTOT
      DATA KEYRM/11/
C
C---- FIND REQUIRED SET
      NSET = NRSET
      CALL GETSET(LURMT,NSET,KEYRM,RFORM,IFAIL)
      IF(IFAIL.NE.0) GO TO 99
      IF(NRSET.EQ.0) NSET = NSET+1
      WRITE(IWRITE,16) RFORM,NSET,LURMT,HEADER
C
      NCHSQ = NCHAN*(NCHAN+1)/2
      NREC = 4
      IF(RFORM.EQ.'FORMATTED') THEN
        IF(ISMAX.GT.0) NREC = NREC+(NCHSQ*ISMAX+3)/4
        NREC = NREC+(nstat+3)/4
        NREC = NREC+(NCHAN*nstat+3)/4
        IF(NPOLE.GT.0) NREC = NREC+(NOCSF*NPOLE+3)/4
        IF(IBUT.GT.0) nrec = nrec+(3*nchan+3)/4
        IF(abs(ibut).gt.1) 
     *              nrec=nrec+1+(nchan+3)/4+(iex+3)/4+(nchan*iex+3)/4
      ELSE
        NREC = NREC+2
        IF(ISMAX.GT.0) NREC=NREC+1
        IF(NPOLE.GT.0) NREC=NREC+1
        IF(IBUT.GT.0) NREC=NREC+1
        if(abs(ibut).gt.1) NREC=NREC+4
      ENDIF
      NINFO = 1
      NDATA = NREC-NINFO
      IF(RFORM.EQ.'FORMATTED') THEN
        WRITE(LURMT,210) KEYRM,NSET,NREC,NINFO,NDATA
        WRITE(LURMT,11) HEADER
        WRITE(LURMT,10) NTARG,NVIB,NDIS,NCHAN
        WRITE(LURMT,12) MGVN,STOT,GUTOT,ION,R,RMASS
        WRITE(LURMT,12) ISMAX,nstat,NPOLE,IBUT,RMATR
        if (abs(ibut).gt.1) write(lurmt,13) nocsf,ezero,iex
        IF(ISMAX.GT.0) WRITE(LURMT,14) ((CF(I,K),I=1,NCHSQ),K=1,ISMAX)
        WRITE(LURMT,14) (EIG(I),I=1,nstat)
        WRITE(LURMT,14) ((WAMP(I,J),I=1,NCHAN),J=1,nstat)
        IF(NPOLE.GT.0) WRITE(LURMT,14) ((VEC(I,J),I=1,NOCSF),J=1,NPOLE)
        IF(IBUT.GT.0) WRITE(LURMT,14) ((BCOEF(I,J),I=1,3),J=1,NCHAN)
        IF(abs(ibut).gt.1) WRITE(LURMT,14) (sfac(J),J=1,NCHAN)
        IF(abs(ibut).gt.1) WRITE(LURMT,14) (ecex(j),j=1,iex)
        IF(abs(ibut).gt.1)WRITE(LURMT,14)((rcex(i,j),I=1,NCHAN),J=1,iex)
      ELSE
        WRITE(LURMT) KEYRM,NSET,NREC,NINFO,NDATA
        WRITE(LURMT) HEADER
        WRITE(LURMT) NTARG,NVIB,NDIS,NCHAN
        WRITE(LURMT) MGVN,STOT,GUTOT,ION,R,RMASS
        WRITE(LURMT) ISMAX,nstat,NPOLE,IBUT,RMATR
        if (abs(ibut).gt.1) write(lurmt) nocsf,ezero,iex
        IF(ISMAX.GT.0) WRITE(LURMT) ((CF(I,K),I=1,NCHSQ),K=1,ISMAX)
        WRITE(LURMT) (EIG(I),I=1,nstat)
        WRITE(LURMT) ((WAMP(I,J),I=1,NCHAN),J=1,nstat)
        IF(NPOLE.GT.0) WRITE(LURMT) ((VEC(I,J),I=1,NOCSF),J=1,NPOLE)
        IF(IBUT.gt.0) WRITE(LURMT) ((BCOEF(I,J),I=1,3),J=1,NCHAN)
        IF(abs(ibut).gt.1) WRITE(LURMT) (sfac(J),J=1,NCHAN)
        IF(abs(ibut).gt.1) WRITE(LURMT) (ecex(j),j=1,iex)
        IF(abs(ibut).gt.1) WRITE(LURMT)((rcex(i,j),I=1,NCHAN),J=1,iex)
      ENDIF
C
      IF(IPRNT.NE.0) THEN
        WRITE(IWRITE,17)
        WRITE(IWRITE,100) KEYRM,NSET,NREC,NINFO,NDATA
        WRITE(IWRITE,110) HEADER
        WRITE(IWRITE,100) NTARG,NVIB,NDIS,NCHAN
        WRITE(IWRITE,120) MGVN,STOT,GUTOT,ION,R,RMASS
        WRITE(IWRITE,120) ISMAX,nstat,NPOLE,IBUT,RMATR
        IF(abs(ibut).gt.1)  WRITE(IWRITE,130) nocsf,ezero,iex
        IF(ISMAX.GT.0) WRITE(IWRITE,140) ((CF(I,K),I=1,NCHSQ),K=1,ISMAX)
        WRITE(IWRITE,140) (EIG(I),I=1,nstat)
        WRITE(IWRITE,140) ((WAMP(I,J),I=1,NCHAN),J=1,nstat)
        IF(NPOLE.GT.0) WRITE(IWRITE,140)((VEC(I,J),I=1,NOCSF),J=1,NPOLE)
        IF(IBUT.gt.0) WRITE(IWRITE,140) ((BCOEF(I,J),I=1,3),J=1,NCHAN)
        IF(abs(ibut).gt.1) WRITE(IWRITE,140) (sfac(J),J=1,NCHAN)
        IF(abs(ibut).gt.1) WRITE(LURMT,140) (ecex(j),j=1,iex)
        IF(abs(ibut).gt.1) WRITE(LURMT,140)
     *                                 ((rcex(i,j),I=1,NCHAN),J=1,iex)
      ENDIF
C
      RETURN
C
 99   WRITE(IWRITE,98) NRSET,KEYRM,LURMT
      IFAIL = 1
      RETURN
 10   FORMAT(16I5)
 11   FORMAT(A80)
 12   FORMAT(4I5,2D20.13)
 13   format(i10,d20.13,i5)
 14   FORMAT(4D20.13)
 16   FORMAT(/' R-matrix data will be written ',A11,' to set number',
     1I3,' on unit number',I3/' Header :',A80)
 17   FORMAT(/' Data written to LURMT :'/)
 98   FORMAT(/' Unable to find R-matrix dataset number',I3,'  key =',I3,
     1' on unit',I3)
 100  FORMAT(1X,16I5)
 110  FORMAT(1X,A80)
 120  FORMAT(1X,4I5,2D20.13)
 130  format(1x,i5,d20.13,i5)
 140  FORMAT(1X,4D20.13)
 210  FORMAT(10I7)
      END
      SUBROUTINE MATTPT (N,A,IPRINT)
      IMPLICIT DOUBLE PRECISION(A-H,O-Z)
C
C***********************************************************************
C
C     PRINTS 2-DIMENSIONAL SYMMETIC MATRIX OF DIMENSION N, THE LOWER
C     TRIANGLE OF WHICH IS STORED IN A (PRESENT CODE IS INEFFICIENT)
C
C***********************************************************************
C
      DIMENSION A(*)
      DATA NCOL/8/
C
      K1 = 1
      K2 = NCOL
C
 1    K = 0
      DO L=1,N
        KL = K+L
        KL1 = K+K1
        KL2 = MIN(K+K2,KL)
        IF(KL1.LE.KL2) WRITE(IPRINT,1001) (A(J),J=KL1,KL2)
        K = KL
      END DO
C
 2    IF (N.LE.K2) RETURN
      K1 = K2 + 1
      K2 = K2 + NCOL
      WRITE(IPRINT,1002)
      GO TO 1
C
C     FORMAT STATEMENTS -
C
 1001 FORMAT(8D15.6)
 1002 FORMAT(/)
C
      END
      SUBROUTINE READAIH(LUAI0,NAISET,NBOUND,
     1NCHAN,NVCHAN,ISMAX,AIFORM0,IPRNT0,IWRIT0,IFAIL)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
C***********************************************************************
C
C     READAIH locates and reads asymptotic data set NAISET on unit LUAI
C
C***********************************************************************
C
      CHARACTER(LEN=11) AIFORM,AIFORM0
      CHARACTER(LEN=80) TITLE
      DIMENSION IPFLG(*),ETHR(*),ICHL(*),LCHL(*),MCHL(*),CF(*),
     1E(*),AMPS1(*)
      DATA KEYBC/11/
      SAVE
C
      AIFORM = AIFORM0
      LUAI = LUAI0
      IPRNT = IPRNT0
      IWRITE= IWRIT0
C
C---- Locate set number NAISET on unit LUAI
      NSET = NAISET
      CALL GETSET(LUAI,NSET,KEYBC,AIFORM,IFAIL)
      IF(IFAIL.NE.0) GO TO 99
C
C---- Read header
      IF(AIFORM.EQ.'FORMATTED') THEN
        READ(LUAI,10) KEYBC,NSET,NREC,NINFO,NDATA
        READ(LUAI,13) TITLE
        READ(LUAI,10) NBOUND,NCHAN,NVCHAN,ISMAX
      ELSE
        READ(LUAI) KEYBC,NSET,NREC,NINFO,NDATA
        READ(LUAI) TITLE
        READ(LUAI) NBOUND,NCHAN,NVCHAN,ISMAX
      ENDIF
C
C---- Print header information
      IF(IPRNT.NE.0)  THEN
        WRITE(IWRITE,14)
        WRITE(IWRITE,100) KEYBC,NSET,NREC,NINFO,NDATA
        WRITE(IWRITE,130) TITLE
        WRITE(IWRITE,100) NBOUND,NCHAN,NVCHAN,ISMAX
      ENDIF
C
      RETURN
C
      ENTRY READAI1(ION,IASY,IWRON,MAXPTS,
     1EPS,RAFEND,RAFIN,DEGENY,EWRON,RMATR,HX,TOL,ETHR,ICHL,LCHL,MCHL,CF,
     1IPFLG,NCHAN,NVCHAN,ISMAX)
C
      IF(AIFORM.EQ.'FORMATTED') THEN
        READ(LUAI,10) ION,IASY,IWRON,MAXPTS
        READ(LUAI,11) EPS,RAFEND,RAFIN,DEGENY,EWRON,RMATR,
     1                 HX,TOL
        READ(LUAI,10) (IPFLG(I),I=1,10)
        READ(LUAI,10) (ICHL(I),I=1,NCHAN)
        READ(LUAI,10) (LCHL(I),I=1,NCHAN)
        READ(LUAI,10) (MCHL(I),I=1,NCHAN)
        READ(LUAI,11) (ETHR(I),I=1,NCHAN)
        READ(LUAI,11) (CF(I),I=1,ISMAX*NCHAN*NCHAN)
      ELSE
        READ(LUAI) ION,IASY,IWRON,MAXPTS
        READ(LUAI) EPS,RAFEND,RAFIN,DEGENY,EWRON,RMATR,
     1             HX,TOL
        READ(LUAI) (IPFLG(I),I=1,10)
        READ(LUAI) (ICHL(I),I=1,NCHAN)
        READ(LUAI) (LCHL(I),I=1,NCHAN)
        READ(LUAI) (MCHL(I),I=1,NCHAN)
        READ(LUAI) (ETHR(I),I=1,NCHAN)
        READ(LUAI) (CF(I),I=1,ISMAX*NCHAN*NCHAN)
      ENDIF
      IF(IPRNT.NE.0) THEN
       WRITE(IWRITE,15)
       WRITE(IWRITE,100) ION,IASY,IWRON,MAXPTS
       WRITE(IWRITE,110) EPS,RAFEND,RAFIN,DEGENY,EWRON,RMATR,
     1                   HX,TOL
       WRITE(IWRITE,100) (IPFLG(I),I=1,10)
       WRITE(IWRITE,100) (ICHL(I),I=1,NCHAN)
       WRITE(IWRITE,100) (LCHL(I),I=1,NCHAN)
       WRITE(IWRITE,100) (MCHL(I),I=1,NCHAN)
       WRITE(IWRITE,110) (ETHR(I),I=1,NCHAN)
       WRITE(IWRITE,110) (CF(I),I=1,ISMAX*NCHAN*NCHAN)
      ENDIF
C
      RETURN
      ENTRY READAI2(NBOUND,E,AMPS1)
C
      DO 146 II=1,NBOUND
        IF(AIFORM.EQ.'FORMATTED') THEN
          READ(LUAI,11) E(II),AMPS1(II)
        ELSE
          READ(LUAI) E(II),AMPS1(II)
        ENDIF
        IF(IPRNT.NE.0) THEN
         WRITE(IWRITE,15)
         WRITE(IWRITE,110) E(II),AMPS1(II)
        ENDIF
 146  CONTINUE
C
      RETURN
C
 99   WRITE(IWRITE,98) NAISET,LUAI
      IFAIL = 1
      RETURN
C
 11   FORMAT(10F20.13)
 10   FORMAT(10I5)
 13   FORMAT(A80)
 14   FORMAT(/' Header on LUAI')
 15   FORMAT(/' Body on LUAI')
 100  FORMAT(1X,10I5)
 110  FORMAT(1X,4E20.13)
 130  FORMAT(1X,A80)
 98   FORMAT(/' UNABLE TO FIND ASYMPTOTIC INFO. SET',I3,' ON UNIT',I3)
      END
      SUBROUTINE WRITTDH(TDFORM,TITLE,NTDSET,LUTRD,IWRITE,NBSET1,NBSET2,
     1LUBND1,LUBND2,NBOUND1,NBOUND2,MGVN1,MGVN2,STOT1,STOT2,GUTOT1,
     1GUTOT2,NSTAT1,NSTAT2,RR1,RR2,NDATA,IFAIL,IPRNT)
C**********************************************************************
C     WRITTDH  writes header of transition dipoles
C     If NTDSET=1 on input then BC are written as first set,
C               0 then written at end of information (eoi).
C     On output NTDSET holds the actual set number
C**********************************************************************
C
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      INTEGER STOT1,GUTOT1,STOT2,GUTOT2
      CHARACTER(LEN=11)  TDFORM
      CHARACTER(LEN=80)  TITLE
      DATA KEYBC/11/
      SAVE
C
C     position file at eoi or end of set number NTDSET
      NSET=NTDSET
      CALL GETSET(LUTRD,NSET,KEYBC,TDFORM,IFAIL)
      IF(IFAIL.NE.0) THEN
        WRITE(IWRITE,666)NTDSET,LUTRD
        RETURN
      ENDIF
      IF(NTDSET.NE.1)  NSET=NSET+1
      WRITE(IWRITE,16) NSET,LUTRD
      NTDSET=NSET
      NINFO = 5
C
C  write header
      WRITE(IWRITE,100)NBSET1,NBSET2,LUBND1,LUBND2
      IF (IPRNT.EQ.1) THEN
         WRITE(IWRITE,101)
      ENDIF
      IF(TDFORM.EQ.'FORMATTED')  THEN
         NREC = NDATA+NINFO
         WRITE(LUTRD,10) KEYBC,NSET,NREC,NINFO,NDATA
         WRITE(LUTRD,13) TITLE
         WRITE(LUTRD,10) NBOUND1,MGVN1,STOT1,GUTOT1,NSTAT1
         WRITE(LUTRD,11) RR1
         WRITE(LUTRD,10) NBOUND2,MGVN2,STOT2,GUTOT2,NSTAT2
         WRITE(LUTRD,11) RR2
      ELSE
         NREC = NINFO+NDATA
         WRITE(LUTRD) KEYBC,NSET,NREC,NINFO,NDATA
         WRITE(LUTRD) TITLE
         WRITE(LUTRD) NBOUND1,MGVN1,STOT1,GUTOT1,NSTAT1
         WRITE(LUTRD) RR1
         WRITE(LUTRD) NBOUND2,MGVN2,STOT2,GUTOT2,NSTAT2
         WRITE(LUTRD) RR2
      ENDIF
C
      RETURN
 666  FORMAT(/' UNABLE TO FIND TRANSITION DIPOLE SET',I3,' ON UNIT',I3)
  10  FORMAT(10I5)
  11  FORMAT(10F20.13)
  13  FORMAT(A80)
  16  FORMAT(/' Transition dipoles and absorption oscillator strengths',
     1/' will be written to set',I3,' on unit',I3)
 100  FORMAT(
     1/'----------------------------------------------------------------
     1---------',
     1/'    INITIAL     |      FINAL      |',
     1/'----------------------------------------------------------------
     1---------',
     1/'SET NUMBER ',I4,' |',I16,' |',
     1/'ON UNIT ',I7,' |',I16,' | TRANSITION | OSCILLATOR |',
     1/'WFN.NO.  ENERGY | WFN.NO.  ENERGY |   DIPOLE   |  STRENGTH  |
     1 V**3 *F',
     1/'----------------------------------------------------------------
     1---------')
 101  FORMAT(
     1/'                                  INNER    OUTER',
     1/'----------------------------------------------------------------
     1---------')
      END
      SUBROUTINE RDTMTH(LUMOM,NMSET,NSTAT1,NSTAT2,NPROP,IOPCDE,
     1        MGVN1,MGVN2,GUTOT1,GUTOT2,STOT1,STOT2,RAB,
     2        IPRNT,IWRITE,IFAIL)
C
C***************************************************************
C     RDTMTH reads the header from UNIT LUMOM and checks the data
C     is consistant. Dipole and other moments computed by TMT are
C     read by entry RDTMT
C***************************************************************
C
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      PARAMETER (MAXOPT=20)
      CHARACTER(LEN=11) TFORM
      CHARACTER(LEN=4) NAME(30)
      LOGICAL SWAP
      INTEGER GUTOT1,GUTOT2,STOT1,STOT2,SYMTYP,GUTOTI,GUTOTJ
      DIMENSION TMTM(NSTAT1*NSTAT2,NPROP),NOB(20),IOPCDE(7,NPROP),
     1          IOPCDS(8,MAXOPT),GEONUC(20),CHARG(20),IPT(MAXOPT)
      SAVE 
      DATA KEYT/50/,TFORM/'UNFORMATTED'/,TINY/1.D-8/
C
C---- FIND REQUIRED SET
      NSET = NMSET
      CALL GETSET(LUMOM,NSET,KEYT,TFORM,IFAIL)
      IF(IFAIL.NE.0) GO TO 99
      NMSET = NSET
C
C---- READ HEADER INFORMATION
C
      READ(LUMOM) KEY,NMSET,NREC,NOPREC,NSTATI,NSTATJ,ISYM,NUCCEN
      WRITE(IWRITE,101) NMSET,LUMOM,ISYM,NSTATI,NSTATJ,NUCCEN
 101  FORMAT(/' Transition moments will be read from set',I3,' on unit',
     1   I3,/,' with ISYM =',I2,' NSTAT1 =',I4,' NSTAT2  =',I4,
     2   ' NUCCEN =',I2)
      IF(KEY.NE.KEYT) GO TO 99
C
      READ(LUMOM) NPT,((IOPCDS(I,J),I=1,8),J=1,NPT)
      IF (IPRNT.GT.0) WRITE(IWRITE,12) NPT,((IOPCDS(I,J),I=1,8),J=1,NPT)
 12   FORMAT(/,7X,'Number of property operators, NPROP =',I3,
     *       /,7X,'Property operator codes, IOPCDS, are as follows:',
     *       /,7X,'  C  N  I  J  K  L  M  *'/(7X,8I3))
C     Check that required moments are present.
      DO 10 K=1,NPROP
      IOPCDE(5,K)=NUCCEN
      DO 20 J=1,NPT
      DO 30 I=1,7
      IF (IOPCDE(I,K) .NE. IOPCDS(I,J)) GOTO 20
   30 CONTINUE
      IPT(K)=J
      GOTO 10
   20 CONTINUE
      WRITE(IWRITE,97)
   97 FORMAT(/,'  Requested moments not present on LUMOM')
      WRITE(IWRITE,96) 'Requested',((IOPCDE(I,J),I=1,7),J=1,NPROP)
      WRITE(IWRITE,96) 'From TMT ',((IOPCDS(I,J),I=1,7),J=1,NPT)
   96 FORMAT(2X,A9,(3X,7I2))
      IFAIL=1
      RETURN
   10 CONTINUE
C
C-----OBTAIN and check PRELIMINARY INFORMATION FROM HEADER
C
      READ(LUMOM) NORB,NSRB,NELT,NSYM,SYMTYP,ISPIN,(NOB(I),I=1,NSYM),
     *            NNUC,(GEONUC(I),I=1,NNUC),(CHARG(I),I=1,NNUC)
C     Compute the bondlength: and check that it is correct
      IF (NUCCEN .EQ. 3) R = ABS(GEONUC(1)-GEONUC(2))
      IF (NUCCEN .EQ. 2) R = ABS(GEONUC(1)-GEONUC(3))
      IF (NUCCEN .EQ. 1) R = ABS(GEONUC(3)-GEONUC(2))
      IF(IPRNT.GE.0) WRITE(IWRITE,106) R
 106  FORMAT(/' Data for R =',F6.3)
      IF (ABS(RAB-R).GT.TINY) THEN
         WRITE(IWRITE,90) RAB,R
         IFAIL = 1
         RETURN
      ENDIF
 90   FORMAT(/' INCONSISTENCIES IN INPUT DATA DETECTED IN RDTMT :',
     1        ' Geometry data does not match',
     2       /' Required R =',F10.5,' read R =',F10.5)
      IF (ISPIN.NE.STOT1 .OR. ISPIN.NE.STOT2) THEN
         WRITE(IWRITE,95) ISPIN,STOT1,STOT2
         IFAIL = 1
         RETURN
      ENDIF
 95   FORMAT(/' INCONSISTENCIES IN INPUT DATA DETECTED IN RDTMT',/,
     1       /' Spin data does not match that input or attempt',
     2       /' to analyse spin non-conserving transition',
     3       /' Input,      STOT1 =',I2,' STOT2 =',I2,
     4       /' From TMT,   STOT  =',I2)
C
      READ(LUMOM) NAME,MGVNI,IREFLI,GUTOTI,NCSFI
      IF(IPRNT.GE.0) WRITE(IWRITE,105) 1,MGVNI,ISPIN,GUTOTI,NAME
 105  FORMAT(/' Symmetry data for state',I2,' MGVN =',I2,3X,'STOT =',I2,
     1       3X,'GUTOT =',I2,/,' Title: ',30A4)
      IF (ISYM.NE.0) THEN
C
         READ(LUMOM) NAME,MGVNJ,IREFLJ,GUTOTJ,NCSFJ
         IF(IPRNT.GE.0) WRITE(IWRITE,105) 2,MGVNJ,ISPIN,GUTOTJ,NAME
      ELSE
C     Are the Bras and Kets matched up correctly ?
        SWAP=.FALSE.
        IF (MGVN1.EQ.MGVNI .AND. GUTOT1.EQ.GUTOTI .AND.
     1      NSTAT1.EQ.NCSFI) RETURN
      ENDIF
      IF (MGVN1.EQ.MGVNI .AND. GUTOT1.EQ.GUTOTI .AND.
     1      NSTAT1.EQ.NCSFI) THEN
        SWAP=.FALSE.
        IF (MGVN2.EQ.MGVNJ .AND. GUTOT2.EQ.GUTOTJ .AND.
     1      NSTAT2.EQ.NCSFJ) GOTO 50
      ELSE
        SWAP=.TRUE.
        IF (MGVN2.EQ.MGVNI .AND. GUTOT2.EQ.GUTOTI .AND.
     1      NSTAT2.EQ.NCSFI .AND. MGVN1.EQ.MGVNJ .AND.
     2      GUTOT1.EQ.GUTOTJ .AND. NSTAT1.EQ.NCSFJ) GOTO 50
      ENDIF
      GOTO 93
C
  50  IF(IPRNT.GE.0 .AND. .NOT. SWAP) WRITE(IWRITE,102) 1,2
      IF(IPRNT.GE.0 .AND.       SWAP) WRITE(IWRITE,102) 2,1
  102 FORMAT(' Wavefunction 1 matched with wavefunction',I2,' from TMT',
     1      /' Wavefunction 2 matched with wavefunction',I2,' from TMT')
      RETURN
C
C     ENTRY which reads the actual transition moments
C
      ENTRY RDTMT(LUMOM,TMTM,NSTAT1,NSTAT2,NPROP,IOPCDE,IPRNT,IWRITE)
C
C     First: how do the Bras and Kets match up?
C
      IF (SWAP) THEN
         NSTATI=NSTAT2
         NSTATJ=NSTAT1
      ELSE
         NSTATI=NSTAT1
         NSTATJ=NSTAT2
      ENDIF
      DO I=1,NSTATI
        DO J=1,NSTATJ
          IF (SWAP) THEN
             II=NSTAT1*(I-1)+J
          ELSE
             II=NSTAT1*(J-1)+I
          END IF
          READ(LUMOM) ISTI,ISTJ,EDIFF
          KK=1
          DO K=1,NPT
            IF (K.EQ.IPT(KK) .AND. KK.LE.NPROP) THEN
               READ(LUMOM) IOUTW2,IOUTW1,TMTM(II,KK)
               KK=KK+1
            ELSE
               READ(LUMOM)
            END IF
          END DO
        END DO
      END DO
C     Print the moments: to be used with caution!
      IF (IPRNT.GE.2) THEN
        DO K=1,NPROP
          WRITE(IWRITE,107) (IOPCDE(I,K),I=1,7)
          IE=0
          DO J=1,NSTAT2
            IB=IE+1
            IE=IE+NSTAT1
            WRITE(IWRITE,108) (TMTM(II,K),II=IB,IE)
          END DO
        END DO
      ENDIF
  107 FORMAT(//,' Transition moments for property',3X,7I2/)
  108 FORMAT(10D13.7)
      RETURN
C
 99   WRITE(IWRITE,98) NMSET,KEYT,LUMOM
 98   FORMAT(/' Unable to find channel dataset number',I3,'  key =',I3,
     1' on unit',I3)
      IFAIL = 1
      RETURN
 93   WRITE(IWRITE,94)
 94   FORMAT(/' INCONSISTENCIES IN INPUT DATA DETECTED IN RDTMT:',
     1        ' Wavefunction data does not match')
      IF (ISYM .EQ. 0) THEN
        WRITE(IWRITE,92) MGVN1,GUTOT1,NSTAT1,MGVNI,GUTOTI,NCSFI
  92  FORMAT(' Input:    MGVN =',I2,' GUTOT =',I2,' NSTAT =',I5,
     1      /' From TMT: MGVN =',I2,' GUTOT =',I2,' NSTAT =',I5)
      ELSE
        WRITE(IWRITE,91) MGVN1,GUTOT1,NSTAT1,MGVN2,GUTOT2,NSTAT2,
     1                   MGVNI,GUTOTI,NCSFI,MGVNJ,GUTOTJ,NCSFJ
  91  FORMAT(' Input    state 1: MGVN =',I2,' GUTOT =',I2,' NSTAT =',I5,
     1      /'          state 2: MGVN =',I2,' GUTOT =',I2,' NSTAT =',I5,
     2      /' From TMT state I: MGVN =',I2,' GUTOT =',I2,' NSTAT =',I5,
     3      /'          state J: MGVN =',I2,' GUTOT =',I2,' NSTAT =',I5)
      ENDIF
      IFAIL = 1
      RETURN
      END
      SUBROUTINE READBH(LUBND0,NBSET,NCHAN,MGVN,STOT,GUTOT,NSTAT,
     1NBOUND,RR,BFORM0,IPRNT0,IWRIT0,IFAIL)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
C***********************************************************************
C
C     READBH locates and reads bound state set NBSET on unit LUBND
C
C***********************************************************************
C
      INTEGER STOT,GUTOT
      CHARACTER(LEN=11) BFORM,BFORM0
      CHARACTER(LEN=80) TITLE
      DIMENSION COEF(*),ETOT(*),VTEMP(*),XVEC(*)
      DATA KEYBC/11/
      SAVE
C
      IFAIL=0
      BFORM = BFORM0
      LUBND = LUBND0
      IPRNT = IPRNT0
      IWRITE= IWRIT0
C
C---- Locate set number NTSET on unit LUBND
      NSET = NBSET
      CALL GETSET(LUBND,NSET,KEYBC,BFORM,IFAIL)
      IF(IFAIL.NE.0) GO TO 99
C
C---- Read header
      IF(BFORM.EQ.'FORMATTED') THEN
        READ(LUBND,10)  KEYBC,NSET
        READ(LUBND,13) TITLE
        READ(LUBND,10) NBOUND,MGVN,STOT,GUTOT,NSTAT,NCHAN
        READ(LUBND,11) RR
      ELSE
        READ(LUBND) KEYBC,NSET
        READ(LUBND) TITLE
        READ(LUBND) NBOUND,MGVN,STOT,GUTOT,NSTAT,NCHAN
        READ(LUBND) RR
      ENDIF
C
C---- Print header information
      IF(IPRNT.NE.0)  THEN
        WRITE(IWRITE,14)
        WRITE(IWRITE,100) KEYBC,NSET
        WRITE(IWRITE,130) TITLE
        WRITE(IWRITE,100) NBOUND,MGVN,STOT,GUTOT,NSTAT,NCHAN
        WRITE(IWRITE,110) RR
      ENDIF
C
      RETURN
C
      ENTRY READBC(NSTAT,ETOT,VTEMP,COEF,NBOUND,NCHAN,XVEC)
C
      DO 146 I=1,NBOUND
        IF(BFORM.EQ.'FORMATTED') THEN
          READ(LUBND,11) ETOT(I),VTEMP(I),(COEF((I-1)*NSTAT+J),
     1J=1,NSTAT)
          READ(LUBND,11) (XVEC((I-1)*NCHAN+J),J=1,NCHAN)
        ELSE
         READ(LUBND) ETOT(I),VTEMP(I),(COEF((I-1)*NSTAT+J),J=1,NSTAT)
         READ(LUBND) (XVEC((I-1)*NCHAN+J),J=1,NCHAN)
        ENDIF
        IF(IPRNT.NE.0) THEN
           WRITE(IWRITE,110) ETOT(I),VTEMP(I),
     1                       (COEF((I-1)*NSTAT+J),J=1,NSTAT)
           WRITE(IWRITE,110) (XVEC((I-1)*NCHAN+J),J=1,NCHAN)
        ENDIF
 146  CONTINUE
C
      RETURN
C
 99   WRITE(IWRITE,98) NBSET,LUBND
      IFAIL = 1
      RETURN
C
 11   FORMAT(10F20.13)
 10   FORMAT(10I5)
 13   FORMAT(A80)
 14   FORMAT(/' Header on LUBND')
 100  FORMAT(1X,10I5)
 110  FORMAT(1X,4E20.13)
 130  FORMAT(1X,A80)
 98   FORMAT(/' UNABLE TO FIND BOUND STATE SET',I3,' ON UNIT',I3)
      END
      SUBROUTINE WRTMT(IWRITE,NSTAT1,NSTAT2,TMTM)
C
C*******************************************************************
C
C     WRTMT PRINTS OUT THE INPUT TRANSITION DIPOLES
C
C*******************************************************************
C
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DIMENSION TMTM(NSTAT1,NSTAT2)
      WRITE(IWRITE,*)'MATRIX OF TRANSITION MOMENTS IS:  '
      DO 1 I=1,NSTAT1
        WRITE(IWRITE,*)'I = ',I
        WRITE(IWRITE,100)(TMTM(I,J),J=1,NSTAT2)
 1    CONTINUE
 100  FORMAT(8D15.6)
      RETURN
      END
      SUBROUTINE RDTGSB(IWRITE,LUTARG,NTSET,R,NDMOM,IMTGTM,TGTM,
     1NUCCEN,ISMAX,IPRNT,IFAIL)
C
C***********************************************************************
C
C     RDTGSB READS TARGET STATE DATA FROM A DUMPFILE ATTACHED TO UNIT
C            LUTARG
C
C     THE DUMPFILE IS DIVIDED INTO DATASETS BY HEADER CARDS BEGINNING
C     WITH KEY = 9
C
C     ALL RECORDS HAVE A FIXED FORMAT ( LRECL = 80 ) :
C
C                   I1, 7I3, D20.12, 2X, A26
C
C     KEY = RECORD KEY = FIRST FIELD (I1)
C
C***********************************************************************
C
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      PARAMETER (MAXTGT=10)
      INTEGER GTARG,STARG
      CHARACTER(LEN=26) HEAD
      DIMENSION MTARG(MAXTGT),STARG(MAXTGT),ITARG(MAXTGT),
     1 GTARG(MAXTGT),ETARG(MAXTGT),idtarg(MAXTGT),
     2 INX(8),RA(20),ZA(20),GEONUC(20),
     3 CHARG(20),AMNUC(20),eshift(MAXTGT),
     4 IP(MAXTGT),irp(MAXTGT),IMTGTM(2*MAXTGT,4),TGTM(2*MAXTGT)
      EQUIVALENCE (INX(1),KEY)
      DATA ZERO,ONE,TWO/0.0D0,1.D0,2.D0/,EPS/1.D-6/
      DATA PI/3.1415926535897932D0/,AMU/1822.832d0/
C
      REWIND LUTARG
      TRPI = TWO*SQRT(PI)
C
C----- SEARCH DUMPFILE FOR REQUIRED DATASET
      IPASS = 1
C
C     SET HEADER    ...    KEY = 9 :
C
C     FIELD
C       2   SET NUMBER
C       3   NUMBER OF RECORDS IN SET
C       4   NUMBER OF GEOMETRY RECORDS ( NNUC )
C       5   NUMBER OF RECORDS OF TARGET DATA
C       6   NUMBER OF RECORDS OF MOMENT DATA
C       7   MOMENT TYPE SWITCH, ISW
C       8
C       9   INTERNUCLEAR SEPARATION, RAB  ( DIATOMICS ONLY )
C      10   26-CHARACTER HEADER FIELD
C
   17 READ(LUTARG,11,END=40)(INX(I),I=1,8),DNX,HEAD
      IF(KEY .NE. 9) GO TO 21
C
      RAB = DNX
      iset   = inx(2)
      ntarg1 = inx(5)
      ntarg1 = inx(5)
      ntarg = NTARG1
      DO i=1,ntarg
        idtarg(i) = i
      END DO
      IF(ABS(RAB-R).GT.EPS.OR.NTARG.gt.ntarg1.OR.
     1               (NTSET.NE.0.AND.Iset.NE.NTSET)) THEN
C
C----- THIS IS NOT THE REQUIRED DATA SET SO SKIP REMAINING RECORDS
         IF(IPRNT.NE.0) WRITE(IWRITE,98) NTSET,NTARG,R,Iset,ntarg1,RAB
         DO J=1,inx(3)
           READ(LUTARG,11,END=21) (INX(I),I=1,8),DNX,HEAD
         END DO
         GO TO 17
      ENDIF
C
C     FILE IS NOW POSITIONED AT REQUIRED GEOMETRY
C
      NNUC=INX(4)
      NMOM=INX(6)
      ISW=INX(7)
C
C     READ DATA DEFINING MOLECULAR GEOMETRY   ...   KEY = 8 :
C
C     FIELD
C       2   NUCLEAR SEQUENCE NUMBER (I)
C       3   SET TO 0 FOR A REAL TARGET NUCLEUS, TO 1 FOR THE SCATTERING
C       4   NUCLEAR CHARGE ( USE NAMELIST INPUT FOR NONINTEGRAL VALUES )
C       5   NUCLEAR MASS ( IN ATOMIC UNITS )
C       6
C       7
C       8
C       9   NUCLEAR POSITION, GEONUC(I)
C      10   26-CHARACTER HEADER FIELD
C
      DO 8 IT=1,NNUC
      READ(LUTARG,11,END=21) (INX(I),I=1,8),DNX,HEAD
      IF(KEY .NE. 8) GO TO 21
      ISEQ=INX(2)
      CHARG(ISEQ)=Dble(INX(4))
      AMNUC(ISEQ)=Dble(INX(5))
      GEONUC(ISEQ)=DNX
 8    CONTINUE
      IQ = 0
      DO 41 I=1,NNUC
      IF(I.EQ.NUCCEN) GO TO 41
      IQ = IQ+1
      ZA(IQ) = CHARG(I)
      RA(IQ) = ABS(GEONUC(I)-GEONUC(NUCCEN))
 41   CONTINUE
      if(charg(nnuc).le.zero) nnuc=nnuc-1
C
C     READ TARGET DATA   ...   KEY = 5  :
C
C     FIELD
C       2   STATE INDEX         (I)
C       3   MANIFOLD INDEX
C       4   INDEX WITHIN MANIFOLD
C       5   |M|            MTARG(I)
C       6   2*S+1          STARG(I)
C       7   +/- INDEX      ITARG(I)
C       8   G/U INDEX      GTARG(I)
C       9   E IN AU        ETARG(I)
C
      ik = 0
      DO 10 ITG=1,NTARG1
      READ(LUTARG,11,END=21) (INX(I),I=1,8),DNX,HEAD
   11 FORMAT(I1,7I3,D20.12,2X,A26)
      IF(KEY .NE. 5) GO TO 21
      iset = inx(2)
      do 24 it=1,ntarg
      if(iset.eq.idtarg(it)) go to 25
 24   continue
      go to 10
 25   ik = ik+1
      MTARG(Ik)=INX(5)
      STARG(Ik)=INX(6)
      ITARG(Ik)=INX(7)
      GTARG(Ik)=INX(8)
      ETARG(Ik)=DNX+eshift(ik)
 10   CONTINUE
      if(ik.ne.ntarg) then
        WRITE(IWRITE,98) ik,NTARG,R,ISET,NTARG1,RAB
        go to 21
      endif
C
C---- Sort target energies into ascending order
c     ( use eshift as work space )
      call sort_outer(ntarg,ip,etarg)
      DO i=1,ntarg
        irp(ip(i)) = i
        eshift(i) = etarg(i)
      END DO
      DO i=1,ntarg
        etarg(i) = eshift(ip(i))
      END DO
      call swap(ntarg,ip,mtarg)
      call swap(ntarg,ip,starg)
      call swap(ntarg,ip,itarg)
      call swap(ntarg,ip,gtarg)
      do 33 i=1,ntarg
      if(ip(i).ne.i) go to 34
 33   continue
      go to 35
 34   write(iwrite,95) (ip(i),i=1,ntarg)
C
C---- CALCULATE REDUCED MASS IN ATOMIC UNITS
 35   SUMM = ZERO
      PRODM = AMU
      DO 2 I=1,NNUC
      AM = AMNUC(I)
      SUMM = SUMM+AM
      IF(AM.GT.ZERO) PRODM=PRODM*AM
 2    CONTINUE
      RMASS = PRODM/SUMM
      WRITE(IWRITE,18)RMASS
C
C     RDTMOM DATA    ...    KEY = 1
C
C     FIELD
C       2   STATE INDEX          (I)
C       3   |M(I)|
C       4   STATE INDEX          (J)
C       5   |M(J)|
C       6   OPERATOR CENTER INDEX  KOP
C       7   OPERATOR L-VALUE       LOP
C       8   OPERATOR |M|-VALUE     MOP
C       9   TRANSITION MOMENT IN AU, ISW CONVENTION DETERMINED BY HEADER
C
      NTG=NTARG*(NTARG+1)
      LU=NTG*ISMAX
C
      WRITE(IWRITE,20)
      NDMOM=0
      DO 60 IM=1,NMOM
      READ(LUTARG,11,END=21)(INX(I),I=1,8),DNX,HEAD
      IF(KEY .NE. 1) GO TO 21
C
      LOP=INX(7)
      IF(LOP .LT. 1 .OR. LOP .GT. ISMAX) GO TO 60
      IF(INX(6) .NE. NUCCEN) GO TO 21
c
c     check that this property corresponds to states being retained
      IT1 = INX(2)
      do 13 it=1,ntarg
      if(it1.eq.idtarg(it)) then
        it1 = it
        go to 14
      endif
 13   continue
      go to 60
 14   IT2 = INX(4)
      do 15 it=1,ntarg
      if(it2.eq.idtarg(it)) then
        it2 = it
        go to 16
      endif
 15   continue
      go to 60
c
 16   MT1 = INX(3)
      MT2 = INX(5)
      it1 = irp(it1)
      it2 = irp(it2)
      IF(MT1.NE.MTARG(IT1) .OR. MT2.NE.MTARG(IT2)) GO TO 21
      MOP=INX(8)
      I1 = MAX(IT1,IT2)
      I2 = MIN(IT1,IT2)
C
C     LOWER TRIANGULAR ARRAY OF ELEMENT PAIRS: |M1-M2|,M1+M2
C
      IQ=(LOP-1)*NTG+I1*(I1-1)+2*I2-1
      IF(INX(8) .NE. IABS(MTARG(I1)-MTARG(I2))) IQ=IQ+1
C
C----- CALCULATE MOMENTS U AS DEFINED BY BURKE,MACKEY AND SHIMAMURA
C      (J.PHYS.B         EQ 29)
C
      UG = DNX
C     The origin of the following 2 lines is mysterious so they have
C     removed
C      IF(MT1.NE.0) UG=UG/SQRT(TWO)
C      IF(MT2.NE.0) UG=UG/SQRT(TWO)
      IF(ISW.EQ.2) UG=TRPI*UG/SQRT(TWO*Dble(LOP)+ONE)
      IF(ISW.NE.0.AND.IT1.EQ.IT2.AND.MT1.EQ.MT2)
     1       UG = UG-ZA(1)*(-RA(1))**LOP-ZA(2)*RA(2)**LOP
      WRITE(IWRITE,96)IT1,MT1,IT2,MT2,LOP,MOP,UG
      IF (LOP.EQ.1) THEN
         NDMOM=NDMOM+1
         IMTGTM(IM,1)=IT1
         IMTGTM(IM,2)=MT1
         IMTGTM(IM,3)=IT2
         IMTGTM(IM,4)=MT2
         TGTM(IM)=UG
      ENDIF
 60   CONTINUE
C
      GO TO 22
C
 40   IF(IPASS.EQ.1) THEN
         REWIND LUTARG
         IPASS = IPASS+1
         GO TO 17
      ELSE
         WRITE(IWRITE,23)NTSET,R
 23      FORMAT(' UNABLE TO FIND REQUIRED TARGET DATA SET',I5,F10.3)
         STOP
      ENDIF
C
 22   CONTINUE
C
      RETURN
C
   21 WRITE(IWRITE,66)
   66 FORMAT(' ERROR in target properties data')
      WRITE(IWRITE,111) (INX(I),I=1,8),DNX,HEAD
  111 FORMAT(1X,I1,7I3,D20.12,2X,A26)
      IFAIL = 1
      RETURN
 98   FORMAT(' Required',2I5,F10.6,5X,'Skipped',2I5,F10.6)
 96   FORMAT(6I5,F10.5)
 18   FORMAT(/' Reduced mass',F9.1,' au')
 20   FORMAT(/' Transition moments')
 95   format(/' Warning, target states on properties file were not in en
     1ergy order'/' They have been reordered to',20i3/' Check that IMCSF
     2 uses energy ordered labels')
      END
      SUBROUTINE WRITAIH(LUAI,NAISET,KFORM,TITLE,NBOUND,
     1NCHAN,NVCHAN,ISMAX,IPRNT,IWRITE,IFAIL)
C
C**********************************************************************
C    WRITAI  writes header of asymptotic data on unit LUAI.
C     If NAISET=1 on input then data is written as first set,
C              0 then written at end of information (eoi).
C     On output NAISET holds the actual set number
C**********************************************************************
C
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      CHARACTER(LEN=11)  KFORM
      CHARACTER(LEN=80)  TITLE
      DATA KEYBC/11/
      SAVE
C
C     position file at eoi or end of set number NAISET
      NSET=NAISET
      CALL GETSET(LUAI,NSET,KEYBC,KFORM,IFAIL)
      IF(IFAIL.NE.0) THEN
        WRITE(IWRITE,666)NAISET,LUAI
        RETURN
      ENDIF
      IF(NAISET.NE.1)  NSET=NSET+1
      WRITE(IWRITE,16) NSET,LUAI
      NAISET=NSET
C
C  write header
      IF(KFORM.EQ.'FORMATTED')  THEN
         TEMP=dble(NCHAN)/10.0d0
         NINF2=NCHAN/10
         IF (TEMP.GT.NINF2) NINF2=NINF2+1
         NINF2=4*NINF2
         TEMP=dble(ISMAX*NCHAN*NCHAN)/10.0d0
         NINF3=(ISMAX*NCHAN*NCHAN)/10
         IF (TEMP.GT.NINF3) NINF3=NINF3+1
         NINFO = 5+NINF2+NINF3
         NDATA = NBOUND
         NREC = NDATA+NINFO
         WRITE(LUAI,10) KEYBC,NSET,NREC,NINFO,NDATA
         WRITE(LUAI,13) TITLE
         WRITE(LUAI,10) NBOUND,NCHAN,NVCHAN,ISMAX
      ELSE
         NINFO = 10
         NDATA = NBOUND
         NREC = NINFO+NDATA
         WRITE(LUAI) KEYBC,NSET,NREC,NINFO,NDATA
         WRITE(LUAI) TITLE
         WRITE(LUAI) NBOUND,NCHAN,NVCHAN,ISMAX
      ENDIF
C
      IF(IPRNT.NE.0)  THEN
         WRITE(IWRITE,14)
         WRITE(IWRITE,100) KEYBC,NSET,NREC,NINFO,NDATA
         WRITE(IWRITE,130) TITLE
         WRITE(IWRITE,100) NBOUND,NCHAN,NVCHAN,ISMAX
      ENDIF
C
      RETURN
  10  FORMAT(10I5)
  11  FORMAT(10F20.13)
  13  FORMAT(A80)
 14   FORMAT(/' Header on LUAI')
  100 FORMAT(1X,10I5)
  101 FORMAT(1X,10F20.6)
  130 FORMAT(1X,A80)
  666 FORMAT(/' UNABLE TO FIND ASYMPTOTIC INT. SET',I3,' ON UNIT',I3)
  16  FORMAT(/' Asymptotic integral info. will be written to set',I3,
     1' on unit',I3)
      END
      SUBROUTINE WRITAI1(LUAI,KFORM,NBOUND,
     1ION,IASY,IWRON,MAXPTS,
     1EPS,RAFEND,RAFIN,DEGENY,EWRON,RMATR,HX,TOL,ETHR,ICHL,LCHL,MCHL,CF,
     1IPFLG,NCHAN,NVCHAN,ISMAX,IPRNT,IWRITE)
C
C**********************************************************************
C    WRITAI  writes body of asymptotic data on unit LUAI.
C**********************************************************************
C
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DIMENSION IPFLG(10),ETHR(NCHAN),LCHL(NCHAN),CF(ISMAX*NCHAN*NCHAN)
      DIMENSION ICHL(NCHAN),MCHL(NCHAN)
      CHARACTER(LEN=11)  KFORM
      DATA KEYBC/11/
      SAVE
C
C  write body
      IF(KFORM.EQ.'FORMATTED')  THEN
         WRITE(LUAI,10) ION,IASY,IWRON,MAXPTS
         WRITE(LUAI,11) EPS,RAFEND,RAFIN,DEGENY,EWRON,RMATR,
     1                   HX,TOL
         WRITE(LUAI,10) (IPFLG(I),I=1,10)
         WRITE(LUAI,10) (ICHL(I),I=1,NCHAN)
         WRITE(LUAI,10) (LCHL(I),I=1,NCHAN)
         WRITE(LUAI,10) (MCHL(I),I=1,NCHAN)
         WRITE(LUAI,11) (ETHR(I),I=1,NCHAN)
         WRITE(LUAI,11) (CF(I),I=1,ISMAX*NCHAN*(NCHAN+1)/2)
      ELSE
         WRITE(LUAI) ION,IASY,IWRON,MAXPTS
         WRITE(LUAI) EPS,RAFEND,RAFIN,DEGENY,EWRON,RMATR,
     1                HX,TOL
         WRITE(LUAI) (IPFLG(I),I=1,10)
         WRITE(LUAI) (ICHL(I),I=1,NCHAN)
         WRITE(LUAI) (LCHL(I),I=1,NCHAN)
         WRITE(LUAI) (MCHL(I),I=1,NCHAN)
         WRITE(LUAI) (ETHR(I),I=1,NCHAN)
         WRITE(LUAI) (CF(I),I=1,ISMAX*NCHAN*(NCHAN+1)/2)
      ENDIF
C
      IF(IPRNT.NE.0)  THEN
         WRITE(IWRITE,14)
         WRITE(IWRITE,100) ION,IASY,IWRON,MAXPTS
         WRITE(IWRITE,101) EPS,RAFEND,RAFIN,DEGENY,EWRON,RMATR,
     1                     HX,TOL
         WRITE(IWRITE,100) (IPFLG(I),I=1,10)
         WRITE(IWRITE,100) (ICHL(I),I=1,NCHAN)
         WRITE(IWRITE,100) (LCHL(I),I=1,NCHAN)
         WRITE(IWRITE,100) (MCHL(I),I=1,NCHAN)
         WRITE(IWRITE,101) (ETHR(I),I=1,NCHAN)
         WRITE(IWRITE,101) (CF(I),I=1,ISMAX*NCHAN*(NCHAN+1)/2)
      ENDIF
C
      RETURN
  10  FORMAT(10I5)
  11  FORMAT(10F20.13)
  13  FORMAT(A80)
 14   FORMAT(/' Body on LUAI')
  100 FORMAT(1X,10I5)
  101 FORMAT(1X,10F20.6)
  130 FORMAT(1X,A80)
      END
      SUBROUTINE WRITAI2(LUAI,KFORM,E,AMPS1,IPRNT,IWRITE)
C
C**********************************************************************
C    WRITAI  writes body of asymptotic data on unit LUAI.
C**********************************************************************
C
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C     DIMENSION IPFLG(10)
      CHARACTER(LEN=11)  KFORM
      DATA KEYBC/11/
      SAVE
C
C  write body
      IF(KFORM.EQ.'FORMATTED')  THEN
         WRITE(LUAI,11) E,AMPS1
      ELSE
         WRITE(LUAI) E,AMPS1
      ENDIF
C
      IF(IPRNT.NE.0)  THEN
         WRITE(IWRITE,14)
         WRITE(IWRITE,101) E,AMPS1
      ENDIF
C
      RETURN
  10  FORMAT(10I5)
  11  FORMAT(10F20.13)
  13  FORMAT(A80)
 14   FORMAT(/' Body on LUAI')
  100 FORMAT(1X,10I5)
  101 FORMAT(1X,10F20.6)
  130 FORMAT(1X,A80)
      END
      SUBROUTINE WRITBH(LUBND,NBSET,KFORM,TITLE,MGVN,STOT,GUTOT,NCHAN,
     1NSTAT,NBOUND,RR,IPRNT,IWRITE,IFAIL)
C
C**********************************************************************
C    WRITBH  writes header of Bound Coefficients(BC) file on unit LUBND.
C     If NBSET=1 on input then BC are written as first set,
C              0 then written at end of information (eoi).
C     On output NBSET holds the actual set number
C**********************************************************************
C
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      INTEGER STOT,GUTOT
      CHARACTER(LEN=11)  KFORM
      CHARACTER(LEN=80)  TITLE
      DATA KEYBC/11/
      SAVE
C
C     position file at eoi or end of set number NBSET
      NSET=NBSET
      CALL GETSET(LUBND,NSET,KEYBC,KFORM,IFAIL)
      IF(IFAIL.NE.0) THEN
        WRITE(IWRITE,666)NBSET,LUBND
        RETURN
      ENDIF
      IF(NBSET.NE.1)  NSET=NSET+1
      WRITE(IWRITE,16) NSET,LUBND
      NBSET=NSET
      NINFO = 3
C
C  write header
      IF(KFORM.EQ.'FORMATTED')  THEN
         TEMP=dble(NSTAT+2)/10.0d0
         NDATA1=(NSTAT+2)/10
         IF (TEMP.GT.NDATA1) NDATA1=NDATA1+1
         TEMP=dble(NCHAN)/10.0d0
         NDATA2=NCHAN/10
         IF (TEMP.GT.NDATA2) NDATA2=NDATA2+1
         NDATA = (NDATA1+NDATA2)*NBOUND
         NREC = NDATA+NINFO
         WRITE(LUBND,10) KEYBC,NSET,NREC,NINFO,NDATA
         WRITE(LUBND,13) TITLE
         WRITE(LUBND,10) NBOUND,MGVN,STOT,GUTOT,NSTAT,NCHAN
         WRITE(LUBND,11) RR
      ELSE
         NDATA = NBOUND*2
         NREC = NINFO+NDATA
         WRITE(LUBND) KEYBC,NSET,NREC,NINFO,NDATA
         WRITE(LUBND) TITLE
         WRITE(LUBND) NBOUND,MGVN,STOT,GUTOT,NSTAT,NCHAN
         WRITE(LUBND) RR
      ENDIF
C
      IF(IPRNT.NE.0)  THEN
         WRITE(IWRITE,14)
         WRITE(IWRITE,100) KEYBC,NSET,NREC,NINFO,NDATA
         WRITE(IWRITE,130) TITLE
         WRITE(IWRITE,100) NBOUND,MGVN,STOT,GUTOT,NSTAT,NCHAN
         WRITE(IWRITE,101) RR
      ENDIF
C
      RETURN
  10  FORMAT(10I5)
  11  FORMAT(10F20.13)
  13  FORMAT(A80)
 14   FORMAT(/' Header on LUBND')
  100 FORMAT(1X,10I5)
  101 FORMAT(1X,10F20.6)
  130 FORMAT(1X,A80)
  666 FORMAT(/' UNABLE TO FIND BOUND STATE SET',I3,' ON UNIT',I3)
  16  FORMAT(/' Bound state coefficients will be written to set',I3,
     1' on unit',I3)
      END
      SUBROUTINE READMH(LUTARG,KEYH,ISET,NRECS,NNUC,NSTAT,NMOM,ISW,RMOI)
C
C**********************************************************************
C     READMH reads the header line from the property file written
C     by DENPROP ("old" or "new" format) or CDENPROP. If the fixed
C     format is not recognized, free format is assumed.
C**********************************************************************
C
      USE precisn_gbl, ONLY: wp
      IMPLICIT NONE
      INTEGER,  INTENT(IN)  :: LUTARG
      INTEGER,  INTENT(OUT) :: KEYH,ISET,NRECS,NNUC,NSTAT,NMOM,ISW
      REAL(wp), INTENT(OUT) :: RMOI(3)
      CHARACTER(LEN=256)    :: LINE
C
      READ (LUTARG, '(A)') LINE
      SELECT CASE (LEN_TRIM(LINE))
         CASE (80)
            READ (LINE, '(I1,6I3,1X,3D20.12)')
     &            KEYH,ISET,NRECS,NNUC,NSTAT,NMOM,ISW,RMOI
         CASE (87)
            READ (LINE, '(I1,I3,I6,I3,I4,I6,I3,1X,3D20.12)')
     &            KEYH,ISET,NRECS,NNUC,NSTAT,NMOM,ISW,RMOI
         CASE (104)
            READ (LINE, '(I1,I3,I9,I9,I9,I9,I3,1X,3D20.12)')
     &            KEYH,ISET,NRECS,NNUC,NSTAT,NMOM,ISW,RMOI
         CASE DEFAULT
            READ (LINE, *) KEYH,ISET,NRECS,NNUC,NSTAT,NMOM,ISW,RMOI
      END SELECT
      END
