! Copyright 2019
!
! For a comprehensive list of the developers that contributed to these codes
! see the UK-AMOR website.
!
! This file is part of UKRmol-out (UKRmol+ suite).
!
!     UKRmol-out is free software: you can redistribute it and/or modify
!     it under the terms of the GNU General Public License as published by
!     the Free Software Foundation, either version 3 of the License, or
!     (at your option) any later version.
!
!     UKRmol-out is distributed in the hope that it will be useful,
!     but WITHOUT ANY WARRANTY; without even the implied warranty of
!     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!     GNU General Public License for more details.
!
!     You should have received a copy of the GNU General Public License
!     along with  UKRmol-out (in source/COPYING). Alternatively, you can also visit
!     <https://www.gnu.org/licenses/>.
!
      subroutine I_MCQD(IFAIL)
C
C***********************************************************************
C
C      IMCQD calculates MultiChannel Quantum Defects from T-matrices
C      by constructing the appropriate S-matrix and assuming
C      extrapolation through threshold.
C      It is intended to be a self contained module which can be
C      run independantly from the main scattering calculation.
C      The argument X is a large array, of dimension MCOR,  to be used
C      as work space.
C      On exit, IFAIL=0 indicates succesful termination, else IFAIL=1
C
C***********************************************************************
C
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      PARAMETER (MAXERN=10,MAXXCH=200)
      DIMENSION EINC(2,MAXERN),NESCAT(MAXERN),IPRNT(3),
     1 IVCHL(MAXXCH),LVCHL(MAXXCH),MVCHL(MAXXCH),EVCHL(MAXXCH),
     2 MAXVI(1),MAXVF(1),NDOPEN(1)
      CHARACTER*80 NAME
      CHARACTER*4 CEUNIT(2)
      CHARACTER*9 FORM
      CHARACTER*1 ITFORM
      CHARACTER*11 TFORM,MODDAT
      CHARACTER*20 DAYTIM
      INTEGER STOT,GUTOT
      integer, allocatable :: ipt(:)
      double precision, allocatable :: tmr(:),tmi(:),alph(:),beta(:)
      double complex, allocatable :: smat(:),val(:),vec(:)
C
C***********************************************************************
C
C     Basic data is input via namelist /MCQD/
C      ITFORM   = Formatted/unformatted switch for unit LUTMT
C      IEUNIT   = UNITS IN WHICH INPUT SCATTERING ENERGIES ARE INPUT
C                 1= Ryd, 2= eV
C      IWRITE   = Logical unit for printed output
C      LUTMT    = Logical unit for T-matrix input
C      NAME     = TITLE FOR ANY OUTPUT
C      NTSET    = SET NUMBER OF T-MATRIX INPUT
C      IPRNT    = Print flag
C      DELTA    = Diagonaliser tolerance
C
      NAMELIST/MCQD/DELTA,IEUNIT,IWRITE,NAME,LUTMT,NTSET,ITFORM,IPRNT
C
C***********************************************************************
C
      DATA LUTMT/12/,NTSET/1/,CEUNIT/' Ryd',' eV '/,
     1     IEUNIT/1/,IPRNT/3*0/,IWRITE/6/,DELTA/1.0D-5/
      DATA RYD/0.073500D0/
      DATA TFORM,FORM/2*'FORMATTED'/,ITFORM/'U'/
      DATA MODDAT/'8-Dec-1995'/
C
      IFAIL = 0
      NEXT = 1
C
C---- Read basic data via namelist /MCQD/
      READ(5,MCQD)
      IF(ITFORM.EQ.'U') TFORM='UN'//FORM
C
C---- Date-stamp run and print title
      CALL DATEST(DAYTIM)
      NAME(61:) = DAYTIM
      WRITE(IWRITE,100)MODDAT,NAME
 100  FORMAT(///' Program MCQD  (last modified ',A,' )'//A/)
C
C----- Find required T-matrix set and read dimension information
      IBUF = NEXT
      CALL READTH(LUTMT,NAME,NTSET,NCHAN,NVIB,NDIS,NTARG,MAXCHI,
     1 MAXCHF,MGVN,STOT,GUTOT,NE,NERANG,NESCAT,EINC,IVCHL,
     2 LVCHL,MVCHL,EVCHL,TFORM,IWRITE,IPRNT(1),IFAIL)
      IF(IFAIL.NE.0) RETURN
      WRITE(IWRITE,104) DELTA
      NETOT = NE
C
C---- Recompute energy parameters NERANG,NESCAT and EINC for energy
C     range [EMIN,EMAX]
      IF(IEUNIT.EQ.2) THEN
        EMIN = RYD*EMIN
        EMAX = RYD*EMAX
      ENDIF
      CALL NEWE(EMIN,EMAX,NE,NERANG,NESCAT,EINC)
C
C---- Allocate space for T-matrices, S-matrix, etc.
      LTMT = MAXCHI*(MAXCHF+NDIS)
      allocate (tmr(ltmt),tmi(ltmt),smat(ltmt),val(nchan),vec(ltmt),
     * alph(nchan),beta(nchan),ipt(nchan))
C
C---- LOOP OVER ENERGIES
C
      NCOLDI=0
      NCOLDF=0
      DO 10 I=1,NETOT
C
C---- Read T-matrices for this energy only
      CALL READT(1,MAXCHI,MAXCHF,NDIS,MAXVI,MAXVF,NDOPEN,TMR,TMI,ENR)
      IF(IEUNIT.EQ.2) ENR=ENR/RYD
C
C---- Have we crossed a threshold since last T-matrix ?
      IF (NCOLDI .LT. MAXVI(1)) THEN
C       If so calculate new Quantum Defects
        NNEW=MAXVI(1)-NCOLDI
        WRITE(IWRITE,108) NNEW,ENR,CEUNIT(IEUNIT)
        IF (NNEW .EQ. MAXVF(1)+NDOPEN(1)-NCOLDF) THEN
          CALL MAKES(TMR,TMI,SMAT,VAL,VEC,ALPH,BETA,MAXCHI,MAXCHF,
     *    NNEW,DELTA,NCOLDI,NCOLDF,IWRITE,IPRNT(2),IPRNT(3),ipt)
        ELSE
          IFAIL=2
          WRITE(IWRITE,94) MAXVF(1)+NDOPEN(1)-NCOLDF
        ENDIF
      ELSE IF (NCOLDF .LT. MAXVF(1)+NDOPEN(1)) THEN
        WRITE(IWRITE,108) 0,ENR,CEUNIT(IEUNIT)
        IFAIL=2
        WRITE(IWRITE,94) MAXVF(1)+NDOPEN(1)-NCOLDF
      ENDIF
      NCOLDI=MAXVI(1)
      NCOLDF=MAXVF(1)+NDOPEN(1)
 10   CONTINUE
c
      deallocate (tmr,tmi,smat,val,vec,alph,beta,ipt)
      CLOSE(UNIT=LUTMT,STATUS='KEEP')
      RETURN
C
  94  FORMAT(I5,' new channels opened for final states'
     1       /' **** S(closed,closed) NOT square - case skipped ****'/)
 104  FORMAT(' Quantum Defects will be computed above each threshold',
     1       ' in grid'
     2     //' S-matrix diagonaliser tolerance DELTA =',D10.3)
 105  FORMAT(10A8)
 108  FORMAT(/I5,' new channels opened at E =',F8.4,A4)
      END
      SUBROUTINE MAKES(TMR,TMI,SMAT,VAL,vec,ALPH,BETA,MAXCHI,MAXCHF,
     1             NNEW,DELTA,NCOLDI,NCOLDF,IWRITE,IPFLG,IPVEC,ipt)
C
C     Construct the open-open part of the S matrix from the
C     corresponding T matrix and diagonalise it.
C
C     INPUT:
C     TMR     Real part of T MATRIX AT CURRENT SCATTERING ENERGY
C     TMI     Real part of T MATRIX AT CURRENT SCATTERING ENERGY
C     NNEW    NUMBER OF newly open CHANNELS
C     NCOLDI  NUMBER OF previously open CHANNELS for the initial state
C     NCOLDF  NUMBER OF previously open CHANNELS for the final   state
C     IPFLG   PRINT FLAG
C
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DIMENSION TMR(MAXCHI,MAXCHF),TMI(MAXCHI,MAXCHF),
     1          ALPH(NNEW),BETA(NNEW),ipt(nnew),work(3*nnew)
      double COMPLEX SMAT(NNEW,NNEW),VAL(NNEW),vec(nnew,nnew),
     *               zwork(64*nnew),lvec(nnew,nnew)
      double precision rwork(2*nnew)     
      DATA ZERO/0.0D0/,ONE/1.0D0/
c
      TWOPI=8.0d0*ATAN(ONE)
c     Construct S at current energy
      IF (IPFLG .GT. 0) WRITE(IWRITE,6)
    6 FORMAT('  CLOSED-CLOSED PART OF SMAT:'/)
      DO 10 J=1,NNEW
      ipt(j)=-1
      JJ=NCOLDI+J
      DO 20 I=1,NNEW
      II=NCOLDF+I
      SMAT(I,J) = CMPLX(TMR(JJ,II),TMI(JJ,II))
 20   continue
      SMAT(J,J) = SMAT(J,J) + CMPLX(ONE,ZERO)
      IF (IPFLG .GT. 0) WRITE(IWRITE,7) (SMAT(I,J),I=1,NNEW)
    7 FORMAT(6D20.9)
   10 CONTINUE
C     diagonalise smat... using CEIGEN (note in this vec should be reset
c     to smat below
c      CALL CEIGEN(SMAT,VAL,NNEW,DELTA,TMR)
!      However, TMR should be replaced by VEC as follow.....(NV 10/03)
!      CALL CEIGEN(smat,VAL,NNEW,DELTA,vec)
!
c      ..........or LAPACK  routine ZGEEV. This routine and dependencies
!      replaces NAG F02GBF that was used erlier in this code (NV 10/03)
!
        call zgeev('n','v',nnew,smat,nnew,val,lvec,nnew,vec,nnew,zwork,
     *             3*nnew,rwork,info)     
      if (info .ne. 0) write(iwrite,444)
  444 format(' LAPACK routine Zgeev returned info =',i4)
      inot=0
      DO 30 I=1,NNEW
      ALPH(I)=aIMAG(LOG(VAL(I)))/TWOPI
      IF (ALPH(I) .LT. ZERO) ALPH(I) = ALPH(I) + ONE
      BETA(I)=-LOG(ABS(VAL(I)))/TWOPI
      jpt=1
      top=abs(vec(1,i))**2
      do 35 j=2,nnew
      comp=abs(vec(j,i))**2
      if (comp .gt. top) then
        jpt=j
        top=comp
      endif
   35 continue
      if (ipt(i) .eq. -1) then
         ipt(i)=jpt
      else
         inot=1
      endif
   30 CONTINUE
      if (inot .ne. 0) then
         write(iwrite,99) 
   99    format(' Attempt to order by channel failed: no re-ordering')
         do 36 i=1,nnew
         ipt(i)=i
 36      continue
      endif
      WRITE(IWRITE,8) (I,I=1,NNEW)
    8 FORMAT(/'  MultiChannel Quantum Defects (in the range 0 to 1)',
     1       /' Channel',i12,5i20)
      WRITE(IWRITE,9) (ALPH(ipt(I)),I=1,NNEW)
    9 FORMAT( ' Alpha  ',6D20.8)
      IF (MAX(NCOLDI,NCOLDF) .GT. 0) 
     1       WRITE(IWRITE,3) (BETA(ipt(I)),I=1,NNEW)
    3 FORMAT( ' Beta   ',6D20.8)
      IF (IPVEC .GT. 0) THEN
        WRITE(IWRITE,11)
   11   FORMAT(/' Channel eigenvectors:')
        DO 40 J=1,NNEW
        WRITE(IWRITE,12) (vec(ipt(I),J),I=1,NNEW)
 40     continue
   12   FORMAT(/(10F12.8))
      ENDIF
      RETURN
      END
      SUBROUTINE CEIGEN(A,VAL,N,DELTA,X)
C
C  DIAGONALISATION OF DOUBLE PRECISION COMPLEX SYMMETRIC MATRIX
C  USING METHOD DESCRIBED BY M.J.SEATON, COMPUTER JOURNAL, VOL.12,
C  PAGE 156, 1969.
C
      IMPLICIT DOUBLE PRECISION(A-H,O-Z)
c      COMPLEX A(N,N),X(N,N),VAL(N),C,S,H,P,Q,CIM
C  **** Warning: on 4-byte word machines this statement should be
      double COMPLEX A(N,N),X(N,N),VAL(N),C,S,H,P,Q,CIM
      data zero,one/0.d0,1.d0/
C
      SQ(H)=H* CONJG(H)
      CIM=(0.d0,1.d0)
      NM1=N-1
C
C  SET IROTM
      IROTM=20
C
C
      IF(N.EQ.1) GO TO 710
      IF(N.EQ.2) GO TO 720
C
C INITIAL D1,D2 AND X
C
      D1=zero
      H=(0.d0,0.d0)
      DO 510 I=1,N
      X(I,I)=(1.d0,0.d0)
      D1=D1+SQ(A(I,I))
      H=H+A(I,I)
 510  continue
      D1=D1-SQ(H)/dble(N)
C
      D2=zero
      DO 520 I=1,NM1
      IP1=I+1
      DO 521 J=IP1,N
      X(I,J)=(0.d0,0.d0)
      X(J,I)=(0.d0,0.d0)
      D2=D2+SQ(A(I,J))
 521  continue
 520  continue
      D1=DELTA*DELTA*0.5d0 * dble(N-1)*(D1+2.d0*D2)/ dble(N+1)
C
C BEGIN ROTATIONS
C
      DO 1010 IROT=1,IROTM
      DO 1011 IP=1,NM1
      IPP1=IP+1
      DO 1000 IQ=IPP1,N
C
C ROTATION CONSTANTS
C
      Q=A(IP,IQ)
      P=0.5d0*(A(IP,IP)-A(IQ,IQ))
      FL= ABS(P**2+Q**2)
      BETA=0.5d0*LOG(SQ(P-CIM*Q)/FL)
      T=( CONJG(P)*Q+P* CONJG(Q))/FL
      D=(SQ(P)-SQ(Q))/FL
      U=-0.25d0* ATAN2(T,D)
C
      T=zero
      D=zero
      DO 560 I=1,N
      IF(I.EQ.IP.OR.I.EQ.IQ) GO TO 560
      D=D+SQ(A(IP,I))+SQ(A(IQ,I))
      T=T+SQ(A(IP,I)+CIM*A(IQ,I))
  560 CONTINUE
      T=D-T
      FN= SQRT(D**2-T**2)
      GAMMA=LOG((D+T)/FN)
C
C ITERATION FOR V
C
      V0=-0.5d0*(BETA+GAMMA)
      DO 570 ITERV=1,100
      V=V0-(FL* SINH(2.d0*(V0+BETA))+FN* SINH(V0+GAMMA))/
     1(2.d0*FL* COSH(2.d0*(V0+BETA))+FN* COSH(V0+GAMMA))
      IF( ABS(V-V0).LT.DELTA*0.1d0) GO TO 580
      V0=V
 570  continue
      WRITE(6,5010) DELTA
      CONTINUE
C
C NEW A,X AND D2
C
  580 V=0.5d0*V
      S= SIN(U)* COSH(V)+CIM* COS(U)* SINH(V)
      C= COS(U)* COSH(V)-CIM* SIN(U)* SINH(V)
      H=2.d0*(S**2*P+S*C*Q)
      A(IP,IP)=A(IP,IP)-H
      A(IQ,IQ)=A(IQ,IQ)+H
      A(IP,IQ)=(C**2-S**2)*Q+2.d0*C*S*P
      D2=D2-SQ(A(IQ,IP))+SQ(A(IP,IQ))
      A(IQ,IP)=A(IP,IQ)
C
      DO 590 I=1,N
      H=X(I,IP)
      X(I,IP)=H*C-X(I,IQ)*S
      X(I,IQ)=H*S+X(I,IQ)*C
      IF(I.EQ. IP.OR.I.EQ.IQ)GO TO 590
      H=A(IP,I)
      A(IP,I)=C*H-A(IQ,I)*S
      A(IQ,I)=S*H+A(IQ,I)*C
      D2=D2-SQ(A(I,IP))-SQ(A(I,IQ))+SQ(A(IP,I))+SQ(A(IQ,I))
      A(I,IP)=A(IP,I)
      A(I,IQ)=A(IQ,I)
  590 CONTINUE
C
C TEST CONVERGENCE
      IF(D2.LT.D1) GO TO 610
C
C END ROTATIONS
C
 1000 CONTINUE
 1011 continue
C
C  RECALCULATE D2
C
      D2=zero
      DO 1002 I=1,NM1
      IP1=I+1
      DO 1001 J=IP1,N
      D2=D2+SQ(A(I,J))
 1001 continue
 1002 continue
 1010 CONTINUE
      WRITE(6,5020)IROTM,DELTA
C
C EIGENVALUES AND EIGENVECTORS
C
  610 DO 620 I=1,N
      VAL(I)=A(I,I)
      DO 621 J=1,N
      A(I,J)=X(I,J)
 621  continue
 620  continue
      RETURN
C
C N=1 AND N=2
C
  710 VAL(1)=A(1,1)
      A(1,1)=one
      RETURN
C
  720 Q=A(1,2)
      P=0.5d0*(A(1,1)-A(2,2))
      FL= ABS(P**2+Q**2)
      V=-0.25d0*LOG(SQ(P-CIM*Q)/FL)
      T= CONJG(P)*Q+P* CONJG(Q)
      U=-0.25d0* ATAN2(T/FL,(SQ(P)-SQ(Q))/FL)
      S= SIN(U)* COSH(V)+CIM* COS(U)* SINH(V)
      C= COS(U)* COSH(V)-CIM* SIN(U)* SINH(V)
      H=2.d0*(S**2*P+S*C*Q)
      VAL(1)=A(1,1)-H
      VAL(2)=A(2,2)+H
      A(1,1)=C
      A(1,2)=S
      A(2,1)=-S
      A(2,2)=C
C
 5010 FORMAT(//10X,'*** SUBROUTINE CEIGEN ***'
     + //' WARNING - NO CONVERGENCE IN ITERATIONS FOR V '/
     +  ' ACCURACY PARAMETER DELTA = ',E9.2/
     +  ' NEXT VALUE OF AVERAGED OMEGA MAY BE INCORRECT'//)
 5020 FORMAT(//10X,' *** SUBROUTINE CEIGEN ***'
     + //' NO CONVERGENCE FOR IROTM = ',I2,/' DELTA = ',E10.2/
     +  ' NEXT VALUE OF AVERAGED OMEGA MAY BE INCORRECT'//)
C
      RETURN
      END
