! Copyright 2019
!
! For a comprehensive list of the developers that contributed to these codes
! see the UK-AMOR website.
!
! This file is part of UKRmol-out (UKRmol+ suite).
!
!     UKRmol-out is free software: you can redistribute it and/or modify
!     it under the terms of the GNU General Public License as published by
!     the Free Software Foundation, either version 3 of the License, or
!     (at your option) any later version.
!
!     UKRmol-out is distributed in the hope that it will be useful,
!     but WITHOUT ANY WARRANTY; without even the implied warranty of
!     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!     GNU General Public License for more details.
!
!     You should have received a copy of the GNU General Public License
!     along with  UKRmol-out (in source/COPYING). Alternatively, you can also visit
!     <https://www.gnu.org/licenses/>.
!
!Z. Masin, 07/2013: module created.
!Z. Masin, 08/2014: integrated with coupling_obj, removed redundant routines
MODULE couplings
use precisn_gbl

   real(kind=wp), parameter, private :: fourpi=  12.5663706143591729538505735331180115_wp
   real(kind=wp), parameter, private :: roneh =  0.70710678118654752440084436210484903_wp !sqrt(0.5)
   real(kind=wp), parameter, private :: rothree =0.57735026918962576450914878050195745_wp !1/sqrt(3)
   complex(kind=wp), parameter, private :: imu=(0.0_wp,1.0_wp)
   
CONTAINS

   !This is a completely rewritten version of the original SWINTERF routine SWASYMC. The coupling potentials for the outer region calculation are defined as:
   !
   !&&V_{ij}(r)=\sum_{\lambda=0}^{\infty}\frac{1}{r^{\lambda+1}}\times \nonumber \\
   !&\times&\underbrace{\sum_{m=-\lambda}^{\lambda}\langle{\cal Y}_{l_{i},m_{i}}\vert {\cal {Y}}_{\lambda ,m}\vert{\cal Y}_{l_{j},m_{j}}\rangle\sqrt{\frac{4\pi}{2\lambda+1}}\underbrace{\sqrt{\frac{4\pi}{2\lambda+1}}\left( T_{ij}^{\lambda m} - \langle \Phi_{i}\vert\Phi_{j}\rangle \sum_{k=1}^{Nuclei}Z_{k}{\cal Y}_{\lambda,m}(\hat{\mathbf{R}}_{k})R_{k}^{\lambda}\right)}_{Q_{ij}^{\lambda m}}}_{a_{ij\lambda}}.
   !
   !This routine calculates the coefficients a_{ij\lambda} using the information for all channels (i,j) and the target permanent and transition multipole moments Q_{ij}^{\lambda m}.
   !The main difference from SWASYMC is that here the coupling coefficients for the real spherical harmonics are calculated independently of the molecular orientation, symmetry and the lambda value.
   !Tests were performed for pyrazine (D2h), uracil (Cs) and water (C2v). The inclusion of the additional polarization potential is also possible, but note that only the spherical part is taken into
   !account at the moment (see the comment for the variable alpha2).
   !
   !\param[in] nchan Total number of channels.
   !\param[in] ismax Maximum lambda value for the multipole potential contribution.
   !\param[in] ntarg Total number of target states.
   !\param[in] ichl(nchan) Integer array holding the sequence number of the target states.
   !\param[in] lchl(nchan) Integer array holding the L values for the scattering channels.
   !\param[in] mchl(nchan) Integer array holding the |M| values for the scattering channels.
   !\param[in] qchl(nchan) Integer array holding the q values for the scattering channels.
   !\param[in] prop(ntarg,ntarg,maxprop) Double prec array holding the permanent and the transition multipole moments of the target electronic states.
   !\param[in] alpha0 Spherical part of the ground state target polarizability.
   !\param[in] alpha2 Non-spherical part of the ground state target polarizability. NOTE that this has not been implemented in this subroutine yet since it is not clear to me what is the convention defining
   !                  this value in the old code.
   !\param[in] use_pol If .true. then the coefficients for the polarization potential will be constructed.
   !\param[inout] a(:) Linear array of dimension at least ismax*nchan*(nchan+1)/2 containing the coupling coefficients a_{ij\lambda}. See the routine for the order of the coefficients.
   subroutine channel_couplings(nchan,ismax,ntarg,ichl,lchl,mchl,qchl,a,prop,alpha0,alpha2,use_pol,ukrmolp_ints)
      use coupling_obj_gbl
      implicit none
      integer, intent(in) :: nchan, ismax, ntarg
      integer, intent(in) :: ichl(nchan), lchl(nchan), mchl(nchan), qchl(nchan)
      real(kind=wp), intent(inout) :: a(:)
      real(kind=wp), intent(in) :: prop(:,:,:), alpha0, alpha2
      logical, intent(in) :: use_pol, ukrmolp_ints
  
      !The Gaunt coefficients for the real spherical harmonics defined below are needed to express the xx, yy, zz angular behaviour in terms of the real spherical harmonics
      !x^2 = x*x ~ X_{11}*X_{11} = x2_X00*X_{00} + x2_X20*X_{20} + x2_X22*X_{22}
      real(kind=wp), parameter :: x2_X00 =  0.282094791773878E+00_wp
      real(kind=wp), parameter :: x2_X20 = -0.126156626101008E+00_wp
      real(kind=wp), parameter :: x2_X22 =  0.218509686118416E+00_wp
      !y^2 = y*y ~ X_{1-1}*X_{1-1} = y2_X00*X_{00} + y2_X20*X_{20} + y2_X22*X_{22}
      real(kind=wp), parameter :: y2_X00 =  0.282094791773878E+00_wp
      real(kind=wp), parameter :: y2_X20 = -0.126156626101008E+00_wp
      real(kind=wp), parameter :: y2_X22 = -0.218509686118416E+00_wp
      !z^2 = z*z ~ X_{10}*X_{10} = z2_X00*X_{00} + z2_X20*X_{20}
      real(kind=wp), parameter :: z2_X00 =  0.282094791773878E+00_wp
      real(kind=wp), parameter :: z2_X20 =  0.252313252202016E+00_wp

      integer :: l1, m1, q1, l2, m2, q2, no_cpl, lqt, iq, isq, it1, it2
      integer :: ch_1, ch_2, lambda, mlambda, lmin, lmax
      real(kind=wp) :: cpl, sph_cpl, fac
      logical :: use_alpha2
      type(couplings_type) :: couplings

         !total number of unique combinations of the scattering channels
         no_cpl = nchan*(nchan+1)/2 

         a(:) = 0.0_wp

         if (use_pol .and. ntarg > 1) then
            write(*,'("WARNING: adding polarization potential while more than one target state is present in the outer region.")')
         endif

         use_alpha2 = .false.
         if (use_pol .and. alpha2 .ne. 0.0_wp) use_alpha2 = .true.

         !check that the array a(:) is large enough, especially if the polarizability is being added.
         lmax = ismax
         if (use_pol) lmax = max(ismax,3) !polarizability corresponds to lambda=3

         !ismax*no_cpl = number of coupling coefficients to calculate: for each lambda we calculate no_cpl couplings corresponding to each unique combination of the scattering channels
         if (size(a) < lmax*no_cpl) then
            stop "Size of the input array a(:) is smaller than the upper limit &
                    &needed for calculation of the coefficients for the coupling potentials."
         endif

         a(:) = 0.0_wp

         do ch_1=1,nchan

            !get the l, |m| values for this channel: the l, |m| values correspond to the l,|m| values of the real spherical harmonics
            l1 = lchl(ch_1)
            m1 = mchl(ch_1)

            !q is sign(m) or 0 if m=0.
            q1 = qchl(ch_1)

            !determine m
            m1 = m1*q1

            !sequence number of the target state corresponding to this channel
            it1 = ichl(ch_1)

            do ch_2=1,ch_1

               !get the l, |m| values for this channel: the l, |m| values correspond to the l,|m| values of the real spherical harmonics
               l2 = lchl(ch_2)
               m2 = mchl(ch_2)

               !q is sign(m) or 0 if m=0.
               q2 = qchl(ch_2)

               !determine m
               m2 = m2*q2

               !sequence number of the target state corresponding to this channel
               it2 = ichl(ch_2)

               !use the selection rules for the real spherical harmonics to determine the range of lambda values which may give non-zero real Gaunt coefficients
               call couplings%bounds_rg(l1,l2,m1,m2,lmin,lmax)

               !don't include potentials with lambda > ismax
               if (lmax > ismax) lmax = ismax

               !lambda = 0 would correspond to the monopole contribution, i.e. taking into account the total (perhaps nonzero) molecular charge. This is taken into account separately in RSOLVE.
               if (lmin .eq. 0) lmin=2

               !linear index corresponding to the current combination of the channels (ch_1,ch_2).
               lqt = ch_1*(ch_1-1)/2+ch_2

               !Loop over the multipole moments which may be non-zero: we loop in steps of two since the selection rules for the r.s.h. imply that only if the sum of all L values is even the coupling may be
               !non-zero.
               do lambda=lmin,lmax,2

                  !linear index corresponding to the current combination of the (lambda, ch_1, ch_2) values.
                  iq = (lambda-1)*no_cpl+lqt

                  !see the formula for V_{ij}: effectively this converts the
                  !inner region property from solid harmonic normalization to
                  !spherical harmonic normalization as needed by the Leg.
                  !expansion.
                  fac = sqrt(fourpi/(2*lambda+1.0_wp))

                  do mlambda=-lambda,lambda

                     !rgaunt: the coupling coefficient for the real spherical harmonics (l1,m1,l2,m2,lambda,mlambda); 
                     !The factor 2.0_wp converts the units of the inner region
                     !properties from Hartree to Rydberg since that's the energy
                     !unit used in the outer region.
                     cpl = 2.0_wp*fac*couplings%rgaunt(l1,lambda,l2,m1,mlambda,m2)

                     !In case of UKRmol+ the boundary amplitudes (and therefore the channels) have angular dependency of the form: (-1)^m*X_{l,m} while the real Gaunt coefficients are calculated
                     !assuming X_{l,m} so the extra phase factors corresponding to the channel functions need to be taken into account here.
                     if (ukrmolp_ints) cpl = cpl*(-1)**(m1+m2)

                     !linear index corresponding to the current (lambda,mlambda) values. lambda*lambda is the number of all (lambda1,mlambda1) combinations for lambda1 = lambda-1.
                     isq = lambda*lambda+lambda+mlambda

                     !increment the value of the coefficient for the multipole coupling potential of order lambda between the target states it1 and it2
                     a(iq) = a(iq)+cpl*prop(it1,it2,isq)

                  enddo !mlambda
               enddo !lambda

               !add polarizabilities for the ground state channels, but only if use_pol == .true.
               if (use_pol .and. it1 == 1 .and. it2 == 1) then

                  !the radial dependence of the polarization potential is r^{-4} which corresponds to lambda = 3
                  lambda = 3

                  !linear index corresponding to the current combination of the (lambda, l1, l2=l1) values.
                  iq = (lambda-1)*no_cpl+lqt

                  !obviously, the spherical polarizability (l=0) couples only the channels with identical angular behaviour
                  if (l1 .eq. l2 .and. m1 .eq. m2) then
                     sph_cpl = 1.0_wp

                     write(*,'("Spherical part of the polarization potential will be added to the &
                             &channel (target state,l,m), coefficient value: (",3i5,") ",e25.15)') it1, l1, m1, -sph_cpl*alpha0

                     !add the spherical part of the polarizability
                     a(iq) = a(iq)-sph_cpl*alpha0
                  else
                     sph_cpl = 0.0_wp
                  endif

                  !Add the non-spherical part of the polarizability; the polarization potential here corresponds to: alpha_{2}*C_{i}*r_{-4}, where the angular behaviour of C_{i} = xy, xz, yz, xx, yy, zz
                  !Note that we assume below that the polarizability tensor has the same values (alpha2) for all components.
                  !Also note that as long as l1,m1 and l2,m2 belong to the same IR, only the totally symmetric components of the pol. tensor (xx,yy,zz) may contribute.
                  if (use_alpha2) then

                     write(*,'("Non-spherical part of the polarization potential &
                             &(target state,l1,m2,l2,m2): ",5i5)') it1, l1, m1, l2, m2

                     !x^2 component
                     cpl = x2_X00*couplings%rgaunt(l1,0,l2,m1,0,m2) + &
                           x2_X20*couplings%rgaunt(l1,2,l2,m1,0,m2) + &
                           x2_X22*couplings%rgaunt(l1,2,l2,m1,2,m2)
                     write(*,'("x^2 coefficient: ",e25.15)') -cpl*alpha2
                     a(iq) = a(iq)-cpl*alpha2

                     !y^2 component
                     cpl = y2_X00*couplings%rgaunt(l1,0,l2,m1,0,m2) + &
                           y2_X20*couplings%rgaunt(l1,2,l2,m1,0,m2) + &
                           y2_X22*couplings%rgaunt(l1,2,l2,m1,2,m2) 
                     write(*,'("y^2 coefficient: ",e25.15)') -cpl*alpha2
                     a(iq) = a(iq)-cpl*alpha2

                     !z^2 component
                     cpl = z2_X00*couplings%rgaunt(l1,0,l2,m1,0,m2) + &
                           z2_X20*couplings%rgaunt(l1,2,l2,m1,0,m2)
                     write(*,'("z^2 coefficient: ",e25.15)') -cpl*alpha2
                     a(iq) = a(iq)-cpl*alpha2

                     !xz component
                     cpl = couplings%rgaunt(l1,2,l2,m1,1,m2)*rothree
                     write(*,'("xz coefficient: ",e25.15)') -cpl*alpha2
                     a(iq) = a(iq)-cpl*alpha2

                     !yz component
                     cpl = couplings%rgaunt(l1,2,l2,m1,-1,m2)*rothree
                     write(*,'("yz coefficient: ",e25.15)') -cpl*alpha2
                     a(iq) = a(iq)-cpl*alpha2

                     !xy component
                     cpl = couplings%rgaunt(l1,2,l2,m1,-2,m2)*rothree
                     write(*,'("xy coefficient: ",e25.15)') -cpl*alpha2
                     a(iq) = a(iq)-cpl*alpha2

                  endif

               endif

            enddo !ch_2
         enddo !ch_1

   end subroutine channel_couplings

END MODULE couplings
