! Copyright 2019
!
! For a comprehensive list of the developers that contributed to these codes
! see the UK-AMOR website.
!
! This file is part of UKRmol-out (UKRmol+ suite).
!
!     UKRmol-out is free software: you can redistribute it and/or modify
!     it under the terms of the GNU General Public License as published by
!     the Free Software Foundation, either version 3 of the License, or
!     (at your option) any later version.
!
!     UKRmol-out is distributed in the hope that it will be useful,
!     but WITHOUT ANY WARRANTY; without even the implied warranty of
!     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!     GNU General Public License for more details.
!
!     You should have received a copy of the GNU General Public License
!     along with  UKRmol-out (in source/COPYING). Alternatively, you can also visit
!     <https://www.gnu.org/licenses/>.
!
      SUBROUTINE GAILIT(ET,L2P,NCHAN,ION,CF,LAMAX,RAFIN,IWRITE,IRAD,
     X IASY,IGAIL,DEGENY,EPS,IPFLG,F,FP,X,MCORE,LUSED,ifail)
C
C***********************************************************************
C
C     GAILIT DETERMINES THE ASYMPTOTIC SCATTERING FUNCTIONS F AND THEIR
C            DERIVATIVES, FP, AT THE RADIAL POINT RAFIN FOR THE
C            N-COUPLED SECOND-ORDER ORDINARY DIFFERENTIAL EQUATIONS
C            CORRESPONDING TO THE RADIAL SCHRODINGER EQUATIONS FOR
C            THE CASE WHERE ALL INTERACTION POTENTIALS MAY BE
C            REPRESENTED IN A SIMPLE MULTIPOLAR FORM.
C
C            VARIOUS ASYMPTOTIC EXPANSIONS ARE ANALYTICALLY CONTINUED
C            USING A CONTINUED FRACTION METHOD EQUIVALENT TO THE
C            USE OF PADE APPROXIMANTS.
C
C     IF IRAD = 0 :
C
C            EITHER THE BURKE-SCHEY EXPANSION METHOD ( IGAIL = 0 )
C            OR THE GAILITIS EXPANSION MAY BE USED   ( IGAIL = 1 )
C
C            COULOMB/BESSEL FUNCTIONS ARE RETURNED IF IGAIL = 2
C
C     IF IRAD = 1 :
C
C            THE RADIUS NECESSARY TO OBTAIN THE REQUIRED ACCURACY,
C            ASSUMING THAT THE SERIES ARE TREATED IN THE STANDARD
C            FASHION FOR ASYMPTOTIC EXPANSIONS, IS PRINTED FOR
C            INFORMATION ONLY
C
C     IF IRAD = 2 :
C
C            THE RADIUS NECESSARY TO OBTAIN THE REQUIRED ACCURACY,
C            ASSUMING THAT THE SERIES ARE TREATED IN THE STANDARD
C            FASHION FOR ASYMPTOTIC EXPANSIONS, IS USED IN THE
C            CALCULATION OF THE ASYMPTOTIC WAVEFUNCTIONS
C
C***********************************************************************
C
C     ET          CHANNEL ENERGIES IN RYDBERGS
C     L2P         CHANNEL ORBITAL ANGULAR MOMENTA
C     NCHAN       NUMBER OF CHANNELS
C     IASY        NUMBER OF TERMS IN THE ASYMPTOTIC EXPANSION
C     ION         RESIDUAL CHARGE ( IONICITY )
C     IRAD        = 0 DETERMINE WAVEFUNCTION AND DERIVATIVES AT RAFIN
C                 = 1 RETURN VALUE OF RAFIN NECESSARY IF THE
C                     ASYMPTOTIC SERIES ARE SIMPLY EVALUATED IN THE
C                     USUAL WAY
C     LAMAX       NUMBER OF MULTIPOLES IN THE EXPANSION OF THE
C                 ASYMPTOTIC POTENTIAL
C     CF          EXPANSION COEFFICIENTS OF THE ASYMPTOTIC POTENTIAL
C                 IN RYDBERG UNITS
C     IWRITE      UNIT NUMBER OF THE PRINTER OR OUTPUT FILE
C     IGAIL       BURKE-SCHEY / GAILITIS METHOD SWITCH
C     DEGENY      CRITERION FOR REGARDING CHANNELS AS DEGENERATE
C     EPS         ACCURACY REQUESTED IN DETERMINING WAVEFUNCTIONS
C     SCALE       FACTOR USED TO SCALE BURKE-SCHEY COEFFICIENTS
C     IPFLG       PRINT FLAGS ...   IPFLG = 1 PRINT DATA
C                                   IPFLG = 0 DO NOT PRINT DATA
C                 FLAG 1    BURKE-SCHEY EXPANSION COEFFICIENTS
C                      2    GAILITIS TRANSFORMATION COEFFICIENTS
C                      3    GAILITIS EXPANSION COEFFICIENTS
C                      4    CONVERGENCE RADII
C                      5    CHANNEL CONVERGENCE RADII AND GLOBAL
C                           CONVERGENCE RADIUS
C                      6    ASYMPTOTIC FUNCTIONS AND DERIVATIVES
C                      7    ASYMPTOTIC SERIES DIAGNOSTIC DATA
C
C     RAFIN       GLOBAL CONVERGENCE RADIUS IN ATOMIC UNITS
C     F, FP       ASYMPTOTIC SCATTERING FUNCTIONS AND DERIVATIVES
C
C     X           not used
C     MCORE       not used
C     LUSED       not used
C
C***********************************************************************
C
      IMPLICIT DOUBLE PRECISION(A-H,O-Z)
C
      DIMENSION F(NCHAN,NCHAN,2),FP(NCHAN,NCHAN,2),ET(NCHAN),
     X  L2P(NCHAN),CF(NCHAN,NCHAN,*),IPFLG(7),
     * ga1(iasy*nchan),ga2(iasy*nchan),gac1(iasy),gac2(iasy)
      CHARACTER(LEN=7) WARN(2)
C
      DATA ZERO/0.0D0/,ONE/1.0D0/,SCALE/1.0D-20/,EPSL/1.D+4/
      DATA WARN/'WARNING','FAILURE'/
C
 1010 FORMAT(' CHANNEL ENERGY ET(',I3,') =',D16.8,' RYDBERGS IS ZERO',
     X       ' TO ACCURACY DEFINED BY DEGENY =',D16.8,/,' THIS CASE',
     X       ' IS NOT ALLOWED FOR IN CFASM .. RETURN TO CALLING CODE')
C
C             **********     STORAGE MANAGEMENT     **********
C
C***** MAJOR MODS 21/12/98 LAM
C
      LMAX=-1
      DO 10 I=1,NCHAN
      LMAX=MAX(LMAX,L2P(I))
   10 CONTINUE
      NFC=LMAX+10
C
C     ASYWF0 :
C
      LEGA=IASY*NCHAN
C
C     INITIALIZE CALCULATION OF ASYMPTOTIC FUNCTIONS
C
      CALL ASYWF0(NCHAN,iasy,nfc,F,FP,EPS,ION,IWRITE,IGAIL,IPFLG(7))
C
C***********************************************************************
C
C     START OF LOOP OVER BOUNDARY CONDITIONS
C
      DO 80 J=1,NCHAN
C
      JCHAN=J
      LJ=L2P(J)
      ETJ=ET(J)
      IF(DABS(ETJ) .LT. DEGENY) GO TO 90
      SKJ=DSQRT(DABS(ETJ))
      ETAJ=-Dble(ION)/SKJ
      SCALX=SCALE
C
C     DETERMINE WHETHER THE CHANNEL IS OPEN OR CLOSED
C
      JOPEN=1
      IF(ETJ .LT. ZERO) JOPEN=0
C
      IF(IGAIL .LT. 2) THEN
C
C     COMPUTE BURKE-SCHEY EXPANSION COEFFICIENTS
C
      CALL BKSHEY(JCHAN,SKJ,ETAJ,ET,L2P,NCHAN,ION,CF,LAMAX,JOPEN,
     X            IASY,DEGENY,IPFLG(1),IWRITE,GA1,GA2,SCALX)
      RSCALX=ONE/SCALX
C
C     CONVERT BURKE-SCHEY COEFFICIENTS TO GAILITIS COEFFICIENTS IF
C     IGAIL=1
C
      IF(IGAIL .EQ. 1) CALL GAIEXP(LJ,SKJ,ETAJ,IASY,NCHAN,IPFLG(2),
     X                             GA1,GA2,GAC1,GAC2,JOPEN,IWRITE)
C
C     IF IRAD=1 ESTIMATE THE CONVERGENCE RADIUS OF THE SELECTED
C     EXPANSION
C
      IF(IRAD.EQ.1) CALL CONRAD(JCHAN,ET,NCHAN,RAFC,IWRITE,IASY,EPSL,
     X                            RSCALX,IPFLG,GAC1,GAC2,GA1,GA2)
C
C     IF IRAD=2 USE ESTIMATED RADIUS IN ASYWFN
      IF(IRAD.EQ.2) RAFIN=RAFC
C
      ENDIF
C
C     COMPUTE THE ASYMPTOTIC FUNCTIONS AT THE REQUIRED MATCHING POINT
C
      CALL ASYWFN(JCHAN,LJ,RAFIN,SKJ,ETJ,ETAJ,JOPEN,GA1,GA2,
     X            NCHAN,IASY,nfc,F,FP,RSCALX,IFAIL)
      IF(IFAIL.NE.0) WRITE(IWRITE,1011) WARN(IFAIL),J,LJ,ET(J),ET(1)
 1011 FORMAT(' ASYWFN ',A,' IN CHANNEL J =',I3,'  LJ =',I3,'  ET(J) = '
     1 ,F10.5,'   ET(1) = ',F10.5,' RYD')
C
   80 CONTINUE
C
C     END OF LOOP OVER SOLUTIONS
C
      RETURN
C
   90 WRITE(IWRITE,1010) J,ETJ,DEGENY
      ifail = 3
      RETURN
C
      END
      SUBROUTINE ASYWF0(NCHAN,iasy,nfc,F,FP,EPS0,ION0,IWRIT0,IGAIL0,
     * IPFLG0)
C
C***********************************************************************
C
C     ASYWFN COMPUTES THE ASYMPTOTIC SCATTERING FUNCTIONS, F, AND
C            THEIR DERIVATIVES, FP.
C
C            NCHAN     IS THE NUMBER OF CHANNELS
C            ION       RESIDUAL CHARGE (IONICITY)
C            IASY      IS THE NUMBER IF TERMS IN THE EXPANSION
C            IWRITE    UNIT NUMBER OF THE PRINTER OR OUTPUT DEVICE
C            NFC       DIMENSION OF WORK ARRAYS FOR FUNCTION ROUTINES
C            IASX      DIMENSION OF WORK ARRAYS FOR CFRACT AND DFRACT
C                      IASX = 2 * ( (IASY+1)/2 )
C            IGAIL     = 0  BURKE-SCHEY ASYMPTOTIC EXPANSION
C                      = 1  GAILITIS ASYMPTOTIC EXPANSION
C                      = 2  COULOMB/BESSEL FUNCTIONS
C            IPFLG     = 0  NO DIAGNOSTIC PRINTOUT
C                      = 1  PRINT DIAGNOSTIC INFORMATION
C            FC,FCP    REGULAR COULOMB FUNCTIONS AND DERIVATIVES
C            GC,GCP    IRREGULAR COULOMB FUNCTIONS AND DERIVATIVES
C
C            XG,YG     WORK ARRAYS FOR ROUTINES CFRACT AND DFRACT
C            XA,YA
C            XC,YC
C            XD,YD
C
C            F,FP      ASYMPTOTIC FUNCTIONS AND DERIVATIVES
C
C            SUBROUTINES CALLED :
C
C                      COULFG     OPEN CHANNEL COULOMB WAVEFUNCTIONS
C                      DECAY      CLOSED CHANNEL NEUTRAL WAVEFUNCTIONS
C                      COULN      CLOSED CHANNEL CHARGED WAVEFUNCTIONS
C                      BKSHWF     BURKE-SCHEY EXPANSION FUNCTIONS
C                      CFRACT     CONVERTS COMPLEX SERIES TO RATIONAL
C                                 FRACTION USING NESBET ALGORITHM
C                      DFRACT     REAL VERSION OF CFRACT
C                      CFEVAL     EVALUATE RATIONAL FORM OF COMPLEX
C                                 CONTINUED FRACTION
C                      DFEVAL     REAL VERSION OF CFEVAL
C                      HSUM       EVALUATE POLYNOMIAL BY HORNER'S
C                                 ALGORITHM
C
C***********************************************************************
C
      IMPLICIT DOUBLE PRECISION(A-H,O-Z)
C
      DIMENSION FC(nfc),GC(nfc),FCP(nfc),GCP(nfc),GA1(NCHAN,IASY),
     X          F(NCHAN,NCHAN,2),FP(NCHAN,NCHAN,2),GA2(NCHAN,IASY),
     X          XG(IASY),YG(IASY),XA(IASY),YA(IASY),XC(iasy+1),
     X          YC(iasy+1),XD(iasy+1),YD(iasy+1)
      save
C
      DATA  ZERO/0.0D0/,ONE/1.0D0/,EPSC/1.D-10/
C
 1000 FORMAT(' COULFG : LJ =',I6,' RHO =',D16.8,' ETAJ =',D16.8)
 1010 FORMAT(' RAFIN =',D16.8,' SKJ =',D16.8,' F =',D16.8,' G =',D16.8,
     X       ' FP =',D16.8,' GP =',D16.8,/,' IFAIL =',I6,/)
 1011 FORMAT(' COULFG FAILED FOR J =',I3,'   LJ =',I3,'   SKJ =',D16.8,
     X       '    IFAIL =',I2,/)
 1020 FORMAT(' NONZERO GAILITIS COEFFICIENTS',I3,' TO',I3,' PASSED',
     X       ' TO CFRACT',/)
 1030 FORMAT(' XA(',I3,') =',D30.15,5X,' YA(',I3,') =',D30.15)
 1040 FORMAT(' ASYWFN WARNING :',/,2I3,' F1  =',D16.8,' DEL =',D16.8,
     X       ' F2  =',D16.8,' DEL =',D16.8,/,6X,' FP1 =',D16.8,
     X       ' DEL =',D16.8,' FP2 =',D16.8,' DEL =',D16.8)
 1050 FORMAT(' COULN : IERR =',I2,' L =',I3,' E=',F10.5,' NLAST =',I3,
     X       ' EPS =',D8.3,' EFX =',D8.3,' FNORM =',D20.12)
 1060 FORMAT(' COUL/DECAY : LJ =',I3,' RAFIN =',D16.8,' ETJ =',D16.8,
     X       ' ZZ =',D16.8)
 1070 FORMAT(' LJ =',I3,' RAFIN =',D30.15,' SKJ =',D30.15,/,
     X       ' F1 =',D30.15,' FP1 =',D30.15)
 1080 FORMAT(' NONZERO GAILITIS COEFFICIENTS',I3,' TO',I3,' PASSED',
     X       ' TO DFRACT',/)
 1090 FORMAT(' XA(',I3,') =',D30.15)
 1100 FORMAT(' ASYWFN WARNING :',/,2I3,' F =',D16.8,' DEL1 =',D16.8,
     X       ' FP =',D16.8,' DEL2 =',D16.8)
C
C     INITIALIZE SOLUTION AND DERIVATIVE ARRAYS, F, FP
C
      EPS = EPS0
      ION = ION0
      IWRITE = IWRIT0
      IGAIL = IGAIL0
      IPFLG = IPFLG0
      DO KDD=1,2
        DO JP=1,NCHAN
          DO IP=1,NCHAN
            F(IP,JP,KDD)=ZERO
            FP(IP,JP,KDD)=ZERO
          END DO
        END DO
      END DO
C
      RETURN
C
C     MAIN ENTRY POINT : COMPUTE ASYMPTOTIC FUNCTIONS FOR CHANNEL J
C
      ENTRY ASYWFN(J,LJ,RAFIN,SKJ,ETJ,ETAJ,JOPEN,GA1,GA2,NCHAN,IASY,
     X       nfc,F,FP,RSCALX,IFAIL)
C
      IFAIL = 0
      IF(JOPEN .EQ. 0) GO TO 120
C
C     OPEN CHANNEL CASE ...
C
      IF(IGAIL .EQ. 0) GO TO 20
C
      ROOTK=DSQRT(SKJ)
      RHO=RAFIN*SKJ
      FLJ=Dble(LJ)
C
C     COMPUTE COULOMB FUNCTIONS
C
      IF(IPFLG .EQ. 1) WRITE(IWRITE,1000)LJ,RHO,ETAJ
      CALL COULFG(RHO,ETAJ,FLJ,FLJ,FC,GC,FCP,GCP,1,0,IFAIL)
      IF(IFAIL.GT.0) THEN
        WRITE(IWRITE,1011) J,LJ,SKJ,IFAIL
        IFAIL = 2
      ENDIF
C
      F1=FC(LJ+1)/ROOTK
      G1=GC(LJ+1)/ROOTK
      FP1=FCP(LJ+1)*ROOTK
      GP1=GCP(LJ+1)*ROOTK
      IF(IPFLG.EQ.1) WRITE(IWRITE,1010)RAFIN,SKJ,F1,G1,FP1,GP1,IFAIL
      GO TO 30
C
C     BURKE-SCHEY EXPANSION FUNCTIONS
C
   20 CALL BKSHWF(LJ,RAFIN,SKJ,ETAJ,ION,JOPEN,F1,G1,FP1,GP1,IWRITE)
C
   30 DO 110 I=1,NCHAN
C
C     NO EXPANSION IF IGAIL = 2
C
      IF(IGAIL .EQ. 2) GO TO 100
C
C     DETERMINE THE LAST ZERO COEFFICIENT
C
      ILZ=0
      DO 40 IP1=1,IASY
      IF(GA1(I,IP1) .NE. ZERO .OR. GA2(I,IP1) .NE. ZERO) GO TO 40
      ILZ=IP1
   40 CONTINUE
      ILZ1=ILZ+1
      ILZ2=ILZ+2
C
      IF(ILZ1 .GT. IASY) GO TO 70
      XG0=GA1(I,ILZ1)
      YG0=GA2(I,ILZ1)
      NCFCT=0
      IF(ILZ1 .EQ. IASY) GO TO 70
C
C     PUT GAILITIS COEFFICIENTS INTO ARRAYS XG AND YG
C
      IF(IPFLG .NE. 0) WRITE(IWRITE,1020)ILZ1,IASY
C
      IY1=0
      DO 50 IP1=ILZ2,IASY
      IY1=IY1+1
      XG(IY1)=GA1(I,IP1)
      YG(IY1)=GA2(I,IP1)
   50 CONTINUE
      NCFCT=IY1
C
C     CALL CONTINUED FRACTION ROUTINE TO CONVERT GAILITIS SERIES
C
      CALL CFRACT(NCFCT,XG0,YG0,XG,YG,XA,YA,XC,YC,XD,YD)
C
      IF(IPFLG .EQ. 0) GO TO 70
C
C     PRINT COEFFICIENTS OF CONTINUED FRACTION
C
      DO 60 IY1=1,NCFCT
      WRITE(IWRITE,1030)IY1,XA(IY1),IY1,YA(IY1)
   60 CONTINUE
C
C     INITIALIZE
C
   70 SUM1=ZERO
      SUM2=ZERO
      DSUM1=ZERO
      DSUM2=ZERO
      EFR=ZERO
      EFI=ZERO
      EFPR=ZERO
      EFPI=ZERO
C
      IF(ILZ .EQ. 0) GO TO 80
C
C     COMPUTE ASYMPTOTIC FUNCTIONS
C
      ILZZ=ILZ-1
      CALL HSUM(ILZZ,NCHAN,RAFIN,GA1(I,1),GA1(I,2),SUM1,DSUM1)
      CALL HSUM(ILZZ,NCHAN,RAFIN,GA2(I,1),GA2(I,2),SUM2,DSUM2)
C
   80 IP=ILZ
      IF(ILZ1 .GT. IASY) GO TO 90
C
C     COMPUTATION OF NUMERATOR AND DENOMINATOR POLYNOMIALS
C
      CALL CFEVAL(NCFCT,IP,RAFIN,XG0,YG0,XC,YC,XD,YD,RFR,RFI,EFR,EFI,
     X            RDR,RDI,EFPR,EFPI,IWRITE,IPFLG)
C
      SUM1=SUM1+RFR
      SUM2=SUM2+RFI
      DSUM1=DSUM1+RDR
      DSUM2=DSUM2+RDI
C
   90 SUM1=RSCALX*SUM1
      SUM2=RSCALX*SUM2
      DSUM1=RSCALX*DSUM1
      DSUM2=RSCALX*DSUM2
C
C     COMPUTE FUNCTIONS
C
      F(I,J,1)=SUM1*F1+SUM2*G1
      F(I,J,2)=SUM1*G1-SUM2*F1
C
C     COMPUTE DERIVATIVES
C
      FP(I,J,1)=SUM1*FP1+SUM2*GP1+DSUM1*F1+DSUM2*G1
      FP(I,J,2)=SUM1*GP1-SUM2*FP1+DSUM1*G1-DSUM2*F1
C
C     ESTIMATE ERRORS
C
      DEL1=DABS(RSCALX*(EFR*F1+EFI*G1))
      DEL2=DABS(RSCALX*(EFR*G1-EFI*F1))
      DEL3=DABS(RSCALX*(EFR*FP1+EFI*GP1+EFPR*F1+EFPI*G1))
      DEL4=DABS(RSCALX*(EFR*GP1-EFI*FP1+EFPR*G1-EFPI*F1))
C
      IF(DEL1 .GT. EPS .OR. DEL2 .GT. EPS .OR. DEL3 .GT. EPS
     X   .OR. DEL4 .GT. EPS) IFAIL = 1
      IF(IPFLG .NE. 0) WRITE(IWRITE,1040)
     X   I,J,F(I,J,1),DEL1,F(I,J,2),DEL2,FP(I,J,1),DEL3,FP(I,J,2),DEL4
      GO TO 110
C
C     COULOMB/BESSEL FUNCTION CASE, IGAIL = 2
C
  100 IF(I .NE. J) GO TO 110
      F(I,J,1)=F1
      F(I,J,2)=G1
      FP(I,J,1)=FP1
      FP(I,J,2)=GP1
C
  110 CONTINUE
C
      RETURN
C
C***********************************************************************
C
C     CLOSED CHANNEL CASE ...
C
C***** CORRECTED LAM 11/1/85
  120 IF(IGAIL .EQ. 0) GO TO 150
C
      ZZ =Dble(ION)
C
C     COMPUTE CLOSED CHANNEL COULOMB FUNCTIONS
C
      IF(IPFLG .EQ. 1) WRITE(IWRITE,1060)LJ,RAFIN,ETJ,ZZ
C
      IF(ION .NE. 0) THEN
        CALL COULN(LJ,ZZ,ETJ,RAFIN,U,UDER,EPSC,EFX,IERR,NLAST,FNORM)
        IF(IERR.EQ.1.OR.IERR.EQ.2 .OR. DABS(FNORM-ONE) .GT. EPSC)
     X    WRITE(IWRITE,1050) IERR,LJ,ETJ,NLAST,EPSC,EFX,FNORM
      ELSE
        CALL DECAY(ETJ,LJ,RAFIN,U,UDER,IWRITE)
      ENDIF
C
      ROOTK=DSQRT(SKJ)
      F1=U/ROOTK
      FP1=UDER/ROOTK
      IF(IPFLG .EQ. 1) WRITE(IWRITE,1070)LJ,RAFIN,SKJ,F1,FP1
      GO TO 160
C
C     BURKE-SCHEY EXPANSION FUNCTIONS
C
  150 CALL BKSHWF(LJ,RAFIN,SKJ,ETAJ,ION,JOPEN,G1,F1,GP1,FP1,IWRITE)
C
C     COMPUTE ASYMPTOTIC FUNCTIONS
C
  160 DO 240 I=1,NCHAN
C
C     NO EXPANSION IF COULOMB/BESSEL FUNCTIONS ARE USED ( IGAIL = 2 )
C
      IF(IGAIL .EQ. 2) GO TO 230
C
C     DETERMINE THE LAST ZERO COEFFICIENT
C
      ILZ=0
      DO 170 IP1=1,IASY
      IF(GA2(I,IP1) .NE. ZERO) GO TO 170
      ILZ=IP1
  170 CONTINUE
      ILZ1=ILZ+1
      ILZ2=ILZ+2
C
      IF(ILZ1 .GT. IASY) GO TO 200
      XG0=GA2(I,ILZ1)
      NCFCT=0
      IF(ILZ1 .EQ. IASY) GO TO 200
C
C     PUT GAILITIS COEFFICIENTS INTO ARRAY XG
C
      IF(IPFLG .NE. 0) WRITE(IWRITE,1080)ILZ1,IASY
C
      IY1=0
      DO 180 IP1=ILZ2,IASY
      IY1=IY1+1
      XG(IY1)=GA2(I,IP1)
  180 CONTINUE
      NCFCT=IY1
C
C     CALL CONTINUED FRACTION ROUTINE TO CONVERT GAILITIS SERIES
C
      CALL DFRACT(NCFCT,XG0,XG,XA,XC,XD)
C
C     PRINT COEFFICIENTS OF CONTINUED FRACTION
C
      IF(IPFLG .EQ. 0) GO TO 200
C
      DO 190 IY1=1,NCFCT
      WRITE(IWRITE,1090)IY1,XA(IY1)
  190 CONTINUE
C
C     INITIALIZE
C
  200 SUM1=ZERO
      DSUM1=ZERO
      EFR=ZERO
      EFPR=ZERO
C
C     COMPUTE SUM OF SERIES
C
      IF(ILZ .EQ. 0) GO TO 210
C
      ILZZ=ILZ-1
      CALL HSUM(ILZZ,NCHAN,RAFIN,GA2(I,1),GA2(I,2),SUM1,DSUM1)
C
  210 IP=ILZ
      IF(ILZ1 .GT. IASY) GO TO 220
C
C     COMPUTATION OF NUMERATOR AND DENOMINATOR POLYNOMIALS
C
      CALL DFEVAL(NCFCT,IP,RAFIN,XG0,XC,XD,RFR,EFR,RDR,EFPR,IWRITE,
     X            IPFLG)
C
      SUM1=SUM1+RFR
      DSUM1=DSUM1+RDR
C
C     RESCALE SERIES
C
  220 SUM1=RSCALX*SUM1
      DSUM1=RSCALX*DSUM1
C
C     COMPUTE ASYMPTOTIC FUNCTIONS
C
      F(I,J,1)=F1*SUM1
      FP(I,J,1)=FP1*SUM1+F1*DSUM1
C
C     ESTIMATE ERRORS
C
      DEL1=DABS(RSCALX*F1*EFR)
      DEL2=DABS(RSCALX*(FP1*EFR+F1*EFPR))
C
      IF(DEL1.GT.EPS .OR. DEL2.GT.EPS) IFAIL = 1
      IF(IPFLG.NE.0) WRITE(IWRITE,1100) I,J,F(I,J,1),DEL1,FP(I,J,1),DEL2
      GO TO 240
C
C     COULOMB/BESSEL FUNCTION CASE, IGAIL = 2
C
  230 IF(I .NE. J) GO TO 240
      F(I,J,1)=F1
      FP(I,J,1)=FP1
C
  240 CONTINUE
C
      RETURN
      END
      SUBROUTINE COULN(LX,Z,E,R,FX,FXP,ACC,EFX,IERR,NLAST,FNORM)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
C***********************************************************************
C
C     COULN COMPUTES THE EXPONENTIALLY DECAYING WHITTAKER FUNCTION
C           AND ITS DERIVATIVE USING THE ASYMPTOTIC EXPANSION CONVERTED
C           TO A CONTINUED FRACTION
C
C                FX = W( KAPPA, MU, ZC )
C
C                KAPPA = 1 / C = Z / DSQRT( -E )
C                MU    = LX + 1/2
C                ZC    = 2 * DSQRT( -E ) * R
C
C      REFERENCE : HANDBOOK OF MATHEMATICAL FUNCTIONS
C                  M. ABRAMOWITZ AND I.A. STEGUN
C                  ( DOVER, NEW YORK, 1965 )   EQUATION : 13.1.33
C
C***********************************************************************
C
C     LX      ORBITAL ANGULAR MOMENTUM ( .GE. ZERO )
C     Z       CHARGE  ( NONZERO )
C     E       ENERGY IN RYDBERGS ( .LT. ZERO )
C     R       RADIAL COORDINATE IN BOHR UNITS
C     ACC     RELATIVE ACCURACY REQUIRED
C
C     FX      WHITTAKER FUNCTION
C     FXP     DERIVATIVE OF WHITTAKER FUNCTION WITH RESPECT TO R
C     FNORM   NORMALIZATION FACTOR MULTIPLYING BOTH FX AND FXP
C     EFX     ESTIMATE OF RELATIVE ERROR IN VALUE OF FX RETURNED
C
C     IERR    = 0   FOR NORMAL RETURN WITH REQUESTED ACCURACY
C             = 1   VALUE RETURNED WITH REDUCED ACCURACY
C             = 2   INPUT ERROR, FX, FXP ZEROED
C             = 3   NORMAL RETURN, SERIES TERMINATES
C     NLAST   NUMBER OF TERMS USED IN THE ASYMPTOTIC EXPANSION
C
C     XG,XA,XC,XD    WORK ARRAYS, LENGTH OF EACH = KKKMAX * NTERM
C                    COEFFICIENTS ARE COMPUTED IN UP TO KKKMAX BLOCKS
C                    EACH BLOCK CONSISTS OF NTERM COEFFICIENTS
C
C     MACHINE DEPENDENT CONSTANTS :
C
C     TMAX     MAGNITUDE OF LARGEST COEFFICIENT TO BE USED
C     SCAL     MAGNITUDE OF SMALLEST NUMBER TO BE USED
C              DETERMINES THE SCALING OF THE SERIES COEFFICIENTS
C     IXMAX    ARGUMENT RANGE PARAMETER OF THE INTRINSIC EXPONENTIAL
C              FUNCTION
C
C     ROUTINES CALLED :       DFRACT, HSUM
C     INTRINSIC FUNCTIONS :   IDINT, MAX0, DABS, DEXP, DBLE, DLOG,
C                             DMIN1, DSQRT
C
C***********************************************************************
C
      parameter (IXMAX=650,KKKMAX=20,NTERM=10)
      DIMENSION XG(kkkmax*nterm),XA(kkkmax*nterm),XC(kkkmax*nterm),
     * XD(kkkmax*nterm)
C
      DATA ZERO/0.0D0/,ONE/1.0D0/,TWO/2.0D0/,TEN/10.0D0/,HALF/0.5D0/
      DATA SCAL/1.0D-50/,TMAX/1.0D+60/,EPS/1.0D-10/,AK5/1.0D+05/
C
      IERR=0
      ITERM=0
      FNORM=ONE
      XMAX=DBLE(IXMAX)
C
      IF(E .GE. ZERO .OR. R .LE. ZERO .OR. Z .EQ. ZERO .OR. LX .LT. 0)
     X   GO TO 140
C
      AK=DSQRT(-E)
      ZI=ONE/Z
      C=AK*ZI
      X=Z*R
      XI=ONE/X
      XI2=XI*XI
      CX=AK*R
      SCALE=SCAL*DEXP(DMIN1(CX,-DLOG(SCAL)))
      RSCALE=ONE/SCALE
      C1=ONE/C
      NC=IDINT(C1+HALF)
C
      NFIRST=1
      NLAST=NTERM
C
C     CHECK FOR FINITE SERIES SPECIAL CASES
C
      IF(DABS(Dble(NC)-C1) .LT. EPS .AND. NC .GE. -LX) ITERM=1
C
      L=LX
      LM=0
      IF(ITERM .EQ. 1) LM=NC+1
      IF(X .GE. ZERO) LM=MAX0(LM,IDINT(DSQRT(X)+HALF))
      IF(LX .GT. LM) L=LM
C
      L1=L+1
      FL=Dble(L)
      FL1=Dble(L1)
C
      IF(ITERM .EQ. 0) GO TO 20
      IF(NC .GE. L1) GO TO 10
      NLAST=L+NC
      GO TO 20
   10 NLAST=NC-L1
C
C     COMPUTE EXPANSION COEFFICIENTS FOR ASYMPTOTIC SERIES L = LM
C     ( IBID. EQUATION : 13.5.2 )
C
   20 Q=SCALE
      XG0=Q
      G1=FL-C1
      G2=-FL1-C1
      G3=-HALF*C1
C
      DO 90 KKK=1,KKKMAX
C
      IF(NLAST .LT. NFIRST) GO TO 50
      DO 30 N=NFIRST,NLAST
      AN=Dble(N)
      Q=(G1+AN)*(G2+AN)*G3*Q/AN
      IF(DABS(Q) .GT. TMAX) GO TO 100
      XG(N)=Q
   30 CONTINUE
      IF(ITERM .EQ. 1) GO TO 50
C
C     OBTAIN COEFFICIENTS OF THE CONTINUED FRACTION USING THE NESBET
C     ALGORITHM
C
      CALL DFRACT(NLAST,XG0,XG,XA,XC,XD)
C
      KMAX=(NLAST+1)/2
      KMAX1=KMAX-1
      NC=KMAX+KMAX-1
C
C     EVALUATE THE CONTINUED FRACTION USING THE RATIONAL FORM
C
      NCP=NC+1
      SN1=XC(NC)
      SN2=XC(NCP)
      AI=Dble(KMAX)
      SNP1=AI*SN1
      SNP2=AI*SN2
C
      SD1=XD(NC)
      SD2=XD(NCP)
      SDP1=AI*SD1
      SDP2=AI*SD2
C
      DO 40 K=1,KMAX1
      NC=NC-2
      NCP=NC+1
      AI=Dble(KMAX-K)
C
      SN1=XC(NC)+XI*SN1
      SN2=XC(NCP)+XI*SN2
      SNP1=AI*XC(NC)+XI*SNP1
      SNP2=AI*XC(NCP)+XI*SNP2
C
      SD1=XD(NC)+XI*SD1
      SD2=XD(NCP)+XI*SD2
      SDP1=AI*XD(NC)+XI*SDP1
      SDP2=AI*XD(NCP)+XI*SDP2
C
      IF(DABS(SN1)+DABS(SN2)+DABS(SD1)+DABS(SD2) .GT. TEN*TMAX*DABS(X))
     X   GO TO 100
      IF(DABS(SNP1)+DABS(SNP2)+DABS(SDP1)+DABS(SDP2) .GT. AK5*TMAX*X*X)
     X   GO TO 100
   40 CONTINUE
C
      SN1=XG0+XI*SN1
      SN2=XG0+XI*SN2
      SNP1=-XI2*SNP1
      SNP2=-XI2*SNP2
C
      SD1=ONE+XI*SD1
      SD2=ONE+XI*SD2
      SDP1=-XI2*SDP1
      SDP2=-XI2*SDP2
C
C     OBTAIN FUNCTIONS AND ERROR ESTIMATES
C
      F=SN1/SD1
      F1=SN2/SD2
      EF=DABS(F-F1)
C
      FP=(SNP1-F*SDP1)/SD1
      FP1=(SNP2-F1*SDP2)/SD2
      EFP=DABS(FP-FP1)
      GO TO 60
C
C     EVALUATE FINITE SERIES CASES USING HORNERS ALGORITHM
C
   50 CALL HSUM(NLAST,1,X,XG0,XG,F,FP)
      EFX=ZERO
      EFXP=ZERO
      IERR=3
C
   60 XARG=-CX+C1*DLOG(TWO*CX)
      IF(XARG .LT. XMAX) GO TO 70
      XARG=XARG-XMAX
      FNORM=DEXP(-XMAX)
      GO TO 80
C
   70 IF(XARG .GT. -XMAX) GO TO 80
      XARG=XARG+XMAX
      FNORM=DEXP(XMAX)
C
   80 FAC=DEXP(XARG)
      FX=FAC*F
      FXP=Z*((ONE/CX-C)*F+FP)*FAC
      IF(ITERM .EQ. 1) GO TO 110
C
      EFX=EF/DMAX1(DABS(F),SCAL)
      EFXP=EFP/DMAX1(DABS(FP),SCAL)
      IF(DABS(EFX) .LT. ACC .AND. DABS(EFXP) .LT. ACC) GO TO 110
C
      NFIRST=NLAST+1
      NLAST=NLAST+NTERM
C
   90 CONTINUE
C
  100 IERR=1
      NLAST=NLAST-NTERM
      IF(KKK .EQ. 1) GO TO 140
C
  110 FX=RSCALE*FX
      FXP=RSCALE*FXP
C
C     UPWARDS RECURRENCE ON L IF LX GREATER THAN LM
C
      IF(LX .LE. LM) GO TO 130
C
      VP=FX
      VQ=FXP
      DO 120 L=L1,LX
      AL=Dble(L)
      AL2=Dble(L*L)
      FX=((ONE-AL2*XI)*VP+AL*VQ*ZI)/(ONE-C*AL)
      FXP=Z*((ONE-AL2*XI)*FX-(ONE+C*AL)*VP)/AL
      VP=FX
      VQ=FXP
  120 CONTINUE
C
  130 RETURN
C
  140 IERR=2
      FX=ZERO
      FXP=ZERO
      RETURN
C
      END
      FUNCTION CPHAZ(L,ETA,IWRITE)
C
C***********************************************************************
C
C     CPHAZ EVALUATES THE COULOMB PHASE, SIGMA(L)
C           SIGMA(L) = ARG (  GAMMA(L+1+I*ETA) )
C
C     NORMALLY, FOR ELECTRON SCATTERING ...
C
C           ETA = -Z / K       WHERE Z IS THE RESIDUAL NUCLEAR CHARGE
C                                    K IS THE ELECTRONIC MOMENTUM
C                              ( ALL QUANTITIES IN ATOMIC UNITS )
C
C***********************************************************************
C
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
      DATA HALF/0.5D0/,ZERO/0.0D0/,TWO/2.0D0/,IP1/9/
      DATA C1/0.83333333333333333D-01/,C2/0.27777777777777778D-02/,
     X     C3/0.79365079365079365D-03/,C4/0.59523809523809523D-03/,
     X     C5/0.84175084175084175D-03/
C
 1000 FORMAT(' COULOMB PHASE, SIGMA(',I3,';',D16.8,' ) =',D30.17)
C
C     ARG ( GAMMA (L+1+I*ETA) ) = IMAG ( LN GAMMA (P+I*ETA) )
C                                 - SUM( S=L+1 TO S=P-1 ) ATAN ( ETA/S )
C     FOR SOME INTEGER P
C
C     HERE TAKE P=10
C
      D1=ZERO
      B=ETA
      K1=L+1
      IF (K1 <= IP1) THEN
        GO TO 10
      ELSE
        GO TO 30
      END IF
C
C     SUM ARC TANGENT TERMS
C
   10 DO 20 J=K1,IP1,1
      A=Dble(J)
      D=DATAN2(B,A)
      D1=D+D1
   20 CONTINUE
      A=Dble(IP1+1)
      GO TO 40
C
   30 A=Dble(L+1)
      D1=ZERO
C
C     EVALUATE LN ( GAMMA (P-I*ETA) ) USING AN ASYMPTOTIC FORMULA
C     REFERENCE: N.B.S. TABLES EQNS 6.1.40 AND 6.1.41
C
   40 C=HALF*DLOG(A*A+B*B)
      D=DATAN2(B,A)
      D=D*(A-HALF)+C*B
      D=D-B
C
C     EVALUATE  1 / (P+I*ETA) AND ( 1 / (P+I*ETA) ) **2
C
      E=A*A+B*B
      A=A/E
      E=-B/E
      F=A*A-E*E
      G=TWO*E*A
C
C     -R1 = H  = 1/1188 F - 1/1680
C     -I1 = H1 = 1/1188 G
C
      H=C5*F-C4
      H1=C5*G
      H2=H
C
C     R2 = H  = 1/1260 + F * ( -R1 ) - G * ( -I1 )
C     I2 = H1 =          F * ( -I1 ) + G * ( -R1 )
C
      H=H*F-H1*G
      H1=H1*F+H2*G
      H=H+C3
      H2=H
C
C     -R3 = H  = -1/360 + F * ( R2 ) - G * ( I2 )
C     -I3 = H1 =          F * ( I2 ) + G * ( R2 )
C
      H=H*F-H1*G
      H1=H1*F+H2*G
      H=H-C2
      H2=H
C
C     R4 = H  = 1/12 + F * ( -R3 ) - G * ( -I3 )
C     I4 = H1 =        F * ( -I3 ) + G * ( -R3 )
C
      H=H*F-H1*G
      H1=H1*F+H2*G
      H=H+C1
      H2=H
C
C     EVALUATE IMAG ( ( R4 + I*I4 ) / ( P + I*ETA ) )
C
      H1=H1*A+H2*E
C
C     CALCULATE FINAL VALUE FOR PHASE SHIFT
C
      CPHAZ=H1+D-D1
C
C      WRITE(IWRITE,1000)L,ETA,CPHAZ
C
      RETURN
      END
      SUBROUTINE BKSHEY(J,SKJ,ETAJ,ET,L2P,NCHAN,ION,CF,LAMAX,JOPEN,
     X                  IASY,DEGENY,IPFLG,IWRITE,GA1,GA2,SCALX)
C
C***********************************************************************
C
C     ROUTINE BKSHEY COMPUTES THE COEFFICIENTS OF THE ASYMPTOTIC
C                    EXPANSION DESCRIBED BY BURKE AND SCHEY
C
C             J       IS THE CHANNEL SEQUENCE NUMBER
C             SKJ     IS THE CHANNEL MOMENTUM IN ATOMIC UNITS
C                     SKJ = SQRT ( ABS(ET(J)) )  .GE. 0
C             ETAJ    IS THE CHANNEL COULOMB PARAMETER
C                     ETAJ = - ION / SKJ
C             ET      IS THE CHANNEL ENERGIES IN RYDBERGS
C             L2P     ARE THE CHANNEL ORBITAL ANGULAR MOMENTA
C             NCHAN   IS THE NUMBER OF CHANNELS
C             ION     IS THE IONICITY (RESIDUAL CHARGE)
C                     ION = Z - N
C             CF      ARRAY OF ASYMPTOTIC POTENTIAL EXPANSION
C                     COEFFICIENTS .. IN RYDBERGS
C                     CF(NCHAN,NCHAN,LAMAX)
C             LAMAX   MAXIMUM MULTIPOLE IN POTENTIAL EXPANSION
C             JOPEN   =1 IF THE CHANNEL IS OPEN
C                     =0 IF THE CHANNEL IS CLOSED
C             IASY    NUMBER OF TERMS OF THE B-S EXPANSION TO BE
C                     CALCULATED
C             DEGENY  CRITERION FOR CONSIDERING CHANNELS DEGENERATE
C             IPFLG   =1 PRINT THE EXPANSION COEFFICIENTS
C                     =0 DO NOT PRINT THE EXPANSION COEFFICIENTS
C             IWRITE  UNIT NUMBER IF THE PRINTER
C
C             GA1     BURKE-SCHEY EXPANSION COEFFICIENTS
C             GA2
C                     GA1(NCHAN,IASY), GA2(NCHAN,IASY)
C
C***********************************************************************
C
C     NOTE :
C           THIS ROUTINE INCORPORATES AN AUTOMATIC SCALING FEATURE
C           THE VALUE OF SCALX IS ADJUSTED TO KEEP THE COMPUTED B.S.
C           COEFFICIENTS WITHIN A RANGE SPECIFIED BY VARIABLES UNFL
C           AND OVFL.
C           IF THE VALUE OF A COEFFICIENT FALLS OUTSIDE THIS RANGE
C           AN ATTEMPT IS MADE TO ALTER THE VALUE OF SCALX ..
C           IF THE NEW VALUE OF SCALX WOULD RESULT IN UNDERFLOWS AN
C           ERROR MESSAGE IS PRINTED AND THE CALCULATION IS TERMINATED.
C
C***********************************************************************
C
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
      DIMENSION ET(NCHAN),L2P(NCHAN),CF(NCHAN,NCHAN,*),
     X          GA1(NCHAN,IASY),GA2(NCHAN,IASY)
C
      DATA ZERO/0.0D0/,SCAL1/1.0D-10/,UNFL/1.0D-158/,OVFL/1.0D+168/
C
 1000 FORMAT(1X,8D14.6)
 1010 FORMAT(/,35X,'EXPANSION COEFFICIENTS WITH J= ',I2)
 1020 FORMAT(35X,'---------------------------------')
 1030 FORMAT(/,35X,'IMAGINARY BURKE AND SCHEY COEFF.',//)
 1040 FORMAT(/,35X,'REAL BURKE AND SCHEY COEFF.')
 1050 FORMAT(' RANGE OVERFLOW IN BKSHEY ...',/,' REDUCE THE NUMBER OF',
     X       ' TERMS IN THE ASYMPTOTIC EXPANSION AT LEAST TO IASY =',I3)
C
      ETJ=ET(J)
      S2K=SKJ+SKJ
      ETA2=ETAJ*ETAJ
C
C     SET RANGE VALUES
C
      CMAX=SCALX
      CMIN=SCALX
C
C     INITIALIZE ARRAYS TO HOLD EXPANSION COEFFICIENTS
C
      DO IP1=1,IASY
        DO I=1,NCHAN
          GA1(I,IP1)=ZERO
          GA2(I,IP1)=ZERO
        END DO
      END DO
      IF(JOPEN .EQ. 0) GO TO 20
      GA1(J,1)=SCALX
      GO TO 30
   20 GA2(J,1)=SCALX
C
C     RECURRENCE PROCEDURE FOR BURKE-SCHEY EXPANSION COEFFICIENTS
C
   30 D1=ZERO
      D2=ETA2
      IF(JOPEN .EQ. 0) D2=-ETA2
      D3=-ETAJ
C
      DO IP1=2,IASY
C
      IP=IP1-1
      IM=IP-1
      DIM=Dble(IM)
      C1=D1
      C2=D2
      C3=D3
      D1=D1+S2K
      D2=D2-DIM-DIM
      D3=D3+ETAJ+ETAJ
C
      DO I=1,NCHAN
      LI=L2P(I)
      LLI=LI*(LI+1)
      DEIJ=ET(I)-ETJ
      IF(DABS(DEIJ) .GT. DEGENY) GO TO 90
C
C     ***********     DEGENERATE ENERGY EXPRESSION    ***********
C
      LM=MIN(IP,LAMAX)
      A1=D2+Dble(LLI)
C
      IF(JOPEN .EQ. 0) GO TO 60
C
C     OPEN CHANNEL CASE ...
C
      SUM1=A1*GA2(I,IP)-D3*GA1(I,IP)
      SUM2=A1*GA1(I,IP)+D3*GA2(I,IP)
C
      IF(LM .EQ. 0) GO TO 50
      DO L=1,LM
        DO IC=1,NCHAN
          A3=CF(I,IC,L)
          SUM1=SUM1+A3*GA2(IC,IP1-L)
          SUM2=SUM2+A3*GA1(IC,IP1-L)
        END DO
      END DO
   50 CF1=-SUM1/D1
      CF2=SUM2/D1
      GA1(I,IP1)=CF1
      GA2(I,IP1)=CF2
      GO TO 170
C
C     CLOSED CHANNEL CASE ...
C
   60 SUM2=(A1-D3)*GA2(I,IP)
C
      IF(LM .EQ. 0) GO TO 80
      DO L=1,LM
        DO IC=1,NCHAN
          SUM2=SUM2+CF(I,IC,L)*GA2(IC,IP1-L)
        END DO
      END DO
   80 CF2=SUM2/D1
      GA2(I,IP1)=CF2
      GO TO 180
C
C     ***********     NON-DEGENERATE ENERGY EXPRESSION    ***********
C
   90 LM=MIN(IM,LAMAX)
      A1=C2+Dble(LLI)
C
C     OPEN CHANNEL CASE ...
C
      IF(JOPEN .EQ. 0) GO TO 130
C
      SUM1=-C1*GA2(I,IP)
      SUM2=C1*GA1(I,IP)
C
      IF(IP1 .EQ. 2) GO TO 100
      SUM1=SUM1+A1*GA1(I,IM)+C3*GA2(I,IM)
      SUM2=SUM2+A1*GA2(I,IM)-C3*GA1(I,IM)
C
  100 IF(LM .EQ. 0) GO TO 120
      DO L=1,LM
        DO IC=1,NCHAN
          A3=CF(I,IC,L)
          SUM1=SUM1+A3*GA1(IC,IP-L)
          SUM2=SUM2+A3*GA2(IC,IP-L)
        END DO
      END DO
C
  120 CF1=SUM1/DEIJ
      CF2=SUM2/DEIJ
      GA1(I,IP1)=CF1
      GA2(I,IP1)=CF2
      GO TO 170
C
C     CLOSED CHANNEL CASE ...
C
  130 SUM2=-C1*GA2(I,IP)
C
      IF(IP1 .EQ. 2) GO TO 140
      SUM2=SUM2+GA2(I,IM)*(A1-C3)
C
  140 IF(LM .EQ. 0) GO TO 160
      DO L=1,LM
        DO IC=1,NCHAN
          SUM2=SUM2+CF(I,IC,L)*GA2(IC,IP-L)
        END DO
      END DO
C
  160 CF2=SUM2/DEIJ
      GA2(I,IP1)=CF2
      GO TO 180
C
C     RESCALING CHECK
C
  170 ACF1=DABS(CF1)
      CMAX=DMAX1(CMAX,ACF1)
      IF(ACF1 .NE. ZERO) CMIN=DMIN1(CMIN,ACF1)
C
  180 ACF2=DABS(CF2)
      CMAX=DMAX1(CMAX,ACF2)
      IF(ACF2 .NE. ZERO) CMIN=DMIN1(CMIN,ACF2)
C
      IF(CMAX .LT. OVFL) CYCLE
C
C     THE COEFFICIENTS MUST BE RESCALED
C
      IF(CMIN .GE. UNFL) GO TO 190
C
C     DYNAMIC RANGE EXCEEDED ... THE NUMBER OF TERMS, IASY,
C                                MUST BE REDUCED
C
      WRITE(IWRITE,1050)IP1
      STOP 62
C
  190 SCALX=SCAL1*SCALX
      CMAX=SCAL1*CMAX
      CMIN=SCAL1*CMIN
C
      DO IQ=1,I
        DO IQ1=1,IP1
          IF(JOPEN .EQ. 0) GO TO 200
          GA1(IQ,IQ1)=SCAL1*GA1(IQ,IQ1)
  200     GA2(IQ,IQ1)=SCAL1*GA2(IQ,IQ1)
        END DO
      END DO
C
      END DO ! I
      END DO ! IP
C
C-----------------------------------------------------------------------
C
C     PRINT EXPANSION COEFFICIENTS IF IPFLG=1
C
      IF(IPFLG .EQ. 0) GO TO 250
C
      WRITE(IWRITE,1010)J
      WRITE(IWRITE,1020)
      WRITE(IWRITE,1030)
      DO 230 I=1,IASY
      WRITE(IWRITE,1000) (GA2(JJ,I),JJ=1,NCHAN)
  230 CONTINUE
C
      IF(JOPEN .EQ. 0) GO TO 250
      WRITE(IWRITE,1040)
      DO 240 I=1,IASY
      WRITE(IWRITE,1000) (GA1(JJ,I),JJ=1,NCHAN)
  240 CONTINUE
C
  250 CONTINUE
      RETURN
      END
      SUBROUTINE GAIEXP(LJ,SKJ,ETAJ,IASY,NCHAN,IPFLG,GA1,GA2,GAC1,GAC2,
     X                  JOPEN,IWRITE)
C
C***********************************************************************
C
C     GAIEXP COMPUTES THE COEFFICIENTS OF THE ASYMPTOTIC EXPANSION
C            DERIVED BY GAILITIS.
C            REFERENCE : M. GAILITIS, J.PHYS.B : AT.MOL.PHYS.9,(1976)
C                        843-854.  EQUATION (15).
C
C            LJ      CHANNEL ORBITAL ANGULAR MOMENTUM
C            SKJ     CHANNEL MOMENTUM (ATOMIC UNITS)
C            ETAJ    COULOMB PARAMETER FOR CHANNEL
C            IASY    NUMBER OF TERMS RETAINED IN BURKE-SCHEY EXPANSION
C            NCHAN   NUMBER OF CHANNELS
C            JOPEN   =1 IF THE CHANNEL IS OPEN
C                    =0 IF THE CHANNEL IS CLOSED
C            IWRITE  UNIT NUMBER OF THE PRINTER OR OUTPUT DEVICE
C            IPFLG   =1 IF PRINTOUT IS REQUIRED
C                    =0 IF PRINTOUT IS NOT REQUIRED
C                     IPFLG(1) REFERS TO THE TRANSFORMATION COEFFICIENTS
C                     IPFLG(2) REFERS TO THE EXPANSION COEFFICIENTS
C
C            GA1,GA2   HOLD BURKE-SCHEY EXPANSION COEFFICIENTS ON INPUT
C                      ON OUTPUT THEY CONTAIN THE GAILITIS COEFFICIENTS
C                      GA1(NCHAN,IASY), GA2(NCHAN,IASY)
C
C            GAC1,GAC2 GAILITIS TRANSFORMATION COEFFICIENTS
C                      GAC1(IASY), GAC2(IASY)
C
C***********************************************************************
C
      IMPLICIT DOUBLE PRECISION(A-H,O-Z)
C
      DIMENSION GA1(NCHAN,IASY),GA2(NCHAN,IASY),GAC1(IASY),GAC2(IASY),
     X          IPFLG(2)
C
      DATA ZERO/0.0D0/,ONE/1.0D0/,CRIT/1.0D-12/
C
 1000 FORMAT(1X,8D14.6)
 1010 FORMAT(/,35X,'IMAGINARY COULOMB COEFF.',//)
 1020 FORMAT(/,35X,'REAL COULOMB COEFF.',//)
 1030 FORMAT(/,35X,'IMAGINARY GAILITIS COEFF.',//)
 1040 FORMAT(/,35X,'REAL GAILITIS COEFF.',//)
 1050 FORMAT(/,35X,'CLOSED CHANNEL GAILITIS COEFF.',//)
C
      LLJ=LJ*(LJ+1)
      S2K=SKJ+SKJ
      ETA2=ETAJ*ETAJ
C
C     GAILITIS TRANSFORMATION COEFFICIENTS
C     REFERENCE : M. GAILITIS, J.PHYS,B: AT.MOL.PHYS 9,(1976)843-854
C                 EQUATION (36)
C
      IF(JOPEN .EQ. 0) GO TO 20
C
C     OPEN CHANNEL CASE ...
C
      GAC1(1)=ONE
      GAC2(1)=ZERO
C
      D1=S2K
      D2=ETA2+Dble(LLJ)
      D3=ETAJ
C
      DO 10 IP1=2,IASY
      IP=IP1-1
      DIP=Dble(IP)
      SUM1=-D2*GAC2(IP)+D3*GAC1(IP)
      SUM2=D2*GAC1(IP)+D3*GAC2(IP)
      GAC1(IP1)=SUM1/D1
      GAC2(IP1)=SUM2/D1
C
      D1=D1+S2K
      D2=D2-DIP-DIP
      D3=D3+ETAJ+ETAJ
   10 CONTINUE
C
      GO TO 40
C
C     CLOSED CHANNEL CASE ...
C
   20 GAC2(1)=ONE
C
      D1=S2K
      D2=-ETA2+Dble(LLJ)
      D3=ETAJ
C
      DO 30 IP1=2,IASY
      IP=IP1-1
      DIP=Dble(IP)
      SUM2=(D2-D3)*GAC2(IP)
      GAC2(IP1)=SUM2/D1
C
      D1=D1+S2K
      D2=D2-DIP-DIP
      D3=D3+ETAJ+ETAJ
C
   30 CONTINUE
C
C-----------------------------------------------------------------------
C
C     PRINT GAILITIS TRANSFORMATION COEFFICIENTS IF IPFLG(1)=1
C
   40 IF(IPFLG(1) .EQ. 0) GO TO 50
C
      WRITE(IWRITE,1010)
      WRITE(IWRITE,1000) (GAC2(I),I=1,IASY)
      IF(JOPEN .EQ. 0) GO TO 50
      WRITE(IWRITE,1020)
      WRITE(IWRITE,1000) (GAC1(I),I=1,IASY)
   50 CONTINUE
C
C-----------------------------------------------------------------------
C
C     DIVIDE THE TWO ASYMPTOTIC SERIES TO GET GAILITIS EXPANSION
C     COEFFICIENTS.
C     REFERENCE : GAILITIS, IBID. EQUATION(38)
C
C-----------------------------------------------------------------------
C
      IF(JOPEN .EQ. 0) GO TO 80
C
C     OPEN CHANNEL CASE ...
C
      DO IP1=2,IASY
      IP=IP1-1
      DO I=1,NCHAN
      SUM1=GA1(I,IP1)
      SUM2=GA2(I,IP1)
      DO 60 IQ1=1,IP
      IQ=IQ1-1
      C1=GA1(I,IQ1)
      C2=GA2(I,IQ1)
      D1=GAC1(IP1-IQ)
      D2=GAC2(IP1-IQ)
      SUM1=SUM1-C1*D1+C2*D2
      SUM2=SUM2-C1*D2-C2*D1
   60 CONTINUE
C
C     AVOID ROUNDOFF ERROR
C
      IF(DABS(SUM1) .LT. DABS(GA1(I,IP1)*CRIT)) SUM1=ZERO
      IF(DABS(SUM2) .LT. DABS(GA2(I,IP1)*CRIT)) SUM2=ZERO
C
      GA1(I,IP1)=SUM1
      GA2(I,IP1)=SUM2
      END DO  ! I
      END DO  ! IP
      GO TO 110
C
C     CLOSED CHANNEL CASE ...
C
   80 DO IP1=2,IASY
      IP=IP1-1
      DO I=1,NCHAN
      SUM2=GA2(I,IP1)
      DO 90 IQ1=1,IP
      A1=GAC2(IP1-IQ1+1)
      A2=GA2(I,IQ1)
      SUM2=SUM2-A1*A2
   90 CONTINUE
C
C     AVOID ROUNDOFF ERROR
C
      IF(DABS(SUM2) .LT. DABS(GA2(I,IP1)*CRIT)) SUM2=ZERO
C
      GA2(I,IP1)=SUM2
      END DO  ! I
      END DO  ! IP
C
C-----------------------------------------------------------------------
C
C     PRINT GAILITIS EXPANSION COEFFICIENTS IF IPFLG(2)=1
C
C
  110 IF(IPFLG(2) .EQ. 0) GO TO 140
C
      IF(JOPEN .EQ. 1) WRITE(IWRITE,1030)
      IF(JOPEN .EQ. 0) WRITE(IWRITE,1050)
      DO 120 I=1,IASY
      WRITE(IWRITE,1000) (GA2(JJ,I),JJ=1,NCHAN)
  120 CONTINUE
      IF(JOPEN .EQ. 0) GO TO 140
      WRITE(IWRITE,1040)
      DO 130  I=1,IASY
      WRITE(IWRITE,1000) (GA1(JJ,I),JJ=1,NCHAN)
  130 CONTINUE
C
  140 CONTINUE
      RETURN
      END
      SUBROUTINE CONRAD(J,ET,NCHAN,RAFIN,IWRITE,IASY,EPSL,RSCALX,IPFLG,
     X                  CVR1,CVR2,GA1,GA2)
C
C***********************************************************************
C
C     CONRAD DETERMINES THE GLOBAL CONVERGENCE RADIUS RAFIN.
C            N.B. THIS IS AN ESTIMATE BASED ON THE USUAL TREATMENT OF
C                 AN ASYMPTOTIC SERIES. THE PADE METHOD MAY IN GENERAL
C                 BE APPLIED AT A MUCH SMALLER RADIUS. THIS ROUTINE IS
C                 FOR INFORMATIONAL PURPOSES ONLY
C
C***********************************************************************
C
C     J           THE CHANNEL NUMBER TO BE PROCESSED
C     ET          CHANNEL ENERGIES IN RYDBERGS
C     NCHAN       NUMBER OF CHANNELS
C     IASY        NUMBER OF TERMS IN THE ASYMPTOTIC EXPANSION
C     IWRITE      UNIT NUMBER OF THE PRINTER OR OUTPUT FILE
C     EPSL        ACCURACY REQUESTED IN DETERMINING GAILITIS RADIUS
C     SCALX       FACTOR USED TO SCALE BURKE-SCHEY COEFFICIENTS
C     IPFLG       PRINT FLAGS ...   IPFLG = 1 PRINT DATA
C                                   IPFLG = 0 DO NOT PRINT DATA
C                 FLAG 4    CONVERGENCE RADII
C                      5    CHANNEL CONVERGENCE RADII AND GLOBAL
C                           CONVERGENCE RADIUS
C
C     RAFIN       GLOBAL CONVERGENCE RADIUS IN ATOMIC UNITS
C
C     GA1,  GA2   GAILITIS COEFFICIENTS
C     CVR1, CVR2  CONVERGE RADII FOR A GIVEN TERM IN EACH ASYMPTOTIC
C                 SERIES
C     RCONV       CONVERGENCE RADII IN EACH CHANNEL
C
C***********************************************************************
C
      IMPLICIT DOUBLE PRECISION(A-H,O-Z)
C
      DIMENSION GA1(NCHAN,IASY),GA2(NCHAN,IASY),CVR1(IASY),CVR2(IASY),
     X          RCONV(NCHAN),IPFLG(7),ET(NCHAN)
C
C     DEFAULT PARAMETERS ...
C
      DATA EPS0/1.0D-12/
      DATA ZERO/0.0D0/,ONE/1.0D0/
C
 1000 FORMAT(1X,8D14.6)
 1010 FORMAT(/,35X,'RADIUS (AU) FOR CONVERGENT ASYMPTOTIC EXPANSION',
     X       2X,F14.7/)
 1020 FORMAT(/,35X,'MAXIMUM RADII IN EACH CHANNEL FOR EACH REAL COEFF.',
     *//)
 1030 FORMAT(/,35X,'MAXIMUM RADII IN EACH CHANNEL FOR EACH IMAGINARY COE
     *FF.',//)
 1040 FORMAT(/,35X,'MAXIMUM RADII IN EACH CHANNEL FOR EACH CLOSED CHANNE
     *L COEFF.',//)
 1050 FORMAT(/,35X,'OPEN CHANNEL J=',I2,9X,'CONVERGENCE RADIUS=',
     XF10.5//)
 1060 FORMAT(/,35X,'CLOSED CHANNEL J=',I2,9X,'CONVERGENCE RADIUS=',
     XF10.5//)
C
      EPS=EPS0/RSCALX
      EPSX=RSCALX*EPSL
      NCHAN1=NCHAN-1
C
      ETJ=ET(J)
C
C     DETERMINE WHETHER THE CHANNEL IS OPEN OR CLOSED
C
      IF(ETJ .LE. ZERO) GO TO 50
C
C***********************************************************************
C
C     OPEN CHANNEL CASE ...
C
C
C     CALCULATE THE MINIMUM RADIUS TO CONVERGE ALL NCHAN CHANNELS
C     GIVEN THE N-TH TERMS OF THE CORRESPONDING ASYMPTOTIC SERIES.
C     REPEAT FOR EACH VALUE OF N STORING THE RESULTS IN ARRAYS CVR1,
C     CVR2
C
      CVR1(1)=ZERO
      CVR2(1)=ZERO
      DO 20 JP=2,IASY
      JP1=JP-1
      RCM1=ZERO
      RCM2=ZERO
      POWER=ONE/(Dble(JP1))
      EPSY=EPSX**POWER
      DO 10 JM=1,NCHAN
      A1=DABS(GA1(JM,JP))
      A2=DABS(GA2(JM,JP))
      IF(A1 .LT. EPS) A1=ZERO
      IF(A2 .LT. EPS) A2=ZERO
      RA1=EPSY*A1**POWER
      RA2=EPSY*A2**POWER
      IF(RA1 .GT. RCM1) RCM1=RA1
      IF(RA2 .GT. RCM2) RCM2=RA2
   10 CONTINUE
      CVR1(JP)=RCM1
      CVR2(JP)=RCM2
   20 CONTINUE
C
C     PRINT CHANNEL CONVERGENCE RADII IF IPFLG(4)=1
C
      IF(IPFLG(4) .EQ. 0) GO TO 30
C
      WRITE(IWRITE,1030)
      WRITE(IWRITE,1000) (CVR1(I),I=1,IASY)
      WRITE(IWRITE,1020)
      WRITE(IWRITE,1000) (CVR2(I),I=1,IASY)
   30 CONTINUE
C
C     FROM THE CONVERGENCE RADII OBTAINED ABOVE CHOOSE MINIMUM VALUES
C
      RCM1=CVR1(IASY)
      RCM2=CVR2(IASY)
      DO 40 I=2,IASY
      I1=I-1
      IF(CVR1(I1) .GT. ZERO .AND. CVR1(I1) .LT. RCM1) RCM1=CVR1(I1)
      IF(CVR2(I1) .GT. ZERO .AND. CVR2(I1) .LT. RCM2) RCM2=CVR2(I1)
   40 CONTINUE
C
C     CHOOSE THE MAXIMUM OF RADII FOR REAL AND IMAGINARY COEFFICIENTS.
C
      RCONV(J)=DMAX1(RCM1,RCM2)
C
C     IF IPFLG(5)=1 PRINT THE CONVERGENCE RADIUS FOR THE CHANNEL
C
      IF(IPFLG(5) .EQ. 0) GO TO 100
C
      WRITE(IWRITE,1050) J,RCONV(J)
      GO TO 100
C
C***********************************************************************
C
C     CLOSED CHANNEL CASE ...
C
C     CALCULATE THE MINIMUM RADIUS TO CONVERGE ALL NCHAN CHANNELS
C     GIVEN THE N-TH TERM OF THE CORRESPONDING ASYMPTOTIC SERIES.
C     REPEAT FOR EACH VALUE OF N STORING THE RESULTS IN ARRAY CVR1.
C
   50 CVR1(1)=ZERO
      DO 70 JP=2,IASY
      JP1=JP-1
      POWER=ONE/(Dble(JP1))
      EPSY=EPSX**POWER
      CRM2=ZERO
      DO 60 JM=1,NCHAN
      A2=DABS(GA2(JM,JP))
      IF(A2 .LT. EPS) GO TO 60
      RA2=EPSY*A2**POWER
      IF(RA2 .GT. CRM2) CRM2=RA2
   60 CONTINUE
      CVR2(JP)=CRM2
   70 CONTINUE
C
C     PRINT CONVERGENCE RADII IF IPFLG(4)=1
C
      IF(IPFLG(4) .EQ. 0) GO TO 80
C
      WRITE(IWRITE,1040)
      WRITE(IWRITE,1000)(CVR2(I),I=1,IASY)
C
C     FROM THE RADII OBTAINED ABOVE SELECT MINIMUM VALUES
C
   80 CRM2=CVR2(IASY)
      DO 90 I=2,IASY
      I1=I-1
      IF(CVR2(I1) .EQ. ZERO) GO TO 90
      IF(CVR2(I1) .LT. CRM2) CRM2=CVR2(I1)
   90 CONTINUE
      RCONV(J)=CRM2
C
C     IF IPFLG(5)=1 PRINT THE CHANNEL CONVERGENCE RADIUS
C
      IF(IPFLG(5) .EQ. 0) GO TO 100
C
      WRITE(IWRITE,1060) J,RCONV(J)
C
C***********************************************************************
C
  100 IF(J .NE. NCHAN) RETURN
C
C     IF J=NCHAN THE MINIMUM RADIUS NECESSARY TO YIELD CONVERGENT
C                SOLUTIONS IN ALL CHANNELS, RAFIN, IS DETERMINED
C
      RAFIN=RCONV(NCHAN)
      DO 110 JP=1,NCHAN1
      IF(RCONV(JP) .GT. RAFIN) RAFIN=RCONV(JP)
  110 CONTINUE
C
C     PRINT GLOBAL CONVERGENCE RADIUS, RAFIN
C
      WRITE(IWRITE,1010) RAFIN
C
      RETURN
      END
      SUBROUTINE BKSHWF(LJ,RA,SKJ,ETAJ,ION,JOPEN,F1,F2,FP1,FP2,IWRITE)
C
C***********************************************************************
C
C      BKSHWF COMPUTES FUNCTIONS REQUIRED FOR THE BURKE-SCHEY ASYMPTOTIC
C             EXPANSION
C
C***********************************************************************
C
C     LJ      CHANNEL ORBITAL ANGULAR MOMENTUM
C     RA      RADIAL POINT AT WHICH THE FUNCTIONS ARE REQUIRED
C     SKJ     CHANNEL MOMENTUM
C     ETAJ    CHANNEL COULOMB PARAMETER
C     ION     IONICITY
C     JOPEN   = 1  FOR AN OPEN CHANNEL
C             = 0  FOR A CLOSED CHANNEL
C     IWRITE  UNIT NUMBER OF PRINTER OR OUTPUT DEVICE
C
C     F1,FP1  REGULAR FUNCTION AND DERIVATIVE, RESPECTIVELY
C     F2,FP2  IRREGULAR FUNCTION AND DERIVATIVE, RESPECTIVELY
C
C***********************************************************************
C
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
      DATA HALF/0.5D0/,PI/3.14159265358979324D+00/,ZERO/0.0D0/
C
      A1=HALF*Dble(LJ)*PI
C
      IF(JOPEN .EQ. 0) GO TO 20
C
C     OPEN CHANNEL CASE ...
C
      SKR=SKJ*RA
      ROOTK=DSQRT(SKJ)
      THETA=SKR-A1
      THETAP=SKJ
C
      IF(ION .EQ. 0) GO TO 10
C
      THETA=THETA-ETAJ*DLOG(SKR+SKR)+CPHAZ(LJ,ETAJ,IWRITE)
      THETAP=THETAP-ETAJ/RA
C
   10 F1=DSIN(THETA)/ROOTK
      F2=DCOS(THETA)/ROOTK
      FP1=THETAP*F2
      FP2=-THETAP*F1
      RETURN
C
C     CLOSED CHANNEL CASE ...
C
   20 SK=DABS(SKJ)
      ETA=DABS(ETAJ)
      SKR=SK*RA
      ROOTK=DSQRT(SK)
      THETA=SKR
      THETAP=SK
C
      IF(ION .EQ. 0) GO TO 30
C
      THETA=THETA-ETA*DLOG(SKR+SKR)
      THETAP=THETAP-ETA/RA
C
   30 F1=ZERO
      F2=DEXP(-THETA)/ROOTK
      FP1=ZERO
      FP2=-THETAP*F2
C
      RETURN
      END
      SUBROUTINE CFRACT(N,XG0,YG0,XG,YG,XA,YA,XC,YC,XD,YD)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
C***********************************************************************
C
C  CFRACT CONVERTS POLYNOMIAL G TO THE CORRESPONDING CONTINUED
C         FRACTION, IN STIELTJES FORM WITH COEFFICIENTS A
C
C   G(Z) = G0+G1*Z+G2*Z**2+G3*Z**3+...+GN*Z**N
C
C   GN(Z)= CN(Z)/DN(Z)
C        = GO/1+ A1*Z/1+A2*Z/1+A3*Z/1+.../1+AN*Z
C
C  DATA:
C     FOR ANY SYMBOL @, X@,Y@ ARE REAL@, IM@, RESPECTIVELY
C   N     ORDER OF G, EVEN OR ODD       INPUT
C   G0    CONSTANT TERM                 INPUT
C   G     VECTOR G(K), K=1,N            INPUT
C   A     VECTOR A(K), K=1,N            OUTPUT
C   C     OUTPUT: NUMERATOR   POLYNOMIALS, CONVERGENTS N AND N-1
C   D     OUTPUT: DENOMINATOR POLYNOMIALS, CONVERGENTS N AND N-1
C         CALLER PROVIDES SPACE FOR A,C,D
C   C AND D ARRAYS CONTAIN NT=2*((N+1)/2) ELEMENTS EACH
C   STORAGE CONVENTIONS FOR C AND D:
C     C0=G0 AND D0=1.0
C     COEFFICIENTS IN SEQUENCE, WITH M=N-1 AND K=(N+1)/2 :
C       CN1,CM1,CN2,CM2,...CNI,CMI,...  I=1,K
C       DN1,DM1,DN2,DM2,...DNI,DMI,...  I=1,K
C     NOTE THAT CNK=0.0 IF N IS ODD
C
C   ALGORITHM: RK NESBET, 82.10.27
C
C***********************************************************************
C
      DIMENSION XG(*),XA(*),XC(*),XD(*)
      DIMENSION YG(*),YA(*),YC(*),YD(*)
      DATA ZERO/0.0D0/,ONE/1.0D0/
C
      NT=2*((N+1)/2)
      DO I=1,NT
        XC(I)=ZERO
        YC(I)=ZERO
        XD(I)=ZERO
        YD(I)=ZERO
      END DO
      XCT=ZERO
      YCT=ZERO
      XDT=ONE
      YDT=ZERO
      XDN= XG0
      YDN= YG0
      DO 40 K=1,N
      XDD=XDN
      YDD=YDN
      XDN=XG(K)
      YDN=YG(K)
      DO 20 I=1,K,2
      IF((I+1)/2.GE.K) GO TO 20
      XDN=XDN+XG(K-(I+1)/2)*XD(I)
     X       -YG(K-(I+1)/2)*YD(I)
      YDN=YDN+XG(K-(I+1)/2)*YD(I)
     X       +YG(K-(I+1)/2)*XD(I)
   20 CONTINUE
      DDD=XDD**2+YDD**2
      XA(K)=-(XDN*XDD+YDN*YDD)/DDD
      YA(K)=-(YDN*XDD-XDN*YDD)/DDD
C
C     PLANT AK=0.0 AND RETURN IF SEQUENCE TRUNCATES
C
      IF(XA(K) .EQ. ZERO .AND. YA(K) .EQ. ZERO ) RETURN
      DO 30 I=1,K,2
      XCI=XC(I)
      YCI=YC(I)
      XC(I)=XCI+XA(K)*XCT -YA(K)*YCT
      YC(I)=YCI+XA(K)*YCT +YA(K)*XCT
      XCT=XC(I+1)
      YCT=YC(I+1)
      XC(I+1)=XCI
      YC(I+1)=YCI
      XDI=XD(I)
      YDI=YD(I)
      XD(I)=XDI+XA(K)*XDT -YA(K)*YDT
      YD(I)=YDI+XA(K)*YDT +YA(K)*XDT
      XDT=XD(I+1)
      YDT=YD(I+1)
      XD(I+1)=XDI
      YD(I+1)=YDI
   30 CONTINUE
      XCT=XG0
      YCT=YG0
      XDT=ONE
      YDT=ZERO
   40 CONTINUE
C
      RETURN
      END
      SUBROUTINE DFRACT(N,G0,G,A,C,D)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
C***********************************************************************
C
C  DFRACT CONVERTS POLYNOMIAL G TO THE CORRESPONDING CONTINUED
C         FRACTION, IN STIELTJES FORM WITH COEFFICIENTS A
C
C   G(Z) = G0+G1*Z+G2*Z**2+G3*Z**3+...+GN*Z**N
C
C   GN(Z)= CN(Z)/DN(Z)
C        = G0/1+ A1*Z/1+A2*Z/1+A3*Z/1+.../1+AN*Z
C
C  DATA:
C   N     ORDER OF G, EVEN OR ODD       INPUT
C   G0    CONSTANT TERM                 INPUT
C   G     VECTOR G(K), K=1,N            INPUT
C   A     VECTOR A(K), K=1,N            OUTPUT
C   C     OUTPUT: NUMERATOR   POLYNOMIALS, CONVERGENTS N AND N-1
C   D     OUTPUT: DENOMINATOR POLYNOMIALS, CONVERGENTS N AND N-1
C         CALLER PROVIDES SPACE FOR A,C,D
C   C AND D ARRAYS CONTAIN NT=2*((N+1)/2) ELEMENTS EACH
C   STORAGE CONVENTIONS FOR C AND D:
C     C0=G0 AND D0=1.0
C     COEFFICIENTS IN SEQUENCE, WITH M=N-1 AND K=(N+1)/2 :
C       CN1,CM1,CN2,CM2,...CNI,CMI,...  I=1,K
C       DN1,DM1,DN2,DM2,...DNI,DMI,...  I=1,K
C     NOTE THAT DNK=0.0 IF N IS ODD
C
C   ALGORITHM: RK NESBET, 82.10.27
C
C***********************************************************************
C
      DIMENSION G(*),A(*),C(*),D(*)
      DATA ZERO/0.0D0/,ONE/1.0D0/
C
      NT=2*((N+1)/2)
      DO I=1,NT
        C(I)=ZERO
        D(I)=ZERO
      END DO
      CT=ZERO
      DT=ONE
      DN= G0
      DO 40 K=1,N
      DD=DN
      DN=G(K)
      DO 20 I=1,K,2
      IF((I+1)/2.GE.K) GO TO 20
      DN=DN+G(K-(I+1)/2)*D(I)
   20 CONTINUE
      A(K)=-DN/DD
C
C     PLANT AK=0.0 AND RETURN IF SEQUENCE TRUNCATES
C
      IF(A(K) .EQ. ZERO) RETURN
      DO 30 I=1,K,2
      CI=C(I)
      C(I)=CI+A(K)*CT
      CT=C(I+1)
      C(I+1)=CI
      DI=D(I)
      D(I)=DI+A(K)*DT
      DT=D(I+1)
      D(I+1)=DI
   30 CONTINUE
      CT=G0
      DT=ONE
   40 CONTINUE
C
      RETURN
      END
      SUBROUTINE DFEVAL(NTERM,IP,X,C0,C,D,F,EF,FP,EFP,IWRITE,IPFLG)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
C***********************************************************************
C
C     DFEVAL USES THE POLYNOMIAL COEFFICIENT ARRAYS C, D
C            PRODUCED BY SUBROUTINE DFRACT TO COMPUTE THE
C            CORRESPONDING FUNCTION VALUE F AND DERIVATIVE VALUE FP
C            AT THE POINT X
C
C            F = 1/X**IP * ( C0 + C1/X + ... + CN/X**N ) / ( D0 + D1/X
C                + ... + DN/X**N)
C
C            NTERM IS THE NUMBER OF COEFFICIENTS SUPPLIED TO DFRACT
C
C            EF AND EFP ARE ERROR ESTIMATES FOR F, FP, RESPECTIVELY
C
C            IF IPFLG IS NONZERO DIAGNOSTIC INFORMATION IS PRINTED ON
C            UNIT IWRITE
C
C***********************************************************************
C
      DIMENSION C(*),D(*)
      DATA ZERO/0.0D0/,ONE/1.0D0/
C
 1000 FORMAT(9X,'I',15X,'C',15X,'D',/,(I10,2D16.8))
 1010 FORMAT(' F  =',D30.17,' F1  =',D30.17,' DELTA =',D16.8,/,
     X       ' FP =',D30.17,' FP1 =',D30.17,' DELTA =',D16.8)
C
      XI=ONE/X
      XI2=XI*XI
      XIP=XI**IP
      XIP1=XI*XIP
C
      KMAX=(NTERM+1)/2
      KMAX1=KMAX-1
      KMAXP=KMAX+IP
      AIP=Dble(KMAX)
      AIQ=Dble(KMAXP)
      NC=KMAX+KMAX-1
      NCP=NC+1
C
      IF(KMAX .GT. 0) GO TO 10
C
      SN1=ZERO
      SN2=ZERO
      SNP1=ZERO
      SNP2=ZERO
C
      SD1=ZERO
      SD2=ZERO
      SDP1=ZERO
      SDP2=ZERO
C
      GO TO 30
C
C     EVALUATE CONTINUED FRACTION
C
   10 IF(IPFLG .NE. 0) WRITE(IWRITE,1000)(I,C(I),D(I),I=1,NTERM)
C
      SN1=C(NC)
      SN2=C(NCP)
      SNP1=AIQ*SN1
      SNP2=AIQ*SN2
C
      SD1=D(NC)
      SD2=D(NCP)
      SDP1=AIP*SD1
      SDP2=AIP*SD2
C
      IF(KMAX1 .LT. 1) GO TO 30
C
      DO 20 K=1,KMAX1
      NC=NC-2
      NCP=NC+1
      AIP=Dble(KMAX-K)
      AIQ=Dble(KMAXP-K)
C
      SN1=C(NC)+XI*SN1
      SN2=C(NCP)+XI*SN2
      SNP1=AIQ*C(NC)+XI*SNP1
      SNP2=AIQ*C(NCP)+XI*SNP2
C
      SD1=D(NC)+XI*SD1
      SD2=D(NCP)+XI*SD2
      SDP1=AIP*D(NC)+XI*SDP1
      SDP2=AIP*D(NCP)+XI*SDP2
C
   20 CONTINUE
C
   30 AIQ=Dble(IP)
      SN1=XIP*(C0+XI*SN1)
      SN2=XIP*(C0+XI*SN2)
      SNP1=-XIP1*(AIQ*C0+XI*SNP1)
      SNP2=-XIP1*(AIQ*C0+XI*SNP2)
C
      SD1=ONE+XI*SD1
      SD2=ONE+XI*SD2
      SDP1=-XI2*SDP1
      SDP2=-XI2*SDP2
C
C     OBTAIN FUNCTIONS AND ERROR ESTIMATES
C
      F=SN1/SD1
      F1=SN2/SD2
      EF=DABS(F-F1)
C
      FP=(SNP1-F*SDP1)/SD1
      FP1=(SNP2-F1*SDP2)/SD2
      EFP=DABS(FP-FP1)
C
      IF(IPFLG .NE. 0) WRITE(IWRITE,1010) F,F1,EF,FP,FP1,EFP
C
      RETURN
      END
      SUBROUTINE CFEVAL(NTERM,IP,X,XC0,YC0,XC,YC,XD,YD,FR,FI,EFR,EFI,
     X                  FPR,FPI,EFPR,EFPI,IWRITE,IPFLG)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
C***********************************************************************
C
C     CFEVAL USES THE POLYNOMIAL COEFFICIENT ARRAYS XC, YC, XD, YD,
C            PRODUCED BY SUBROUTINE CFRACT TO COMPUTE THE
C            CORRESPONDING FUNCTION VALUE F AND DERIVATIVE VALUE FP
C            AT THE POINT X
C            F = ( FR, FI ) AND FP = ( FPR, FPI )
C
C            F = 1/X**IP * ( C0 + C1/X + ... + CN/X**N ) / ( D0 + D1/X
C                            + ... + DN/X**N)
C
C            NTERM IS THE NUMBER OF COEFFICIENTS SUPPLIED TO CFRACT
C
C            EF AND EFP ARE ERROR ESTIMATES FOR F, FP, RESPECTIVELY
C
C            IF IPFLG IS NONZERO DIAGNOSTIC INFORMATION IS PRINTED ON
C            UNIT IWRITE
C
C***********************************************************************
C
      DIMENSION XC(*),YC(*),XD(*),YD(*)
      DATA ZERO/0.0D0/,ONE/1.0D0/
C
 1000 FORMAT(9X,'I',14X,'XC',14X,'YC',14X,'XD',14X,'YD',/,
     X      (I10,4D16.8))
 1010 FORMAT(' FR  =',D30.17,' FR1  =',D30.17,' DELTA =',D16.8,/,
     X       ' FI  =',D30.17,' FI1  =',D30.17,' DELTA =',D16.8,/,
     X       ' FPR =',D30.17,' FPR1 =',D30.17,' DELTA =',D16.8,/,
     X       ' FPI =',D30.17,' FPI1 =',D30.17,' DELTA =',D16.8)
C
      XI=ONE/X
      XI2=XI*XI
      XIP=XI**IP
      XIP1=XI*XIP
C
      KMAX=(NTERM+1)/2
      KMAX1=KMAX-1
      KMAXP=KMAX+IP
C
      IF(KMAX .GT. 0) GO TO 10
C
      SN1=ZERO
      SN2=ZERO
      SNP1=ZERO
      SNP2=ZERO
C
      SD1=ZERO
      SD2=ZERO
      SDP1=ZERO
      SDP2=ZERO
      GO TO 20
C
C     EVALUATE CONTINUED FRACTION
C
   10 IF(IPFLG .NE. 0) WRITE(IWRITE,1000)(I,XC(I),YC(I),XD(I),YD(I),
     X                                  I=1,NTERM)
C
   20 DO 100 I=1,2
C
      AIP=Dble(KMAX)
      AIQ=Dble(KMAXP)
      NC=KMAX+KMAX-1
      NCP=NC+1
C
      IF(KMAX .EQ. 0) GO TO 80
      IF(I .EQ. 2) GO TO 30
C
      C0=XC0
      SN1=XC(NC)
      SN2=XC(NCP)
      SD1=XD(NC)
      SD2=XD(NCP)
      GO TO 40
   30 C0=YC0
      SN1=YC(NC)
      SN2=YC(NCP)
      SD1=YD(NC)
      SD2=YD(NCP)
C
   40 SNP1=AIQ*SN1
      SNP2=AIQ*SN2
      SDP1=AIP*SD1
      SDP2=AIP*SD2
C
      IF(KMAX1 .LT. 1) GO TO 80
C
      DO 70 K=1,KMAX1
      NC=NC-2
      NCP=NC+1
      AIP=Dble(KMAX-K)
      AIQ=Dble(KMAXP-K)
C
      IF(I .EQ. 2) GO TO 50
C
      C1=XC(NC)
      C2=XC(NCP)
      D1=XD(NC)
      D2=XD(NCP)
      GO TO 60
C
   50 C1=YC(NC)
      C2=YC(NCP)
      D1=YD(NC)
      D2=YD(NCP)
C
   60 SN1=C1+XI*SN1
      SN2=C2+XI*SN2
      SNP1=AIQ*C1+XI*SNP1
      SNP2=AIQ*C2+XI*SNP2
C
      SD1=D1+XI*SD1
      SD2=D2+XI*SD2
      SDP1=AIP*D1+XI*SDP1
      SDP2=AIP*D2+XI*SDP2
C
   70 CONTINUE
C
   80 AIQ=Dble(IP)
      SN1=XIP*(C0+XI*SN1)
      SN2=XIP*(C0+XI*SN2)
      SNP1=-XIP1*(AIQ*C0+XI*SNP1)
      SNP2=-XIP1*(AIQ*C0+XI*SNP2)
C
      SDP1=-XI2*SDP1
      SDP2=-XI2*SDP2
C
      IF(I .EQ. 2) GO TO 90
C
      SXN1=SN1
      SXN2=SN2
      SXD1=ONE+XI*SD1
      SXD2=ONE+XI*SD2
      SXNP1=SNP1
      SXNP2=SNP2
      SXDP1=SDP1
      SXDP2=SDP2
      GO TO 100
C
   90 SYN1=SN1
      SYN2=SN2
      SYD1=XI*SD1
      SYD2=XI*SD2
      SYNP1=SNP1
      SYNP2=SNP2
      SYDP1=SDP1
      SYDP2=SDP2
C
  100 CONTINUE
C
C     OBTAIN FUNCTIONS AND ERROR ESTIMATES
C
      DEN1=SXD1*SXD1+SYD1*SYD1
      DEN2=SXD2*SXD2+SYD2*SYD2
      FR =(SXN1*SXD1+SYN1*SYD1)/DEN1
      FR1=(SXN2*SXD2+SYN2*SYD2)/DEN2
      FI =(SYN1*SXD1-SXN1*SYD1)/DEN1
      FI1=(SYN2*SXD2-SXN2*SYD2)/DEN2
C
      EFR=DABS(FR-FR1)
      EFI=DABS(FI-FI1)
C
C     OBTAIN DERIVATIVES AND ERROR ESTIMATES
C
      RSAV1=SXNP1-FR *SXDP1+FI *SYDP1
      RSAV2=SXNP2-FR1*SXDP2+FI1*SYDP2
      RSAV3=SYNP1-FR *SYDP1-FI *SXDP1
      RSAV4=SYNP2-FR1*SYDP2-FI1*SXDP2
C
      FPR =(SXD1*RSAV1+SYD1*RSAV3)/DEN1
      FPR1=(SXD2*RSAV2+SYD2*RSAV4)/DEN2
      FPI =(SXD1*RSAV3-SYD1*RSAV1)/DEN1
      FPI1=(SXD2*RSAV4-SYD2*RSAV2)/DEN2
      EFPR=DABS(FPR-FPR1)
      EFPI=DABS(FPI-FPI1)
C
      IF(IPFLG .NE. 0) WRITE(IWRITE,1010) FR,FR1,EFR,FI,FI1,EFI,
     X                 FPR,FPR1,EFPR,FPI,FPI1,EFPI
C
      RETURN
      END
      SUBROUTINE HSUM(N,M,X,C0,C,S,SP)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
C
C***********************************************************************
C
C     HSUM EVALUATES THE SUMS
C
C           S  = C0 + C(1)/X + C(2)/X**2 + ... + C(N)/X**N
C           SP =  - ( C(1)/X + 2*C(2)/X**2 + ... + N*C(N)/X**N ) / X
C
C***********************************************************************
C
      DIMENSION C(*)
      DATA ZERO/0.0D0/,ONE/1.0D0/
C
      IF(N .EQ. 0) GO TO 30
C
      N1=N-1
      NC=N
C
      XI=ONE/X
C
      NQ=N1*M+1
      S=C(NQ)
      AI=Dble(N)
      SP=AI*S
      IF(N .EQ. 1) GO TO 20
C
      DO 10 K=1,N1
      NC=NC-1
      NQ=NQ-M
      AI=Dble(NC)
      S=C(NQ)+XI*S
      SP=AI*C(NQ)+XI*SP
   10 CONTINUE
C
   20 S=C0+XI*S
      SP=-XI*XI*SP
      RETURN
C
   30 S=C0
      SP=ZERO
      RETURN
C
      END
      SUBROUTINE COULFG(XX,ETA1,XLMIN,XLMAX, FC,GC,FCP,GCP,
     X                  MODE1,KFN,IFAIL)
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C                                                                      C
C  REVISED COULOMB WAVEFUNCTION PROGRAM USING STEED'S METHOD           C
C                                                                      C
C  A. R. BARNETT           MANCHESTER  MARCH   1981                    C
C                                                                      C
C  ORIGINAL PROGRAM 'RCWFN'      IN    CPC  8 (1974) 377-395           C
C                 + 'RCWFF'      IN    CPC 11 (1976) 141-142           C
C  FULL DESCRIPTION OF ALGORITHM IN    CPC 21 (1981) 297-314           C
C  THIS VERSION WRITTEN UP       IN    CPC 27 (1982) 147-ZZZ           C
C                                                                      C
C  COULFG RETURNS F,G,F',G', FOR REAL XX.GT.0,REAL ETA1 (INCLUDING 0), C
C   AND REAL LAMBDA(XLMIN) .GT. -1 FOR INTEGER-SPACED LAMBDA VALUES    C
C   THUS GIVING POSITIVE-ENERGY SOLUTIONS TO THE COULOMB SCHRODINGER   C
C   EQUATION,TO THE KLEIN-GORDON EQUATION AND TO SUITABLE FORMS OF     C
C   THE DIRAC EQUATION ,ALSO SPHERICAL & CYLINDRICAL BESSEL EQUATIONS  C
C                                                                      C
C  FOR A RANGE OF LAMBDA VALUES (XLMAX - XLMIN) MUST BE AN INTEGER,    C
C  STARTING ARRAY ELEMENT IS M1 = MAX0(IDINT(XLMIN+ACCUR),0) + 1       C
C      SEE TEXT FOR MODIFICATIONS FOR INTEGER L-VALUES                 C
C                                                                      C
C  IF 'MODE' = 1  GET F,G,F',G'   FOR INTEGER-SPACED LAMBDA VALUES     C
C            = 2      F,G      UNUSED ARRAYS MUST BE DIMENSIONED IN    C
C            = 3      F               CALL TO AT LEAST LENGTH (1)      C
C  IF 'KFN'  = 0 REAL        COULOMB FUNCTIONS ARE RETURNED            C
C            = 1 SPHERICAL   BESSEL      "      "     "                C
C            = 2 CYLINDRICAL BESSEL      "      "     "                C
C  THE USE OF 'MODE' AND 'KFN' IS INDEPENDENT                          C
C                                                                      C
C  PRECISION:  RESULTS TO WITHIN 2-3 DECIMALS OF 'MACHINE ACCURACY'    C
C   IN OSCILLATING REGION X .GE. ETA1 + SQRT(ETA1**2 + XLM(XLM+1))     C
C   COULFG IS CODED FOR REAL*8 ON IBM OR EQUIVALENT  ACCUR = 10**-16   C
C   USE AUTODBL + EXTENDED PRECISION ON HX COMPILER  ACCUR = 10**-33   C
C   FOR MANTISSAS OF 56 & 112 BITS. FOR SINGLE PRECISION CDC (48 BITS) C
C   REASSIGN DSQRT=SQRT ETC.  SEE TEXT FOR COMPLEX ARITHMETIC VERSION  C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      IMPLICIT double precision (A-H,O-Z)
      DIMENSION    FC(*),GC(*),FCP(*),GCP(*)
      LOGICAL      ETANE0,XLTURN
C***  COULFG HAS CALLS TO: DSQRT,DABS,DMOD,IDINT,DSIGN,DBLE,DMIN1
      DATA ZERO,ONE,TWO,TEN2,ABORT /0.0D0, 1.0D0, 2.0D0, 1.0D2, 2.0D4/
      DATA HALF,TM30 / 0.5D0, 1.0D-30 /
      DATA RT2DPI /0.79788 45608 02865 35587 98921 19868 76373 D0/
C *** THIS CONSTANT IS  DSQRT(TWO/PI):  USE Q0 FOR IBM REAL*16: D0 FOR
C ***  REAL*8 & CDC DOUBLE P:  E0 FOR CDC SINGLE P; AND TRUNCATE VALUE.
C
C ZM for double precision ACCUR must be 1.0D-16 not 1.0D-32 as before!
                        ACCUR = 1.0D-16
c                        ACCUR = 1.0D-32
C ***            CHANGE ACCUR TO SUIT MACHINE AND PRECISION REQUIRED
      FJWKB = 0.0D0
      MODE  = 1
      IF(MODE1 .EQ. 2 .OR. MODE1 .EQ. 3 ) MODE = MODE1
      IFAIL = 0
      IEXP  = 1
      NPQ   = 0
      ETA   = ETA1
      GJWKB = ZERO
      PACCQ = ONE
      IF(KFN .NE. 0) ETA = ZERO
                 ETANE0  = ETA .NE. ZERO
      ACC   = ACCUR
      ACC4  = ACC*TEN2*TEN2
      ACCH  = DSQRT(ACC)
C ***    TEST RANGE OF XX, EXIT IF.LE.DSQRT(ACCUR) OR IF NEGATIVE
C
      IF(XX .LE. ACCH)                          GO TO 120
      X     = XX
      XLM   = XLMIN
      IF(KFN .EQ. 2)  XLM = XLM - HALF
      IF(XLM .LE. -ONE .OR. XLMAX .LT. XLMIN)   GO TO 130
      E2MM1 = ETA*ETA + XLM*XLM + XLM
      XLTURN= X*(X - TWO*ETA) .LT. XLM*XLM + XLM
      DELL  = XLMAX - XLMIN + ACC
C      IF(DABS(DMOD(DELL,ONE)) .GT. ACC) WRITE(6,1060)XLMAX,XLMIN,DELL
C ZM improve the test on integer
      IF (ABS(XLMAX-XLMIN-nint(XLMAX - XLMIN))>ACCUR) WRITE(6,1060)XLMAX
     X ,XLMIN,DELL
      LXTRA = IDINT(DELL)
      XLL   = XLM + DBLE(LXTRA)
C ***       LXTRA IS NUMBER OF ADDITIONAL LAMBDA VALUES TO BE COMPUTED
C ***       XLL  IS MAX LAMBDA VALUE, OR 0.5 SMALLER FOR J,Y BESSELS
C ***         DETERMINE STARTING ARRAY ELEMENT (M1) FROM XLMIN
      M1  = MAX0(IDINT(XLMIN + ACC),0) + 1
      L1  = M1 + LXTRA
C
C ***    EVALUATE CF1  =  F   =  FPRIME(XL,ETA,X)/F(XL,ETA,X)
C
      XI  = ONE/X
      FCL = ONE
      PK  = XLL + ONE
      PX  = PK  + ABORT
   10 EK  = ETA / PK
      F   = (EK + PK*XI)*FCL + (FCL - ONE)*XI
      PK1 =  PK + ONE
C ***   TEST ENSURES B1 .NE. ZERO FOR NEGATIVE ETA; FIXUP IS EXACT.
             IF(DABS(ETA*X + PK*PK1) .GT. ACCH)  GO TO 20
             FCL  = (ONE + EK*EK)/(ONE + (ETA/PK1)**2)
             PK   =  TWO + PK
      GO TO 10
   20 D   =  ONE/((PK + PK1)*(XI + EK/PK1))
      DF  = -FCL*(ONE + EK*EK)*D
            IF(FCL .NE. ONE )  FCL = -ONE
            IF(D   .LT. ZERO)  FCL = -FCL
      F   =  F  + DF
C
C ***   BEGIN CF1 LOOP ON PK = K = LAMBDA + 1
C
      P     = ONE
   30 PK    = PK1
        PK1 = PK1 + ONE
        EK  = ETA / PK
        TK  = (PK + PK1)*(XI + EK/PK1)
        D   =  TK - D*(ONE + EK*EK)
              IF(DABS(D) .GT. ACCH)             GO TO 40
              WRITE (6,1000) D,DF,ACCH,PK,EK,ETA,X
              P = P  +   ONE
              IF( P .GT. TWO )                  GO TO 140
   40 D     = ONE/D
              IF (D .LT. ZERO) FCL = -FCL
        DF  = DF*(D*TK - ONE)
        F   = F  + DF
              IF(PK .GT. PX)                    GO TO 140
      IF(DABS(DF) .GE. DABS(F)*ACC)             GO TO 30
                  NFP = PK - XLL - 1
      IF(LXTRA .EQ. 0)                          GO TO 60
C
C *** DOWNWARD RECURRENCE TO LAMBDA = XLM. ARRAY GC,IF PRESENT,STORES RL
C
      FCL = FCL*TM30
      FPL = FCL*F
      IF(MODE .EQ. 1) FCP(L1) = FPL
                      FC (L1) = FCL
      XL  = XLL
      RL  = ONE
      EL  = ZERO
      DO LP = 1,LXTRA
         IF(ETANE0) EL = ETA/XL
         IF(ETANE0) RL = DSQRT(ONE + EL*EL)
         SL    =  EL  + XL*XI
         L     =  L1  - LP
         FCL1  = (FCL *SL + FPL)/RL
         FPL   =  FCL1*SL - FCL *RL
         FCL   =  FCL1
         FC(L) =  FCL
         IF(MODE .EQ. 1) FCP(L)  = FPL
         IF(MODE .NE. 3 .AND. ETANE0) GC(L+1) = RL
         XL = XL - ONE
      END DO
      IF(FCL .EQ. ZERO) FCL = ACC
      F  = FPL/FCL
C ***    NOW WE HAVE REACHED LAMBDA = XLMIN = XLM
C ***    EVALUATE CF2 = P + I.Q  AGAIN USING STEED'S ALGORITHM
C ***    SEE TEXT FOR COMPACT COMPLEX CODE FOR SP CDC OR NON-ANSI IBM
C
   60 IF( XLTURN ) CALL JWKB(X,ETA,DMAX1(XLM,ZERO),FJWKB,GJWKB,IEXP)
      IF( IEXP .GT. 1 .OR. GJWKB .GT. ONE/(ACCH*TEN2))  GO TO 80
          XLTURN = .FALSE.
      TA =  TWO*ABORT
      PK =  ZERO
      WI =  ETA + ETA
      P  =  ZERO
      Q  =  ONE - ETA*XI
      AR = -E2MM1
      AI =  ETA
      BR =  TWO*(X - ETA)
      BI =  TWO
      DR =  BR/(BR*BR + BI*BI)
      DI = -BI/(BR*BR + BI*BI)
      DP = -XI*(AR*DI + AI*DR)
      DQ =  XI*(AR*DR - AI*DI)
   70 P     = P  + DP
         Q  = Q  + DQ
         PK = PK + TWO
         AR = AR + PK
         AI = AI + WI
         BI = BI + TWO
         D  = AR*DR - AI*DI + BR
         DI = AI*DR + AR*DI + BI
         C  = ONE/(D*D + DI*DI)
         DR =  C*D
         DI = -C*DI
         A  = BR*DR - BI*DI - ONE
         B  = BI*DR + BR*DI
         C  = DP*A  - DQ*B
         DQ = DP*B  + DQ*A
         DP = C
         IF(PK .GT. TA)                         GO TO 150
      IF(DABS(DP)+DABS(DQ).GE.(DABS(P)+DABS(Q))*ACC)   GO TO 70
                      NPQ   = PK/TWO
                      PACCQ = HALF*ACC/DMIN1(DABS(Q),ONE)
                      IF(DABS(P) .GT. DABS(Q)) PACCQ = PACCQ*DABS(P)
C
C *** SOLVE FOR FCM = F AT LAMBDA = XLM,THEN FIND NORM FACTOR W=W/FCM
C
      GAM = (F - P)/Q
            IF(Q .LE. ACC4*DABS(P))             GO TO 160
      W   = ONE/DSQRT((F - P)*GAM + Q)
            GO TO 90
C *** ARRIVE HERE IF G(XLM) .GT. 10**6 OR IEXP .GT. 70 & XLTURN = .TRUE.
   80 W   = FJWKB
      GAM = GJWKB*W
      P   = F
      Q   = ONE
C
C *** NORMALISE FOR SPHERICAL OR CYLINDRICAL BESSEL FUNCTIONS
C
   90                     ALPHA = ZERO
          IF(KFN  .EQ. 1) ALPHA = XI
          IF(KFN  .EQ. 2) ALPHA = XI*HALF
                          BETA  = ONE
          IF(KFN  .EQ. 1) BETA  = XI
          IF(KFN  .EQ. 2) BETA  = DSQRT(XI)*RT2DPI
      FCM  = DSIGN(W,FCL)*BETA
           FC(M1)  = FCM
                      IF(MODE .EQ. 3)           GO TO 100
           IF(.NOT. XLTURN)   GCL =  FCM*GAM
           IF(      XLTURN)   GCL =  GJWKB*BETA
           IF( KFN .NE. 0 )   GCL = -GCL
           GC(M1)  = GCL
           GPL =  GCL*(P - Q/GAM) - ALPHA*GCL
                      IF(MODE .EQ. 2)           GO TO 100
           GCP(M1) = GPL
           FCP(M1) = FCM*(F - ALPHA)
  100 IF(LXTRA .EQ. 0 ) RETURN
C *** UPWARD RECURRENCE FROM GC(M1),GCP(M1)  STORED VALUE IS RL
C *** RENORMALISE FC,FCP AT EACH LAMBDA AND CORRECT REGULAR DERIVATIVE
C ***    XL   = XLM HERE  AND RL = ONE , EL = ZERO FOR BESSELS
         W    = BETA*W/DABS(FCL)
         MAXL = L1 - 1
      DO L = M1,MAXL
                      IF(MODE .EQ. 3)           CYCLE
                      XL = XL + ONE
         IF(ETANE0)   EL = ETA/XL
         IF(ETANE0)   RL = GC(L+1)
                      SL = EL + XL*XI
         GCL1     = ((SL - ALPHA)*GCL - GPL)/RL
         GPL      =   RL*GCL -  (SL + ALPHA)*GCL1
         GCL      = GCL1
         GC(L+1)  = GCL1
                      IF(MODE .EQ. 2)           CYCLE
         GCP(L+1) = GPL
         FCP(L+1) = W*(FCP(L+1) - ALPHA*FC(L+1))
      FC(L+1)     = W* FC(L+1)
      END DO
      RETURN
 1000 FORMAT(/,' CF1 ACCURACY LOSS: D,DF,ACCH,K,ETA/K,ETA,X = ',1P,
     X       7D9.2,/)
C
C ***    ERROR MESSAGES
C
  120 IFAIL = -1
      WRITE(6,1010) XX,ACCH
 1010 FORMAT(' FOR XX = ',1P,D12.3,' TRY SMALL-X  SOLUTIONS',
     X' OR X NEGATIVE',/ ,' SQUARE ROOT ACCURACY PARAMETER =  ',D12.3,/)
      RETURN
  130 IFAIL = -2
      WRITE (6,1020) XLMAX,XLMIN,XLM
 1020 FORMAT(/,' PROBLEM WITH INPUT ORDER VALUES:XLMAX,XLMIN,XLM = ',
     X1P,3D15.6,/)
      RETURN
  140 IFAIL =  1
      WRITE (6,1030) ABORT,F ,DF,PK,PX,ACC
 1030 FORMAT(' CF1 HAS FAILED TO CONVERGE AFTER ',F10.0,' ITERATIONS',/
     X,' F,DF,PK,PX,ACCUR =  ',1P,5D12.3,//)
      RETURN
  150 IFAIL =  2
      WRITE (6,1040) ABORT,P,Q,DP,DQ,ACC
 1040 FORMAT(' CF2 HAS FAILED TO CONVERGE AFTER ',F7.0,' ITERATIONS',/
     X,' P,Q,DP,DQ,ACCUR =  ',1P,4D17.7,D12.3,//)
      RETURN
  160 IFAIL =  3
      WRITE (6,1050) P,Q,ACC,DELL,LXTRA,M1
 1050 FORMAT(' FINAL Q.LE.DABS(P)*ACC*10**4 , P,Q,ACC = ',1P,3D12.3,4X,
     X' DELL,LXTRA,M1 = ',D12.3,2I5,/)
      RETURN
 1060 FORMAT(' XLMAX - XLMIN = DELL NOT AN INTEGER ',1P,3D20.10,/)
      END
      SUBROUTINE JWKB(XX,ETA1,XL,FJWKB,GJWKB,IEXP)
      DOUBLE PRECISION          XX,ETA1,XL,FJWKB,GJWKB,DZERO
C *** COMPUTES JWKB APPROXIMATIONS TO COULOMB FUNCTIONS    FOR DL.GE. 0
C *** AS MODIFIED BY BIEDENHARN ET AL. PHYS REV 97 (1955) 542-554
C *** CALLS DMAX1,SQRT,ALOG,EXO,ATAN2,DBLE,INT        BARNETT FEB 1981
      DATA   ZERO,HALF,ONE,SIX,TEN/ 0.0d0, 0.5d0, 1.0d0, 6.0d0, 10.0d0 /
      DATA  DZERO, RL35, ALOGE  /0.0D0, 35.0d0, 0.43429 45 d0 /
      X     = XX
      ETA   = ETA1
      GH2   = X*(ETA + ETA - X)
      XLL1  = DMAX1(XL*XL + XL,DZERO)
      IF(GH2 + XLL1 .LE. ZERO) RETURN
       HLL  = XLL1 + SIX/RL35
       HL   = SQRT(HLL)
       SL   = ETA/HL + HL/X
       RL2  = ONE + ETA*ETA/HLL
       GH   = SQRT(GH2 + HLL)/X
       PHI  = X*GH - HALF*( HL*ALOG((GH + SL)**2/RL2) - ALOG(GH) )
          IF(ETA .NE. ZERO) PHI = PHI - ETA*ATAN2(X*GH,X - ETA)
      PHI10 = -PHI*ALOGE
      IEXP  =  INT(PHI10)
      IF(IEXP .GT. 70) GJWKB = TEN**(PHI10 - DBLE(IEXP))
      IF(IEXP .LE. 70) GJWKB = EXP(-PHI)
      IF(IEXP .LE. 70) IEXP  = 0
      FJWKB = HALF/(GH*GJWKB)
      RETURN
      END
      SUBROUTINE DECAY(SKJ,L,R,U,UDER,IWRITE)
      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
      DIMENSION A(31)
C
C***********************************************************************
C
C     GIVES THE EXPONENTIALLY DECAYING SOLUTION AT R OF THE EQUATION
C
C      2     2    2
C     D U /DR +(-K-L(L+1)/(R*R))U =0
C        L                       L
C
C     WHERE K*K IS A REAL POSITIVE QUANTITY,L IS THE ANGULAR MOMENTUM
C
C     THE SOLUTION IS OBTAINED BY EXPANDING THE SOLUTION AS
C
C     U = EXP(-K*R)*((A(1)+A(2)/R+A(3)/R**2+ ::::::: ))
C     AND USING THE RECURSION RELATION
C
C     A     =((L*L+L-N*N-N)/((2*K*(N+1))*A  WITH A   =1
C      N+1                                N       0
C
C     THE SERIES TERMINATES WHEN N=L
C
C***********************************************************************
C
      DATA ZERO/0.0D0/,ONE/1.0D0/,TWO/2.0D0/
      DATA LMAX/30/
C
 1000 FORMAT(' ERROR IN DECAY ... SK =',D16.8,/)
 1010 FORMAT(' ERROR IN DECAY ... L =',I10,' LMAX =',I10,/)
C
      SK=-SKJ
C
      IF(SK .LT. ZERO) GO TO 50
      IF(L .GT. LMAX) GO TO 60
C
      ROOTK=DSQRT(SK)
      TWOR=TWO*ROOTK
      FACTOR=DEXP(-ROOTK*R)
      IF(L .EQ. 0) GO TO 30
      A(1)=ONE
      XLP=DBLE(L*(L+1))
      N=L
      DO J=1,N
        A(J+1)=A(J)*(XLP-Dble((J-1)*J))/(TWOR*Dble(J))
      END DO
C
      SUM1=ONE
      SUM2=ZERO
      DO JJ=1,N
        SUM1=SUM1+A(JJ+1)/(R**JJ)
        SUM2=SUM2+(A(JJ+1)*Dble(JJ))/(R**(JJ+1))
      END DO
      GO TO 40
   30 U=FACTOR
      UDER=-ROOTK*U
      RETURN
C
   40 U=FACTOR*SUM1
      UDER=-FACTOR*SUM2-ROOTK*U
      RETURN
C
   50 WRITE(IWRITE,1000)SK
      GO TO 70
   60 WRITE(IWRITE,1010)L,LMAX
C
   70 STOP 70
      END
